/**
 * This class SacSmaState was originally written by Chip Gobs for SSHP and is used in dhm. For using in FEWS, Xiaoshen
 * Li has modified it: mainly got rid of fields that are not used to make the class smaller(9/18/2008).
 * 
 * @author Chip Gobs
 * @version 6/20/03
 */
package ohd.hseb.model.sacsma;

import java.text.DecimalFormat;
import java.text.NumberFormat;

import ohd.hseb.db.DbTimeHelper;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.fews.OHDConstants;

final public class SacSmaState
{
    private long _validTime = 0; //the time corresponding to current state 

    /** --------------all state default initial values are 0.0 --------------- */
    private double _uztwc = 0.0;
    private double _uzfwc = 0.0;
    private double _lztwc = 0.0;
    private double _lzfsc = 0.0;
    private double _lzfpc = 0.0;
    private double _adimc = 0.0;

    //RSUM, PPE, PSC in card 6A(optional) are not used

    //Frozen Ground Calculation
    private double _fgix = 0.0; //sometimes called FINDX. We decided to consolidate to FGIX

    //PTA, PWE in card F3 are not used

    /**
     * Default constructor
     */
    public SacSmaState()
    {
        //do nothing
    }

    /**
     * A constructor which takes the current state time(long) as parameter.
     */
    public SacSmaState(final long currentTime)
    {
        this.setValidTime(currentTime);
    }

    public void setValidTime(final long validTime)
    {
        _validTime = validTime;
    }

    public long getValidTime()
    {
        return _validTime;
    }

    public void setUztwc(final double value)
    {
        if(value == OHDConstants.MISSING_DATA)
        {
            return;
        }
        this._uztwc = value;
    }

    public double getUztwc()
    {
        return _uztwc;
    }

    public void setUzfwc(final double value)
    {
        if(value == OHDConstants.MISSING_DATA)
        {
            return;
        }
        this._uzfwc = value;
    }

    public double getUzfwc()
    {
        return _uzfwc;
    }

    public void setLztwc(final double value)
    {
        if(value == OHDConstants.MISSING_DATA)
        {
            return;
        }
        this._lztwc = value;
    }

    public double getLztwc()
    {
        return _lztwc;
    }

    public void setLzfsc(final double value)
    {
        if(value == OHDConstants.MISSING_DATA)
        {
            return;
        }
        this._lzfsc = value;
    }

    public double getLzfsc()
    {
        return _lzfsc;
    }

    public void setLzfpc(final double value)
    {
        if(value == OHDConstants.MISSING_DATA)
        {
            return;
        }
        this._lzfpc = value;
    }

    public double getLzfpc()
    {
        return _lzfpc;
    }

    public void setAdimc(final double value)
    {
        if(value == OHDConstants.MISSING_DATA)
        {
            return;
        }
        this._adimc = value;
    }

    public double getAdimc()
    {
        return _adimc;
    }

    /**
     * Return frost index, DEGC.
     */
    public double getFgix()
    {
        return _fgix;
    }

    /**
     * Set frozen ground index, DEGC.
     */
    public void setFgix(final double fgix)
    {
        if(fgix == OHDConstants.MISSING_DATA)
        {
            return;
        }
        _fgix = fgix;
    }


    /**
     * A generic method to re-set 7 kinds of state values: 6 SacSma state plus frozen ground state FGIX.
     * 
     * @param stateName - state string name
     * @param newValue - new value of the state
     */
    public void updateState(final String stateName, final double newValue)
    {

        if(stateName.equals(OHDConstants.SAC_STATE_UZTWC))
        {
            this.setUztwc(newValue);
        }
        else if(stateName.equals(OHDConstants.SAC_STATE_UZFWC))
        {
            this.setUzfwc(newValue);
        }
        else if(stateName.equals(OHDConstants.SAC_STATE_LZTWC))
        {
            this.setLztwc(newValue);
        }
        else if(stateName.equals(OHDConstants.SAC_STATE_LZFSC))
        {
            this.setLzfsc(newValue);
        }
        else if(stateName.equals(OHDConstants.SAC_STATE_LZFPC))
        {
            this.setLzfpc(newValue);
        }
        else if(stateName.equals(OHDConstants.SAC_STATE_ADIMC))
        {
            this.setAdimc(newValue);
        }
        else if(stateName.equals(OHDConstants.SAC_STATE_FGIX))
        {
            this.setFgix(newValue);
        }

        //if state name does not equal to one of the 7 names above, this method cannot re-set its value.
    }

    @Override
    public String toString()
    {
        final String formatString = "#####.#####";
        final NumberFormat format = new DecimalFormat(formatString);
        final StringBuffer sb = new StringBuffer();
        sb.append("valid time = ").append(DateTime.getDateTimeStringFromLong(_validTime,OHDConstants.GMT_TIMEZONE)).append(" uztwc = ").append(format.format(_uztwc)).
        append(" uzfwc = ").append(format.format(_uzfwc)).append(" lztwc = ").append(format.format(_lztwc)).
        append(" lzfsc = ").append(format.format(_lzfsc)).append("\n").append(" lzfpc = ").append(format.format(_lzfpc)).append(" adimc = ").
        append(format.format(_adimc));
//        final String string = "valid time = " + DbTimeHelper.getDateTimeStringFromLongTime(_validTime) + " uztwc = "
//            + format.format(_uztwc) + " uzfwc = " + format.format(_uzfwc) + " lztwc = " + format.format(_lztwc)
//            + " lzfsc = " + format.format(_lzfsc) + "\n" + " lzfpc = " + format.format(_lzfpc) + " adimc = "
//            + format.format(_adimc);

        return sb.toString();
    }

}
