package ohd.hseb.ohdfewsadapter.util;

import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.Stack;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;

/**
 *  XML 3w DOM utilities traverse the top node to all its children nodes
 *  and their attributes
 * @author <a href="mailto:Raymond.Chui@noaa.gov">Raymond.Chui@noaa.gov</a>
 * @since CHPS created 2008, April
 * @version 1.0
 * @see RandomAccessFile org.w3c.dom.*
 */
public class FEWSXMLDOMUtilities {
	/** top node */
	private Node _node;
	/** print out */
	private RandomAccessFile _out;
	/** stack for the indentation */
	private Stack<Character> _stack;
	/** 
	 * The constructor
	 * @param out -- the buffered writer output
	 */
	public FEWSXMLDOMUtilities(RandomAccessFile out) {
		_out = out;
		_stack = new Stack<Character>();
	}
	/**
	 * Constructor
	 */
	public FEWSXMLDOMUtilities() {
		_stack = new Stack<Character>();
	}
	/**
	 * Set file write to
	 * @param out -- file output
	 */
	public void setRandomAccessFile(RandomAccessFile out) {
		_out = out;
	}
	/**
	 * Get file write to
	 * @return file output
	 */
	public RandomAccessFile getRandomAccessFile() {
		return _out;
	}
	/**
	 * Set the character stack
	 * @param stack set this stack
	 */
	public void setStack(Stack<Character> stack) {
		_stack = stack;
	}
	/**
	 * Get the character stack
	 * @return stack
	 */
	public Stack<Character> getStack() {
		return _stack;
	}
	/**
	 * Clear the stack
	 */
	public void clearStack() {
		_stack.clear();
	}
	/**
	 * Push a character to stack
	 * @param character -- push this
	 */
	public void pushStack(Character character) {
		_stack.push(character);
	}
	/**
	 * Pop a character from stack
	 * @return character
	 */
	public Character popStack() {
		return _stack.pop();
	}
	/**
	 * Is is an empty stack
	 * @return true when it empty; false otherwise
	 */
	public boolean isEmptyStack() {
		return _stack.isEmpty();
	}
	/**
	 * Peek the top of stack without popping it
	 * @return the top of stack
	 */
	public Character peekStack() {
		return _stack.peek();
	}
	/**
	 * Get stack size
	 * @return the size of stack
	 */
	public int getStackSize() {
		return _stack.size();
	}
	/**
	 * Set the top node
	 * @param node -- top node
	 */
	public void setNode(Node node) { _node = node; }
	/**
	 * Get the top node
	 * @return the top node
	 */
	public Node getNode() { return _node; }
	/**
	 * Write text string to the output
	 * @param stringBuilder - write this object's string
	 * @throws IOException
	 */
	public void writeBytes(StringBuilder stringBuilder) throws IOException {
		_out.writeBytes(stringBuilder.toString());
	}
	  
    /**
	 * Traverse a list of attributes
	 * @param namedNodeMap -- the attribute named of node mape
	 * @param stringBuilder -- append the XML document into this string builder/buffer
	 */
    public void traverseAttributes(NamedNodeMap namedNodeMap, 
    		StringBuilder stringBuilder,
    		String attributeHead) 
    {
    	for (int i = 0; i < namedNodeMap.getLength(); i++) {
			printAttribute((Attr)namedNodeMap.item(i), stringBuilder, attributeHead);
		}
    } // end method
     /**
     * Print an attribute name and value pair
     * @param attribute -- the attribute
     * @param stringBuilder -- append the XML document into this string builder/buffer
     */
    public void printAttribute(Attr attribute, StringBuilder stringBuilder, String attributeHead) 
    {
    	stringBuilder.append(attributeHead + attribute.getName() + 
    			"=\"" + attribute.getValue() + "\"");
    } // end of method
	
    /**
     * Get the parent node from an element node
     * @param node -- from this element node
     * @return its parent node; null if the node type is document
     */
    public Node getParentNodes(Node node) {
    	if (node.getNodeType() == Node.DOCUMENT_NODE)
    		return null;
    	return node.getParentNode();
    }
    /**
     * Get the hierarchy level count from a node, stop counting at the top node
     * @param node -- from this node
     * @param count -- start from the count number
     * @return the count number
     */
    public int getTopNodeLevel(Node node, int count) {
    	if (node.getParentNode() != null) {
    		count++;
    		count = getTopNodeLevel(node.getParentNode(), count); // recursive call
    	}
    	    return count;
    }
   
	 /**
	 * Traverse a node for document, element plus attributes and text node types
	 * @param node -- a node
	 * @param stringBuilder -- append the XML document into this string builder/buffer
	 * @param indents -- indentation, i.e 4-space, tab
	 */
	public void traverseXMLDocument(Node node, StringBuilder stringBuilder, String indents) {
		// first check the node type
		switch (node.getNodeType()) {
		case Node.DOCUMENT_NODE:	
			stringBuilder.append("<?xml");
			Document document = (Document)node;
			stringBuilder.append(" version=\"" + document.getXmlVersion() + "\"" +
					" standalone=\"" + (document.getXmlStandalone() ? "yes" : "no")  
				//	+ "\"" +
				//	" encoding=\"" + 
				//	(document.getXmlEncoding() == null ? "UTF-8" :  document.getXmlEncoding())
					+ "\" ?>");
			stringBuilder.append('\n');
			// recursive call
			traverseXMLDocument((Node)document.getDocumentElement(), stringBuilder, indents);
			break;
		case Node.ELEMENT_NODE:
			Element element = (Element)node;	
			int nodes = getTopNodeLevel(node, 0);
			String attributeHead = " ";
			if (nodes == 1) { // it is the top node
				attributeHead = "\n    "; // the top node usually has XML schema attributes
			}
			stringBuilder.append("<" + element.getTagName());
			if (element.hasAttributes()) { // traverse the attributes 
				traverseAttributes(element.getAttributes(), stringBuilder, attributeHead);
			}
			
			if (! element.hasChildNodes() && element.getTextContent().length() <= 0) {
			    stringBuilder.append(" />"); // an empty element (no close tag)
			    stringBuilder.append('\n');     
			} else if (element.hasChildNodes() &&
					element.getFirstChild().getNodeType() != Node.TEXT_NODE) {
				//System.out.println(element.getNodeName() + " has no next sibling");
				stringBuilder.append(">\n"); // a tag without text content
			} else {
				stringBuilder.append(">"); // a tag with text content
			}
			
			if (element.hasChildNodes()) {
				_stack.push('\t'); // push a tab every time has a child
			    NodeList nodeList = element.getChildNodes();
			    for (int i = 0; i < nodeList.getLength(); i++) {
			    	if (element.getFirstChild().getNodeType() != Node.TEXT_NODE) {
			    		// print the tabs for this case only
			    		for (int j = 0; j < _stack.size(); j++) {
			    			stringBuilder.append(indents);
			    		} // end little for loop
			    	} // end if
			    	// recursive call
			    	traverseXMLDocument(nodeList.item(i), stringBuilder, indents);			    	 
			    } //end inner for loop
			  
			    // end of an element
			    if (element.getFirstChild().getNodeType() != Node.TEXT_NODE) {
			        // print the tabs for this case only
			        for (int j = 1; j < _stack.size(); j++) {
			        	stringBuilder.append(indents);
			        } // end little for loop
			    } // end if
			    stringBuilder.append("</" + element.getTagName() + ">");
			    stringBuilder.append('\n');
			    if (!_stack.empty())
		            _stack.pop(); // pop a tab back when a child end
			}  // end outer if
			break;
		case Node.TEXT_NODE:
			Text text = (Text)node;
			//The nodeName always is #text; Therefore, no need _out.write(text.getNodeName());
			stringBuilder.append(text.getNodeValue());
			break;		
		default:
			System.out.println("Other Node");
			break;
		} // end switch
	} // end method
}