package ohd.hseb.ohdfewsadapter.util;

import java.util.LinkedHashMap;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.ext.DefaultHandler2;

/**
 * This class is a handler read through a time series XML file by 3W SAX and store it in 3W DOM Document object
 * 
 * @author <a href="mailto:Raymond.Chui@noaa.gov">Raymond.Chui@noaa.gov</a>
 * @since CHPS, created July 2008
 * @version 1.0
 */
public class FEWSXMLSAXHandler extends DefaultHandler2
{
    /** An empty tag */
    private boolean _anEmptyTag = true;
    /** The document */
    private Document _document;
    /** Linked hash map for the attributes */
    private LinkedHashMap<String, String> _linkHashMap;
    /** Text content */
    private String _textContent = "";
    /** Time series element */
    private Element _timeSeriesTag;
    /** Series element */
    private Element _seriesTag;
    /** Series header element */
    private Element _headerTag;
    /** Debug level 0 - 9 */
    protected int FEWSDEBUG = -1;

    /**
     * The Constructor
     */
    public FEWSXMLSAXHandler()
    {
        super();
    }

    /**
     * Get document
     * 
     * @return the document
     */
    public Document getDocument()
    {
        return _document;
    }

    /**
     * Set document
     * 
     * @param document - set this document
     */
    public void setDocument(Document document)
    {
        _document = document;
    }

    /**
     * Set debug level
     * 
     * @param level - debug level 0 - 9
     */
    public void setFEWSDebug(int level)
    {
        FEWSDEBUG = level;
    }

    /**
     * Get debug level
     * 
     * @return debug level 0 - 9
     */
    public int getFEWSDebug()
    {
        return FEWSDEBUG;
    }

    @Override
    public void startDocument() throws SAXException
    {
        if(FEWSDEBUG > 0)
            System.out.println("Start the document ************************");
        DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
        try
        { // create the document if it hasn't built, yet.
            DocumentBuilder documentBuilder = documentBuilderFactory.newDocumentBuilder();
            if(_document == null)
                _document = documentBuilder.newDocument();
            _document.setXmlVersion("1.0");
            _document.setXmlStandalone(true);
            // initial time series, series and header elements
            _timeSeriesTag = _document.createElement("TimeSeries");
            _seriesTag = _document.createElement("series");
            _headerTag = _document.createElement("header");
        }
        catch(ParserConfigurationException e)
        {
            e.printStackTrace();
        }
    }

    @Override
    public void endDocument() throws SAXException
    {
        if(FEWSDEBUG > 0)
            System.out.println("end document *****************");
    }

    @Override
    public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException
    {
        if(FEWSDEBUG > 0)
            System.out.print("<" + qName);
        if(qName.equalsIgnoreCase("event") || qName.equalsIgnoreCase("timeStep") || qName.equalsIgnoreCase("startDate")
            || qName.equalsIgnoreCase("endDate"))
            _anEmptyTag = true; // these are empty tags in time series XML file
        else
            _anEmptyTag = false;

        if(qName != null)
        {
            if(qName.equals("series"))
            { // the series tags are under time series tag
                _seriesTag = _document.createElement("series");
                _timeSeriesTag.appendChild(_seriesTag);
            }
            if(qName.equals("header"))
            { // a header tag is under a series tag
                _headerTag = _document.createElement("header");
                _seriesTag.appendChild(_headerTag);
            }
            if(qName.equals("TimeSeries"))
            { // time series tag is the top node of the document
                _timeSeriesTag = _document.createElement("TimeSeries");
                _document.appendChild(_timeSeriesTag);
            }
        }

        _linkHashMap = new LinkedHashMap<String, String>();
        _textContent = new String();
        if(attributes.getLength() >= 1)
        { // if there are attributes, save them in linked hash map
            for(int i = 0; i < attributes.getLength(); i++)
            {
                if(FEWSDEBUG > 0)
                    System.out.print(" " + attributes.getQName(i) + "=\"" + attributes.getValue(i) + "\"");
                _linkHashMap.put(attributes.getQName(i), attributes.getValue(i));
                if(qName.equals("TimeSeries")) // set the attributes for the time series tag, now
                    _timeSeriesTag.setAttribute(attributes.getQName(i), attributes.getValue(i));
            } // end for loop
        }

        if(_anEmptyTag)
        {
            if(FEWSDEBUG > 0)
                System.out.println(" />");
        }
        else
        {
            if(FEWSDEBUG > 0)
                System.out.print(">");
        }
    }

    @Override
    public void endElement(String uri, String localName, String qName) throws SAXException
    {
        Element element = null;
        if(qName.equalsIgnoreCase("event") || qName.equalsIgnoreCase("timeStep") || qName.equalsIgnoreCase("startDate")
            || qName.equalsIgnoreCase("endDate"))
        {
            _anEmptyTag = true; // these are empty tags
        }
        else
            _anEmptyTag = false;

        if(!_anEmptyTag && !_linkHashMap.isEmpty())
        { // a tag has text content and attributes
            if(FEWSDEBUG > 0)
                System.out.print("</" + qName + ">");
            element = createAnElement(qName, _textContent, _linkHashMap, _document);
        }
        else if(!_anEmptyTag && _linkHashMap.isEmpty())
        { // a tag has text content only
            if(FEWSDEBUG > 0)
                System.out.print("</" + qName + ">");
            element = createAnElement(qName, _textContent, null, _document);
        }
        else if(_anEmptyTag && !_linkHashMap.isEmpty())
        { // an empty tag has attributes
            element = createAnElement(qName, null, _linkHashMap, _document);
        }
        else if(_anEmptyTag && _linkHashMap.isEmpty())
        { // an empty tag no attributes
            ; // not apply
        }

        if(FEWSDEBUG > 2)
        {
            System.out.println("element node name = " + element.getNodeName());
        }

        if(qName.equals("event"))
        {
            _seriesTag.appendChild(element); // event tags are children of a series tag
        }
        else if(qName.equals("TimeSeries") || qName.equals("series") || qName.equals("header"))
        {
            ; // do nothing
        }
        else if(qName.equals("timeZone"))
        { // time zone tag is child of time series tag
            _timeSeriesTag.appendChild(element);
        }
        else
        { // header tag is a parent of other tags
            _headerTag.appendChild(element);
        }
    }

    @Override
    public void characters(char[] ch, int start, int length) throws SAXException
    { // create the new string object
        _textContent = new String(ch, start, length);

        if(ch.length > 0 && !_anEmptyTag)
        {
            if(FEWSDEBUG > 0)
                System.out.print(_textContent);
        }
    }

    /**
     * Create a DOM element
     * 
     * @param tagName -- element tag name
     * @param textContent -- element text content, null if it is an empty element
     * @param attributes -- element attributes in order, null if it has no attributes
     * @param document - the document for create element
     * @return an element
     */
    public static Element createAnElement(String tagName,
                                          String textContent,
                                          LinkedHashMap<String, String> attributes,
                                          Document document)
    {
        Element element = document.createElement(tagName);
        if(textContent != null)
            element.setTextContent(textContent);
        if(attributes != null && !attributes.isEmpty())
        {
            for(String key: attributes.keySet())
            {
                element.setAttribute(key, attributes.get(key));
            }
//            for (Iterator<String> iterator = attributes.keySet().iterator(); iterator.hasNext();)
//            {
//                String key = iterator.next();
//                element.setAttribute(key, attributes.get(key));
//            }
        }
        return element;
    }

}