package ohd.hseb.ohdfewsadapter.util.readWriteBinaryData;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.Locale;
import java.util.TimeZone;

import javax.xml.parsers.ParserConfigurationException;

import ohd.hseb.ohdfewsadapter.util.FEWSXMLDOMUtilities;
import ohd.hseb.ohdfewsadapter.util.FEWSXMLSAXHandler;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * This class will let you read a binary data file and a time series
 * header XML file. Then combine or merge them into an XML 3W DOM document object
 * @author <a href="mailto:Raymond.Chui@noaa.gov">Raymond.Chui@noaa.gov</a>
 * @since CHPS, August, 2008
 * @version 1.0
 * @see ReadTimeSeriesXMLFile, ReadTimeSeriesBinaryFile
 */

public class MergeTimeSeriesBinaryData2XMLFile {
	/** Time series header XML file reader */
	private ReadTimeSeriesXMLFile _readTimeSeriesXMLFile;
	/** Time series binary reader */
	private ReadTimeSeriesBinaryFile _readTimeSeriesBinaryFile;
	private StringBuilder _oneStringBuilder;
	private StringBuilder _combineStringBuilder;
	/** debug level */
	private int _FEWSDEBUG = -1;
	
	/** A constructor */
	public MergeTimeSeriesBinaryData2XMLFile() {
			
		_readTimeSeriesBinaryFile = new ReadTimeSeriesBinaryFile();
		_readTimeSeriesXMLFile = new ReadTimeSeriesXMLFile();
	}
	/**
	 * Set debug level
	 * @param FEWSDEBUG - debug level 0 - 9
	 */
	public void setFEWSDebug(int FEWSDEBUG) {
		_FEWSDEBUG =FEWSDEBUG; 
	}
	/**
	 * Get debug level
	 * @return debug level
	 */
	public int getEWSDebug() {
		return _FEWSDEBUG;
	}
	
	public void setOneBuilder(StringBuilder oneStringBuilder) {
		_oneStringBuilder = oneStringBuilder;
	}
	public void setCombineBuilder(StringBuilder combineStringBuilder) {
		_combineStringBuilder = combineStringBuilder;
	}
	public StringBuilder getOneBuilder() {
		return _oneStringBuilder;
	}
	public StringBuilder getCombineBuilder() {
		return _combineStringBuilder;
	}
	/**
	 * Read the time series binary data file
	 * @param binaryFile -- binary file name
	 * @throws FileNotFoundException
	 * @throws IOException
	 */
	public void readBinaryFile(final String binaryFile) throws 
	    FileNotFoundException, IOException {
		
		_readTimeSeriesBinaryFile.readBinaryFile(binaryFile);
		
	}
	
	/**
	 * Read the time series headers without event data from a XML file
	 * @param xmlFile -- XML file name
	 * @throws FileNotFoundException
	 * @throws ParserConfigurationException
	 * @throws SAXException
	 * @throws IOException
	 */
	public void readXMLFile(final String xmlFile) throws 
	FileNotFoundException, 
	ParserConfigurationException, 
	SAXException, 
	IOException {	
		_readTimeSeriesXMLFile.readXMLFile(xmlFile);	
	}
	/**
	 * Write the combined or merged XML document to a XML file
	 * @param xmlFileName -- XML file name
	 * @throws IOException
	 */
	public void writeXMLFile(final String xmlFileName) throws IOException {
		File outFile = new File(xmlFileName);
    	if (outFile.exists())
    		outFile.delete();
    	RandomAccessFile out = new RandomAccessFile(outFile, "rw");
    	FEWSXMLDOMUtilities domUtilites = new FEWSXMLDOMUtilities(out);
    	StringBuilder stringBuilder = new StringBuilder();
    	domUtilites.clearStack();
		domUtilites.traverseXMLDocument(getReadTimeSeriesXMLFile().getXMLDocument(), 
				stringBuilder, "    ");
		domUtilites.writeBytes(stringBuilder);
		out.close();
	}
	/**
	 * Set binary file object
	 * @param readTimeSeriesBinaryFile - bianry object
	 */
	public void setReadTimeSeriesBinaryFile(ReadTimeSeriesBinaryFile readTimeSeriesBinaryFile) {
		_readTimeSeriesBinaryFile = readTimeSeriesBinaryFile;
	}
	/**
	 * Set XML document object
	 * @param readTimeSeriesXMLFile -- XML document object
	 */
	public void setReadTimeSeriesXMLFile(ReadTimeSeriesXMLFile readTimeSeriesXMLFile) {
		_readTimeSeriesXMLFile = readTimeSeriesXMLFile;
	}
	/**
	 * Get binary file object
	 * @return binary data object
	 */
	public ReadTimeSeriesBinaryFile getReadTimeSeriesBinaryFile() {
		return _readTimeSeriesBinaryFile;
	}
	/**
	 * Get XML 3W DOM Document object
	 * @return Document object
	 */
	public ReadTimeSeriesXMLFile getReadTimeSeriesXMLFile() {
		return _readTimeSeriesXMLFile;
	}
	/**
	 * Set debug level
	 * @param level -- debug level 0 -9
	 */
	public void setDebugLevel(int level) {
		_readTimeSeriesXMLFile.setDebugLevel(level);
		setFEWSDebug(level);
	}
	
	/**
	 * Check out whether the combined binary data and XML header file is
	 * the same as XML with data file.
	 * 
	 * @param binaryDataFile - The binary data file
	 * @param binaryHeaderFile - The XML header without event data file
	 * @param xmlFile - The XML file with event data
	 * @return true if the compare results are the same; false otherwise
	 * @throws FileNotFoundException
	 * @throws IOException
	 * @throws ParserConfigurationException
	 * @throws SAXException
	 */
	public boolean isTheMergeTheSameAsXML(
			String binaryDataFile,
			String binaryHeaderFile,
			String xmlFile) throws 
			FileNotFoundException, 
			IOException, 
			ParserConfigurationException, 
			SAXException
	{
		// read time series event data binary file
		readBinaryFile(binaryDataFile);
		// read XML series header file without event data
		readXMLFile(binaryHeaderFile);
		doMerge(); // combine them
		Document combineDocument = getReadTimeSeriesXMLFile().getXMLDocument();
		removeEventFlag(combineDocument);
		_combineStringBuilder = new StringBuilder();
		
		// read time series XML file with event data and save into a XML document
		ReadTimeSeriesXMLFile readTimeSeriesXMLFile = new ReadTimeSeriesXMLFile();		
		readTimeSeriesXMLFile.readXMLFile(xmlFile);		
		Document oneDocument = readTimeSeriesXMLFile.getXMLDocument();
		removeEventFlag(oneDocument);		
		_oneStringBuilder = new StringBuilder();
		
		 // save both documents into two string builder/buffer respectively	 
        FEWSXMLDOMUtilities domUtilites = new FEWSXMLDOMUtilities();
        domUtilites.clearStack();
        domUtilites.traverseXMLDocument(combineDocument, _combineStringBuilder, "    ");
        domUtilites.clearStack();
        domUtilites.traverseXMLDocument(oneDocument, _oneStringBuilder, "    ");
        
        if (_FEWSDEBUG > 0)
        {
        	System.out.println("One document ******\n" + _oneStringBuilder.toString());
        	System.out.println("Combine document ******\n" + _combineStringBuilder.toString());
        }
        
        // Do string comparison
        return _oneStringBuilder.toString().equals(_combineStringBuilder.toString());
	}
	
	/**
     * Remove the flag attribute in event tag, because we ignore it
     * 
     * @param document -- the XML 3W DOM document
     */
    public void removeEventFlag(Document document)
    {
        Element timeSeries = (Element)document.getFirstChild();
        NodeList serieses = timeSeries.getChildNodes();
        for(int i = 0; i < serieses.getLength(); i++)
        {
            Element series = (Element)serieses.item(i);
            NodeList events = series.getElementsByTagName("event");
            for(int j = 0; j < events.getLength(); j++)
            {
                Element event = (Element)events.item(j);
                event.removeAttribute("flag");
            }
        }
    }
	
	/**
	 * Combine or merge time series binary data and headers 
	 * into an XML 3W DOM document object
	 * @throws NumberFormatException
	 * @throws IndexOutOfBoundsException
	 */
	public void doMerge() throws NumberFormatException, 
	    IndexOutOfBoundsException {
		Element timeStep = null, startDate = null, endDate = null;
		
		Document document = getReadTimeSeriesXMLFile().getXMLDocument();
		ArrayList<Float> arrayList = getReadTimeSeriesBinaryFile().getArrayLinkList();
		if (document == null) {
			System.out.println("Null XML document object");
			return;
		}
		if (arrayList == null || arrayList.size() <= 0) {
			System.out.println("Null binary data");
			return;
		}
		int arrayListIndex = 0;
		// The first child is TimeSeries
		Element timeSeries = (Element)document.getFirstChild();
		NodeList serieses = timeSeries.getElementsByTagName("series");
		if (this._FEWSDEBUG > 0)
		    System.out.println("number of seriese = " + serieses.getLength());
		
		// The first child under each series is header
		// There are timeStep, startDate and endDate under each header
		for (int i = 0; i < serieses.getLength(); i++) {
			Element series = (Element)serieses.item(i);
			if (this._FEWSDEBUG > 0)
			    System.out.println(series.getNodeName() + " " + i);
			Element header = (Element)series.getFirstChild();
			NodeList headerElements = header.getChildNodes();
			// get timeStep, startDate and endDate
			for (int j = 0; j < headerElements.getLength(); j++) {
				Element element = (Element)headerElements.item(j);
				String nodeName = element.getNodeName();
				if (nodeName.equals("timeStep"))
					timeStep = element;
				else if (nodeName.equals("startDate"))
					startDate = element;
				else if (nodeName.equals("endDate"))
					endDate = element;
			} // end inner for loop
			if (this._FEWSDEBUG > 0)
			    System.out.print(timeStep.getNodeName() + " ");
			Attr unit = timeStep.getAttributeNode("unit");
			Attr multiplier = timeStep.getAttributeNode("multiplier");
			if (this._FEWSDEBUG > 0) {
			    System.out.print(unit.getNodeName() + " = " + unit.getNodeValue() + " ");
			    System.out.println(multiplier.getNodeName() + " = " + multiplier .getNodeValue());
			}
			// convert the time interval into milliseconds
			long timeInterval = 1000; // milliseconds
			if (unit.getNodeValue().equalsIgnoreCase("second")) {
				timeInterval *= (long)Integer.parseInt(multiplier .getNodeValue());
			} else if (unit.getNodeValue().equalsIgnoreCase("minute")) {
				timeInterval *= (long)60 * Integer.parseInt(multiplier .getNodeValue());
			} else if (unit.getNodeValue().equalsIgnoreCase("hour")) {
				timeInterval *= (long)60 * 60 * Integer.parseInt(multiplier .getNodeValue());
			}
			if (this._FEWSDEBUG > 0)			
			    System.out.println("time interval = " + timeInterval);
			Attr sDate = startDate.getAttributeNode("date");
			Attr sTime = startDate.getAttributeNode("time");
			if (this._FEWSDEBUG > 0) {
			    System.out.print(sDate.getNodeName() + " = " + sDate.getNodeValue() + " ");
			    System.out.println(sTime.getNodeName() + " = " + sTime .getNodeValue());
			}
			Calendar startCalendar = Calendar.getInstance(TimeZone.getTimeZone("GMT"), 
					Locale.getDefault());
			
			
			startCalendar.clear();
			startCalendar.set(Integer.parseInt(sDate.getNodeValue().substring(0, 4)), // year
					Integer.parseInt(sDate.getNodeValue().substring(5, 7))-1, //month, Jan is 0
					Integer.parseInt(sDate.getNodeValue().substring(8)), // day of month
					Integer.parseInt(sTime.getNodeValue().substring(0, 2)), // hour of day
					Integer.parseInt(sTime.getNodeValue().substring(3, 5)), // minute
					Integer.parseInt(sTime.getNodeValue().substring(7))); // second
			
			
			
			if (this._FEWSDEBUG > 0)
			    System.out.print(endDate.getNodeName() + " ");
			Attr eDate = endDate.getAttributeNode("date");
			Attr eTime = endDate.getAttributeNode("time");
			if (this._FEWSDEBUG > 0) {
			    System.out.print(eDate.getNodeName() + " = " + eDate.getNodeValue() + " ");
			    System.out.println(eTime.getNodeName() + " = " + eTime .getNodeValue());
			}
			Calendar endCalendar = Calendar.getInstance(TimeZone.getTimeZone("GMT"), 
					Locale.getDefault());
			endCalendar.clear();
			endCalendar.set(Integer.parseInt(eDate.getNodeValue().substring(0, 4)), // year
					Integer.parseInt(eDate.getNodeValue().substring(5, 7))-1, //month, Jan is 0
					Integer.parseInt(eDate.getNodeValue().substring(8)), // day of month
					Integer.parseInt(eTime.getNodeValue().substring(0, 2)), // hour of day
					Integer.parseInt(eTime.getNodeValue().substring(3, 5)), // minute
					Integer.parseInt(eTime.getNodeValue().substring(7))); // second
			
			SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");
			dateFormat.setTimeZone(TimeZone.getTimeZone("GMT"));
			SimpleDateFormat timeFormat = new SimpleDateFormat("HH:mm:ss");
			timeFormat.setTimeZone(TimeZone.getTimeZone("GMT"));
			
			if (this._FEWSDEBUG > 0) {
			    System.out.println("start calendar = " +
			    	dateFormat.format(startCalendar.getTime()) + " " +
			    	timeFormat.format(startCalendar.getTime())
				);
			    System.out.println("end calendar = " +
			    	dateFormat.format(endCalendar.getTime()) + " " +
				    timeFormat.format(endCalendar.getTime())
				);
			}
			
			while(startCalendar.getTime().getTime() <= endCalendar.getTime().getTime()) {
				LinkedHashMap<String, String> linkedHashMap = new LinkedHashMap<String, String>();
				
				if (this._FEWSDEBUG > 0) {
				    System.out.print("<event> ");
				    System.out.print("start calendar = " +
				    	dateFormat.format(startCalendar.getTime()) + " " +
					    timeFormat.format(startCalendar.getTime())
					);			    
				    //Float floatValue = ;
				    System.out.println(" data value = " + 
				    		arrayList.get(arrayListIndex).floatValue()
				    		);
				} // end if FEWSDEBUG
				linkedHashMap.put("date", 
						dateFormat.format(startCalendar.getTime())
						);
				linkedHashMap.put("time", 
						timeFormat.format(startCalendar.getTime())
						);
				linkedHashMap.put("value", arrayList.get(arrayListIndex).toString());
				// there is no way we know the value of the flag. just put a default value 1
				linkedHashMap.put("flag", "0");
				Element event = FEWSXMLSAXHandler.createAnElement("event", 
						null, // no text content, it is an empty tag
						linkedHashMap, // attributes
						document);
				series.appendChild(event);
				// increment the time interval
				startCalendar.setTime(new Date(timeInterval + startCalendar.getTime().getTime()));
				arrayListIndex++;
			} // end while loop
		} // end outer for loop	
	} // end method
} // end class