package ohd.hseb.ohdfewsadapter.util.timeseriesconverter;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.sql.Timestamp;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Locale;
import java.util.Properties;
import java.util.Scanner;
import java.util.StringTokenizer;
import java.util.TimeZone;
import java.util.Vector;
import java.util.regex.Pattern;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import ohd.hseb.ohdfewsadapter.util.FEWSXMLDOMUtilities;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.Diagnostics;
import ohd.hseb.util.fews.FewsAdapterDAO;
import ohd.hseb.util.fews.FewsRegularTimeSeries;
import ohd.hseb.util.fews.FewsXmlValidation;
import ohd.hseb.util.fews.NwsrfsDataTypeMappingReader;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.OHDFewsAdapterConstants;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Mainly convert time series text file and/or its card text file into XML file format
 * 
 * @author <a href="mailto:Raymond.Chui@noaa.gov">Raymond.Chui@noaa.gov</a>
 * @version 1.0
 * @since CHPS created April, 2008
 * @see FewsTimeSeriesConverterConstants
 */
public class FewsTimeseriesConverter
{
    /** this is for java.util.Scanner class */
    private final Pattern _pattern = Pattern.compile("\\s");
    /** the XML document top node */
    private Document _document;
    /** the XML root element tag under the document */
    private Element _timeSeries;
    /** the XML element modes under the XML header element tag */
    private LinkedHashMap<String, String> _headerElements;
    /** Random access file to an output XML file */
    private RandomAccessFile _out;
    /** existedXMLFile */
    private boolean _existedXMLFile = false;
    /** The date format */
    private final SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");
    /** The time format */
    private final SimpleDateFormat simpleTimeFormat = new SimpleDateFormat("HH:mm:ss");
    /** starting timestamp for XML output */
    private Calendar _startTimestamp;
    /** ending timestamp for XML output */
    private Calendar _endTimestamp;
    /** Forecast timestamp */
    private Calendar _forecastTimestamp;
    /** A vector for input line */
    private final Vector<String> _inputLinesVec;
    /** The decimal format */
    private final DecimalFormat _decimalFormat = new DecimalFormat(".000000");
    private int _valuesStep = 1;
    /** logger */
    private static Diagnostics _logger;
    /** the root tag name for the XML file */
    public final static String rootTag = "TimeSeries";
    protected int FEWSDEBUG = -1;

    /**
     * Constructor creates a DOM document object and initialized it
     * 
     * @param outputFile -- output XML file
     */
    public FewsTimeseriesConverter(final String outputFile)
    {
        _inputLinesVec = new Vector<String>();
        final File xmlOutFile = new File(outputFile);
        _logger = new Diagnostics();

        final DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
        try
        {
            final DocumentBuilder documentBuilder = documentBuilderFactory.newDocumentBuilder();

            if(_document == null)
            {
                _document = documentBuilder.newDocument();
                if(_timeSeries == null)
                {
                    _timeSeries = _document.createElement(rootTag);
                }

                if(xmlOutFile.exists())
                {
                    this.setExistedXMLFile(true);
                }
                else
                {
                    this.setExistedXMLFile(false);
                    _document.setXmlVersion("1.0");
                    _document.setXmlStandalone(true);

                    _timeSeries.setAttribute("version", "1.2");
                    _timeSeries.setAttribute("xmlns", "http://www.wldelft.nl/fews/PI");
                    _timeSeries.setAttribute("xmlns:xsi", "http://www.w3.org/2001/XMLSchema-instance");
                    _timeSeries.setAttribute("xsi:schemaLocation", "http://www.wldelft.nl/fews/PI" + " "
                        + "http://fews.wldelft.nl/schemas/version1.0/pi-schemas/pi_timeseries.xsd");

                    _document.appendChild(_timeSeries); // create a root tag

                    final int rawOffset = OHDConstants.GMT_TIMEZONE.getRawOffset() / (1000 * 60 * 60);
                    _timeSeries.appendChild(createAnElement("timeZone", Integer.toString(rawOffset), null));
                } // end else
            } // end outer if
              // create a random access file for read/write mode
            _out = new RandomAccessFile(xmlOutFile, "rw");
            // if write to an existed file, then go to 4th byte from bottom
            if(getExistedXMLFile() && xmlOutFile.length() > 4)
            {
                final long fileLength = _out.length();
                final Document existedDocument = documentBuilder.parse(xmlOutFile);
                final Element firstChild = (Element)existedDocument.getFirstChild();
                final String tagName = firstChild.getTagName();
                // push back tag length plus 4 characters, they are <,/, > and \n
                _out.seek(fileLength - (4 + tagName.length()));
                _out.writeBytes("    "); // indent 4 spaces
            }
        }
        catch(final Exception e)
        {
            _logger.log(Logger.ERROR, e.getMessage());
            e.printStackTrace();
        }

        simpleDateFormat.setTimeZone(OHDConstants.GMT_TIMEZONE);
        simpleTimeFormat.setTimeZone(OHDConstants.GMT_TIMEZONE);

        _startTimestamp = null;
        _endTimestamp = null;
        _forecastTimestamp = null;
    }

    public SimpleDateFormat getSimpleDateFormat()
    {
        return this.simpleDateFormat;
    }

    public SimpleDateFormat getSimpleTimeFormat()
    {
        return this.simpleTimeFormat;
    }

    /**
     * Set debug level
     * 
     * @param FEWSDEBUG -- debug level 0 - 9
     */
    public void setDebug(final int FEWSDEBUG)
    {
        this.FEWSDEBUG = FEWSDEBUG;
    }

    public Element getTimeseries()
    {
        return this._timeSeries;
    }

    /**
     * Get debug level
     * 
     * @return the debug level
     */
    public int getDebug()
    {
        return FEWSDEBUG;
    }

    public Diagnostics getLogger()
    {
        return _logger;
    }

    /**
     * This method set starting offset timestamp for XML output, it is different from the start date and time
     * 
     * @param startTimestamp -- the offset timestamp
     * @throws IllegalArgumentException if it is not yyyy-mm-dd HH:MM:SS format
     */
    public void setStartTimestamp(final String startTimestamp) throws IllegalArgumentException
    {
        //System.out.println("startTimestamp = " + startTimestamp);
        final Timestamp timestamp = Timestamp.valueOf(startTimestamp + ".0");
        if(_startTimestamp == null)
        {
            _startTimestamp = Calendar.getInstance(OHDConstants.GMT_TIMEZONE, Locale.getDefault());
        }
        final long offset = TimeZone.getDefault().getOffset(timestamp.getTime());

        _startTimestamp.setTimeInMillis(timestamp.getTime() + offset);
        _startTimestamp.set(Calendar.MILLISECOND, 0);
        if(FEWSDEBUG > -1)
        {
            System.out.println("default raw offset = " + offset);
            System.out.println("_startTimestamp = " + this.simpleDateFormat.format(_startTimestamp.getTime()) + " "
                + this.simpleTimeFormat.format(_startTimestamp.getTime()));
        }
    }

    /**
     * Set start time stamp
     * 
     * @param startTimestamp - start time stamp
     */
    public void setStartTimestamp(final Calendar startTimestamp)
    {
        _startTimestamp = startTimestamp;
    }

    /**
     * This method set ending offset timestamp for XML output, it is different from the start date and time
     * 
     * @param endTimestamp -- the offset timestamp
     * @throws IllegalArgumentException if it is not yyyy-mm-dd HH:MM:SS format
     */
    public void setEndTimestamp(final String endTimestamp) throws IllegalArgumentException
    {
        //System.out.println("startTimestamp = " + endTimestamp);
        final Timestamp timestamp = Timestamp.valueOf(endTimestamp + ".0");
        if(_endTimestamp == null)
        {
            _endTimestamp = Calendar.getInstance(OHDConstants.GMT_TIMEZONE, Locale.getDefault());
        }
        final long offset = TimeZone.getDefault().getOffset(timestamp.getTime());
        _endTimestamp.setTimeInMillis(timestamp.getTime() + offset);
        _endTimestamp.set(Calendar.MILLISECOND, 0);
    }

    /**
     * Set end time stamp
     * 
     * @param endTimestamp - end time stamp
     */
    public void setEndTimestamp(final Calendar endTimestamp)
    {
        _endTimestamp = endTimestamp;
    }

    /**
     * This method set ending offset timestamp for XML output, it is different from the start date and time
     * 
     * @param forecastTimestamp -- the offset timestamp
     * @throws IllegalArgumentException if it is not yyyy-mm-dd HH:MM:SS format
     */
    public void setForecastTimestamp(final String forecastTimestamp) throws IllegalArgumentException
    {
        //System.out.println("startTimestamp = " + endTimestamp);
        final Timestamp timestamp = Timestamp.valueOf(forecastTimestamp + ".0");
        if(_forecastTimestamp == null)
        {
            _forecastTimestamp = Calendar.getInstance(OHDConstants.GMT_TIMEZONE, Locale.getDefault());
        }
        final long offset = TimeZone.getDefault().getOffset(timestamp.getTime());
        _forecastTimestamp.setTimeInMillis(timestamp.getTime() + offset);
        _forecastTimestamp.set(Calendar.MILLISECOND, 0);
    }

    /**
     * Set forecast time stamp
     * 
     * @param forecastTimestamp - forecast time stamp
     */
    public void setForecastTimestamp(final Calendar forecastTimestamp)
    {
        _forecastTimestamp = forecastTimestamp;
    }

    /**
     * Reset the start time stamp
     */
    public void clearStartTimestamp()
    {
        this._startTimestamp = null;
    }

    /**
     * Reset the end time stamp
     */
    public void clearEndTimestamp()
    {
        this._endTimestamp = null;
    }

    /**
     * Reset the forecast time stamp
     */
    public void clearForecastTimestamp()
    {
        this._forecastTimestamp = null;
    }

    /**
     * Get starting offset time stamp
     * 
     * @return the offset time stamp
     */
    public Calendar getStartTimestamp()
    {
        return _startTimestamp;
    }

    /**
     * Get ending offset time stamp
     * 
     * @return the offset time stamp
     */
    public Calendar getEndTimestamp()
    {
        return _endTimestamp;
    }

    /**
     * Get forecast offset time stamp
     * 
     * @return the offset time stamp
     */
    public Calendar getForecastTimestamp()
    {
        return _forecastTimestamp;
    }

    /**
     * Is the forecast time stamp before (less than) start time stamp
     * 
     * @return true if forecast time stamp before start time stamp; false otherwise
     */
    public boolean isForecastTimestampBeforeStartTimestamp()
    {
        boolean isBefore = false;
        if(_startTimestamp != null && _forecastTimestamp != null)
        {
            if(_forecastTimestamp.getTime().getTime() < _startTimestamp.getTime().getTime())
            {
                isBefore = true; // forecast date and time can't less than start date and time
                System.err.println("Forecast date and time " + simpleDateFormat.format(_forecastTimestamp.getTime())
                    + " " + simpleTimeFormat.format(_forecastTimestamp.getTime())
                    + " must greater or equals to start date and time "
                    + simpleDateFormat.format(_startTimestamp.getTime()) + " "
                    + simpleTimeFormat.format(_startTimestamp.getTime()));
            }
        }
        return isBefore;
    }

    /**
     * Is the forecast time stamp after (greater than) end time stamp
     * 
     * @return true if forecast time stamp after end time stamp; false otherwise
     */
    public boolean isForecastTimestampAfterEndTimestamp()
    {
        boolean isAfter = false;
        if(_endTimestamp != null && _forecastTimestamp != null)
        {
            if(_forecastTimestamp.getTime().getTime() > _endTimestamp.getTime().getTime())
            {
                isAfter = true; // forecast date and time can't greater than end date and time
                System.err.println("Forecast date and time " + simpleDateFormat.format(_forecastTimestamp.getTime())
                    + " " + simpleTimeFormat.format(_forecastTimestamp.getTime())
                    + " must less or equals to end date and time " + simpleDateFormat.format(_endTimestamp.getTime())
                    + " " + simpleTimeFormat.format(_endTimestamp.getTime()));
            }
        }
        return isAfter;
    }

    /**
     * Is the forecast time stamp must between start time stamp and end time stamp
     * 
     * @return false if forecast time stamp is not between start/end; true otherwise
     */
    public boolean isForecastTimestampBetweenStartAndEnd()
    {
        boolean isInRange = true;

        if(_startTimestamp != null && _endTimestamp != null && _forecastTimestamp != null)
        {
            if(_forecastTimestamp.getTime().getTime() < _startTimestamp.getTime().getTime())
            {
                isInRange = false; // forecast date and time can't less than start date and time
                System.err.println("Forecast date and time " + simpleDateFormat.format(_forecastTimestamp.getTime())
                    + " " + simpleTimeFormat.format(_forecastTimestamp.getTime())
                    + " must greater or equals to start date and time "
                    + simpleDateFormat.format(_startTimestamp.getTime()) + " "
                    + simpleTimeFormat.format(_startTimestamp.getTime()));
            }
            if(_forecastTimestamp.getTime().getTime() > _endTimestamp.getTime().getTime())
            {
                isInRange = false; // forecast date and time can't greater than end date and time
                System.err.println("Forecast date and time " + simpleDateFormat.format(_forecastTimestamp.getTime())
                    + " " + simpleTimeFormat.format(_forecastTimestamp.getTime())
                    + " must less or equals to end date and time " + simpleDateFormat.format(_endTimestamp.getTime())
                    + " " + simpleTimeFormat.format(_endTimestamp.getTime()));
            }
            if(_forecastTimestamp.getTime().getTime() >= _startTimestamp.getTime().getTime()
                && _forecastTimestamp.getTime().getTime() <= _endTimestamp.getTime().getTime())
            { // forecast date and time lie between start and end date and time
                isInRange = true;
            }
        }
        return isInRange;
    }

    /**
     * Set status for existed XML file
     * 
     * @param existedXMLFile -- true if it is existed; Otherwise false
     */
    public void setExistedXMLFile(final boolean existedXMLFile)
    {
        _existedXMLFile = existedXMLFile;
    }

    /**
     * Get existed XML file
     * 
     * @return true if it is existed; Otherwise false
     */
    public boolean getExistedXMLFile()
    {
        return _existedXMLFile;
    }

    /**
     * Set the header elements
     * 
     * @param headerElements -- the header elements
     */
    public void setHeaderElements(final LinkedHashMap<String, String> headerElements)
    {
        _headerElements = headerElements;
    }

    /**
     * Get the header elements
     * 
     * @return the header elements
     */
    public LinkedHashMap<String, String> getHeaderElements()
    {
        return _headerElements;
    }

    /**
     * Create an DOM element
     * 
     * @param tagName -- element tag name
     * @param textContent -- element text content, null if it is an empty element
     * @param attributes -- element attributes in order, null if it has no attributes
     * @return an element
     */
    public Element createAnElement(final String tagName,
                                   final String textContent,
                                   final LinkedHashMap<String, String> attributes)
    {
        final Element element = _document.createElement(tagName);
        if(textContent != null)
        {
            element.setTextContent(textContent);
        }
        if(attributes != null && !attributes.isEmpty())
        {
            for(final Iterator<String> iterator = attributes.keySet().iterator(); iterator.hasNext();)
            {
                final String key = iterator.next();
                element.setAttribute(key, attributes.get(key));
            }
        }
        return element;
    }

    /**
     * Add the elements to the header
     * 
     * @param header -- the elements add to this header
     */
    protected void addHeaderElements(final Element header)
    {
        for(final Iterator<String> iterator = _headerElements.keySet().iterator(); iterator.hasNext();)
        {
            final String key = iterator.next();
            if(key.equals("DEBUG"))
            {
                continue; // debugging tag is not in our XML schema
            }
            header.appendChild(createAnElement(key, _headerElements.get(key), null));
        }
    }

    /**
     * Clear the Line Vector from super class
     */
    public void clearTheLinesVector()
    {
        if(_inputLinesVec != null && !_inputLinesVec.isEmpty())
        {
            _inputLinesVec.clear();
        }
    }

    /**
     * Search an element from a node list by the tag name
     * 
     * @param tagName -- search this tag
     * @param nodeList -- the node list
     * @return the found element, null if not found
     */
    public Element searchNodeList(final String tagName, final NodeList nodeList)
    {
        Element headerNode = null;
        for(int i = 0; i < nodeList.getLength(); i++)
        {
            headerNode = (Element)nodeList.item(i);
            if(headerNode.getTagName().equals(tagName))
            {
                break;
            }
            else
            {
                headerNode = null;
            }
        }
        return headerNode;
    }

    /**
     * Put the time series header in FEWS order
     * 
     * @param header -- time series header
     */
    public void putTimeSeriesHeaderInOrder(final Element header)
    {
        final NodeList nodeList = header.getChildNodes();
        final LinkedHashMap<String, Element> linkedHashMap = new LinkedHashMap<String, Element>();
        // save the header elements into a linked hash map in FEWS order
        for(int i = 0; i < FewsTimeseriesConverterConstants.timeSeriesHeader.length; i++)
        {
            final Element headerNode = searchNodeList(FewsTimeseriesConverterConstants.timeSeriesHeader[i], nodeList);
            if(headerNode != null)
            {
                linkedHashMap.put(FewsTimeseriesConverterConstants.timeSeriesHeader[i], headerNode);
                /*
                 * if (headerNode.getNodeName().equals("startDate")) System.out.println("header node " + i + " = " +
                 * headerNode.getNodeName() + " date = " + headerNode.getAttribute("date") + " time = " +
                 * headerNode.getAttribute("time"));
                 */
            }
        }
        linkedHashMap.remove("DEBUG");
        // clear the header
        for(int i = 0; i < nodeList.getLength(); i++)
        {
            final Element element = (Element)nodeList.item(i);
            header.removeChild(element);
        }
        // restore the header elements back from the saved link hash map
        for(final Iterator<String> iterator = linkedHashMap.keySet().iterator(); iterator.hasNext();)
        {
            final String key = iterator.next();
            final Element element = linkedHashMap.get(key);
            if(element == null)
            {
                continue;
            }
            header.appendChild(element);

            if(this.FEWSDEBUG > 0)
            { // for debugging only
                final String textContent = element.getTextContent();
                if(textContent != null)
                {
                    System.out.println(element.getTagName() + " " + textContent);
                }
                else
                {
                    System.out.println(element.getTagName() + " null");
                }

                if(element.hasAttributes())
                {
                    final NamedNodeMap attributes = element.getAttributes();
                    for(int i = 0; i < attributes.getLength(); i++)
                    {
                        final Attr attribute = (Attr)attributes.item(i);
                        System.out.println(attribute.getName() + " " + attribute.getValue());
                    } // end inner for loop
                } // end inner if
            } // end outer FEWSDEBUG if
        } // end outer for loop
    } // end method

    /**
     * Read a time series text file and create XML tags
     * 
     * @param fileName -- input text file name
     * @return success if true; otherwise false
     */
    public void readTSFile(final String fileName) throws Exception
    {
        clearTheLinesVector();
        readFile(fileName);
        final LinkedHashMap<String, String> linkedHashMap = new LinkedHashMap<String, String>();
        final Calendar calendar = Calendar.getInstance(OHDConstants.GMT_TIMEZONE);
        final Calendar cal = Calendar.getInstance(OHDConstants.GMT_TIMEZONE);
        Integer flag = Integer.valueOf(-1);
        Integer timeStep = Integer.valueOf(0);
        Element series = null;
        Element header = null;
        boolean createdStartDate = false;
        boolean hasDataStarted = false;
        int calendarDiff = 0;
        boolean hasFutureData = false;

        series = _document.createElement("series");
        _timeSeries.appendChild(series);

        header = _document.createElement("header");

        addHeaderElements(header);

        header.appendChild(createAnElement("creationDate", simpleDateFormat.format(calendar.getTime()), null));
        header.appendChild(createAnElement("creationTime", simpleTimeFormat.format(calendar.getTime()), null));
        final int thisYear = calendar.get(Calendar.YEAR);

        series.appendChild(header);
        final NodeList nList = header.getElementsByTagName("missVal");
        if(nList == null || nList.getLength() <= 0)
        {
            header.appendChild(createAnElement("missVal", "-999.0", null));
        }

        String aLine = "";
        for(int i = 0; i < _inputLinesVec.size(); i++)
        {
            aLine = _inputLinesVec.get(i);
            //System.out.println(aLine);
            if(aLine.toUpperCase().indexOf("ERROR") >= 0)
            {
                //continue; // skip error line
            }
            String aString = "";
            if(aLine.indexOf("BOTH") > 0)
            {
                hasFutureData = true;
            }
            if(aLine.indexOf("NWSRFS") >= 0)
            { // 1st if, beginning of file
                final StringTokenizer token = new StringTokenizer(aLine, "=");
//                String createdDate = "";
                token.nextElement();
                token.nextElement();
//                createdDate = (String)
                token.nextElement();
                //System.out.println("created date = " + createdDate.trim());
            } // end 1st if
            if(aLine.indexOf("FUTURE") >= 0)
            {
                if(this.FEWSDEBUG > 0)
                {
                    System.out.println("has future data = " + hasFutureData);
                }
                if(hasFutureData)
                {
                    if(_forecastTimestamp != null && _startTimestamp != null
                        && !isForecastTimestampBeforeStartTimestamp()) // good shape
                    {
                        if(this.FEWSDEBUG > 0)
                        {
                            System.out.println("forecast time stamp is after start time stamp");
                        }
                    }
                    else if(_forecastTimestamp != null && _startTimestamp != null
                        && isForecastTimestampBeforeStartTimestamp()) // bad shape
                    {
                        if(this.FEWSDEBUG > 0)
                        {
                            System.out.println("forecast time stamp is before start time stamp");
                        }
                        setForecastTimestamp(_startTimestamp); // Therefore, ignore it 
                    }

                    if(_forecastTimestamp != null && _endTimestamp != null && !isForecastTimestampAfterEndTimestamp()) // good shape
                    {
                        if(this.FEWSDEBUG > 0)
                        {
                            System.out.println("forecast time stamp is before end time stamp");
                        }
                    }
                    else if(_forecastTimestamp != null && _endTimestamp != null
                        && isForecastTimestampAfterEndTimestamp()) // bad shape
                    {
                        if(this.FEWSDEBUG > 0)
                        {
                            System.out.println("forecast time stamp is after end time stamp");
                        }
                        setForecastTimestamp(_endTimestamp); // Therefore, ignore it
                    }

                    if(_forecastTimestamp != null && _startTimestamp != null && _endTimestamp != null
                        && isForecastTimestampBetweenStartAndEnd()) // good shape
                    {
                        if(this.FEWSDEBUG > 0)
                        {
                            System.out.println("forecast time stamp is between start and end time stamp");
                        }
                    }
                    else if(_forecastTimestamp != null && _startTimestamp != null && _endTimestamp != null
                        && !isForecastTimestampBetweenStartAndEnd()) // bad shape
                    {
                        if(this.FEWSDEBUG > 0)
                        {
                            System.out.println("forecast time stamp is not between start and end time stamp");
                        }
                        _forecastTimestamp = null; // Therefore, ignore it
                    }

                    if(_forecastTimestamp != null && _startTimestamp == null && _endTimestamp == null)
                    {
                        if(this.FEWSDEBUG > 0)
                        {
                            System.out.println("since the end time stamp is not defined, then let it equals to forecast");
                        }
                        setEndTimestamp(_forecastTimestamp);
                    }
                    else if(_forecastTimestamp != null && _startTimestamp == null && _endTimestamp != null)
                    {
                        if(this.FEWSDEBUG > 0)
                        {
                            System.out.println("the end time stamp is defined");
                        }
                        ;
                    }
                    else if(_forecastTimestamp != null && _startTimestamp != null && _endTimestamp == null)
                    {
                        if(this.FEWSDEBUG > 0)
                        {
                            System.out.println("since the end time stamp is not defined, then let it equals to forecast");
                        }
                        setEndTimestamp(_forecastTimestamp);
                    }
                    else if(_forecastTimestamp == null && (_startTimestamp == null || _endTimestamp == null))
                    {
                        if(this.FEWSDEBUG > 0)
                        {
                            System.out.println("no forecast time stamp nor start/end time stamp defined, take all data");
                        }
                        ;
                    }

                    continue; // continue to next line for the future data
                }
                else
                {
                    break; // stop for the future data if regular
                }
            }
            if(aLine.indexOf("ID=") >= 0)
            { // 2nd if, the header information
                final Scanner scanner = new Scanner(aLine).useDelimiter(_pattern);
                aString = "";
                while(scanner.hasNext())
                {
                    aString = scanner.next().trim();
                    if(aString == null || aString.length() <= 0)
                    {
                        continue;
                    }
                    if(aString.indexOf("ID=") >= 0)
                    {
                        //System.out.println("ID .... " + aString);
                        final NodeList aNodeList = header.getElementsByTagName(FewsTimeseriesConverterConstants.headerCard1[FewsTimeseriesConverterConstants.statationID]);
                        // if the location Id is not existed, then create a new one
                        if(aNodeList.getLength() <= 0)
                        {
                            header.appendChild(createAnElement(FewsTimeseriesConverterConstants.headerCard1[FewsTimeseriesConverterConstants.statationID],
                                                               aString.substring(aString.indexOf('=') + 1),
                                                               null));
                            FewsTimeseriesConverter._logger.log(Logger.DEBUG,
                                                                "ID = " + aString.substring(aString.indexOf('=') + 1));
                        }
                        else if(_headerElements.containsKey("qualifierId"))
                        {
                            ; // don't take the qualifier id as location id
                        }
                        else
                        { // if it is existed, then change the text content
                            final Element anElement = (Element)aNodeList.item(0);
                            anElement.setTextContent(aString.substring(aString.indexOf('=') + 1));
                            FewsTimeseriesConverter._logger.log(Logger.DEBUG,
                                                                "ID = " + aString.substring(aString.indexOf('=') + 1));
                        }
                    }
                    else if(aString.indexOf("TYPE") >= 0)
                    {
                        final String value = aString.substring(aString.indexOf('=') + 1);

                        header.appendChild(createAnElement("type",
                                                           FewsRegularTimeSeries.getFewsTimeCode(NwsrfsDataTypeMappingReader.getTimeCode(value,
                                                                                                                                         _logger))
                                                                                .getTypeName(),
                                                           null));

                        header.appendChild(createAnElement(FewsTimeseriesConverterConstants.headerCard1[FewsTimeseriesConverterConstants.dataType],
                                                           value,
                                                           null));
                        FewsTimeseriesConverter._logger.log(Logger.DEBUG, "type = " + value);
                    }
                    else if(aString.indexOf("UNITS") >= 0)
                    {
                        header.appendChild(createAnElement(FewsTimeseriesConverterConstants.headerCard1[FewsTimeseriesConverterConstants.unit],
                                                           aString.substring(aString.indexOf('=') + 1),
                                                           null));
                        FewsTimeseriesConverter._logger.log(Logger.DEBUG,
                                                            "unit = " + aString.substring(aString.indexOf('=') + 1));
                    }
                } // end while loop
                try
                {
                    timeStep = Integer.parseInt(aLine.substring(aLine.indexOf("STEP") + 5, aLine.indexOf("VALUE"))
                                                     .trim());
                }
                catch(final StringIndexOutOfBoundsException se)
                {
                    System.err.println("aLine ... " + aLine);
                    System.err.println(se.getMessage());
                }
                catch(final NumberFormatException nfe)
                {
                    System.err.println("aLine ... " + aLine);
                    System.err.println(nfe.getMessage());
                }
                linkedHashMap.put("unit", "hour");
                linkedHashMap.put("multiplier", timeStep.toString());
                header.appendChild(createAnElement(FewsTimeseriesConverterConstants.headerCard1[FewsTimeseriesConverterConstants.timeStep],
                                                   null, //timeStep.toString(), 
                                                   linkedHashMap));
                FewsTimeseriesConverter._logger.log(Logger.DEBUG, "time step = " + timeStep.toString());

                try
                {
                    flag = Integer.parseInt(aLine.substring(aLine.lastIndexOf("STEP") + 5, aLine.indexOf("DESCRIPTION"))
                                                 .trim());
                    _valuesStep = 1;
                }
                catch(final StringIndexOutOfBoundsException se)
                {
                    System.err.println(se.getMessage());
                    _logger.log(Logger.DEBUG, se.getMessage());
                }
                catch(final NumberFormatException nfe)
                {
                    System.err.println(nfe.getMessage());
                    _logger.log(Logger.DEBUG, nfe.getMessage());
                }
                try
                {
                    int qpfIndex = aLine.indexOf("QPF");
                    if(qpfIndex < 0)
                    {
                        qpfIndex = aLine.trim().length();
                    }
                    header.appendChild(createAnElement(FewsTimeseriesConverterConstants.headerCard1[FewsTimeseriesConverterConstants.statationDescription],
                                                       aLine.substring(aLine.indexOf("DESCRIPTION") + 12, qpfIndex)
                                                            .trim(),
                                                       null));
                }
                catch(final StringIndexOutOfBoundsException se)
                {
                    System.err.println(se.getMessage());
                    _logger.log(Logger.DEBUG, se.getMessage());
                }
                catch(final IndexOutOfBoundsException ie)
                {
                    System.err.println(ie.getMessage());
                    _logger.log(Logger.DEBUG, ie.getMessage());
                }
            } // end 2nd if

            //final long timeIncrement = this.incrementTimeStep(timeStep.intValue(), flag.intValue());
            final long timeIncrement = this.incrementTimeStep(timeStep.intValue());

            //3rd if, the data
            if((aLine.indexOf('Z') >= 0 && aLine.indexOf('/') >= 0) || // a line has 'Z' and '/'
                (hasDataStarted == true && // or ((has '.' and no 'DATE=') or has NONE) and started of data
                    ((aLine.indexOf('.') >= 0 && aLine.indexOf("DATE=") < 0)) || (aLine.indexOf("NONE") >= 0)))
            {
                //System.out.println(aLine);
                hasDataStarted = true;
                final Scanner scanner = new Scanner(aLine).useDelimiter(_pattern);
                aString = "";
                while(scanner.hasNext())
                {
                    aString = scanner.next().trim();
                    if(aString == null || aString.length() <= 0)
                    {
                        continue;
                    }
                    if(aString.trim().endsWith("Z") && aString.indexOf('/') > 0 && !createdStartDate)
                    { // create the start date
                        setACalendar(calendar, aString, thisYear);
                        linkedHashMap.clear();
                        if(_startTimestamp != null && _startTimestamp.after(calendar))
                        { // no error, store the date and time
                            linkedHashMap.put("date", simpleDateFormat.format(_startTimestamp.getTime()));
                            linkedHashMap.put("time", simpleTimeFormat.format(_startTimestamp.getTime()));
                        }
                        else if(_startTimestamp != null && _startTimestamp.before(calendar))
                        { // error handle start date and time before first data date and time
                            System.err.println("The earliest available data is "
                                + simpleDateFormat.format(calendar.getTime()) + " "
                                + simpleTimeFormat.format(calendar.getTime()));
                            System.err.println("****WARNING: Invalid start timestamp for "
                                + simpleDateFormat.format(_startTimestamp.getTime()) + " "
                                + simpleTimeFormat.format(_startTimestamp.getTime()));

                            _logger.log(Logger.WARNING,
                                        "Invalid start timestamp for "
                                            + simpleDateFormat.format(_startTimestamp.getTime()) + " "
                                            + simpleTimeFormat.format(_startTimestamp.getTime()));

                            return;
                        }
                        else if(_endTimestamp != null && _endTimestamp.before(calendar))
                        { // error handle, end date and time before first data date and time
                            System.err.println("The earliest available data is "
                                + simpleDateFormat.format(calendar.getTime()) + " "
                                + simpleTimeFormat.format(calendar.getTime()));
                            System.err.println("****WARNING: Invalid end timestamp for "
                                + simpleDateFormat.format(_endTimestamp.getTime()) + " "
                                + simpleTimeFormat.format(_endTimestamp.getTime()));

                            _logger.log(Logger.WARNING,
                                        "Invalid end timestamp for " + simpleDateFormat.format(_endTimestamp.getTime())
                                            + " " + simpleTimeFormat.format(_endTimestamp.getTime()));

                            return;
                        }
                        else
                        { // no error, store the date and time
                            linkedHashMap.put("date", simpleDateFormat.format(calendar.getTime()));
                            linkedHashMap.put("time", simpleTimeFormat.format(calendar.getTime()));
                        }
                        // create startDate element
                        header.appendChild(createAnElement("startDate", null, linkedHashMap));
                        createdStartDate = true;
                        // after create the startDate tag, then go to next scann
                        continue; // end if createdStartData 
                    }
                    else if(aString.trim().endsWith("Z") && aString.indexOf('/') > 0 && createdStartDate)
                    {
                        // Check each time at the left column has date and hour in Z
                        setACalendar(cal, aString, thisYear);
                        calendarDiff = cal.compareTo(calendar);
                        if(this.FEWSDEBUG > 0)
                        {
                            System.out.print(this.simpleDateFormat.format(cal.getTime()) + " "
                                + this.simpleTimeFormat.format(cal.getTime()));

                            if(cal.after(calendar))
                            {
                                System.out.println(" compared = " + calendarDiff + " after "
                                    + this.simpleDateFormat.format(calendar.getTime()) + " "
                                    + this.simpleTimeFormat.format(calendar.getTime()));
                            }
                            else if(cal.before(calendar))
                            {
                                System.out.println(" compared = " + calendarDiff + " before "
                                    + this.simpleDateFormat.format(calendar.getTime()) + " "
                                    + this.simpleTimeFormat.format(calendar.getTime()));
                            }
                            else
                            {
                                System.out.println(" compared = " + calendarDiff + " the same "
                                    + this.simpleDateFormat.format(calendar.getTime()) + " "
                                    + this.simpleTimeFormat.format(calendar.getTime()));
                            }
                        }
                    }
                    if(aString.trim().endsWith("/"))
                    {
                        aString = aString.trim().substring(0, aString.trim().lastIndexOf('/'));
                    }
                    // sometime two data without a space between, we replace '/' with '#' in here
                    if(aString.trim().indexOf('/') >= 0 && createdStartDate)
                    {
                        aString = aString.replace('/', '#').trim(); //  
                    }
                    if(aString.trim().indexOf('/') >= 0 && !createdStartDate)
                    {
                        continue;
                    }
                    else if(aString.trim().indexOf('/') < 0)
                    { // create data
                        String[] stringArray = null;
                        if(aString.indexOf('#') >= 0 && aString.indexOf('Z') < 0)
                        {
                            stringArray = aString.split("#"); // split it into array of two
                            aString = stringArray[0];
                            /*
                             * System.out.println("split strings " + stringArray[0] + " " + stringArray[1]);
                             */
                        }

                        Double doubleString = null;
                        try
                        { // make sure it is a valid number
                          //System.out.println("A String is " + aString);
                            doubleString = Double.valueOf(aString);
                        }
                        catch(final NumberFormatException nfe)
                        {

                            if(aString.indexOf("NONE") >= 0)
                            {
                                //System.err.println(aString + " is not a double value, assigned it -999.0");
                                doubleString = Double.valueOf(-999.0);
                            }
                            else
                            {
                                continue; // skip that if it is an invalid number 
                            }
                        }
//                        System.out.println(simpleDateFormat.format(calendar.getTime()) + " "
//                            + simpleTimeFormat.format(calendar.getTime()) + " " + doubleString);
// check whether the data is overlap or not
                        if(calendarDiff < 0)
                        { // if it is overlap, then reset the cal time and compare it with calendar
                            cal.setTime(new Date(timeIncrement + cal.getTimeInMillis()));
                            calendarDiff = cal.compareTo(calendar);
                            continue; // this is an overlap data, skip it
                        }
                        addDataEvent(linkedHashMap, calendar, flag, doubleString, series, header, timeIncrement);

                        if(stringArray != null && stringArray.length > 1)
                        { // 2nd array string, some times has 2nd array string
                            aString = stringArray[1];
                            //System.out.println("2nd aString " + aString);
                            doubleString = null;
                            try
                            { // make sure it is a valid number
                                doubleString = Double.valueOf(aString);
                            }
                            catch(final NumberFormatException nfe)
                            {

                                if(aString.indexOf("pts") >= 0)
                                {
                                    //System.err.println(aString + " is not a double value, throught it away");
                                    continue; // skip that if it is an invalid number 
                                }
                                else
                                {
                                    //System.err.println(aString + " is not a double value");
                                    doubleString = Double.valueOf(999.0);
                                }
                            }
                            addDataEvent(linkedHashMap, calendar, flag, doubleString, series, header, timeIncrement);
                        } // end if for 2nd array string
                    } // end else if create data
                } // end while loop
            } // end 3rd if
        } // end for loop
        if(_startTimestamp != null && _startTimestamp.after(calendar))
        { // error handle, the start date and time is after the last data date and time
            System.err.println("The latest available data is " + simpleDateFormat.format(calendar.getTime()) + " "
                + simpleTimeFormat.format(calendar.getTime()));
            System.err.println("****WARNING: Invalid start timestamp for "
                + simpleDateFormat.format(_startTimestamp.getTime()) + " "
                + simpleTimeFormat.format(_startTimestamp.getTime()));

            _logger.log(Logger.WARNING,
                        "Invalid start timestamp for " + simpleDateFormat.format(_startTimestamp.getTime()) + " "
                            + simpleTimeFormat.format(_startTimestamp.getTime()));

        }
        if(_endTimestamp != null && _endTimestamp.after(calendar))
        { // error handle, the end date and time is after the last data date and time
            System.err.println("The latest available data is " + simpleDateFormat.format(calendar.getTime()) + " "
                + simpleTimeFormat.format(calendar.getTime()));
            System.err.println("****WARNING: Invalid end timestamp for "
                + simpleDateFormat.format(_endTimestamp.getTime()) + " "
                + simpleTimeFormat.format(_endTimestamp.getTime()));

            _logger.log(Logger.WARNING, "Invalid end timestamp for " + simpleDateFormat.format(_endTimestamp.getTime())
                + " " + simpleTimeFormat.format(_endTimestamp.getTime()));

        }

        putTimeSeriesHeaderInOrder(header);
        if(flag > 1)
        {
            reorder(flag);
        }
    } // end of method

    /**
     * Reorder the series element in XML
     * 
     * @param flag -- number of values step
     */
    protected void reorder(final int flag)
    {
        //System.out.println("values step = " + flag);

        Element series = null;
        final Element timeSeries = _timeSeries;

        try
        {
            series = (Element)timeSeries.getLastChild();
        }
        catch(final Exception e)
        {
            System.err.println("Can't get series elements to reorder");
            return;
        }
        //System.out.println("start reodering");
        final Element serieses[] = new Element[flag];
        for(int i = 0; i < serieses.length; i++)
        {
            serieses[i] = (Element)series.cloneNode(true);
            // get the header
            final Element header = (Element)serieses[i].getFirstChild();
            final NodeList nodeList = header.getElementsByTagName("parameterId");
            Element parameterId = null;
            for(int j = 0; j < nodeList.getLength(); j++)
            {
                parameterId = (Element)nodeList.item(j);
            }
            final String textContent = parameterId.getTextContent() + Integer.valueOf(i + 1).toString();
            // reset the parameter Id
            parameterId.setTextContent(textContent);

            NodeList events = serieses[i].getElementsByTagName("event");
            final ArrayList<Element> arrayList = new ArrayList<Element>();
            int eventLength = events.getLength();
            for(int k = 0; k < eventLength; k++)
            {
                final Element event = (Element)events.item(k);
                final int flagInt = Integer.parseInt(event.getAttribute("flag"));
                //System.out.println("i = " + i + " k = " + k + " flag int = " + flagInt);

                if(flagInt == i) // save it for later use
                {
                    arrayList.add((Element)event.cloneNode(true));
                    //	System.out.println("added i = " + i + " k = " + k + " flag int = " + flagInt);
                }
            }
            events = serieses[i].getElementsByTagName("event");
            eventLength = events.getLength();
            // remove the old events
            for(int l = 0; l < eventLength; l++)
            {
                final Element event = (Element)events.item(0);
                serieses[i].removeChild(event);
            }
            // the the new events
            for(int m = 0; m < arrayList.size(); m++)
            {
                final Element event = arrayList.get(m);
                event.setAttribute("flag", "0");
                serieses[i].appendChild(event);
            }
            // append the new serieses
            timeSeries.appendChild(serieses[i]);
        }
        // remove the old series
        timeSeries.removeChild(series);
    }

    /**
     * Set a string format mm/dd/yy/hhZ into a calendar
     * 
     * @param calendar - set to this calendar
     * @param aString - set this string format
     * @param thisYear - an integer for year
     */
    protected void setACalendar(final Calendar calendar, String aString, final int thisYear)
    {
        //System.out.println("aString = " + aString);
        aString = aString.trim().substring(0, aString.trim().lastIndexOf('Z'));
        final StringTokenizer token = new StringTokenizer(aString, "/");
        final Integer month = Integer.parseInt((String)token.nextElement());
        final Integer day = Integer.parseInt((String)token.nextElement());
        final Integer year = Integer.parseInt((String)token.nextElement());
        final Integer hour = Integer.parseInt((String)token.nextElement());

        int theYear = year.intValue() + 2000;
        if(theYear > thisYear)
        {
            theYear -= 100;
        }
        calendar.set(Calendar.YEAR, theYear);
        calendar.set(Calendar.MONTH, month.intValue() - 1);
        calendar.set(Calendar.DAY_OF_MONTH, day.intValue());
        calendar.set(Calendar.HOUR_OF_DAY, hour.intValue());
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
    }

    /**
     * check which one of four cases
     * 
     * @return which one
     */
    private int checkCases()
    {
        int fourCases = 0;
        if(_startTimestamp != null && _endTimestamp == null)
        {
            fourCases = 1;
        }
        else if(_startTimestamp == null && _endTimestamp != null)
        {
            fourCases = 2;
        }
        else if(_startTimestamp != null && _endTimestamp != null)
        {
            fourCases = 3;
        }
        else
        {
            fourCases = 0;
        }
        return fourCases;
    }

    /**
     * Create a event tag for the XML document
     * 
     * @param linkedHashMap -- linked hash map for the event attributes
     * @param calendar -- timestamp for the event
     * @param flag -- flag attribute in the event
     * @param doubleString -- the data value for the event
     * @param series -- An element for the events
     * @param header -- header element for the endDate tag
     * @param timeIncrement -- a constant for the time increment in each event
     */
    private void addDataEvent(final LinkedHashMap<String, String> linkedHashMap,
                              final Calendar calendar,
                              final Integer flag,
                              final Double doubleString,
                              final Element series,
                              final Element header,
                              final long timeIncrement)
    {
        //System.out.println(simpleDateFormat.format(calendar.getTime()) +
        //		" " + simpleTimeFormat.format(calendar.getTime()));
        linkedHashMap.clear();
        linkedHashMap.put("date", simpleDateFormat.format(calendar.getTime()));
        // for now we force the flag to zero until we find out what is that mean
        //linkedHashMap.put("flag", "0");
        linkedHashMap.put("flag", Integer.valueOf(_valuesStep - 1).toString());
        linkedHashMap.put("time", simpleTimeFormat.format(calendar.getTime()));
        linkedHashMap.put("value", _decimalFormat.format(doubleString.doubleValue()));

        final int fourCases = checkCases();
        boolean willAppend = false;
        switch(fourCases)
        {
            case 0:
            default:
                willAppend = true;
                break;
            case 1:
                if(calendar.getTime().getTime() >= _startTimestamp.getTime().getTime())
                {
                    willAppend = true;
                }
                break;
            case 2:
                if(calendar.getTime().getTime() <= _endTimestamp.getTime().getTime())
                {
                    willAppend = true;
                }
                break;
            case 3:
                if(calendar.getTime().getTime() >= _startTimestamp.getTime().getTime()
                    && calendar.getTime().getTime() <= _endTimestamp.getTime().getTime())
                {
                    willAppend = true;
                }
                break;
        } // end switch
        if(willAppend)
        {
            series.appendChild(createAnElement("event", null, linkedHashMap));
            // create or update the endDate tag
            linkedHashMap.clear();
            linkedHashMap.put("date", simpleDateFormat.format(calendar.getTime()));
            linkedHashMap.put("time", simpleTimeFormat.format(calendar.getTime()));
            final NodeList nodeList = header.getElementsByTagName("endDate");
            if(nodeList.getLength() > 0)
            { // replace the endDate if it is existed
                final Element newElement = createAnElement("endDate", null, linkedHashMap);
                final Element oldElement = (Element)nodeList.item(0);
                header.replaceChild(newElement, oldElement);
            }
            else
            {
                // create the endDate if it doesn't existed
                header.appendChild(createAnElement("endDate", null, linkedHashMap));
            }
        } // end if
        if(_valuesStep == flag.intValue())
        {
            calendar.setTime(new Date(timeIncrement + calendar.getTimeInMillis()));
            _valuesStep = 1;
        }
        else
        {
            _valuesStep++;
        }
    } // end method

    public void readGeneralTextFile(final String fileName)
    {
        clearTheLinesVector();
        readFile(fileName);

        final Calendar calendar = Calendar.getInstance(OHDConstants.GMT_TIMEZONE);
        final Calendar startDate = Calendar.getInstance(OHDConstants.GMT_TIMEZONE);
        //Calendar endDate = Calendar.getInstance(OHDConstants.GMT_TIMEZONE);

        Integer timeStep = Integer.valueOf(0);
        Element series = null;
        Element header = null;
        LinkedHashMap<String, String> attributes = null;

        series = _document.createElement("series");
        _timeSeries.appendChild(series);

        header = _document.createElement("header");

        addHeaderElements(header);

        header.appendChild(createAnElement("creationDate", simpleDateFormat.format(calendar.getTime()), null));
        header.appendChild(createAnElement("creationTime", simpleTimeFormat.format(calendar.getTime()), null));
        //final int thisYear = calendar.get(Calendar.YEAR);

        series.appendChild(header);
        final NodeList nList = header.getElementsByTagName("missVal");
        if(nList == null || nList.getLength() <= 0)
        {
            header.appendChild(createAnElement("missVal", "-999.0", null));
        }

        String aLine = "";
        if(this._inputLinesVec.size() > 4)
        {
            // get first line
            aLine = this._inputLinesVec.get(1);
            StringTokenizer token = new StringTokenizer(aLine);
            int tokenCount = 0;
            String tmpStr = "";
            while(token.hasMoreElements())
            {
                tmpStr = (String)token.nextElement();
                //System.out.println(tokenCount + " = " + tmpStr + " ");
                switch(tokenCount)
                {
                    default:
                        header.appendChild(createAnElement("type", "instantaneous", null));
                        break;
                    case 0:
                        header.appendChild(createAnElement("stationName", tmpStr, null));
                        break;
                    case 1:
                        header.appendChild(createAnElement("parameterId", tmpStr, null));
                        break;
                    case 3:
                        header.appendChild(createAnElement("units", tmpStr, null));
                        break;
                    case 4:
                        attributes = new LinkedHashMap<String, String>();
                        timeStep = string2Integer(tmpStr);
                        attributes.put("multiplier", timeStep.toString());
                        attributes.put("unit", "hour");
                        header.appendChild(createAnElement("timeStep", null, attributes));
                        break;
                    case 5:
                        header.appendChild(createAnElement("locationId", tmpStr, null));
                        break;
                    case 6:
                        header.appendChild(createAnElement("longName", tmpStr, null));
                        break;
                } // end switch
                tokenCount++;
            } // end 1st while loop
              // get second line
            aLine = this._inputLinesVec.get(2);
            tokenCount = 0;
            token = new StringTokenizer(aLine);

            Integer tmpInteger = Integer.valueOf(0);
            while(token.hasMoreElements())
            {
                tmpStr = (String)token.nextElement();
                switch(tokenCount)
                {
                    case 0:
                        tmpInteger = string2Integer(tmpStr);
                        startDate.set(Calendar.MONTH, tmpInteger.intValue() - 1);
                        break;
                    case 1:
                        tmpInteger = string2Integer(tmpStr);
                        startDate.set(Calendar.YEAR, tmpInteger.intValue());
                        break;
//                    case 2:
//                        tmpInteger = string2Integer(tmpStr);
//                        endDate.set(Calendar.MONTH, tmpInteger.intValue() - 1);
//                        break;
//                    case 3:
//                        tmpInteger = string2Integer(tmpStr);
//                        endDate.set(Calendar.YEAR, tmpInteger.intValue());
//                        break;
                } // end switch
                tokenCount++;
            } // end 2nd while loop
            attributes = new LinkedHashMap<String, String>();
            startDate.set(Calendar.DAY_OF_MONTH, 1);
            startDate.set(Calendar.HOUR_OF_DAY, 0);
            startDate.set(Calendar.MINUTE, 0);
            startDate.set(Calendar.SECOND, 0);
            startDate.set(Calendar.MILLISECOND, 0);
            attributes.put("date", this.simpleDateFormat.format(startDate.getTime()));
            attributes.put("time", this.simpleTimeFormat.format(startDate.getTime()));
            header.appendChild(createAnElement("startDate", null, attributes));

            //attributes = new LinkedHashMap<String, String>();
            //endDate.set(Calendar.DAY_OF_MONTH, 1);
//            endDate.set(Calendar.HOUR_OF_DAY, 12);
//            endDate.set(Calendar.MINUTE, 0);
//            endDate.set(Calendar.SECOND, 0);
//            endDate.set(Calendar.MILLISECOND, 0);
//            attributes.put("date", this.simpleDateFormat.format(endDate.getTime()));
//            attributes.put("time", this.simpleTimeFormat.format(endDate.getTime()));
//            header.appendChild(createAnElement("endDate", null, attributes));

            // read rest of data
            final int lineSize = this._inputLinesVec.size();
            final long incrementTime = this.incrementTimeStep(timeStep);
            for(int i = 3; i < lineSize; i++)
            {
                aLine = this._inputLinesVec.get(i);
                tokenCount = 0;
                token = new StringTokenizer(aLine);

                attributes = new LinkedHashMap<String, String>();
                attributes.put("flag", "0");
                attributes.put("date", simpleDateFormat.format(startDate.getTime()));
                attributes.put("time", simpleTimeFormat.format(startDate.getTime()));
                while(token.hasMoreElements())
                {
                    tmpStr = (String)token.nextElement();

                    switch(tokenCount)
                    {
                        case 3:
                            try
                            {
                                final Double doubleValue = Double.valueOf(tmpStr);
                                attributes.put("value", doubleValue.toString());
                            }
                            catch(final NumberFormatException e)
                            {
                                attributes.put("value", "-999.000");
                            }
                            series.appendChild(createAnElement("event", null, attributes));
                            break;
                    } // end switch
                    tokenCount++;
                } // end 3rd while loop               
                startDate.setTime(new Date(incrementTime + startDate.getTimeInMillis()));
            } // end for loop
            startDate.setTime(new Date(startDate.getTimeInMillis() - incrementTime));
            attributes = new LinkedHashMap<String, String>();
            attributes.put("date", this.simpleDateFormat.format(startDate.getTime()));
            attributes.put("time", this.simpleTimeFormat.format(startDate.getTime()));
            header.appendChild(createAnElement("endDate", null, attributes));
        } // end big if

        putTimeSeriesHeaderInOrder(header);
    } // end method

    private Integer string2Integer(final String stringInt)
    {
        Integer integer = Integer.valueOf(0);
        try
        {
            integer = Integer.valueOf(stringInt);
        }
        catch(final NumberFormatException e)
        {
            ;
        }
        return integer;
    }

    /**
     * Read an input file and store the lines into a vector
     * 
     * @param fileName -- file name
     */
    public void readFile(final String fileName)
    {
        try
        {
            final RandomAccessFile randomAccessFile = new RandomAccessFile(new File(fileName), "r");
            String aLine = "";
            while((aLine = randomAccessFile.readLine()) != null)
            {
                _inputLinesVec.add(aLine);
            }
            randomAccessFile.close();
        }
        catch(final FileNotFoundException e)
        {
            _logger.log(Logger.ERROR, e.getMessage());
            e.printStackTrace();
        }
        catch(final IOException ioe)
        {
            _logger.log(Logger.ERROR, ioe.getMessage());
            ioe.printStackTrace();
        }
    }

    /**
     * Read input card text file and store data into XML
     * 
     * @param fileName -- input file
     * @return success if true; otherwise false
     */
    public void readCardFile(final String fileName) throws Exception
    {
        clearTheLinesVector();
        //super.readLines(fileName);
        readFile(fileName);
        //boolean success = true;

        final Calendar startCalendar = Calendar.getInstance(OHDConstants.GMT_TIMEZONE);
        final Calendar endCalendar = Calendar.getInstance(OHDConstants.GMT_TIMEZONE);
        final Calendar eventCalendar = Calendar.getInstance(OHDConstants.GMT_TIMEZONE);

        String aString = "";
        String aLine = "";
        String scanString = "";
        String tokenString = "";
        String dataFormat = "";

        int dataLine = 0;
        for(dataLine = 0; dataLine < _inputLinesVec.size(); dataLine++)
        {
            aLine = _inputLinesVec.get(dataLine).trim();
            if(!aLine.startsWith("$"))
            {
                break;
            }
            if(aLine.indexOf("OUTPUT FORMAT") > 0)
            {
                dataFormat = aLine.substring(aLine.indexOf('(') + 1, aLine.indexOf(')'));
                //System.out.println("OUTPUT FORMAT = " + dataFormat);
            }
        }
        //this.FEWSDEBUG = 2;

        final String lineSegment_1 = aLine.substring(0, FewsTimeseriesConverterConstants.headerCard1Index[1]);
        final String lineSegment_2 = aLine.substring(FewsTimeseriesConverterConstants.headerCard1Index[1]);
        String lineSegment_3 = "";
        if(aLine.length() >= FewsTimeseriesConverterConstants.headerCard1Index[5])
        {
            lineSegment_3 = aLine.substring(FewsTimeseriesConverterConstants.headerCard1Index[5]);
        }
        if(this.FEWSDEBUG > 1)
        {
            System.out.println("aLine = " + aLine);
            System.out.println("aLine_1 = " + lineSegment_1);
            System.out.println("aLine_2 = " + lineSegment_2);
            System.out.println("aLine_3 = " + lineSegment_3);
        }
        Scanner scanner = new Scanner(lineSegment_2).useDelimiter(_pattern);
        final StringTokenizer token = new StringTokenizer(lineSegment_2);

        int count = 0;
        Element element = null;
        Element series = null;
        Element header = null;

        series = _document.createElement("series");
        _timeSeries.appendChild(series);

        header = _document.createElement("header");

        addHeaderElements(header);
        header.appendChild(createAnElement("creationDate", simpleDateFormat.format(startCalendar.getTime()), null));
        header.appendChild(createAnElement("creationTime", simpleTimeFormat.format(startCalendar.getTime()), null));

        series.appendChild(header);
        Integer timeStep = Integer.valueOf(0);

        // for the headers 
        for(int i = 0; i < FewsTimeseriesConverterConstants.headerCard1.length; i++)
        {
            element = null;
            aString = "";
            try
            {
                //if (scanner.hasNext())
                switch(i)
                {
                    case FewsTimeseriesConverterConstants.fileName:
                        element = _document.createElement(FewsTimeseriesConverterConstants.headerCard1[FewsTimeseriesConverterConstants.fileName]);
                        aString = aLine.substring(0, FewsTimeseriesConverterConstants.headerCard1Index[1]).trim();
                        //scanString = scanner.next().trim();
                        element.setTextContent(aString);
                        if(this.FEWSDEBUG > 1)
                        {
                            System.out.println(FewsTimeseriesConverterConstants.headerCard1[FewsTimeseriesConverterConstants.fileName]
                                + ": " + aString + "...");
                        }
                        break;
                    case FewsTimeseriesConverterConstants.dataType:

                        aString = aLine.substring(FewsTimeseriesConverterConstants.headerCard1Index[1],
                                                  FewsTimeseriesConverterConstants.headerCard1Index[1] + 3).trim();

                        if(token.hasMoreElements())
                        {
                            tokenString = ((String)token.nextElement()).trim();
                        }
                        // FCamacho - set the type element based on the parameterId value.
                        element = _document.createElement("type");
                        element.setTextContent(FewsRegularTimeSeries.getFewsTimeCode(NwsrfsDataTypeMappingReader.getTimeCode(tokenString,
                                                                                                                             _logger))
                                                                    .getTypeName());
                        header.appendChild(element);

                        element = _document.createElement(FewsTimeseriesConverterConstants.headerCard1[FewsTimeseriesConverterConstants.dataType]);
                        element.setTextContent(tokenString);
                        if(this.FEWSDEBUG > 1)
                        {
                            System.out.println(FewsTimeseriesConverterConstants.headerCard1[FewsTimeseriesConverterConstants.dataType]
                                + ": " + aString + "..." + "...." + tokenString);
                        }
                        FewsTimeseriesConverter._logger.log(2,
                                                            FewsTimeseriesConverterConstants.headerCard1[FewsTimeseriesConverterConstants.dataType]
                                                                + ": " + aString + "..." + "...." + tokenString);
                        break;
                    case FewsTimeseriesConverterConstants.dimension:
                        // no dimension defined in XML file
                        token.nextElement();
                        break;
                    case FewsTimeseriesConverterConstants.unit:
                        element = _document.createElement(FewsTimeseriesConverterConstants.headerCard1[FewsTimeseriesConverterConstants.unit]);
                        aString = aLine.substring(FewsTimeseriesConverterConstants.headerCard1Index[3],
                                                  FewsTimeseriesConverterConstants.headerCard1Index[3] + 3).trim();
                        while((tokenString = (String)token.nextElement()).trim() == null)
                        {
                            tokenString = ((String)token.nextElement()).trim();
                        }
                        element.setTextContent(tokenString);
                        if(this.FEWSDEBUG > 1)
                        {
                            System.out.println(FewsTimeseriesConverterConstants.headerCard1[FewsTimeseriesConverterConstants.unit]
                                + ": " + aString + "..." + "...." + tokenString);
                        }
                        FewsTimeseriesConverter._logger.log(2,
                                                            FewsTimeseriesConverterConstants.headerCard1[FewsTimeseriesConverterConstants.unit]
                                                                + ": " + aString + "..." + "...." + tokenString);
                        break;
                    case FewsTimeseriesConverterConstants.timeStep:
                        element = _document.createElement(FewsTimeseriesConverterConstants.headerCard1[FewsTimeseriesConverterConstants.timeStep]);
                        int endIndex = FewsTimeseriesConverterConstants.headerCard1Index[4] + 1;
                        if(aLine.length() > endIndex)
                        {
                            endIndex += 1;
                        }
                        aString = aLine.substring(FewsTimeseriesConverterConstants.headerCard1Index[4], endIndex)
                                       .trim();
                        try
                        {

                            while((tokenString = (String)token.nextElement()).trim() == null)
                            {
                                tokenString = ((String)token.nextElement()).trim();
                                //System.out.println("token String = " + tokenString);
                            }
                        }
                        catch(final Exception exp)
                        {
                            tokenString = "6";
                            FewsTimeseriesConverter._logger.log(2, "token String exception, forced to 6");
                        }
                        if(this.FEWSDEBUG > 1)
                        {
                            System.out.println("aLine = " + aLine
                                + " ==== FewsTimeseriesConverterConstants.headerCard1Index[4] == "
                                + FewsTimeseriesConverterConstants.headerCard1Index[4]);
                            System.out.println("time step = " + aString);
                        }
                        aString = "6";
                        //System.out.println("token String = " + tokenString);
                        try
                        {
                            timeStep = Integer.valueOf(tokenString);
//                            if(timeStep.intValue() != 6)
//                            {
//                                timeStep = Integer.valueOf(6);
//                            }
                        }
                        catch(final StringIndexOutOfBoundsException siobe)
                        {
                            timeStep = Integer.valueOf(6);
                            _logger.log(Logger.WARNING, "Invalid time step " + siobe.getMessage());
                            System.err.println(siobe.getMessage() + " Invalid time step ");
                        }
                        catch(final NumberFormatException nfe)
                        {
                            timeStep = Integer.valueOf(6);
                            _logger.log(Logger.WARNING, "Invalid time step " + nfe.getMessage());
                            System.err.println(nfe.getMessage());
                        } // end catch
                        element.setAttribute("unit", "hour");
                        //System.out.println("multiplier = " + timeStep.toString());
                        element.setAttribute("multiplier", timeStep.toString());
                        element.setTextContent(null);
                        if(this.FEWSDEBUG > 1)
                        {
                            System.out.println(FewsTimeseriesConverterConstants.headerCard1[FewsTimeseriesConverterConstants.timeStep]
                                + ": " + aString + "......." + timeStep.toString());
                        }
                        FewsTimeseriesConverter._logger.log(Logger.DEBUG,
                                                            FewsTimeseriesConverterConstants.headerCard1[FewsTimeseriesConverterConstants.timeStep]
                                                                + ": " + aString + "......." + timeStep.toString());
                        break;
                    case FewsTimeseriesConverterConstants.statationID:
                        // there is no locationId in data card file, we defined it from other place
                        break;
                    case FewsTimeseriesConverterConstants.statationDescription:
                        element = _document.createElement(FewsTimeseriesConverterConstants.headerCard1[FewsTimeseriesConverterConstants.statationDescription]);
                        aString = aLine.substring(FewsTimeseriesConverterConstants.headerCard1Index[5],
                                                  aLine.length() - 1).trim();
                        scanString = scanner.next().trim();
                        element.setTextContent(aString);
                        if(this.FEWSDEBUG > 1)
                        {
                            System.out.println(FewsTimeseriesConverterConstants.headerCard1[FewsTimeseriesConverterConstants.statationDescription]
                                + ": " + aString + "..." + scanString);
                        }
                        FewsTimeseriesConverter._logger.log(Logger.DEBUG,
                                                            FewsTimeseriesConverterConstants.headerCard1[FewsTimeseriesConverterConstants.statationDescription]
                                                                + ": " + aString + "..." + scanString);
                        break;
                } // end switch  
            } // end try
            catch(final StringIndexOutOfBoundsException e)
            {
                aString = " ";
                element.setTextContent(aString);
                System.err.println(e.getMessage());
                FewsTimeseriesConverter._logger.log(Logger.WARNING, e.getMessage());
            }
            if(element != null)
            {
                header.appendChild(element);
            }
        } // end for loop
        final NodeList aNodeList = header.getElementsByTagName("locationId");
        if(aNodeList.getLength() <= 0)
        { // locationId tag isn't existed
            element = _document.createElement("locationId");
            element.setTextContent("none"); // create a dummy locationId
            header.appendChild(element);
        }
        count = 0;
        // for the start/end dates
        aLine = _inputLinesVec.get(++dataLine).trim();
        if(this.FEWSDEBUG > 1)
        {
            System.out.println("start/end dates = " + aLine);
        }
        scanner = new Scanner(aLine).useDelimiter(_pattern);

        Integer flag = null;
        while(scanner.hasNext())
        {
            aString = scanner.next().trim();
            if(aString != null && aString.length() > 0)
            {
                switch(count)
                {
                    case 0:
                        startCalendar.set(Calendar.MONTH, Integer.parseInt(aString) - 1);
                        if(this.FEWSDEBUG > 1)
                        {
                            System.out.println("start month " + aString);
                        }
                        break;
                    case 1:
                        startCalendar.set(Calendar.YEAR, Integer.parseInt(aString));
                        if(this.FEWSDEBUG > 1)
                        {
                            System.out.println("start year " + aString);
                        }
                        break;
                    case 2:
                        endCalendar.set(Calendar.MONTH, Integer.parseInt(aString) - 1);
                        eventCalendar.set(Calendar.MONTH, Integer.parseInt(aString) - 1);
                        if(this.FEWSDEBUG > 1)
                        {
                            System.out.println("end month " + aString);
                        }
                        break;
                    case 3:
                        endCalendar.set(Calendar.YEAR, Integer.parseInt(aString));
                        eventCalendar.set(Calendar.YEAR, Integer.parseInt(aString));
                        if(this.FEWSDEBUG > 1)
                        {
                            System.out.println("end year " + aString);
                        }
                        break;
                    case 4: // will use this flag later
                        flag = Integer.parseInt(aString);
                        break;
                    case 5: // for now, we don't use string format, yet
                        //format = aString;
                        break;
                }
                count++;
            }
        }
        startCalendar.set(Calendar.DAY_OF_MONTH, 1);
        startCalendar.set(Calendar.HOUR_OF_DAY, timeStep.intValue());
        startCalendar.set(Calendar.MINUTE, 0);
        startCalendar.set(Calendar.SECOND, 0);
        startCalendar.set(Calendar.MILLISECOND, 0);
        // make end calendar the same as start calendar first, it'll change later
        endCalendar.set(Calendar.YEAR, startCalendar.get(Calendar.YEAR));
        endCalendar.set(Calendar.MONTH, startCalendar.get(Calendar.MONTH));
        endCalendar.set(Calendar.DAY_OF_MONTH, 1);
        endCalendar.set(Calendar.HOUR_OF_DAY, timeStep.intValue());
        endCalendar.set(Calendar.MINUTE, 0);
        endCalendar.set(Calendar.SECOND, 0);
        endCalendar.set(Calendar.MILLISECOND, 0);
        // count it when an event occur
        eventCalendar.set(Calendar.YEAR, startCalendar.get(Calendar.YEAR));
        eventCalendar.set(Calendar.MONTH, startCalendar.get(Calendar.MONTH));
        eventCalendar.set(Calendar.DAY_OF_MONTH, 1);
        eventCalendar.set(Calendar.HOUR_OF_DAY, timeStep.intValue());
        eventCalendar.set(Calendar.MINUTE, 0);
        eventCalendar.set(Calendar.SECOND, 0);
        eventCalendar.set(Calendar.MILLISECOND, 0);

        //this.FEWSDEBUG = -2;

        if(FEWSDEBUG > -1)
        {
            System.out.println("Start " + simpleDateFormat.format(startCalendar.getTime()) + " "
                + simpleTimeFormat.format(startCalendar.getTime()));
            System.out.println("End " + simpleDateFormat.format(endCalendar.getTime()) + " "
                + simpleTimeFormat.format(endCalendar.getTime()));
            System.out.println("Event " + simpleDateFormat.format(eventCalendar.getTime()) + " "
                + simpleTimeFormat.format(eventCalendar.getTime()));
        }
        FewsTimeseriesConverter._logger.log(Logger.DEBUG, "Start " + simpleDateFormat.format(startCalendar.getTime())
            + " " + simpleTimeFormat.format(startCalendar.getTime()));
        FewsTimeseriesConverter._logger.log(Logger.DEBUG, "End " + simpleDateFormat.format(endCalendar.getTime()) + " "
            + simpleTimeFormat.format(endCalendar.getTime()));

        final LinkedHashMap<String, String> linkedHashMap = new LinkedHashMap<String, String>();
        if(getStartTimestamp() != null && getStartTimestamp().after(startCalendar))
        {
            linkedHashMap.put("date", simpleDateFormat.format(getStartTimestamp().getTime()));
            linkedHashMap.put("time", simpleTimeFormat.format(getStartTimestamp().getTime()));
        }
        else if(getStartTimestamp() != null && getStartTimestamp().before(startCalendar))
        {
            System.err.println("The earliest available data is " + simpleDateFormat.format(startCalendar.getTime())
                + " " + simpleTimeFormat.format(startCalendar.getTime()));
            System.err.println("****WARNING: Invalid start timestamp for "
                + simpleDateFormat.format(getStartTimestamp().getTime()) + " "
                + simpleTimeFormat.format(getStartTimestamp().getTime()));

            _logger.log(Logger.WARNING,
                        "Invalid start timestamp for " + simpleDateFormat.format(getStartTimestamp().getTime()) + " "
                            + simpleTimeFormat.format(getStartTimestamp().getTime()));

            return;
        }
        else if(getEndTimestamp() != null && getEndTimestamp().before(startCalendar))
        {
            System.err.println("The earliest available data is " + simpleDateFormat.format(startCalendar.getTime())
                + " " + simpleTimeFormat.format(startCalendar.getTime()));
            System.err.println("****WARNING: Invalid end timestamp for "
                + simpleDateFormat.format(getEndTimestamp().getTime()) + " "
                + simpleTimeFormat.format(getEndTimestamp().getTime()));

            _logger.log(Logger.WARNING,
                        "Invalid end timestamp for " + simpleDateFormat.format(getEndTimestamp().getTime()) + " "
                            + simpleTimeFormat.format(getEndTimestamp().getTime()));

            return;
        }
        else
        { // no error, 
            linkedHashMap.put("date", simpleDateFormat.format(startCalendar.getTime()));
            linkedHashMap.put("time", simpleTimeFormat.format(startCalendar.getTime()));
        }
        header.appendChild(createAnElement("startDate", null, linkedHashMap));

        //System.out.println("timeStep = " + timeStep.intValue() + " flag = " + flag.intValue());
        //final long timeIncrement = incrementTimeStep(timeStep.intValue(), 1);
        final long timeIncrement = incrementTimeStep(timeStep.intValue());

        boolean isFirstEvent = true;
        boolean isFirstData = true;
        final ArrayList<String> arrayList = new ArrayList<String>();
        // ready to get data event       
        //System.out.println("OUTPUT FORMAT = " + dataFormat);
        for(++dataLine; dataLine < _inputLinesVec.size(); dataLine++)
        {
            aLine = _inputLinesVec.get(dataLine);
            //System.out.println(dataLine + " -- " + aLine);
            if(dataFormat != null && !dataFormat.equals(""))
            {
                aLine = formatLine(dataFormat, aLine);
                //System.out.println(aLine);
            }
            else
            {
                aLine = aLine.trim();
                //System.out.println(aLine);
            }
            //System.out.println(aLine);
            scanner = new Scanner(aLine).useDelimiter(_pattern);
            count = 0;
            arrayList.clear();

            while(scanner.hasNext())
            {
                aString = scanner.next().trim();
                if(aString != null && aString.length() > 0)
                {
                    arrayList.add(aString);
                }
            } // end inner while loop

            /*
             * StringTokenizer dataToken = new StringTokenizer(aLine); while(dataToken.hasMoreElements()) { aString =
             * (String)dataToken.nextElement(); if(aString != null && aString.length() > 0) {
             * arrayList.add(aString.trim()); } }
             */
            final int arrayListSize = arrayList.size();
            int index = 0;
            //System.out.println("array list size = " + arrayListSize + " flag = " + flag.intValue());
            if(arrayListSize > flag.intValue())
            {
                index = arrayListSize - flag.intValue();
            }
            //System.out.println("index starts at " + index);
            for(; index < arrayListSize; index++)
            {
                aString = arrayList.get(index);
                if(aString.indexOf('.') < 0 && aString.indexOf('*') < 0)
                {
                    continue; // no decimal
                }
                Double doubleString = null;
                try
                { // make sure that string is a valid double number
                    doubleString = Double.valueOf(aString);
                    //System.out.println("double string = " + doubleString);
                }
                catch(final NumberFormatException nfe)
                {
                    System.err.println(fileName + " error at line " + dataLine + " with invalid data " + aString);
                    FewsTimeseriesConverter._logger.log(Logger.WARNING, fileName + " error at line " + dataLine
                        + " with invalid data " + aString);
                    // _logger.log(fileName + " error at line " + dataLine + " with invalid data " + aString);
                    doubleString = Double.valueOf(-999.0);
                }
                linkedHashMap.clear();
                linkedHashMap.put("date", simpleDateFormat.format(endCalendar.getTime()));
                //linkedHashMap.put("flag", flag.toString());
                // for now we force the flag to zero until we find out what is that mean
                linkedHashMap.put("flag", "0");
                linkedHashMap.put("time", simpleTimeFormat.format(endCalendar.getTime()));
                linkedHashMap.put("value", _decimalFormat.format(doubleString.doubleValue()));

                /*
                 * System.out.println("date = " + simpleDateFormat.format(endCalendar.getTime()) + " " +
                 * simpleTimeFormat.format(endCalendar.getTime()) + " " +
                 * _decimalFormat.format(doubleString.doubleValue()));
                 */

                final int fourCases = checkCases();
                //int fourCases = checkCases();
                boolean willAppend = false;

                switch(fourCases)
                {
                    default:
                    case 0:
                        willAppend = true;
                        break;
                    case 1:
                        if(endCalendar.getTime().getTime() >= _startTimestamp.getTime().getTime())
                        {
                            willAppend = true;
                        }
                        break;
                    case 2:
                        if(endCalendar.getTime().getTime() <= _endTimestamp.getTime().getTime())
                        {
                            willAppend = true;
                        }
                        break;
                    case 3:
                        if(endCalendar.getTime().getTime() >= _startTimestamp.getTime().getTime()
                            && endCalendar.getTime().getTime() <= _endTimestamp.getTime().getTime())
                        {
                            willAppend = true;
                        }
                        break;
                } // end inner switch 
                  //System.out.print("willAppend = " + willAppend + " ");
                  //willAppend = true;
                if(willAppend)
                {
                    series.appendChild(createAnElement("event", null, linkedHashMap));
                    if(isFirstEvent)
                    {
                        eventCalendar.setTime(new Date(endCalendar.getTimeInMillis()));
                        isFirstEvent = false;
                        if(FEWSDEBUG > 1)
                        {
                            System.out.println("first event date " + simpleDateFormat.format(eventCalendar.getTime())
                                + " " + simpleTimeFormat.format(eventCalendar.getTime()));
                        }
                    }
                    // increment the event calendar 
                    eventCalendar.setTime(new Date(timeIncrement + endCalendar.getTimeInMillis()));
                    if(FEWSDEBUG == 6)
                    {
                        System.out.println("event date " + simpleDateFormat.format(eventCalendar.getTime()) + " "
                            + simpleTimeFormat.format(eventCalendar.getTime()));
                    }
                } // end if willAppend
                if(isFirstData)
                {// 1st data
                    endCalendar.setTime(new Date(endCalendar.getTimeInMillis()));
                    isFirstData = false;
                }
                // increment the end calendar 
                endCalendar.setTime(new Date(timeIncrement + endCalendar.getTimeInMillis()));
                if(FEWSDEBUG > 1)
                {
                    System.out.println("end date " + simpleDateFormat.format(endCalendar.getTime()) + " "
                        + simpleTimeFormat.format(endCalendar.getTime()));
                }
            } // end inner for loop
        } // end outer for loop

        if(getStartTimestamp() != null && getStartTimestamp().after(endCalendar))
        {
            System.err.println("The latest available data is " + simpleDateFormat.format(endCalendar.getTime()) + " "
                + simpleTimeFormat.format(endCalendar.getTime()));
            System.err.println("****WARNING: Invalid start timestamp for "
                + simpleDateFormat.format(getStartTimestamp().getTime()) + " "
                + simpleTimeFormat.format(getStartTimestamp().getTime()));
            _logger.log(Logger.WARNING,
                        "Invalid start timestamp for " + simpleDateFormat.format(getStartTimestamp().getTime()) + " "
                            + simpleTimeFormat.format(getStartTimestamp().getTime()));
        }
        if(getEndTimestamp() != null && getEndTimestamp().after(endCalendar))
        {
            System.err.println("The latest available data is " + simpleDateFormat.format(endCalendar.getTime()) + " "
                + simpleTimeFormat.format(endCalendar.getTime()));
            System.err.println("****WARNING: Invalid end timestamp for "
                + simpleDateFormat.format(getEndTimestamp().getTime()) + " "
                + simpleTimeFormat.format(getEndTimestamp().getTime()));
            _logger.log(Logger.WARNING,
                        "Invalid end timestamp for " + simpleDateFormat.format(getEndTimestamp().getTime()) + " "
                            + simpleTimeFormat.format(getEndTimestamp().getTime()));

        }
        linkedHashMap.clear();
        eventCalendar.setTimeInMillis(eventCalendar.getTimeInMillis() - timeIncrement);
        linkedHashMap.put("date", simpleDateFormat.format(eventCalendar.getTime()));
        linkedHashMap.put("time", simpleTimeFormat.format(eventCalendar.getTime()));

        header.appendChild(createAnElement("endDate", null, linkedHashMap));
        final NodeList nList = header.getElementsByTagName("missVal");
        if(nList == null || nList.getLength() <= 0)
        {
            header.appendChild(createAnElement("missVal", "-999.0", null));
        }

        putTimeSeriesHeaderInOrder(header);
    } // end method

    /**
     * Reformat the line with a specific format
     * 
     * @param format -- the specific format
     * @param aLine -- input line
     * @return the reformat line
     */
    protected String formatLine(final String format, String aLine)
    {
        //System.out.println(format);
        //System.out.println(aLine);
        final StringTokenizer token = new StringTokenizer(format, ",");
        int skipSpaces = 0;
        int formatSpaces = 0;
        int numberOfFormats = 0;
        String returnLine = "";

        while(token.hasMoreElements())
        {
            String tmpstr = (String)token.nextElement();
            if(tmpstr.startsWith("A"))
            {
                tmpstr = tmpstr.substring(1);
                skipSpaces += Integer.parseInt(tmpstr);
            }
            else if(tmpstr.indexOf('A') > 0)
            {
                final int multiple = Integer.parseInt(tmpstr.substring(0, tmpstr.indexOf('A')));
                final int aSpace = Integer.parseInt(tmpstr.substring(tmpstr.indexOf('A') + 1));
                skipSpaces += multiple * aSpace;
            }
            else if(tmpstr.startsWith("I"))
            {
                tmpstr = tmpstr.substring(1);
                skipSpaces += Integer.parseInt(tmpstr);
            }
            else if(tmpstr.indexOf('I') > 0)
            {
                final int multiple = Integer.parseInt(tmpstr.substring(0, tmpstr.indexOf('I')));
                final int aSpace = Integer.parseInt(tmpstr.substring(tmpstr.indexOf('I') + 1));
                skipSpaces += multiple * aSpace;
            }
            else if(tmpstr.startsWith("F"))
            {
                tmpstr = tmpstr.substring(1);
                formatSpaces += Double.valueOf(tmpstr).intValue();
            }
            else if(tmpstr.indexOf('F') > 0)
            {
                numberOfFormats = Integer.parseInt(tmpstr.substring(0, tmpstr.indexOf('F')));
                final int aSpace = Double.valueOf(tmpstr.substring(tmpstr.indexOf('F') + 1)).intValue();
                formatSpaces += aSpace;
            }
            //System.out.println(tmpstr);
        }
        //System.out.println("Skip spaces = " + skipSpaces + " number of formats = " + numberOfFormats + " format spaces = " + formatSpaces);
        aLine = aLine.substring(skipSpaces);
        //System.out.println(aLine);
        for(int i = 0; i < numberOfFormats; i++)
        {
            try
            {
                final String aString = aLine.substring(i * formatSpaces, formatSpaces * (i + 1));
                //System.out.println(aString.trim());
                returnLine += aString.trim() + " ";
            }
            catch(final StringIndexOutOfBoundsException e)
            {
                ;
            }
        }
        //System.out.println("return Line = " + returnLine.trim());
        return returnLine.trim();
    }

    /**
     * calculate the increment time
     * 
     * @param timeStep -- number of time steps
     * @param step -- step of increment
     * @return the increment of time step in milliseconds
     */
    public long incrementTimeStep(final int timeStep, final int step)
    {
        return (timeStep * step * 60 * 60 * 1000);
    }

    public long incrementTimeStep(final int timeStep)
    {
        return (timeStep * 60 * 60 * 1000);
    }

    /**
     * Get the DOM document
     * 
     * @return the document
     */
    public Document getDocument()
    {
        return _document;
    }

    /**
     * Get all children nodes from time series element
     * 
     * @return its children node list
     */
    public NodeList getSerieses()
    {
        return _timeSeries.getChildNodes();
    }

    /**
     * close the buffered writer
     * 
     * @throws IOException
     */
    public void closeWriter() throws IOException
    {
        //_out.flush();
        _out.close();
    }

    /**
     * write a close tag and close the writer
     * 
     * @param closeTag -- close tag name
     * @throws IOException
     */
    public void closeWriter(final String closeTag) throws IOException
    {
        _out.writeBytes("</" + closeTag + ">\n");
        _out.close();
    }

    /**
     * Use the XML DOM utilities to traverse the whole document
     * 
     * @param node -- top node
     */
    public void traverseXMLDocument(final Node node, final Character character)
    {
        final FEWSXMLDOMUtilities utilities = new FEWSXMLDOMUtilities(_out);
        if(character != null)
        {
            utilities.pushStack(character);
        }
        final StringBuilder stringBuilder = new StringBuilder();
        utilities.traverseXMLDocument(node, stringBuilder, "    "); // 4 spaces for indentation 
        if(character != null)
        {
            utilities.popStack();
        }
        try
        {
            utilities.writeBytes(stringBuilder);
        }
        catch(final IOException e)
        {
            e.printStackTrace();
        }
    }

    /**
     * Get all the properties from a property file
     * 
     * @param propertyFile -- property file
     * @return all properties
     */
    public Properties getProperties(final String propertyFile)
    {
        final Properties properties = new Properties();
        try
        {
            final FileInputStream fis = new FileInputStream(new File(propertyFile));
            final BufferedInputStream bis = new BufferedInputStream(fis);
            //properties.load(new BufferedInputStream(new FileInputStream(new File(propertyFile))));
            properties.load(bis);
            fis.close();
            bis.close();
        }
        catch(final IOException ioe)
        {
            System.err.println(ioe.getMessage());
        }
        if(FEWSDEBUG > -1)
        {
            final Enumeration<?> enumeration = properties.propertyNames();
            while(enumeration.hasMoreElements())
            {
                final String key = (String)enumeration.nextElement();
                System.out.println(key + " = " + properties.getProperty(key));
            }
        }
        return properties;
    }

    /**
     * write the logs into a diagnostics file
     * 
     * @param diagFileName -- diagnostics file name
     */
    public void writeLogger(final String diagFileName)
    {

        try
        {
            if(!_logger.getList().isEmpty())
            {
                FewsAdapterDAO.writeLog(_logger, diagFileName);
            }
        }
        catch(final Exception e)
        {
            e.printStackTrace();
        }
    }

    public static void main(final String args[])
    {
        String outFile = null;
        try
        {
            outFile = System.getProperty("outfile");
            String tsInput = System.getProperty("tsinput");
            String cardInput = System.getProperty("cardinput");

            final String locationId = System.getProperty("locationId");
            final String qualifierId = System.getProperty("qualifierId");
            String propertyFile = System.getProperty("properties");
            if(propertyFile == null || propertyFile.length() <= 0)
            {
                propertyFile = "fews_xml_timeseries_converter.properties";
            }

            String startTimestamp = "";
            String endTimestamp = "";
            String forecastTimestamp = "";
            startTimestamp = System.getProperty("startTimestamp");
            endTimestamp = System.getProperty("endTimestamp");
            forecastTimestamp = System.getProperty("forecastTimestamp");

            final Properties properties = new Properties();
            try
            {
                properties.load(new BufferedInputStream(new FileInputStream(new File(propertyFile))));
            }
            catch(final IOException ioe)
            {
                System.err.println(ioe.getMessage());
            }

            final LinkedHashMap<String, String> headerElements = new LinkedHashMap<String, String>();
            if(locationId != null && locationId.length() > 0)
            {
                headerElements.put("locationId", locationId);
            }
            if(qualifierId != null && qualifierId.length() > 0)
            {
                headerElements.put("qualifierId", qualifierId);
            }
            if(outFile == null || outFile.length() <= 0)
            {
                outFile = properties.getProperty("outfile");
                properties.remove("outfile"); // no longer need
            }
            if(tsInput == null || tsInput.length() <= 0)
            {
                tsInput = properties.getProperty("tsinput");
                properties.remove("tsinput");
            }
            if(cardInput == null || cardInput.length() <= 0)
            {
                cardInput = properties.getProperty("cardinput");
                properties.remove("cardinput");
            }
            if(startTimestamp == null || startTimestamp.length() <= 0)
            {
                startTimestamp = properties.getProperty("startTimestamp");
                properties.remove("startTimestamp");
            }
            if(endTimestamp == null || endTimestamp.length() <= 0)
            {
                endTimestamp = properties.getProperty("endTimestamp");
                properties.remove("endTimestamp");
            }
            if(forecastTimestamp == null || forecastTimestamp.length() <= 0)
            {
                forecastTimestamp = properties.getProperty("forecastTimestamp");
                properties.remove("forecastTimestamp");
            }
            final Enumeration<Object> enumeration = properties.keys();
            while(enumeration.hasMoreElements())
            {
                final String key = (String)enumeration.nextElement();
                if(!headerElements.containsKey(key))
                { // prevent duplication
                    headerElements.put(key, properties.getProperty(key));
                }
            }

            if(outFile == null)
            {
                throw new ArrayIndexOutOfBoundsException();
            }
            final Integer Debug = Integer.parseInt(System.getProperty("DEBUG", "-1"));
            final FewsTimeseriesConverter converter = new FewsTimeseriesConverter(outFile);
            converter.setDebug(Debug.intValue());
            headerElements.remove("DEBUG");

            converter.clearStartTimestamp();
            if(startTimestamp != null && startTimestamp.length() == 19)
            { // timestamp is 'yyyy-mm-dd HH:MM:SS', total length is 19
                try
                {
                    converter.setStartTimestamp(startTimestamp);
                }
                catch(final IllegalArgumentException iae)
                {
                    System.err.println(iae.getMessage());
                }
            }
            converter.clearEndTimestamp();
            if(endTimestamp != null && endTimestamp.length() == 19)
            {
                try
                {
                    converter.setEndTimestamp(endTimestamp);
                }
                catch(final IllegalArgumentException iae)
                {
                    System.err.println(iae.getMessage());
                }
            }
            converter.clearForecastTimestamp();
            if(forecastTimestamp != null && forecastTimestamp.length() == 19)
            {
                try
                {
                    converter.setForecastTimestamp(forecastTimestamp);
                }
                catch(final IllegalArgumentException iae)
                {
                    System.err.println(iae.getMessage());
                }
            }

            try
            {
                if(tsInput != null)
                {
                    converter.setHeaderElements(headerElements);
                    converter.readTSFile(tsInput);
                    converter.writeLogger("tsDiag.xml");
                }
                if(cardInput != null)
                {
                    converter.setHeaderElements(headerElements);
                    converter.readCardFile(cardInput);
                    converter.writeLogger("dataCardDiag.xml");
                }
            }
            catch(final Exception e)
            {

                e.printStackTrace();
                System.err.println(e.getMessage());
            }

            writeAndValidateXml(outFile, converter);

        }
        catch(final Exception e)
        {
            System.err.println("Usage java FewsTimeseriesConverter -Doutfile=filename "
                + "-Dtsinput=filename -Dcardinput=filename " + "[-DstartTimestamp='yyyy-mm-dd HH:MM:SS']"
                + "[-DendTimestamp='yyyy-mm-dd HH:MM:SS']");
            e.printStackTrace();

        }

        System.out.println(" XML file succesfully created!!");
        // create txt file for stand alone model
        String createTextFile = System.getProperty("createTextFile");
        //System.out.println("create text file property = " + createTextFile);
        if(createTextFile == null)
        {
            createTextFile = OHDConstants.TRUE_STR;
        }
        else if(createTextFile != null && !createTextFile.equalsIgnoreCase(OHDConstants.TRUE_STR))
        {
            createTextFile = OHDConstants.FALSE_STR;
        }
        else if(createTextFile != null && createTextFile.equalsIgnoreCase(OHDConstants.TRUE_STR))
        {
            createTextFile = OHDConstants.TRUE_STR;
        }
        final boolean notCreateTextFile = Boolean.parseBoolean(createTextFile);
        //System.out.println(",,,,,,,,,,,, will create text file = " + notCreateTextFile);
        if(notCreateTextFile)
        {
            System.out.println("creating stand alone .txt file is no longer supported");
        }
    }

    public static void writeAndValidateXml(final String outFile, final FewsTimeseriesConverter converter) throws IOException,
                                                                                                         Exception
    {
        if(!converter.getExistedXMLFile()) // write to a new XML file
        {
            converter.traverseXMLDocument(converter.getDocument(), null);
            converter.closeWriter();
            FewsXmlValidation.validateXmlFileAgainsXMLSchema(outFile,
                                                             OHDFewsAdapterConstants.PI_TIMESERIES_XML_SCHEMA_TAG,
                                                             _logger);
        }
        else
        // write to an existed XML file
        {
            final NodeList serieses = converter.getSerieses();
            for(int i = 0; i < serieses.getLength(); i++)
            {
                converter.traverseXMLDocument(serieses.item(i), '\t');
            }
            converter.closeWriter(rootTag);
            FewsXmlValidation.validateXmlFileAgainsXMLSchema(outFile,
                                                             OHDFewsAdapterConstants.PI_TIMESERIES_XML_SCHEMA_TAG,
                                                             _logger);
        }
    }
}
