package ohd.hseb.ohdgriddedmodels;

import java.io.File;

import javax.management.timer.Timer;

import ohd.hseb.grid.NetcdfConstants;
import ohd.hseb.grid.NetcdfTemplateCreator;
import ohd.hseb.grid.RfcGridWithTime;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.Driver;
import ohd.hseb.util.fews.OHDUtilities;

public abstract class GriddedDriver extends Driver
{

    protected RfcGridWithTime _inputTsGrid;
    protected RfcGridWithTime _outputTsGrid;

    protected long _computationStartTimeInMillis; //= runInfo runStartTime + drivingTS intervalInMillis 

    protected int _inputTsInterval;

    protected int _hrapRow;
    protected int _hrapCol;

    protected int _processedGridNum = 0;

    @Override
    public void executeDriver() throws Exception
    {
        //super._state has been loaded in OHDFewsAdapter. Based on _state, set the RFC HRAP Row and Col range:
        final GriddedState griddedState = ((GriddedState)(super._state));
        final int westColumn = griddedState._rfcGridWithTime.getRfcOrigHrapColumn();
        final int eastColumn = westColumn + griddedState._rfcGridWithTime.getColumnNum() - 1;
        final int southRow = griddedState._rfcGridWithTime.getRfcOrigHrapRow();
        final int northRow = southRow + griddedState._rfcGridWithTime.getRowNum() - 1;

        //check the state time is equal to run_info.xml run start time: if not, log it
        if(griddedState.getDateTime() != _runInfo.getRunStartTimeLong())
        {
            _logger.log(Logger.WARNING,
                        "Gridded State time[" + DateTime.getDateTimeStringFromLong(griddedState.getDateTime(), null)
                            + "] is different from run start time["
                            + DateTime.getDateTimeStringFromLong(_runInfo.getRunStartTimeLong(), null) + "]");
        }

        //parse parameter NetCDF file
        final String paramsNcFile = this.findParamNcFile();
        final GriddedParameters griddedParams = ((GriddedParameters)super._parameters);
        griddedParams.loadParameters(paramsNcFile);

        //parse input NetCDF time series file. 
        final String tsNcFile = getRunInfo().getInputNetCdfFile();

        _inputTsGrid = new RfcGridWithTime(tsNcFile, _logger);

        //all input TSs inside _inputTsGrids3D assuming have same interval
        _inputTsInterval = _inputTsGrid.getTimeSeriesIntervalInHR();
        _computationStartTimeInMillis = _runInfo.getRunStartTimeLong() + _inputTsInterval * Timer.ONE_HOUR;

        this.checkTsWithEnoughData();

        this.createOutputTimeseriesGrids(_inputTsInterval);

        if(_logger.getPrintDebugInfo() > 0 && _logger.getPrintDebugInfo() <= 4)
        {//so if printDeBugInfo is 5, the logging will not be turned off -- useful during error situation
            _logger.log(Logger.INFO,
                        "Due to huge number of logging messages at debug level, re-set PrintDebugInfo to 0. It was "
                            + _logger.getPrintDebugInfo());

            _logger.setPrintDebugInfo(0);
        }

        /*
         * Going through each pixel within the range of NetCDF file and select the ones within RFC boundary. Check that
         * pixel's input TSs have missing value or not. Then do the model computation at that pixel.
         */
        for(_hrapRow = southRow; _hrapRow <= northRow; _hrapRow++)
        {
            for(_hrapCol = westColumn; _hrapCol <= eastColumn; _hrapCol++)
            {
                if(griddedParams.getGrid().isValueMissing(NetcdfConstants.MASK_VAR_NAME, _hrapCol, _hrapRow))
                {//this pixel is outside of RFC, skip it
                    continue;
                }

                _logger.log(Logger.DEBUG, "Run model computation at the pixel: HRAP_ROW=" + _hrapRow + " HRAP_COL="
                    + _hrapCol);

                this.execute(); //run child class's method, doing model computation at this pixel

            }//close hrapCol for loop

        }//close hrapRow for loop

        if(_processedGridNum == 0)
        {//none of the grids have been calculated, i.e. all grids have been skipped

            throw new Exception("All grids have been skipped. None of them has been calculated!");
        }

        //write _outputTsGrids3D to NetCDF file
        _outputTsGrid.writeOutNetcdfFile(_runInfo.getOutputNetCdfFile());

        griddedState.setDateTimeInGrid(_runInfo.getRunEndTimeLong());//set "time" dim value in NetCDF file statesO.nc
        griddedState.setDateTime(_runInfo.getRunEndTimeLong());//set the warm state time in states meta file states.xml

        super.writeStatesMetaFile();

        //write _state which values have been updated to NetCDF file
        _state.writeState(getOutputStateFile(), _logger);

    }

    /**
     * Find the NetCDF file within the directory {@link NetcdfConstants#INPUT_NETCDF_PARAMSFILE_DIR}, specified as a
     * property in run_info.xml. If didn't find it, throw an Exception.
     */
    private String findParamNcFile() throws Exception
    {
        //parse NetCDF param now
        if(_runInfo.getProperties().containsKey(NetcdfConstants.INPUT_NETCDF_PARAMSFILE_DIR) == false)
        {
            throw new Exception("Within Run Info file, the property[" + NetcdfConstants.INPUT_NETCDF_PARAMSFILE_DIR
                + "] is missing.");
        }

        final String paramFileDirName = _runInfo.getProperties()
                                                .getProperty(NetcdfConstants.INPUT_NETCDF_PARAMSFILE_DIR);

        final File paramFileDir = new File(paramFileDirName);

        final File[] filesInDir = paramFileDir.listFiles();

        String paramNcFile = null;

        //It is expected to have one NetCDF file in the directory ("work/") which contains parameters
        for(int i = 0; i < filesInDir.length; i++)
        {
            final String fileName = filesInDir[i].getPath();

            if(OHDUtilities.isNetcdfFile(fileName))
            {
                paramNcFile = fileName;

                break;
            }
        }

        if(paramNcFile == null)
        {//failed to locate the NC file
            throw new Exception("Could not find a NetCDF file containing parameters in the directory "
                + paramFileDirName);
        }

        return paramNcFile;
    }

    /**
     * Check TSs have enough data by checking "time" dimension start time and number of elements. If not, throw an
     * Exception.
     * 
     * @throws Exception
     */
    private void checkTsWithEnoughData() throws Exception
    {
        final int expectedTsEventsNum = (int)((_runInfo.getRunEndTimeLong() - _computationStartTimeInMillis) / (_inputTsInterval * Timer.ONE_HOUR));

        if(_inputTsGrid.getStartTimeInMillis() > _computationStartTimeInMillis)
        {
            final String computeStartTimeStr = DateTime.getDateTimeStringFromLong(_computationStartTimeInMillis, null);
            final String tsStartTimeStr = DateTime.getDateTimeStringFromLong(_inputTsGrid.getStartTimeInMillis(), null);

            throw new Exception("No enough data in input time series file: computation starts at "
                + computeStartTimeStr + " but time series starts at " + tsStartTimeStr);
        }
        else if(expectedTsEventsNum > _inputTsGrid.getTimeOrDurationDimSize())
        {
            final String computeEndTimeStr = DateTime.getDateTimeStringFromLong(_runInfo.getRunEndTimeLong(), null);

            final long tsEndTimeInMillis = _inputTsGrid.getStartTimeInMillis()
                + (_inputTsGrid.getTimeOrDurationDimSize() - 1) * _inputTsInterval * Timer.ONE_HOUR;

            final String tsEndTimeStr = DateTime.getDateTimeStringFromLong(tsEndTimeInMillis, null);
            throw new Exception("No enough data in input time series file: computation starts at " + computeEndTimeStr
                + " but time series ends at " + tsEndTimeStr);
        }
    }

    /**
     * First create output time series NetCDF file template, then open it in {@link #_outputTsGrid}, which will be
     * inserted values in each loop of model computation. The output time series names are from
     * {@link #getOutputTsNameArray()}.
     */
    private void createOutputTimeseriesGrids(final int tsInterval) throws Exception
    {
        //create NetCDF template with specific output time series names

        final int southRow = _inputTsGrid.getRfcOrigHrapRow();
        final int rowNum = _inputTsGrid.getRowNum();
        final int westCol = _inputTsGrid.getRfcOrigHrapColumn();
        final int colNum = _inputTsGrid.getColumnNum();
        final int timeSteps = (int)((_runInfo.getRunEndTimeLong() - _computationStartTimeInMillis) / Timer.ONE_HOUR / _inputTsInterval) + 1;

        //note: the output TS start time, end time are based on Run Info, not input TS, which could have any start time or end time
        //the output TS interval is based on input TS interval - they are same

        final String outputNcFile = _runInfo.getOutputNetCdfFile();
        if(outputNcFile == null)
        {
            throw new Exception("Within the Run Info file, the element[" + NetcdfConstants.OUTPUT_NETCDF_FILE
                + "] is missing.");
        }

        final NetcdfTemplateCreator outputNcFileCreator = new NetcdfTemplateCreator(southRow,
                                                                                    rowNum,
                                                                                    westCol,
                                                                                    colNum,
                                                                                    timeSteps);

        outputNcFileCreator.createRfcTimeSeriesTemplateNCfiles(outputNcFile,
                                                               getOutputTsNameArray(),
                                                               _computationStartTimeInMillis,
                                                               tsInterval,
                                                               timeSteps);

        _outputTsGrid = new RfcGridWithTime(outputNcFile, _logger);

    }

    /**
     * Each subclass must define the output time series names, e.g. for GriddedSnow17ModelDriver, it is ["RAIM","SASC"].
     */
    abstract protected String[] getOutputTsNameArray();

    /**
     * A helper method. The grids _outputTsGrids3D must have the variable same as the time series type and the "time"
     * dimension size in _outputTsGrids3D must be the same or greater than the time series time steps.
     */
    protected void insertTsToGrids(final RegularTimeSeries rts, final int hrapRow, final int hrapCol) throws Exception
    {
        for(int i = 0; i < rts.getMeasurementCount(); i++)
        {
            _outputTsGrid.setDoubleVariable(rts.getTimeSeriesType(), //
                                            i, //
                                            hrapCol, //
                                            hrapRow, //
                                            rts.getMeasurementValueByIndex(i, rts.getMeasuringUnit()));
        }
    }

}
