package ohd.hseb.ohdgriddedmodels;

import java.util.HashMap;
import java.util.Map;
import java.util.TimeZone;

import javax.management.timer.Timer;

import ohd.hseb.grid.RfcGridWithTime;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.IState;
import ohd.hseb.util.fews.OHDUtilities.StateLocation;
import ucar.ma2.Array;
import ucar.ma2.Index;

/**
 * This class is for handling gridded state. It is loaded from a NETCDF file and the state information of each pixel are
 * stored at {@link #_rfcGridWithTime}.
 */
public class GriddedState implements IState
{
    protected RfcGridWithTime _rfcGridWithTime;

    private long _dateTime;

    private String _version;

    protected Logger _logger; //set by OHDFewsAdapter

    private String _id = null;

    private String _name = null;

    private TimeZone _timeZone;

    private String _daylightSavingObservatingTimeZone;

    private Map<String, StateLocation> _stateLocation;

    public GriddedState()
    {
        //
    }

    public GriddedState(final String stateNetcdfFileName, final Logger logger) throws Exception
    {
        this.loadState(stateNetcdfFileName, logger);
    }

    /**
     * Process the initial state NETCDF file(statesI.nc) and stores each pixel's states at {@link #_rfcGridWithTime}.
     */
    public void loadState(final String stateNetcdfFileName, final Logger logger) throws Exception
    {

        _logger = logger;

        _rfcGridWithTime = new RfcGridWithTime(stateNetcdfFileName, _logger);

        _dateTime = _rfcGridWithTime.getStartTimeInMillis();

        _logger.log(Logger.DEBUG, "Loaded the gridded state from the NETCDF file: " + stateNetcdfFileName
            + " warm state time:" + DateTime.getDateTimeStringFromLong(_dateTime, null));

    }

    /**
     * Output each pixel's states to a NETCDF file.
     */
    @Override
    public void writeState(final String outputStateNetcdfFileName, final Logger logger) throws Exception
    {
        _rfcGridWithTime.writeOutNetcdfFile(outputStateNetcdfFileName);

        _logger.log(Logger.DEBUG, "Output the state to the NETCDF file: " + outputStateNetcdfFileName);
    }

    /**
     * Prints out the following information to console:<br>
     * State time = 2011-05-24 13:48:00 GMT<br>
     * West column= 850<br>
     * South row= 470<br>
     * Colunm number= 200<br>
     * Row number= 200<br>
     * 
     * @throws Exception
     */
    public void printVariableRange() throws Exception
    {
        _rfcGridWithTime.printVariableRange();
    }

    /**
     * Prints out every pixel containing non-missing values in the format like:<br>
     * HRAP_ROW= 620 HRAP_COLUMN= 924 TINDEX= -4.515737<br>
     * HRAP_ROW= 620 HRAP_COLUMN= 924 WE= 13.221888<br>
     * HRAP_ROW= 620 HRAP_COLUMN= 924 SBAESC= 0.188781<br>
     * HRAP_ROW= 620 HRAP_COLUMN= 924 EXLAG_ARRAY= "0.0 0.0"<br>
     * 
     * @throws Exception
     */
    public void printOutGridsValues() throws Exception
    {
        _rfcGridWithTime.printOutGridsValues();
    }

    @Override
    /**
     * used to set state time of state grid; Note: state grids only have one time
     * it represents the start or end of the run
     * 
     */
    public void setDateTime(final long dateTime)
    {
        _dateTime = dateTime;
    }

    public void setDateTimeInGrid(final long dateTimeInMilliSeconds)
    {
        final Array timeArray = _rfcGridWithTime.getTimeArray();
        final Index timeIndex = timeArray.getIndex();
        final long dateTimeInMinutesSince1970 = dateTimeInMilliSeconds / Timer.ONE_MINUTE;
        timeArray.setDouble(timeIndex.set(0), dateTimeInMinutesSince1970);
    }

    @Override
    public long getDateTime()
    {
        return _dateTime;
    }

    @Override
    public void setVersion(final String value)
    {
        _version = value;
    }

    @Override
    public String getVersion()
    {
        return _version;
    }

    @Override
    public void setStateLocation(final StateLocation value)
    {
        if(_stateLocation == null)
        {
            _stateLocation = new HashMap<String, StateLocation>();
        }

        if(value != null)
        {
            final String pathFileName = value.getReadLocation();
            final int fileSeparatorIndex = pathFileName.lastIndexOf(System.getProperty("file.separator"));
            final String fileName = pathFileName.substring(fileSeparatorIndex + 1);
            this._stateLocation.put(fileName, value);
        }
    }

    @Override
    public Map<String, StateLocation> getStateLocation()
    {
        if(_stateLocation == null)
        {
            _stateLocation = new HashMap<String, StateLocation>();
        }
        return this._stateLocation;
    }

    @Override
    public void setLogger(final Logger logger)
    {
        _logger = logger;
    }

    @Override
    public void setId(final String id)
    {
        _id = id;
    }

    @Override
    public String getId()
    {
        return _id;
    }

    @Override
    public void setName(final String name)
    {
        _name = name;
    }

    public String getName()
    {
        return _name;
    }

    public Logger getLogger()
    {
        return _logger;
    }

    public void setTimeZone(final TimeZone timeZone)
    {
        _timeZone = timeZone;
    }

    public TimeZone getTimeZone()
    {
        return _timeZone;
    }

    /**
     * Returns the internal grid.
     */
    public RfcGridWithTime getGrid()
    {
        return _rfcGridWithTime;
    }

    @Override
    public void setDaylightSavingObservatingTimeZone(final String daylightSavingObservatingTimeZone)
    {
        _daylightSavingObservatingTimeZone = daylightSavingObservatingTimeZone;

    }

    @Override
    public String getDaylightSavingObservatingTimeZone()
    {
        // TODO Auto-generated method stub
        return _daylightSavingObservatingTimeZone;
    }
}
