package ohd.hseb.ohdgriddedmodels.snow17;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.management.timer.Timer;

import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.ohdgriddedmodels.GriddedDriver;
import ohd.hseb.ohdmodels.snow17.Snow17Model;
import ohd.hseb.ohdmodels.snow17.Snow17ModelConstants;
import ohd.hseb.ohdmodels.snow17.Snow17ModelDriver;
import ohd.hseb.ohdmodels.snow17.Snow17ModelParameters;
import ohd.hseb.ohdmodels.snow17.Snow17ModelState;
import ohd.hseb.ohdmodels.snow17.Snow17Technique;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.DataType;
import ohd.hseb.util.fews.FewsRegularTimeSeries;

/**
 * This ModelDriver subclass is called by {@link FewsAdapter} through
 * {@link FewsAdapter#setUpAndExecuteModel(Properties)} to conduct Snow17 computation for the time period in the input
 * fews.xml. super._parameters and super._state in the super class ModelDriver are also loaded inside the method
 * {@link FewsAdapter#setUpAndExecuteModel(Properties)}. The Snow17ModelDriver method
 * {@link #execute(String, String, Properties, Map)} is the entry point from FewsAdapter. The results, four output time
 * series(raim, swe, sasc and snsg), are stored in the argument Map and returned. Finally, they are written to the xml
 * file {@link FewsAdapter#OUTPUT_FILE_NAME}.
 * <p>
 * The method {@link #ex19()} loops through the whole computation time period, with step of MAT input time series
 * interval. During each loop, {@link #ex19()} calls {@link Snow17Model}'s method
 * {@link Snow17Model#pack19(double, double[], double[], double, long, double, double)} once, which does the Snow17
 * computation.
 * <p>
 * All the date and time are in GMT timezone. For input data in non-GMT time zone, the time is converted to GMT time in
 * FewsAdapter.java. When needs to output the computation results, the time is switched back to the user's timezone.
 * 
 * @author FewsPilot Team
 */
final public class GriddedSnow17ModelDriver extends GriddedDriver
{

    /** alias to super._parameters to save type casting */
    private final Snow17GriddedParameters _griddedSnow17ModelParams;

    /** alias to super._state to save type casting */
    private final Snow17GriddedModelState _griddedSnow17ModelState;

    /**
     * Default constructor. From the view of Fews, it is implicitly called by {@link FewsAdapter} in the method
     * {@link FewsAdapter#setUpAndExecuteModel(Properties arguments)}:
     * <p>
     * Class modelClass = Class.forName(arguments.getProperty(_MODEL_NAME_TAG));<br>
     * Object obj = modelClass.newInstance(); //calls ModelDriver subclass(e.g. Snow17ModelDriver.java) constructor
     * super._parameters and super._state are created by default constructors. They are not loaded yet (from params xml
     * file and statesI file)!
     * 
     * @throws IOException
     */
    public GriddedSnow17ModelDriver() throws Exception
    {

        _griddedSnow17ModelState = new Snow17GriddedModelState();
        super._state = _griddedSnow17ModelState;

        _griddedSnow17ModelParams = new Snow17GriddedParameters();
        super._parameters = _griddedSnow17ModelParams;

    } // close constructor

    /**
     * Called from {@link GriddedDriver#executeDriver()}.
     */
    @Override
    public void execute() throws Exception
    {

        //do computation for this pixel
        this.ex19GridToLumped(_hrapCol, _hrapRow);

    }// back to GriddedDriver executeDriver()

    /**
     * Do snow17 computation at this pixel: first extract input TSs(MAP, MAT), parameters and states for this pixel. If
     * there is missing values in them, skip the computation. If no missing values, call existing Snow17ModelDriver code
     * to compute, then save the output time series into _outputTsGrids3D, snow17 state into _griddedSnow17ModelState
     */
    private void ex19GridToLumped(final int hrapCol, final int hrapRow) throws Exception
    {

        final List<RegularTimeSeries> tsListAtPixel = this.extractInputTSsAtPixel(hrapCol, hrapRow);

        if(tsListAtPixel == null)
        {//the input TSs have missing values, log it and exit method, so the model computation is not done at this pixel
            _logger.log(Logger.DEBUG,
                        "Input time series(MAP or MAT) has missing values at HRAP_COL=" + hrapCol + " HRAP_ROW="
                            + hrapRow + " within "
                            + DateTime.getDateTimeStringFromLong(_computationStartTimeInMillis, null) + " to "
                            + DateTime.getDateTimeStringFromLong(_runInfo.getRunEndTimeLong(), null));

            return;
        }

        final Snow17ModelDriver snow17ModelDriver = new Snow17ModelDriver();
        
        

        final RegularTimeSeries matTs = tsListAtPixel.get(0);
        final int matTsInterval = matTs.getIntervalInHours();
        snow17ModelDriver.setDrivingRTS(matTs); //first TS in list is MAT, see setInputTsAtPixel()

        final Snow17ModelParameters snow17ModelParamsAtPixel = _griddedSnow17ModelParams.getSnow17ModelParameters(hrapCol,
                                                                                                                  hrapRow);
        final Snow17ModelState snow17ModelStateAtPixel = _griddedSnow17ModelState.getSnow17ModelState(hrapCol, hrapRow);

        if(snow17ModelParamsAtPixel == null || snow17ModelStateAtPixel == null)
        {
            _logger.log(Logger.WARNING, "Skip computation at HRAP_COL[" + hrapCol + "], HRAP_ROW[" + hrapRow
                + "] due to missing value in snow17 parameters or states.");

            return;
        }

        snow17ModelParamsAtPixel.setLogger(_logger);
        snow17ModelStateAtPixel.setLogger(_logger);

        snow17ModelStateAtPixel.setDateTime(_runInfo.getRunStartTimeLong());

        snow17ModelDriver.setParameters(snow17ModelParamsAtPixel);
        snow17ModelDriver.setState(snow17ModelStateAtPixel);
        snow17ModelDriver.setTsList(tsListAtPixel);

        snow17ModelDriver.setRunInfo(_runInfo);

        snow17ModelDriver.setLogger(_logger);

        // for now we don't handle cox set to false
        snow17ModelDriver.setNeedCarryoverTransfer(false);
        
        // check to see if mat interval matches parameters
        // if not adjust parameters
        
        if( matTsInterval != snow17ModelParamsAtPixel.getAirTempInterval()) {
        	snow17ModelParamsAtPixel.modifyParametersForDifferentTimestep(matTsInterval);
        }
        
        if(matTsInterval != snow17ModelStateAtPixel.getITPX())
        {
        	snow17ModelStateAtPixel.modifyStatesForDifferentTimestep(matTsInterval);
        }
        
        _processedGridNum++; //counting num of grids being calculated

        snow17ModelDriver.execute();

        final List<FewsRegularTimeSeries> resultTsListAtPixel = snow17ModelDriver.getFewsOutputTimeseriesList();

        for(final RegularTimeSeries rts: resultTsListAtPixel)
        {
            if(rts.getTimeSeriesType().equals(DataType.SNSG_DATATYPE)
                || rts.getTimeSeriesType().equals(Snow17ModelState.SNDPT_TAG))
            {//SNSG and SNDPT are in units of CM, in the output NetCDF file, they are in MM
                rts.setMeasuringUnit(MeasuringUnit.mm);
            }

            super.insertTsToGrids(rts, hrapRow, hrapCol);
        }

        //save output state
        _griddedSnow17ModelState.setSnow17ModelState(snow17ModelStateAtPixel, hrapCol, hrapRow);

    }

    /**
     * Extract MAT and MAP input time series at the pixel from the NetCDF file. This method also does these tasks:<br>
     * 1)Trim the two input TSs the beginning and the end based on computation start time and end time from Run Info.
     * This way takes care of both cases that too much data or not enough data.<br>
     * 2)After trimming, check if there is missing value inside the input TSs: if so, return null. Both MAT and MAP ts
     * are not allowed to have missing value.<br>
     * If reaching to the end, form a ts list and return.
     */
    private List<RegularTimeSeries> extractInputTSsAtPixel(final int hrapCol, final int hrapRow) throws Exception
    {

        final List<RegularTimeSeries> tsListAtPixel = new ArrayList<RegularTimeSeries>(2);

        final long tsStartTime = _inputTsGrid.getStartTimeInMillis();
        final long tsEndTime = tsStartTime + _inputTsInterval * Timer.ONE_HOUR
            * (_inputTsGrid.getTimeOrDurationDimSize() - 1);

        /*
         * the objects in _tsList are actually FewsRegularTimeSeries, not just RegularTimeSeries. This is important,
         * because later(in getFewsOutputTimeseriesList()) when converting result time series from RegularTimeSeries
         * type to FewsRegualrTimeSeries type, _tsList.get(0) will be casted to FewsRegularTimeSeries. Otherwise, that
         * casting will crash.
         */
        String unitStr = _inputTsGrid.getUnitAsString(DataType.MAP_DATATYPE);
        final RegularTimeSeries mapTs = new FewsRegularTimeSeries(tsStartTime, //
                                                                  tsEndTime,//fine with _inputTsGrids3D's TSs extended end time
                                                                  _inputTsInterval,
                                                                  MeasuringUnit.getMeasuringUnit(unitStr));
        mapTs.setTimeSeriesType(DataType.MAP_DATATYPE);

        unitStr = _inputTsGrid.getUnitAsString(DataType.MAT_DATATYPE);
        final RegularTimeSeries matTs = new FewsRegularTimeSeries(tsStartTime, //
                                                                  tsEndTime, //fine with _inputTsGrids3D's TSs extended end time
                                                                  _inputTsInterval,
                                                                  MeasuringUnit.getMeasuringUnit(unitStr));
        matTs.setTimeSeriesType(DataType.MAT_DATATYPE);

        double map;
        double mat;
        for(int i = 0; i < mapTs.getMeasurementCount(); i++)
        {
            map = _inputTsGrid.getDoubleVariable(DataType.MAP_DATATYPE, i, hrapCol, hrapRow);
            mat = _inputTsGrid.getDoubleVariable(DataType.MAT_DATATYPE, i, hrapCol, hrapRow);

            mapTs.setMeasurementByIndex(map, i);
            matTs.setMeasurementByIndex(mat, i);
        }

        //trim TSs based on Run Info specified period
        matTs.trimTimeSeriesAtStartWithCheck(_computationStartTimeInMillis);
        matTs.trimTimeSeriesAtEndWithCheck(getRunInfo().getRunEndTimeLong());

        mapTs.trimTimeSeriesAtStartWithCheck(_computationStartTimeInMillis);
        mapTs.trimTimeSeriesAtEndWithCheck(getRunInfo().getRunEndTimeLong());

        if(matTs.hasMissingValue() || mapTs.hasMissingValue())
        {
            return null;
        }

        tsListAtPixel.add(matTs); //first ts is MAT
        tsListAtPixel.add(mapTs); //second ts is MAP

        return tsListAtPixel;

    }

    /**
     * Returns the output time series names, depending on running slim version or full version, controlled by TECHNIQUE
     * SACSNOW.
     */
    @Override
    protected String[] getOutputTsNameArray()
    {
        final String[] primTsNames = new String[]{DataType.RAIM_DATATYPE, DataType.SASC_DATATYPE, DataType.SWE_DATATYPE};
        final String[] allTsNames = new String[]{DataType.RAIM_DATATYPE, // 22 in total
            DataType.SASC_DATATYPE, //
            DataType.SWE_DATATYPE, //
            DataType.SNSG_DATATYPE, //
            DataType.PRAIN_DATATYPE, //
            DataType.PSFALL_DATATYPE, //
            DataType.PSNWRO_DATATYPE, //
            DataType.PROBG_DATATYPE, //
            Snow17ModelState.ACCMAX_TAG, //
            Snow17ModelState.AEADJ_TAG, //
            Snow17ModelState.LIQW_TAG, //
            Snow17ModelState.NEGHS_TAG, //
            Snow17ModelState.SB_TAG,//
            Snow17ModelState.SBAESC_TAG, //
            Snow17ModelState.SBWS_TAG, //
            Snow17ModelState.SNDPT_TAG, //
            Snow17ModelState.SNTMP_TAG, //
            Snow17ModelState.STORGE_TAG,//
            Snow17ModelState.TAPREV_TAG, //
            Snow17ModelState.TINDEX_TAG, //
            Snow17ModelState.WE_TAG, //
            Snow17ModelState.PQNET_TAG};

        final Snow17Technique snow17Tech = new Snow17Technique(getDriverProperties(), _logger);

        if(snow17Tech.getBoolValue(Snow17ModelConstants.SACSNOW_FULL_VERSION)) //running in full version
        {
            return allTsNames;
        }

        return primTsNames;
    }
    
} // close class
