package ohd.hseb.ohdmodels.apicont;

import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import ohd.hseb.measurement.AbsTimeMeasurement;
import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.MeasuringUnitType;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.SortedProperties;
import ohd.hseb.util.fews.DataType;
import ohd.hseb.util.fews.ModelException;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.State;
import ohd.hseb.util.fews.ohdmodels.LegacyModelAdapter;
import ohd.hseb.util.fews.ohdmodels.ModelDriver;

final public class APIContModelDriver extends ModelDriver
{

    private static final String APICONT_EXE_NAME = "apicont";

    private final List<RegularTimeSeries> _modsTsList;

    //MOD TSs: 10 types
    private final static String[] _APICONT_MOD_TYPES = new String[]{"AEICQN", "AIADJ", "APICBASF", "API", "SMI",
        "SMID", "BFSC", "BFI", "FI", "FEI"};

    private final static String _DRIVING_TS_TYPE = DataType.RAIM_DATATYPE;

    private static final String MAPE_TIMESTEP = "mapeTimeStep";

    private String _mapeTimeStep = "24";

    public APIContModelDriver()
    {
        super(); // calling ModelDriver constructor

        _state = new APIContModelState();

        _parameters = new APIContModelParameters();
        _parameters.setLogger(_logger);

        _savedParameters = new APIContModelParameters();
        _savedParameters.setLogger(_logger);

        _modsTsList = new ArrayList<RegularTimeSeries>();

    }

    @Override
    public void execute() throws Exception
    {

        // Send in work dir and state dir
        final LegacyModelAdapter apicontAdapter = new LegacyModelAdapter(this, getWorkDir(), _logger);

        apicontAdapter.writeStandAloneFiles();

        // apicont needs an extra argument - mod file name
        final SortedProperties properties = new SortedProperties();
        properties.put(LegacyModelAdapter.MOD_FILE_NAME, getWorkDir() + "/" + LegacyModelAdapter.MODS_TXT_FILE);

        // Write AEICQN, AIADJ, APICBASF and APICCO mods for API-CONT
        // model
        apicontAdapter.writeModsToFile(_modsTsList);

        //Clear out state Map before storing the new data (bug 244)
        ((State)_state).getStateMap().clear();

        //get user sets up property mape time interval from runinfo

        _mapeTimeStep = _runInfo.getProperties().getProperty(MAPE_TIMESTEP);

        if(_runInfo.getProperties().getProperty(MAPE_TIMESTEP) == null)
        {
            _mapeTimeStep = String.valueOf(24);
        }
        else
        {
            _mapeTimeStep = _runInfo.getProperties().getProperty(MAPE_TIMESTEP);
        }

        properties.put("mapeTimeStep", String.valueOf(_mapeTimeStep));

        // if there are any model specific arguments to be sent,
        // fill in a String [] array and send to executeModel as last argument
        apicontAdapter.executeModel(getDriverProperties().getProperty(OHDConstants.LEGACY_LOCATION_DIR) + "/"
            + APICONT_EXE_NAME, properties);

        // load the results
        final List<RegularTimeSeries> resultsList = apicontAdapter.readOutputFlatFilesToLists();

        for(int i = 0; i < resultsList.size(); i++)
        {
            addTStoResultMap(resultsList.get(i));
        }

        getState().loadState(apicontAdapter.getOutputStateFileName(), _logger);
        //SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm");
        //sdf.setTimeZone(TimeZone.getTimeZone("GMT"));
        //System.out.println("ComputationEndTime = " + sdf.format(getComputationEndTime()));
        getState().setDateTime(getComputationEndTime());

    }

    /**
     * Overriding the method. Be called in {@link LegacyModelAdapter#LegacyModelAdapter(ModelDriver, String, Logger)}
     */
    @Override
    final protected void runModelDriverValidation() throws Exception
    {

        final Iterator<RegularTimeSeries> ite = getTsList().iterator();

        RegularTimeSeries inputRts;

        String tsType;

        while(ite.hasNext())
        {
            inputRts = ite.next();

            tsType = inputRts.getTimeSeriesType();

            /*---------------try to find _drivingTs ------------------*/
            if(_drivingTs == null)
            {
                // only ts containing MM data should be considered as driving time series for this model
                if((inputRts.getMeasuringUnit().getType() == MeasuringUnitType.length)
                    && (tsType.equalsIgnoreCase(_DRIVING_TS_TYPE)))
                {
                    _drivingTs = inputRts;
                    _logger.log(Logger.DEBUG, "Driving Time Series set to " + _drivingTs.getLocationId() + " "
                        + _drivingTs.getTimeSeriesType() + " " + _drivingTs.getIntervalInHours());

                    continue; //skip next for loop in this run. Move on to next RTS
                }
            }

            //fish out MOD TSs from _tsList and put them in _modsTsList
            for(final String type: _APICONT_MOD_TYPES)
            {
                if(tsType.equalsIgnoreCase(type))
                {
                    _modsTsList.add(inputRts.clone());

                    ite.remove();

                    break; //jump out of for loop
                }
            }

        } //close while loop

        if(_drivingTs == null)
        {
            _logger.log(Logger.DEBUG, "Driving Time Series is null...");
        }

        super.runModelDriverValidation();

        if(_modsTsList.size() > 0)
        {
            super.checkInputTsInSync(_modsTsList);
        }

        final String statesStartTimeStr = DateTime.getTimeStringFromLong(getState().getDateTime(),
                                                                         OHDConstants.GMT_TIMEZONE);
        final String hourStr = statesStartTimeStr.substring(0, 2);
        final int stateStartHour = Integer.valueOf(hourStr);

        // APICont model State must starts at previous 12Z

        if(this.getStartHourOfDay() != stateStartHour)
        {
            throw new ModelException("APICONT model: Start time of Day is not the same as Model state Start time (update state is not at 12z)");
        }

    } //close method

    /*
     * The following method is used to create a APICONT.txt mod file each value in the inputModList time series is
     * converted into a line in the text file For example: AEICQN_UNKNOWN_817416=3.4. In the code below, numIdentifiers
     * is the Number of fields separated by underscore not including the mod name and date as julian hr in the above
     * line numIdentifiers is 1 which is "UNKNOWN" and it does not include AEICQN and 817416
     */
    @Override
    public String getModsStringFromTS(final List<RegularTimeSeries> inputModsList) throws Exception
    {
        final int numIdentifiers = 1;
        final long effDate = new DateTime(_runInfo.getRunLastObservationTimeLong(), getStartHourOfDay()).convertToJulianHours();

        final String header = numIdentifiers + " " + Long.toString(effDate) + "\n";

        final List<ApicontModEntry> allApicontMods = new ArrayList<ApicontModEntry>();

        for(final RegularTimeSeries ts: inputModsList)
        {
            createEntriesInModFileForOneTimeseriesType(allApicontMods, ts);
        }

        Collections.sort(allApicontMods);

        String modsString = header;

        for(int i = 0; i < allApicontMods.size(); i++)
        {
            modsString = modsString + allApicontMods.get(i).getModLine();
        }

        return modsString;

    }

    private void createEntriesInModFileForOneTimeseriesType(final List<ApicontModEntry> myStringsCollection,
                                                            final RegularTimeSeries ts) throws Exception
    {
        String oneModLine;
        String idkey;
        String modName;
        Double modValue;
        long time;
        DateTime sdt;

        int dateAsInteger;
        ApicontModEntry modEntry;
        final int measureMentCount = ts.getMeasurementCount();
        final MeasuringUnit regulationUnit = ts.getMeasuringUnit();

        final DecimalFormat df = new DecimalFormat("0.0####");

        modName = ts.getTimeSeriesType();
        idkey = "UNKNOWN";

        if(!modName.equals("AEICQN") && !modName.equals("AIADJ") && !modName.equals("APICBASF"))
        {
            modName = "APICCO";
            idkey = ts.getTimeSeriesType();
        }

        for(int index = 0; index < measureMentCount; index++)
        {
            oneModLine = modName + "_" + idkey;

            modValue = ts.getMeasurementValueByIndex(index, regulationUnit);

            if(modValue == ts.getMissingValue())
            {
                continue;
            }

            // convert unit in metric (MM or DEGC) to English (INCHES or DEGF) 
            // because model is used units in ENGLISH (Inches and DEGC) for 
            // the AEICQN, AIADJ, APICBASF and APICCO MODs

            if(modName.equals("APICCO") || modName.equals("AEICQN") || modName.equals("AIADJ"))
            {

                if(regulationUnit.equals(MeasuringUnit.mm))
                {
                    ts.convert(MeasuringUnit.inches);
                    modValue = ts.getMeasurementValueByIndex(index, ts.getMeasuringUnit());
                }
                else if(regulationUnit.equals(MeasuringUnit.degreesCelsius))
                {
                    ts.convert(MeasuringUnit.degreesFahrenheit);
                    modValue = ts.getMeasurementValueByIndex(index, ts.getMeasuringUnit());
                }

            }

            final AbsTimeMeasurement absTime = ts.getAbsTimeMeasurementByIndex(index);
            time = absTime.getTime();

            sdt = new DateTime(time, getStartHourOfDay());

            dateAsInteger = sdt.convertToJulianHours();

            oneModLine = oneModLine + "_" + dateAsInteger + "=" + df.format(modValue) + "\n";

            //System.out.printf("%s", oneModLine);

            modEntry = new ApicontModEntry(oneModLine, dateAsInteger);

            myStringsCollection.add(modEntry);
        }

    }

    @Override
    public APIContModelParameters getParameters()
    {
        return (APIContModelParameters)_parameters;
    }

    @Override
    public APIContModelParameters getPreviousParameters()
    {
        return (APIContModelParameters)_savedParameters;
    }

    @Override
    public APIContModelState getState()
    {
        return (APIContModelState)_state;
    }
}
