package ohd.hseb.ohdmodels.baseflow;

import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import javax.management.timer.Timer;

import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.MeasuringUnitType;
import ohd.hseb.measurement.RegularTimeSeries;
// import ohd.hseb.ohdmodels.chanloss.ChanlossJavaModelParameters;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.ohdmodels.ModelDriver;

final public class BaseflowJavaModelDriver extends ModelDriver
{

    public static final String BASEFLOW_MODEL_TIME_INTERVAL = "baseflowModelTimeInterval";

    private final BaseflowJavaModelParameters _baseFlowModelParameters; // alias to super._parameters to save type
                                                                        // casting

    private final BaseflowJavaModelParameters _baseFlowModelSavedParameters; // alias to super._savedParameters to save
                                                                             // type casting

    private final BaseflowJavaModelState _baseFlowModelState;

    private int _timeInterval; // set based on run_info.xml, during runModelDriverValidation

    public BaseflowJavaModelDriver()
    {
        super(); // calling ModelDriver constructor

        _state = new BaseflowJavaModelState();
        _baseFlowModelState = (BaseflowJavaModelState)super.getState();

        _parameters = new BaseflowJavaModelParameters();
        _baseFlowModelParameters = (BaseflowJavaModelParameters)super.getParameters(); // use alias to save casting

        _savedParameters = new BaseflowJavaModelParameters();
        _baseFlowModelSavedParameters = (BaseflowJavaModelParameters)super.getPreviousParameters(); // use alias to
                                                                                                    // save casting

    }

    @Override
    public BaseflowJavaModelParameters getParameters()
    {
        return (BaseflowJavaModelParameters)_parameters;
    }

    @Override
    public BaseflowJavaModelState getState()
    {
        return (BaseflowJavaModelState)_state;
    }

    @Override
    public BaseflowJavaModelParameters getPreviousParameters()
    {
        return (BaseflowJavaModelParameters)_savedParameters;
    }

    /**
     * Override (@link ModelDriver#runInputTsValidation}. The trimming at the start needs to be done at
     * _modelRunInfo.getRunStartTimeLong() + getDrivingTsIntervalInMillis(), not getComputationStartTime(). In addition,
     * there is at most only one input TS, -- could be none, no need to check in sync.
     */
    @Override
    protected void runInputTsValidation() throws Exception
    {
        if(_drivingTs != null) // BASEFLOW could have none input TSs
        {
            _drivingTs.trimTimeSeriesAtStartWithCheck(_runInfo.getRunStartTimeLong() + getDrivingTsIntervalInMillis());
            _drivingTs.trimTimeSeriesAtEndWithCheck(getComputationEndTime());

            /* ----------------validate the RTS in inputs.xml has enough data------------ */
            for(final RegularTimeSeries rts: _tsList)
            {
                // check if has enough data
                rts.checkHasEnoughData(getInitialStateTime(), getComputationEndTime());
                _logger.log(Logger.DEBUG, "Input time series " + rts.getTimeSeriesType() + " has enough data.");
            }
        }
    }

    /**
     * Override {@link ModelDriver#getComputationStartTime()} which uses _drivingTs interval. Here, use the property
     * "baseflowModelTimeInterval" in run_info.xml, because BASEFLOW could have none input TSs.
     */
    @Override
    public long getComputationStartTime()
    {
        return _runInfo.getRunStartTimeLong() + (_timeInterval * Timer.ONE_HOUR);
    }

    @Override
    public void execute() throws Exception
    {
        // if input Time series is present
        try
        {
            runModelDriverValidation();
            addTStoResultMap(this.computeBaseFlow());
            if(_drivingTs != null)
            {
                addTStoResultMap(_drivingTs);
            }
        }
        catch(final Exception e)
        {

            final Exception x = e;

            throw x;

        }

        getState().setDateTime(getComputationEndTime());

    }

    /**
     * Actually does more than validation: assign individual input TS to a reference with more meaningful name; Check if
     * the needed TS(s) are present; validate snow17Parameters and snow17ModelState.
     */
    @Override
    final protected void runModelDriverValidation() throws Exception
    {
        final Properties modelProperties = getDriverProperties();
        _timeInterval = Integer.valueOf(modelProperties.getProperty(BaseflowModelDriver.BASEFLOW_MODEL_TIME_INTERVAL));

        // only timeseries containing DLES data should be considered as driving time
        // series for this model
        for(final RegularTimeSeries timeSeries: _tsList)
        {
            if(timeSeries.getMeasuringUnit().getType() == MeasuringUnitType.unitless)
            {
                _drivingTs = timeSeries;
            }
        } // close for loop

        _baseFlowModelParameters.validateParams();

        if(super.needCarryoverTransfer())
        {
            _baseFlowModelState.validateState();
            _baseFlowModelState.doCarryOverTransfer(_baseFlowModelSavedParameters, _baseFlowModelParameters);
        }
        else
        {
            _baseFlowModelState.validateState();
        }

        super.runModelDriverValidation();
    } // close method

    /**
     * BaseFlow model is special regarding to set _outputLocationId in ModelDriver.java: it may not have any input TS.
     * And the properties of "outputLocationId" may not exist in run_info.xml. Get the value from params.xml "TS_ID".
     * 
     * @throws Exception
     */
    @Override
    protected void setOutputLocationId() throws Exception
    {
        if(_drivingTs != null)
        {
            _outputLocationId = _drivingTs.getLocationId();
        }
        else
        {
            if(getParameters().isParamExisting(OHDConstants.TS_ID) == false)
            {
                _outputLocationId = null;
            }
            else
            {// the parameter exists
                _outputLocationId = getParameters().getStringDataParameter(OHDConstants.TS_ID);
            }
        }

        if(getDriverProperties().containsKey(OHDConstants.OUTPUT_LOCATION_ID))
        {
            _outputLocationId = getDriverProperties().getProperty(OHDConstants.OUTPUT_LOCATION_ID);
        }
    }

    /**
     * @return
     */
    private RegularTimeSeries computeBaseFlow()
    {
        // set this values and we don't need to retrieve several times in loop
        final long drivingTsIntervalInMillis = _timeInterval * Timer.ONE_HOUR;
        final long computationStartTime = this.getComputationStartTime();
        final long computationEndTime = _runInfo.getRunEndTimeLong();// getComputationEndTime();
        final int computationTimeStepInterval = this._timeInterval;

        DateTime helpDate = null;
        int hour = 0;
        if(_drivingTs != null && !_drivingTs.isEmpty())
        {
            helpDate = new DateTime(super.getComputationStartTime());
            helpDate.setStartOfDay(0);
            hour = new DateTime(super.getComputationStartTime()).getNwsrfsHour()
                + super.getRunInfo().getTimeZoneRawOffsetInHours();
        }

        // Variable baseflow option indicator
        final String varBaseFlowOptionIndicator = getParameters().getVariableOptionIndicator();

        double baseflowIn = 0.0;
        double baseflowOut = 0.0;
        double recessionCoefficient = 0;

        // Output TS
        final RegularTimeSeries baseFlowOutputTs = new RegularTimeSeries(computationStartTime,
                                                                         computationEndTime,
                                                                         computationTimeStepInterval,
                                                                         // MeasuringUnit.percentDecimal);
                                                                         OHDConstants.DISCHARGE_UNIT);
        baseFlowOutputTs.setTimeSeriesType(getParameters().getTimeSerieDataType());
        baseFlowOutputTs.setLocationId(getParameters().getTimeSeriesId());

        final List<String> qualifiersIds = new ArrayList<String>();
        qualifiersIds.add(getParameters().getTimeSeriesId());// super.getDrivingRTS().getLocationId());
        baseFlowOutputTs.setQualifierIds(qualifiersIds);

        final double constantBaseFlow = this.getParameters().getConstantAmount();
        double previousBaseFlow = this.getState().getPreviousBaseflow();

        final double timeSerieInterval = this.getParameters().getTimeSeriesInterval() / 24;
        double timeIntervalRecession = 0;

        if(!varBaseFlowOptionIndicator.equals(BaseflowJavaModelParameters.NO_VARIABLE_BASEFLOW))
        {
            previousBaseFlow = previousBaseFlow - constantBaseFlow;
            if(previousBaseFlow < 0.0)
            {
                previousBaseFlow = 0.0;
            }
        }

        if(varBaseFlowOptionIndicator.equals(BaseflowJavaModelParameters.RECESSION_COEFF_SUPLIED_AS_TIMESERIE))
            // Get initial recession coefficient
            recessionCoefficient = this.getState().getBaseflowRecessionCoefficient();//
        else if(varBaseFlowOptionIndicator.equals(BaseflowJavaModelParameters.SINGLE_BASEFLOW_RECESSION_COEFF))
            // Get initial recession coefficient
            recessionCoefficient = this.getState().getBaseflowRecessionCoefficient();
        //recessionCoefficient = this.getParameters().getPreviousRecessionCoefficient();

        for(long timeStep = computationStartTime; timeStep <= computationEndTime; timeStep += drivingTsIntervalInMillis)
        {

            helpDate = new DateTime(timeStep);

            if(_drivingTs != null && !_drivingTs.isEmpty())
            {
                final int myDate = helpDate.getNwsrfsHour() + super.getRunInfo().getTimeZoneRawOffsetInHours();
                if(super.getComputationStartTime() >= timeStep && myDate == hour)
                {
                    baseflowIn = _drivingTs.getMeasurementValueByTime(timeStep, MeasuringUnit.percentDecimal);
                    recessionCoefficient = baseflowIn;
                }
            }

            baseflowOut = 0.0;

            if(!varBaseFlowOptionIndicator.isEmpty())
            {
                // IS IOPT 1 - Constant baseflow recession coefficient; 2 - recession
                // coefficient supplied as time series
                if(varBaseFlowOptionIndicator.equalsIgnoreCase(BaseflowJavaModelParameters.SINGLE_BASEFLOW_RECESSION_COEFF)
                    || varBaseFlowOptionIndicator.equalsIgnoreCase(BaseflowJavaModelParameters.RECESSION_COEFF_SUPLIED_AS_TIMESERIE))
                {
                    // Compute time interval recession
                    timeIntervalRecession = Math.pow(recessionCoefficient, timeSerieInterval);
                    previousBaseFlow = previousBaseFlow * timeIntervalRecession;
                }

                baseflowOut = constantBaseFlow + previousBaseFlow;
                baseFlowOutputTs.setMeasurementByTime(baseflowOut, timeStep);

            }
        }

        if(varBaseFlowOptionIndicator.equals(BaseflowJavaModelParameters.SINGLE_BASEFLOW_RECESSION_COEFF)
            || varBaseFlowOptionIndicator.equals(BaseflowJavaModelParameters.RECESSION_COEFF_SUPLIED_AS_TIMESERIE))
        {
            // set the carryover value - END OF RUN PERIOD--UPDATE CO IF REQUESTED
            this.getState().setPreviousBaseflow(baseflowOut);
        }
        if(varBaseFlowOptionIndicator.equals(BaseflowJavaModelParameters.RECESSION_COEFF_SUPLIED_AS_TIMESERIE))
        {
            this.getState().setBaseflowRecessionCoefficient(recessionCoefficient);
        }
        // update the states values if needed.
        this.getState().setCarryOverValuesToMap();

        return baseFlowOutputTs;

    }
}