package ohd.hseb.ohdmodels.baseflow;

import java.util.List;
import java.util.Properties;

import javax.management.timer.Timer;

import ohd.hseb.measurement.MeasuringUnitType;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.util.Logger;
import ohd.hseb.util.SortedProperties;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.State;
import ohd.hseb.util.fews.ohdmodels.LegacyModelAdapter;
import ohd.hseb.util.fews.ohdmodels.ModelDriver;

final public class BaseflowModelDriver extends ModelDriver
{

    private static final String BASEFLOW_EXE_NAME = "baseflow";
    public static final String BASEFLOW_MODEL_TIME_INTERVAL = "baseflowModelTimeInterval";

    private int _timeInterval; //set based on run_info.xml, during runModelDriverValidation

    public BaseflowModelDriver()
    {
        super(); //calling ModelDriver constructor

        _state = new BaseflowModelState();
        _parameters = new BaseflowModelParameters();
        _savedParameters = new BaseflowModelParameters();

    }

    @Override
    public BaseflowModelParameters getParameters()
    {
        return (BaseflowModelParameters)_parameters;
    }

    @Override
    public BaseflowModelState getState()
    {
        return (BaseflowModelState)_state;
    }

    @Override
    public BaseflowModelParameters getPreviousParameters()
    {
        return (BaseflowModelParameters)_savedParameters;
    }

    /**
     * Override (@link ModelDriver#runInputTsValidation}. The trimming at the start needs to be done at
     * _modelRunInfo.getRunStartTimeLong() + getDrivingTsIntervalInMillis(), not getComputationStartTime(). In addition,
     * there is at most only one input TS, -- could be none, no need to check in sync.
     */
    @Override
    protected void runInputTsValidation() throws Exception
    {
        if(_drivingTs != null) //BASEFLOW could have none input TSs
        {
            _drivingTs.trimTimeSeriesAtStartWithCheck(_runInfo.getRunStartTimeLong() + getDrivingTsIntervalInMillis());
            _drivingTs.trimTimeSeriesAtEndWithCheck(getComputationEndTime());

            /* ----------------validate the RTS in inputs.xml has enough data------------ */
            for(final RegularTimeSeries rts: _tsList)
            {
                //check if has enough data
                rts.checkHasEnoughData(getInitialStateTime(), getComputationEndTime());
                _logger.log(Logger.DEBUG, "Input time series " + rts.getTimeSeriesType() + " has enough data.");
            }
        }
    }

    /**
     * Override {@link ModelDriver#getComputationStartTime()} which uses _drivingTs interval. Here, use the property
     * "baseflowModelTimeInterval" in run_info.xml, because BASEFLOW could have none input TSs.
     */
    @Override
    public long getComputationStartTime()
    {
        return _runInfo.getRunStartTimeLong() + (_timeInterval * Timer.ONE_HOUR);
    }

    @Override
    public void execute() throws Exception
    {

        //Send in work dir and state dir
        final LegacyModelAdapter baseflowAdapter = new LegacyModelAdapter(this, getWorkDir(), _logger);

        baseflowAdapter.writeStandAloneFiles();

        final SortedProperties properties = new SortedProperties();
        properties.put(BaseflowModelDriver.BASEFLOW_MODEL_TIME_INTERVAL, String.valueOf(_timeInterval));

        //Clear out state Map before storing the new data (bug 244)
        ((State)_state).getStateMap().clear();

        //if there are any model specific arguments to be sent, 
        //fill in a String [] array and send to executeModel as last argument
        baseflowAdapter.executeModel(getDriverProperties().getProperty(OHDConstants.LEGACY_LOCATION_DIR) + "/"
            + BASEFLOW_EXE_NAME, properties);

        //load the results
        final List<RegularTimeSeries> resultsList = baseflowAdapter.readOutputFlatFilesToLists();

        for(int i = 0; i < resultsList.size(); i++)
        {
            addTStoResultMap(resultsList.get(i));
        }

        getState().loadState(baseflowAdapter.getOutputStateFileName(), _logger);
        //getState().setDateTime(_drivingTs.getEndTime()); // chg to read end time from model run info for all models
        getState().setDateTime(_runInfo.getRunEndTimeLong());

    }

    /**
     * Actually does more than validation: assign individual input TS to a reference with more meaningful name; Check if
     * the needed TS(s) are present; validate snow17Parameters and snow17ModelState.
     */
    @Override
    final protected void runModelDriverValidation() throws Exception
    {
        final Properties modelProperties = getDriverProperties();
        _timeInterval = Integer.valueOf(modelProperties.getProperty(BaseflowModelDriver.BASEFLOW_MODEL_TIME_INTERVAL));

        // only timeseries containing DLES data should be considered as driving time series for this model
        for(final RegularTimeSeries timeSeries: _tsList)
        {
            if(timeSeries.getMeasuringUnit().getType() == MeasuringUnitType.unitless)
            {
                _drivingTs = timeSeries;
            }
        } //close for loop

        super.runModelDriverValidation();
    } //close method

    /**
     * BaseFlow model is special regarding to set _outputLocationId in ModelDriver.java: it may not have any input TS.
     * And the properties of "outputLocationId" may not exist in run_info.xml. Get the value from params.xml "TS_ID".
     * 
     * @throws Exception
     */
    @Override
    protected void setOutputLocationId() throws Exception
    {
        if(_drivingTs != null)
        {
            _outputLocationId = _drivingTs.getLocationId();
        }
        else
        {
            if(getParameters().isParamExisting(OHDConstants.TS_ID) == false)
            {
                _outputLocationId = null;
            }
            else
            {//the parameter exists
                _outputLocationId = getParameters().getStringDataParameter(OHDConstants.TS_ID);
            }
        }

        if(getDriverProperties().containsKey(OHDConstants.OUTPUT_LOCATION_ID))
        {
            _outputLocationId = getDriverProperties().getProperty(OHDConstants.OUTPUT_LOCATION_ID);
        }
    }
}
