package ohd.hseb.ohdmodels.chanloss;

import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.ohdmodels.ModelDriver;

final public class ChanlossJavaModelDriver extends ModelDriver
{

    private final ChanlossJavaModelParameters _ChanlossModelParameters; //alias to super._parameters to save type casting

    private final ChanlossJavaModelParameters _chanLossModelSavedParameters; //alias to super._savedParameters to save type casting

    private final ChanlossJavaModelState _chanLossModelState; //alias to super._state to save type casting    

    private RegularTimeSeries _peTs = null; //optional -  Potential Evaporation (PE)- input time series.
    private int _peTime = 0;
    private int _outputTSTime = 0;
    // This table represents the fixed diurnal variation for evaporation, 
    // expressed as percent/100 of daily evaporation, that is applied each hour.
    static final double DAILYDISTRIBUTIONFACTOR[] = {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.02, 0.05, 0.10, 0.16,
        0.20, 0.18, 0.14, 0.09, 0.05, 0.01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    
    public static final String DAY_OF_MONTH_VALUE = "DAY_OF_MONTH";
    public static final int DEFAULT_DAY_OF_MONTH = 16;

    public ChanlossJavaModelDriver()
    {
        super(); //calling ModelDriver constructor       

        super._parameters = new ChanlossJavaModelParameters();
        _ChanlossModelParameters = (ChanlossJavaModelParameters)super.getParameters(); //use alias to save casting

        super._savedParameters = new ChanlossJavaModelParameters();
        _chanLossModelSavedParameters = (ChanlossJavaModelParameters)super._savedParameters; //use alias to save casting

        super._state = new ChanlossJavaModelState();
        _chanLossModelState = (ChanlossJavaModelState)super._state; //use alias to save casting    
    }

    @Override
    public ChanlossJavaModelParameters getParameters()
    {
        return (ChanlossJavaModelParameters)_parameters;
    }

    @Override
    public ChanlossJavaModelState getState()
    {
        return (ChanlossJavaModelState)_state;
    }

    @Override
    public ChanlossJavaModelParameters getPreviousParameters()
    {
        return (ChanlossJavaModelParameters)_savedParameters;
    }

    @Override
    public void execute() throws Exception
    {

        // all input timeseries have the same time step. Any timeseries is a driver.
        _drivingTs = _tsList.get(0);

        if(_drivingTs == null)
        {
            _logger.log(Logger.FATAL, "Driving Time Series is null.Exiting...");

            throw new Exception();
        }
        // if PE Time series is present
        if(_tsList.size() > 1)
        {
            _peTs = _tsList.get(1);
            _ChanlossModelParameters.setPeTimeSeriesAvailability(true);
            final DateTime peDateTime = new DateTime(_peTs.getStartTime());
            _peTime = peDateTime.hour();
            final DateTime outputDateTime = new DateTime(getComputationStartTime());
            _outputTSTime = outputDateTime.hour();
        }

        _logger.log(Logger.DEBUG,
                    "Driving Time Series set to " + _drivingTs.getLocationId() + " " + _drivingTs.getTimeSeriesType()
                        + " " + _drivingTs.getIntervalInHours());

        try
        {
            runLocalModelDriverValidation();
            addTStoResultMap(this.computeChanLoss());
        }
        catch(final Exception e)
        {

            final Exception x = e;

            throw x;

        }

        getState().setDateTime(getComputationEndTime());

    }

    /**
     * This Operation accounts for losses or gains of water that occur along a channel reach as a result of flow through
     * the channel bottom and evaporation from the stream surface. Even though channel losses are actually distributed
     * along the length of the reach the Operation adjusts instantaneous discharges at a flow point for such losses.
     * Parameter SSOUT specifies the sub-surface loss and is defined as the rate in CMS of sub-surface outflow along the
     * stream channel. A positive SSOUT value indicates a loss of water from the stream and a negative value indicates a
     * gain of water by the stream. SSOUT can be constant throughout the year or it can vary from month to month.
     * Constant SSOUT is defined with one value. Variable SSOU T is defined by twelve values at the 16th of each month.
     * When variable SSOUT is used daily values are obtained through linear interpolation between mid-month values. The
     * computational data time interval for the Operation is equal to the data time interval of the instantaneous
     * discharge data.
     * 
     * @return a TimeSeries with the compute chanloss values.
     * @throws Exception
     */
    private RegularTimeSeries computeChanLoss() throws Exception
    {

        double flowin = 0.0;
        double peFlowIn = 0.0;
        double flowOut = 0.0;

        // set this values and we don't need to retrieve several times in loop
        final long drivingTsIntervalInMillis = getDrivingTsIntervalInMillis();
        final long computationStartTime = getComputationStartTime();
        final long computationEndTime = getComputationEndTime();
        final int computationTimeStepInterval = super.getDrivingTsInterval();
        final double SSOUT = _ChanlossModelParameters.getSSOUTParameterConstant();
        final double[] ssoutValues = _ChanlossModelParameters.getSSOUTValues();
        final double[] ssoutIncrements = _ChanlossModelParameters.getSSOUTIncrements();

        final double[] peValues = _ChanlossModelParameters.getAverageEvaporationValues();
        final double[] peIncrements = _ChanlossModelParameters.getAverageEvaporationIncrements();

        final double waterSurfaceArea = _ChanlossModelParameters.getWaterSurfaceArea();

        final int numberIntervals = 24 / computationTimeStepInterval;
        final double peadj = _ChanlossModelParameters.getConstantEvaporationAdjustmentFactor();
        final String evaporationDistributionType = _ChanlossModelParameters.getDailyEvaporationDistributionType();

        final String typeOfChannelBottonLoss = _ChanlossModelParameters.getSSOUTParameterType(); //SSWCH
        DateTime currentDate = null;
        final DateTime endDate = new DateTime(computationEndTime);

        final int IDT = computationTimeStepInterval;
        double[] epdist = null;
        double carryoverPEadjust = 0.0;
        // Get the defined day of the month value from the parameter value, if not defined in the parameters file it will use the 
        // default value of 16th of the month.
        final int DAY_OF_MONTH_VALUE = _ChanlossModelParameters.getDayOfMonth();

        // Output TS
        final RegularTimeSeries chanLossOutputTs = new RegularTimeSeries(computationStartTime,
                                                                         computationEndTime,
                                                                         computationTimeStepInterval,
                                                                         OHDConstants.DISCHARGE_UNIT);
        chanLossOutputTs.setTimeSeriesType(super.getDrivingRTS().getTimeSeriesType());
        chanLossOutputTs.setLocationId(_drivingTs.getLocationId());

        final List<String> qualifiersIds = new ArrayList<String>();
        qualifiersIds.add(super.getDrivingRTS().getLocationId());
        chanLossOutputTs.setQualifierIds(qualifiersIds);

        DateTime helpDate = null;
        // determine daily evap-distribution
        if(waterSurfaceArea > 0.0)
        {
            epdist = this.computeDailyEvaporationDistribution(evaporationDistributionType,
                                                              numberIntervals,
                                                              computationTimeStepInterval);
        }

        for(long timeStep = computationStartTime; timeStep <= computationEndTime; timeStep += drivingTsIntervalInMillis)
        {
            flowin = _drivingTs.getMeasurementValueByTime(timeStep, OHDConstants.DISCHARGE_UNIT);
            flowOut = 0.0;

            currentDate = new DateTime(timeStep);
            helpDate = new DateTime(timeStep);
            helpDate.setStartOfDay(0);

            if(!typeOfChannelBottonLoss.isEmpty())

                // IS SSOUT CONSTANT, A FIXED PERCENTAGE, VARIABLE OR A VARIABLE % ?
                if(typeOfChannelBottonLoss.equalsIgnoreCase(ChanlossJavaModelParameters.VARIABLE_SSOUT_CMS)
                    || typeOfChannelBottonLoss.equalsIgnoreCase(ChanlossJavaModelParameters.VARIABLE_SSOUT_PERCENTAGE))
                {

                    int dayOfMonth = currentDate.dayOfMonth() - DAY_OF_MONTH_VALUE;

                    // Set the help date "myDate" to use NWSRFC time format + Time Zone to get correct time for iteration. 
                    int myDate = helpDate.getNwsrfsHour() + super.getRunInfo().getTimeZoneRawOffsetInHours();
                    if(myDate == 0)
                    {
                        myDate = 24;
                    }
                    if(helpDate.getNwsrfsHour() == 24 || myDate == 24)
                    {
                        dayOfMonth = dayOfMonth - 1;
                    }
                    int monthCurrent = currentDate.month() - 1;
                    if(monthCurrent == 0)
                    {
                        monthCurrent = 12;
                    }
                    final int monthValues = currentDate.month();
                    int monthIncrements = monthCurrent;
                    if(currentDate.dayOfMonth() > DAY_OF_MONTH_VALUE)
                    {
                        monthIncrements = currentDate.month();
                    }

                    final double ss = ssoutValues[monthValues - 1]
                        + (ssoutIncrements[monthIncrements - 1] * dayOfMonth);
                    if(typeOfChannelBottonLoss.equalsIgnoreCase(ChanlossJavaModelParameters.VARIABLE_SSOUT_CMS))
                    {
                        flowOut = flowin - ss;
                    }
                    if(typeOfChannelBottonLoss.equalsIgnoreCase(ChanlossJavaModelParameters.VARIABLE_SSOUT_PERCENTAGE))
                    {
                        flowOut = flowin - (ss * flowin);
                    }
                    if(flowOut < 0.0)
                        flowOut = 0.0;
                }
                // SSOUT is a fixed percentage
                else if(typeOfChannelBottonLoss.equalsIgnoreCase(ChanlossJavaModelParameters.CONSTANT_SSOUT_PERCENTAGE))
                {
                    flowOut = flowin - (flowin * SSOUT);
                }
                // SSOUT is a constant
                // NOTE: A negative SSOUT indicates a gain instead of a loss.
                else
                {
                    flowOut = flowin - SSOUT;
                }

            if(waterSurfaceArea > 0.0)
            {
                // Compute EVAP-DEMAND/PE-ADJ curve vales for current day
                int day = currentDate.dayOfMonth() - DAY_OF_MONTH_VALUE;

                // Set the help date "myDate" to use NWSRFC time format + Time Zone to get correct time for iteration. 
                int myDate = helpDate.getNwsrfsHour() + super.getRunInfo().getTimeZoneRawOffsetInHours();
                if(myDate == 0)
                {
                    myDate = 24;
                }
                if(helpDate.getNwsrfsHour() == 24 || myDate == 24)
                {
                    day = day - 1;
                }

                int month = currentDate.month() - 1;
                if(month == 0)
                {
                    month = 12;
                }
                final int monthValues = currentDate.month();
                int monthIncrement = month;
                if(currentDate.dayOfMonth() > DAY_OF_MONTH_VALUE)
                    monthIncrement = currentDate.month();

                final double evaporation = peValues[monthValues - 1] + (peIncrements[monthIncrement - 1] * day);

                // compute actual EVAP-DEMAND for period (MM/IDT)
                final int period = (currentDate.hour() / computationTimeStepInterval);
                double ET = 0.0;
                if(_ChanlossModelParameters.getPeTimeSeriesAvailability())
                {
                    // PE data is used                   
                    carryoverPEadjust = _chanLossModelState.getLastPeValue() * peadj;

                    if(computationStartTime > computationEndTime)
                    {
                        if(endDate.hour() != 0)
                        {
                            ET = carryoverPEadjust * evaporation * epdist[period];
                        }
                    }
                    else
                    {

                        day = currentDate.dayOfMonth();

                        if(currentDate.hour() == _outputTSTime)
                        {
                            day = day + 1;
                        }
                        final DateTime peDateTime = new DateTime(currentDate.year(), currentDate.month(), day, _peTime);

                        peFlowIn = _peTs.getMeasurementValueByTime(peDateTime.getTimeInMillis(),
                                                                   OHDConstants.RUNOFF_UNIT);

                        ET = peFlowIn * peadj * evaporation * epdist[period];

                    }
                }
                else
                {
                    // No PE data
                    ET = peadj * evaporation * epdist[period];
                }
                // convert ET to instantaneous losss rate in cms.
                ET = (0.27778 / IDT) * ET * waterSurfaceArea;

                flowOut = flowOut - ET;

            }
            if(flowOut < 0.0)
            {
                flowOut = 0.0;
            }
            //save the value to output ts
            chanLossOutputTs.setMeasurementByTime(flowOut, timeStep);
        }
        /** --------------------- prepare carryover for storing-------------------- */
        if(_ChanlossModelParameters.getPeTimeSeriesAvailability())
        {
            _chanLossModelState.setCarryOverValuesToMap(_peTs.getMeasurementValueByTime(_peTs.getEndTime(),
                                                                                        OHDConstants.RUNOFF_UNIT));
//            _ChanlossModelParameters.setPeCarryOverValue(_peTs.getMeasurementValueByTime(_peTs.getEndTime(),
//                                                                                         OHDConstants.RUNOFF_UNIT));
        }

        return chanLossOutputTs;
    }

    /**
     * The second step in the loss Operation is to account for evaporation from the water surface area (Aw) of the
     * stream. This step is performed only when Aw in KM2 is assigned a value greater than 0.0 . Daily evaporation data
     * can be obtained from either of two sources: (1) a computed day-by-day potential evaporation (PE) record can be
     * generated from daily data, synoptic data or pan data (2) long-term average evaporation (Ea) curve
     * 
     * @param evaporationDistributionType
     * @param numberIntervals
     * @return
     */
    private double[] computeDailyEvaporationDistribution(final String evaporationDistributionType,
                                                         final int numberIntervals,
                                                         final int timeStepInterval)
    {
        final double[] epdist = new double[numberIntervals];
        // diurnal daily evaporation distribution
        if(evaporationDistributionType.equalsIgnoreCase(ChanlossJavaModelParameters.DIURNAL_EVAPORATION_DISTRIBUTION))
        {
            for(int i = 0; i < numberIntervals; i++)
            {
                final int i2 = (i + 1) * timeStepInterval;
                final int i1 = i * timeStepInterval;
                epdist[i] = 0.0;
                for(int j = i1; j < i2; j++)
                {
                    epdist[i] = epdist[i] + DAILYDISTRIBUTIONFACTOR[j];
                }
            }
        }
        else
        // even evaporation distribution
        {
            final double v = (1.0 / numberIntervals);
            for(int i = 0; i < numberIntervals; i++)
            {
                epdist[i] = v;
            }
        }
        return epdist;
    }

    /**
     * @throws Exception
     */
    private void runLocalModelDriverValidation() throws Exception
    {

        //validate chanLossModelParamters and chanLossModelState
        _ChanlossModelParameters.validateParams();

        if(super.needCarryoverTransfer())
        {
            _chanLossModelState.validateState(_chanLossModelSavedParameters);
            _chanLossModelState.doCarryOverTransfer(_chanLossModelSavedParameters, _ChanlossModelParameters);
        }
        else
        {
            _chanLossModelState.validateState(_ChanlossModelParameters);
        }

        super.runModelDriverValidation();
    }
    
    /**
     * Get the defined day of the month value from the run_info property value, if not defined it will use the 
     * default value of 16th of the month.
     * @return the day of the month read from the run_info.xml if defined, otherwise, the default value of 16.
     * @throws Exception
     */
    private int getDayOfMonthValue() throws Exception
    {
    	int dailyValue = DEFAULT_DAY_OF_MONTH;
    	Properties properties = super.getRunInfo().getProperties();

            if(properties.containsKey(DAY_OF_MONTH_VALUE))
            {
            	dailyValue = Integer.parseInt(properties.getProperty(DAY_OF_MONTH_VALUE));
            }
        
    	return dailyValue;
    }

}
