package ohd.hseb.ohdmodels.glacier;

import java.util.ArrayList;
import java.util.List;

import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.MeasuringUnitType;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.ohdmodels.ModelDriver;

final public class GlacierModelDriver extends ModelDriver
{

//    private static final String GLACIER_EXE_NAME = "glacier";

    private final GlacierModelParameters _glacierModelParameters; //alias to super._parameters to save type casting

    private final GlacierModelParameters _glacierModelSavedParameters; //alias to super._savedParameters to save type casting

    private final GlacierModelState _glacierModelState; //alias to super._state to save type casting

    public GlacierModelDriver()
    {
        super(); //calling ModelDriver constructor

        super._parameters = new GlacierModelParameters();
        _glacierModelParameters = (GlacierModelParameters)super.getParameters(); //use alias to save casting

        super._savedParameters = new GlacierModelParameters();
        _glacierModelSavedParameters = (GlacierModelParameters)super._savedParameters; //use alias to save casting

        super._state = new GlacierModelState();
        _glacierModelState = (GlacierModelState)super._state; //use alias to save casting     

    }

    @Override
    public GlacierModelParameters getParameters()
    {
        return (GlacierModelParameters)_parameters;
    }

    @Override
    public GlacierModelState getState()
    {
        return (GlacierModelState)_state;
    }

    @Override
    public GlacierModelParameters getPreviousParameters()
    {
        return (GlacierModelParameters)_savedParameters;
    }

    @Override
    public void execute() throws Exception
    {

        // only timeseries containing L3/T data should be considered as driving time series for this model
        for(final RegularTimeSeries timeSeries: _tsList)
        {

            if(timeSeries.getMeasuringUnit().getType() == MeasuringUnitType.length)
            {
                _drivingTs = timeSeries;
                _logger.log(Logger.DEBUG,
                            "Driving Time Series set to " + _drivingTs.getLocationId() + " "
                                + _drivingTs.getTimeSeriesType() + " " + _drivingTs.getIntervalInHours());
            }
        }

        if(_drivingTs == null)
        {
            _logger.log(Logger.FATAL, "Driving Time Series is null.Exiting...");
            throw new Exception();
        }

        try
        {
            runLocalModelDriverValidation();
            this.computeGlacier();
        }
        catch(final Exception e)
        {
            final Exception x = e;
            throw x;
        }

        //Set state date and time
        getState().setDateTime(getComputationEndTime());

    }

    /**
     * @throws Exception
     */
    private void runLocalModelDriverValidation() throws Exception
    {

        //validate TatumModelParamters and TatumModelState
        _glacierModelParameters.validateParams(_drivingTs.getMeasuringUnit().getName());

        if(super.needCarryoverTransfer())
        {
            _glacierModelState.validateState(_glacierModelSavedParameters);
            _glacierModelState.doCarryOverTransfer(_glacierModelSavedParameters, _glacierModelParameters);
        }
        else
        {
            _glacierModelState.validateState(_glacierModelParameters);

        }

        super.runModelDriverValidation();
    }

    /**
     * 
     */
    private void computeGlacier()
    {
        // set this values and we don't need to retrieve several times in loop
        final long drivingTsIntervalInMillis = super.getDrivingTsIntervalInMillis();
        final long computationStartTime = super.getComputationStartTime();
        final long computationEndTime = super.getComputationEndTime();
        double inflow = 0.0;
        double flowout = 0.0;
        double dstore = 0.0;

        final int dataTimeInterval = _glacierModelParameters.getTimeSeriesRainMeltOutputTimeInterval();
        final double cg1Param = _glacierModelParameters.getCg1Parameter();
        final double cg2Param = _glacierModelParameters.getCg2Parameter();
        double cg3Param = _glacierModelParameters.getAfiDecayParameter();
        double kg1Param = _glacierModelParameters.getKg1Parameter();
        double kg2Param = _glacierModelParameters.getKg2Parameter();
        double storage = _glacierModelState.getInitialGlacierStorage();
        double antecedentFlowIndex = _glacierModelState.getInitialAfiDecayParameter();
        double fAntecedentFlowIndex;
        double kgParam;
        int antecedentFlow;

        if(_glacierModelParameters.getAntecedentFlowIndexId().isEmpty())
        {
            antecedentFlow = 0;
        }
        else
            antecedentFlow = 1;

        kg1Param = 1.0 - Math.pow((1.0 - kg1Param), ((double)dataTimeInterval / 24));
        kg2Param = 1.0 - Math.pow((1.0 - kg2Param), ((double)dataTimeInterval / 24));
        cg3Param = Math.pow(cg3Param, ((double)dataTimeInterval / 24));

        // Output TS
        final RegularTimeSeries glacierOutputTs = new RegularTimeSeries(computationStartTime,
                                                                        computationEndTime,
                                                                        _glacierModelParameters.getTimeSeriesRainMeltOutputTimeInterval(),
                                                                        MeasuringUnit.mm);
        glacierOutputTs.setTimeSeriesType(_glacierModelParameters.getTimeSeriesOutputDataType());
        glacierOutputTs.setLocationId(_drivingTs.getLocationId());

        final List<String> qualifiersIds = new ArrayList<String>();
        qualifiersIds.add(_glacierModelParameters.getTimeSeriesOutputId());
        glacierOutputTs.setQualifierIds(qualifiersIds);

        final RegularTimeSeries antecedentFlowTs = new RegularTimeSeries(computationStartTime,
                                                                         computationEndTime,
                                                                         _glacierModelParameters.getTimeSeriesRainMeltOutputTimeInterval(),
                                                                         MeasuringUnit.mm);
        antecedentFlowTs.setTimeSeriesType(_glacierModelParameters.getAntecedentFlowIndexDataType());
        antecedentFlowTs.setLocationId(_drivingTs.getLocationId());
        final List<String> antecedentQualifiersIds = new ArrayList<String>();
        antecedentQualifiersIds.add(_glacierModelParameters.getAntecedentFlowIndexId());
        antecedentFlowTs.setQualifierIds(antecedentQualifiersIds);

        double num;

        for(long timeStep = computationStartTime; timeStep <= computationEndTime; timeStep += drivingTsIntervalInMillis)
        {
            inflow = _drivingTs.getMeasurementValueByTime(timeStep, MeasuringUnit.mm);
            antecedentFlowIndex = cg3Param * antecedentFlowIndex + inflow;
            num = Math.exp(cg1Param + cg2Param * antecedentFlowIndex);

            fAntecedentFlowIndex = num / (1 + num);
            kgParam = kg1Param + (kg2Param - kg1Param) * fAntecedentFlowIndex;
            if(antecedentFlow == 1)
            {
                antecedentFlowTs.setMeasurementByTime(fAntecedentFlowIndex, timeStep);
            }
            flowout = kgParam * storage;

            //save the value to output ts
            glacierOutputTs.setMeasurementByTime(flowout, timeStep);
            dstore = inflow - flowout;
            storage = storage + dstore;
        }
        _glacierModelState.setInitialGlacierStorage(storage);
        _glacierModelState.setInitialAfiDecayParameter(antecedentFlowIndex);
        _glacierModelState.setCarryOverValues();

        addTStoResultMap(glacierOutputTs);
        if(antecedentFlow == 1)
        {
            addTStoResultMap(antecedentFlowTs);
        }
    }
}
