package ohd.hseb.ohdmodels.glacier;

import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.ModelException;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.Parameters;
import ohd.hseb.util.fews.ohdmodels.ModelParameters;
import ohd.hseb.util.fews.ohdmodels.ModelState;

/**
 * 
 */
public class GlacierModelState extends ModelState
{
    public final static String INITIAL_GLACIER_STORAGE_TAG = "GLACIER_STORAGE";
    public final static String INITIAL_AFI_DECAY_PARAMETER_TAG = "INITIAL_AFI_DECAY_PARAMETER";

    private String _unitType = OHDConstants.UNIT_METRIC;
    private double _initialGlacierStorage;
    private double _initialAfiDecayParameter;

    public GlacierModelState()
    {
        super();
    }

    /*
     * (non-Javadoc)
     * @see ohd.hseb.ohdmodels.ModelState#doCarryOverTransfer(ohd.hseb.ohdmodels.ModelParameters,
     * ohd.hseb.ohdmodels.ModelParameters)
     */
    @Override
    public void doCarryOverTransfer(final ModelParameters savedParams, final ModelParameters params) throws Exception
    {
        if(_logger.getPrintDebugInfo() > 0)
        {
            _logger.log(Logger.DEBUG, "GlacierModelState before carry over transfer " + this.getStateMap());
        }
        final GlacierModelParameters glacierSavedParams = (GlacierModelParameters)savedParams;
        final GlacierModelParameters glacierParams = (GlacierModelParameters)params;
        final double cg1ParamOld = glacierSavedParams.getCg1Parameter();
        final double cg2ParamOld = glacierSavedParams.getCg2Parameter();
        final double cg1ParamNew = glacierParams.getCg1Parameter();
        final double cg2ParamNew = glacierParams.getCg2Parameter();
        final double kg1Param = glacierSavedParams.getKg1Parameter();
        final double kg2Param = glacierSavedParams.getKg2Parameter();
        final double antecedentFlowIndexOld = this.getInitialAfiDecayParameter();

        final int oneN = 1;
        //Transfer carryover - adjust storage 
        //calculate kgParamOld
        int num = (int)Math.exp(cg1ParamOld + cg2ParamOld * antecedentFlowIndexOld);
        double fAntecedentFlowIndex = num / (oneN + num);
        final double kgParamOld = kg1Param + (kg2Param - kg1Param) * fAntecedentFlowIndex;

        //Calculate kgParamNew        
        num = (int)Math.exp(cg1ParamNew + cg2ParamNew * antecedentFlowIndexOld);
        fAntecedentFlowIndex = Math.abs(Math.round((double)num / (double)(oneN + num)));
        final double kgParamNew = kg1Param + (kg2Param - kg1Param) * fAntecedentFlowIndex;
        final double initGlacierStorage = (kgParamOld / kgParamNew) * this.getInitialGlacierStorage();

        this.setInitialGlacierStorage(initGlacierStorage);
        //Transfer carryover - no adjustment for InitialAfiDecayParameter
        super.setStateValue(INITIAL_GLACIER_STORAGE_TAG, initGlacierStorage);

        if(_logger.getPrintDebugInfo() > 0)
        {
            _logger.log(Logger.DEBUG, "GlacierModelState after carry over transfer " + this.getStateMap());
        }
    }

    /*
     * (non-Javadoc)
     * @see ohd.hseb.util.fews.State#extractValuesFromMap()
     */
    @Override
    public void extractValuesFromMap() throws Exception
    {
        _initialGlacierStorage = super.getDoubleDataState(INITIAL_GLACIER_STORAGE_TAG);
        ;
        _initialAfiDecayParameter = super.getDoubleDataState(INITIAL_AFI_DECAY_PARAMETER_TAG);
    }

    public void validateState(final Parameters params) throws Exception
    {

        this.setUnitType(OHDConstants.UNIT_METRIC);
        //this.extractValuesFromMap();

        if(getInitialGlacierStorage() < 0.0)
        {
            _logger.log(Logger.FATAL, "STORAGE CAN NOT BE LESS THAN ZERO");
            throw new ModelException("STORAGE CAN NOT BE LESS THAN ZERO");
        }

        if(getInitialAfiDecayParameter() < 0.0)
        {
            _logger.log(Logger.FATAL, "ERROR: ANTECEDENT FLOW INDEX CAN NOT BE LESS THAN ZERO");
            throw new ModelException("ERROR: ANTECEDENT FLOW INDEX CAN NOT BE LESS THAN ZERO");
        }
    }

    /**
     * @return the unitType
     */
    public String getUnitType()
    {
        return _unitType;
    }

    /**
     * @param unitType the unitType to set
     */
    public void setUnitType(final String unitType)
    {
        _unitType = unitType;
        super.setStateValue(OHDConstants.UNIT_TAG, unitType);
    }

    public double getInitialGlacierStorage()
    {
        return _initialGlacierStorage;
    }

    public void setInitialGlacierStorage(final double initialGlacierStorage)
    {
        this._initialGlacierStorage = initialGlacierStorage;
    }

    public double getInitialAfiDecayParameter()
    {
        return _initialAfiDecayParameter;
    }

    public void setInitialAfiDecayParameter(final double initialAfiDecayParameter)
    {
        this._initialAfiDecayParameter = initialAfiDecayParameter;
    }

    /**
     * Set the values from local variables to map in parent class
     */
    public void setCarryOverValues()
    {
        super.setStateValue(INITIAL_GLACIER_STORAGE_TAG, _initialGlacierStorage);
        super.setStateValue(INITIAL_AFI_DECAY_PARAMETER_TAG, _initialAfiDecayParameter);

    }
}