package ohd.hseb.ohdmodels.lagkj;

import java.util.ArrayList;

class LagKJCommonMethods
{

    private int _newNumPair;
    private boolean _qdt = false;
    double _quarterInterval;

    /**
     * This method linearly interpolate in the second column of a two dimensional matrix and returns the value from the
     * first column which corresponds to the location in the second column which matches a supplied value.
     * 
     * @author: ORIGINALLY PROGRAMMED BY GEORGE F. SMITH - HRL DECEMBER 1979.
     * @see FORTRAN - fserc7.f & fslag7.f
     * @param maxLag - maximum value of Lag //LY
     * @param matchValue - The value to be matched in second column of table //Q
     * @param numberPairsOfValues - The number of pairs of values in table //N
     * @param pairKQ - The table to be searched
     * @param locQT - index to start
     * @return
     */
    double linearInterpolator(final int maxLag,
                              final double matchValue,
                              final int numberPairsOfValues,
                              final double[] pairKQ,
                              final int locQT)
    {
        double val = 0.0;

        if(matchValue > pairKQ[locQT + 1])
        {
            if(numberPairsOfValues != 1)
            {
                int i = 0;
                int j1 = 1;

                if(_newNumPair > maxLag)
                    j1 = _newNumPair - maxLag; //LY - only use for NO Double Back

                for(int j = j1; j < numberPairsOfValues; j++)
                {
                    i = j;

                    final int oldp = i * 2 + locQT;

                    if(i >= numberPairsOfValues || oldp >= pairKQ.length)
                        return 0.;

                    if(matchValue <= pairKQ[oldp + 1])
                    {
                        _newNumPair = i - 1;

                        final int newp = _newNumPair * 2 + locQT;

                        val = pairKQ[newp] + (matchValue - pairKQ[newp + 1]) / (pairKQ[oldp + 1] - pairKQ[newp + 1])
                            * (pairKQ[oldp] - pairKQ[newp]);

                        return val;
                    }
                } // end j loop
            }

            val = pairKQ[(numberPairsOfValues - 1) * 2];
            _newNumPair = numberPairsOfValues;
        }
        else
        {
            val = pairKQ[locQT];
            _newNumPair = 1;
        }

        return val;
    }

    /**
     * This method computes the K values by iteration, with four possible results 1. Success within 21 iterations,
     * returning calculated outflow 2. A decision to split the interval into quarters - 1/8 of orig interval 3. If the
     * interval is already split, and the interval is still too large for the K calculated, set outflow = inflow 4.
     * Failure to converge within 21 iterations, returning the value at the 21st iteration
     * 
     * @author: ORIGINALLY PROGRAMMED BY GEORGE F. SMITH - HRL DECEMBER 1979
     * @see: FORTRAN - fcmpk7.f
     * @param begIntervalInflow - INFLOW AT BEGINNING OF INTERVAL //X1
     * @param endIntervalInflow - INFLOW AT END OF INTERVAL //X2
     * @param previousOutflow - PREVIOUS OUTFLOW //Y0
     * @param begIntervalOutflow - OUTFLOW AT BEGINNING OF INTERVAL // Y1
     * @param table - K/Q PAIRS VALUES // TAB
     * @param halfInflowInterval - HALF INFLOW TIME INTERVAL // XTA
     * @param constK - LOGICAL VARIABLE TRUE IF K ENABLED // CONK
     * @return THE OUTFLOW AT END OF INTERVAL // Y2
     */
    double computeOutflowEndingOfInterval(final double begIntervalInflow,
                                          final double endIntervalInflow,
                                          final double previousOutflow,
                                          final double begIntervalOutflow,
                                          final double[] table,
                                          final double halfInflowInterval,
                                          final boolean constK)
    {
        double valueYAtTheEndOfTimeStep = (3. * begIntervalOutflow - previousOutflow) / 2.; //ye

        double endIntervalOutflow = 0.;
        int knt = 0;
        final int tableLength = table.length / 2;
        _quarterInterval = halfInflowInterval / 4.;

        while(true)
        {
            this.setNewNumPair(0);

            final double interpolatedK = this.linearInterpolator(0, valueYAtTheEndOfTimeStep, tableLength, table, 0);//xk

            if(!((interpolatedK > _quarterInterval) || (constK == false)))
                return endIntervalInflow;

            if(interpolatedK <= _quarterInterval)
            {
                if(this.getFlagQDT() == true)
                {
                    this.setFlagQDT(true);

                    return endIntervalInflow;
                }
            }

            // Store outflow in endIntervalOutflow and check convergence with valueYAtTheEndOfTimeStep
            final double tr = 0.5 * halfInflowInterval;

            endIntervalOutflow = (tr * (begIntervalInflow + endIntervalInflow) + begIntervalOutflow
                * (interpolatedK - tr))
                / (interpolatedK + tr);

            if((1.02 * valueYAtTheEndOfTimeStep > endIntervalOutflow)
                && (0.98 * valueYAtTheEndOfTimeStep < endIntervalOutflow))
            {
                return endIntervalOutflow;
            }

            valueYAtTheEndOfTimeStep = endIntervalOutflow;

            // IF ESTIMATE OF AVG Q DOES NOT CONVERGE USE LAST GUESS AFTER TWENTY TRIES.
            if(knt > 20)
                break;

            knt++;
        }

        return endIntervalOutflow;
    }

    /**
     * THIS FUNCTION IS CALLED IF VARIABLE K IS ENABLED AND K FOUND FROM INTERPOLATION OF K,Q TABLE IS LT ROUTING
     * INTERVAL/4.0. IT TAKES THE ROUTING INTERVAL AND QUARTERS IT, CALCULATING THE OUTFLOW (Y2) AFTER LAGGING AND K. *
     * 
     * @author: ORIGINALLY PROGRAMMED BY GEORGE F. SMITH - HRL DECEMBER 1979.
     * @see: Fortran - fqdt7.f.
     * @param begIntervalInflow - INFLOW AT BEGINNING OF INTERVAL //X1
     * @param endIntervalInflow - INFLOW AT END OF INTERVAL //X2
     * @param previousOutflow - PREVIOUS OUTFLOW //Y0
     * @param begIntervalOutflow - OUTFLOW AT BEGINNING OF INTERVAL //Y1
     * @param table - K/Q PAIRS VALUES // TAB
     * @param halfInflowInterval - HALF INFLOW TIME INTERVAL // XTA
     * @param constK - LOGICAL VARIABLE TRUE IF K ENABLED // CONK
     * @return THE OUTFLOW AT END OF INTERVAL //Y2
     */
    double computeKfromQuarterRoutingInterval(final double begIntervalInflow,
                                              final double endIntervalInflow,
                                              final double previousOutflow,
                                              final double begIntervalOutflow,
                                              final boolean constK,
                                              final double halfInterval,
                                              final double[] table)
    {
        _quarterInterval = halfInterval / 4.;

        double x2t = begIntervalInflow;
        double y1t = previousOutflow * 0.25 + begIntervalOutflow * 0.75;
        double y2t = begIntervalOutflow;

        double frac, x1t, y0t;
        // LOOP THROUGH ROUTING INTERVAL ONE QUARTER PERIOD AT A TIME.
        // IF K STILL GT ROUTING INTERVAL DO NOT ATTENUATE THIS INTERVAL.
        for(int i = 0; i < 4; i++)
        {
            frac = (i + 1) / 4.;

            x1t = x2t;
            x2t = begIntervalInflow * (1. - frac) + endIntervalInflow * frac;

            y0t = y1t;
            y1t = y2t;
            y2t = this.computeOutflowEndingOfInterval(x1t, x2t, y0t, y1t, table, _quarterInterval, constK);

        }//end i loop

        return y2t;

    }

    /**
     * THIS FUNCTION PERFORMS THE FORT WORTH FLOW TRANSMISSION LOSS COMPUTATIONS.
     * 
     * @author: ORIGINALLY PROGRAMMED BY GEORGE SMITH - HRL JUNE 1989.
     * @see: Fortran - ftwtl7.f
     * @param tlrc
     * @param qbntl
     * @param currLaggedInflow
     * @param prevAttenuatedFlow
     * @param routedFlowBeforeLossComputations
     * @return the intercept values array
     */
    double performsFortWorthFlowTransmissionLossComputations(final double tlrc,
                                                             final double qbntl,
                                                             final double currLaggedInflow,
                                                             final double prevAttenuatedFlow,
                                                             final double routedFlowBeforeLossComputations)
    {

        double outedFlowIncludingLoss;

        double qtl = -999.;

        outedFlowIncludingLoss = routedFlowBeforeLossComputations;

        // SEE IF WE ARE ON RECESSION PORTION OF HYDROGRAPH
        if(currLaggedInflow < prevAttenuatedFlow)
        {
            // YES - COMPUTE FLOW INCLUDING TRANSMISSION LOSS (QTL) AND CHECK FOR RANGE TO APPLY LOSS
            qtl = prevAttenuatedFlow * tlrc;

            if(!((qtl >= routedFlowBeforeLossComputations) || (qtl <= qbntl)))
                outedFlowIncludingLoss = qtl;
        }

        return outedFlowIncludingLoss;
    }

    /**
     * This function will convert double to Double
     * 
     * @param doubleArray
     * @return the Double array
     */
    Double[] convertdouble2Double(final double[] doubleArray)
    {
        final Double[] tempArray = new Double[doubleArray.length];
        int i = 0;
        for(final double d: doubleArray)
        {
            tempArray[i] = d;
            i++;
        }

        return tempArray;
    }

    /**
     * This function will convert Double array list to primitive double array
     * 
     * @param arrayListDouble
     * @return
     */
    double[] convertArrayListDouble2double(final ArrayList<Double> arrayListDouble)
    {

        final double[] tempArray = new double[arrayListDouble.size()];
        int i = 0;
        for(final double d: arrayListDouble)
        {
            tempArray[i] = d;
            i++;
        }

        return tempArray;
    }

    /**
     * This function will shift values to right and add new values for first two index
     * 
     * @param shiftValueArray
     * @param val0
     * @param val1
     * @return the new array values
     */
    ArrayList<Double> shiftValuesToRight(final ArrayList<Double> shiftValueArray, final double val0, final double val1)
    {
        for(int i = shiftValueArray.size() - 1; i > 0; i -= 2)
        {
            if((i + 2) > shiftValueArray.size())
            {
                shiftValueArray.add(shiftValueArray.get(i - 1));
                shiftValueArray.add(shiftValueArray.get(i));
            }
            else
            {
                shiftValueArray.set(i + 2, shiftValueArray.get(i));
                shiftValueArray.set(i + 1, shiftValueArray.get(i - 1));
            }
        }

        // Set first pair to new values
        shiftValueArray.set(0, val0);
        shiftValueArray.set(1, val1);

        return shiftValueArray;
    }

    /**
     * @param newPair
     */
    void setNewNumPair(final int newPair)
    {
        _newNumPair = newPair;
    }

    /**
     * @return the _newNumPair
     */
    int getNewNumPair()
    {
        return this._newNumPair;
    }

    /**
     * @return the _qdt
     */
    boolean getFlagQDT()
    {
        return _qdt;
    }

    /**
     * @param flagQDT
     */
    void setFlagQDT(final boolean flagQDT)
    {
        _qdt = flagQDT;
    }

}
