package ohd.hseb.ohdmodels.lagkj;

import java.util.ArrayList;
import java.util.List;

import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.NwsrfsDataTypeMappingReader;
import ohd.hseb.util.fews.OHDConstants;

class LagKJModel
{
    private final LagKJCommonMethods _commonMethods = new LagKJCommonMethods();

    private final Logger _logger;

    private double[] _kAtlantaValues;
    private final double[] _paramPairsValueKQ;
    private final double[] _paramPairsValueLagQ;

    final long _intervalInMillis;
    final long _startTime;
    final long _endTime;

    private final LagKJModelParameters _lagkParams;
    private final LagKJModelState _lagkStates;

    private int _carryoverTime;
    private int _numDataTime;
    private int _timeStep;
    private int _NumPairsQTCarryover;
    private int _NumPairsLagQ; //*2
    private int _NumPairsKQ; //*2
    private final int _pairsLagQ;
    private int _ipass = 1;

    // Input Ts
    private RegularTimeSeries _inflowTs = null; //QINE (Discharge - L3/T) or QNE (Volume - L3)

    // Output Ts
    private double[] _outputTS;

    LagKJModel(final LagKJModelParameters lagkParams,
               final LagKJModelState lagkStates,
               final long startTime,
               final long endTime,
               final long interval,
               final RegularTimeSeries drivingRTS,
               final Logger logger)
    {
        _logger = logger;

        _intervalInMillis = interval;
        _startTime = startTime;
        _endTime = endTime;

        _lagkParams = lagkParams;
        _lagkStates = lagkStates;

        _inflowTs = drivingRTS;

        _paramPairsValueKQ = _lagkParams.getPairsValuesKQ();
        _paramPairsValueLagQ = _lagkParams.getPairsValuesLagQ();
        _pairsLagQ = _lagkParams.getNumPairsLagQ();
    }

    /**
     * THIS METHOD PERFORMS THE LAG AND/OR K FUNCTIONS AND RETURNS A TIME SERIES OF OUTFLOW WITH THE TIME STEP EQUALS TO
     * THE TIME STEP OF THE INFLOW TIME SERIES.
     * 
     * @return The outflow time series
     * @throws Exception
     */
    RegularTimeSeries runLagKModel() throws Exception
    {
        _numDataTime = _inflowTs.getMeasurementCount();
        _timeStep = _inflowTs.getIntervalInHours();
        _carryoverTime = _numDataTime * _timeStep;

        _NumPairsLagQ = _lagkParams.getNumPairsLagQ() * 2;
        _NumPairsKQ = _lagkParams.getNumPairsKQ() * 2;
        _outputTS = new double[_numDataTime];// Initialize _outputTS variable

        // Compute K by Atlanta Method if K option is enable =============================
        if(_lagkParams.isKOptEnable())
        {
            if(_logger.getPrintDebugInfo() >= 5)
            {
                _logger.log(Logger.DEBUG, "K OPTION ENABLED");
                _logger.log(Logger.DEBUG, "C(1) - length carryover array = " + _lagkStates.getLengthCarryoverArr());
                _logger.log(Logger.DEBUG, "C(2)--currentLaggedInflow,C(3)--currentOutflow,C(4)--currentStorage = " + _lagkStates.getCurrentLaggedInflow() + "  "
                    + _lagkStates.getCurrentOutflow() + "  " + _lagkStates.getCurrentStorage());
            }

            if(_lagkParams.getAtlantaMethod() == 1)
            {
                // Call PINA7.f
                final ArrayList<Double> tempArrayList = this.computeKbyAtlantaMethod();

                // Define array of Double
                _kAtlantaValues = new double[tempArrayList.size()];

                // Convert a list to array
                _kAtlantaValues = _commonMethods.convertArrayListDouble2double(tempArrayList);

                if(_logger.getPrintDebugInfo() >= 5)
                    for(int i = 0; i < _kAtlantaValues.length; i++)
                        _logger.log(Logger.DEBUG, "==> kAtlantaValues(" + i + ") = " + _kAtlantaValues[i]);

                tempArrayList.clear();
            }
        }//if (_lagkParams.isKOptEnable())

        //Call EX7
        if(_logger.getPrintDebugInfo() >= 5)
        {
            _logger.log(Logger.DEBUG, "\n========= ex7 =============");
            _logger.log(Logger.DEBUG, "_startTime =  " + _startTime + " _endTime + " + _endTime + " timeInterval = "
                + _intervalInMillis);
            _logger.log(Logger.DEBUG, " unit" + _inflowTs.getMeasuringUnit());
        }

        // Perform LAG operation =============================================
        if(_lagkParams.isLagOptEnable())
        {
            _outputTS = this.LagOperation(); //flag7.f
        }
        else
        //If LAG not enable, copy _inflowTs to _outputTS
        {
            for(int i = 0; i < _numDataTime; i++)
            {
                _outputTS[i] = _inflowTs.getMeasurementValueByIndex(i, _inflowTs.getMeasuringUnit());
            }
        }

        // Perform K Operation ================================================
        _outputTS = this.KOperation();

        // Output TS
        final RegularTimeSeries lagkOutflowTS = new RegularTimeSeries(_startTime,
                                                                      _endTime,
                                                                      _lagkParams.getOutflowTimeInterval(),
                                                                      OHDConstants.DISCHARGE_UNIT);

        lagkOutflowTS.setTimeSeriesType(_lagkParams.getOutflowTimeSeriesType());

        // Extra step for old legacy models converter to Java need to define a qualifier id which is NWSRFS TSID
        final List<String> qualifierIds = new ArrayList<String>();

        qualifierIds.add(_lagkParams.getOutflowTimeSeriesId());
        lagkOutflowTS.setQualifierIds(qualifierIds);

        // Save the value to output ts
        for(int i = 0; i < _numDataTime; i++)
            lagkOutflowTS.setMeasurementByIndex(_outputTS[i], i);

        return lagkOutflowTS;
    }

    /**
     * THIS METHOD CONTROLS K OPERATION
     * 
     * @author ORIGINALLY PROGRAMMED BY GEORGE F. SMITH - HRL DECEMBER 1979
     * @see FORTRAN - fexa7.f
     * @return the output time series
     * @throws Exception
     */
    private double[] KOperation() throws Exception
    {
        if(_logger.getPrintDebugInfo() >= 0)
        {
            _logger.log(Logger.DEBUG, "  Starting K Option ....");
            _logger.log(Logger.DEBUG,
                        "IBK-P18 -- ConstantK =" + _lagkParams.getConstantK() + "  TLRC -- Transmission Loss Recession Coef = "
                            + _lagkParams.getTransmissionLossRecessionCoef() + " IATL -- Atlanta method = "
                            + _lagkParams.getAtlantaMethod());
            _logger.log(Logger.DEBUG, "koption = " + _lagkParams.isKOptEnable());
        }

        if(_lagkParams.isKOptEnable() && _lagkParams.getAtlantaMethod() == 0)
        {
            //fk7
            return (this.controlAttenuationK());
        }

        if(_lagkParams.isKOptEnable() && _lagkParams.getAtlantaMethod() == 1)
        {
            //fka7
            return (this.computeAttenuationK());
        }

        if(_lagkParams.isKOptEnable() == false && _numDataTime > 0)
        {
            _lagkStates.setCurrentOutflow(_outputTS[_numDataTime - 1]);

            if(_logger.getPrintDebugInfo() >= 4)
                _logger.log(Logger.DEBUG, "++++++++ reset CT(3) -- current outflow = " + _lagkStates.getCurrentOutflow());
        }

        if(_logger.getPrintDebugInfo() > 0)
            _logger.log(Logger.DEBUG, "\n ======= DONE K OPTION ==========\n");

        return _outputTS;
    }

    /**
     * THIS METHOD CONTROLS THE LAG OPERATION. IT SOLVES THE LAG FOR BOTH VARIABLE AND CONSTANT LAGGING.
     * 
     * @author ORIGINALLY PROGRAMMED BY GEORGE F. SMITH - HRL DECEMBER 1979 MODIFIED TO ADD NEW INTERPOLATION FUNCTION
     *         TO BE USED FOR THE MULTIPLE INTERCEPT PROBLEM - JML - HRL 10/92
     * @see Fortran - flag7.f
     * @return outflowTimeSeries - The outflow time series
     */
    private double[] LagOperation()
    {
        ArrayList<Double> QT = new ArrayList<Double>();

        double inflow = 0.0;
        boolean constantLag = false;
        final double[] constantQT = new double[2];

        int numPairQT = 0;

        final double[] outflowTimeSeries = new double[_numDataTime]; //output TS

        if(_logger.getPrintDebugInfo() >= 0)
        {
            _logger.log(Logger.DEBUG, "  Starting LAG Option ....");
            _logger.log(Logger.DEBUG, "NDT = " + _numDataTime + " carryoverTime = " + _carryoverTime);
        }

        //Clear out QT and lagQTcarryover before store new data
        QT.clear();

        _NumPairsQTCarryover = _lagkStates.getNumPairsQTCarryover() * 2;

        final double lagQTcarryover[] = new double[_NumPairsQTCarryover];
        int idxco = 0;

        // Copy old carryover to lagQTcarryover
        if(_lagkStates.getNumPairsQTCarryover() > 0)
        {
            if(_logger.getPrintDebugInfo() >= 5)
                _logger.log(Logger.DEBUG, "MXLCO -- number pairs QT carryover = " + _lagkStates.getNumPairsQTCarryover());

            for(int i = 0; i < _NumPairsQTCarryover; i += 2)
            {
                if(_lagkStates.getPairsQTCarryoverValues()[i + 1] == 0.0) //time
                {
                    if(_logger.getPrintDebugInfo() >= 5)
                    {
                        _logger.log(Logger.DEBUG, "**#C(" + (i + 1) + ") = "
                            + _lagkStates.getPairsQTCarryoverValues()[i + 1]);
                    }
                    break;
                }

                if(_lagkStates.getPairsQTCarryoverValues()[i + 1] > _carryoverTime)
                {
                    if(idxco >= lagQTcarryover.length)
                        break;

                    lagQTcarryover[idxco] = _lagkStates.getPairsQTCarryoverValues()[i];
                    lagQTcarryover[idxco + 1] = _lagkStates.getPairsQTCarryoverValues()[i + 1] - _carryoverTime;

                    if(_logger.getPrintDebugInfo() >= 5)
                    {
                        _logger.log(Logger.DEBUG, "QTcarryover " + _lagkStates.getPairsQTCarryoverValues()[i + 1]
                            + " > cotime -- carryover time " + _carryoverTime);
                    }

                    idxco += 2;
                }
                else
                {
                    QT.add(_lagkStates.getPairsQTCarryoverValues()[i]);
                    QT.add(_lagkStates.getPairsQTCarryoverValues()[i + 1]);

                    if(_logger.getPrintDebugInfo() >= 5)
                        _logger.log(Logger.DEBUG, "**?QT(" + (QT.size() - 2) + ")=" + QT.get(QT.size() - 2));
                }
            }
        }

        //Interpolate to find LAGS
        if(_pairsLagQ == 0.0)
        {
            constantLag = true;
            constantQT[0] = _lagkParams.getConstantLag(); //time
            constantQT[1] = 1.E20; //cms
            numPairQT = 1;
        }

        double qTime = 0.0;
        int index = 0;
        int indexco = idxco;

        if(_logger.getPrintDebugInfo() >= 5)
            _logger.log(Logger.DEBUG, "lengthQT = " + lagQTcarryover.length + " constantLag = " + constantLag);

        for(long timeStep = _startTime; timeStep <= _endTime; timeStep += _intervalInMillis)
        {
            final int current = (index + 1) * _inflowTs.getIntervalInHours();

            inflow = _inflowTs.getMeasurementValueByTime(timeStep, OHDConstants.DISCHARGE_UNIT); //cms ; note should check cmsd???

            if(constantLag == true)
            {
                if(_logger.getPrintDebugInfo() >= 6)
                {
                    _logger.log(Logger.DEBUG, "11- QA = " + inflow + " NPLQ-- number pair QT = " + numPairQT + " CONLQ1--constant QT1, CONLQ2--constant QT2 = "
                        + constantQT[0] + " " + constantQT[1]);
                }

                _commonMethods.setNewNumPair(0);
                qTime = _commonMethods.linearInterpolator(0, inflow, numPairQT, constantQT, 0);

                if(_logger.getPrintDebugInfo() >= 6)
                    _logger.log(Logger.DEBUG, "*** TIME = " + qTime);
            }
            else
            {
                if(_logger.getPrintDebugInfo() >= 6)
                    _logger.log(Logger.DEBUG, "22- QA--inflow = " + inflow + " NPLQ--pairsLagQ = " + _pairsLagQ);

                _commonMethods.setNewNumPair(0);

                qTime = _commonMethods.linearInterpolator(0, inflow, _pairsLagQ, _paramPairsValueLagQ, 0);

                if(_logger.getPrintDebugInfo() >= 6)
                    _logger.log(Logger.DEBUG, "*** TIME = " + qTime);
            }

            if((qTime + current) > _carryoverTime)
            {
                if(indexco >= lagQTcarryover.length)
                    break;

                lagQTcarryover[indexco] = inflow;
                lagQTcarryover[indexco + 1] = qTime + current - _carryoverTime;

                if(_logger.getPrintDebugInfo() >= 6)
                    _logger.log(Logger.DEBUG, "--> C(" + indexco + ")=" + lagQTcarryover[indexco] + " C("
                        + (indexco + 1) + ")=" + lagQTcarryover[indexco + 1] + "\n");

                indexco += 2;
            }
            else if(QT.size() <= (_lagkStates.getNumPairsQTCarryover() + (_numDataTime * 2)))
            {
                QT.add(inflow);
                QT.add(qTime + current);

                if(_logger.getPrintDebugInfo() >= 6)
                    _logger.log(Logger.DEBUG,
                                "here Q(" + (QT.size() - 2) + ")=" + QT.get(QT.size() - 2) + "  T(" + (QT.size() - 1)
                                    + ")=" + QT.get(QT.size() - 1));
            }
            else
                break;

            index++;
        }

        if(_logger.getPrintDebugInfo() >= 5)
            _logger.log(Logger.DEBUG, "========== 200 LQT = " + (QT.size()));

        // SHIFT VALUES IN QT ARRAY AHEAD AND INSERT CURRENT OUTFLOW VALUE FROM CARRYOVER.
        if(QT.size() > 0)
            QT = _commonMethods.shiftValuesToRight(QT, _lagkStates.getCurrentOutflow(), 0.0);

        // IF THE FIRST CARRYOVER TIME IS NONZERO INSERT IT AND Q VALUE AT THE END OF THE QT ARRAY.
        final double value0 = lagQTcarryover[0];
        final double value1 = lagQTcarryover[1];

        if(_logger.getPrintDebugInfo() >= 5)
            _logger.log(Logger.DEBUG, "C(6) =  " + value0 + " C(7) = " + value1);

        if(value1 != 0.0)
        {
            QT.add(value0);
            QT.add(value1 + _carryoverTime);

            if(_logger.getPrintDebugInfo() >= 5)
                _logger.log(Logger.DEBUG, "C7 NE 0 ==> QT(" + (QT.size() - 2) + ")= " + QT.get(QT.size() - 2) + " QT("
                    + (QT.size() - 1) + ")= " + QT.get(QT.size() - 1));
        }

        /*
         * PROCESS QT ARRAY USING ATLANTA METHOD OF REMOVING DOUBLE BACK VALUES BY SUMMING ALL PARTS OF THE ARRAY THAT
         * FALL 'INSIDE' THE CURVE AT ANY GIVEN TIME.
         */
        // FIRST REMOVE ANY SEQUENTIAL FLOWS WHICH HAVE EXACTLY THE SAME TIMES ASSOCIATED WITH THEM.  AVERAGE ALL SEQUENTIAL
        // FLOWS AT SAME TIME AND STORE AS ONE FLOW AT THAT TIME.
        int sizeQT = QT.size(); //don't use
        int l = 0;

        if((sizeQT - 2) > 0)
        {
            for(int i = 1; i < sizeQT; i += 2)
            {
                if(i >= (sizeQT - 1))
                    break;

                if(i > l)
                {
                    int k = i + 2;
                    if(_logger.getPrintDebugInfo() >= 6)
                        _logger.log(Logger.DEBUG, "QT-k(" + k + ")=" + QT.get(k) + " QT-i(" + i + ")=" + QT.get(i));

                    if(Double.compare(QT.get(k), QT.get(i)) == 0)
                    {
                        while(true)
                        {
                            l = k;
                            if((k + 2) > (sizeQT - 1)) //K = i+4
                                break;

                            if(Double.compare(QT.get(k), QT.get(k + 2)) != 0)
                            {
                                if(_logger.getPrintDebugInfo() >= 6)
                                    _logger.log(Logger.DEBUG, "QT(" + (k + 2) + ") = " + QT.get(k + 2) + " != QT(" + k
                                        + ") = " + QT.get(k) + "  GO235...");

                                break;
                            }

                            k += 2;
                        }//end while(true)

                        double sum = 0.0;

                        for(int kk = i; kk < k + 1; kk += 2)
                            sum += QT.get(kk - 1);

                        QT.set((i - 1), sum / ((l - i) / 2.0 + 1.0));

                        if(_logger.getPrintDebugInfo() >= 6)
                            _logger.log(Logger.DEBUG, "+++ QT(" + (i - 1) + ") =" + QT.get(i - 1) + " SUM, L, I = "
                                + sum + " " + l + " " + i);

                        final int ibeg = i + 2;

                        for(int kk = ibeg; kk < l + 1; kk += 2)
                        {
                            if(kk >= QT.size())
                                break;

                            QT.set(kk, -999.0);

                            if(_logger.getPrintDebugInfo() >= 6)
                                _logger.log(Logger.DEBUG, "== QT(" + kk + ") =" + QT.get(kk));
                        }
                    }//if(QT.get(k) == QT.get(i))
                }//end if(i > l )
            }

            int maxSizeQT = sizeQT;

            if(_logger.getPrintDebugInfo() >= 5)
                _logger.log(Logger.DEBUG, "====== 245 ====== LQT--sizeQT, LQTN--Maximum size QT = " + sizeQT + " " + maxSizeQT);

            for(int kk = 1; kk < sizeQT; kk += 2)
            {
                if(kk >= (sizeQT - 1))
                    break;

                final double val = QT.get(kk);

                if(_logger.getPrintDebugInfo() >= 5)
                    _logger.log(Logger.DEBUG, "-- val(" + kk + ") = " + val);

                if(val <= -998.5) //USE DISCHARGE VALUE
                {
                    if(_logger.getPrintDebugInfo() >= 5)
                        _logger.log(Logger.DEBUG, "TEST ME AT -998.5");

                    maxSizeQT = sizeQT - 2;
                    final int iend = sizeQT - 1;

                    if(_logger.getPrintDebugInfo() >= 5)
                        _logger.log(Logger.DEBUG, "LQTN--maximum size QT = " + maxSizeQT + " LQTEND = " + iend + " KK = " + kk);

                    for(int kkk = kk; kkk < iend; kkk++)
                    {
                        if(kkk >= QT.size())
                            break;

                        QT.set((kkk - 1), QT.get(kkk + 1));

                        if(_logger.getPrintDebugInfo() >= 5)
                            _logger.log(Logger.DEBUG, "QT-1(" + (kkk - 1) + ")=" + QT.get(kkk - 1) + " QT+1("
                                + (kkk + 1) + ")=" + QT.get(kkk + 1));
                    }

                    if(_logger.getPrintDebugInfo() >= 5)
                        _logger.log(Logger.DEBUG, "go back 245...");
                }

                sizeQT = maxSizeQT;
            }//for(int kk = 1; kk < maxSizeQT; kk += 2)

            // CHECK FOR DOUBLE BACK OF QT ARRAY - REPEATED SHORT WAVES WITH LONG LAGS.
            // IF NO DOUBLE BACK CAN PROCESS MORE EFFICIENTLY IN NEXT SECTION OF CODING.
            int maxLag = -999;

            if(_pairsLagQ == 0.0) //use constant Lag value
            {
                maxLag = (int)_lagkParams.getConstantLag();
            }
            else
            {
                for(int i = 0; i < _NumPairsLagQ; i += 2)
                    if(_paramPairsValueLagQ[i] > maxLag)
                        maxLag = (int)_paramPairsValueLagQ[i];
            }

            if(_logger.getPrintDebugInfo() >= 5)
                _logger.log(Logger.DEBUG, " after MXLAG-P(20) = " + maxLag);

            final int maxLag2 = (maxLag * 2);
            boolean doubleBack = false;
            final double[] qtValues = _commonMethods.convertArrayListDouble2double(QT);
            final int numpairQT = (int)(qtValues.length / 2.);

            if((sizeQT - 2) != 0) //need to use sizeQT=maxSizeQT.  Do not use qtValues.length
            {
                for(int i = 1; i < sizeQT; i += 2)
                {
                    if(i >= (sizeQT - 1))
                        break;

                    if(_logger.getPrintDebugInfo() >= 5)
                        _logger.log(Logger.DEBUG, "QT(" + i + ")=" + qtValues[i] + " >= QT(" + (i + 2) + ")="
                            + qtValues[i + 2]);

                    if(Double.compare(qtValues[i], qtValues[i + 2]) >= 0)
                    {
                        if(_logger.getPrintDebugInfo() >= 5)
                            _logger.log(Logger.DEBUG, "set DBLBK = true");

                        doubleBack = true;
                        break;
                    }
                }
            }

            // HAVE DOUBLEBACKS TO GET HERE  ====================================================
            double timeStart = 0; //timestart is the first time values will be summed for
            final int maxNumPairQT = _lagkStates.getNumPairsQTCarryover();

            if(doubleBack == true)
            {
                _logger.log(Logger.DEBUG, "HAVE DOUBLEBACKS TO GET HERE......");
                //  NOW CHECK QT VALUE AGAINST TIMES WANTED FOR OUTFLOW TIME SERIES
                final int QTsize = qtValues.length;
                double tp2 = 0.0;
                int jst = 2;
                boolean add = true;
                int lastj = 0;
                double tlast = 0.;
                boolean goto912 = false;

                for(int i = 0; i < _numDataTime; i++)
                {
                    timeStart = (i + 1) * _timeStep;

                    if(timeStart > maxLag)
                    {   
                    	// put back (- maxNumPairQT) to fix bug976, by RHC 1/4/2013
                        jst = (int)((i + 1) * 2. - maxLag / _timeStep) - (maxNumPairQT); 
                        //Bug 857 - 9/18/12
                        //Reset Time QT pair index if it is out of bounds or it should be even number.(like Fortran does)
                        if(jst <= 0)
                            jst = 2;
                        else if((jst % 2) == 1)
                            jst += 1;

                        if(_logger.getPrintDebugInfo() >= 6)
                            _logger.log(Logger.DEBUG, "i = " + (i + 1) + " maxLag, maxCol = " + maxLag + " "
                                + maxNumPairQT + " ==> jst = " + jst);
                    }

                    tp2 = timeStart + maxLag2;

                    double Q = 0.0;
                    jst -= 1; //java start index at 0

                    for(int j = jst; j < QTsize; j += 2)
                    {
                        if((j + 2) < QTsize)
                        {
                            if(_logger.getPrintDebugInfo() >= 6)
                                _logger.log(Logger.DEBUG, "-- QT(" + (j + 2) + ") = " + qtValues[j + 2] + "  > tp2 = "
                                    + tp2 + " add = " + add);

                            if(Double.compare(qtValues[j + 2], tp2) > 0)
                            {
                                if(_logger.getPrintDebugInfo() >= 6)
                                {
                                    _logger.log(Logger.DEBUG, "GO TO 295 ....");
                                    _logger.log(Logger.DEBUG, "++ ADD = " + add);
                                }

                                break;
                            }
                        }
                        // IF TIME IN QT ARRAY (EVEN LOCATIONS - PAIRS ARE Q,T) IS EQ TO
                        // TSTAR CHECK WHETHER SURROUNDING VALUES ARE GT OR LT
                        if(_logger.getPrintDebugInfo() >= 7)
                        {
                            _logger.log(Logger.DEBUG, "111 - QT(" + (j + 1) + ") = " + qtValues[j] + " tstar = "
                                + timeStart);
                        }

                        if(qtValues[j] == timeStart)
                        {
                            // IF AT 1ST PAIR IN QT ONLY CHECK NEXT PAIR NOT PREVIOUS PAIR
                            if(j > 1)
                            {
                                if(_logger.getPrintDebugInfo() >= 6)
                                    _logger.log(Logger.DEBUG, "222 - J = " + j);

                                // IF AT LAST PAIR IN QT ONLY CHECK PREVIOUS PAIR, NOT NEXT PAIR
                                if(j < (QTsize - 1) && (j + 2 < QTsize))
                                {
                                    if(_logger.getPrintDebugInfo() >= 6)
                                        _logger.log(Logger.DEBUG, "3333 - J < LQT--QT size = " + (j + 1) + " " + QTsize);

                                    // GET HERE IF qtValues(J) EQ TSTAR AND NOT AT BEGINNING OR END OF qtValues ARRAY
                                    // IF TIME OF PREVIOUS QT PAIR IS LT TSTAR AND TIME OF NEXT 
                                    // QT PAIR IS GT TSTAR  -- ADD CURRENT FLOW
                                    if(!((qtValues[j - 2] >= timeStart) || (qtValues[j + 2] <= timeStart)))
                                    {
                                        if(_logger.getPrintDebugInfo() >= 6)
                                            _logger.log(Logger.DEBUG, "444 - Q = " + Q + " QT(" + (j - 1) + ")="
                                                + qtValues[j - 1]);

                                        Q += qtValues[j - 1];
                                        add = true;
                                    }
                                    // IF TIME OF PREVIOUS QT PAIR IS GT TSTAR AND TIME OF NEXT
                                    // QT PAIR IS LT TSTAR -- SUBT CURRENT FLOW
                                    else if(!((qtValues[j - 2] <= timeStart) || (qtValues[j + 2] >= timeStart)))
                                    {
                                        if(_logger.getPrintDebugInfo() >= 6)
                                            _logger.log(Logger.DEBUG, "555 - Q = " + Q + " QT(" + (j - 1) + ")="
                                                + qtValues[j - 1]);

                                        Q -= qtValues[j - 1];
                                        add = false;
                                        lastj = j - 1;
                                        tlast = timeStart;
                                    }
                                }
                                // IF NEITHER OF THE ABOVE CASES THEN AT EDGE OF QT CURVE -- 
                                // SO DON'T ADD OR SUBT CURRENT FLOW
                                else if(qtValues[j - 2] > timeStart)
                                {
                                    if(_logger.getPrintDebugInfo() >= 6)
                                        _logger.log(Logger.DEBUG, "6666 - Q = " + Q + " QT(" + (j - 1) + ")="
                                            + qtValues[j - 1]);

                                    Q -= qtValues[j - 1];
                                    add = false;
                                    lastj = j - 1;
                                    tlast = timeStart;
                                }
                                else if(qtValues[j - 2] < timeStart)
                                {
                                    if(_logger.getPrintDebugInfo() >= 6)
                                        _logger.log(Logger.DEBUG, "777 - Q = " + Q + " QT(" + (j - 1) + ")="
                                            + qtValues[j - 1]);

                                    Q += qtValues[j - 1];
                                    add = true;
                                }
                            }
                            // IF ALSO AT LAST PAIR SET outflowTimeSeries(I) TO QT(1)
                            else if(j >= (QTsize - 1))
                            {
                                goto912 = true;
                                if(_logger.getPrintDebugInfo() >= 6)
                                {
                                    _logger.log(Logger.DEBUG, "888 - GO TO 912 ....");
                                    _logger.log(Logger.DEBUG, " ====> Q = " + Q);
                                }
                                break;
                            }
                            else if((j + 2 < QTsize) && qtValues[j + 2] > timeStart)
                            {
                                if(_logger.getPrintDebugInfo() >= 6)
                                    _logger.log(Logger.DEBUG, "999 - Q = " + Q + " QT(" + (j - 1) + ")="
                                        + qtValues[j - 1]);

                                Q += qtValues[j - 1];
                                add = true;
                            }
                            else if((j + 2 <= QTsize) && qtValues[j + 2] < timeStart)
                            {
                                if(_logger.getPrintDebugInfo() >= 6)
                                    _logger.log(Logger.DEBUG, "1010 - Q = " + Q + " QT(" + (j - 1) + ")="
                                        + qtValues[j - 1]);

                                Q -= qtValues[j - 1];
                                add = false;
                                lastj = j - 1;
                                tlast = timeStart;
                            }

                            if(j >= (QTsize - 1))
                                break;
                        }
                        // HERE QT(J) NE TSTAR  -- IF QT(J) LT TSTAR AND QT(J+2) GT TSTAR -- ADD CURRENT FLOW
                        // IF AT END OF QT ARRAY CONTINUE
                        else if((j + 2 < QTsize) && j != (QTsize - 1))
                        {
                            if(_logger.getPrintDebugInfo() >= 6)
                                _logger.log(Logger.DEBUG, "aaa - J != LQT--QT size = " + j + " " + QTsize + "  QT(j)="
                                    + qtValues[j] + " QT(j+2)=" + qtValues[j + 2]);

                            if(!((qtValues[j] >= timeStart) || (qtValues[j + 2] <= timeStart)))
                            {
                                final double Qadd = this.solveInterceptProblem(timeStart, qtValues, j - 1);

                                if(_logger.getPrintDebugInfo() >= 6)
                                {
                                    _logger.log(Logger.DEBUG, "bbb - TSTAR--time start = " + timeStart + " QT(" + (j - 1) + ")="
                                        + qtValues[j - 1]);
                                    _logger.log(Logger.DEBUG, " =======> Qadd = " + Qadd);
                                }

                                Q += Qadd;
                                add = true;
                            }
                            // IF CURRENT TIME IS GT TSTAR AND NEXT TIME IS LT TSTAR -- SUBT CURRENT FLOW
                            else if(!((qtValues[j] <= timeStart) || (qtValues[j + 2] >= timeStart)))
                            {
                                final double Qsub = this.solveInterceptProblem(timeStart, qtValues, j - 1);
                                if(_logger.getPrintDebugInfo() >= 6)
                                {
                                    _logger.log(Logger.DEBUG, "ccc - TSTAR -- time start = " + timeStart + " QT(" + (j - 1) + ")="
                                        + qtValues[j - 1]);
                                    _logger.log(Logger.DEBUG, " =======> Qsub = " + Qsub);
                                }

                                Q -= Qsub;
                                add = false;
                                lastj = j - 1;
                                tlast = timeStart;
                            }
                        }// end if(qtValues[j] == timeStart)
                    } // end j loop

                    if(goto912 == true)
                    {
                        Q += qtValues[0];

                        if(_logger.getPrintDebugInfo() >= 6)
                            _logger.log(Logger.DEBUG, " |||||||| Q = " + Q);
                    }

                    if(_logger.getPrintDebugInfo() >= 7)
                        _logger.log(Logger.DEBUG, "==> ADD, TLAST, LASTJ, QT(LASTJ) = " + add + " " + tlast + " "
                            + (lastj) + " " + qtValues[lastj]);

                    if(add == false)
                    {
                        if(_logger.getPrintDebugInfo() >= 7)
                            _logger.log(Logger.DEBUG, "TEST ME AT (ADD == false)");

                        _commonMethods.setNewNumPair(0);

                        Q += _commonMethods.linearInterpolator(0, tlast, 1, qtValues, lastj - 1);
                    }

                    outflowTimeSeries[i] = Q;
                    jst++;

                    if(_logger.getPrintDebugInfo() >= 5)
                        _logger.log(Logger.DEBUG, "++++++++++++++++++++++++++++ QB(" + (i + 1) + ") = "
                            + outflowTimeSeries[i] + " jst = " + jst);
                } // end i loop
            }
            //NO DOUBLEBACKS IF AT THIS CODING ==========================================================================
            else
            {
                _logger.log(Logger.DEBUG, "NO DOUBLEBACKS IF AT THIS CODING..........");
                int LY = 2;
                int locQT = 1;

                if(maxLag2 > 2.)
                    LY = maxLag2;

                _commonMethods.setNewNumPair(0);

                for(int i = 0; i < _numDataTime; i++)
                {
                    timeStart = ((i + 1) * _timeStep);

                    if(_logger.getPrintDebugInfo() >= 5)
                    {
                        _logger.log(Logger.DEBUG, "i = " + (i + 1) + " timeStar = " + timeStart + " LOCQT = " + locQT);
                        _logger.log(Logger.DEBUG, "+++ qtValue(" + locQT + ")= " + qtValues[locQT - 1] + "  qtValue("
                            + (locQT + 1) + ") = " + qtValues[locQT]);
                    }

                    outflowTimeSeries[i] = _commonMethods.linearInterpolator(LY,
                                                                             timeStart,
                                                                             numpairQT,
                                                                             qtValues,
                                                                             locQT - 1);

                    if(_logger.getPrintDebugInfo() >= 5)
                        _logger.log(Logger.DEBUG, "=> QB(" + (i + 1) + ") = " + outflowTimeSeries[i] + "  LX = "
                            + (_commonMethods.getNewNumPair() + 1) + "\n");

                    locQT = (_commonMethods.getNewNumPair() + 1) * 2 - 1;
                }
            }
        }
        else
        {
            for(int i = 0; i < _numDataTime; i++)
            {
                outflowTimeSeries[i] = QT.get(0);

                if(_logger.getPrintDebugInfo() >= 5)
                    _logger.log(Logger.DEBUG, "&&& => QB(" + i + ") = " + outflowTimeSeries[i]);
            }
        }

        //Update state QTPairsCarryover
        if(lagQTcarryover.length > 0)
        {
            if(_NumPairsQTCarryover < lagQTcarryover.length)
                _NumPairsQTCarryover = lagQTcarryover.length;

            final double[] pairsQTLagCarryover = new double[_NumPairsQTCarryover];
            System.arraycopy(lagQTcarryover, 0, pairsQTLagCarryover, 0, lagQTcarryover.length);

            _lagkStates.setNumPairsQTCarryover(_NumPairsQTCarryover / 2);
            _lagkStates.setPairsQTCarryoverValues(pairsQTLagCarryover);

        }
        if(_logger.getPrintDebugInfo() >= 5)
        {
            _logger.log(Logger.DEBUG,
                        "C = " + _lagkStates.getLengthCarryoverArr() + " " + _lagkStates.getCurrentLaggedInflow() + " "
                            + _lagkStates.getCurrentOutflow() + " " + _lagkStates.getCurrentStorage() + "  "
                            + _lagkStates.getNumPairsQTCarryover());

            for(int t = 0; t < _NumPairsQTCarryover; t++)
                _logger.log(Logger.DEBUG, " " + _lagkStates.getPairsQTCarryoverValues()[t]);
        }

        if(_logger.getPrintDebugInfo() >= 0)
            _logger.log(Logger.DEBUG, "\n ======= DONE LAG OPTION ==========\n");

        return outflowTimeSeries;
    }

    /**
     * THIS METHOD DOES THE ATTENUATION (K) COMPUTATIONS FOR THE ATLANTA METHOD. THIS METHOD HANDLES NON-LINEAR AND
     * NON-UNIQUE K VERSUS TIME RELATIONSHIPS.
     * 
     * @author ORIGINALLY PROGRAMMED BY GEORGE F. SMITH - HRL DECEMBER 1979
     * @see FORTRAN - fka7.f
     * @return outflowTimeSeries - the outflow time series
     * @throws Exception
     */
    private double[] computeAttenuationK() throws Exception
    {
        final double[] outflowTimeSeries = new double[_numDataTime]; //output TS

        boolean constantK = false;
        final String dimension = NwsrfsDataTypeMappingReader.getNwsrfsDim(_lagkParams.getInflowTimeSeriesType(),
                                                                          _logger);
        final String inflowUnit = NwsrfsDataTypeMappingReader.getNwsrfsUnit(_lagkParams.getInflowTimeSeriesType(),
                                                                            _logger);
        if(_logger.getPrintDebugInfo() >= 0)
            _logger.log(Logger.DEBUG, "**  computeAttenuationK() FKA7 entered ....");

        // If constant is used then initialize variables
        final int numpairKQ = _lagkParams.getNumPairsKQ(); //JLAG - NPKQ
        double firstHalfSegForK = 0.0f; //xk1
        double secondHalfSegForK = 0.0f; //xk2
        double firstHalfOfCurrQuaterSegforK = 0.0; //xk14
        double secondHalfOfCurrQuaterSegforK = 0.0; //xk24
        final double constanttKValue = _lagkParams.getConstantK();

        //if(_lagkParams.getConstantK() > 0.0)
        if (constanttKValue > 0.0)
        {
            constantK = true;
            firstHalfSegForK = _lagkParams.getConstantK(); //xk1
            secondHalfSegForK = firstHalfSegForK;
            firstHalfOfCurrQuaterSegforK = firstHalfSegForK;
            secondHalfOfCurrQuaterSegforK = firstHalfSegForK;
        }

        if(_logger.getPrintDebugInfo() >= 5)
        {
            _logger.log(Logger.DEBUG, " DataType = " + _lagkParams.getInflowTimeSeriesType() + " Dimension = "
                + dimension + " Unit" + inflowUnit);

            _logger.log(Logger.DEBUG, "==> constantK = " + constantK + " its value = " + constanttKValue);
        }

        double fact = 1.0;
        double finalInflow = _lagkStates.getCurrentLaggedInflow(); //X2
        double initialOutflow = _lagkStates.getCurrentOutflow(); //Y1

        final boolean meanQ = _lagkParams.getMeanQ();
        double storageTerm = _lagkStates.getCurrentStorage() * 2. / _timeStep; //s2odt
        final double quarterTimeStep = _timeStep / 4.;

        if(_logger.getPrintDebugInfo() >= 5)
            _logger.log(Logger.DEBUG,
                        "Initial Values For timeStep, quarterTimeStep, prevLaggedInflow, prevOutflow, S2ODT--storage term, meanQ Are: "
                            + _timeStep + " " + quarterTimeStep + " " + finalInflow + " " + initialOutflow + " "
                            + storageTerm + " " + meanQ);

        // BEGIN OVERALL LOOP FOR THIS PASS THROUGH K OPERATION
        double initialInflow = 0.0; //X1
        double solvedOutflow = 0.0; //Y2

        double prevQ = 0.0; //Y1

        for(int i = 0; i < _numDataTime; i++)
        {
            double value = 0.0;
            initialInflow = finalInflow; //curLaggedInflow - X1

            finalInflow = _outputTS[i]; // curOutflow (X2)

            prevQ = initialOutflow; //Y1

            if(_logger.getPrintDebugInfo() >= 5)
                _logger.log(Logger.DEBUG, "X1--initial inflow, X2--final inflow, QPREV, meanQ = " + initialInflow + " " + finalInflow + " " + prevQ
                    + " " + meanQ);

            value = initialInflow + finalInflow + storageTerm - initialOutflow; //meanQ ==false

            if(meanQ == true)
                value = 2. * finalInflow * 24 / _timeStep + storageTerm - initialOutflow;

            if(value < 1.0E-7)
                value = 0.0;

            if(_logger.getPrintDebugInfo() >= 6)
                _logger.log(Logger.DEBUG, " KAtlanta value=" + value + " IBOS1 = 0" + "lenghtAt = " + _kAtlantaValues.length
                    + " _kAvalue0 = " + _kAtlantaValues[0]);

            // There are two set kAtlantaValue which depends on ipass = 1 or 2 
            int pass1KAtSize = _kAtlantaValues.length / 2;

            if(_ipass > 1)
                pass1KAtSize = _kAtlantaValues.length / 4;

            _commonMethods.setNewNumPair(0);

            solvedOutflow = _commonMethods.linearInterpolator(0, value, pass1KAtSize, _kAtlantaValues, 0);

            if(_logger.getPrintDebugInfo() >= 1)
                _logger.log(Logger.DEBUG, "IN DO 10 LOOP, VALUE = " + value);

            if(_logger.getPrintDebugInfo() >= 6)
            {
                _logger.log(Logger.DEBUG, " ===> Y2--solved outflow = " + solvedOutflow + " value = " + value);
                _logger.log(Logger.DEBUG, "-- Y1--initial outflow, Y2--solved outflow, NPKQ--numberPairs KQ = " + initialOutflow + " " + solvedOutflow + " "
                    + _lagkParams.getNumPairsKQ());
            }

            if(meanQ == false)
            {
                if(constantK == false)
                {
                    _commonMethods.setNewNumPair(0);
                    firstHalfSegForK = _commonMethods.linearInterpolator(0,
                                                                         initialOutflow,
                                                                         numpairKQ,
                                                                         _paramPairsValueKQ,
                                                                         0);

                    _commonMethods.setNewNumPair(0);
                    secondHalfSegForK = _commonMethods.linearInterpolator(0,
                                                                          solvedOutflow,
                                                                          numpairKQ,
                                                                          _paramPairsValueKQ,
                                                                          0);

                    if(_logger.getPrintDebugInfo() >= 6)
                        _logger.log(Logger.DEBUG, "++ Y1--initial outflow, XK1--1st half segFork = " + initialOutflow + " " + firstHalfSegForK
                            + " -- Y2--solved outflow, XK2--2nd half segFork = " + solvedOutflow + " " + secondHalfSegForK + "\n");
                }

                if(_logger.getPrintDebugInfo() >= 1)
                {
                    _logger.log(Logger.DEBUG, "IN DO 10 LOOP, X1--initialInflow,X2--finalInflow,VALUE,Y2--solvedOutflow,XK1--1st 1/2 segfork,XK2--2nd 1/2 segfork =");
                    _logger.log(Logger.DEBUG, initialInflow + " " + finalInflow + " " + value + " " + solvedOutflow
                        + " " + firstHalfSegForK + " " + secondHalfSegForK);
                }

                if(!((firstHalfSegForK >= (quarterTimeStep * 2.0)) && (secondHalfSegForK >= (quarterTimeStep * 2.0))))
                {
                    if(_logger.getPrintDebugInfo() >= 6)
                        _logger.log(Logger.DEBUG, "??? xk1--1st 1/2 segFork, xk2--2nd 1/2 segFork, quarterTimeStep = " + firstHalfSegForK + " "
                            + secondHalfSegForK + " " + quarterTimeStep);

                    if((firstHalfSegForK > (quarterTimeStep / 2.0)) || (secondHalfSegForK > (quarterTimeStep / 2.0)))
                    {
                        //GET HERE IF K FOR Y1 GT DT/2 AND K FOR Y2 LE DT/2 OR VICE VERSA SOLVE EQUATIONS IN THIS LOOP WITH DT=ORIGINAL DT/4
                        storageTerm = storageTerm * 4.0;

                        if(storageTerm < -0.5)
                            _logger.log(Logger.WARNING, " s2odt < -0.5 ............");

                        final double deltaX = finalInflow - initialInflow;

                        if(_logger.getPrintDebugInfo() >= 6)
                            _logger.log(Logger.DEBUG, "x2--finalInflow, x1--initialInflow,s2odt--storageTerm,dx--deltaX = " + finalInflow + " " + initialInflow + " "
                                + storageTerm + " " + deltaX);

                        final double quarterOfDeltaX = deltaX / 4.0; //dxovr4 = dx / 4.0
                        final double oneEighthTimeSeries = quarterTimeStep / 2.0; //xitao2
                        double initialInflowForCurrQuater = 0.0; //x14

                        final int pass2KAtSize = _kAtlantaValues.length / 2;

                        int indexStart = 0;

                        if(this.getIpass() > 1)
                            indexStart = pass2KAtSize;

                        for(int j = 0; j < 4; j++)
                        {
                            initialInflowForCurrQuater = initialInflow + j * quarterOfDeltaX;
                            finalInflow = initialInflowForCurrQuater + quarterOfDeltaX; //X2
                            value = initialInflowForCurrQuater + finalInflow + storageTerm - initialOutflow; //x14 + X2 + so2dt - Y1

                            _commonMethods.setNewNumPair(0);
                            solvedOutflow = _commonMethods.linearInterpolator(0,
                                                                              value,
                                                                              pass2KAtSize,
                                                                              _kAtlantaValues,
                                                                              indexStart);

                            if(_logger.getPrintDebugInfo() >= 6)
                                _logger.log(Logger.DEBUG, "!!!!!Y2--solvedOutflow = " + solvedOutflow + " with value (x14+X2+so2dt-Y1) = " + value);

                            if(constantK == false)
                            {
                                _commonMethods.setNewNumPair(0);
                                firstHalfOfCurrQuaterSegforK = _commonMethods.linearInterpolator(0,
                                                                                                 initialOutflow,
                                                                                                 numpairKQ,
                                                                                                 _paramPairsValueKQ,
                                                                                                 0);
                                _commonMethods.setNewNumPair(0);
                                secondHalfOfCurrQuaterSegforK = _commonMethods.linearInterpolator(0,
                                                                                                  solvedOutflow,
                                                                                                  numpairKQ,
                                                                                                  _paramPairsValueKQ,
                                                                                                  0);
                            }

                            // IF EITHER K FOR Y1 OR K FOR Y2 IS STILL LT NEW DT/2
                            // (I.E. ORIGINAL DT/8) SET OUTFLOW=MIN(INFLOW,VALUE) AND CONTINUE IN
                            // QUARTER PERIOD LOOP
                            if(firstHalfOfCurrQuaterSegforK < oneEighthTimeSeries
                                || secondHalfOfCurrQuaterSegforK < oneEighthTimeSeries)
                            { //TODO
                                if(_logger.getPrintDebugInfo() >= 1)
                                {
                                    _logger.log(Logger.DEBUG, "K LT DT/8 IN DO 5 LOOP");
                                    _logger.log(Logger.DEBUG, "Y1--initialOutflow, " + "Y2--solvedOutflow, " +
                                    		"XK14--1st 1/2 current 1/4 segFork,XK24--2nd 1/2 current 1/4 segFork, " +
                                    		"X2--finalInflow = " + 
                                    			initialOutflow + " "+ solvedOutflow + " " 
                                    		+ firstHalfOfCurrQuaterSegforK + " " + secondHalfOfCurrQuaterSegforK 
                                    		+ " " + finalInflow);
                                }

                                solvedOutflow = finalInflow; //y2 = x2
                                if(_logger.getPrintDebugInfo() >= 6)
                                	_logger.log(Logger.DEBUG, "Is value less than solvedOutflow " + value + " ? " + solvedOutflow);
                                if(value < solvedOutflow)
                                    solvedOutflow = value;

                                if(_logger.getPrintDebugInfo() >= 6)
                                    _logger.log(Logger.DEBUG, "RRRRRRRRRRRRRR -- solvedOutflow = " + solvedOutflow);
                            }
                            else
                            {
                            	//solvedOutflow = finalInflow;
                            	if(_logger.getPrintDebugInfo() >= 1)
                                {
                                    _logger.log(Logger.DEBUG, "XK14--1st 1/2 current 1/4 segFork,XK24--2nd 1/2 current 1/4 segFork, solvedOutflow, VALUE = "
                                        + firstHalfOfCurrQuaterSegforK + " "
                                        + secondHalfOfCurrQuaterSegforK + " " + solvedOutflow + " " + value);
                                }
                            }

                            storageTerm = value - solvedOutflow; // value - y2

                            if(storageTerm < -0.5)
                                _logger.log(Logger.WARNING, "s2odt (storageTerm) < -0.5.........");

                            if(_logger.getPrintDebugInfo() >= 1)
                            {
                                _logger.log(Logger.DEBUG, "IN DO 5 LOOP - J,X14--initialInflowCurrent1/4,X2--finalInflow,VALUE,Y2--solvedOutflow,S2ODT--storageTerm = ");
                                _logger.log(Logger.DEBUG, " " + j + " " + initialInflowForCurrQuater + " "
                                    + finalInflow + " " + value + " " + solvedOutflow + " " + storageTerm);
                            }

                            initialOutflow = solvedOutflow; //Y1 = Y2
                        }// end for j loop

                        fact = 4.0;
                    }
                    // GET HERE IF K FOR Y1 AND K FOR Y2 ARE BOTH LT DT/2; SET OUTFLOW = MINIMUM OF (INFLOW,VALUE)
                    else
                    {
                        solvedOutflow = finalInflow; // Y2 = X2
                        if(_logger.getPrintDebugInfo() >= 6)
                        	_logger.log(Logger.DEBUG, "Is value also less than solvedOutflow " + value + " ? " + solvedOutflow);
                        if(value < solvedOutflow)
                            solvedOutflow = value;

                        if(_logger.getPrintDebugInfo() >= 6)
                            _logger.log(Logger.DEBUG, "HHHHHHHHHHHHHH -- solvedOutflow =" + solvedOutflow);

                        if(_logger.getPrintDebugInfo() >= 1)
                            _logger.log(Logger.DEBUG, "K LT DT/2 - Y2 -- solvedOutflow= " + solvedOutflow);
                    }
                } //if(!(xk1 >= (quarterTimeStep * 2.0)) && (xk2 >= (quarterTimeStep * 2.0)))
            }// if(meanQ == false)

            outflowTimeSeries[i] = solvedOutflow; //QB(i) = y2

            if(_logger.getPrintDebugInfo() >= 5)
                _logger.log(Logger.DEBUG, "$$$$$ QB(" + (i + 1) + ") = " + outflowTimeSeries[i]);

            storageTerm = (value - solvedOutflow) / fact;

            if(storageTerm < -0.5)
                _logger.log(Logger.WARNING, " s2odt(storageTerm) < - 0.5");

            fact = 1.0;

            if(_logger.getPrintDebugInfo() >= 1)
                _logger.log(Logger.DEBUG, "IN DO 10 LOOP - Y2--solvedOutflow,S2ODT--storageTerm = " + solvedOutflow + " " + storageTerm);

            initialOutflow = solvedOutflow; //Y1 = Y2

        }// end numDataTime loop =============

        // STORE CARRYOVER - IF C(4), CURRENT STORAGE, IS LT ZERO BECAUSE OF ROUNDOFF ERROR SET TO ZERO
        _lagkStates.setCurrentLaggedInflow(finalInflow); //C(2) = X2
        _lagkStates.setCurrentOutflow(solvedOutflow); //c(3) = Y2
        _lagkStates.setCurrentStoreage(storageTerm * _timeStep / 2.0);

        if(_lagkStates.getCurrentStorage() < 0.0)
            _lagkStates.setCurrentStoreage(0.0);

        if(_logger.getPrintDebugInfo() >= 1)
        {
            _logger.log(Logger.DEBUG, "AFTER DO 10 LOOP -States C(2)--currentLaggedInflow,C(3)--currentOutflow,C(4)--currentStorage = " + _lagkStates.getCurrentLaggedInflow()
                + " " + _lagkStates.getCurrentOutflow() + " " + _lagkStates.getCurrentStorage());

            _logger.log(Logger.DEBUG, "IN FKA7 - QB = ");

            for(int i = 0; i < _numDataTime; i++)
                if(i < 100)
                    _logger.log(Logger.DEBUG, " " + outflowTimeSeries[i]);
        }

        if(_logger.getPrintDebugInfo() >= 0)
            _logger.log(Logger.DEBUG, "Exit  computeAttenuationK() function ...");

        return outflowTimeSeries;
    }

    /**
     * THIS METHOD CONTROLS THE ATTENUATION (K) OPERATION.
     * 
     * @author ORIGINALLY PROGRAMMED BY GEORGE F. SMITH - HRL DECEMBER 1979
     * @see FORTRAN - fk7.f
     * @return the output time series outflowTimeSeries[]
     */
    private double[] controlAttenuationK()
    {
        final double[] outflowTimeSeries = new double[_numDataTime]; //output TS

        if(_logger.getPrintDebugInfo() > 0)
            _logger.log(Logger.DEBUG, "**  computeAttenuationK() FK7 entered ....");

        if(_numDataTime > 0)
        {
            final double[] pairsKQ = {_lagkParams.getConstantK(), 1.E20};

            final double halfInterval = _lagkParams.getInflowTimeInterval() / 2.;

            //TWO VARIABLES FOR FT. WORTH TRANSMISSION LOSS COMPUTATIONS
            double laggedInflowAtTheEndOfTimeStep = _lagkStates.getCurrentLaggedInflow();//x2
            double attenuatedOutflow = _lagkStates.getCurrentOutflow(); //y2
            double lastOutflow = _lagkStates.getCurrentStorage();//y1
            double lastStorage = 0.; //y0
            double qprev = 0.; //last outflow for Fort Worth calc
            double averageAcrossCurrSeg = 0.; //x12
            double averageLastStorageAndLastOutflow = 0.; //y01

            // PERFORM ATTENUATION ON INFLOW TIME SERIES.  ACTUALLY ROUTE AT ONE HALF ORIGINAL INTERVAL 
            // - BUT ONLY STORE OUTFLOW AT WHOLE INTERVALS.
            boolean flagConstK = false;

            for(int i = 0; i < _numDataTime; i++)
            {
                _commonMethods.setFlagQDT(false);

                final double laggedInflowAtTheStartOfTimeStep = laggedInflowAtTheEndOfTimeStep; //x1
                laggedInflowAtTheEndOfTimeStep = _outputTS[i];
                lastStorage = lastOutflow;
                lastOutflow = attenuatedOutflow;
                qprev = attenuatedOutflow;

                averageAcrossCurrSeg = (laggedInflowAtTheStartOfTimeStep + laggedInflowAtTheEndOfTimeStep) / 2.;
                averageLastStorageAndLastOutflow = (lastStorage + lastOutflow) / 2.;

                if(_lagkParams.getConstantK() > 0.)
                {
                    flagConstK = true;

                    // COMPUTE ONCE FOR EACH HALF ROUTING INTERVAL WITH CONSTANT K.  See fcmpk7.f and fcmpk7();
                    final double segmentRoutingResult = _commonMethods.computeOutflowEndingOfInterval(laggedInflowAtTheStartOfTimeStep, //x1
                                                                                                      averageAcrossCurrSeg, //x12
                                                                                                      averageLastStorageAndLastOutflow, //y01
                                                                                                      lastOutflow, //y1
                                                                                                      pairsKQ,
                                                                                                      halfInterval,
                                                                                                      flagConstK);
                    attenuatedOutflow = _commonMethods.computeOutflowEndingOfInterval(averageAcrossCurrSeg, //x12
                                                                                      laggedInflowAtTheEndOfTimeStep, //x2
                                                                                      lastOutflow, //y1
                                                                                      segmentRoutingResult,//y12
                                                                                      pairsKQ,
                                                                                      halfInterval,
                                                                                      flagConstK);
                }
                else
                {
                    // COMPUTE ONCE FOR EACH HALF ROUTING INTERVAL WITH VARIABLE K.
                    // IF K LT (HALF INTERVAL)/4.0 - CALL FQDT7 WHICH ROUTES AT ONE QUARTER OF CURRENT ROUTINE INTERVAL - I.E. AT ONE EIGHTH
                    // OF WHOLE (ORIGINAL) ROUTING INTERVAL. See fcmpk7.f and fcmpk7();
                    double segmentRoutingResult = _commonMethods.computeOutflowEndingOfInterval(laggedInflowAtTheStartOfTimeStep, //x1
                                                                                                averageAcrossCurrSeg, //x12
                                                                                                averageLastStorageAndLastOutflow, //y01
                                                                                                lastOutflow, //y1
                                                                                                pairsKQ,
                                                                                                halfInterval,
                                                                                                flagConstK);

                    if(_commonMethods.getFlagQDT() == true)
                    {
                        segmentRoutingResult = _commonMethods.computeKfromQuarterRoutingInterval(laggedInflowAtTheStartOfTimeStep, //x1
                                                                                                 averageAcrossCurrSeg, //x12
                                                                                                 averageLastStorageAndLastOutflow, //y01
                                                                                                 lastOutflow, //y1
                                                                                                 flagConstK,
                                                                                                 halfInterval,
                                                                                                 pairsKQ);
                    }

                    //fcmpk7();
                    attenuatedOutflow = _commonMethods.computeOutflowEndingOfInterval(averageAcrossCurrSeg, //x12
                                                                                      laggedInflowAtTheEndOfTimeStep, //x2
                                                                                      lastOutflow, //y1
                                                                                      segmentRoutingResult,//y12
                                                                                      pairsKQ,
                                                                                      halfInterval,
                                                                                      flagConstK);

                    if(_commonMethods.getFlagQDT() == true)
                    {
                        //fqdt7();
                        attenuatedOutflow = _commonMethods.computeKfromQuarterRoutingInterval(averageAcrossCurrSeg, //x12
                                                                                              laggedInflowAtTheEndOfTimeStep, //x2
                                                                                              lastOutflow, //y1
                                                                                              segmentRoutingResult, //y12
                                                                                              flagConstK,
                                                                                              halfInterval,
                                                                                              pairsKQ);
                    }
                }

                // STORE THE COMPUTED OUTFLOW IN outflowTimeSeries AT THE END OF EACH WHOLE ROUTING INTERVAL.
                if(_lagkParams.getTransmissionLossRecessionCoef() > 0.)
                {
                    //ftwtl7();
                    attenuatedOutflow = _commonMethods.performsFortWorthFlowTransmissionLossComputations(_lagkParams.getTransmissionLossRecessionCoef(),
                                                                                                         _lagkParams.getFlowAboveTLRC(),
                                                                                                         _outputTS[i],
                                                                                                         qprev,
                                                                                                         attenuatedOutflow);
                }

                outflowTimeSeries[i] = attenuatedOutflow;

                if(_logger.getPrintDebugInfo() >= 5)
                    _logger.log(Logger.DEBUG, "==> QB(" + i + ") = " + outflowTimeSeries[i]);

            }//end i loop

            // STORE CARRYOVER VALUES FOR K OPERATION.
            _lagkStates.setCurrentLaggedInflow(laggedInflowAtTheEndOfTimeStep); //x2
            _lagkStates.setCurrentStoreage(_lagkStates.getCurrentOutflow());

            if(_numDataTime > 1)
                _lagkStates.setCurrentStoreage(outflowTimeSeries[_numDataTime - 2]);

            _lagkStates.setCurrentOutflow(attenuatedOutflow);

            if(_logger.getPrintDebugInfo() >= 5)
                _logger.log(Logger.DEBUG,
                            "c2, c3, c4 = " + _lagkStates.getCurrentLaggedInflow() + " "
                                + _lagkStates.getCurrentOutflow() + " " + _lagkStates.getCurrentStorage());
        }

        return outflowTimeSeries;
    }

    /**
     * THIS METHOD COMPUTES THE 2*S/DT+O VS O AND 2*S/(DT/4)+O VS. O TABLES NEEDED BY THE ATLANTA K PROCEDURE, AND
     * STORES THESE TABLE IN KvalueArray.
     * 
     * @author ORIGINALLY PROGRAMMED BY GEORGE F. SMITH - HRL DECEMBER 1979; UPDATED MARCH 1982 TO ALLOW METRIC AND
     *         ENGLISH UNITS
     * @see FORTRAN - pina7.f
     * @return KvalueArr
     */
    private ArrayList<Double> computeKbyAtlantaMethod()
    {
        ArrayList<Double> kAtalantaMethodArr = new ArrayList<Double>();
        final ArrayList<Double> KvalueArr = new ArrayList<Double>();

        if(_logger.getPrintDebugInfo() >= 0)
            _logger.log(Logger.DEBUG, "=== Entered computeKbyAtlantaMethod() - pina7.f =========");

        double inflowTimeInterval = _lagkParams.getInflowTimeInterval();
        boolean goto1020 = false;
        double outflowAfterKAtEndTimeStep = 0.0;
        double QBAR = 0.0;
        double delK = 0.0; //time
        double delQ = 0.0; //cms

        while(true)
        {
            if(_logger.getPrintDebugInfo() >= 5)
            {
                _logger.log(Logger.DEBUG, "_lagkParams.getNumPairsKQ() = " + _lagkParams.getNumPairsKQ());
                _logger.log(Logger.DEBUG, "NPKQ - P(18) = " + _lagkParams.getNumPairsKQ());
            }

            if(_lagkParams.getNumPairsKQ() > 0)
            {
                int npos = 0;
                double valueSum = 0.0;
                double outflowAfterKAtStartTimeStep = 0.0;
                int i, j;
                final int k = 0;
                int isegs;
                int ipart;

                for(i = 0; i < _NumPairsKQ; i += 2)
                {
                    if(((i / 2) + 1) >= _lagkParams.getNumPairsKQ())
                    {
                        if(_logger.getPrintDebugInfo() >= 5)
                            _logger.log(Logger.DEBUG, "I = " + i + " .GE. NPKQ =" + _lagkParams.getNumPairsKQ());

                        break;
                    }

                    j = i + 2;

                    if(_logger.getPrintDebugInfo() >= 5)
                        _logger.log(Logger.DEBUG, "\n\n==> J = " + j + "  I = " + i);

                    delK = Math.abs(_paramPairsValueKQ[j] - _paramPairsValueKQ[i]);

                    delQ = Math.abs(_paramPairsValueKQ[j + 1] - _paramPairsValueKQ[i + 1]);

                    if(_logger.getPrintDebugInfo() >= 5)
                    {
                        _logger.log(Logger.DEBUG, "p(" + j + ") = " + _paramPairsValueKQ[j] + "" + "\t\t\tp(" + i
                            + ") = " + _paramPairsValueKQ[i] + "\t==>delK = " + delK);
                        _logger.log(Logger.DEBUG, "p(" + (j + 1) + ") = " + _paramPairsValueKQ[j + 1] + "\tp("
                            + (i + 1) + ") = " + _paramPairsValueKQ[i + 1] + "\t==>delQ = " + delQ);
                    }

                    isegs = 1;

                    if(delK != 0.0)
                        isegs = (int)((delQ + LagKJConstants.C1 * delK) / LagKJConstants.C2 + 1.5);

                    if(isegs > 20)
                        isegs = 20;

                    if(_logger.getPrintDebugInfo() >= 5)
                        _logger.log(Logger.DEBUG, "--> ISEGS = " + isegs);

                    ipart = 0;

                    while(true)
                    {
                        outflowAfterKAtEndTimeStep = _paramPairsValueKQ[i + 1] + delQ * ipart / isegs;

                        kAtalantaMethodArr.add(outflowAfterKAtEndTimeStep);

                        if(_logger.getPrintDebugInfo() >= 5)
                        {
                            _logger.log(Logger.DEBUG, "Q2 = " + outflowAfterKAtEndTimeStep + " p(" + (i + 1) + ")="
                                + _paramPairsValueKQ[i + 1] + " + DELQ = " + delQ + " * IPART = " + ipart
                                + " / ISEGS =" + isegs);
                            _logger.log(Logger.DEBUG, "*** kAtlanta - P(" + (npos * 2) + ") = "
                                + outflowAfterKAtEndTimeStep);
                        }

                        QBAR = (outflowAfterKAtStartTimeStep + outflowAfterKAtEndTimeStep) / 2.0;

                        //Linear interpolate fserc1.f
                        if(_logger.getPrintDebugInfo() >= 5)
                        {
                            _logger.log(Logger.DEBUG, "QBAR = " + QBAR + " (Q1 + Q2)/2 = "
                                + outflowAfterKAtStartTimeStep + " " + outflowAfterKAtEndTimeStep);
                            _logger.log(Logger.DEBUG, "Before fserc1 - QBAR,NPKQ,P(IBK1),Q1,Q2,S = " + QBAR + " "
                                + _lagkParams.getNumPairsKQ() + " " + _paramPairsValueKQ[0] + " "
                                + outflowAfterKAtStartTimeStep + " " + outflowAfterKAtEndTimeStep + " " + valueSum);
                        }

                        _commonMethods.setNewNumPair(0);

                        valueSum += _commonMethods.linearInterpolator(0,
                                                                      QBAR,
                                                                      _lagkParams.getNumPairsKQ(),
                                                                      _paramPairsValueKQ,
                                                                      0)
                            * (outflowAfterKAtEndTimeStep - outflowAfterKAtStartTimeStep);
                        if(inflowTimeInterval == 0)
                            _logger.log(Logger.DEBUG, "????????? inflow timestep = " + inflowTimeInterval);

                        kAtalantaMethodArr.add(2. * valueSum / inflowTimeInterval + outflowAfterKAtEndTimeStep);

                        if(_logger.getPrintDebugInfo() >= 5)
                        {
                            _logger.log(Logger.DEBUG, "After fserc1 - S = " + valueSum + " inflowtsinterval =  "
                                + inflowTimeInterval);
                            _logger.log(Logger.DEBUG, "*** kAtlanta - P(" + (npos * 2 + 1) + ") = "
                                + (2. * valueSum / inflowTimeInterval + outflowAfterKAtEndTimeStep));
                        }

                        outflowAfterKAtStartTimeStep = outflowAfterKAtEndTimeStep;
                        npos++;
                        ipart++;

                        if(_logger.getPrintDebugInfo() >= 5)
                            _logger.log(Logger.DEBUG, "NPOS, IPART, ISEGS = " + npos + " " + ipart + " " + isegs);

                        if(ipart >= isegs)
                            break;

                    }//end while(true)
                }//end for loop NPKQ

                outflowAfterKAtEndTimeStep = _paramPairsValueKQ[i + 1];

                kAtalantaMethodArr.add(outflowAfterKAtEndTimeStep);

                if(_logger.getPrintDebugInfo() >= 5)
                    _logger.log(Logger.DEBUG, "100 - Q2 = P(" + (i + 1) + " )= " + outflowAfterKAtEndTimeStep
                        + " *** kAtlanta - P(" + (npos * 2) + ") =  " + outflowAfterKAtEndTimeStep);

                QBAR = (outflowAfterKAtStartTimeStep + outflowAfterKAtEndTimeStep) / 2.0;

                _commonMethods.setNewNumPair(0);

                valueSum += _commonMethods.linearInterpolator(0,
                                                              QBAR,
                                                              _lagkParams.getNumPairsKQ(),
                                                              _paramPairsValueKQ,
                                                              0)
                    * (outflowAfterKAtEndTimeStep - outflowAfterKAtStartTimeStep);

                kAtalantaMethodArr.add(2. * valueSum / inflowTimeInterval + outflowAfterKAtEndTimeStep);

                if(_logger.getPrintDebugInfo() >= 5)
                    _logger.log(Logger.DEBUG, "*** kAtlanta - P(" + (npos * 2 + 1) + ") = "
                        + (2. * valueSum / inflowTimeInterval + outflowAfterKAtEndTimeStep));

                outflowAfterKAtStartTimeStep = outflowAfterKAtEndTimeStep;

                npos++;

                if(_logger.getPrintDebugInfo() >= 5)
                {
                    _logger.log(Logger.DEBUG, "==> PAIR Q - P(" + (k + 1) + ") = " + _paramPairsValueKQ[k + 1]);
                    _logger.log(Logger.DEBUG, "========= NPOS = " + npos + "=========");
                }

                if(_paramPairsValueKQ[k + 1] != 0.0)
                    kAtalantaMethodArr = _commonMethods.shiftValuesToRight(kAtalantaMethodArr, 0.0, 0.0);

                outflowAfterKAtEndTimeStep = 1.E6;

                kAtalantaMethodArr.add(outflowAfterKAtEndTimeStep);

                if(_logger.getPrintDebugInfo() >= 5)
                {
                    _logger.log(Logger.DEBUG, "'   Q2, IP+2 = " + outflowAfterKAtEndTimeStep + " " + npos);
                    _logger.log(Logger.DEBUG, "*** kAtlanta - P(" + (npos * 2) + ") = " + outflowAfterKAtEndTimeStep);
                }

                QBAR = (outflowAfterKAtStartTimeStep + outflowAfterKAtEndTimeStep) / 2.0;

                _commonMethods.setNewNumPair(0);

                valueSum += _commonMethods.linearInterpolator(-1,
                                                              QBAR,
                                                              _lagkParams.getNumPairsKQ(),
                                                              _paramPairsValueKQ,
                                                              0)
                    * (outflowAfterKAtEndTimeStep - outflowAfterKAtStartTimeStep);

                kAtalantaMethodArr.add(2.0 * valueSum / inflowTimeInterval + outflowAfterKAtEndTimeStep);

                if(_logger.getPrintDebugInfo() >= 5)
                    _logger.log(Logger.DEBUG, "*** kAtlanta - P(" + (npos * 2 + 1) + ") = "
                        + (2.0 * valueSum / inflowTimeInterval + outflowAfterKAtEndTimeStep));

                npos++;
            }//if(_lagkParams.getNumPairsKQ() > 0)
            else
            {
                kAtalantaMethodArr.add(0.0);
                kAtalantaMethodArr.add(0.0);
                kAtalantaMethodArr.add(1.E6);

                final double S = 1.E6 * _lagkParams.getConstantK();

                if(_logger.getPrintDebugInfo() >= 5)
                {
                    _logger.log(Logger.DEBUG, "\n\n++++++++++++++++\nkAtlanta length = 2");
                    _logger.log(Logger.DEBUG, "==> 1.E6 * P(1) = " + _lagkParams.getConstantK() + " S = " + S);
                }

                kAtalantaMethodArr.add(2.0 * S / inflowTimeInterval + 1.E6);
            }

            KvalueArr.addAll(kAtalantaMethodArr);

            if(_ipass == 2 || _lagkParams.getNumPairsKQ() == 0)
                break;

            for(int l = 0; l < _NumPairsKQ; l += 2)
            {
                if((inflowTimeInterval / 2.0) > _paramPairsValueKQ[l])
                {
                    _ipass = 2;
                    inflowTimeInterval /= 4.0;

                    goto1020 = true;
                    kAtalantaMethodArr.clear();
                    break;
                }
            }

            if(goto1020 == false)
                break;

        }//end while(true)

        if(_logger.getPrintDebugInfo() >= 0)
            _logger.log(Logger.DEBUG, "Exit computeKbyAtlantaMethod() function ...");

        return KvalueArr;
    }

    /**
     * THIS METHOD DOES A LINEAR INTERPOLATION FOR THE MULTIPLE INTERCEPT PROBLEM.
     * 
     * @author ORIGINALLY PROGRAMMED BY JANICE M. LEWIS - HRL OCTOBER 1992.
     * @see FORTRAN - fintp7.f
     * @return the value after solve intercept problem
     */
    private double solveInterceptProblem(final double time, final double[] qtValues, final int indexstart)
    {
        if(_logger.getPrintDebugInfo() >= 5)
            _logger.log(Logger.DEBUG, "Entered solveInterceptProblem() function ...");

        final double ratio = (qtValues[indexstart + 2] - qtValues[indexstart])
            / (qtValues[indexstart + 3] - qtValues[indexstart + 1]);

        if(_logger.getPrintDebugInfo() >= 5)
            _logger.log(Logger.DEBUG, "Exit solveInterceptProblem() function ...");

        return (qtValues[indexstart] + ratio * (time - qtValues[indexstart + 1]));
    }

    /**
     * @return the _ipass
     */
    private int getIpass()
    {
        return this._ipass;
    }

}
