package ohd.hseb.ohdmodels.lagkj;

import ohd.hseb.measurement.MeasuringUnitType;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.ModelException;
import ohd.hseb.util.fews.NwsrfsDataTypeMappingReader;
import ohd.hseb.util.fews.ohdmodels.ModelParameters;

/**
 * ChamP 11/25/2014 - FB1617 Cleanup LagK parameter definitions
 */
public class LagKJModelParameters extends ModelParameters
{
    private String _inflowTSId; //TSIDA
    private String _inflowTSType; //DTA
    private int _inflowTimeStep; //ITA
    private String _outflowTSId = ""; //TSITB
    private String _outflowTSType = ""; //DTB
    private int _outflowTimeStep = 0; //ITB
    private int _numPairsLagQ = 0; //JLAG
    private int _numPairsKQ = 0; //JK
    private String _unitsType = "METR"; //METENG  - Default is METRIC (i.e METR or ENGL)
    private double _transmissionLostRecesssionCoef = 0.0; //TLRC
    private double _flowAboveTLRC = 0.0; //QBNTL

    private double[] _pairsValuesLagQ = new double[2];
    private double[] _pairsValuesKQ = new double[2];
    private double _constantLag;
    private double _constantK;
    private boolean _isLagOptionUse = true;
    private boolean _isKOptionUse = true;

    private int _maxPairQTLagCarryover;
    private int _atlantaMethod = 1;
    private boolean _meanQ = false;

    public LagKJModelParameters()
    {
        //empty
    }

    /*
     * (non-Javadoc)
     * @see ohd.hseb.ohdmodels.ModelParameters#extractValuesFromMap()
     */
    @Override
    public void extractValuesFromMap() throws Exception
    {
        //card 1 ========================================

        //Get input time series (QINE or SQIN) (required)
        String newParTag = null;

        newParTag = LagKJParIdentifiers.TSIDA.getNewName();
        if(isParamExisting(newParTag))
            _inflowTSId = getStringDataParameter(newParTag);
        else {
        	_logger.log(Logger.WARNING, "*** WARNING: From OHD-CORE 4.5.a, the old " + LagKJParIdentifiers.TSIDA.getOldName()
                + " parameter is no longer used. Please run the script cleanUpParDefinitions included in the OHD-CORE-CHPS package.");
            _inflowTSId = getStringDataParameter(LagKJParIdentifiers.TSIDA.getOldName());
        }
        newParTag = LagKJParIdentifiers.DTA.getNewName();
        if(isParamExisting(newParTag))
            _inflowTSType = getStringDataParameter(newParTag);
        else {
        	_logger.log(Logger.WARNING, "*** WARNING: From OHD-CORE 4.5.a, the old " + LagKJParIdentifiers.DTA.getOldName()
                + " parameter is no longer used.  Please run the script cleanUpParDefinitions included in the OHD-CORE-CHPS package.");
            _inflowTSType = getStringDataParameter(LagKJParIdentifiers.DTA.getOldName());
        }
        
        newParTag = LagKJParIdentifiers.ITA.getNewName();
        if(isParamExisting(newParTag))
            _inflowTimeStep = getIntDataParameter(newParTag);
        else {
        	_logger.log(Logger.WARNING,"*** WARNING: From OHD-CORE 4.5.a, the old " + LagKJParIdentifiers.ITA.getOldName()
                + " parameter is no longer used.  Please run the script cleanUpParDefinitions included in the OHD-CORE-CHPS package.");
            _inflowTimeStep = getIntDataParameter(LagKJParIdentifiers.ITA.getOldName());
        }
        
        //Get output time series (optional)
        newParTag = LagKJParIdentifiers.DTB.getNewName();
        if(isParamExisting(newParTag))
            _outflowTSType = getStringDataParameter(newParTag);
        else if(isParamExisting(LagKJParIdentifiers.DTB.getOldName()))
        {
            _outflowTSType = getStringDataParameter(LagKJParIdentifiers.DTB.getOldName());
            _logger.log(Logger.WARNING, "*** WARNING: From OHD-CORE 4.5.a, the old " + LagKJParIdentifiers.DTB.getOldName()
                + " parameter is no longer used.  Please run the script cleanUpParDefinitions included in the OHD-CORE-CHPS package.");
        }
        else
        {
            if(NwsrfsDataTypeMappingReader.getNwsrfsDim(_inflowTSType, _logger)
                                          .equalsIgnoreCase(MeasuringUnitType.volume.getName()))
            {
                _logger.log(Logger.ERROR, "*** ERROR - The inflow time series has dimension "
                    + NwsrfsDataTypeMappingReader.getNwsrfsDim(_inflowTSType, _logger)
                    + " the outflow time series MUST be specified.");
                throw new ModelException("*** ERROR - The inflow time series has dimension "
                    + NwsrfsDataTypeMappingReader.getNwsrfsDim(_inflowTSType, _logger)
                    + " the outflow time series MUST be specified.");
            }
        }

        newParTag = LagKJParIdentifiers.TSIDB.getNewName();
        if(isParamExisting(newParTag))
            _outflowTSId = getStringDataParameter(newParTag);
        else if(isParamExisting(LagKJParIdentifiers.TSIDB.getOldName()))
        {
            _logger.log(Logger.WARNING, "*** WARNING: From OHD-CORE 4.5.a, the old " + LagKJParIdentifiers.TSIDB.getOldName()
                + " parameter is no longer used.  Please run the script cleanUpParDefinitions included in the OHD-CORE-CHPS package.");
            _outflowTSId = getStringDataParameter(LagKJParIdentifiers.TSIDB.getOldName());
        }
    
        newParTag = LagKJParIdentifiers.ITB.getNewName();
        if(isParamExisting(newParTag))
            _outflowTimeStep = getIntDataParameter(newParTag);
        else
        {
            _logger.log(Logger.WARNING, "*** WARNING: From OHD-CORE 4.5.a, the old " + LagKJParIdentifiers.ITB.getOldName()
                + " parameter is no longer used.  Please run the script cleanUpParDefinitions included in the OHD-CORE-CHPS package.");
            _outflowTimeStep = getIntDataParameter(LagKJParIdentifiers.ITB.getOldName());
        }
        
        //Get # of pairs of Lag and Q (JLAG)
        newParTag = LagKJParIdentifiers.JLAG.getNewName();
        if(isParamExisting(newParTag))
            _numPairsLagQ = getIntDataParameter(newParTag);
        else
        {
            _logger.log(Logger.WARNING, "*** WARNING: From OHD-CORE 4.5.a, the old " + LagKJParIdentifiers.JLAG.getOldName()
                + " parameter is no longer used.  Please run the script cleanUpParDefinitions included in the OHD-CORE-CHPS package.");
            _numPairsLagQ = getIntDataParameter(LagKJParIdentifiers.JLAG.getOldName());
        }
        
        //Get # of pairs of K and Q (JK)
        newParTag = LagKJParIdentifiers.JK.getNewName();
        if(isParamExisting(newParTag))
            _numPairsKQ = getIntDataParameter(newParTag);
        else
        {
            _logger.log(Logger.WARNING, "*** WARNING: From OHD-CORE 4.5.a, the old " + LagKJParIdentifiers.JK.getOldName()
                + " parameter is no longer used.  Please run the script cleanUpParDefinitions included in the OHD-CORE-CHPS package.");
            _numPairsKQ = getIntDataParameter(LagKJParIdentifiers.JK.getOldName());
        }

        //Get unit type
        newParTag = LagKJParIdentifiers.METENG.getNewName();
        if(isParamExisting(newParTag))
            _unitsType = getStringDataParameter(newParTag);
        else
        {
            _logger.log(Logger.WARNING, "*** WARNING: From OHD-CORE 4.5.a, the old " + LagKJParIdentifiers.METENG.getOldName()
                + " parameter is no longer used.  Please run the script cleanUpParDefinitions included in the OHD-CORE-CHPS package.");
            _unitsType = getStringDataParameter(LagKJParIdentifiers.METENG.getOldName());
        }

        //Get Transmission Lost recession coefficient (TLRC)
        newParTag = LagKJParIdentifiers.TLRC.getNewName();
        if(isParamExisting(newParTag))
            _transmissionLostRecesssionCoef = getDoubleDataParameter(newParTag);
        else if(isParamExisting(LagKJParIdentifiers.TLRC.getOldName()))
        {
            _logger.log(Logger.WARNING, "*** WARNING: From OHD-CORE 4.5.a, the old " + LagKJParIdentifiers.TLRC.getOldName()
                + " parameter is no longer used.  Please run the script cleanUpParDefinitions included in the OHD-CORE-CHPS package.");
            _transmissionLostRecesssionCoef = getDoubleDataParameter(LagKJParIdentifiers.TLRC.getOldName());
        }

        //Get flow above TLRC(QBNTL)
        newParTag = LagKJParIdentifiers.QBNTL.getNewName();
        if(isParamExisting(newParTag))
            _flowAboveTLRC = getDoubleDataParameter(newParTag);
        else if(isParamExisting(LagKJParIdentifiers.QBNTL.getOldName()))
        {
            _logger.log(Logger.WARNING, "*** WARNING: From OHD-CORE 4.5.a, the old " + LagKJParIdentifiers.QBNTL.getOldName()
                + " parameter is no longer used.  Please run the script cleanUpParDefinitions included in the OHD-CORE-CHPS package.");
            _flowAboveTLRC = getDoubleDataParameter(LagKJParIdentifiers.QBNTL.getOldName());
        }

        //Card 2 ==============================
        if(_numPairsLagQ > 0) //#pairs of LagQ used to define the variable Lag vs Q curve
        {
            _pairsValuesLagQ = getDoubleArrayFromTableParameter("LAGQ_PAIRS");

            if(_numPairsLagQ != (_pairsValuesLagQ.length / 2))
                throw new ModelException("Check parameter xml file - NUMBER_OF_LAGQ_PAIRS = " + _numPairsLagQ
                    + " does not match with number of LAGQ_PAIRS = " + (_pairsValuesLagQ.length / 2));

            //_numPairsLagQ = _pairsValuesLagQ.length / 2;
        }
        else
        //constant lag will be used and if it eq to 0.0 then Lag option turn off
        {
            newParTag = LagKJParIdentifiers.SETLAG.getNewName();
            if(isParamExisting(newParTag))
                _constantLag = getDoubleDataParameter(newParTag);
            else if(isParamExisting(LagKJParIdentifiers.SETLAG.getOldName()))
            {
                _logger.log(Logger.WARNING, "*** WARNING: From OHD-CORE 4.5.a, the old "
                    + LagKJParIdentifiers.SETLAG.getOldName() + " parameter is no longer used.  Please run the script cleanUpParDefinitions included in the OHD-CORE-CHPS package.");
                _constantLag = getDoubleDataParameter(LagKJParIdentifiers.SETLAG.getOldName());
            }
            else
                throw new ModelException("Check parameter xml file - CONSTANT_LAG_VALUE should have existed because NUMBER_OF_LAGQ_PAIRS = 0");

            if(_constantLag == 0.0)
                _isLagOptionUse = false;
        }

        //Card 3 =============================
        if(_numPairsKQ > 0) //#pairs of KQ used to define the variable K vs Q curve
        {
            _pairsValuesKQ = getDoubleArrayFromTableParameter("KQ_PAIRS");

            if(_numPairsKQ != (_pairsValuesKQ.length / 2))
                throw new ModelException("Check parameter xml file - NUMBER_OF_KQ_PAIRS = " + _numPairsKQ
                    + " does not match with number of KQ_PAIRS =" + (_pairsValuesKQ.length / 2));

            //_numPairsKQ = _pairsValuesKQ.length / 2;
        }
        else
        {
            newParTag = LagKJParIdentifiers.SETK.getNewName();
            if(isParamExisting(newParTag))
                _constantK = getDoubleDataParameter(newParTag);
            else if(isParamExisting(LagKJParIdentifiers.SETK.getOldName()))
            {
                _logger.log(Logger.WARNING, "*** WARNING: From OHD-CORE 4.5.a, the old " + LagKJParIdentifiers.SETK.getOldName()
                    + " parameter is no longer used.  Please run the script cleanUpParDefinitions included in the OHD-CORE-CHPS package.");
                _constantK = getDoubleDataParameter(LagKJParIdentifiers.SETK.getOldName());
            }
            else
                throw new ModelException("Check parameter xml files - CONSTANT_K_VALUE should have existed because NUMBER_OF_KQ_PAIRS = 0");

            if(_constantK == 0.0)
                _isKOptionUse = false;
        }

        //Card 4 ==============================

        //Card 5 - contains pairs of Q and time carryover values
        int maxPairQTLagCarryover = 0;

        if(_isLagOptionUse == true)
        {
            _logger.log(Logger.DEBUG, "LAG OPTION ENABLED");

            //Determine maximum LAG carryover
            if(_numPairsLagQ == 0)
            {
                maxPairQTLagCarryover = (int)(_constantLag / this.getInflowTimeInterval()) + 2;

                if(_logger.getPrintDebugInfo() >= 5)
                    _logger.log(Logger.DEBUG, "NUMBER_OF_LAGQ_PAIRS = " + this.getNumPairsLagQ() + " P(20) = "
                        + _constantLag + " MXCOL = " + maxPairQTLagCarryover);
            }
            else
            {
                double maxLagQValues = -1.0;

                for(int i = 0; i < _numPairsLagQ * 2; i += 2)
                {
                    if(i == _numPairsLagQ * 2)
                        break;

                    if(_logger.getPrintDebugInfo() >= 5)
                        _logger.log(Logger.DEBUG, "+++ " + _pairsValuesLagQ[i]);

                    if(_pairsValuesLagQ[i] > maxLagQValues)
                        maxLagQValues = _pairsValuesLagQ[i];
                }

                maxPairQTLagCarryover = (int)(maxLagQValues / this.getInflowTimeInterval()) + 2;

                if(_logger.getPrintDebugInfo() >= 5)
                    _logger.log(Logger.DEBUG, "NUMBER_OF_LAGQ_PAIRS = " + _numPairsLagQ + " X = " + maxLagQValues
                        + " MXCOL = " + maxPairQTLagCarryover);
            }
        }

        // Set number pair QT/Lag Carryover to max number pair QT/Lag carryover calculated.
        _maxPairQTLagCarryover = maxPairQTLagCarryover;

        this.setMaxPairQTLagCarryover(_maxPairQTLagCarryover);

        // Validate parameters
        this.validateParams();

        if(_logger.getPrintDebugInfo() > 0)
            _logger.log(Logger.DEBUG, "Completed extracting param from map ...");
    }

    /**
     * @throws ModelException
     * @throws Exception
     */
    private void validateParams() throws ModelException, Exception
    {
        String dimension = null;

        double cfsm = 1.0;

        if(_logger.getPrintDebugInfo() >= 5)
        {
            _logger.log(Logger.DEBUG, "Start validate parameters........");
            _logger.log(Logger.DEBUG, "TSIDA=" + _inflowTSId + " DTA=" + _inflowTSType + " ITA=" + _inflowTimeStep
                + " TSIDB=" + _outflowTSId + " DTB=" + _outflowTSType + " ITB=" + _outflowTimeStep + " JLAG="
                + _numPairsLagQ + " JK=" + _numPairsKQ + " METENG=" + this.getUnitsType() + " TLRC="
                + _transmissionLostRecesssionCoef + " QBNTL=" + _flowAboveTLRC);
        }

        if(this.getUnitsType().equals("ENGL"))
            cfsm = LagKJConstants.CFSM;
//            cfdm = LagKJConstants.CFDM; //unused???

        //Check the outflow time series 
        if((!this.getOutflowTimeSeriesId().isEmpty() && !this.getOutflowTimeSeriesType().isEmpty())
            && this.getOutflowTimeInterval() == 0)
        {
            this.setOutflowTimeInterval(this.getInflowTimeInterval());
        }
        else if(this.getOutflowTimeSeriesId().isEmpty() && this.getOutflowTimeSeriesType().isEmpty())
        {
            this.setOutflowTimeSeriesId(this.getInflowTimeSeriesId());
            this.setOutputTimeSeriesType(this.getInflowTimeSeriesType());
            this.setOutflowTimeInterval(this.getInflowTimeInterval());
        }

        //Transmission lost recession coefficient out of valid range - set to zero
        if(this.getTransmissionLossRecessionCoef() < 0.0 || this.getTransmissionLossRecessionCoef() > 1.0)
        {
            _logger.log(Logger.WARNING,
                        "**WARNING** VALUE OF THE RECESSION COEFFICIENT FOR THE FT. WORTH TRANSMISSION LOSS "
                            + "COMPUTATIONS (" + this.getTransmissionLossRecessionCoef()
                            + ") CANNOT BE LESS THAN ZERO OR GREATER THAN ONE. VALUE WILL BE SET TO ZERO.");

            this.setTransmissionLossRecessionCoef(0.0);
        }

        if(_logger.getPrintDebugInfo() >= 5)
            _logger.log(Logger.DEBUG, "1 - TRANSMISSION_LOSS_COEFFICIENT = " + this.getTransmissionLossRecessionCoef());

        //Flow above of transmission lost recession coefficient less than zero - set to zero
        if(this.getFlowAboveTLRC() < 0.0)
        {
            _logger.log(Logger.WARNING, "**WARNING** MINIMUM FLOW ENTERED FOR THE "
                + " FT. WORTH TRANSMISSION LOSS COMPUTATIONS (" + this.getFlowAboveTLRC()
                + ") CANNOT BE LESS THAN ZERO. VALUE WILL BE SET TO ZERO.");

            this.setFlowAboveTLRC(0.0);
        }
        else
            //convert cfs to cms if unit type is ENGL
            this.setFlowAboveTLRC(this.getFlowAboveTLRC() / cfsm);

        if(_logger.getPrintDebugInfo() >= 5)
            _logger.log(Logger.DEBUG, "2 - TRANSMISSION_LOSS_THRESHOLD_FLOW = " + this.getFlowAboveTLRC());

        //Set atlanta RFC method - default 1 - method is used
        if(this.getTransmissionLossRecessionCoef() > 0.0)
            this.setAlantaMethod(0); //atlanta method not use

        if(_logger.getPrintDebugInfo() >= 5)
            _logger.log(Logger.DEBUG, "3 - IATL = " + this.getAtlantaMethod());

        //Check if LAG/K is being used within a RES-SNGL
        if(this.getInflowTimeSeriesId().equalsIgnoreCase("RES-SNGL"))
        {
            _logger.log(Logger.DEBUG, "RES-SNG identify.........");
            this.setInflowTimeInterval(0);
            this.setOutflowTimeInterval(this.getInflowTimeInterval());
            if(this.getInflowTimeSeriesType().equalsIgnoreCase("QME"))
            {
                _meanQ = true;
                this.setInflowTimeSeriesType("QINE"); //dimension discharge L3/T
            }
        }
        else
        {
            _meanQ = false;
            dimension = NwsrfsDataTypeMappingReader.getNwsrfsDim(this.getInflowTimeSeriesType(), _logger);

            if(!(dimension.equalsIgnoreCase(MeasuringUnitType.discharge.getName()) || dimension.equalsIgnoreCase(MeasuringUnitType.volume.getName())))
            {
                _logger.log(Logger.ERROR, "**ERROR** INVALID DIMENSIONS (" + dimension
                    + "') FOR INFLOW TIME SERIES. VALID DIMENSIONS ARE L3/T AND L3.");
                throw new ModelException("**ERROR** INVALID DIMENSIONS (" + dimension
                    + "') FOR INFLOW TIME SERIES. VALID DIMENSIONS ARE L3/T AND L3.");
            }

            if(dimension.equalsIgnoreCase(MeasuringUnitType.volume.getName()))
                _meanQ = true;

            if(_logger.getPrintDebugInfo() >= 5)
                _logger.log(Logger.DEBUG, "### 2 MEANQ, IATL = " + _meanQ + "   " + this.getAtlantaMethod());

            if(!(_meanQ == false || this.getAtlantaMethod() == 1))
            {
                _logger.log(Logger.ERROR,
                            "**ERROR** CANNOT USE PERIOD AVERAGE INFLOW WITH 'MCP2 ATTENUATION (K) TECHNIQUE.");
                throw new ModelException("**ERROR** CANNOT USE PERIOD AVERAGE INFLOW WITH 'MCP2 ATTENUATION (K) TECHNIQUE.");
            }

            if(_logger.getPrintDebugInfo() >= 5)
                _logger.log(Logger.DEBUG, "### 3 ITB, MEANQ = " + this.getOutflowTimeInterval() + "  " + _meanQ);

            if(!(this.getOutflowTimeInterval() > 0 || _meanQ == false))
            {
                _logger.log(Logger.ERROR,
                            "**ERROR** CANNOT WRITE OUTFLOW TIME SERIES OVER INFLOW TIME SERIES WHEN INFLOW IS PERIOD AVERAGED.");
                throw new ModelException("**ERROR** CANNOT WRITE OUTFLOW TIME SERIES OVER INFLOW TIME SERIES WHEN INFLOW IS PERIOD AVERAGED.");
            }

            if(!(this.getInflowTimeInterval() <= this.getOutflowTimeInterval() || this.getOutflowTimeInterval() == 0))
            {
                _logger.log(Logger.ERROR, "**ERROR** INFLOW TIME STEP (" + this.getInflowTimeInterval()
                    + ") MUST NOT BE GREATER THAN THE OUTFLOW TIME STEP (" + this.getOutflowTimeInterval() + ").");
                throw new ModelException("**ERROR** INFLOW TIME STEP (" + this.getInflowTimeInterval()
                    + ") MUST NOT BE GREATER THAN THE OUTFLOW TIME STEP (" + this.getOutflowTimeInterval() + ").");
            }
        } //end if RES-SNGL

        if(_logger.getPrintDebugInfo() >= 5)
            _logger.log(Logger.DEBUG,
                        "### 6 jlag = " + this.getNumPairsLagQ() + " NUMBER_OF_KQ_PAIRS = " + this.getNumPairsKQ());

        if(this.getNumPairsLagQ() < 0 || this.getNumPairsKQ() < 0)
        {
            _logger.log(Logger.ERROR,
                        "**ERROR** NUMBER OF PAIRS OF VALUES IN THE " + "(LAG,Q) TABLE (" + this.getNumPairsLagQ()
                            + ") OR (K,Q) TABLE (" + this.getNumPairsKQ() + ") CANNOT BE LESS THAN ZERO.");
            throw new ModelException("**ERROR** NUMBER OF PAIRS OF VALUES IN THE " + "(LAG,Q) TABLE ("
                + this.getNumPairsLagQ() + ") OR (K,Q) TABLE (" + this.getNumPairsKQ() + ") CANNOT BE LESS THAN ZERO.");
        }
        else
        {
            if(_logger.getPrintDebugInfo() >= 5)
            {
                _logger.log(Logger.DEBUG,
                            "p2 = " + this.getInflowTimeSeriesId() + " p4 = " + this.getInflowTimeSeriesType()
                                + " p5 = " + this.getInflowTimeInterval());
                _logger.log(Logger.DEBUG,
                            "p6 = " + this.getOutflowTimeSeriesId() + " p8 = " + this.getOutflowTimeSeriesType()
                                + " p9 = " + this.getOutflowTimeInterval());
                _logger.log(Logger.DEBUG, "p10 = " + this.getUnitsType());
                _logger.log(Logger.DEBUG, "p19 NUMBER_OF_LAGQ_PAIRS = " + this.getNumPairsLagQ());
            }

            //Check Variable LAG values
            int npairs;

            if(this.getNumPairsLagQ() > 0)
            {
                npairs = 1;
                for(int i = 1; i < this.getNumPairsLagQ() * 2; i += 2)
                {
                    if((i < (this.getNumPairsLagQ() * 2) - 1) && (_pairsValuesLagQ[i] > _pairsValuesLagQ[i + 2]))
                    {
                        _logger.log(Logger.ERROR, "**ERROR** DISCHARGES IN THE LAG VS Q TABLE MUST "
                            + "BE IN ASCENDING ORDER.");

                        throw new ModelException("**ERROR** DISCHARGES IN THE LAG VS Q TAB" + "BE IN ASCENDING ORDER.");
                    }

                    // Convert cfs to cms if unit type is ENGL
                    if(this.getUnitsType().equalsIgnoreCase("ENGL"))
                    {
                        if(_logger.getPrintDebugInfo() >= 5)
                            _logger.log(Logger.DEBUG, "before conv LAG/Q to cms p(ip+i)= " + _pairsValuesLagQ[i]);

                        _pairsValuesLagQ[i] = _pairsValuesLagQ[i] / cfsm;

                        if(_logger.getPrintDebugInfo() >= 5)
                            _logger.log(Logger.DEBUG, "after conv LAG/Q to cms p(" + i + ")= " + _pairsValuesLagQ[i]);
                    }

                    if(_pairsValuesLagQ[i] > 1.E6)
                    {
                        _logger.log(Logger.WARNING, "**WARNING** FLOW FOR PAIR " + npairs
                            + " IN THE LAG VS Q TABLE WAS ENTERED AS " + _pairsValuesLagQ[i] + " CMS.\n"
                            + "THIS VALUE HAS BEEN RESET TO 1000000 CMS.");

                        _pairsValuesLagQ[i] = 1.E6;
                    }

                    if(i == (this.getNumPairsLagQ() * 2) - 1)
                        break;

                    npairs++;
                }
            }

            // Check Variable K values
            if(_logger.getPrintDebugInfo() >= 5)
                _logger.log(Logger.DEBUG,
                            "NUMBER_OF_KQ_PAIRS = " + this.getNumPairsKQ() + " XITA = " + this.getInflowTimeInterval());

            if(this.getNumPairsKQ() > 0)
            {
                npairs = 1;
                boolean firstpair = true;

                for(int i = 1; i < this.getNumPairsKQ() * 2; i += 2)
                {
                    if(firstpair) //Fortran check only for first pair???? - It should be check all
                    {
                        if((i < (this.getNumPairsKQ() * 2) - 1) && (_pairsValuesKQ[i] > _pairsValuesKQ[i + 2]))
                        {
                            _logger.log(Logger.ERROR,
                                        "**ERROR** DISCHARGES IN THE K VS Q TABLE MUST BE IN ASCENDING ORDER.");
                            throw new ModelException("**ERROR** DISCHARGES IN THE K VS Q TABLE MUST BE IN ASCENDING ORDER.");
                        }
                        firstpair = false;
                    }

                    // Convert cfs to cms if unit type is ENGL
                    if(this.getUnitsType().equalsIgnoreCase("ENGL"))
                    {
                        if(_logger.getPrintDebugInfo() >= 5)
                            _logger.log(Logger.DEBUG, "before KQ conv to cms = " + _pairsValuesKQ[i]);

                        _pairsValuesKQ[i] = _pairsValuesKQ[i] / cfsm;

                        if(_logger.getPrintDebugInfo() >= 5)
                            _logger.log(Logger.DEBUG, "after KQ conv to cms = " + _pairsValuesKQ[i]);
                    }

                    if(_pairsValuesKQ[i] > 1.E6)
                    {
                        _logger.log(Logger.WARNING, "**WARNING** FLOW FOR PAIR " + npairs
                            + " IN THE K VS Q TABLE WAS ENTERED AS " + _pairsValuesKQ[i] + " CMS.\n"
                            + "THIS VALUE HAS BEEN RESET TO 1000000 CMS.");

                        _pairsValuesKQ[i] = 1.E6;
                    }

                    if(i == (this.getNumPairsKQ() * 2) - 1)
                        break;

                    npairs++;
                }

                if(_logger.getPrintDebugInfo() >= 5)
                    _logger.log(Logger.DEBUG, "### 9 MEANQ = " + _meanQ);

                if(_meanQ == true)
                {
                    final double halfRoutingInterval = this.getInflowTimeInterval() / 2.;
                    npairs = 1;

                    for(int i = 0; i < this.getNumPairsKQ() * 2; i += 2)
                    {
                        if(_pairsValuesKQ[i] < halfRoutingInterval)
                        {
                            _logger.log(Logger.WARNING,
                                        "**WARNING** VARIABLE K FOR PAIR "
                                            + npairs
                                            + " IN THE K VS Q TABLE HAS BEEN RESET TO "
                                            + halfRoutingInterval
                                            + ".  THIS VALUE CANNOT BE LESS THAN ONE HALF THE ROUTING INTERVAL WHEN PERIOD AVERAGE FLOW IS INPUT.");

                            _pairsValuesKQ[i] = halfRoutingInterval;
                        }
                        npairs++;
                    }
                }
            }//end if(_numPairsKQ > 0)
             //Check constant K value , NO pairs Value KQ ===================
            else
            {
                if(_logger.getPrintDebugInfo() >= 5)
                {
                    _logger.log(Logger.DEBUG,
                                "### 8 P(22 or 27) = " + this._constantK + " XITA = " + this.getInflowTimeInterval()
                                    + " IATL = " + this.getAtlantaMethod());
                }

                if(this.getConstantK() != 0.0)
                {
                    double timeStep = this.getInflowTimeInterval();

                    if(this.getAtlantaMethod() == 0)
                        timeStep /= 2.0;

                    if(this.getConstantK() < (timeStep / 2.0))
                    {
                        double setK = 0.;

                        if(_meanQ == true)
                            setK = timeStep / 2.0;

                        if(this.getConstantK() > (timeStep / 4.0))
                            setK = timeStep / 2.0;

                        _logger.log(Logger.WARNING, "**WARNING** CONSTANT K VALUE READ IN (" + this.getConstantK()
                            + ") IS LESS THAN ONE HALF OF THE COMPUTATIONAL INTERVAL VALUE HAS BEEN SET TO " + setK);

                        this.setConstantK(setK);
                    }
                }
            }//end check constant K
        }//end elseif(this.getNumPairsLagQ() >= 0 || this.getNumPairsKQ() >= 0) block

        if(_logger.getPrintDebugInfo() > 0)
            _logger.log(Logger.DEBUG, "completed validate parameters....");
    }

    /**
     * @return the _inflowTSId
     */
    String getInflowTimeSeriesId()
    {
        return _inflowTSId;
    }

    /**
     * @return the _inflowTSType
     */
    String getInflowTimeSeriesType()
    {
        return _inflowTSType;
    }

    /**
     * @param dta
     */
    void setInflowTimeSeriesType(final String dta)
    {
        this._inflowTSType = dta;
    }

    /**
     * @return the _inflowTimeStep
     */
    int getInflowTimeInterval()
    {
        return _inflowTimeStep;
    }

    /**
     * @param ita
     */
    void setInflowTimeInterval(final int ita)
    {
        this._inflowTimeStep = ita;
    }

    /**
     * @return the _outflowTSId
     */
    String getOutflowTimeSeriesId()
    {
        return _outflowTSId;
    }

    /**
     * @param tsId
     */
    void setOutflowTimeSeriesId(final String tsId)
    {
        this._outflowTSId = tsId;
    }

    /**
     * @return the _outflowTSType
     */
    String getOutflowTimeSeriesType()
    {
        return _outflowTSType;
    }

    /**
     * @param tsType
     */
    void setOutputTimeSeriesType(final String tsType)
    {
        this._outflowTSType = tsType;
    }

    /**
     * @return the _outflowTimeStep
     */
    int getOutflowTimeInterval()
    {
        return _outflowTimeStep;
    }

    /**
     * @param timestep
     */
    void setOutflowTimeInterval(final int timestep)
    {
        this._outflowTimeStep = timestep;
    }

    /**
     * @return the _numPairsLagQ (JLAG)
     */
    int getNumPairsLagQ()
    {
        return _numPairsLagQ;
    }

    /**
     * @return the _pairsValuesLagQ[]
     */
    double[] getPairsValuesLagQ()
    {
        return _pairsValuesLagQ;
    }

    /**
     * @return the _numPairsKQ (JK)
     */
    int getNumPairsKQ()
    {
        return _numPairsKQ;
    }

    /**
     * @return the pairsValuesKQ[]
     */
    double[] getPairsValuesKQ()
    {
        return _pairsValuesKQ;
    }

    /**
     * @return the _transmissionLostRecesssionCoef
     */
    double getTransmissionLossRecessionCoef()
    {
        return _transmissionLostRecesssionCoef;
    }

    /**
     * @param tlrc
     */
    void setTransmissionLossRecessionCoef(final double tlrc)
    {
        this._transmissionLostRecesssionCoef = tlrc;
    }

    /**
     * @return the _flowAboveTLRC
     */
    double getFlowAboveTLRC()
    {
        return _flowAboveTLRC;
    }

    /**
     * @param qbntl
     */
    void setFlowAboveTLRC(final double qbntl)
    {
        this._flowAboveTLRC = qbntl;
    }

    /**
     * @return the _unitsType
     */
    String getUnitsType()
    {
        return _unitsType;
    }

    /**
     * @return the _isLagOptionUse
     */
    boolean isLagOptEnable()
    {
        return _isLagOptionUse;
    }

    /**
     * @return the _isKOptionUse
     */
    boolean isKOptEnable()
    {
        return _isKOptionUse;
    }

    /**
     * @return the _constantLag
     */
    double getConstantLag()
    {
        return _constantLag;
    }

    /**
     * @return the conatantK
     */
    double getConstantK()
    {
        return _constantK;
    }

    void setConstantK(final double setK)
    {
        this._constantK = setK;
    }

    /**
     * @return the atlantaMethod
     */
    int getAtlantaMethod()
    {
        return _atlantaMethod;
    }

    /**
     * @param iatl
     */
    void setAlantaMethod(final int iatl)
    {
        this._atlantaMethod = iatl;
    }

    /**
     * @return the _meanQ
     */
    boolean getMeanQ()
    {
        return _meanQ;
    }

    /**
     * @return _maxPairQTLagCarryover
     */
    int getMaxPairQTLagCarryover()
    {
        return this._maxPairQTLagCarryover;
    }

    /**
     * @param maxpair
     */
    void setMaxPairQTLagCarryover(final int maxpair)
    {
        _maxPairQTLagCarryover = maxpair;

    }

}
