package ohd.hseb.ohdmodels.lagkj;

import java.util.ArrayList;
import java.util.Arrays;

import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.ModelException;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.Parameters;
import ohd.hseb.util.fews.ohdmodels.ModelParameters;
import ohd.hseb.util.fews.ohdmodels.ModelState;

/**
 * ChamP 11/25/2014 - FB1617 Cleanup LagK parameter definitions
 * ChamP 05/22/2019 - RM54565 Lag/K mod error when default lag is zero
 */
public class LagKJModelState extends ModelState
{

    private String _stateUnits;
    private int _lengthCarryoverArray;
    private double _currentLaggedInflow;
    private double _currentOutflow;
    private double _currentStorage;
    private int _pairQTLagCarryover = 0;
    private double[] _LagDischargeCarryoverValues;
    private boolean _doCarryOver=false;

    public LagKJModelState()
    {
        super();
    }

    public LagKJModelState(final int whichMap)
    {
        super(whichMap);
    }

    /*
     * (non-Javadoc)
     * @see ohd.hseb.util.fews.State#extractValuesFromMap()
     */
    @Override
    public void extractValuesFromMap() throws Exception
    {
        _stateUnits = getStateAsString("UNIT");
        _lengthCarryoverArray = (int)getDoubleDataState(LagKJConstants.LENGTH_CARRYOVER_ARRAY);
        _currentLaggedInflow = getDoubleDataState(LagKJConstants.INFLOW_BEGINNING_ROUTING_INTERVAL);
        _currentOutflow = getDoubleDataState(LagKJConstants.OUTFLOW_BEGINNING_ROUTING_INTERVAL);
        _currentStorage = getDoubleDataState(LagKJConstants.STORAGE_BEGINNING_ROUTING_INTERVAL);

        if(_lengthCarryoverArray > LagKJConstants.DEFAULT_LENGTH_CARRYOVER)
            _pairQTLagCarryover = (int)getDoubleDataState(LagKJConstants.QT_PAIRS);

              
        if(_pairQTLagCarryover < 0)
        {
            _logger.log(Logger.ERROR, "**ERROR** DEFAULT CARRYOVER FLAG (" + this.getNumPairsQTCarryover()
                + " ) CANNOT BE LESS THAN ZERO.");
            throw new ModelException("**ERROR** DEFAULT CARRYOVER FLAG (" + this.getNumPairsQTCarryover()
                + " ) CANNOT final BE LESS THAN ZERO.");
        }
        // Added this else block for RM54565 - If constantLag applies MODs, but statsI.txt PAIR_QT_LAG_CARRYOVER=0, 
        // then model fails. Better initialize default to 2-pairs than forecaster has to modify the statesI.txt file.
        else if ((_doCarryOver == true) && (_pairQTLagCarryover == 0 && _lengthCarryoverArray == LagKJConstants.DEFAULT_LENGTH_CARRYOVER))
        {
        	final double initQTCarryoverValues[]  = new double[2 * 2]; //initial lag carryover for 2-pairs
            Arrays.fill(initQTCarryoverValues, 0.0);
            this.setLengthCarryoverArr(LagKJConstants.DEFAULT_LENGTH_CARRYOVER + 4);          
            this.setNumPairsQTCarryover(2);
            this.setPairsQTCarryoverValues(initQTCarryoverValues);  
            _logger.log(Logger.INFO, "The default Lag is zero in params_previous.xml and PAIR_QT_LAG_CARRYOVER=0 in StatesI.txt.  To make a Lag mod, the state Q and Time carryover values are initialized to 0.0 before transferring LAG carryover.");

        }
        else if(_pairQTLagCarryover > 0 && _lengthCarryoverArray > LagKJConstants.DEFAULT_LENGTH_CARRYOVER)
        {
            if(_pairQTLagCarryover > 0)
            {
                _LagDischargeCarryoverValues = new double[_pairQTLagCarryover * 2];
                int j = 0;
                for(int i = 0; i < _pairQTLagCarryover; i++)
                {
                    final String dischargeStr = "DISCHARGE#" + i;
                    final String lagStr = "LAG#" + i;

                    _LagDischargeCarryoverValues[j] = getDoubleDataState(dischargeStr);
                    _LagDischargeCarryoverValues[j + 1] = getDoubleDataState(lagStr);

                    j += 2;
                }
            }
        }
    }

    /**
     * This function validates the states
     * 
     * @param params
     * @throws Exception
     */
    void validateState(final Parameters params, boolean doCarryOverTransfer) throws Exception
    {
        if(_logger.getPrintDebugInfo() > 0)
            _logger.log(Logger.DEBUG, "Start validate states.........");

        final LagKJModelParameters lagkParams = (LagKJModelParameters)params;

        // Extract state values from map 
        _doCarryOver = doCarryOverTransfer;
        this.extractValuesFromMap();

        // Read in Lag/QT carryover values from statesI.txt
        if(lagkParams.getNumPairsLagQ() >= 0 && lagkParams.getNumPairsKQ() >= 0)
        {
            if(_logger.getPrintDebugInfo() >= 5)
            {
                _logger.log(Logger.DEBUG, "### 10 NDFLT = " + this.getNumPairsQTCarryover());
                _logger.log(Logger.DEBUG, "==> STATES <== P(17) = " + this.getNumPairsQTCarryover());
            }

            if(lagkParams.isLagOptEnable())
            {
                //Reset max number of pair QT Lag carryover values
                if(lagkParams.getMaxPairQTLagCarryover() >= this.getNumPairsQTCarryover())
                {
                    final double newQTCarryoverValues[] = new double[lagkParams.getMaxPairQTLagCarryover() * 2];
                    Arrays.fill(newQTCarryoverValues, 0.0);

                    if(_logger.getPrintDebugInfo() >= 0)
                        _logger.log(Logger.DEBUG,
                                    "Number of lagk carryovers changed from " + this.getNumPairsQTCarryover() + " to "
                                        + lagkParams.getMaxPairQTLagCarryover());

                    if(this.getNumPairsQTCarryover() * 2 > 0)
                        for(int i = 0; i < this.getNumPairsQTCarryover() * 2; i++)
                            newQTCarryoverValues[i] = this.getPairsQTCarryoverValues()[i];

                    this.setPairsQTCarryoverValues(newQTCarryoverValues);
                    this.setLengthCarryoverArr(LagKJConstants.DEFAULT_LENGTH_CARRYOVER
                        + lagkParams.getMaxPairQTLagCarryover() * 2);
                    this.setNumPairsQTCarryover(lagkParams.getMaxPairQTLagCarryover());
                }
                else
                {
                    // Fill in default LAG carryover values to zero
                    if((lagkParams.isLagOptEnable() == false) && (lagkParams.isKOptEnable() == true))
                        this.setNumPairsQTCarryover(0);
                }
            }//if(lagkParams.isLagOptEnable)
        }

        if(_logger.getPrintDebugInfo() > 0)
            _logger.log(Logger.DEBUG, "Completed validate states...........");
    }

    /**
     * Over-written the super class method. This method is used to output states into the generated statesO.txt file.
     * The units are always in METRIC.
     */
    /*
     * (non-Javadoc)
     * @see ohd.hseb.util.fews.State.#getStringsFromState()
     */
    @Override
    protected String getStringsFromState()
    {
        final StringBuilder resultStr = new StringBuilder();
        resultStr.append(OHDConstants.UNIT_TAG)
                 .append("=")
                 .append(OHDConstants.UNIT_METRIC)
                 .append(OHDConstants.NEW_LINE); //units are in METRIC always
        resultStr.append(LagKJConstants.INFLOW_BEGINNING_ROUTING_INTERVAL)
                 .append("=")
                 .append(_currentLaggedInflow)
                 .append(OHDConstants.NEW_LINE);
        resultStr.append(LagKJConstants.OUTFLOW_BEGINNING_ROUTING_INTERVAL)
                 .append("=")
                 .append(_currentOutflow)
                 .append(OHDConstants.NEW_LINE);
        resultStr.append(LagKJConstants.STORAGE_BEGINNING_ROUTING_INTERVAL)
                 .append("=")
                 .append(_currentStorage)
                 .append(OHDConstants.NEW_LINE);
        resultStr.append(LagKJConstants.LENGTH_CARRYOVER_ARRAY)
                 .append("=")
                 .append(_lengthCarryoverArray)
                 .append(OHDConstants.NEW_LINE);
        resultStr.append(LagKJConstants.QT_PAIRS + "=").append(_pairQTLagCarryover).append(OHDConstants.NEW_LINE);

        if(_pairQTLagCarryover > 0 && _lengthCarryoverArray > 5)
        {
            for(int i = 0; i < _pairQTLagCarryover * 2; i++)
            {
                if(i == 0 || (i % 2) == 0)
                {
                    resultStr.append("DISCHARGE#")
                             .append((i - (i / 2)))
                             .append("=")
                             .append(this.getPairsQTCarryoverValues()[i])
                             .append(OHDConstants.NEW_LINE);
                }
                else if((i % 2 != 0))
                {
                    resultStr.append("LAG#")
                             .append((i - 1 - (i / 2)) + "=")
                             .append(this.getPairsQTCarryoverValues()[i])
                             .append(OHDConstants.NEW_LINE);
                }
            }
        }
        return resultStr.toString();
    }

    /*
     * (non-Javadoc)
     * @see ohd.hseb.ohdmodels.ModelState#doCarryOverTransfer(ohd.hseb.ohdmodels.ModelParameters,
     * ohd.hseb.ohdmodels.ModelParameters)
     */
    /**
     * THIS FUNCTION TRANSFERS CARRYOVER VALUES FROM ONE SET OF PARAMETERS TO ANOTHER FOR THE LAG/K OPERATION.
     */
    @Override
    public void doCarryOverTransfer(final ModelParameters savedParameters, final ModelParameters parameters) throws Exception
    {

        final LagKJModelParameters oldParams = (LagKJModelParameters)savedParameters;
        final LagKJModelParameters newParams = (LagKJModelParameters)parameters;
        final LagKJCommonMethods commonMethods = new LagKJCommonMethods();

        final int oldNumPairQTCaryyover = this.getNumPairsQTCarryover();
        int newNumPairQTCaryyover = oldNumPairQTCaryyover;

        final int oldCarryoverLength = this.getLengthCarryoverArr();
        int newCarryoverLength = oldCarryoverLength;

        final double oldCurrInflow = this.getCurrentLaggedInflow();
        final double oldCurrOutflow = this.getCurrentOutflow();

        double newCurrInflow = this.getCurrentLaggedInflow();
        double newCurrOutflow = this.getCurrentOutflow();
        double newCurrStorage = this.getCurrentStorage();

        final ArrayList<Double> newQTLagCarryoverValues = new ArrayList<Double>();

        // Copy pair QT/Lag carryover values from 1d-array to Arraylist 
        if(newParams.getMaxPairQTLagCarryover() != newNumPairQTCaryyover)//new FB1671
            newNumPairQTCaryyover = newParams.getMaxPairQTLagCarryover();//FB1671

        if(newNumPairQTCaryyover > 0)
            for(final double d: this.getPairsQTCarryoverValues())
                newQTLagCarryoverValues.add(d);

        int flagsLag = 0;
        int NumPairQTCaryyover = 0;

        if(_logger.getPrintDebugInfo() >= 5)
        {
            _logger.log(Logger.DEBUG, "old#pairQTCarry = " + oldNumPairQTCaryyover + " new#pairQTCarry = "
                + newNumPairQTCaryyover);
            _logger.log(Logger.DEBUG, "oldCOLength = " + oldCarryoverLength + " newCOLength = " + newCarryoverLength);
            _logger.log(Logger.DEBUG, "\n========================\nSTART DO CARRYOVER TRANSFER ....");
            _logger.log(Logger.DEBUG, " MXCON = " + newCarryoverLength + " MXCOO = " + oldCarryoverLength);
        }
        // ........ TRANSFER LAG CARRYOVER .........
        if(!(oldParams.isLagOptEnable() == false || newParams.isLagOptEnable() == false || newCarryoverLength < 7))
        {
            _logger.log(Logger.DEBUG, "........ TRANSFER LAG CARRYOVER .........");

            // In case K is not enable - transfer current outflow
            if (newParams.isKOptEnable() == false)
            {
                newCurrInflow = 0;
                newCurrStorage = 0;
                _logger.log(Logger.DEBUG,"Transfer LAG carryover with K option is turned off then set newCurrInflow and newCurrStorage to 0.");
                newCurrOutflow = oldCurrOutflow;
            }

            if(_logger.getPrintDebugInfo() >= 4)
            {
                _logger.log(Logger.DEBUG, "old#PairQTLag = " + oldNumPairQTCaryyover + " new#PairQTLag = "
                    + newNumPairQTCaryyover);
                _logger.log(Logger.DEBUG, "new Curr Outflow = " + newCurrOutflow + " MXLCOO Old Pair QT carryover = "
                    + oldNumPairQTCaryyover);
            }
            // Initialize old/new pair LAG/Q value to default constant Lag and 1.E20
            double[] oldParamPairsLagQ = new double[2];
            double[] newParamPairsLagQ = new double[2];

            if(oldParams.getConstantLag() == 0.)
            {
                oldParamPairsLagQ = new double[oldParams.getPairsValuesLagQ().length];
                oldParamPairsLagQ = oldParams.getPairsValuesLagQ();
            }
            else
            {
                oldParamPairsLagQ[0] = oldParams.getConstantLag();
                oldParamPairsLagQ[1] = 1.E20;
            }

            if(newParams.getConstantLag() == 0.)
            {
                newParamPairsLagQ = new double[newParams.getPairsValuesLagQ().length];
                newParamPairsLagQ = newParams.getPairsValuesLagQ();
            }
            else
            {
                newParamPairsLagQ[0] = newParams.getConstantLag();
                newParamPairsLagQ[1] = 1.E20;
            }

            // Set Lag Flags
            if(this.getNumPairsQTCarryover() > 0)
            { // flagsLag = 4 - FOR OLD AND NEW PARAMETERS CONSTANT LAG
                if(oldParams.getConstantLag() > 0. && newParams.getConstantLag() > 0.)
                    flagsLag = 4;

                // flagsLag = 3 -  FOR OLD CONSTANT LAG AND NEW VARIABLE LAG
                if(oldParams.getConstantLag() > 0. && newParams.getConstantLag() == 0.)
                    flagsLag = 3;

                // flagsLag = 2 - FOR OLD VARIABLE LAG AND NEW CONSTANT LAG
                if(oldParams.getConstantLag() == 0 && newParams.getConstantLag() > 0.)
                    flagsLag = 2;

                // flagsLag = 1 - FOR OLD AND NEW VARIABLE LAG
                if(oldParams.getConstantLag() == 0 && newParams.getConstantLag() == 0.)
                    flagsLag = 1;

                if(_logger.getPrintDebugInfo() >= 5)
                    _logger.log(Logger.DEBUG, " LAG FLAG = " + flagsLag + " MXLCOO =" + this.getNumPairsQTCarryover());

                int lengthCarryover = 6;
                double lagTime = 0.;
                int n = 0;
                final int NumOldParamPairsLagQ = oldParamPairsLagQ.length / 2;
                final int NumNewParamPairsLagQ = newParamPairsLagQ.length / 2;

                for(int i = 0; i < this.getNumPairsQTCarryover() * 2; i += 2)
                {
                    if(_logger.getPrintDebugInfo() >= 5)
                        _logger.log(Logger.DEBUG, "CO(" + (i + 6) + ") = " + this.getPairsQTCarryoverValues()[i]
                            + " CO(" + (i + 7) + ") = " + this.getPairsQTCarryoverValues()[i + 1]);

                    //if lag value (time) less than 0 then get out the loop
                    if(this.getPairsQTCarryoverValues()[i + 1] <= 0.)
                    {
                        _logger.log(Logger.DEBUG, "break at this.getPairsQTCarryoverValues()[" + (i + 1) + "] = "
                            + this.getPairsQTCarryoverValues()[i + 1] + " <= 0");
                        break;
                    }

                    commonMethods.setNewNumPair(0);
                    final double oldLinearInterp = commonMethods.linearInterpolator(0,
                                                                                    this.getPairsQTCarryoverValues()[i],
                                                                                    NumOldParamPairsLagQ,
                                                                                    oldParamPairsLagQ,
                                                                                    0);
                    commonMethods.setNewNumPair(0);
                    final double newLinearInterp = commonMethods.linearInterpolator(0,
                                                                                    this.getPairsQTCarryoverValues()[i],
                                                                                    NumNewParamPairsLagQ,
                                                                                    newParamPairsLagQ,
                                                                                    0);

                    lagTime = this.getPairsQTCarryoverValues()[i + 1] - oldLinearInterp + newLinearInterp;

                    if(_logger.getPrintDebugInfo() >= 4)
                    {
                        _logger.log(Logger.DEBUG, "111 ===> TIME = " + lagTime);
                        _logger.log(Logger.DEBUG, "IN DO 100 LOOP - I,K,J,CO(K),CO(J),LAG= " + lengthCarryover + " "
                            + (5 + i + 2) + " " + (5 + i + 1) + " " + this.getPairsQTCarryoverValues()[i + 1] + " "
                            + this.getPairsQTCarryoverValues()[i]);
                        _logger.log(Logger.DEBUG, "  TIME= " + lagTime);
                    }

                    if(lagTime > 0.)
                    {
                        if(lengthCarryover + 1 > newCarryoverLength)
                        {
                            if(_logger.getPrintDebugInfo() >= 5)
                                _logger.log(Logger.DEBUG, "Length CN =" + lengthCarryover + " > MXCON = "
                                    + newCarryoverLength);
                            break;
                        }

                        if(n >= newQTLagCarryoverValues.size())
                            break;

                        newQTLagCarryoverValues.set(n, this.getPairsQTCarryoverValues()[i]);
                        newQTLagCarryoverValues.set(n + 1, lagTime);

                        if(_logger.getPrintDebugInfo() >= 5)
                            _logger.log(Logger.DEBUG, "CN(" + (n + 6) + ")=" + newQTLagCarryoverValues.get(n) + " CN("
                                + (n + 7) + ")=" + newQTLagCarryoverValues.get(n + 1));

                        // Increase index to 2
                        n += 2;
                        lengthCarryover += 2;
                    }
                }// end i loop

                if(_logger.getPrintDebugInfo() >= 5)
                {
                    _logger.log(Logger.DEBUG, "After Length CN = " + lengthCarryover + " TIME = " + lagTime);
                    _logger.log(Logger.DEBUG, "LAG = " + flagsLag + " CN2 = " + newCurrInflow);
                }

                if (flagsLag != 4 && newParams.isKOptEnable() != false)
                {
                   commonMethods.setNewNumPair(0);
                   final double oldLinearInterp = commonMethods.linearInterpolator(0,
                                                                                newCurrInflow,
                                                                                NumOldParamPairsLagQ,
                                                                                oldParamPairsLagQ,
                                                                                0);
                   commonMethods.setNewNumPair(0);
                   final double newLinearInterp = commonMethods.linearInterpolator(0,
                                                                                newCurrInflow,
                                                                                NumNewParamPairsLagQ,
                                                                                newParamPairsLagQ,
                                                                                0);
                   lagTime = 0.0 - oldLinearInterp + newLinearInterp;

                   if (_logger.getPrintDebugInfo() >= 5)
                       _logger.log(Logger.DEBUG, "222 ===> TIME = " + lagTime);

                   // Re-initialize newCarryoverLength
                   NumPairQTCaryyover = newNumPairQTCaryyover * 2;
                   newCarryoverLength = LagKJConstants.DEFAULT_LENGTH_CARRYOVER + NumPairQTCaryyover;//newNumPairQTCaryyover * 2;

                   if(lagTime > 0.0)
                   {
                       lengthCarryover -= 1;

                       if(_logger.getPrintDebugInfo() >= 5)
                           _logger.log(Logger.DEBUG, "222 time = " + lagTime + " LCN = " + lengthCarryover + " MXCON = "
                            + newCarryoverLength);

                       if(lengthCarryover > LagKJConstants.DEFAULT_LENGTH_CARRYOVER)
                       {
                           commonMethods.shiftValuesToRight(newQTLagCarryoverValues, newCurrInflow, lagTime);

                           if(_logger.getPrintDebugInfo() >= 5)
                           {
                               for(int l = 0; l < NumPairQTCaryyover; l += 2)
                               {
                                   if(l + 1 > newNumPairQTCaryyover)
                                       break;

                                   _logger.log(Logger.DEBUG, "CN(" + (l + 1) + ") = " + newQTLagCarryoverValues.get(l + 1)
                                       + " CN(" + (l) + ") = " + newQTLagCarryoverValues.get(l));

                               }
                           }
                       } // if(lengthCarryover > 5)
                   }//if(lagTime > 0.0)
                }//if (flagsLag != 4 && newParams.isKOptEnable() != false)
            }//if(this.getNumPairsQTCarryover() > 0)
        }// End transfer LAG carryover

        if(!(oldParams.isKOptEnable() == false || newParams.isKOptEnable() == false))
        {
            _logger.log(Logger.DEBUG, "........ TRANSFER K CARRYOVER .........");

            newCurrOutflow = this.getCurrentOutflow();
            newCurrStorage = this.getCurrentStorage();

            if(!(oldParams.getMeanQ() == true || newParams.getMeanQ() == true))
                newCurrInflow = oldCurrInflow;
            else if(!(oldParams.getMeanQ() == false || newParams.getMeanQ() == true))
                newCurrInflow = newCurrOutflow;

            if(_logger.getPrintDebugInfo() >= 5)
                _logger.log(Logger.DEBUG, "==> newCurrentInflow = CN(2) = " + newCurrInflow);
        }// End transfer K carryover

        // Update states after transfer carryover
        this.setLengthCarryoverArr(newCarryoverLength);
        this.setCurrentLaggedInflow(newCurrInflow);
        this.setCurrentOutflow(newCurrOutflow);
        this.setCurrentStoreage(newCurrStorage);

        if(_logger.getPrintDebugInfo() > 0)
        {
            _logger.log(Logger.DEBUG, "cn(1) = " + this.getLengthCarryoverArr());
            _logger.log(Logger.DEBUG, "cn(2) = " + this.getCurrentLaggedInflow());
            _logger.log(Logger.DEBUG, "cn(3) = " + this.getCurrentOutflow());
            _logger.log(Logger.DEBUG, "cn(4) = " + this.getCurrentStorage());
        }

        if(NumPairQTCaryyover > 0)
        {
            final double[] newpairsQTLagCarryover = new double[NumPairQTCaryyover];

            for(int i = 0; i < NumPairQTCaryyover; i++)
            {
                if(i + 1 > newQTLagCarryoverValues.size())
                    break;

                newpairsQTLagCarryover[i] = newQTLagCarryoverValues.get(i);

                if(_logger.getPrintDebugInfo() > 0)
                    _logger.log(Logger.DEBUG, "cn(" + (i + 6) + ") = " + newpairsQTLagCarryover[i]);
            }

            this.setNumPairsQTCarryover(newNumPairQTCaryyover);
            this.setPairsQTCarryoverValues(newpairsQTLagCarryover);

            if(_logger.getPrintDebugInfo() > 0)
                _logger.log(Logger.DEBUG, "cn(5) = " + this.getNumPairsQTCarryover());
        }

        if(_logger.getPrintDebugInfo() > 0)
            _logger.log(Logger.DEBUG, "DONE CARRYOVER TRANSFER...");
    }

    /**
     * @return the _LagDischargeCarryoverValues
     */
    double[] getPairsQTCarryoverValues()
    {
        return this._LagDischargeCarryoverValues;

    }

    /**
     * @param carryover
     */
    void setPairsQTCarryoverValues(final double[] carryover)
    {
        _LagDischargeCarryoverValues = carryover;
    }

    /**
     * @return the _pairQTLagCarryover
     */
    int getNumPairsQTCarryover()
    {
        return this._pairQTLagCarryover;
    }

    /**
     * @param numPairsQT
     */
    void setNumPairsQTCarryover(final int numPairsQT)
    {
        _pairQTLagCarryover = numPairsQT;
    }

    /**
     * @return the unitType
     */
    String getUnitType()
    {
        return _stateUnits;
    }

    /**
     * @param unitType the unitType to set
     */
    public void setUnitType(final String unitType)
    {
        _stateUnits = unitType;
        super.setStateValue(OHDConstants.UNIT_TAG, _stateUnits);
    }

    /**
     * @return the _currentLaggedInflow
     */
    double getCurrentLaggedInflow()
    {
        return this._currentLaggedInflow;
    }

    /**
     * @param currentLaggedInflow
     */
    void setCurrentLaggedInflow(final double currentLaggedInflow)
    {
        _currentLaggedInflow = currentLaggedInflow;
    }

    /**
     * @return the _currentOutflow
     */
    double getCurrentOutflow()
    {
        return this._currentOutflow;
    }

    /**
     * @param currentOutflow
     */
    void setCurrentOutflow(final double currentOutflow)
    {
        _currentOutflow = currentOutflow;
    }

    /**
     * @return the _currentStoreage
     */
    double getCurrentStorage()
    {
        return this._currentStorage;
    }

    /**
     * @param currentStoreage
     */
    void setCurrentStoreage(final double currentStoreage)
    {
        _currentStorage = currentStoreage;
    }

    /**
     * @return the_lengthCarryoverArray
     */
    int getLengthCarryoverArr()
    {
        return this._lengthCarryoverArray;
    }

    /**
     * @param lengthCarryover
     */
    void setLengthCarryoverArr(final int lengthCarryover)
    {
        _lengthCarryoverArray = lengthCarryover;
    }

}
