package ohd.hseb.ohdmodels.laycoef;

import java.util.ArrayList;
import java.util.List;

import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.ohdmodels.ModelDriver;

/**
 * @author FewsPilot Team
 */
final public class LayCoefModelDriver extends ModelDriver
{

    private final LayCoefModelParameters _layCoefModelParameters; //alias to super._parameters to save type casting

    private final LayCoefModelParameters _layCoefModelSavedParameters; //alias to super._savedParameters to save type casting

    private final LayCoefModelState _layCoefModelState; //alias to super._state to save type casting

    public LayCoefModelDriver()
    {
        super(); //calling ModelDriver constructor       

        super._parameters = new LayCoefModelParameters();
        _layCoefModelParameters = (LayCoefModelParameters)super.getParameters(); //use alias to save casting

        super._savedParameters = new LayCoefModelParameters();
        _layCoefModelSavedParameters = (LayCoefModelParameters)super._savedParameters; //use alias to save casting

        super._state = new LayCoefModelState();
        _layCoefModelState = (LayCoefModelState)super._state; //use alias to save casting     

    }

    @Override
    public LayCoefModelParameters getParameters()
    {
        return (LayCoefModelParameters)_parameters;
    }

    @Override
    public LayCoefModelState getState()
    {
        return (LayCoefModelState)_state;
    }

    @Override
    public LayCoefModelParameters getPreviousParameters()
    {
        return (LayCoefModelParameters)_savedParameters;
    }

    @Override
    public void execute() throws Exception
    {

        // all input timeseries have the same time step. Any timeseries is a driver.
        _drivingTs = _tsList.get(0);

        if(_drivingTs == null)
        {
            _logger.log(Logger.FATAL, "Driving Time Series is null.Exiting...");

            throw new Exception();
        }

        _logger.log(Logger.DEBUG,
                    "Driving Time Series set to " + _drivingTs.getLocationId() + " " + _drivingTs.getTimeSeriesType()
                        + " " + _drivingTs.getIntervalInHours());

        try
        {
            runLocalModelDriverValidation();
            addTStoResultMap(this.computeLayerCoefficientRouting());
        }
        catch(final Exception e)
        {
            final Exception x = e;
            throw x;
        }

        _layCoefModelState.setDateTime(getComputationEndTime());

    }

    /**
     * @return
     * @throws Exception
     */
    private RegularTimeSeries computeLayerCoefficientRouting() throws Exception
    {

        // set this values and we don't need to retrieve several times in loop
        final long drivingTsIntervalInMillis = getDrivingTsIntervalInMillis();
        final long computationStartTime = getComputationStartTime();
        final long computationEndTime = getComputationEndTime();
        double flowin = 0.0;
        double flow = 0.0;
        final int numberOfLayers = _layCoefModelParameters.getNumberLayers();
        final double[] upperLimitFlowLayer = _layCoefModelParameters.getUpperLimitFlowByLayer();
        final double[] coefficientsByLayers = _layCoefModelParameters.getLayerCoefficients();
        final double[] carryOverByLayers = _layCoefModelState.getCarryOverValuesByLayer();
        double flowTotal = 0.0;

        // Output TS
        final RegularTimeSeries layCoefOutputTs = new RegularTimeSeries(computationStartTime,
                                                                        computationEndTime,
                                                                        _layCoefModelParameters.getOutflowTimeSerieTimeStep() != 0 ? _layCoefModelParameters.getOutflowTimeSerieTimeStep() : _drivingTs.getIntervalInHours(),
                                                                        OHDConstants.DISCHARGE_UNIT);
        layCoefOutputTs.setLocationId(_drivingTs.getLocationId());

        if(_layCoefModelParameters.getOutflowTimeSerieType() != null)
        {
            layCoefOutputTs.setTimeSeriesType(_layCoefModelParameters.getOutflowTimeSerieType());
        }
        else
        {
            layCoefOutputTs.setTimeSeriesType(_drivingTs.getTimeSeriesType());
        }

        if(_layCoefModelParameters.getOutflowTimeSerieId() != null)
        {
            final List<String> qualifiersIds = new ArrayList<String>();
            qualifiersIds.add(_layCoefModelParameters.getOutflowTimeSerieId());
            layCoefOutputTs.setQualifierIds(qualifiersIds);
        }
        else
        {
            layCoefOutputTs.setQualifierIds(_drivingTs.getQualifierIds());
        }

        for(long timeStep = computationStartTime; timeStep <= computationEndTime; timeStep += drivingTsIntervalInMillis)
        {
            flowTotal = 0.0;
            flowin = _drivingTs.getMeasurementValueByTime(timeStep, OHDConstants.DISCHARGE_UNIT);

            int layer = 0;
            // Determine layer inflow
            if(numberOfLayers > 0)
            {

                for(int layerNumber = 0; layerNumber < numberOfLayers; layerNumber++)
                {
                    // Subtract 1 to make this condition to be possible. as layerNumber will be never equals to numberOfLayers
                    if(layerNumber == numberOfLayers - 1)
                    {
                        layer = layerNumber;
                        break;
                    }
                    if(flowin <= upperLimitFlowLayer[layerNumber])
                    {
                        layer = layerNumber;
                        break;
                    }
                }

                // Route inflow for one layer at the time
                for(int layerNumber = 0; layerNumber < numberOfLayers; layerNumber++)
                {

                    if(layerNumber <= 0)
                    {
                        if(layer <= 0)
                        {
                            flow = flowin + carryOverByLayers[layerNumber];
                        }
                        else
                        {
                            flow = upperLimitFlowLayer[0] + carryOverByLayers[layerNumber];
                        }
                    }
                    else
                    {
                        if(layerNumber == layer)
                        {
                            flow = flowin - upperLimitFlowLayer[layerNumber - 1] + carryOverByLayers[layerNumber];
                        }
                        else
                        {

                            if(layerNumber > layer)
                            {
                                flow = carryOverByLayers[layerNumber];
                            }
                            else
                            {
                                flow = upperLimitFlowLayer[layerNumber] - upperLimitFlowLayer[layerNumber - 1]
                                    + carryOverByLayers[layerNumber];
                            }
                        }
                    }
                    flowTotal = flowTotal + flow * coefficientsByLayers[layerNumber];
                    if(flowTotal < 0.0)
                    {
                        flowTotal = 0.0;
                    }
                    carryOverByLayers[layerNumber] = flow * (1.0 - coefficientsByLayers[layerNumber]);
                    if(carryOverByLayers[layerNumber] < 0.00001)
                    {
                        carryOverByLayers[layerNumber] = 0.0;
                    }
                }
            }//end if(numberOfLayers > 0)
             //save the value to output ts
            layCoefOutputTs.setMeasurementByTime(flowTotal, timeStep);
        }
        /** --------------------- prepare carryover for storing-------------------- */
        _layCoefModelState.setCarryOverValuesToMap(carryOverByLayers);
        _layCoefModelState.setCarryOverValuesByLayer(carryOverByLayers);
        _layCoefModelState.setUnitType(OHDConstants.UNIT_METRIC);

        return layCoefOutputTs;
    }

    private void runLocalModelDriverValidation() throws Exception
    {

        //validate LayCoeffModelParamters and LayCoeffModelState
        _layCoefModelParameters.validateParams(_drivingTs.getMeasuringUnit());

        if(super.needCarryoverTransfer())
        {
            _layCoefModelState.validateState(_layCoefModelSavedParameters);
            _layCoefModelState.doCarryOverTransfer(_layCoefModelSavedParameters, _layCoefModelParameters);
        }
        else
        {
            _layCoefModelState.validateState(_layCoefModelParameters);

        }

        super.runModelDriverValidation();
    }
}
