package ohd.hseb.ohdmodels.muskrout;

import java.util.ArrayList;
import java.util.List;

import ohd.hseb.measurement.MeasuringUnitType;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.ohdmodels.ModelDriver;

final public class MuskroutModelDriver extends ModelDriver
{

    private final MuskroutModelParameters _muskroutModelParameters; //alias to super._parameters to save type casting

    private final MuskroutModelParameters _muskroutModelSavedParameters; //alias to super._savedParameters to save type casting

    private final MuskroutModelState _muskroutModelState; //alias to super._state to save type casting    

    public MuskroutModelDriver()
    {
        super(); //calling ModelDriver constructor

        _state = new MuskroutModelState();
        _muskroutModelState = (MuskroutModelState)_state;

        _parameters = new MuskroutModelParameters();
        _muskroutModelParameters = (MuskroutModelParameters)_parameters;

        _savedParameters = new MuskroutModelParameters();
        _muskroutModelSavedParameters = (MuskroutModelParameters)_savedParameters;
    }

    @Override
    public MuskroutModelParameters getParameters()
    {
        return (MuskroutModelParameters)_parameters;
    }

    @Override
    public MuskroutModelState getState()
    {
        return (MuskroutModelState)_state;
    }

    @Override
    public MuskroutModelParameters getPreviousParameters()
    {
        return (MuskroutModelParameters)_savedParameters;
    }

    @Override
    public void execute() throws Exception
    {

        runModelDriverValidation();

        try
        {

            addTStoResultMap(this.computeMuskrout());
        }
        catch(final Exception e)
        {
            final Exception x = e;
            throw x;
        }

        _muskroutModelState.setDateTime(getComputationEndTime());

    }

    /**
     * Muskingum routing is a storage routing method based on the storage equation which is an expression of continuity:<br>
     * I- O = dS / dt<br>
     * where I is the inflow rate O is the outflow rate S is storage t is time The expression for storage in a reach of
     * a stream used in the Muskingum method is:<br>
     * S=k[xI+(1-x)O] <br>
     * where K and x represent storage parameters.<br>
     * To apply Muskingum routing to a reach it is first necessary to determine values for the parameters K and x which
     * are used to describe the storage characteristics of the reach. Values of K and x are normally derived
     * analytically from historical flood records. The parameter K is a storage constant expressing the ratio between
     * storage and discharge and is usually expressed in hours. It may also be viewed as the lag or travel time through
     * the reach. The dimensionless parameter x is indicative of the relative importance of inflow and outflow to
     * storage. Like K it is also referred to as a storage constant. Although the Muskingum method is not as inherently
     * flexible as the Lag and K method of routing it is adequate for many reaches and routing situations.
     * 
     * @return RegularTimeSeries with Muskingum routing compute values.
     */
    private RegularTimeSeries computeMuskrout()
    {
        double inflowCarryover1 = 0.0;
        double outflowCarryover1 = 0.0;
        double inflowCarryover2 = 0.0;
        double outflowCarryover2 = 0.0;
        double routingCoefficientC0 = 0.0;
        double routingCoefficientC1 = 0.0;
        double routingCoefficientC2 = 0.0;
        int iCount = 0;
        int iCterr = 0;
        final double divisor = 4.0;
        double delq = 0.0;
        double muskroutFlowin = 0.0;
        final List<String> qualifiersIds = new ArrayList<String>();

        // set this values and we don't need to retrieve several times in loop
        final long drivingTsIntervalInMillis = getDrivingTsIntervalInMillis();
        final long computationStartTime = getComputationStartTime();
        final long computationEndTime = getComputationEndTime();

        double inflowCarryover = _muskroutModelState.getInitialInflowCarryover();
        double outflowCarryover = _muskroutModelState.getInitialOutflowCaryover();

        final double kParam = _muskroutModelParameters.getkParameter();
        final double xParam = _muskroutModelParameters.getxParameter();
        final double inputTsInterval = _muskroutModelParameters.getInflowTimeSerieTimeStep();

        // get coefficients c0, c1 and c2 computed from k, x and time series interval input values.
        double xkParam = kParam * xParam;
        double halfTimeStep = inputTsInterval * 0.5;
        double denominator = kParam - xkParam + halfTimeStep;
        final double coefficient0 = (halfTimeStep - xkParam) / denominator;
        final double coefficient1 = (xkParam + halfTimeStep) / denominator;
        final double coefficient2 = (kParam - xkParam - halfTimeStep) / denominator;

        // Output TS
        final RegularTimeSeries muskroutOutputTs = new RegularTimeSeries(computationStartTime,
                                                                         computationEndTime,
                                                                         _muskroutModelParameters.getOutflowTimeSerieTimeStep(),
                                                                         OHDConstants.DISCHARGE_UNIT);
        muskroutOutputTs.setTimeSeriesType(_muskroutModelParameters.getOutflowTimeSerieDataType());
        muskroutOutputTs.setLocationId(_drivingTs.getLocationId());
        qualifiersIds.add(_muskroutModelParameters.getOutflowTimeSerieId());
        muskroutOutputTs.setQualifierIds(qualifiersIds);

        //Part of the Musktingum routing algorithm. This section is always the same and was removed from the loop.
        halfTimeStep = inputTsInterval * 0.5 / divisor;
        xkParam = kParam * xParam;
        denominator = kParam - xkParam + halfTimeStep;
        routingCoefficientC0 = (halfTimeStep - xkParam) / denominator;
        routingCoefficientC1 = (xkParam + halfTimeStep) / denominator;
        routingCoefficientC2 = (kParam - xkParam - halfTimeStep) / denominator;

        for(long timeStep = computationStartTime; timeStep <= computationEndTime; timeStep += drivingTsIntervalInMillis)
        {
            iCount++;
            muskroutFlowin = _drivingTs.getMeasurementValueByTime(timeStep, OHDConstants.DISCHARGE_UNIT);
            double muskroutFlowout = coefficient0 * muskroutFlowin + coefficient1 * inflowCarryover + coefficient2
                * outflowCarryover;
            if(muskroutFlowout <= 0.0000001)
            {
                muskroutFlowout = 0.0;
            }
            if(muskroutFlowout < 0.0)
            {

                delq = (muskroutFlowin - inflowCarryover) / divisor;
                inflowCarryover1 = inflowCarryover;
                outflowCarryover1 = outflowCarryover;
                for(int i = 1; i <= 4; i++)
                {
                    inflowCarryover2 = inflowCarryover1 + delq;
                    outflowCarryover2 = routingCoefficientC0 * inflowCarryover2 + routingCoefficientC1
                        * inflowCarryover1 + routingCoefficientC2 * outflowCarryover1;

                    if(outflowCarryover2 <= 0.0000001)
                    {
                        outflowCarryover2 = 0.0;
                    }
                    inflowCarryover1 = inflowCarryover2;
                    outflowCarryover1 = outflowCarryover2;
                }
                muskroutFlowout = outflowCarryover2;
                if(muskroutFlowout < 0.0)
                {
                    //FOR LINEAR INTERPOLATION OF FIRST CALCULATION
                    if(iCount == 1)
                    {
                        muskroutFlowout = _muskroutModelState.getInitialOutflowCaryover();

                    }
                    // INTERPOLATION FOR SECOND POINT
                    else if(iCount == 2)
                    {
                        muskroutFlowout = 2.0
                            * muskroutOutputTs.getMeasurementValueByTime((timeStep - drivingTsIntervalInMillis),
                                                                         OHDConstants.DISCHARGE_UNIT)
                            - _muskroutModelState.getInitialOutflowCaryover();
                        if(muskroutFlowout < 0.0)
                        {
                            muskroutFlowout = 0.0;
                            iCterr++;
                        }
                    }
                    else
                    {
                        muskroutFlowout = 2
                            * muskroutOutputTs.getMeasurementValueByTime((timeStep - drivingTsIntervalInMillis),
                                                                         OHDConstants.DISCHARGE_UNIT)
                            - muskroutOutputTs.getMeasurementValueByTime((timeStep - (2 * drivingTsIntervalInMillis)),
                                                                         OHDConstants.DISCHARGE_UNIT);
                        if(muskroutFlowout < 0.0)
                        {
                            muskroutFlowout = 0.0;
                            iCterr++;
                        }
                    }
                }
            }
            //save the value to output ts
            muskroutOutputTs.setMeasurementByTime(muskroutFlowout, timeStep);

            inflowCarryover = muskroutFlowin;
            outflowCarryover = muskroutFlowout;

        }

        if(iCterr > 2)
        {
            _logger.log(Logger.WARNING, "In Muskingum Routing for" + _muskroutModelParameters.getGeneralInfo() + " "
                + _muskroutModelParameters.getInflowTimeSerieId()
                + " calculated outflows were negatives and set to zero " + iCterr + " times");
        }

        /** --------------------- prepare carryover for storing-------------------- */
        _muskroutModelState.setInitialInflowCarryover(inflowCarryover);
        _muskroutModelState.setInitialOutflowCaryover(outflowCarryover);
        _muskroutModelState.setCarryOverValues();

        return muskroutOutputTs;

    }

    /**
     * Validation of Parameters and States are implicit in the extractValuesFromMap() method. There is not need to
     * validate that required states/parameters are set. For Muskrout model all required states/parameters values are
     * set as part of the extractValuesFromMap process, if values are not present an exception will throw and it be
     * logged.
     */
    @Override
    protected void runModelDriverValidation() throws Exception
    {
        for(final RegularTimeSeries timeSeries: _tsList)
        {
            if(timeSeries.getMeasuringUnit().getType() == MeasuringUnitType.discharge)
            {
                _drivingTs = timeSeries;
                _logger.log(Logger.DEBUG,
                            "Driving Time Series set to " + _drivingTs.getLocationId() + " "
                                + _drivingTs.getTimeSeriesType() + " " + _drivingTs.getIntervalInHours());
            }
        }
        if(_drivingTs == null)
        {
            _logger.log(Logger.FATAL, "Driving Time Series is null.Exiting...");

            throw new Exception();
        }
        _logger.log(Logger.DEBUG,
                    "Driving Time Series set to " + _drivingTs.getLocationId() + " " + _drivingTs.getTimeSeriesType()
                        + " " + _drivingTs.getIntervalInHours());

        _muskroutModelParameters.validateParams();

        super.runModelDriverValidation();

    }

}
