package ohd.hseb.ohdmodels.ressngl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import ohd.hseb.measurement.AbsTimeMeasurement;
import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.MeasuringUnitType;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.SortedProperties;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.State;
import ohd.hseb.util.fews.ohdmodels.LegacyModelAdapter;
import ohd.hseb.util.fews.ohdmodels.ModelDriver;

/**
 * @author champ
 */
public class RessnglModelDriver extends ModelDriver
{
    private static final String _RESSNGL_EXE_NAME = "ressngl";
    private static String _MOD_NAME = "SETQMEAN";
    private static String _IREG_TECHNIQUE = "REGULATE";

    private final List<RegularTimeSeries> _modsTsList;

    public RessnglModelDriver()
    {
        super(); // calling ModelDriver constructor

        _parameters = new RessnglModelParameters();
        _state = new RessnglModelState();
        _savedParameters = new RessnglModelParameters();

        _modsTsList = new ArrayList<RegularTimeSeries>(1); //only 1 MOD ts
    }

    @Override
    public void execute() throws Exception
    {

        // Convert the inputs in XML file to flat file (ts.txt, params.txt and
        // stateI.txt) to work directory
        final LegacyModelAdapter ressnglAdapter = new LegacyModelAdapter(this, getWorkDir(), _logger);

        ressnglAdapter.writeStandAloneFiles();

        // ressngl needs an extra argument - mod file name,
        // fill in a String [] array and send to executeModel as last argument
        final SortedProperties properties = new SortedProperties();
        properties.put(LegacyModelAdapter.MOD_FILE_NAME, getWorkDir() + "/" + LegacyModelAdapter.MODS_TXT_FILE);

        int ireg = 0; //default 0 for not apply REGULATE technique 

        if(_runInfo.getProperties().getProperty(_IREG_TECHNIQUE) == null)
        {
            ireg = 0;
        }
        else
        {
            if(_runInfo.getProperties().getProperty(_IREG_TECHNIQUE).equalsIgnoreCase("TRUE"))
            {
                ireg = 1;
            }
        }

        properties.put("ressnglTechnique", String.valueOf(ireg));

        // write ressngl mods if any to mods.txt file
        ressnglAdapter.writeModsToFile(_modsTsList);

        //Clear out state Map before storing the new data (bug 244)
        ((State)_state).getStateMap().clear();

        // Execute RESS-SNGL model
        ressnglAdapter.executeModel(getDriverProperties().getProperty(OHDConstants.LEGACY_LOCATION_DIR) + "/"
            + _RESSNGL_EXE_NAME, properties);

        // Load the state results
        getState().loadState(ressnglAdapter.getOutputStateFileName(), _logger);

        // Set state date and time
        getState().setDateTime(getComputationEndTime());

        // Read time series (ts.txt)
        final List<RegularTimeSeries> resultsList = ressnglAdapter.readOutputFlatFilesToLists();

        // qualifier id is part of unique key NOT location id
        // this is because can have multiple outputs with same type

        for(int i = 0; i < resultsList.size(); i++)
        {
            addTStoResultMap(resultsList.get(i));
        }
    }

    /**
     * Overriding the method. Be called in {@link LegacyModelAdapter#LegacyModelAdapter(ModelDriver, String, Logger)}
     */
    @Override
    final protected void runModelDriverValidation() throws Exception
    {
        final Iterator<RegularTimeSeries> ite = getTsList().iterator();

        RegularTimeSeries inputRts;

        RegularTimeSeries tempDrivingTs = null;

        String tsType;

        while(ite.hasNext())
        {
            inputRts = ite.next();

            tsType = inputRts.getTimeSeriesType();

            if(tsType.equalsIgnoreCase(_MOD_NAME)) //only 1 MOD ts
            {
                _modsTsList.add(inputRts.clone());

                ite.remove();

            }
            else if(inputRts.getMeasuringUnit().getType() == MeasuringUnitType.discharge)
            {
                // All input time series don't have the same time step
                // pick the input ts with DISCHARGE dimension code and smallest time
                // step

                if(tempDrivingTs == null)
                {
                    tempDrivingTs = inputRts; //first DISCHARGE ts
                }
                else if(inputRts.getIntervalInHours() < tempDrivingTs.getIntervalInHours())
                {//find the finest TS

                    tempDrivingTs = inputRts;
                }

            }

        }// close while loop

        if(tempDrivingTs != null)
        {
            _drivingTs = tempDrivingTs;
        }
        else
        {
            _logger.log(Logger.FATAL, "Driving Time Series is null.Exiting...");

            throw new Exception();
        }

        super.runModelDriverValidation();
        if(_modsTsList.size() > 0)
        {
            super.checkInputTsInSync(_modsTsList);
        }
    }

    @Override
    /*
     * The following method is used to create a SETQMEAN .txt mod file each value in the inputModList time series is
     * converted into a line in the text file For example: SETQMEAN_817416=42.566630. We set effective date (i.e
     * validDate for mod) equal to Last observation time (i.e. LSTCMPDY) to make sure mod is used. This assumes FEWS has
     * done the necessary logic to disqualify invalid mods
     */
    public String getModsStringFromTS(final List<RegularTimeSeries> inputModsList) throws Exception
    {
        final int numIdentifiers = 0;

        final List<RessnglModEntry> allRessnglMods = new ArrayList<RessnglModEntry>();

        final long effDate = new DateTime(_runInfo.getRunLastObservationTimeLong(), getStartHourOfDay()).convertToJulianHours();

        final String header = numIdentifiers + " " + Long.toString(effDate) + "\n";

        for(final RegularTimeSeries ts: inputModsList)
        {
            createEntriesInModFileForOneTimeseriesType(_MOD_NAME, allRessnglMods, ts);
        }

        Collections.sort(allRessnglMods);

        String modsString = "";

        // when all values are missing return an empty string
        // NO MODS
        if(allRessnglMods.size() == 0)
        {
            return modsString;
        }

        modsString = header;
        for(int i = 0; i < allRessnglMods.size(); i++)
        {
            modsString = modsString + allRessnglMods.get(i).getModLine();
        }

        return modsString;
    }

    private void createEntriesInModFileForOneTimeseriesType(final String modName,
                                                            final List<RessnglModEntry> myStringsCollection,
                                                            final RegularTimeSeries ts) throws Exception
    {
        String oneModLine;
        double modValue;
        long time;
        DateTime sdt;

        int dateAsInteger;
        RessnglModEntry modEntry;
        final int measureMentCount = ts.getMeasurementCount();
        int newMeasureMentCount=0;
        final MeasuringUnit regulationUnit = ts.getMeasuringUnit();

        for (int index = measureMentCount-1; index >=0; index--)
        {
        	double mvalue=ts.getMeasurementValueByIndex(index, regulationUnit);
        	if ( ! Double.toString(mvalue).equalsIgnoreCase(Double.toString(ts.getMissingValue())))
        	{
        		newMeasureMentCount = index;
        		
        		break;
        	}	
        }
        int istart=0;                                   
        for(int index = 0; index <= newMeasureMentCount; index++)
        {
            oneModLine = modName;
            modValue = ts.getMeasurementValueByIndex(index, regulationUnit);
            
            // convert unit in CFS (English) to CMS (metric) because model used 
            // units for the SETQMEAN MOD are Metric
            if(!regulationUnit.equals(MeasuringUnit.cms))
            {
                ts.convert(MeasuringUnit.cms);
                modValue = ts.getMeasurementValueByIndex(index, ts.getMeasuringUnit());
            }

            // this handles case where missing is not a number but a string (e.g. "NaN")
            if(istart == 0 && Double.toString(modValue).equalsIgnoreCase(Double.toString(ts.getMissingValue())))
            {
                continue;
            }
            else 
            {
            	if (istart == 1 && Double.toString(modValue).equalsIgnoreCase(Double.toString(ts.getMissingValue()))) 
            	{
            	   modValue = -888.0;
            	   
            	}
            	istart = 1;
            }

            final AbsTimeMeasurement absTime = ts.getAbsTimeMeasurementByIndex(index);
            time = absTime.getTime();
            sdt = new DateTime(time, getStartHourOfDay());

            dateAsInteger = sdt.convertToJulianHours();
            oneModLine = oneModLine + "_" + dateAsInteger + "=" + modValue + "\n";

            modEntry = new RessnglModEntry(oneModLine, dateAsInteger);

            myStringsCollection.add(modEntry);
        }

    }

    @Override
    public RessnglModelParameters getParameters()
    {
        return (RessnglModelParameters)_parameters;
    }

    @Override
    public RessnglModelParameters getPreviousParameters()
    {
        return (RessnglModelParameters)_savedParameters;
    }

    @Override
    public RessnglModelState getState()
    {
        return (RessnglModelState)_state;
    }

}
