package ohd.hseb.ohdmodels.rsnwelev;

import java.util.ArrayList;
import java.util.List;

import javax.management.timer.Timer;

import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.util.fews.DataType;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.ohdmodels.ModelDriver;

public class RsnwelevModelDriver extends ModelDriver
{

    private final RsnwelevModelParameters _rsnwelevParameters;
    private final RsnwelevModelState _rsnwelevState;
    private int _timeStep;
    private double _elevationOfAirTemperaturesInMeters;
    private double _lapseRate;
    private double _thresholdTemp;
    private boolean _usingFreezingLevelData = false;
    private double _initialFreezingLevel = OHDConstants.MISSING_DATA;
    private double _endOfPeriodFreezingLevel = OHDConstants.MISSING_DATA;
    private String _qualifierId;

    // Input TS
    RegularTimeSeries _matTs = null;
    RegularTimeSeries _freezingLevelTs = null;

    public RsnwelevModelDriver()
    {
        super(); //calling ModelDriver constructor

        _rsnwelevState = new RsnwelevModelState();
        super._state = _rsnwelevState;
        _rsnwelevParameters = new RsnwelevModelParameters();
        super._parameters = _rsnwelevParameters;
        _savedParameters = new RsnwelevModelParameters();
    }

    @Override
    protected void runModelDriverValidation() throws Exception
    {
        _rsnwelevParameters.validateParams();
        _timeStep = _rsnwelevParameters.getIntervalInHours();
        _elevationOfAirTemperaturesInMeters = _rsnwelevParameters.getElevationOfAirTempInMeters();
        _lapseRate = _rsnwelevParameters.getLapseRate();
        _thresholdTemp = _rsnwelevParameters.getThresholdTemperature();
        _qualifierId = _rsnwelevParameters.getRainSnowElevationTimeseriesId();

        for(final RegularTimeSeries rts: _tsList)
        {
            if(rts.getTimeSeriesType().equalsIgnoreCase("ZELV"))
            {
                _freezingLevelTs = rts;
                _usingFreezingLevelData = true;
            }

            if(rts.getTimeSeriesType().equalsIgnoreCase("MAT"))
            {
                _matTs = rts;
            }
        }

        // error check
        if(_matTs == null)
            throw new Exception("MAT TS not defined");

        _drivingTs = _matTs;

        super.runModelDriverValidation();

        if(_usingFreezingLevelData)
            _initialFreezingLevel = _rsnwelevState.getDoubleDataState(RsnwelevModelState.FREEZING_LEVEL_TS);

    }

    /**
     * The Operation computes the elevation that separates rain from snow. Input variables are air temperature and
     * optionally the freezing level (elevation where the temperature is 0 DEGC). Parametric input consists of the
     * threshold temperature (temperature that divides rain from snow), the lapse rate during precipitation periods, and
     * the elevation associated with the air temperature data.
     * <p>
     * The rain-snow elevation is computed as: Ers = Ev + ((Tv - PXTEMP) * (100/Lp))<br>
     * where: <br>
     * Ers is the elevation separating rain from snow (M)<br>
     * Ev is the elevation of input variable (M): <br>
     * Ev = Ze when input variable is freezing level <br>
     * Ev = Te when input variable is air temperature <br>
     * Ze is the freezing level (M) <br>
     * Te is the elevation associated with air temperature data (M)<br>
     * Tv is the temperature at Ev (DEGC): <br>
     * Tv = 0 DEGC when input variable is freezing level <br>
     * Tv = Ta when input variable is air temperature Ta is the air temperature (DEGC) <br>
     * PXTEMP is the threshold temperature (DEGC) <br>
     * Lp is the lapse rate during precipitation periods (DEGC/100M)
     * <p>
     * If freezing level data are available, it is used to compute the rainsnow elevation. If freezing level data are
     * not available or if a given freezing level value is missing, then the rain-snow elevation is computed using the
     * air temperature. A mean rain-snow elevation is computed for each time interval.
     */
    @Override
    public void execute() throws Exception
    {

        runModelDriverValidation();

        final long intervalInMillis = _timeStep * Timer.ONE_HOUR;
        final long startTime = getComputationStartTime();
        final long endTime = getComputationEndTime();
        double averageFreezingLevel = 0.0;
        double currentTemp = 0.0;

        // Output TS
        final RegularTimeSeries rainsnowElevationTS = new RegularTimeSeries(startTime,
                                                                            endTime,
                                                                            _timeStep,
                                                                            MeasuringUnit.meters);
        rainsnowElevationTS.setTimeSeriesType(DataType.RSEL_DATATYPE);

        double rainSnowElevationValue = OHDConstants.MISSING_DATA;

        for(long time = startTime; time <= endTime; time += intervalInMillis)
        {
            if(_usingFreezingLevelData)
            {
                _endOfPeriodFreezingLevel = _freezingLevelTs.getMeasurementValueByTime(time, MeasuringUnit.meters);

                if((_initialFreezingLevel != OHDConstants.MISSING_DATA && _endOfPeriodFreezingLevel != OHDConstants.MISSING_DATA))
                {
                    averageFreezingLevel = (_initialFreezingLevel + _endOfPeriodFreezingLevel) * 0.5f;
                    rainSnowElevationValue = averageFreezingLevel - (_thresholdTemp * (100.0f / _lapseRate));
                }
                _initialFreezingLevel = _endOfPeriodFreezingLevel;
            }

            if(rainSnowElevationValue == OHDConstants.MISSING_DATA)
            {
                // temperature approach
                currentTemp = _matTs.getMeasurementValueByTime(time, MeasuringUnit.degreesCelsius);
                rainSnowElevationValue = _elevationOfAirTemperaturesInMeters
                    + ((currentTemp - _thresholdTemp) * (100.0f / _lapseRate));
            }

            // check to make sure value not LT -100
            if(rainSnowElevationValue < -100.0f)
            {
                rainSnowElevationValue = -100.0f;
            }

            rainsnowElevationTS.setMeasurementByTime(rainSnowElevationValue, time);

            rainSnowElevationValue = OHDConstants.MISSING_DATA;

        }

        // extra step for old legacy models converter to Java
        // need to define a qualifier id which is NWSRFS TSID
        final List<String> qualifierIds = new ArrayList<String>();
        qualifierIds.add(_qualifierId);
        rainsnowElevationTS.setQualifierIds(qualifierIds);

        addTStoResultMap(rainsnowElevationTS);
        _rsnwelevState.setDateTime(getComputationEndTime());

        // set state value if using freezing level option
        if(_usingFreezingLevelData)
        {
            _rsnwelevState.getStateMap().put(RsnwelevModelState.FREEZING_LEVEL_TS, _endOfPeriodFreezingLevel);
        }

    }

}
