package ohd.hseb.ohdmodels.sacsma;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import ohd.hseb.time.DateTime;
import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.model.sacsma.SacSmaModsTSHolder;
import ohd.hseb.model.sacsma.SacSma_MODS;
import ohd.hseb.ohdmodels.snow17.Snow17ModelConstants;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.DataType;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.OHDUtilities;
import ohd.hseb.util.fews.PRECIP_DATATYPE;
import ohd.hseb.util.fews.RUNOFF_COMPONENT;
import ohd.hseb.util.fews.ohdmodels.ModelDriver;

/**
 * @author FewsPilot Team
 */
final public class SacSmaModelDriver extends ModelDriver
{

    /** alias to super._parameters to save type casting */
    private final SacSmaModelParameters _sacModelParameters;

    /** alias to super._parameters to save type casting */
    private final SacSmaModelParameters _sacModelSavedParameters;

    /** alias to super._state to save type casting */
    private final SacSmaModelState _sacModelState;

    //input TSs:
    private RegularTimeSeries _mapeTs = null; //optional, precipTs is the required and is the driving Ts
    private RegularTimeSeries _sascTs = null; //optional, but is also used in FRZE
    private RegularTimeSeries _matTs = null; //required for FRZE
    private RegularTimeSeries _weTs = null;
    /*
     * optional for FRZE, controlled by the boolean parameter "WE_INPUT_OPTION"(default is false): if false, ignore WE
     * ts in inputs.xml; if true, WE ts is required
     */

    private boolean _printLog = false;

    private SacSmaTechnique _sacSmaTechnique;

    private SacSmaModsTSHolder _sacsmaModsTsHolder;

    public SacSmaModelDriver()
    {
        super(); //calling ModelDriver constructor

        super._state = new SacSmaModelState();
        super._parameters = new SacSmaModelParameters();
        super._savedParameters = new SacSmaModelParameters();

        _sacModelParameters = (SacSmaModelParameters)super._parameters; // use alias to save casting
        _sacModelSavedParameters = (SacSmaModelParameters)super._savedParameters;
        _sacModelState = (SacSmaModelState)super._state; // use alias to save casting

    }

    @Override
    public void execute() throws Exception
    {

        if(_logger.getPrintDebugInfo() > 0)
        {
            _printLog = true;
        }
        /* return gracefully, errors are logged */
        runModelDriverValidation();

        _logger.log(Logger.DEBUG,
                    "Running SAC-SMA program from GMT time: "
                        + DateTime.getDateTimeStringFromLong(getComputationStartTime(),OHDConstants.GMT_TIMEZONE) + " to "
                        + DateTime.getDateTimeStringFromLong(getComputationEndTime(),OHDConstants.GMT_TIMEZONE));

        final SacSmaRainfallRunoffModelAdapter sAdapter = new SacSmaRainfallRunoffModelAdapter(this);

        /** -------------------------------run model--------------------------------- */
        sAdapter.runSacSmaRainfallRunoffModel(_resultMap, _sacsmaModsTsHolder);

        if(_sacSmaTechnique.getBoolValue(Snow17ModelConstants.SACSNOW_FULL_VERSION))
        {

            /* ----------------- Add 6 state percentage TSs----------------------- */
            /*
             * the code for creating 6 state percentage TSs need be here first, because later, within _resultMap, it is
             * possible that there are two TSs of LZFSC, LZFPC and UZFWC, each with different interval(one copy is for
             * output state TS, another copy is for SMZC). Here, there is only a copy of each state, and the TS interval
             * is RAIM interval.
             */
            final RegularTimeSeries stateUztwcPerc = OHDUtilities.getRTSFromResultMap(_resultMap,
                                                                                      OHDConstants.SAC_STATE_UZTWC)
                                                                 .clone();
            stateUztwcPerc.scaleRegularTimeSeries(1.0 / _sacModelParameters.getUztwm());
            stateUztwcPerc.setMeasuringUnitNoChangeData(MeasuringUnit.percentDecimal);
            stateUztwcPerc.setTimeSeriesType(SacSmaModelConstants.UZTWC_PERC);
            addTStoResultMap(stateUztwcPerc);

            final RegularTimeSeries stateUzfwcPerc = OHDUtilities.getRTSFromResultMap(_resultMap,
                                                                                      OHDConstants.SAC_STATE_UZFWC)
                                                                 .clone();
            stateUzfwcPerc.scaleRegularTimeSeries(1.0 / _sacModelParameters.getUzfwm());
            stateUzfwcPerc.setMeasuringUnitNoChangeData(MeasuringUnit.percentDecimal);
            stateUzfwcPerc.setTimeSeriesType(SacSmaModelConstants.UZFWC_PERC);
            addTStoResultMap(stateUzfwcPerc);

            final RegularTimeSeries stateLztwcPerc = OHDUtilities.getRTSFromResultMap(_resultMap,
                                                                                      OHDConstants.SAC_STATE_LZTWC)
                                                                 .clone();
            stateLztwcPerc.scaleRegularTimeSeries(1.0 / _sacModelParameters.getLztwm());
            stateLztwcPerc.setMeasuringUnitNoChangeData(MeasuringUnit.percentDecimal);
            stateLztwcPerc.setTimeSeriesType(SacSmaModelConstants.LZTWC_PERC);
            addTStoResultMap(stateLztwcPerc);

            final RegularTimeSeries stateLzfscPerc = OHDUtilities.getRTSFromResultMap(_resultMap,
                                                                                      OHDConstants.SAC_STATE_LZFSC)
                                                                 .clone();
            stateLzfscPerc.scaleRegularTimeSeries(1.0 / _sacModelParameters.getLzfsm());
            stateLzfscPerc.setMeasuringUnitNoChangeData(MeasuringUnit.percentDecimal);
            stateLzfscPerc.setTimeSeriesType(SacSmaModelConstants.LZFSC_PERC);
            addTStoResultMap(stateLzfscPerc);

            final RegularTimeSeries stateLzfpcPerc = OHDUtilities.getRTSFromResultMap(_resultMap,
                                                                                      OHDConstants.SAC_STATE_LZFPC)
                                                                 .clone();
            stateLzfpcPerc.scaleRegularTimeSeries(1.0 / _sacModelParameters.getLzfpm());
            stateLzfpcPerc.setMeasuringUnitNoChangeData(MeasuringUnit.percentDecimal);
            stateLzfpcPerc.setTimeSeriesType(SacSmaModelConstants.LZFPC_PERC);
            addTStoResultMap(stateLzfpcPerc);

            final RegularTimeSeries stateAdimcPerc = OHDUtilities.getRTSFromResultMap(_resultMap,
                                                                                      OHDConstants.SAC_STATE_ADIMC)
                                                                 .clone();
            stateAdimcPerc.scaleRegularTimeSeries(1.0 / (_sacModelParameters.getUztwm() + _sacModelParameters.getLztwm()));
            stateAdimcPerc.setMeasuringUnitNoChangeData(MeasuringUnit.percentDecimal);
            stateAdimcPerc.setTimeSeriesType(SacSmaModelConstants.ADIMC_PERC);
            addTStoResultMap(stateAdimcPerc);

            //FGIX TS
            if(_sacModelParameters.useFrozenGroundCalc()
                && _sacModelParameters.getFgixTsInterval() != getDrivingTsInterval())
            {
                final RegularTimeSeries fgixRTS = OHDUtilities.removeRTSFromResultMap(_resultMap,
                                                                                      DataType.FGIX_DATATYPE);
                fgixRTS.integrateToIntervalInst(_sacModelParameters.getFgixTsInterval());
                super.addTStoResultMap(fgixRTS);
            }

            //6 ROCL output TS, accumulated type
            if(_sacModelParameters.getRunoffComponentTsInterval() != getDrivingTsInterval())
            {
                //going through the 6 runoff component TS inside _resultMap
                RegularTimeSeries roclRTS;
                for(final RUNOFF_COMPONENT runoffComp: RUNOFF_COMPONENT.values())
                {
                    final String roclName = runoffComp.getTypeName();
                    roclRTS = OHDUtilities.removeRTSFromResultMap(_resultMap, roclName);
                    roclRTS.integrateToIntervalAccum(_sacModelParameters.getRunoffComponentTsInterval());
                    super.addTStoResultMap(roclRTS);
                }
            }

            /** --------construct 5 soil moisture storages (SMZC) TSs by using output state TSs --------- */
            //because data modification, has to get independent copies

            final RegularTimeSeries smzcUztdefTs = OHDUtilities.getRTSFromResultMap(_resultMap,
                                                                                    OHDConstants.SAC_STATE_UZTWC)
                                                               .clone();
            smzcUztdefTs.setTimeSeriesType(SOIL_MOISTURE_STORAGE.UZTDEF.getTypeName()); //UZTDEF = UZTWM - UZTWC
            smzcUztdefTs.scaleRegularTimeSeries(-1.0);
            smzcUztdefTs.addToRegularTimeSeries(_sacModelParameters.getUztwm());

            final RegularTimeSeries smzcLztdefTs = OHDUtilities.getRTSFromResultMap(_resultMap,
                                                                                    OHDConstants.SAC_STATE_LZTWC)
                                                               .clone();
            smzcLztdefTs.setTimeSeriesType(SOIL_MOISTURE_STORAGE.LZTDEF.getTypeName()); //LZTDEF = LZTWM - LZTWC
            smzcLztdefTs.scaleRegularTimeSeries(-1.0);
            smzcLztdefTs.addToRegularTimeSeries(_sacModelParameters.getLztwm());

            /*
             * 5 soil moisture storages(SMZC): 1)UZTDEF(UZTWM-UZTWC); 2)UZFWC; 3)LZTDEF(LZTWM-LZTWC); 4)LZFSC; 5)LZFPC;
             * So there are three of them overlap with state(UZFWC, LZFSC and LZFPC) If parameter specification of SMZC
             * TS interval is not equal to drivingTS interval, then convert the three overlapping SMZC TS to new
             * interval. Note: SMZC in INST type.
             */
            if(_sacModelParameters.getSmzcTsInterval() != getDrivingTsInterval())
            {
                //adjust 2 TS interval
                smzcUztdefTs.integrateToIntervalInst(_sacModelParameters.getSmzcTsInterval());
                smzcLztdefTs.integrateToIntervalInst(_sacModelParameters.getSmzcTsInterval());

                //save a copy of the 3 overlapping TSs first, before they are modified
                final RegularTimeSeries stateUzfwcTs = OHDUtilities.getRTSFromResultMap(_resultMap,
                                                                                        SOIL_MOISTURE_STORAGE.UZFWC.getTypeName())
                                                                   .clone();
                stateUzfwcTs.integrateToIntervalInst(_sacModelParameters.getSmzcTsInterval());
                super.addTStoResultMap(stateUzfwcTs);

                final RegularTimeSeries stateLzfscTs = OHDUtilities.getRTSFromResultMap(_resultMap,
                                                                                        SOIL_MOISTURE_STORAGE.LZFSC.getTypeName())
                                                                   .clone();
                stateLzfscTs.integrateToIntervalInst(_sacModelParameters.getSmzcTsInterval());
                super.addTStoResultMap(stateLzfscTs);

                final RegularTimeSeries stateLzfpcTs = OHDUtilities.getRTSFromResultMap(_resultMap,
                                                                                        SOIL_MOISTURE_STORAGE.LZFPC.getTypeName())
                                                                   .clone();
                stateLzfpcTs.integrateToIntervalInst(_sacModelParameters.getSmzcTsInterval());
                super.addTStoResultMap(stateLzfpcTs);

            }
            //adding the 2 TSs to the resultMpa. Their interval may or may not be adjusted
            addTStoResultMap(smzcUztdefTs);
            addTStoResultMap(smzcLztdefTs);

        } //if(_isFullVersion)
        else
        {//slimVersion
            if(_printLog)
            {
                _logger.log(Logger.DEBUG, "Running in slim version, so only output TCI time series.");
            }

            /*
             * Remove all the secondary output TSs from _resultMap: A)(6 runoff components -- ROCL) 1)DIR_RO, 2)IMP_RO,
             * 3)INT_RO, 4)PRI_RO, 5)SUP_RO, 6)SUR_RO; B) (5 soil moisture storages -- SMZC) 1)LZFPC, 2)LZFSC, 3)LZTDEF,
             * 4)UZFWC, 5)UZTDEF; C)FGIX when using FRZE(It was optional output in NWSRFS, but is required now in FEWS).
             * So there are 11 or 12 output TSs belongs to the secondary group. Or in another words, remove all the
             * output TSs, except TCI
             */

            final Iterator<Map.Entry<String, RegularTimeSeries>> ite = _resultMap.entrySet().iterator();

            while(ite.hasNext())
            {
                final Map.Entry<String, RegularTimeSeries> entry = ite.next();

                //remove all the output TSs from _resultMap, except TCI
                if(entry.getKey().contains(DataType.CHANNEL_INFLOW_DATATYPE) == false)
                {
                    ite.remove();
                }
            } //close while loop

        }//close else        

        _logger.log(Logger.DEBUG, "SacSmaModel has successfully finished.");

    }//close execute() 

    /**
     * Besides validation, also retrieve individual input time series from _tsList.
     */
    @Override
    protected void runModelDriverValidation() throws Exception
    {
        _sacSmaTechnique = new SacSmaTechnique(getDriverProperties(), _logger);

        //TECHNIQUE FROST only be meaningful when using FRZE
        if(_sacModelParameters.useFrozenGroundCalc())
        {
            if(_sacSmaTechnique.getBoolValue(SacSmaModelConstants.FROST_TAG) == false)
            {
                _sacModelParameters.setUseFrozenGroundCalc(false); //FROST is false, turn off FRZE
                _sacModelState.setFgix(0.0); //over-write FGIX to 0.0

                _logger.log(Logger.DEBUG,
                            "The TECHNIQUE FROST is false so that FROZEN GROUND is not used. The state FGIX is set to 0.0");
            }

            //FROST is true, do not do anything
        }
        else
        {
            if(_sacSmaTechnique.getBoolValue(SacSmaModelConstants.FROST_TAG))
            {
                _logger.log(Logger.DEBUG,
                            "The property FROST is ignored since FRZE is not defined at all. No FRZE in model calculation.");
            }
        }

        final Iterator<RegularTimeSeries> ite = getTsList().iterator();

        RegularTimeSeries inputRts;

        String tsType;

        final List<RegularTimeSeries> modsTsList = new ArrayList<RegularTimeSeries>(SacSma_MODS.values().length);

        while(ite.hasNext())
        {
            inputRts = ite.next();

            tsType = inputRts.getTimeSeriesType();

            if(PRECIP_DATATYPE.isPrecipType(tsType)) //mandatory input TS
            {
                _drivingTs = inputRts;

                _sacModelParameters.setRaimTsInterval(_drivingTs.getIntervalInHours());
            }
            else if(tsType.equals(DataType.POTENTIAL_ET_DATATYPE))
            {//optional input TS

                if(_sacModelParameters.useMAPEInput())
                {
                    _mapeTs = inputRts;

                    /*
                     * NWSRFS requires the interval be 24 hours, but for FEWS, no need for this requirement; but must be
                     * multiple of driving TS(RAIM) interval. This will be automatically validated during disIntegrate
                     * below.
                     */
                    if(_printLog)
                    {
                        _logger.log(Logger.DEBUG, "Using optional input MAPE time series.");
                    }
                }
                else
                {//since not using it, delete it from _tsList
                    ite.remove();
                    if(_printLog)
                    {
                        _logger.log(Logger.DEBUG, "Not use optional input MAPE time series.");
                    }
                }
            }
            else if(tsType.equals(DataType.SASC_DATATYPE))
            {//optional input TS

                if(_sacModelParameters.useSascInputTS())
                {
                    _sascTs = inputRts;
                    _logger.log(Logger.DEBUG, "Using optional input SASC time series.");
                }
                else
                {//since not using it, delete it from _tsList
                    ite.remove();
                    if(_printLog)
                    {
                        _logger.log(Logger.DEBUG, "Not use optional input SASC time series.");
                    }
                }
            }

            /** -----------------TWO INPUT TSs for FROZEN GROUND CALCULATION(one required, one optional) ---------- */

            else if(tsType.equals(DataType.MAT_DATATYPE))
            {//MAT TS is required for FRZE

                if(_sacModelParameters.useFrozenGroundCalc())
                {
                    _matTs = inputRts;
                }
                else
                {//since not using it, delete it from _tsList
                    ite.remove();
                }

            }
            else if(tsType.equals(DataType.SWE_DATATYPE))
            { //SWE TS is optional for FRZE

                if(_sacModelParameters.useFrozenGroundCalc() && _sacModelParameters.useWEinputTS())
                {
                    _weTs = inputRts;
                    if(_printLog)
                    {
                        _logger.log(Logger.DEBUG, "Use optional input SWE time series in Frozen Ground Calculation.");
                    }
                }
                else
                {//since not using it, delete it from _tsList
                    ite.remove();
                    if(_printLog)
                    {
                        _logger.log(Logger.DEBUG,
                                    "Not use optional input SWE time series in Frozen Ground Calculation.");
                    }
                }

            }
            /*
             * FEWS will always provide input MODS time series xml file regardless of having MODS or not: when not using
             * a MOD, that MOD time series still have a header in the xml file. It is parsed and stored in _tsList with
             * all values be -999.0. To let model computation not read these empty time series, they are removed now.
             */
            else if(inputRts.isEmpty())//if passed all "if" checks above, must be MOD ts
            {
                if(_printLog)
                {
                    _logger.log(Logger.DEBUG, "Input " + tsType
                        + " MOD time series is empty. Removed it now so not using it anymore.");
                }

                ite.remove(); //delete the RTS from the list
            }
            else if(tsType.equalsIgnoreCase(SacSma_MODS.UZTWC.getDataType()))
            {
                if(inputRts.getStartTime() <= getSwitchFromObservedToForecastTime())
                {
                    inputRts.trimTimeSeriesAtEnd(getSwitchFromObservedToForecastTime());
                    if(inputRts.getMeasuringUnit() != MeasuringUnit.mm)
                    {
                        inputRts.setMeasuringUnit(MeasuringUnit.mm);
                    }
                    inputRts.setMaxValue(_sacModelParameters.getUztwm());
                    modsTsList.add(inputRts.clone());

                    _logger.log(Logger.DEBUG, "Input " + tsType + " MOD time series is present.");
                }
                ite.remove();
            }
            else if(tsType.equalsIgnoreCase(SacSma_MODS.UZFWC.getDataType()))
            {
                if(inputRts.getStartTime() <= getSwitchFromObservedToForecastTime())
                {
                    inputRts.trimTimeSeriesAtEnd(getSwitchFromObservedToForecastTime());
                    if(inputRts.getMeasuringUnit() != MeasuringUnit.mm)
                    {
                        inputRts.setMeasuringUnit(MeasuringUnit.mm);
                    }
                    inputRts.setMaxValue(_sacModelParameters.getUzfwm());
                    modsTsList.add(inputRts.clone());

                    _logger.log(Logger.DEBUG, "Input " + tsType + " MOD time series is present.");
                }
                ite.remove();
            }
            else if(tsType.equalsIgnoreCase(SacSma_MODS.LZTWC.getDataType()))
            {
                if(inputRts.getStartTime() <= getSwitchFromObservedToForecastTime())
                {
                    inputRts.trimTimeSeriesAtEnd(getSwitchFromObservedToForecastTime());
                    if(inputRts.getMeasuringUnit() != MeasuringUnit.mm)
                    {
                        inputRts.setMeasuringUnit(MeasuringUnit.mm);
                    }
                    inputRts.setMaxValue(_sacModelParameters.getLztwm());
                    modsTsList.add(inputRts.clone());

                    _logger.log(Logger.DEBUG, "Input " + tsType + " MOD time series is present.");
                }
                ite.remove();
            }
            else if(tsType.equalsIgnoreCase(SacSma_MODS.LZFPC.getDataType()))
            {
                if(inputRts.getStartTime() <= getSwitchFromObservedToForecastTime())
                {
                    inputRts.trimTimeSeriesAtEnd(getSwitchFromObservedToForecastTime());
                    if(inputRts.getMeasuringUnit() != MeasuringUnit.mm)
                    {
                        inputRts.setMeasuringUnit(MeasuringUnit.mm);
                    }
                    inputRts.setMaxValue(_sacModelParameters.getLzfpm());
                    modsTsList.add(inputRts.clone());

                    _logger.log(Logger.DEBUG, "Input " + tsType + " MOD time series is present.");
                }
                ite.remove();
            }
            else if(tsType.equalsIgnoreCase(SacSma_MODS.LZFSC.getDataType()))
            {
                if(inputRts.getStartTime() <= getSwitchFromObservedToForecastTime())
                {
                    inputRts.trimTimeSeriesAtEnd(getSwitchFromObservedToForecastTime());
                    if(inputRts.getMeasuringUnit() != MeasuringUnit.mm)
                    {
                        inputRts.setMeasuringUnit(MeasuringUnit.mm);
                    }
                    inputRts.setMaxValue(_sacModelParameters.getLzfsm());
                    modsTsList.add(inputRts.clone());

                    _logger.log(Logger.DEBUG, "Input " + tsType + " MOD time series is present.");
                }
                ite.remove();
            }
            else if(tsType.equalsIgnoreCase(SacSma_MODS.ADIMC.getDataType()))
            {
                if(inputRts.getStartTime() <= getSwitchFromObservedToForecastTime())
                {
                    inputRts.trimTimeSeriesAtEnd(getSwitchFromObservedToForecastTime());
                    if(inputRts.getMeasuringUnit() != MeasuringUnit.mm)
                    {
                        inputRts.setMeasuringUnit(MeasuringUnit.mm);
                    }

                    final double maxValue = _sacModelParameters.getUztwm() + _sacModelParameters.getLztwm();
                    final double minValue = _sacModelState.getUztwc();
                    inputRts.setMaxValue(maxValue);
                    inputRts.setMinValue(minValue);

                    modsTsList.add(inputRts.clone());

                    _logger.log(Logger.DEBUG, "Input " + tsType + " MOD time series is present.");
                }

                ite.remove();
            }
            //Added for FB131 NCRFC requested - CP 10/22/14
            else if(tsType.equalsIgnoreCase(SacSma_MODS.UZTWD.getDataType()))
            {
                if(inputRts.getStartTime() <= getSwitchFromObservedToForecastTime())
                {
                    inputRts.trimTimeSeriesAtEnd(getSwitchFromObservedToForecastTime());
                    if(inputRts.getMeasuringUnit() != MeasuringUnit.mm)
                    {
                        inputRts.setMeasuringUnit(MeasuringUnit.mm);
                    }
                    modsTsList.add(inputRts.clone());

                    _logger.log(Logger.DEBUG, "Input " + tsType + " MOD time series is present.");
                }
                ite.remove();
            }
            else if(tsType.equalsIgnoreCase(SacSma_MODS.LZTWD.getDataType()))
            {
                if(inputRts.getStartTime() <= getSwitchFromObservedToForecastTime())
                {
                    inputRts.trimTimeSeriesAtEnd(getSwitchFromObservedToForecastTime());
                    if(inputRts.getMeasuringUnit() != MeasuringUnit.mm)
                    {
                        inputRts.setMeasuringUnit(MeasuringUnit.mm);
                    }
                    modsTsList.add(inputRts.clone());

                    _logger.log(Logger.DEBUG, "Input " + tsType + " MOD time series is present.");
                }
                ite.remove();
            }
            else if(tsType.equalsIgnoreCase(SacSma_MODS.UZTWCADJ.getDataType()))
            {
                if(inputRts.getStartTime() <= getSwitchFromObservedToForecastTime())
                {
                    inputRts.trimTimeSeriesAtEnd(getSwitchFromObservedToForecastTime());
                    if(inputRts.getMeasuringUnit() != MeasuringUnit.mm)
                    {
                        inputRts.setMeasuringUnit(MeasuringUnit.mm);
                    }
                    modsTsList.add(inputRts.clone());

                    _logger.log(Logger.DEBUG, "Input " + tsType + " MOD time series is present.");
                }
                ite.remove();
            }
            else if(tsType.equalsIgnoreCase(SacSma_MODS.LZTWCADJ.getDataType()))
            {
                if(inputRts.getStartTime() <= getSwitchFromObservedToForecastTime())
                {
                    inputRts.trimTimeSeriesAtEnd(getSwitchFromObservedToForecastTime());
                    if(inputRts.getMeasuringUnit() != MeasuringUnit.mm)
                    {
                        inputRts.setMeasuringUnit(MeasuringUnit.mm);
                    }
                    modsTsList.add(inputRts.clone());

                    _logger.log(Logger.DEBUG, "Input " + tsType + " MOD time series is present.");
                }
                ite.remove();
            }
            // End FB 131
            else if(tsType.equalsIgnoreCase(SacSma_MODS.FGIX.getDataType()))
            {
                if(inputRts.getStartTime() <= getSwitchFromObservedToForecastTime())
                {
                    inputRts.trimTimeSeriesAtEnd(getSwitchFromObservedToForecastTime());
                    if(inputRts.getMeasuringUnit() != MeasuringUnit.degreesCelsius)
                    {
                        inputRts.setMeasuringUnit(MeasuringUnit.degreesCelsius);
                    }
                    modsTsList.add(inputRts.clone());

                    _logger.log(Logger.DEBUG, "Input " + tsType + " MOD time series is present.");
                }
                ite.remove();
            }
            else if(tsType.equalsIgnoreCase(SacSma_MODS.SACBASEF.getDataType()))
            {
                if(inputRts.getStartTime() <= getSwitchFromObservedToForecastTime())
                {
                    if(inputRts.getMeasuringUnit() != MeasuringUnit.unitlessReal)
                    {
                        throw new Exception("Error: input SACBASEF MOD time series unit is not "
                            + MeasuringUnit.unitlessReal.getName());
                    }

                    inputRts.trimTimeSeriesAtEnd(getSwitchFromObservedToForecastTime());
                    modsTsList.add(inputRts.clone());

                    _logger.log(Logger.DEBUG, "Input " + tsType + " MOD time series is present.");
                }
                ite.remove();
            }

        } //close while loop, finished all the extraction of input TSs

        if(_drivingTs == null)
        {
            throw new Exception("There is no driving TS " + DataType.RAIM_DATATYPE + " in inputs.xml.");
        }

        //validate each input MODS ts and check if it in sync with _drivingTs

        super.checkInputTsInSync(modsTsList);

        _sacsmaModsTsHolder = new SacSmaModsTSHolder(modsTsList);

        /* ----- check if input TSs are present as they should be and convert their intervals if needed ------ */
        if(_sacModelParameters.useSascInputTS())
        {

            if(_sascTs == null)
            {
                throw new Exception("The input SASC time series is required since the parameter "
                    + SacSmaModelConstants.SASC_INPUT_TAG + " is true.");
            }

            //If SASC interval != RAIM interval(but must be even multiple of it), convert to that small interval
            if((_sascTs.getIntervalInHours() != getDrivingTsInterval()))
            {
                _sascTs.disIntegrateToIntervalInst(getDrivingTsInterval());
            }
        }

        if(_sacModelParameters.useMAPEInput())
        {

            if(_mapeTs == null)
            {
                throw new Exception("The input MAPE time series is required since the parameter "
                    + SacSmaModelConstants.MAPE_INPUT_TAG + " is true.");
            }

            //if _mapeTs interval is greater than _raim(but must be even multiple of it), convert _mapeTs to that smaller interval
            if((_mapeTs.getIntervalInHours() != getDrivingTsInterval()))
            {
                _mapeTs.disIntegrateToIntervalAccum(getDrivingTsInterval());
            }
        }

        /** ---------------------Check input TS intervals constraints as in frdfg1.f---------- */

        if(_sacModelParameters.useFrozenGroundCalc())
        { //for Frozen Ground Calculation

            if(_matTs == null)
            {
                throw new Exception("For Frozen Ground calculation, the input MAT time series is required.");
            }

            //MAT TS is mandatory for Frozen Ground Calculation
            if(_matTs.getIntervalInHours() > getDrivingTsInterval())
            {//If MAT interval is not equal to RAIM interval, convert MAT TS to RAIM interval

                _matTs.disIntegrateToIntervalInst(getDrivingTsInterval());
            }

            //SWE TS is optional for Frozen Ground Calculation
            if(_sacModelParameters.useWEinputTS())
            { //convert SWE to smaller interval

                if(_weTs == null)
                {
                    throw new Exception("For Frozen Ground calculation, the input SWE time series is required since the parameter "
                        + SacSmaModelConstants.WE_INPUT_TAG + " is true.");
                }

                if((_weTs.getIntervalInHours() != getDrivingTsInterval()))
                {
                    _weTs.disIntegrateToIntervalInst(getDrivingTsInterval());
                }
            }

        } //close if(_sacModelParameters.useFrozenGroundCalc())

        super.runModelDriverValidation();

        //parameters and states are validated
        _sacModelParameters.validateParams(getComputationEndTime() - getInitialStateTime());

        if(super.needCarryoverTransfer())
        {//do carryover transfer

            _sacModelState.validateState(_sacModelSavedParameters);
            _sacModelState.doCarryOverTransfer(_sacModelSavedParameters, _sacModelParameters);
        }
        else
        {//no need to do carryover transfer
            _sacModelState.validateState(_sacModelParameters);
        }

    } //close method

    /**
     * @return - The precipitation time series from list of input time series
     */
    RegularTimeSeries getPrecipTs()
    {
        return _drivingTs;
    }

    RegularTimeSeries getMapeTs()
    {
        return _mapeTs;
    }

    public void setMapeTs(final RegularTimeSeries mapeTS)
    {
        _mapeTs = mapeTS;
    }

    RegularTimeSeries getSascTs()
    {
        return _sascTs;
    }

    RegularTimeSeries getMatTs()
    {
        return _matTs;
    }

    public void setMatTs(final RegularTimeSeries matTS)
    {
        _matTs = matTS;
    }

    RegularTimeSeries getWeTs()
    {
        return _weTs;
    }

    @Override
    public SacSmaModelParameters getParameters()
    {
        return (SacSmaModelParameters)super._parameters;
    }

    @Override
    public SacSmaModelState getState()
    {
        return (SacSmaModelState)super._state;
    }

}
