package ohd.hseb.ohdmodels.sacsma;

import java.util.Arrays;

import javax.management.timer.Timer;

import ohd.hseb.util.fews.ohdmodels.ModelException;
import ohd.hseb.util.fews.ohdmodels.ModelParameters;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.OHDUtilities;
import ohd.hseb.util.fews.Parameters;

/**
 * This class holds all the SAC-SMA Model parameters. It is a subclass of {@link ModelParameters} and interface
 * {@link IModelParameters}. They are specified in the opt file:
 * <p>
 * <ul>
 * <li>Card #1,
 * <li>Card #2(could be completely blank),
 * <li>Card #3,
 * <li>Card #4,
 * <li>Card #5,
 * <li>Card #F1,(F1 and F2 are for FRZE)
 * <li>Card #F2
 * </ul>
 * Values are initially loaded into super._parasmMap, then extracted into instance variables to avoid frequent accessing
 * the Map. Setter methods are not needed, since the model does not change parameters. All values in METRIC unit.
 * <p>
 * {@link SacSmaOptFileConverter} sets the parameter values by using various insertParameter(..) methods in
 * {@link Parameters}.
 * 
 * @author FewsPilot Team
 */
final public class SacSmaModelParameters extends ModelParameters
{

    //these Tags are used by sacsmaHT project and nwsrfsadapter, that is the only reason this class be public

    private double _pxadj;
    private double _rserv;
    private double _side;
    private double _pfree;
    private double _lzpk;
    private double _lzsk;
    private double _efc;
    private double _rexp;
    private double _zperc;
    private double _riva;
    private double _adimp;
    private double _pctim;
    private double _uzk;
    private double[] _ett;

    private double _lzfpm;
    private double _lzfsm;
    private double _lztwm;
    private double _uzfwm;
    private double _peadj;
    private double _uztwm;

    private boolean _useMAPEInput = false; //default is NO
    private boolean _useSASCInput = false; //default is NO
    private int _runoffComponentTsInterval = 24;
    private int _smzcTsInterval = 24;

    /** ------------------------------FRZE -------------------------------------- */
    private boolean _useFrozenGroundCalc = false; //default value

    //card F1
    private boolean _useWeInputInFRZE = false; //default    
    private int _fgixInterval = 0; //default is _raimTsInterval; "0" is used as a flag to indicate it is set from params.xml or not

    //card F2
    private double _csoil;
    private double _csnow;
    private double _gch;
    private double _rthaw;
    private double _frtemp;
    private double _satr;
    private double _frexp;

    private int _raimTsInterval; //required input TS, not in params.xml, set in SacSmaModelDriver

    /**
     * Instantiates a new sacsma model parameters. Values are set by parsing params.xml file by FewsXMLParser.java(in
     * package ohdfewsadapter) method public void parseParameters(final String paramFileName).
     */
    public SacSmaModelParameters()
    {
        //empty
    }

    /**
     * Over-loaded method to take an extra parameter of the computation period(long) for validation. Inside the method,
     * calls {@link #validateParams(Logger)} to do most parameter validation.
     */
    void validateParams(final long computPeriod) throws Exception
    {
        this.validateParams();

        //check if the running period is too short(shorter than runoff component or soil moisture component interval)
        if(getRunoffComponentTsInterval() * Timer.ONE_HOUR > computPeriod
            || getSmzcTsInterval() * Timer.ONE_HOUR > computPeriod)
        {
            final StringBuffer errorMessage = new StringBuffer();
            errorMessage.append("Running period(")
                        .append(computPeriod / Timer.ONE_HOUR)
                        .append("hr) is shorter than either RUNOFF_COMPONENT_INTERVAL(")
                        .append(getRunoffComponentTsInterval())
                        .append("hr) or SMZC interval(")
                        .append(getSmzcTsInterval())
                        .append("hr).");

            throw new ModelException(errorMessage.toString());
        }

    }

    /**
     * Extract values from super._parametersMap into instance variables. The values in super._parametersMap are set by
     * parsing parameter xml file in OHDFewsAdapter.java.
     * 
     * @throws Exception
     */
    @Override
    public void extractValuesFromMap() throws Exception
    {

        _pxadj = getDoubleDataParameter(SacSmaModelConstants.PXADJ_TAG);

        _rserv = getDoubleDataParameter(SacSmaModelConstants.RSERV_TAG);

        _side = getDoubleDataParameter(SacSmaModelConstants.SIDE_TAG);

        _pfree = getDoubleDataParameter(SacSmaModelConstants.PFREE_TAG);

        _lzpk = getDoubleDataParameter(SacSmaModelConstants.LZPK_TAG);

        _lzsk = getDoubleDataParameter(SacSmaModelConstants.LZSK_TAG);

        _efc = getDoubleDataParameter(SacSmaModelConstants.EFC_TAG);

        _rexp = getDoubleDataParameter(SacSmaModelConstants.REXP_TAG);

        _zperc = getDoubleDataParameter(SacSmaModelConstants.ZPERC_TAG);

        _riva = getDoubleDataParameter(SacSmaModelConstants.RIVA_TAG);

        _adimp = getDoubleDataParameter(SacSmaModelConstants.ADIMP_TAG);

        _pctim = getDoubleDataParameter(SacSmaModelConstants.PCTIM_TAG);

        _uzk = getDoubleDataParameter(SacSmaModelConstants.UZK_TAG);

        _lzfpm = getDoubleDataParameter(SacSmaModelConstants.LZFPM_TAG);

        _lzfsm = getDoubleDataParameter(SacSmaModelConstants.LZFSM_TAG);

        _lztwm = getDoubleDataParameter(SacSmaModelConstants.LZTWM_TAG);

        _uzfwm = getDoubleDataParameter(SacSmaModelConstants.UZFWM_TAG);

        _peadj = getDoubleDataParameter(SacSmaModelConstants.PEADJ_TAG);

        _uztwm = getDoubleDataParameter(SacSmaModelConstants.UZTWM_TAG);

        //extract ett array from super._paramsMap            

        if(isParamExisting(SacSmaModelConstants.ET_DEMAND_CURVE_TAG))
        {//new format
            _ett = super.getDoubleArrayParameter(SacSmaModelConstants.ET_DEMAND_CURVE_TAG);
        }
        else
        {//old format, to be backward compatible
            _ett = new double[OHDConstants.NUMBER_OF_MONTHS_IN_YEAR];
            for(int i = 0; i < _ett.length; i++)
            {
                switch(i)
                {
                    case 0:
                        _ett[i] = getDoubleDataParameter(SacSmaModelConstants.ETTJAN_TAG);
                        break;
                    case 1:
                        _ett[i] = getDoubleDataParameter(SacSmaModelConstants.ETTFEB_TAG);
                        break;
                    case 2:
                        _ett[i] = getDoubleDataParameter(SacSmaModelConstants.ETTMAR_TAG);
                        break;
                    case 3:
                        _ett[i] = getDoubleDataParameter(SacSmaModelConstants.ETTAPR_TAG);
                        break;
                    case 4:
                        _ett[i] = getDoubleDataParameter(SacSmaModelConstants.ETTMAY_TAG);
                        break;
                    case 5:
                        _ett[i] = getDoubleDataParameter(SacSmaModelConstants.ETTJUN_TAG);
                        break;
                    case 6:
                        _ett[i] = getDoubleDataParameter(SacSmaModelConstants.ETTJUL_TAG);
                        break;
                    case 7:
                        _ett[i] = getDoubleDataParameter(SacSmaModelConstants.ETTAUG_TAG);
                        break;
                    case 8:
                        _ett[i] = getDoubleDataParameter(SacSmaModelConstants.ETTSEP_TAG);
                        break;
                    case 9:
                        _ett[i] = getDoubleDataParameter(SacSmaModelConstants.ETTOCT_TAG);
                        break;
                    case 10:
                        _ett[i] = getDoubleDataParameter(SacSmaModelConstants.ETTNOV_TAG);
                        break;
                    case 11:
                        _ett[i] = getDoubleDataParameter(SacSmaModelConstants.ETTDEC_TAG);
                        break;
                    default:
                        //should not reach here
                        _ett[i] = 0.0;
                        break;
                } //close switch loop

            } //close for loop
        }

        if(isParamExisting(SacSmaModelConstants.RUNOFF_COMPOENT_INTERVAL_TAG)
            && getIntDataParameter(SacSmaModelConstants.RUNOFF_COMPOENT_INTERVAL_TAG) != 0)
        {
            _runoffComponentTsInterval = getIntDataParameter(SacSmaModelConstants.RUNOFF_COMPOENT_INTERVAL_TAG);
        }

        if(isParamExisting(SacSmaModelConstants.SMZC_INTERVAL_TAG) == true
            && getIntDataParameter(SacSmaModelConstants.SMZC_INTERVAL_TAG) != 0)
        {
            _smzcTsInterval = getIntDataParameter(SacSmaModelConstants.SMZC_INTERVAL_TAG);
        }

        if(isParamExisting(SacSmaModelConstants.MAPE_INPUT_TAG))
        {
            _useMAPEInput = getBooleanDataParameter(SacSmaModelConstants.MAPE_INPUT_TAG);
        }

        if(isParamExisting(SacSmaModelConstants.SASC_INPUT_TAG))
        {
            _useSASCInput = getBooleanDataParameter(SacSmaModelConstants.SASC_INPUT_TAG);
        }

        /** ------------------------------FRZE -------------------------------------- */
        if(isParamExisting(SacSmaModelConstants.FRZE_INPUT_TAG)
            && getBooleanDataParameter(SacSmaModelConstants.FRZE_INPUT_TAG))
        {
            _useFrozenGroundCalc = true;

            _csoil = getDoubleDataParameter(SacSmaModelConstants.CSOIL_TAG);
            _csnow = getDoubleDataParameter(SacSmaModelConstants.CSNOW_TAG);
            _gch = getDoubleDataParameter(SacSmaModelConstants.GCH_TAG);
            _rthaw = getDoubleDataParameter(SacSmaModelConstants.RTHAW_TAG);
            _frtemp = getDoubleDataParameter(SacSmaModelConstants.FRTEMP_TAG);
            _satr = getDoubleDataParameter(SacSmaModelConstants.SATR_TAG);
            _frexp = getDoubleDataParameter(SacSmaModelConstants.FREXP_TAG);

            if(isParamExisting(SacSmaModelConstants.WE_INPUT_TAG))
            {
                _useWeInputInFRZE = getBooleanDataParameter(SacSmaModelConstants.WE_INPUT_TAG);
            }

            if(isParamExisting(SacSmaModelConstants.FGIX_INTERVAL_TAG) == true
                && getIntDataParameter(SacSmaModelConstants.FGIX_INTERVAL_TAG) != 0)
            {
                _fgixInterval = getIntDataParameter(SacSmaModelConstants.FGIX_INTERVAL_TAG);
            }
        }
        else
        {
            _logger.log(Logger.DEBUG, "Not using Frozen Ground Calculation.");
        }

    } //close extractValuesFromMap()

    /**
     * Set values from instance variables into super._parametersMap Map. The values in super._parametersMap are set by
     * parsing parameter xml file in OHDFewsAdapter.java.
     * 
     * @throws Exception
     */
    @Override
    public void setValuesToMap() throws Exception
    {

        super.insertParameter(SacSmaModelConstants.PXADJ_TAG, _pxadj);

        super.insertParameter(SacSmaModelConstants.RSERV_TAG, _rserv);

        super.insertParameter(SacSmaModelConstants.SIDE_TAG, _side);

        super.insertParameter(SacSmaModelConstants.PFREE_TAG, _pfree);

        super.insertParameter(SacSmaModelConstants.LZPK_TAG, _lzpk);

        super.insertParameter(SacSmaModelConstants.LZSK_TAG, _lzsk);

        super.insertParameter(SacSmaModelConstants.EFC_TAG, _efc);

        super.insertParameter(SacSmaModelConstants.REXP_TAG, _rexp);

        super.insertParameter(SacSmaModelConstants.ZPERC_TAG, _zperc);

        super.insertParameter(SacSmaModelConstants.RIVA_TAG, _riva);

        super.insertParameter(SacSmaModelConstants.ADIMP_TAG, _adimp);

        super.insertParameter(SacSmaModelConstants.PCTIM_TAG, _pctim);

        super.insertParameter(SacSmaModelConstants.UZK_TAG, _uzk);

        super.insertParameter(SacSmaModelConstants.LZFPM_TAG, _lzfpm);

        super.insertParameter(SacSmaModelConstants.LZFSM_TAG, _lzfsm);

        super.insertParameter(SacSmaModelConstants.LZTWM_TAG, _lztwm);

        super.insertParameter(SacSmaModelConstants.UZFWM_TAG, _uzfwm);

        super.insertParameter(SacSmaModelConstants.PEADJ_TAG, _peadj);

        super.insertParameter(SacSmaModelConstants.UZTWM_TAG, _uztwm);

        //extract ett array from super._paramsMap            

        if(isParamExisting(SacSmaModelConstants.ET_DEMAND_CURVE_TAG))
        {//new format
            super.insertParameter(SacSmaModelConstants.ET_DEMAND_CURVE_TAG, _ett);
        }
        else
        {//old format, to be backward compatible

            super.insertParameter(SacSmaModelConstants.ETTJAN_TAG, _ett[0]);

            super.insertParameter(SacSmaModelConstants.ETTFEB_TAG, _ett[1]);

            super.insertParameter(SacSmaModelConstants.ETTMAR_TAG, _ett[2]);

            super.insertParameter(SacSmaModelConstants.ETTAPR_TAG, _ett[3]);

            super.insertParameter(SacSmaModelConstants.ETTMAY_TAG, _ett[4]);

            super.insertParameter(SacSmaModelConstants.ETTJUN_TAG, _ett[5]);

            super.insertParameter(SacSmaModelConstants.ETTJUL_TAG, _ett[6]);

            super.insertParameter(SacSmaModelConstants.ETTAUG_TAG, _ett[7]);

            super.insertParameter(SacSmaModelConstants.ETTSEP_TAG, _ett[8]);

            super.insertParameter(SacSmaModelConstants.ETTOCT_TAG, _ett[9]);

            super.insertParameter(SacSmaModelConstants.ETTNOV_TAG, _ett[10]);

            super.insertParameter(SacSmaModelConstants.ETTDEC_TAG, _ett[11]);

        }

        if(isParamExisting(SacSmaModelConstants.RUNOFF_COMPOENT_INTERVAL_TAG)
            && getIntDataParameter(SacSmaModelConstants.RUNOFF_COMPOENT_INTERVAL_TAG) != 0)
        {
            super.insertParameter(SacSmaModelConstants.RUNOFF_COMPOENT_INTERVAL_TAG, _runoffComponentTsInterval);
        }

        if(isParamExisting(SacSmaModelConstants.SMZC_INTERVAL_TAG) == true
            && getIntDataParameter(SacSmaModelConstants.SMZC_INTERVAL_TAG) != 0)
        {
            super.insertParameter(SacSmaModelConstants.SMZC_INTERVAL_TAG, _smzcTsInterval);
        }

        if(isParamExisting(SacSmaModelConstants.MAPE_INPUT_TAG))
        {
            super.insertParameter(SacSmaModelConstants.MAPE_INPUT_TAG, _useMAPEInput);
        }

        if(isParamExisting(SacSmaModelConstants.SASC_INPUT_TAG))
        {
            super.insertParameter(SacSmaModelConstants.SASC_INPUT_TAG, _useSASCInput);
        }

        /** ------------------------------FRZE -------------------------------------- */
        if(isParamExisting(SacSmaModelConstants.FRZE_INPUT_TAG)
            && getBooleanDataParameter(SacSmaModelConstants.FRZE_INPUT_TAG))
        {
            _useFrozenGroundCalc = true;

            super.insertParameter(SacSmaModelConstants.CSOIL_TAG, _csoil);
            super.insertParameter(SacSmaModelConstants.CSNOW_TAG, _csnow);
            super.insertParameter(SacSmaModelConstants.GCH_TAG, _gch);
            super.insertParameter(SacSmaModelConstants.RTHAW_TAG, _rthaw);
            super.insertParameter(SacSmaModelConstants.FRTEMP_TAG, _frtemp);
            super.insertParameter(SacSmaModelConstants.SATR_TAG, _satr);
            super.insertParameter(SacSmaModelConstants.FREXP_TAG, _frexp);

            if(isParamExisting(SacSmaModelConstants.WE_INPUT_TAG))
            {
                super.insertParameter(SacSmaModelConstants.WE_INPUT_TAG, _useWeInputInFRZE);
            }

            if(isParamExisting(SacSmaModelConstants.FGIX_INTERVAL_TAG) == true
                && getIntDataParameter(SacSmaModelConstants.FGIX_INTERVAL_TAG) != 0)
            {
                super.insertParameter(SacSmaModelConstants.FGIX_INTERVAL_TAG, _fgixInterval);
            }
        }
        else
        {
            _logger.log(Logger.DEBUG, "Not using Frozen Ground Calculation.");
        }

    } //close setValuesToMap()

    /**
     * Does SAC-SMA parameter validation.
     */
    public void validateParams() throws ModelException, Exception
    {
        if(_fgixInterval == 0)
        {
            _fgixInterval = _raimTsInterval; //_raimTsInterval must already set in SacSmaModelDriver validation
        }
        else
        {
            //_fgixInterval was set from params.xml, verify it be multiple of RAIM TS
            if(OHDUtilities.isEvenMultiple(_fgixInterval, _raimTsInterval) == false)
            {
                throw new ModelException("Output FGIX TS interval is not multiple of RAIM TS interval");
            }
        }

        _logger.log(Logger.DEBUG, "SacSmaModelParameters validation has passed");

    } //close method

    /**
     * @return the _adimp
     */
    public double getAdimp()
    {
        return _adimp;
    }

    public void setAdimp(final double adimpValue)
    {
        _adimp = adimpValue;
    }

    /**
     * @return the _efc
     */
    public double getEfc()
    {
        return _efc;
    }

    public void setEfc(final double efcValue)
    {
        _efc = efcValue;
    }

    /**
     * @return the _lzfpm
     */
    public double getLzfpm()
    {
        return _lzfpm;
    }

    public void setLzfpm(final double lzfpmValue)
    {
        _lzfpm = lzfpmValue;
    }

    /**
     * @return the _lzfsm
     */
    public double getLzfsm()
    {
        return _lzfsm;
    }

    public void setLzfsm(final double lzfsmValue)
    {
        _lzfsm = lzfsmValue;
    }

    /**
     * @return the _lzpk
     */
    public double getLzpk()
    {
        return _lzpk;
    }

    public void setLzpk(final double lzpkValue)
    {
        _lzpk = lzpkValue;
    }

    /**
     * @return the _lzsk
     */
    public double getLzsk()
    {
        return _lzsk;
    }

    public void setLzsk(final double lzskValue)
    {
        _lzsk = lzskValue;
    }

    /**
     * @return the _lztwm
     */
    public double getLztwm()
    {
        return _lztwm;
    }

    public void setLztwm(final double lztwmValue)
    {
        _lztwm = lztwmValue;
    }

    /**
     * @return the _pctim
     */
    public double getPctim()
    {
        return _pctim;
    }

    public void setPctim(final double pctimValue)
    {
        _pctim = pctimValue;
    }

    /**
     * @return the _peAdj
     */
    public double getPeAdj()
    {
        return _peadj;
    }

    public void setPeAdj(final double peAdjValue)
    {
        _peadj = peAdjValue;
    }

    /**
     * @return the _pfree
     */
    public double getPfree()
    {
        return _pfree;
    }

    public void setPfree(final double pfreeValue)
    {
        _pfree = pfreeValue;
    }

    /**
     * @return the _pxAdj
     */
    public double getPxAdj()
    {
        return _pxadj;
    }

    public void setPxAdj(final double pxadjValue)
    {
        _pxadj = pxadjValue;
    }

    /**
     * @return the _rexp
     */
    public double getRexp()
    {
        return _rexp;
    }

    public void setRexp(final double rexpValue)
    {
        _rexp = rexpValue;
    }

    /**
     * @return the _riva
     */
    public double getRiva()
    {
        return _riva;
    }

    public void setRiva(final double rivaValue)
    {
        _riva = rivaValue;
    }

    /**
     * @return the _rserv
     */
    public double getRserv()
    {
        return _rserv;
    }

    public void setRserv(final double rservValue)
    {
        _rserv = rservValue;
    }

    /**
     * @return the _side
     */
    public double getSide()
    {
        return _side;
    }

    public void setSide(final double sideValue)
    {
        _side = sideValue;
    }

    /**
     * @return the _uzfwm
     */
    public double getUzfwm()
    {
        return _uzfwm;
    }

    public void setUzfwm(final double uzfwmValue)
    {
        _uzfwm = uzfwmValue;
    }

    /**
     * @return the _uzk
     */
    public double getUzk()
    {
        return _uzk;
    }

    public void setUzk(final double uzkValue)
    {
        _uzk = uzkValue;
    }

    /**
     * @return the _uztwm
     */
    public double getUztwm()
    {
        return _uztwm;
    }

    public void setUztwm(final double uztwmValue)
    {
        _uztwm = uztwmValue;
    }

    /**
     * @return the _zperc
     */
    public double getZperc()
    {
        return _zperc;
    }

    public void setZperc(final double zpercValue)
    {
        _zperc = zpercValue;
    }

    /**
     * See comments on {@link #useMAPEInput()}.
     */
    public double[] getEtDemand()
    {
        return _ett;
    }

    /**
     * Default is false, which means that MAPE ts is not used and etDemand 12 values represent ET-demand 16th of each
     * month, daily values are computed by linear interpolation; In the condition of true, etDemand 12 values becomes
     * PE-adjustment factor for the input MAPE ts
     */
    public boolean useMAPEInput()
    {
        return _useMAPEInput;
    }

    public void setUseMAPEInput(final boolean useMape)
    {
        _useMAPEInput = useMape;
    }

    /**
     * By default, not using input SASC TS. The presence of the TS in the fews xml is ignored; If true, the presence of
     * the SASC TS in fews xml is required.
     */
    public boolean useSascInputTS()
    {
        return _useSASCInput;
    }

    public void setUseSascInputTS(final boolean useSasc)
    {
        _useSASCInput = useSasc;
    }

    /**
     * By default, not using input SWE TS. The presence of the TS in the fews xml is ignored; If true, the presence of
     * the SWE TS in fews xml is required.
     */
    boolean useWEinputTS()
    {
        return _useWeInputInFRZE;
    }

    /**
     * Gets the data time interval of the output time series FGIX. If specification does not want the output time series
     * and does not specify its data interval, take RAIM TS interval {@link #_raimTsInterval}.
     * 
     * @return the FGIX interval
     */
    int getFgixTsInterval()
    {
        return _fgixInterval;
    }

    /**
     * Gets the data time interval of the five output time series SMZC. If not specified, default value is 24hr.
     * 
     * @return the SMZC TS interval
     */
    int getSmzcTsInterval()
    {
        return _smzcTsInterval;
    }

    /**
     * Gets the data time interval of the six output time series ROCL. If not specified, default value is 24hr.
     * 
     * @return the ROCL TS interval
     */
    int getRunoffComponentTsInterval()
    {
        return _runoffComponentTsInterval;
    }

    public double getCsoil()
    {
        return _csoil;
    }

    public void setCsoil(final double csoilValue)
    {
        _csoil = csoilValue;
    }

    public double getCsnow()
    {
        return _csnow;
    }

    public void setCsnow(final double csnowValue)
    {
        _csnow = csnowValue;
    }

    public double getGch()
    {
        return _gch;
    }

    public void setGch(final double gchValue)
    {
        _gch = gchValue;
    }

    public double getRthaw()
    {
        return _rthaw;
    }

    public void setRthaw(final double rthawValue)
    {
        _rthaw = rthawValue;
    }

    public double getFrtemp()
    {
        return _frtemp;
    }

    public void setFrtemp(final double frtempValue)
    {
        _frtemp = frtempValue;
    }

    public double getSatr()
    {
        return _satr;
    }

    public void setSatr(final double satrValue)
    {
        _satr = satrValue;
    }

    public double getFrexp()
    {
        return _frexp;
    }

    public void setFrexp(final double frexpValue)
    {
        _frexp = frexpValue;
    }

    public boolean useFrozenGroundCalc()
    {
        return _useFrozenGroundCalc;
    }

    public void setUseFrozenGroundCalc(final boolean frozenGroundCalc)
    {
        _useFrozenGroundCalc = frozenGroundCalc;
    }

    void setRaimTsInterval(final int raimTsInterval)
    {
        _raimTsInterval = raimTsInterval;
    }

    @Override
    public String toString()
    {
        final StringBuilder resultStr = new StringBuilder();

        resultStr.append(SacSmaModelConstants.PXADJ_TAG + "=" + _pxadj).append(OHDConstants.NEW_LINE);

        resultStr.append(SacSmaModelConstants.RSERV_TAG + "=" + _rserv).append(OHDConstants.NEW_LINE);

        resultStr.append(SacSmaModelConstants.SIDE_TAG + "=" + _side).append(OHDConstants.NEW_LINE);

        resultStr.append(SacSmaModelConstants.PFREE_TAG + "=" + _pfree).append(OHDConstants.NEW_LINE);

        resultStr.append(SacSmaModelConstants.LZPK_TAG + "=" + _lzpk).append(OHDConstants.NEW_LINE);

        resultStr.append(SacSmaModelConstants.LZSK_TAG + "=" + _lzsk).append(OHDConstants.NEW_LINE);

        resultStr.append(SacSmaModelConstants.EFC_TAG + "=" + _efc).append(OHDConstants.NEW_LINE);

        resultStr.append(SacSmaModelConstants.REXP_TAG + "=" + _rexp).append(OHDConstants.NEW_LINE);

        resultStr.append(SacSmaModelConstants.ZPERC_TAG + "=" + _zperc).append(OHDConstants.NEW_LINE);

        resultStr.append(SacSmaModelConstants.RIVA_TAG + "=" + _riva).append(OHDConstants.NEW_LINE);

        resultStr.append(SacSmaModelConstants.ADIMP_TAG + "=" + _adimp).append(OHDConstants.NEW_LINE);

        resultStr.append(SacSmaModelConstants.PCTIM_TAG + "=" + _pctim).append(OHDConstants.NEW_LINE);

        resultStr.append(SacSmaModelConstants.UZK_TAG + "=" + _uzk).append(OHDConstants.NEW_LINE);

        resultStr.append(SacSmaModelConstants.LZFPM_TAG + "=" + _lzfpm).append(OHDConstants.NEW_LINE);

        resultStr.append(SacSmaModelConstants.LZFSM_TAG + "=" + _lzfsm).append(OHDConstants.NEW_LINE);

        resultStr.append(SacSmaModelConstants.LZTWM_TAG + "=" + _lztwm).append(OHDConstants.NEW_LINE);

        resultStr.append(SacSmaModelConstants.UZFWM_TAG + "=" + _uzfwm).append(OHDConstants.NEW_LINE);

        resultStr.append(SacSmaModelConstants.PEADJ_TAG + "=" + _peadj).append(OHDConstants.NEW_LINE);

        resultStr.append(SacSmaModelConstants.UZTWM_TAG + "=" + _uztwm).append(OHDConstants.NEW_LINE);

        if(_ett != null)
        {
            resultStr.append(SacSmaModelConstants.ET_DEMAND_CURVE_TAG + "=" + Arrays.toString(_ett))
                     .append(OHDConstants.NEW_LINE);
        }

        resultStr.append(SacSmaModelConstants.RUNOFF_COMPOENT_INTERVAL_TAG + "=" + _runoffComponentTsInterval)
                 .append(OHDConstants.NEW_LINE);

        resultStr.append(SacSmaModelConstants.SMZC_INTERVAL_TAG + "=" + _smzcTsInterval).append(OHDConstants.NEW_LINE);

        resultStr.append(SacSmaModelConstants.MAPE_INPUT_TAG + "=" + _useMAPEInput).append(OHDConstants.NEW_LINE);

        resultStr.append(SacSmaModelConstants.SASC_INPUT_TAG + "=" + _useSASCInput).append(OHDConstants.NEW_LINE);

        /** ------------------------------FRZE -------------------------------------- */
        if(_useFrozenGroundCalc)
        {

            resultStr.append(SacSmaModelConstants.CSOIL_TAG + "=" + _csoil).append(OHDConstants.NEW_LINE);
            resultStr.append(SacSmaModelConstants.CSNOW_TAG + "=" + _csnow).append(OHDConstants.NEW_LINE);
            resultStr.append(SacSmaModelConstants.GCH_TAG + "=" + _gch).append(OHDConstants.NEW_LINE);
            resultStr.append(SacSmaModelConstants.RTHAW_TAG + "=" + _rthaw).append(OHDConstants.NEW_LINE);
            resultStr.append(SacSmaModelConstants.FRTEMP_TAG + "=" + _frtemp).append(OHDConstants.NEW_LINE);
            resultStr.append(SacSmaModelConstants.SATR_TAG + "=" + _satr).append(OHDConstants.NEW_LINE);
            resultStr.append(SacSmaModelConstants.FREXP_TAG + "=" + _frexp).append(OHDConstants.NEW_LINE);

            resultStr.append(SacSmaModelConstants.WE_INPUT_TAG + "=" + _useWeInputInFRZE).append(OHDConstants.NEW_LINE);
            resultStr.append(SacSmaModelConstants.FGIX_INTERVAL_TAG + "=" + _fgixInterval)
                     .append(OHDConstants.NEW_LINE);
        }

        return resultStr.toString();
    }
}
