package ohd.hseb.ohdmodels.sacsma;

import ohd.hseb.util.fews.ohdmodels.ModelParameters;
import ohd.hseb.util.fews.ohdmodels.ModelState;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.Parameters;
import ohd.hseb.util.fews.State;

/**
 * This class holds all the SacSma Model state variables. It is a subclass of the abstract class ModelState and
 * interface IModelState. The state variable values are loaded from a text file with properties format through
 * {@link #loadState(String, Logger)}. The values are are initially loaded into the Map<String, Object> _statesMap in
 * ModelState, then the values are extracted to the corresponding instance variables.
 * 
 * @author CHPS
 */
final public class SacSmaModelState extends ModelState
{
    private double _uztwc;
    private double _uzfwc;
    private double _lztwc;
    private double _lzfsc;
    private double _lzfpc;
    private double _adimc;
    /*
     * If not known, use UZTWC + LZTWC. But actually, ADIMC is always punched out into the opt file. So all statesI.txt
     * generated by SacSmaOptFileConverter.java always have ADIMC.
     */

    private double _fgix; //always mandatory now, regardless of using FRZE or not

    /**
     * Default constructor. All the values in _statesMap are set by {@link State#loadState(String, Logger)}.
     */
    public SacSmaModelState()
    {
        super();
    }

    /**
     * Extract values from super._statesMap into instance variables. The values in super._statessMap are set by parsing
     * the initial state file in OHDFewsAdapter.java.
     */
    @Override
    public void extractValuesFromMap() throws Exception
    {
        _uztwc = getDoubleDataState(OHDConstants.SAC_STATE_UZTWC);

        _uzfwc = getDoubleDataState(OHDConstants.SAC_STATE_UZFWC);

        _lztwc = getDoubleDataState(OHDConstants.SAC_STATE_LZTWC);

        _lzfsc = getDoubleDataState(OHDConstants.SAC_STATE_LZFSC);

        _lzfpc = getDoubleDataState(OHDConstants.SAC_STATE_LZFPC);

        _adimc = getDoubleDataState(OHDConstants.SAC_STATE_ADIMC);

        _fgix = getDoubleDataState(OHDConstants.SAC_STATE_FGIX);

    }

    /**
     * Validate all required states are present. Confirm input unit is METRIC. Modify states as in fckco1.f
     */
    public void validateState(final Parameters params) throws Exception
    {
        super.confirmUnitMetric();

        final SacSmaModelParameters sacModelParams = (SacSmaModelParameters)params;

        /** -------------modify states as in fckco1.f ------------------ */
        if(getUztwc() > sacModelParams.getUztwm())
        {
            _logger.log(Logger.WARNING,
                        "For the initial state, the value for Upper Zone Tension Water Contents was changed from "
                            + getUztwc() + " to " + sacModelParams.getUztwm());
            setUztwc(sacModelParams.getUztwm());
        }

        if(getUzfwc() > sacModelParams.getUzfwm())
        {
            _logger.log(Logger.WARNING,
                        "For the initial state, the value for Upper Zone Free Water Contents was changed from "
                            + getUzfwc() + " to " + sacModelParams.getUzfwm());
            setUzfwc(sacModelParams.getUzfwm());
        }

        if(getLztwc() > sacModelParams.getLztwm())
        {
            _logger.log(Logger.WARNING,
                        "For the initial state, the value for Lower Zone Tension Water Contents was changed from "
                            + getLztwc() + " to " + sacModelParams.getLztwm());
            setLztwc(sacModelParams.getLztwm());
        }

        if(getLzfsc() > sacModelParams.getLzfsm())
        {
            _logger.log(Logger.WARNING,
                        "For the initial state, the value for Lower Zone Free Secondary Contents was changed from "
                            + getLzfsc() + " to " + sacModelParams.getLzfsm());
            setLzfsc(sacModelParams.getLzfsm());
        }

        if(getLzfpc() > sacModelParams.getLzfpm())
        {
            _logger.log(Logger.WARNING,
                        "For the initial state, the value for Lower Zone Free Primary Contents was changed from "
                            + getLzfpc() + " to " + sacModelParams.getLzfpm());
            setLzfpc(sacModelParams.getLzfpm());
        }

        if(getAdimc() > (sacModelParams.getUztwm() + sacModelParams.getLztwm()))
        {
            _logger.log(Logger.WARNING,
                        "For the initial state, the value for additional impervious contents was changed from "
                            + getAdimc() + " to " + (sacModelParams.getUztwm() + sacModelParams.getLztwm()));
            setAdimc(sacModelParams.getUztwm() + sacModelParams.getLztwm());
        }

        if(getAdimc() < getUztwc())
        {
            _logger.log(Logger.WARNING,
                        "For the initial state, the value for additional impervious contents was changed from "
                            + getAdimc() + " to " + getUztwc());
            setAdimc(getUztwc());
        }

        _logger.log(Logger.DEBUG, "SacSmaModelState validation has passed");

    } //close validateState()

    /**
     * Over-written the super class method. This method is used to output states into the generated state txt file. The
     * units are always in METRIC.
     */
    @Override
    protected String getStringsFromState()
    {
        final StringBuilder resultStr = new StringBuilder();

        resultStr.append(OHDConstants.SAC_STATE_ADIMC + "=" + _adimc + OHDConstants.NEW_LINE);
        resultStr.append(OHDConstants.SAC_STATE_FGIX + "=" + _fgix + OHDConstants.NEW_LINE);
        resultStr.append(OHDConstants.SAC_STATE_LZFPC + "=" + _lzfpc + OHDConstants.NEW_LINE);
        resultStr.append(OHDConstants.SAC_STATE_LZFSC + "=" + _lzfsc + OHDConstants.NEW_LINE);
        resultStr.append(OHDConstants.SAC_STATE_LZTWC + "=" + _lztwc + OHDConstants.NEW_LINE);
        resultStr.append(OHDConstants.SAC_STATE_UZFWC + "=" + _uzfwc + OHDConstants.NEW_LINE);
        resultStr.append(OHDConstants.SAC_STATE_UZTWC + "=" + _uztwc + OHDConstants.NEW_LINE);

        resultStr.append(OHDConstants.UNIT_TAG + "=" + OHDConstants.UNIT_METRIC + OHDConstants.NEW_LINE);
        //units are in METRIC always

        return resultStr.toString();
    }

    /**
     * Add carryover transfer logic to sacsma model. Note: when the two parameters have the same value, nothing will be
     * changed.
     */
    @Override
    public void doCarryOverTransfer(final ModelParameters savedParams, final ModelParameters params)
    {
        final SacSmaModelParameters savedParameters = (SacSmaModelParameters)savedParams;
        final SacSmaModelParameters parameters = (SacSmaModelParameters)params;
        if(_logger.getPrintDebugInfo() > 0)
        {
            _logger.log(Logger.DEBUG, "SacSmaModelState before carry over transfer " + this.getStringsFromState());
        }

        if(this.getUzfwc() > parameters.getUzfwm())
        {
            this.setUzfwc(parameters.getUzfwm());
        }

        this.setUztwc(parameters.getUztwm() - savedParameters.getUztwm() + this.getUztwc());
        if(this.getUztwc() < 0.0)
        {
            this.setUztwc(0.0);
        }

        this.setLztwc(parameters.getLztwm() - savedParameters.getLztwm() + this.getLztwc());
        if(this.getLztwc() < 0.0)
        {
            this.setLztwc(0.0);
        }

        if(parameters.getLzsk() != 0.0)
        {
            this.setLzfsc(this.getLzfsc() * (savedParameters.getLzsk() / parameters.getLzsk())
                * ((1.0 + parameters.getSide()) / (1.0 + savedParameters.getSide())));

            if(this.getLzfsc() > parameters.getLzfsm())
            {
                this.setLzfsc(parameters.getLzfsm());
            }
        }

        if(parameters.getLzpk() != 0.0)
        {
            this.setLzfpc(this.getLzfpc() * (savedParameters.getLzpk() / parameters.getLzpk())
                * ((1.0 + parameters.getSide()) / (1.0 + savedParameters.getSide())));

            if(this.getLzfpc() > parameters.getLzfpm())
            {
                this.setLzfpc(parameters.getLzfpm());
            }
        }

        this.setAdimc(parameters.getUztwm() + parameters.getLztwm() - savedParameters.getUztwm()
            - savedParameters.getLztwm() + this.getAdimc());

        if(this.getAdimc() < this.getUztwc())
        {
            this.setAdimc(this.getUztwc());
        }

        if(_logger.getPrintDebugInfo() > 0)
        {
            _logger.log(Logger.DEBUG, "SacSmaModelState after carry over transfer " + this.getStringsFromState());
        }

    }

    public void setUztwc(final double uztwc)
    {
        _uztwc = uztwc;
    }

    public double getUztwc()
    {
        return _uztwc;
    }

    public void setUzfwc(final double uzfwc)
    {
        _uzfwc = uzfwc;
    }

    public double getUzfwc()
    {
        return _uzfwc;
    }

    public void setLztwc(final double lztwc)
    {
        _lztwc = lztwc;
    }

    public double getLztwc()
    {
        return _lztwc;
    }

    public void setLzfsc(final double lzfsc)
    {
        _lzfsc = lzfsc;
    }

    public double getLzfsc()
    {
        return _lzfsc;
    }

    public void setLzfpc(final double lzfpc)
    {
        _lzfpc = lzfpc;
    }

    public double getLzfpc()
    {
        return _lzfpc;
    }

    public void setAdimc(final double adimc)
    {
        _adimc = adimc;
    }

    public double getAdimc()
    {
        return _adimc;
    }

    public void setFgix(final double fgix)
    {
        _fgix = fgix;
    }

    public double getFgix()
    {
        return _fgix;
    }

}
