package ohd.hseb.ohdmodels.sacsma;

import java.util.Map;

import javax.management.timer.Timer;

import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.model.MonthlyValues;
import ohd.hseb.model.sacsma.SacSmaModsTSHolder;
import ohd.hseb.model.sacsma.SacSmaParameters;
import ohd.hseb.model.sacsma.SacSmaRainfallRunoffModel;
import ohd.hseb.model.sacsma.SacSmaState;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.ModelException;
import ohd.hseb.util.fews.OHDConstants;

/**
 * @author FewsPilot Team
 */
public class SacSmaRainfallRunoffModelAdapter
{

    private final SacSmaRainfallRunoffModel _toModel;
    private final SacSmaModelDriver _fromModel;

    private boolean _useStaticMape; //if true: don't use MAPE TS; if false: use MAPE TS

    private String _ffhHeaderLine;

    private final boolean _printSmaTechnique;

    public SacSmaRainfallRunoffModelAdapter(final SacSmaModelDriver fromModel) throws Exception
    {

        _fromModel = fromModel; //alias, both are the same thing

        //decision for using MAPE input TS or not
        if(_fromModel.getParameters().useMAPEInput() == false) //Do not use MAPE input TS, no matter it is present or not in input fews.xml
        {
            _useStaticMape = true;
            if(_fromModel.getLogger() != null && _fromModel.getLogger().getPrintDebugInfo() > 0)
            {
                _fromModel.getLogger().log(Logger.DEBUG,
                                           "Not using MAPE input time series to obtain evaporation amount.");
            }
        }
        else
        // sacModel.getParameters().hasMAPEInput() == true. Use MAPE input TS.
        {
            //first, check to see if MAPE input TS is present in the input fews.xml
            if(_fromModel.getMapeTs() == null)
            {
                throw new ModelException("User specified to use MAPE input time series, but it is not available in the input xml file.");
            }
            if(_fromModel.getLogger() != null && _fromModel.getLogger().getPrintDebugInfo() > 0)
            {
                _fromModel.getLogger().log(Logger.DEBUG, "Using MAPE input time series to obtain evaporation amount.");
            }

            _useStaticMape = false;

        } //close big else block: parameter set to use MAPE input TS

        /** --------construct SacSmaRainfallRunoffModel obj by using FEWS Constructor----------------- */
        _toModel = new SacSmaRainfallRunoffModel(_adaptModelStateToSacSmaState(_fromModel.getState()),
                                                 _adaptModelParametersToSacSmaParameters(_fromModel.getParameters()),
                                                 _useStaticMape,
                                                 new MonthlyValues(_fromModel.getParameters().getEtDemand()),
                                                 _fromModel.getMapeTs(),
                                                 _fromModel.getStartHourOfDay(),
                                                 _fromModel.getSwitchFromObservedToForecastTime(),
                                                 _fromModel.getLogger());

        //pass optional input SASC TS to SacSmaRainfallRunoffModel if using it
        if(_fromModel.getParameters().useSascInputTS() == true)
        {
            _toModel.setSascTs(_fromModel.getSascTs());
        }

        //if using FRZE, pass MAT TS(required) and WE TS(optional)
        if(_fromModel.getParameters().useFrozenGroundCalc() == true)
        {
            _toModel.setMatTS(_fromModel.getMatTs()); //required TS

            //WE TS is optional input when using Frozen Ground 
            if(_fromModel.getParameters().useWEinputTS())
            {
                _toModel.setWeTS(_fromModel.getWeTs());
            }

        }

        final SacSmaTechnique sacSmaTechnique = new SacSmaTechnique(_fromModel.getDriverProperties(),
                                                                    _fromModel.getLogger());

        _printSmaTechnique = sacSmaTechnique.getBoolValue(SacSmaModelConstants.PRINTSMA);

        if(_printSmaTechnique)
        {

            String headerLine;

            if(_fromModel.getDrivingRTS() != null)
            {//running SAC-SMA alone
                headerLine = "'" + _fromModel.getDrivingRTS().getLocationId() + "    TIME ZONE=Z"; //table hour info is shown in Z, so hard-coded TZ=Z
            }
            else
            {//running for FFH
//                headerLine = "DETAILED SOIL-MOISTURE ACCOUNTING OUTPUT FOR TECUMSEH (DCP)           11/2010   TIME ZONE=CDT           UNITS ARE MM.";

                final DateTime dateTime = new DateTime(fromModel.getSwitchFromObservedToForecastTime());
                final String monYearStr = String.valueOf(dateTime.month()) + "/" + String.valueOf(dateTime.year());
                final String tzStr = "TIME ZONE=" + fromModel.getRunInfo().getTimeZone().getID();

                _ffhHeaderLine = "DETAILED SOIL-MOISTURE ACCOUNTING OUTPUT FOR " + _fromModel.getState().getId()
                    + "     " + monYearStr + "   " + tzStr + "    UNITS ARE MM.";

                headerLine = _ffhHeaderLine;
            }

            _toModel.setPrintTableLogHeader(headerLine);
        }

    } //close constructor

    /**
     * Use dhm's SacSmaRainfallRunoffModel to calculate the runoff time series, based on precip TS. All the returned
     * time series are inside resultMap, except the 6 SacSmaState time series.
     */
    void runSacSmaRainfallRunoffModel(final Map<String, RegularTimeSeries> resultMap,
                                      final SacSmaModsTSHolder modsTsHolder)
    {

        //calculating the run off 
        _toModel.runSacSmaModel(_fromModel.getDrivingRTS(), //precip TS(RAIN + MELT)
                                _fromModel.getInitialStateTime(),
                                resultMap,
                                modsTsHolder);

        //1)re-assigned outputModsMap; 2)update _state in _fromModel(SacSmaModelDriver) state values and time from _state(SacSmaState) in _toModel
        updateModelDriverState();

    }

    /**
     * convert FEWS ModelState object to Chip's SacSmaState. Note: not using SacSmaModelState super._statesMap because
     * the values inside are not updated by COX.
     */
    private SacSmaState _adaptModelStateToSacSmaState(final SacSmaModelState sacSmaModelState)
    {
        //all states have been initialized to 0.0, copy state time(initial state time)
        final SacSmaState newSacSmaState = new SacSmaState(sacSmaModelState.getDateTime());

        /** -----------------------copy 6 states ----------------------------- */
        newSacSmaState.setUztwc(sacSmaModelState.getUztwc());
        newSacSmaState.setUzfwc(sacSmaModelState.getUzfwc());
        newSacSmaState.setLztwc(sacSmaModelState.getLztwc());
        newSacSmaState.setLzfsc(sacSmaModelState.getLzfsc());
        newSacSmaState.setLzfpc(sacSmaModelState.getLzfpc());
        newSacSmaState.setAdimc(sacSmaModelState.getAdimc());

        /** -----------------------FRZE part ----------------------------- */
        newSacSmaState.setFgix(sacSmaModelState.getFgix()); //always mandatory now, regardless of using FRZE or not

        return newSacSmaState;

    }

    private SacSmaParameters _adaptModelParametersToSacSmaParameters(final SacSmaModelParameters modelParams)
    {

        final SacSmaParameters newSacSmaParams = new SacSmaParameters();
        newSacSmaParams.setUztwm(modelParams.getUztwm());
        newSacSmaParams.setUzfwm(modelParams.getUzfwm());
        newSacSmaParams.setUzk(modelParams.getUzk());
        newSacSmaParams.setPctim(modelParams.getPctim());
        newSacSmaParams.setAdimp(modelParams.getAdimp());
        newSacSmaParams.setRiva(modelParams.getRiva());
        newSacSmaParams.setZperc(modelParams.getZperc());
        newSacSmaParams.setRexp(modelParams.getRexp());
        newSacSmaParams.setLztwm(modelParams.getLztwm());
        newSacSmaParams.setLzfsm(modelParams.getLzfsm());
        newSacSmaParams.setLzfpm(modelParams.getLzfpm());
        newSacSmaParams.setLzsk(modelParams.getLzsk());
        newSacSmaParams.setLzpk(modelParams.getLzpk());
        newSacSmaParams.setPfree(modelParams.getPfree());
        newSacSmaParams.setRserv(modelParams.getRserv());
        newSacSmaParams.setSide(modelParams.getSide());
        newSacSmaParams.setEfc(modelParams.getEfc());
        newSacSmaParams.setPeadj(modelParams.getPeAdj());
        newSacSmaParams.setPxadj(modelParams.getPxAdj());

        //frze feature
        if(modelParams.useFrozenGroundCalc() == true)
        {
            newSacSmaParams.setUseFronzeGroundCalc(true);

            newSacSmaParams.setCsoil(modelParams.getCsoil());
            newSacSmaParams.setCsnow(modelParams.getCsnow());
            newSacSmaParams.setGhc(modelParams.getGch());
            newSacSmaParams.setRthaw(modelParams.getRthaw());
            newSacSmaParams.setFrtemp(modelParams.getFrtemp());
            newSacSmaParams.setSatr(modelParams.getSatr());
            newSacSmaParams.setFrexp(modelParams.getFrexp());
        }

        return newSacSmaParams;
    }

    /**
     * Do two things:<br>
     * 1)Set the 6 SacSma state RTS map to the ModelDriver outputModsMap;<br>
     * 2)Update 7 state variables inside _fromModel with values and time in _toModel SacSmaState obj. Other instance
     * variables are un-changed.
     */
    private void updateModelDriverState()
    {
        //update 7 state variables
        _fromModel.getState().setUztwc(_toModel.getState().getUztwc());
        _fromModel.getState().setUzfwc(_toModel.getState().getUzfwc());
        _fromModel.getState().setLztwc(_toModel.getState().getLztwc());
        _fromModel.getState().setLzfpc(_toModel.getState().getLzfpc());
        _fromModel.getState().setLzfsc(_toModel.getState().getLzfsc());
        _fromModel.getState().setAdimc(_toModel.getState().getAdimc());

        //for FRZE
        if(_fromModel.getParameters().useFrozenGroundCalc())
        {
            _fromModel.getState().setFgix(_toModel.getState().getFgix());
        }

        _fromModel.getState().setDateTime(_toModel.getState().getValidTime());
    }

    /**
     * Get the runoff (mm) based on the amount of precipitation using SAC-SMA model. This method is for FFG(FFH and
     * Gridded FFG). evapotranspiration is hard-coded as 0.0, so the parameter "ET_DEMAND_CURVE" or input MAPE ts for
     * SACSMA alone is not needed.
     * 
     * @param precipAmount - could be rain only or if using snow model, rain + melt. in MM unit.
     * @return - the runoff(TCI) in unit of MM.
     */
    public double getRunoffBySacSma(final double precipAmount, final int modelTimeInterval)
    {

        //always use the state at the initial state time
        _toModel.setSacSmaState(_adaptModelStateToSacSmaState(_fromModel.getState()));
        _toModel.setSacSmaParams(_adaptModelParametersToSacSmaParameters(_fromModel.getParameters())); //maybe not needed

//        System.out.println("Similar at top FLAND1. states:\n" + _toModel.getSacSmaState().toString());
        _fromModel.getLogger().log(Logger.DEBUG,
                                   "SAC-SMA Model states: UZTWC=" + _toModel.getSacSmaState().getUztwc() + " UZFWC="
                                       + _toModel.getSacSmaState().getUzfwc() + " LZTWC="
                                       + _toModel.getSacSmaState().getLztwc() + " LZFSC="
                                       + _toModel.getSacSmaState().getLzfsc() + " LZFPC="
                                       + _toModel.getSacSmaState().getLzfpc() + " ADIMC="
                                       + _toModel.getSacSmaState().getAdimc() + " FGIX="
                                       + _toModel.getSacSmaState().getFgix());

        final long timeSinceT0InLong = _fromModel.getRunInfo().getRunLastObservationTimeLong() + modelTimeInterval
            * Timer.ONE_HOUR;

        final double intervalPortionOfDay = (modelTimeInterval * 1.0) / OHDConstants.HOURS_PER_DAY;

        final double evaporationAmount = 0.0; //MM. zero-out for FFH, ignoring MAPE TS or SASC TS

        /** ---------------------calculate runoff in the time step---------------------- */
        final double tci = _toModel.calculate(intervalPortionOfDay, precipAmount, evaporationAmount, timeSinceT0InLong);

        if(_printSmaTechnique)
        {
            _toModel.printOutTableLog();

            _toModel.setPrintTableLogHeader(_ffhHeaderLine); //re-set to get rid of last run's message
        }

        return tci; //MM
    }

} //close class
