package ohd.hseb.ohdmodels.sacsmaHT;

import ohd.hseb.util.DoubleHolder;
import ohd.hseb.util.Logger;
import ohd.hseb.util.MathHelper;
import ohd.hseb.util.fews.ModelException;
import ohd.hseb.util.fews.OHDConstants;

/**
 * @author FewsPilot Team
 */
final class SacSmaHTModel
{

    SacSmaHTData _sacHTData;
    HeatTransferSimulation _heatTransferSim;

    private double _uztwc0 = 0;
    private double _uzfwc0 = 0;
    private double _lztwc0 = 0;
    private double _lzfsc0 = 0;
    private double _lzfpc0 = 0;
    private int _frozenGroundSimulationTimeStepAsInt = 0;
    private double _modelTimeStepInSeconds = 0.0;
    private final SacSmaHTResults _currentResult = new SacSmaHTResults();
    private double _changeInPercolationRate;
    private double _changeInInterflowWithdrawlRate;
    private double _frostDepth = OHDConstants.MISSING_DATA;

    public SacSmaHTModel(final SacSmaHTData sacHTData)
    {
        _sacHTData = sacHTData;
        _heatTransferSim = new HeatTransferSimulation(_sacHTData);
    }

    /*
     * fLand1 executes the 'sac-sma' operation for one time period frozen ground changes uztwc,uzfwc,lztwc,lzfsc,lzfpc
     * are total water storages uztwh,uzfwh,lztwh,lzfsh,lzfph are unfrozen water storages tci (runoff) is output dt is
     * in days
     */
    public SacSmaHTResults fLand1(final double inputTimeStep,
                                  final double inputPrecipitation,
                                  final double airTemperature,
                                  final double snowWaterEquivalent,
                                  final double snowCoverAsPercent,
                                  final double snowDepth,
                                  double etDemand,
                                  final int icall) throws Exception
    {
        double eused, dlzp, lowerZonePrimaryRatio, lowerZoneSecondaryRatio, bfs, tet;
        double lztwh, lzfsh, lzfph, addro, percm, upperZoneDeficitRatio, percolation, lowerZoneDeficitRatio, check, hpl, fracp, bfcc;
        double percf = 0.0;

        double saved, parea, adsur, ratio, xx1, perct, percp, percs;
        double excess, uztwh, uzfwh, ruzice, rlzice, ruzperc, twx, roimp, ninc, dinc, pinc, duz, dlzs;
        double bf = 0.;
        double duztwc, e2, e3, e4, e5, lowerZoneTensionWaterRatio, lowerZoneRatio, sbf, sur, ssur, sif, sperc, sdro, spbf, tbf, bfp;
        double del = 0.;
        double sfh = 0.;
        double runoff = 0.;
        double upperZoneRatio;

        _uztwc0 = _sacHTData.getSacSmaHtState().getUztwc();
        _uzfwc0 = _sacHTData.getSacSmaHtState().getUzfwc();
        _lztwc0 = _sacHTData.getSacSmaHtState().getLztwc();
        _lzfsc0 = _sacHTData.getSacSmaHtState().getLzfsc();
        _lzfpc0 = _sacHTData.getSacSmaHtState().getLzfpc();

        saved = _sacHTData.getSacSmaHtParameters().getRserv()
            * (_sacHTData.getSacSmaHtParameters().getLzfpm() + _sacHTData.getSacSmaHtParameters().getLzfsm());
        parea = 1.0 - _sacHTData.getSacSmaHtParameters().getPctim() - _sacHTData.getSacSmaHtParameters().getAdimp();

        /*
         * CVK New frozen ground version: use estimated unfrozen water CVK Redefine unfrozen water variables if they are
         * negative Old frozen ground not involved here
         */
        uztwh = (_sacHTData.getSacSmaHtState().getUztwh() < 0. ? 0.0 : _sacHTData.getSacSmaHtState().getUztwh());
        uzfwh = (_sacHTData.getSacSmaHtState().getUzfwh() < 0. ? 0. : _sacHTData.getSacSmaHtState().getUzfwh());
        lztwh = (_sacHTData.getSacSmaHtState().getLztwh() < 0. ? 0. : _sacHTData.getSacSmaHtState().getLztwh());
        lzfsh = (_sacHTData.getSacSmaHtState().getLzfsh() < 0. ? 0. : _sacHTData.getSacSmaHtState().getLzfsh());
        lzfph = (_sacHTData.getSacSmaHtState().getLzfph() < 0. ? 0. : _sacHTData.getSacSmaHtState().getLzfph());

        /*
         * ruzice & rlzice are reduction of free water movement based on kulik's theory: kfrz =
         * kunfrz/(1+fgpm(4)*ice)**2
         */
        ruzice = 0.001
            * _sacHTData.getFrdStfg().rTuz
            * (_sacHTData.getSacSmaHtState().getUztwc() - _sacHTData.getSacSmaHtState().getUztwh()
                + _sacHTData.getSacSmaHtState().getUzfwc() - _sacHTData.getSacSmaHtState().getUzfwh())
            / _sacHTData.getFrdStfg().dzUp;
        rlzice = 0.001
            * _sacHTData.getFrdStfg().rTlz
            * (_sacHTData.getSacSmaHtState().getLztwc() - _sacHTData.getSacSmaHtState().getLztwh()
                + _sacHTData.getSacSmaHtState().getLzfsc() - _sacHTData.getSacSmaHtState().getLzfsh())
            / _sacHTData.getFrdStfg().dzLow;
        ruzperc = 1.0;

        if(ruzice == 0)
        {
            ruzice = _sacHTData.getSacSmaHtParameters().getUzk();
        }
        else
        {
            ruzperc = 1.0 / (Math.pow((1. + _sacHTData.getSacSmaHtParameters().getRunoffEffect() * ruzice), 2));
            ruzice = 1. - Math.exp(Math.log(1. - _sacHTData.getSacSmaHtParameters().getUzk()) * ruzperc);
        }

        if(rlzice == 0.)
        {
            rlzice = _sacHTData.getSacSmaHtParameters().getLzsk();
        }
        else
        {
            rlzice = 1.0 / (Math.pow((1. + _sacHTData.getSacSmaHtParameters().getRunoffEffect() * rlzice), 2));
            rlzice = 1. - Math.exp((Math.log(1. - _sacHTData.getSacSmaHtParameters().getLzsk()) * rlzice));
        }

        /*
         * compute evapotranspiration loss for the time interval. edmnd is the et-demand for the time interval
         * edmnd=ep*epdist(kint) adjust edmnd for effect of snow & forest cover. from ex1 ofs subroutine
         * edmnd=efc*edmnd+(1.0-efc)*(1.0-aesc)*edmnd
         */

        etDemand = _sacHTData.getSacSmaHtParameters().getEfc() * etDemand
            + (1.0f - _sacHTData.getSacSmaHtParameters().getEfc()) * (1.0f - snowCoverAsPercent) * etDemand;

        /*
         * C compute et from upper zone. CVK e1=edmnd*(uztwc/uztwm) CVK only unfrozen water can be evaporated
         */

        double e1 = etDemand * (uztwh / _sacHTData.getSacSmaHtParameters().getUztwm());
        double red = etDemand - e1;

        /*
         * C red is residual evap demand CVK uztwc=uztwc-e1
         */

        uztwh -= e1;
        e2 = 0.0;

        if(uztwh >= 0.)
        {
            //CV.K    Subtract et from total water storage
            _sacHTData.getSacSmaHtState().setUztwc(_sacHTData.getSacSmaHtState().getUztwc() - e1);
            // go to 220
            if((_sacHTData.getSacSmaHtState().getUztwc() / _sacHTData.getSacSmaHtParameters().getUztwm()) >= (_sacHTData.getSacSmaHtState()
                                                                                                                        .getUzfwc() / _sacHTData.getSacSmaHtParameters()
                                                                                                                                                .getUzfwm()))
            {
                ;
            }
            else
            {
                //Upper zone free water ratio exceeds upper zone
                //Tension water ratio, thus transfer free water to tension
                upperZoneRatio = (_sacHTData.getSacSmaHtState().getUztwc() + _sacHTData.getSacSmaHtState().getUzfwc())
                    / (_sacHTData.getSacSmaHtParameters().getUztwm() + _sacHTData.getSacSmaHtParameters().getUzfwm());

                //CV.K  account for ratio of unfrozen water only
                //CV.K  and adjust four soil states
                //CV.K      uztwc=uztwm*uzrat
                //CV.K      uzfwc=uzfwm*uzrat
                duztwc = _sacHTData.getSacSmaHtParameters().getUztwm() * upperZoneRatio
                    - _sacHTData.getSacSmaHtState().getUztwc();
                if(duztwc > uzfwh)
                {
                    duztwc = uzfwh;
                }
                //CV.K  transfered water can not exceed unfrozen free water
                _sacHTData.getSacSmaHtState().setUztwc(_sacHTData.getSacSmaHtState().getUztwc() + duztwc);
                uztwh += duztwc;
                _sacHTData.getSacSmaHtState().setUzfwc(_sacHTData.getSacSmaHtState().getUzfwc() - duztwc);
                uzfwh -= duztwc;
                //CV.K  check unfrozen water storages too
            }
        }
        else
        {
            //e1 can not exceed uztwc
            e1 += uztwh;
            uztwh = 0.0;
            //Reduce total tension water by actual e1
            _sacHTData.getSacSmaHtState().setUztwc(_sacHTData.getSacSmaHtState().getUztwc() - e1);
            red = etDemand - e1;

            if(uzfwh >= red)
            {
                //221
                e2 = red;
                // CVK Subtract E2 from TOTAL & unffrozen water storages
                _sacHTData.getSacSmaHtState().setUzfwc(_sacHTData.getSacSmaHtState().getUzfwc() - e2);
                uzfwh -= e2;
                red = 0.0;
                //fall through to 220
                if((_sacHTData.getSacSmaHtState().getUztwc() / _sacHTData.getSacSmaHtParameters().getUztwm()) >= (_sacHTData.getSacSmaHtState()
                                                                                                                            .getUzfwc() / _sacHTData.getSacSmaHtParameters()
                                                                                                                                                    .getUzfwm()))
                {
                    ;
                }
                else
                {
                    //Upper zone free water ratio exceeds upper zone
                    //Tension water ratio, thus transfer free water to tension
                    upperZoneRatio = (_sacHTData.getSacSmaHtState().getUztwc() + _sacHTData.getSacSmaHtState()
                                                                                           .getUzfwc())
                        / (_sacHTData.getSacSmaHtParameters().getUztwm() + _sacHTData.getSacSmaHtParameters()
                                                                                     .getUzfwm());

                    //CV.K  account for ratio of unfrozen water only
                    //CV.K  and adjust four soil states
                    //CV.K  uztwc=uztwm*uzrat
                    //CV.K  uzfwc=uzfwm*uzrat
                    duztwc = _sacHTData.getSacSmaHtParameters().getUztwm() * upperZoneRatio
                        - _sacHTData.getSacSmaHtState().getUztwc();
                    if(duztwc >= uzfwh)
                    {
                        duztwc = uzfwh;
                    }
                    //CV.K  transfered water can not exceed unfrozen free water
                    _sacHTData.getSacSmaHtState().setUztwc(_sacHTData.getSacSmaHtState().getUztwc() + duztwc);
                    uztwh += duztwc;
                    _sacHTData.getSacSmaHtState().setUzfwc(_sacHTData.getSacSmaHtState().getUzfwc() - duztwc);
                    uzfwh -= duztwc;
                }
            }
            else if(uzfwh < red)
            {
                //e2 is evap from uzfwc.
                e2 = uzfwh;
                uzfwh = 0.0;
                //reduce total free water by actual e2
                _sacHTData.getSacSmaHtState().setUzfwc(_sacHTData.getSacSmaHtState().getUzfwc() - e2);
                red -= e2;
                //go to 225
            }
        }

        //225 
        if(_sacHTData.getSacSmaHtState().getUztwc() < 0.00001)
        {
            _sacHTData.getSacSmaHtState().setUztwc(0.0);
            uztwh = 0.0;
        }
        if(_sacHTData.getSacSmaHtState().getUzfwc() < 0.00001)
        {
            _sacHTData.getSacSmaHtState().setUzfwc(0.0);
            uzfwh = 0.0;
        }

        //Compute ET from the lower zone
        // Compute et from lztwc (E3)	
        //CV.K Only Unfrozen water can be evaporated

        e3 = red
            * (lztwh / (_sacHTData.getSacSmaHtParameters().getUztwm() + _sacHTData.getSacSmaHtParameters().getLztwm()));
        lztwh -= e3;

        if(lztwh >= 0.0)
        {
            _sacHTData.getSacSmaHtState().setLztwc(_sacHTData.getSacSmaHtState().getLztwc() - e3);
            //go to 226
        }
        else
        {
            //e3 can not exceed lztwc

            e3 += lztwh;
            lztwh = 0.0;
            //CV.K   reduce total tension water by e3
            _sacHTData.getSacSmaHtState().setLztwc(_sacHTData.getSacSmaHtState().getLztwc() - e3);
        }
        //226

        lowerZoneTensionWaterRatio = _sacHTData.getSacSmaHtState().getLztwc()
            / _sacHTData.getSacSmaHtParameters().getLztwm();
        lowerZoneRatio = (_sacHTData.getSacSmaHtState().getLztwc() + _sacHTData.getSacSmaHtState().getLzfpc()
            + _sacHTData.getSacSmaHtState().getLzfsc() - saved)
            / (_sacHTData.getSacSmaHtParameters().getLztwm() + _sacHTData.getSacSmaHtParameters().getLzfpm()
                + _sacHTData.getSacSmaHtParameters().getLzfsm() - saved);
        if(lowerZoneTensionWaterRatio >= lowerZoneRatio)
        {
            ;//go to 230
        }
        else
        {
            //Resupply lower zone tension water from lower
            //Zone free water if more water available there.
            del = (lowerZoneRatio - lowerZoneTensionWaterRatio) * _sacHTData.getSacSmaHtParameters().getLztwm();
            //CV.K  Only unfrozen water can be transfered
            sfh = lzfsh + lzfph;
            if(del > sfh)
            {
                del = sfh;
            }
            lzfsh -= del;
            if(lzfsh >= 0.0)
            {
                //Transfer from lzfsc to lztwc.
                _sacHTData.getSacSmaHtState().setLzfsc(_sacHTData.getSacSmaHtState().getLzfsc() - del);
            }
            else
            {
                //if transfer exceeds lzfsc then remainder comes from lzfpc
                _sacHTData.getSacSmaHtState().setLzfpc(_sacHTData.getSacSmaHtState().getLzfpc() + lzfsh);
                lzfph += lzfsh;
                _sacHTData.getSacSmaHtState().setLzfsc(_sacHTData.getSacSmaHtState().getLzfsc() - (lzfsh + del));
                lzfsh = 0.0;
            }
            _sacHTData.getSacSmaHtState().setLztwc(_sacHTData.getSacSmaHtState().getLztwc() + del);
            lztwh += del;
        }

        //CV.K  Check unfrozen water storage
        //230 
        if(_sacHTData.getSacSmaHtState().getLztwc() < 0.00001)
        {
            _sacHTData.getSacSmaHtState().setLztwc(0.0);
            lztwh = 0.0;
        }

        //Compute et from adimp area.-e5
        e5 = e1
            + (red + e2)
            * ((_sacHTData.getSacSmaHtState().getAdimc() - e1 - _sacHTData.getSacSmaHtState().getUztwc()) / (_sacHTData.getSacSmaHtParameters()
                                                                                                                       .getUztwm() + _sacHTData.getSacSmaHtParameters()
                                                                                                                                               .getLztwm()));
        //Adjust adimc,additional impervious area storage, for evaporation.
        _sacHTData.getSacSmaHtState().setAdimc(_sacHTData.getSacSmaHtState().getAdimc() - e5);
        if(_sacHTData.getSacSmaHtState().getAdimc() < 0.0)
        {
            //E5 can not exceed adimc.
            e5 = e5 + _sacHTData.getSacSmaHtState().getAdimc();
            _sacHTData.getSacSmaHtState().setAdimc(0.0);
        }
        //231
        e5 *= _sacHTData.getSacSmaHtParameters().getAdimp();
        //E5 is et from the area adimp.
        //Compute percolation and runoff amounts.
        twx = inputPrecipitation + _sacHTData.getSacSmaHtState().getUztwc()
            - _sacHTData.getSacSmaHtParameters().getUztwm();

        //Twx is the time interval available moisture in excess
        //Of uztw requirements.
        if(twx >= 0.0) //GO TO 232
        {
            //232
            uztwh += (_sacHTData.getSacSmaHtParameters().getUztwm() - _sacHTData.getSacSmaHtState().getUztwc());
            _sacHTData.getSacSmaHtState().setUztwc(_sacHTData.getSacSmaHtParameters().getUztwm());
        }
        else
        {
            //All moisture held in uztw--no excess.
            _sacHTData.getSacSmaHtState().setUztwc(_sacHTData.getSacSmaHtState().getUztwc() + inputPrecipitation);
            //CV.K  Adjust unfrozen tension water
            uztwh += inputPrecipitation;
            twx = 0.0;
            //GO TO 233
        }

        //Moisture available in excess of uztwc storage.
        //CV.K  232 uztwc=uztwm
        //233 
        _sacHTData.getSacSmaHtState().setAdimc(_sacHTData.getSacSmaHtState().getAdimc() + inputPrecipitation - twx);
        //Compute impervious area runoff.
        roimp = inputPrecipitation * _sacHTData.getSacSmaHtParameters().getPctim();
        //Roimp is runoff from the minimum impervious area.
        //Initialize time interval sums.
        sbf = ssur = sif = sperc = sdro = spbf = 0.0;

        //Determine computational time increments for the basic time
        //interval
        //CV.K      NINC=1.0+0.2*(UZFWC+TWX)
        //CV.K  Percolate unfrozen water only
        ninc = (int)(1.0 + 0.2 * (uzfwh + twx));
        /*
         * NINC=Number of time increments that the time interval is divided into for further soil-moisture accounting.
         * no one increment will exceed 5.0 millimeters of uzfwc+pav
         */
        dinc = (1.0 / ninc) * inputTimeStep;

        //dinc=length of each increment in days.
        pinc = twx / ninc;

        /*
         * PINC=Amount of available moisture for each increment. compute free water depletion fractions for the time
         * increment being used-basic depletions are for one day introduced reduction (ruzice & rlzice) due frozen
         * ground however, primary runoff is unchanged DUZ=1.0-((1.0-UZK)**DINC) DLZS=1.0-((1.0-LZSK)**DINC)
         */
        dlzp = 1.0 - Math.pow((1.0 - _sacHTData.getSacSmaHtParameters().getLzpk()), dinc);

        /*
         * CVK Linear transformation for frozen ground DUZ=1.0-((1.0-UZK*RUZICE)**DINC)
         * DLZS=1.0-((1.0-LZSK*RLZICE)**DINC) CVK Non-linear (correct) transformation for frozen ground
         */

        duz = 1.0 - Math.pow((1.0 - ruzice), dinc);
        dlzs = 1.0 - Math.pow((1.0 - rlzice), dinc);

        //CVK  Adjustment to depletions due to frozen water

        //Start incremental do loop for the time interval.
        for(int i = 0; i < ninc; i++)
        {
            adsur = 0.0;
            //Compute direct runoff (from adimp area).
            ratio = (_sacHTData.getSacSmaHtState().getAdimc() - _sacHTData.getSacSmaHtState().getUztwc())
                / _sacHTData.getSacSmaHtParameters().getLztwm();
            if(ratio < 0.0)
            {
                ratio = 0.0;
            }
            addro = pinc * Math.pow(ratio, 2);

            /*
             * addro is the amount of direct runoff from the area adimp. C compute baseflow and keep track of time
             * interval sum. CV.K bf=lzfpc*dlzp CV.K lzfpc=lzfpc-bf CV.K if (lzfpc.gt. 0.0001) go to 234 CV.K
             * bf=bf+lzfpc CV.K lzfpc=0.0 CV.K baseflow from unfrozen water only
             */

            bf = lzfph * dlzp;
            lzfph -= bf;

            if(lzfph > 0.0001)
            {
                _sacHTData.getSacSmaHtState().setLzfpc(_sacHTData.getSacSmaHtState().getLzfpc() - bf);
            }
            else
            {
                bf += lzfph;
                lzfph = 0.0;
                _sacHTData.getSacSmaHtState().setLzfpc(_sacHTData.getSacSmaHtState().getLzfpc() - bf);
                if(_sacHTData.getSacSmaHtState().getLzfpc() <= 0.0001)
                {
                    _sacHTData.getSacSmaHtState().setLzfpc(0.0);
                }
            }
            //234
            sbf += bf;
            spbf += bf;

            /*
             * cv.k supplemental flow from unfrozen water only (note, dlzs cv.k note, dlzs is reduced due frozen ground
             * cv.k BF=LZFSC*DLZS cv.k LZFSC=LZFSC-BF cv.k IF(LZFSC.GT.0.0001) GO TO 235 cv.k BF=BF+LZFSC cv.k LZFSC=0.0
             */

            bf = lzfsh * dlzs;
            lzfsh -= bf;

            if(lzfsh > 0.0001)
            {
                _sacHTData.getSacSmaHtState().setLzfsc(_sacHTData.getSacSmaHtState().getLzfsc() - bf);
                //go to 2	35
            }
            else
            {
                bf += lzfsh;
                lzfsh = 0.0;
                _sacHTData.getSacSmaHtState().setLzfsc(_sacHTData.getSacSmaHtState().getLzfsc() - bf);
                if(_sacHTData.getSacSmaHtState().getLzfsc() <= 0.0001)
                {
                    _sacHTData.getSacSmaHtState().setLzfsc(0.0);
                }
            }

            // CV.K--------------------------------------------
            //235 
            sbf += bf;
            /*
             * C C compute percolation-if no water available then skip ccvk if((pinc+uzfwc).gt.0.01) go to 251
             */

            xx1 = pinc + uzfwh;

            if(xx1 > 0.01)
            {
                //go to 251
                //251 
                percm = _sacHTData.getSacSmaHtParameters().getLzfpm() * dlzp
                    + _sacHTData.getSacSmaHtParameters().getLzfsm() * dlzs;

                /*
                 * CVK perc=percm*(uzfwc/uzfwm) CV.K use only unfrozen water ratios ccvk new change: percolation reduced
                 * by ruzperc CC perc=percm*(uzfwh/uzfwm)*ruzice
                 */

                percolation = percm * (uzfwh / _sacHTData.getSacSmaHtParameters().getUzfwm()) * ruzperc;

                /*
                 * CV.K defr=1.0-((lztwc+lzfpc+lzfsc)/(lztwm+LZFPM+LZFSM)) cvk 6/22/00
                 * defr=1.0-((lztwh+lzfph+lzfsh)/(lztwm+LZFPM+LZFSM)) cvk better to keep original definition of DEFR
                 * using total water
                 */

                lowerZoneDeficitRatio = 1.0 - ((_sacHTData.getSacSmaHtState().getLztwc()
                    + _sacHTData.getSacSmaHtState().getLzfpc() + _sacHTData.getSacSmaHtState().getLzfsc()) / (_sacHTData.getSacSmaHtParameters()
                                                                                                                        .getLztwm()
                    + _sacHTData.getSacSmaHtParameters().getLzfpm() + _sacHTData.getSacSmaHtParameters().getLzfsm()));

                _changeInPercolationRate = 1.0;
                _changeInInterflowWithdrawlRate = 1.0;

                upperZoneDeficitRatio = 1.0 - ((_sacHTData.getSacSmaHtState().getUztwc() + _sacHTData.getSacSmaHtState()
                                                                                                     .getUzfwc()) / (_sacHTData.getSacSmaHtParameters()
                                                                                                                               .getUztwm() + _sacHTData.getSacSmaHtParameters()
                                                                                                                                                       .getUzfwm()));

                /*
                 * CVK call fgfr1(defr,fr,uzdefr,fi) CVK if( ivers .eq. 1) then CVK if ivers=1, old version; if ivers=2,
                 * new vers. frost index, CVK but old vers. of percolat. and interflow reduction
                 */

                //239
                percolation *= ((1.0 + _sacHTData.getSacSmaHtParameters().getZperc()
                    * Math.pow(lowerZoneDeficitRatio, _sacHTData.getSacSmaHtParameters().getRexp())) * _changeInPercolationRate);
                //C     Note...percolation occurs from uzfwc before pav is added.
                //CV.K  IF(PERC.LT.UZFWC) GO TO 241

                /*
                 * sudha - adding the isNaN check,because behaviour in FORTRAN differs slightly, causing the first part
                 * of the condition to fail in this case, but not in FORTRAN - don't just assume that that condition
                 * isn't being caught by the comparison to uzfwh
                 */

                if((percolation >= uzfwh) || (Double.isNaN(percolation)))
                {
                    //C      percolation rate exceeds uzfwh.
                    //CV.K      perc=uzfwc
                    //C     percolation rate is less than uzfwh.
                    percolation = uzfwh;
                }

                //241 
                _sacHTData.getSacSmaHtState().setUzfwc(_sacHTData.getSacSmaHtState().getUzfwc() - percolation);

                //CV.K  adjust unfrozen storage also  

                uzfwh -= percolation;

                //check to see if percolation exceeds lower zone deficiency.
                check = _sacHTData.getSacSmaHtState().getLztwc() + _sacHTData.getSacSmaHtState().getLzfpc()
                    + _sacHTData.getSacSmaHtState().getLzfsc() + percolation
                    - _sacHTData.getSacSmaHtParameters().getLztwm() - _sacHTData.getSacSmaHtParameters().getLzfpm()
                    - _sacHTData.getSacSmaHtParameters().getLzfsm();
                if(check > 0.0)
                {
                    percolation -= check;
                    _sacHTData.getSacSmaHtState().setUzfwc(_sacHTData.getSacSmaHtState().getUzfwc() + check);
                    //CV.K  adjust unfrozen starage also
                    uzfwh += check;
                }

                //242 
                sperc += percolation;

                /*
                 * Sperc is the time interval summation of perc C compute interflow and keep track of time interval sum.
                 * C note...pinc has not yet been added CV.K del=uzfwc*duz*fi CVK interflow also reduced due frofen
                 * ground (duz reduced by ruzice) CVK additional reduction due impervious frozen areas (fi) is optional
                 * CVK in the new version. basic option is fi=1
                 */

                del = uzfwh * duz * _changeInInterflowWithdrawlRate;
                sif += del;
                _sacHTData.getSacSmaHtState().setUzfwc(_sacHTData.getSacSmaHtState().getUzfwc() - del);
                //CV.K  Adjust unfrozen storage also
                uzfwh -= del;

                /*
                 * C Distribe percolated water into the lower zones C tension water must be filled first except for the
                 * pfree area. C perct is percolation to tension water and percf is percolation C going to free water.
                 */

                perct = percolation * (1.0 - _sacHTData.getSacSmaHtParameters().getPfree());
                xx1 = perct + _sacHTData.getSacSmaHtState().getLztwc();

                if(xx1 > _sacHTData.getSacSmaHtParameters().getLztwm())
                { //GO TO 243
                    percf = perct + _sacHTData.getSacSmaHtState().getLztwc()
                        - _sacHTData.getSacSmaHtParameters().getLztwm();
                    //CV.K  Change unfrozen water storage
                    lztwh += (_sacHTData.getSacSmaHtParameters().getLztwm() - _sacHTData.getSacSmaHtState().getLztwc());
                    _sacHTData.getSacSmaHtState().setLztwc(_sacHTData.getSacSmaHtParameters().getLztwm());

                    //continue with 244
                }
                else
                {
                    _sacHTData.getSacSmaHtState().setLztwc(_sacHTData.getSacSmaHtState().getLztwc() + perct);
                    //CV.K  adjust unfrozen storage also
                    lztwh += perct;
                    percf = 0.0;
                    //GO TO 244
                }

                /*
                 * Distribute percolation in excess of tension Requirements among the free water storages.
                 */
                percf = percf + percolation * _sacHTData.getSacSmaHtParameters().getPfree();
                // if(percf == 0.0) go to 245
                if(percf != 0.0)
                {

                    hpl = _sacHTData.getSacSmaHtParameters().getLzfpm()
                        / (_sacHTData.getSacSmaHtParameters().getLzfpm() + _sacHTData.getSacSmaHtParameters()
                                                                                     .getLzfsm());
                    /*
                     * Hpl is the relative size of the primary storage as compared with total lower zone free water
                     * storage.
                     */
                    lowerZonePrimaryRatio = _sacHTData.getSacSmaHtState().getLzfpc()
                        / _sacHTData.getSacSmaHtParameters().getLzfpm();
                    lowerZoneSecondaryRatio = _sacHTData.getSacSmaHtState().getLzfsc()
                        / _sacHTData.getSacSmaHtParameters().getLzfsm();
                    /*
                     * ratlp and ratls are content to capacity ratios, or in other words, the relative fullness of each
                     * storage
                     */
                    fracp = (hpl * 2.0 * (1.0 - lowerZonePrimaryRatio))
                        / ((1.0 - lowerZonePrimaryRatio) + (1.0 - lowerZoneSecondaryRatio));
                    //fracp is the fraction going to primary.
                    if(fracp > 1.0)
                    {
                        fracp = 1.0;
                    }
                    percp = percf * fracp;
                    percs = percf - percp;
                    /*
                     * Percp and percs are the amount of the excess percolation going to primary and supplemental
                     * storges,respectively.
                     */
                    _sacHTData.getSacSmaHtState().setLzfsc(_sacHTData.getSacSmaHtState().getLzfsc() + percs);
                    //CV.K      if(lzfsc.le.lzfsm) go to 246
                    if(_sacHTData.getSacSmaHtState().getLzfsc() <= _sacHTData.getSacSmaHtParameters().getLzfsm())
                    {
                        lzfsh += percs;
                        //GO TO 246
                    }
                    else
                    {
                        percs -= _sacHTData.getSacSmaHtState().getLzfsc()
                            + _sacHTData.getSacSmaHtParameters().getLzfsm();
                        //CV.K  adjust unfrozen storage also
                        lzfsh += percs;
                        _sacHTData.getSacSmaHtState().setLzfsc(_sacHTData.getSacSmaHtParameters().getLzfsm());
                    }
                    //246 
                    _sacHTData.getSacSmaHtState().setLzfpc(_sacHTData.getSacSmaHtState().getLzfpc() + (percf - percs));
                    //C     CHECK TO MAKE SURE LZFPC DOES NOT EXCEED LZFPM.
                    //CV.K      if lzfpc.le.lzfpm) go to 245
                    if(_sacHTData.getSacSmaHtState().getLzfpc() <= _sacHTData.getSacSmaHtParameters().getLzfpm())
                    {
                        lzfph += (percf - percs);
                        //	GO TO 245
                    }
                    else
                    {
                        excess = _sacHTData.getSacSmaHtState().getLzfpc()
                            - _sacHTData.getSacSmaHtParameters().getLzfpm();
                        _sacHTData.getSacSmaHtState().setLztwc(_sacHTData.getSacSmaHtState().getLztwc() + excess);
                        //CV.K  adjust unfrozen storages also
                        lztwh += excess;
                        lzfph += (percf - percs) - excess;
                        _sacHTData.getSacSmaHtState().setLzfpc(_sacHTData.getSacSmaHtParameters().getLzfpm());
                    }
                }

                /* C distribute pinc between uzfwc and surface runoff. */
                //245 
                //IF(PINC.EQ.0.0) GO TO 249
                if(pinc != 0.0)
                {
                    //C check if pinc exceeds uzfwm
                    xx1 = pinc + _sacHTData.getSacSmaHtState().getUzfwc();
                    if(xx1 <= _sacHTData.getSacSmaHtParameters().getUzfwm())
                    {
                        //No surface runoff
                        _sacHTData.getSacSmaHtState().setUzfwc(_sacHTData.getSacSmaHtState().getUzfwc() + pinc);
                        //CV.K  Adjust unfrozen storage also
                        uzfwh += pinc;
                        /*
                         * GO TO 249 C Compute surface runoff (sur) and keep track of time interval sum.
                         */
                    }
                    else
                    {
                        //248 
                        sur = pinc + _sacHTData.getSacSmaHtState().getUzfwc()
                            - _sacHTData.getSacSmaHtParameters().getUzfwm();
                        _sacHTData.getSacSmaHtState().setUzfwc(_sacHTData.getSacSmaHtParameters().getUzfwm());
                        //CV.K adjust unfrozen storage also
                        uzfwh += (pinc - sur);
                        ssur = ssur + sur * parea;
                        adsur = sur * (1.0 - addro / pinc);
                        /*
                         * C adsur is the amount of surface runoff which comes C from that portion of adimp which is not
                         * C currently generating direct runoff. addro/pinc C is the fraction of adimp currently
                         * generating C direct runoff.
                         */
                        ssur += (adsur * _sacHTData.getSacSmaHtParameters().getAdimp());
                    }
                }
            }
            else
            {
                //IF(xx1 < 0.01) 
                _sacHTData.getSacSmaHtState().setUzfwc(_sacHTData.getSacSmaHtState().getUzfwc() + pinc);
                //CV.K  add to unfrozen water also
                uzfwh += pinc;
                //go to 249
            }

            // adimp area water balance -- sdro is the 6 hr sum of direct runoff. 
            // 249 
            _sacHTData.getSacSmaHtState().setAdimc(_sacHTData.getSacSmaHtState().getAdimc() + pinc - addro - adsur);
            xx1 = _sacHTData.getSacSmaHtParameters().getUztwm() + _sacHTData.getSacSmaHtParameters().getLztwm();
            //if (adimc.le.xx1) go to 247
            if(_sacHTData.getSacSmaHtState().getAdimc() > xx1)
            {
                addro += (_sacHTData.getSacSmaHtState().getAdimc() - xx1);
                _sacHTData.getSacSmaHtState().setAdimc(xx1);
            }
            //247 
            sdro += (addro * _sacHTData.getSacSmaHtParameters().getAdimp());
            if(_sacHTData.getSacSmaHtState().getAdimc() < 0.00001)
            {
                _sacHTData.getSacSmaHtState().setAdimc(0.0);
                //240 continue
                //end of incremental do loop. 
            }
        }

        /*
         * C compute sums and adjust runoff amounts by the area over C which they are generated.
         */
        eused = e1 + e2 + e3;
        // C     eused is the et from parea which is 1.0-adimp-pctim
        sif *= parea;
        /*
         * C separate channel component of baseflow C from the non-channel component
         */
        tbf = sbf * parea;
        //tbf is total baseflow
        bfcc = tbf * (1.0 / (1.0 + _sacHTData.getSacSmaHtParameters().getSide()));
        //bfcc is baseflow, channel component
        bfp = spbf * parea / (1.0 + _sacHTData.getSacSmaHtParameters().getSide());
        bfs = bfcc - bfp;
        if(bfs < 0.0)
        {
            bfs = 0.0;
            /*
             * Bfncc is baseflow,non-channel component Add to monthly sums.
             */
        }

        //Compute total channel inflow for the time interval. 
        runoff = roimp + sdro + ssur + sif + bfcc;

        // Compute e4-et from riparian vegetation.
        e4 = (etDemand - eused) * _sacHTData.getSacSmaHtParameters().getRiva();
        //Subtract e4 from channel inflow
        runoff -= e4;

        //IF(tci.GE.0.0) GO TO 250
        if(runoff < 0.0)
        {
            e4 += runoff;
            runoff = 0.0;
        }

        //250 
        //Compute total evapotranspiration-tet
        eused *= parea;
        tet = eused + e5 + e4;

        //Check that adimc.ge.uztwc
        if(_sacHTData.getSacSmaHtState().getAdimc() < _sacHTData.getSacSmaHtState().getUztwc())
        {
            _sacHTData.getSacSmaHtState().setAdimc(_sacHTData.getSacSmaHtState().getUztwc());
        }

        /*
         * Compute new frost index and moisture transfer. CVK Eric's version of frost index
         */
        if(_sacHTData.getSacSmaHtState().getUztwh() < 0.)
        {
            _sacHTData.getSacSmaHtState().setUztwh((_sacHTData.getSacSmaHtState().getUztwh() + uztwh));
        }
        else
        {
            _sacHTData.getSacSmaHtState().setUztwh(uztwh);
        }

        if(_sacHTData.getSacSmaHtState().getUzfwh() < 0.)
        {
            _sacHTData.getSacSmaHtState().setUzfwh(_sacHTData.getSacSmaHtState().getUzfwh() + uzfwh);
        }
        else
        {
            _sacHTData.getSacSmaHtState().setUzfwh(uzfwh);
        }
        if(_sacHTData.getSacSmaHtState().getLztwh() < 0.)
        {
            _sacHTData.getSacSmaHtState().setLztwh((float)_sacHTData.getSacSmaHtState().getLztwh() + (float)lztwh);
        }
        else
        {
            _sacHTData.getSacSmaHtState().setLztwh((float)lztwh);
        }

        if(_sacHTData.getSacSmaHtState().getLzfsh() < 0.)
        {
            _sacHTData.getSacSmaHtState().setLzfsh(_sacHTData.getSacSmaHtState().getLzfsh() + lzfsh);
        }
        else
        {
            _sacHTData.getSacSmaHtState().setLzfsh(lzfsh);
        }

        if(_sacHTData.getSacSmaHtState().getLzfph() < 0.)
        {
            _sacHTData.getSacSmaHtState().setLzfph(_sacHTData.getSacSmaHtState().getLzfph() + lzfph);
        }
        else
        {
            _sacHTData.getSacSmaHtState().setLzfph(lzfph);
        }

        _frostDepth = OHDConstants.MISSING_DATA;

        frost2_1(inputPrecipitation,
                 airTemperature,
                 snowWaterEquivalent,
                 snowCoverAsPercent,
                 snowDepth,
                 inputTimeStep,
                 icall);
        /*
         * double frzdInCm = OHDModelConstants.MISSING_DATA; int layerIndex; for(layerIndex = _numberOfSoilLayers - 1;
         * layerIndex >= 0; layerIndex--) { System.out.println("temp = " +
         * _sacHTData.getSacSmaHtState().getSoilLayerTemperature(layerIndex));
         * if(_sacHTData.getSacSmaHtState().getSoilLayerTemperature(layerIndex) <= 0.0) { break; } } if(layerIndex < 0.)
         * { frzdInCm = OHDModelConstants.MISSING_DATA; if(_sacHTData.getSacSmaHtState().getFindex() < 0.) { frzdInCm =
         * -(float)_frostDepth; } } else { double dz1 = 0; double dz2 = 0; if(layerIndex == 0) { dz1 = -0.5f *
         * (float)_sacHTData.getSacSmaHtState().getSoilLayerDepth(0); } else { dz1 = 0.5f *
         * ((float)_sacHTData.getSacSmaHtState().getSoilLayerDepth(layerIndex - 1) -
         * (float)_sacHTData.getSacSmaHtState() .getSoilLayerDepth(layerIndex)); }
         * if(_sacHTData.getSacSmaHtState().getSoilLayerTemperature(layerIndex) == 0) { frzdInCm = (float)dz1 +
         * (float)_sacHTData.getSacSmaHtState().getSoilLayerDepth(layerIndex) * 100.f; } else { dz2 =
         * _sacHTData.getSacSmaHtState().getSoilLayerDepth(layerIndex) -
         * _sacHTData.getSacSmaHtState().getSoilLayerDepth(layerIndex + 1); if(layerIndex != _numberOfSoilLayers) { dz2
         * *= 0.5; } final double frzdNumerator =
         * -(float)_sacHTData.getSacSmaHtState().getSoilLayerTemperature(layerIndex) ((float)dz1 + (float)dz2); final
         * double frzdDenominator = ((float)_sacHTData.getSacSmaHtState() .getSoilLayerTemperature(layerIndex + 1) -
         * (float)_sacHTData.getSacSmaHtState() .getSoilLayerTemperature(layerIndex)); final double frzdDatum =
         * (float)dz1 + (float)_sacHTData.getSacSmaHtState().getSoilLayerDepth(layerIndex); frzdInCm = 100.f *
         * ((float)frzdDatum - ((float)frzdNumerator / (float)frzdDenominator)); } } if((layerIndex <
         * _previousLayerIndex) && (_sacHTData.getSacSmaHtState().getFindex() <= _previousFrozenGroundIndex)) { frzdInCm
         * = _previousFrozenGroundDepth; } _previousLayerIndex = layerIndex; _previousFrozenGroundDepth = frzdInCm;
         * _previousFrozenGroundIndex = _sacHTData.getSacSmaHtState().getFindex();
         */
        _sacHTData.setChannelInflow(runoff);
        _sacHTData.setFrozenDepth(_frostDepth);
        _sacHTData.setRunoffComponents(roimp, sdro, ssur, sif, bfp, bfs);
        _currentResult.setResults(_sacHTData);

        return _currentResult;
    } //close fLand1 method

    public void frost2_1(final double inputPrecipitation,
                         final double airTemperatureInCelsius,
                         final double snowWaterEquivalentInMm,
                         final double snowCoverAsPercentage,
                         final double snowDepthInCm,
                         final double modelTimeStepInHours,
                         final int iopt) throws Exception
    {
        //  Input arguments
        //  px,dt,ta,we,aesc,sh,dt,iprint,iout,iopt
        //  Output arguments
        //  default constants
        //  dtsim - simulation time step of frozen ground model, in sec.
        //  it can be less than sac-sma time step, but not greater.

        final double rHsts[] = new double[SacSmaHTConstants.MAX_NUMBER_OF_USER_DEFINED_PROFILES];
        double inputSoilTemperature[] = new double[SacSmaHTConstants.MAX_NUMBER_OF_USER_DEFINED_PROFILES];
        double outputSoilTemperature[] = new double[SacSmaHTConstants.MAX_NUMBER_OF_USER_DEFINED_PROFILES];
        double maximumPorisity, dwf, snowDepthInMeters, xxs, snowDensity, airTemperatureInKelvins, frost;
        int nup, ifrz, isfrz, itfrz;
        double dwt;
        double frozenGroundSimulationTimeStepInSeconds;

        //		calculate characteristics of the first thin layer (residue)
        //		3cm thickness was assigned for this layer
        maximumPorisity = SacSmaHTConstants.MAX_SOIL_POROSITY;
        //		default max porosity of this layer is 0.58

        if(iopt == 0)
        {

            //		define temperature at the bottom boundary 
            //		default temperature at _frzCnst._ZBOT m depths is 10.0 C

            //		define frozen ground simulation time step as a fraction of sac dt
            //		which is in days
            frozenGroundSimulationTimeStepInSeconds = modelTimeStepInHours * 3600 * 24
                / SacSmaHTConstants.ONE_HALF_HOUR_IN_SECONDS;
            //original IDT=RDT+0.5;
            //round RDT to achieve the same thing as above
            _frozenGroundSimulationTimeStepAsInt = (int)(Math.round(frozenGroundSimulationTimeStepInSeconds));
            _modelTimeStepInSeconds = SacSmaHTConstants.ONE_HALF_HOUR_IN_SECONDS;
            if(frozenGroundSimulationTimeStepInSeconds < 1.)
            {
                _modelTimeStepInSeconds = modelTimeStepInHours * 3600;
            }
            else
            {
                if(((modelTimeStepInHours * 3600 * 24) % _modelTimeStepInSeconds) != 0.)
                {
                    _modelTimeStepInSeconds = modelTimeStepInHours * 3600 * 24 / _frozenGroundSimulationTimeStepAsInt;
                }
            }

            //			no any states to recalculate
            //			c  Cox subroutine should be added
            //			get soil layers definition

            //			generate initial states assuming uniform distribution of total water.
            //			unfrozen water will be distributed depending on soil temperature
            //			uztwh, uzfwh, lztwh, lzfsh, lzfph can be negative which means that
            //			liquid water are below wilting point

        }
        //		end of initial loop

        //		frost index and soil temperature states
        frost = _sacHTData.getSacSmaHtState().getFindex();

        for(int i = 0; i < _sacHTData.getSacSmaHtState().getNumberOfSoilLayers(); i++)
        {
            inputSoilTemperature[i] = (float)_sacHTData.getSacSmaHtState().getSoilLayerTemperature(i)
                + (float)SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS;
        }

        //		call subroutine to recalculate sac-sma states into frozen
        //		ground model states:
        //		upper zone storages
        dwt = _sacHTData.getFrdStfg().rTuz * (_sacHTData.getSacSmaHtState().getUztwc() - _uztwc0);
        dwf = _sacHTData.getFrdStfg().rTuz * (_sacHTData.getSacSmaHtState().getUzfwc() - _uzfwc0);
        nup = 2;
        sac2Frz1(dwt,
                 dwf,
                 nup,
                 _sacHTData.getFrdStfg().nUpl,
                 _sacHTData.getSacSmaHtState().getSoilLayerDepthArray(),
                 _sacHTData.getFrdStfg().maximumSoilMoisture);

        //		lower zone storages
        dwt = _sacHTData.getFrdStfg().rTlz * (_sacHTData.getSacSmaHtState().getLztwc() - _lztwc0);
        dwf = _sacHTData.getFrdStfg().rTlz
            * (_sacHTData.getSacSmaHtState().getLzfsc() + _sacHTData.getSacSmaHtState().getLzfpc() - _lzfsc0 - _lzfpc0);
        nup = _sacHTData.getFrdStfg().nUpl + 1;
        sac2Frz1(dwt,
                 dwf,
                 nup,
                 _sacHTData.getFrdStfg().nsac,
                 _sacHTData.getSacSmaHtState().getSoilLayerDepthArray(),
                 _sacHTData.getFrdStfg().maximumSoilMoisture);

        // If only soil moisture calculations to be done, can ignore the frozen ground calculations below
        if(!_sacHTData.getSacSmaHtParameters().getModelRunType().equals(SacSmaHTConstants.ONLY_SOIL_RUN_STR))
        {

            //		calculate snow density
            //		sh - snow depth in cm
            //		sr - snow density in g/cm3
            //		we - snow water equivalent in mm
            if(snowWaterEquivalentInMm == 0.)
            {
                snowDepthInMeters = 0.;
                snowDensity = 0.2;
            }
            else
            {
                snowDensity = 0.1 * snowWaterEquivalentInMm / snowDepthInCm;
                snowDepthInMeters = 0.01 * snowDepthInCm;
            }
            //		convert air temperature into kelvin units
            airTemperatureInKelvins = airTemperatureInCelsius + SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS;
            //		frozen ground simulation loop 
            for(int it = 0; it < _frozenGroundSimulationTimeStepAsInt; it++)
            {
                //			hrt1 routine calcs the right hand side of the soil temp dif eqn
                _heatTransferSim.heatTransferSimulation(rHsts,
                                                        inputSoilTemperature,
                                                        _sacHTData.getFrdStfg().maximumSoilMoisture,
                                                        _sacHTData.getSacSmaHtState().getNumberOfSoilLayers(),
                                                        _sacHTData.getSacSmaHtState().getSoilLayerDepthArray(),
                                                        airTemperatureInKelvins,
                                                        _sacHTData.getSacSmaHtParameters().getTbot(),
                                                        SacSmaHTConstants.DEPTH_AT_BOTTOM_LAYER_IN_METERS,
                                                        _sacHTData.getFrdStfg().psisat,
                                                        _modelTimeStepInSeconds,
                                                        _sacHTData.getFrdStfg().brt,
                                                        snowDepthInMeters,
                                                        snowDensity,
                                                        snowCoverAsPercentage,
                                                        _sacHTData.getFrdStfg().soilType,
                                                        _sacHTData.getFrdStfg().quartz,
                                                        maximumPorisity,
                                                        _sacHTData.getSacSmaHtParameters().getRunoffEffect());

                //			hstep routine calcs/updates soil temps based on rhsts

                outputSoilTemperature = _heatTransferSim.hstep(inputSoilTemperature,
                                                               rHsts,
                                                               _modelTimeStepInSeconds,
                                                               _sacHTData.getSacSmaHtState().getNumberOfSoilLayers());

                //			double calling hrt1 & hstep to reduce noise. 
                //			it runs only if there is an ice in one soil layer at least
                ifrz = 0;
                for(int I = 0; I < _sacHTData.getSacSmaHtState().getNumberOfSoilLayers(); I++)
                {
                    xxs = _sacHTData.getSacSmaHtState().getSoilLayerMoistureContent(I)
                        - _sacHTData.getSacSmaHtState().getSoilLayerUnfrozenWater(I);

                    if(MathHelper.isGreaterThanZero(xxs))
                    // chnanged to match fortran; precision diffs
                    //                    if(xxs > 0.0)
                    {
                        ifrz = 1;
                    }
                }
                if(ifrz == 1)
                {

                    //				second call of hrt1 and hstep
                    for(int I = 0; I < _sacHTData.getSacSmaHtState().getNumberOfSoilLayers(); I++)
                    {
                        inputSoilTemperature[I] = 0.5f * ((float)outputSoilTemperature[I] + (float)inputSoilTemperature[I]);
                    }

                    _heatTransferSim.heatTransferSimulation(rHsts,
                                                            inputSoilTemperature,
                                                            _sacHTData.getFrdStfg().maximumSoilMoisture,
                                                            _sacHTData.getSacSmaHtState().getNumberOfSoilLayers(),
                                                            _sacHTData.getSacSmaHtState().getSoilLayerDepthArray(),
                                                            airTemperatureInKelvins,
                                                            _sacHTData.getSacSmaHtParameters().getTbot(),
                                                            SacSmaHTConstants.DEPTH_AT_BOTTOM_LAYER_IN_METERS,
                                                            _sacHTData.getFrdStfg().psisat,
                                                            _modelTimeStepInSeconds,
                                                            _sacHTData.getFrdStfg().brt,
                                                            snowDepthInMeters,
                                                            snowDensity,
                                                            snowCoverAsPercentage,
                                                            _sacHTData.getFrdStfg().soilType,
                                                            _sacHTData.getFrdStfg().quartz,
                                                            maximumPorisity,
                                                            _sacHTData.getSacSmaHtParameters().getRunoffEffect());

                    //				hstep routine calcs/updates soil temps based on rhsts
                    inputSoilTemperature = _heatTransferSim.hstep(inputSoilTemperature,
                                                                  rHsts,
                                                                  _modelTimeStepInSeconds,
                                                                  _sacHTData.getSacSmaHtState().getNumberOfSoilLayers());

                }
                else
                {
                    //				skip second call of hrt1 and hstep
                    for(int i = 0; i < _sacHTData.getSacSmaHtState().getNumberOfSoilLayers(); i++)
                    {
                        inputSoilTemperature[i] = (float)outputSoilTemperature[i];
                    }

                }

            }

            //		store soil temperature states in celsius
            itfrz = 0;
            for(int i = 0; i < _sacHTData.getSacSmaHtState().getNumberOfSoilLayers(); i++)
            {
                _sacHTData.getSacSmaHtState()
                          .setSoilLayerTemperature(i,
                                                   (float)inputSoilTemperature[i]
                                                       - (float)SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS);

                if(_sacHTData.getSacSmaHtState().getSoilLayerTemperature(i) < 0.)
                {
                    itfrz += 1;
                }
            }

            //		recalculate unfrozen moisture states into sac-sma storages 
            isfrz = 0;
            if(_sacHTData.getSacSmaHtState().getUztwc() != _sacHTData.getSacSmaHtState().getUztwh())
            {
                isfrz = isfrz + 1;
            }
            if(_sacHTData.getSacSmaHtState().getUzfwc() != _sacHTData.getSacSmaHtState().getUzfwh())
            {
                isfrz = isfrz + 1;
            }
            if(_sacHTData.getSacSmaHtState().getLztwc() != _sacHTData.getSacSmaHtState().getLztwh())
            {
                isfrz = isfrz + 1;
            }
            if(_sacHTData.getSacSmaHtState().getLzfsc() != _sacHTData.getSacSmaHtState().getLzfsh())
            {
                isfrz = isfrz + 1;
            }
            if(_sacHTData.getSacSmaHtState().getLzfpc() != _sacHTData.getSacSmaHtState().getLzfph())
            {
                isfrz = isfrz + 1;
            }
            if(isfrz != 0 || itfrz != 0)
            {
                frz2Sac1(inputPrecipitation,
                         _sacHTData.getSacSmaHtState().getSoilLayerDepthArray(),
                         _sacHTData.getFrdStfg().nUpl,
                         _sacHTData.getFrdStfg().nsac,
                         _sacHTData.getFrdStfg().soilWiltingPoint,
                         _sacHTData.getFrdStfg().rTuz,
                         _sacHTData.getFrdStfg().rTlz);

                if(_sacHTData.getFrdStfg().nsac != _sacHTData.getSacSmaHtState().getNumberOfSoilLayers()
                    && _sacHTData.getSacSmaHtState().getNumberOfSoilLayers() > 0)
                {
                    _sacHTData.getSacSmaHtState()
                              .setFindex(((float)_sacHTData.getSacSmaHtState().getFindex() - 1000.f
                                  * ((float)_sacHTData.getSacSmaHtState()
                                                      .getSoilLayerMoistureContent(_sacHTData.getSacSmaHtState()
                                                                                             .getNumberOfSoilLayers() - 1) - (float)_sacHTData.getSacSmaHtState()
                                                                                                                                              .getSoilLayerUnfrozenWater(_sacHTData.getSacSmaHtState()
                                                                                                                                                                                   .getNumberOfSoilLayers() - 1))
                                  * ((float)_sacHTData.getSacSmaHtState()
                                                      .getSoilLayerDepth(_sacHTData.getSacSmaHtState()
                                                                                   .getNumberOfSoilLayers() - 2) - (float)_sacHTData.getSacSmaHtState()
                                                                                                                                    .getSoilLayerDepth(_sacHTData.getSacSmaHtState()
                                                                                                                                                                 .getNumberOfSoilLayers() - 1))
                                  / (float)SacSmaHTConstants.FRST_FACT));
                }
                //			frost depth calculation                
                _frostDepth = frzInd1(_sacHTData.getSacSmaHtState().getSoilLayerDepthArray(),
                                      _sacHTData.getSacSmaHtState().getNumberOfSoilLayers(),
                                      _sacHTData.getFrdStfg().soilWiltingPoint);
            }
        }
        // post process step to compute layer dependent values at user requested depths
        computeInterpolatedValues();

        return;
    }

    private void computeInterpolatedValues() throws Exception
    {
        final int numberOfModelDefinedLayers = _sacHTData.getSacSmaHtState().getNumberOfSoilLayers() + 1;

        final double modelDefinedTemperature[] = new double[numberOfModelDefinedLayers];
        final double modelDefinedSoilProfile[] = new double[numberOfModelDefinedLayers];
        final double modelDefinedDimensionlessSoilMoisture[] = new double[numberOfModelDefinedLayers];
        final double modelDefinedSoilMoistureAsPercent[] = new double[numberOfModelDefinedLayers];
        final double modelDefinedUnfrozenWaterContent[] = new double[numberOfModelDefinedLayers];

        //define top layer
        modelDefinedSoilProfile[0] = -0.5 * _sacHTData.getSacSmaHtState().getSoilLayerDepth(0);
        modelDefinedTemperature[0] = _sacHTData.getSacSmaHtState().getSoilLayerTemperature(0);
        modelDefinedDimensionlessSoilMoisture[0] = _sacHTData.getSacSmaHtState().getSoilLayerMoistureContent(1);
        modelDefinedSoilMoistureAsPercent[0] = (modelDefinedDimensionlessSoilMoisture[0] - _sacHTData.getFrdStfg().soilWiltingPoint)
            / (_sacHTData.getFrdStfg().maximumSoilMoisture - _sacHTData.getFrdStfg().soilWiltingPoint);
        modelDefinedUnfrozenWaterContent[0] = _sacHTData.getSacSmaHtState().getSoilLayerUnfrozenWater(1);

        //define bottom layer
        modelDefinedSoilProfile[numberOfModelDefinedLayers - 1] = -SacSmaHTConstants.DEPTH_AT_BOTTOM_LAYER_IN_METERS;
        modelDefinedTemperature[numberOfModelDefinedLayers - 1] = _sacHTData.getSacSmaHtParameters().getTbot()
            - SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS;
        modelDefinedDimensionlessSoilMoisture[numberOfModelDefinedLayers - 1] = _sacHTData.getFrdStfg().maximumSoilMoisture;
        modelDefinedSoilMoistureAsPercent[numberOfModelDefinedLayers - 1] = (modelDefinedDimensionlessSoilMoisture[numberOfModelDefinedLayers - 1] - _sacHTData.getFrdStfg().soilWiltingPoint)
            / (_sacHTData.getFrdStfg().maximumSoilMoisture - _sacHTData.getFrdStfg().soilWiltingPoint);
        modelDefinedUnfrozenWaterContent[numberOfModelDefinedLayers - 1] = _sacHTData.getFrdStfg().maximumSoilMoisture;

        //define middle layers
        for(int i = 1; i < numberOfModelDefinedLayers - 1; i++)
        {
            modelDefinedSoilProfile[i] = -0.5
                * (_sacHTData.getSacSmaHtState().getSoilLayerDepthArray()[i - 1] + _sacHTData.getSacSmaHtState()
                                                                                             .getSoilLayerDepthArray()[i]);
            modelDefinedTemperature[i] = _sacHTData.getSacSmaHtState().getSoilLayerTemperature(i);
            modelDefinedDimensionlessSoilMoisture[i] = _sacHTData.getSacSmaHtState().getSoilLayerMoistureContent(i);
            modelDefinedSoilMoistureAsPercent[i] = (modelDefinedDimensionlessSoilMoisture[i] - _sacHTData.getFrdStfg().soilWiltingPoint)
                / (_sacHTData.getFrdStfg().maximumSoilMoisture - _sacHTData.getFrdStfg().soilWiltingPoint);
            modelDefinedUnfrozenWaterContent[i] = _sacHTData.getSacSmaHtState().getSoilLayerUnfrozenWater(i);
        }

        final double[] userDefinedDepths = _sacHTData.getSacSmaHtParameters().getUserDefinedSoilLayerDepths();
        double[] result;

        // if only soil moisture calculations to be done, no need to interpolate temperature stuff
        if(!_sacHTData.getSacSmaHtParameters().getModelRunType().equals(SacSmaHTConstants.ONLY_SOIL_RUN_STR))
        {
            result = SacSmaHtCommonMethods.interpolateSoilVariableValue(modelDefinedTemperature,
                                                                modelDefinedSoilProfile,
                                                                userDefinedDepths,
                                                                _sacHTData.getLogger());
            _sacHTData.setInterpolatedSoilTemperature(result);
        }

        result = SacSmaHtCommonMethods.interpolateSoilVariableValue(modelDefinedDimensionlessSoilMoisture,
                                                            modelDefinedSoilProfile,
                                                            userDefinedDepths,
                                                            _sacHTData.getLogger());
        _sacHTData.setInterpolatedTotalSoilMoisture(result);

        result = SacSmaHtCommonMethods.interpolateSoilVariableValue(modelDefinedSoilMoistureAsPercent,
                                                            modelDefinedSoilProfile,
                                                            userDefinedDepths,
                                                            _sacHTData.getLogger());
        _sacHTData.setInterpolatedTotalSoilMoistureAsPercent(result);

        result = SacSmaHtCommonMethods.interpolateSoilVariableValue(modelDefinedUnfrozenWaterContent,
                                                            modelDefinedSoilProfile,
                                                            userDefinedDepths,
                                                            _sacHTData.getLogger());
        _sacHTData.setInterpolatedUnfrozenSoilMoisture(result);
    }

    public void frz2Sac1(final double inputPrecipitation,
                         final double soilLayerDepths[],
                         final int nUpl,
                         final int nSac,
                         final double soilWiltingPoint,
                         final double rtUz,
                         final double rtLz)
    {

        final DoubleHolder lztwhHolder = new DoubleHolder(0.0);
        final DoubleHolder lzfshHolder = new DoubleHolder(0.0);
        final DoubleHolder lzfphHolder = new DoubleHolder(0.0);
        final DoubleHolder frost2Holder = new DoubleHolder(0.0);
        final DoubleHolder frost1Holder = new DoubleHolder(0.0);
        final DoubleHolder xhHolder = new DoubleHolder(0.0);
        final DoubleHolder uztwhHolder = new DoubleHolder(_sacHTData.getSacSmaHtState().getUztwh());
        final DoubleHolder uzfwhHolder = new DoubleHolder(_sacHTData.getSacSmaHtState().getUzfwh());

        final double xc = 0.;
        int nup;

        fst2Sac1(soilLayerDepths,
                 1,
                 (nUpl - 1),
                 soilWiltingPoint,
                 rtUz,
                 uztwhHolder,
                 uzfwhHolder,
                 xhHolder,
                 _sacHTData.getSacSmaHtState().getUztwc(),
                 _sacHTData.getSacSmaHtState().getUzfwc(),
                 xc,
                 frost1Holder,
                 inputPrecipitation);

        if(frost1Holder.getValue() == 0.0)
        {
            _sacHTData.getSacSmaHtState().setFindex(0.);
        }
        else
        {
            _sacHTData.getSacSmaHtState()
                      .setFindex((-(float)frost1Holder.getValue() / (float)SacSmaHTConstants.FRST_FACT));
        }
        _sacHTData.getSacSmaHtState().setUztwh((float)uztwhHolder.getValue());
        _sacHTData.getSacSmaHtState().setUzfwh((float)uzfwhHolder.getValue());

        //  GET LOWER ZONE STATES
        lztwhHolder.setValue((float)_sacHTData.getSacSmaHtState().getLztwh());
        lzfshHolder.setValue((float)_sacHTData.getSacSmaHtState().getLzfsh());
        lzfphHolder.setValue((float)_sacHTData.getSacSmaHtState().getLzfph());
        nup = nUpl;

        fst2Sac1(soilLayerDepths,
                 nup,
                 (nSac - 1),
                 soilWiltingPoint,
                 rtLz,
                 lztwhHolder,
                 lzfshHolder,
                 lzfphHolder,
                 _sacHTData.getSacSmaHtState().getLztwc(),
                 _sacHTData.getSacSmaHtState().getLzfsc(),
                 _sacHTData.getSacSmaHtState().getLzfpc(),
                 frost2Holder,
                 0.);
        if(frost2Holder.getValue() != 0.0)
        {
            _sacHTData.getSacSmaHtState().setFindex((float)_sacHTData.getSacSmaHtState().getFindex()
                - (float)frost2Holder.getValue() / (float)SacSmaHTConstants.FRST_FACT);
        }

        _sacHTData.getSacSmaHtState().setLztwh((float)lztwhHolder.getValue());
        _sacHTData.getSacSmaHtState().setLzfsh((float)lzfshHolder.getValue());
        _sacHTData.getSacSmaHtState().setLzfph((float)lzfphHolder.getValue());

        return;
    }

    /**
     * subroutine recalculates soil moisture states into sac-sma storages
     */
    public void fst2Sac1(final double soilLayerDepths[],
                         final int nup,
                         final int nlw,
                         final double soilWiltingPoint,
                         final double rt,
                         final DoubleHolder twhVal,
                         final DoubleHolder fwhVal,
                         final DoubleHolder f1wHVal,
                         final double twc,
                         final double fwc,
                         final double f1wc,
                         final DoubleHolder frostVal,
                         final double inputPrecipitation)
    {

        int i;
        double stot, sliq, sx, swh, dz, dH2O, dsh, alp, s;
        double twh, fwh, f1wh, frost;

        twh = twhVal.getValue();
        fwh = fwhVal.getValue();
        f1wh = f1wHVal.getValue();

        frost = 0.;
        stot = 0.;
        sliq = 0.;
        sx = 0.;
        swh = (float)twh + (float)fwh + (float)f1wh + 1000.f * (float)soilWiltingPoint
            * ((float)soilLayerDepths[nup - 1] - (float)soilLayerDepths[nlw]) / (float)rt;

        for(i = nup; i <= nlw; i++)
        {
            dz = (float)soilLayerDepths[i - 1] - (float)soilLayerDepths[i];
            sx = (float)sx + 1000.f * (float)_sacHTData.getSacSmaHtState().getSoilLayerUnfrozenWater(i) * (float)dz
                / (float)rt;
            frost = (float)frost
                + (1000.f
                    * ((float)_sacHTData.getSacSmaHtState().getSoilLayerMoistureContent(i) - (float)_sacHTData.getSacSmaHtState()
                                                                                                              .getSoilLayerUnfrozenWater(i))
                    * (float)dz / (float)rt);
            stot = (float)stot
                + (1000.f
                    * ((float)_sacHTData.getSacSmaHtState().getSoilLayerMoistureContent(i) - (float)soilWiltingPoint)
                    * (float)dz / (float)rt);
            sliq = (float)sliq
                + (1000.f
                    * ((float)_sacHTData.getSacSmaHtState().getSoilLayerUnfrozenWater(i) - (float)soilWiltingPoint)
                    * (float)dz / (float)rt);
        }

        //  DH2O IS LIQUID WATER CHANGE DUE TO FREEZING/THAWING
        dH2O = (float)sx - (float)swh;

        if((frost != 0.) || (twh != twc) || (fwh != fwc) || (f1wh != f1wc))
        {
            if((frost == 0.) && (dH2O == 0.))
            {
                //  no frozen ground, no liquid water change
                twh = (float)twc;
                fwh = (float)fwc;
                f1wh = (float)f1wc;
            }

            else
            {
                //  change liquid water storages
                swh = (float)twh + (float)fwh;
                if(dH2O < 0.)
                {
                    //  unfrozen water reduction (freezing)       
                    if(twh >= 0.)
                    {
                        dsh = (float)swh + (float)dH2O;
                        if(dsh >= 0.)
                        {
                            if(swh > 1E-04)
                            {
                                if((inputPrecipitation > 0.) && (sliq < stot))
                                {
                                    fwh += (float)dH2O;
                                    if(fwh < 0.)
                                    {
                                        twh += (float)fwh;
                                        fwh = 0.;
                                    }
                                }
                                else
                                {
                                    alp = (float)dH2O * (float)twh / (float)swh;
                                    twh += (float)alp;
                                    fwh += ((float)dH2O - (float)alp);
                                }
                            }
                            else
                            {
                                twh += (float)dH2O;
                            }
                        }
                        else
                        {
                            twh = (float)dsh;
                            fwh = 0.;
                        }
                    }
                    else
                    {
                        fwh += (float)dH2O;
                        if(fwh < 0.)
                        {
                            twh += (float)fwh;
                            fwh = 0.;
                        }
                    }
                }
                else
                {
                    //  unfrozen water increase (thawing)                      
                    if(twh < 0.)
                    {
                        twh += (float)dH2O;
                    }
                    else
                    {
                        if(swh > 1E-04)
                        {
                            alp = (float)dH2O * (float)twh / (float)swh;
                            twh += (float)alp;

                            fwh += ((float)dH2O - (float)alp);
                            if(fwh > fwc)
                            {
                                twh += ((float)fwh - (float)fwc);
                                fwh = (float)fwc;
                            }
                        }
                        else
                        {
                            twh += (float)dH2O;
                        }
                    }

                    if(twh > twc)
                    {
                        fwh += ((float)twh - (float)twc);
                        twh = (float)twc;
                    }
                }
            }
        }

        // check consistency of the water balance between sac-sma and
        // frozen ground states 
        s = twc + fwc + f1wc;
        if(Math.abs(stot - s) > 0.5)
        {

            _sacHTData.getLogger().log(Logger.WARNING,
                                       "No total water balance " + nup + " " + stot + " " + twc + " " + fwc + " "
                                           + f1wc);
        }
        s = twh + fwh + f1wh;
        if(Math.abs(sliq - s) > 0.5)
        {
            _sacHTData.getLogger().log(Logger.WARNING,
                                       "No liquid water balance " + nup + " " + sliq + " " + twh + " " + fwh + " "
                                           + f1wh);
        }

        twhVal.setValue((float)twh);
        fwhVal.setValue((float)fwh);
        f1wHVal.setValue((float)f1wh);
        frostVal.setValue((float)frost);
    }

    public double frzInd1(final double[] depthOfSoilLayers, final int numberOfSoilLayers, final double soilWiltingPoint)
    {
        // returns frozen ground depth for case when no layers are frozen but fgindex < 0

        double iceDifferential;
        double dzice;

        for(int i = numberOfSoilLayers - 1; i >= 1; i--)
        {
            iceDifferential = (float)_sacHTData.getSacSmaHtState().getSoilLayerMoistureContent(i)
                - (float)_sacHTData.getSacSmaHtState().getSoilLayerUnfrozenWater(i);

            if(MathHelper.isGreater(iceDifferential, SacSmaHTConstants.SICEMN))
            {

//                System.out.println("iceDifferential "
//                    + iceDifferential
//                    + "  > "
//                    + SacSmaHTConstants.SICEMN
//                    + " "
//                    + (_sacHTData.getSacSmaHtState().getSoilLayerMoistureContent(i) - _sacHTData.getSacSmaHtState()
//                                                                                                .getSoilLayerUnfrozenWater(i)));

                dzice = (float)iceDifferential * ((float)depthOfSoilLayers[i - 1] - (float)depthOfSoilLayers[i])
                // A negative value (-) was introduced to return negatived frozen ground deeps. (100 moved to -100) 
                    / ((float)_sacHTData.getSacSmaHtState().getSoilLayerMoistureContent(i) - (float)soilWiltingPoint);

//                System.out.println("dzice = "
//                    + (dzice - +(iceDifferential * (depthOfSoilLayers[i - 1] - depthOfSoilLayers[i]) / (_sacHTData.getSacSmaHtState()
//                                                                                                                  .getSoilLayerMoistureContent(i) - soilWiltingPoint)))
//                    + " return = "
//                    + ((-100.0f * ((float)dzice - (float)depthOfSoilLayers[i - 1])) - +(-100.0f * ((float)dzice - (float)depthOfSoilLayers[i - 1]))));

                return -100.0f * ((float)dzice - (float)depthOfSoilLayers[i - 1]);
            }
            //CamachoF - This break is wrong, it will not allow to iterate the numberOfSoilLayers and giving us wrong results.
            //break;
        }
        return OHDConstants.MISSING_DATA;
    }

    /**
     * estimation of impermeable area fraction due to frozen ground
     */
    //Note: becase FGPM(5)=-1 in line423 of fland1.f, this has not called
    public static double surFrz1(final double frostIndex, final double iceContent, final double crFrz)
    {
        double impermeableAreaFraction = 0.0;
        double sice, acrt, ialp1, sum, x;
        int k, j, ipow;

        if(iceContent == 0.)
        {
            return impermeableAreaFraction;
        }

        sice = -SacSmaHTConstants.FRST_FACT * frostIndex;

        if(sice > 1.E-2)
        {
            acrt = iceContent * crFrz / sice;
            ialp1 = iceContent + 0.5;
            sum = Math.exp(-acrt);
            k = 1;
            for(j = (int)ialp1; j <= 2; j--)
            {
                ipow = (int)ialp1 - j + 1;
                k *= ipow;
                x = -acrt + ipow * Math.log(acrt) - Math.log(k);
                sum += Math.exp(x);
            }
            impermeableAreaFraction = sum;
        }
        return impermeableAreaFraction;
    }

    public void sac2Frz1(double changeInTensionWaterPerTimeStepInMm,
                         final double changeInFreeWaterPerTimeStepInMm,
                         final int numberOfUpperSoilLayers,
                         final int numberOfLowerSoilLayers,
                         final double[] soilLayerDepthsInM,
                         final double maximumSoilMoisture) throws Exception
    {
        //substitute nUpl-1 for nUpl and nLowl-1 for nLowl when used as an index & in loops
        final Integer[] ms = new Integer[SacSmaHTConstants.MAX_NUMBER_OF_USER_DEFINED_PROFILES];
        //initialize xx1 so loop is entered at least once
        double a, incrementalSoilLayerDepthInM, changeInFreeWaterPerUnitDepth, dwx, delta = 0., depthOfLayerInM, sAvg, alp, ddtx, dMax, dztx, xx1 = 1;
        int m = 0, nn = -1, n;

        /*
         * Split free water between soil layers free water is split equally retween layers
         */
        if(numberOfUpperSoilLayers > 5)
        {
            //the original model stops at this point..
            _sacHTData.getLogger().log(Logger.ERROR, "No. of Upper Soil Layers exceeds 5");

            throw new ModelException("number of Upper Soil Layers is greater than 5");
        }

        for(int i = numberOfUpperSoilLayers - 1; i <= numberOfLowerSoilLayers - 1; i++)
        {
            ms[i] = 1;
        }

        incrementalSoilLayerDepthInM = (float)soilLayerDepthsInM[(numberOfUpperSoilLayers - 1) - 1]
            - (float)soilLayerDepthsInM[numberOfLowerSoilLayers - 1];
        changeInFreeWaterPerUnitDepth = 0.001f * (float)changeInFreeWaterPerTimeStepInMm
            / (float)incrementalSoilLayerDepthInM;

        while(nn != 0)
        {
            //100   	 	
            nn = 0;
            m = m + 1;

            if(m <= (numberOfLowerSoilLayers - numberOfUpperSoilLayers + 1))
            {
                dwx = 0.;

                if(m > 1)
                {
                    incrementalSoilLayerDepthInM = 0.f;
                    for(int i = numberOfUpperSoilLayers - 1; i <= numberOfLowerSoilLayers - 1; i++)
                    {
                        incrementalSoilLayerDepthInM += ((float)soilLayerDepthsInM[i - 1] - (float)soilLayerDepthsInM[i])
                            * (float)ms[i];
                    }
                }

                for(int i = numberOfUpperSoilLayers - 1; i <= numberOfLowerSoilLayers - 1; i++)
                {
                    if(ms[i] != 0)
                    {
                        _sacHTData.getSacSmaHtState()
                                  .setSoilLayerMoistureContent(i,
                                                               (float)_sacHTData.getSacSmaHtState()
                                                                                .getSoilLayerMoistureContent(i)
                                                                   + (float)changeInFreeWaterPerUnitDepth);
                        _sacHTData.getSacSmaHtState()
                                  .setSoilLayerUnfrozenWater(i,
                                                             (float)_sacHTData.getSacSmaHtState()
                                                                              .getSoilLayerUnfrozenWater(i)
                                                                 + (float)changeInFreeWaterPerUnitDepth);
                        delta = (float)_sacHTData.getSacSmaHtState().getSoilLayerMoistureContent(i)
                            - (float)maximumSoilMoisture;
                        //if((delta > 0.0001f) || (_sacHTData.getSacSmaHtState().getSoilLayerMoistureContent(i) < 0.))
                        if(MathHelper.isGreater(delta, 0.0001)
                            || (_sacHTData.getSacSmaHtState().getSoilLayerMoistureContent(i) < 0.))
                        {
                            nn = i;
                            ms[i] = 0;
                            depthOfLayerInM = (float)soilLayerDepthsInM[i - 1] - (float)soilLayerDepthsInM[i];
                            incrementalSoilLayerDepthInM = (float)incrementalSoilLayerDepthInM - (float)depthOfLayerInM;
                            if(_sacHTData.getSacSmaHtState().getSoilLayerMoistureContent(i) >= 0.)
                            {
                                dwx += ((float)delta * (float)depthOfLayerInM * 1000.f);
                                _sacHTData.getSacSmaHtState()
                                          .setSoilLayerUnfrozenWater(i,
                                                                     _sacHTData.getSacSmaHtState()
                                                                               .getSoilLayerUnfrozenWater(i)
                                                                         - delta);
                                _sacHTData.getSacSmaHtState().setSoilLayerMoistureContent(i, maximumSoilMoisture);
                            }
                            else
                            {
                                dwx += ((float)_sacHTData.getSacSmaHtState().getSoilLayerMoistureContent(i)
                                    * (float)depthOfLayerInM * 1000.f);
                                _sacHTData.getSacSmaHtState().setSoilLayerMoistureContent(i, 0.0);
                                _sacHTData.getSacSmaHtState().setSoilLayerUnfrozenWater(i, 0.0);
                            }
                        }
                    }
                }

                if(nn != 0)
                {
                    changeInFreeWaterPerUnitDepth = 0.001f * (float)dwx / (float)incrementalSoilLayerDepthInM;
                    //goto 100 - replaced by while
                }
            }
            else
            {
                final String logMsg = "WARN: NO BALANCE IN SAC2FRZ " + "M= " + m + " NUPL= " + numberOfUpperSoilLayers
                    + " NLOWL=" + numberOfLowerSoilLayers;
                _sacHTData.getLogger().log(Logger.WARNING, logMsg);
                // end while loop
                break;
            }
        } // end while

        /*
         * Split tension water: if tension water reduction, split by a ratio of unfrozen water from previous time step;
         * if tension water increase, split by an inverse ratio of total water deficit. calculate a ratio of unfrozen
         * water or ratio of total water deficit
         */

        n = 1;
        //77    
        while(n < 2)
        {
            sAvg = 0.;

            for(int i = numberOfUpperSoilLayers - 1; i <= numberOfLowerSoilLayers - 1; i++)
            {
                //if(changeInTensionWaterPerTimeStepInMm < 0.)
                if(MathHelper.isLessThan(changeInTensionWaterPerTimeStepInMm, 0.))
                {
                    sAvg += (float)_sacHTData.getSacSmaHtState().getSoilLayerUnfrozenWater(i);
                }
                else
                {
                    sAvg += ((float)maximumSoilMoisture - (float)_sacHTData.getSacSmaHtState()
                                                                           .getSoilLayerMoistureContent(i));
                }
            }

            sAvg /= ((float)numberOfLowerSoilLayers - (float)numberOfUpperSoilLayers + 1);
            //if(sAvg < .00001f)
            if(MathHelper.isLessThan(sAvg, .00001))
            {
                //GOTO 7
                return;
            }
            else
            {
                alp = 0.;

                for(int i = numberOfUpperSoilLayers - 1; i <= numberOfLowerSoilLayers - 1; i++)
                {
                    depthOfLayerInM = (float)soilLayerDepthsInM[i - 1] - (float)soilLayerDepthsInM[i];
                    //if(changeInTensionWaterPerTimeStepInMm < 0.)
                    if(MathHelper.isLessThan(changeInTensionWaterPerTimeStepInMm, 0.))
                    {
                        //unfrozen water ratio
                        alp += ((float)depthOfLayerInM
                            * (float)_sacHTData.getSacSmaHtState().getSoilLayerUnfrozenWater(i) / (float)sAvg);
                    }
                    else
                    {
                        //total water deficit ratio
                        alp += ((float)depthOfLayerInM
                            * ((float)maximumSoilMoisture - (float)_sacHTData.getSacSmaHtState()
                                                                             .getSoilLayerMoistureContent(i)) / (float)sAvg);
                    }
                }

                alp = 1.f / (float)alp;

                //run redistribution of water between soil layers
                ddtx = 0.;

                for(int i = numberOfUpperSoilLayers - 1; i <= numberOfLowerSoilLayers - 1; i++)
                {
                    depthOfLayerInM = (float)soilLayerDepthsInM[i - 1] - (float)soilLayerDepthsInM[i];
                    //if(changeInTensionWaterPerTimeStepInMm < 0.)
                    if(MathHelper.isLessThan(changeInTensionWaterPerTimeStepInMm, 0.))
                    {
                        //reduction in tension water. use a ratio of unfrozen water
                        dMax = 1000.f * (float)_sacHTData.getSacSmaHtState().getSoilLayerUnfrozenWater(i)
                            * (float)depthOfLayerInM;
                        dztx = (float)depthOfLayerInM
                            * ((float)changeInTensionWaterPerTimeStepInMm
                                * ((float)_sacHTData.getSacSmaHtState().getSoilLayerUnfrozenWater(i) / (float)sAvg)
                                * (float)alp + (float)ddtx
                                / ((float)soilLayerDepthsInM[i - 1] - (float)soilLayerDepthsInM[numberOfLowerSoilLayers - 1]));

                        xx1 = Math.abs(dztx);
                        //if(xx1 > dMax)
                        if(MathHelper.isGreater(xx1, dMax))
                        {
                            ddtx = (float)dztx + (float)dMax;
                            dztx = 0 - (float)dMax;
                        }
                    }
                    else
                    {
                        //increase in tension water. use an inverse ratio of unfrozen water
                        dMax = 1000.f
                            * ((float)maximumSoilMoisture - (float)_sacHTData.getSacSmaHtState()
                                                                             .getSoilLayerMoistureContent(i))
                            * (float)depthOfLayerInM;
                        dztx = (float)depthOfLayerInM
                            * ((float)changeInTensionWaterPerTimeStepInMm
                                * (((float)maximumSoilMoisture - (float)_sacHTData.getSacSmaHtState()
                                                                                  .getSoilLayerMoistureContent(i)) / (float)sAvg)
                                * (float)alp + (float)ddtx
                                / ((float)soilLayerDepthsInM[i - 1] - (float)soilLayerDepthsInM[numberOfLowerSoilLayers - 1]));
                        //if(dztx > dMax)
                        if(MathHelper.isGreater(dztx, dMax))
                        {
                            ddtx = (float)dztx - (float)dMax;
                            dztx = (float)dMax;
                        }
                    }

                    a = 0.001f * (float)dztx / (float)depthOfLayerInM;
                    _sacHTData.getSacSmaHtState()
                              .setSoilLayerUnfrozenWater(i,
                                                         (float)_sacHTData.getSacSmaHtState()
                                                                          .getSoilLayerUnfrozenWater(i)
                                                             + (float)a);
                    _sacHTData.getSacSmaHtState()
                              .setSoilLayerMoistureContent(i,
                                                           (float)_sacHTData.getSacSmaHtState()
                                                                            .getSoilLayerMoistureContent(i)
                                                               + (float)a);
                    if(_sacHTData.getSacSmaHtState().getSoilLayerUnfrozenWater(i) < 0.)
                    {
                        _sacHTData.getSacSmaHtState().setSoilLayerUnfrozenWater(i, 0.0);
                    }
                    if(_sacHTData.getSacSmaHtState().getSoilLayerMoistureContent(i) < 0.)
                    {
                        _sacHTData.getSacSmaHtState().setSoilLayerMoistureContent(i, 0.0);
                    }
                }

                xx1 = Math.abs(ddtx);
                //if(xx1 > .01f)
                if(MathHelper.isGreater(xx1, .01))
                {
                    changeInTensionWaterPerTimeStepInMm = ddtx;
                    n += 1;
                    if(n > 2)
                    {
                        _sacHTData.getLogger().log(Logger.WARNING, "WARN: NO BALANCE IN SAC-SMA WATER CHANGE. ");
                    }
                    //else goto 77
                }
                else
                {
                    return;
                }
            }
        }
    }
}
