package ohd.hseb.ohdmodels.sacsmaHT;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import ohd.hseb.db.DbTimeHelper;
import ohd.hseb.measurement.Measurement;
import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.model.MonthlyValues;
import ohd.hseb.ohdmodels.sacsma.SacSmaModelConstants;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.MathHelper;
import ohd.hseb.util.fews.DataType;
import ohd.hseb.util.fews.ModelException;
import ohd.hseb.util.fews.NwsrfsDataTypeMappingReader;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.OHDUtilities;
import ohd.hseb.util.fews.Parameters;
import ohd.hseb.util.fews.ohdmodels.ModelDriver;
import ohd.hseb.util.fews.ohdmodels.ModelParameters;
import ohd.hseb.util.fews.ohdmodels.ModelState;

/**
 * @author FewsPilot Team
 */
final public class SacSmaHTModelDriver extends ModelDriver
{

    private final SacSmaHTData _sacHTData;
    private FinFg0 _finFg0; // just driver

    //input TSs:
    private RegularTimeSeries _snsgTs;
    private RegularTimeSeries _matTs;
    private RegularTimeSeries _sascTs;
    private RegularTimeSeries _sweTs;
    private RegularTimeSeries _mapeTs;

    private List<RegularTimeSeries> _sstTsList = null;
    private List<RegularTimeSeries> _ssmTsList = null;

    private final List<RegularTimeSeries> _modsTsList;

    public SacSmaHTModelDriver()
    {
        super(); // calling ModelDriver constructor

        _sacHTData = new SacSmaHTData();
        _finFg0 = new FinFg0();

        _state = new SacSmaHTModelState();
        _parameters = new SacSmaHTModelParameters();
        _savedParameters = new SacSmaHTModelParameters();
        _sacHTData.setSacSmaHtParameters((SacSmaHTModelParameters)_parameters);
        _sacHTData.setSacSmaHtSavedParameters((SacSmaHTModelParameters)_savedParameters);
        _sacHTData.setSacSmaHtState((SacSmaHTModelState)_state);

        _modsTsList = new ArrayList<RegularTimeSeries>();
    }

    /**
     * validates the parameters, state and all the time series needed for the model exist.
     */
    @Override
    protected void runModelDriverValidation() throws Exception
    {
        //validate state
        if(super.needCarryoverTransfer())
        {
            _sacHTData.getSacSmaHtState().validateState(_savedParameters); //use _savedParameters for carryover transfer

            if(!_sacHTData.getSacSmaHtState().isColdStart())
            {
                _sacHTData.getSacSmaHtState().setNumberOfSoilLayers(0); //re-set it to 0 so later calculation will be trigged
            }
        }
        else
        {
            _sacHTData.getSacSmaHtState().validateState((Parameters)_parameters); //use _parameters
        }

        //extract individual input TS from _tsList.
        final Iterator<RegularTimeSeries> ite = getTsList().iterator();

        RegularTimeSeries inputRts;

        String tsType;

        while(ite.hasNext())
        {
            inputRts = ite.next();

            tsType = inputRts.getTimeSeriesType();

            if(tsType.equals(DataType.RAIM_DATATYPE)) //mandatory input TS
            {
                _drivingTs = inputRts;
            }
            else if(tsType.equals(DataType.POTENTIAL_ET_DATATYPE))
            {//optional input TS

                if(_sacHTData.getSacSmaHtParameters().useMAPEInput())
                {
                    _mapeTs = inputRts;

                    _logger.log(Logger.DEBUG, "Using optional input MAPE time series.");

                    /*
                     * NWSRFS requires the interval be 24 hours, but for FEWS, no need for this requirement; but must be
                     * multiple of driving TS(RAIM) interval. Will be validated when disintegrated to RAIM interval
                     * below.
                     */
                }
                else
                {//since not using it, delete it from _tsList
                    ite.remove();

                    _logger.log(Logger.DEBUG, "Not use optional input MAPE time series.");
                }
            }
            else if(tsType.equals(DataType.SASC_DATATYPE))
            {
                _sascTs = inputRts;
            }
            else if(tsType.equals(DataType.SNSG_DATATYPE))
            {
                _snsgTs = inputRts;
            }
            else if(tsType.equals(DataType.MAT_DATATYPE))
            {
                _matTs = inputRts;
            }
            else if(tsType.equals(DataType.SWE_DATATYPE))
            {
                _sweTs = inputRts;
            }
            /*
             * FEWS will always provide input MODS time series xml file regardless of having MODS or not: when not using
             * a MOD, that MOD time series still have a header in the xml file. It is parsed and stored in
             * getInputModsList() with all values be -999.0. To let model computation not read these empty time series,
             * they are removed now.
             */
            else if(inputRts.isEmpty())//if passed all "if" checks above, must be MOD ts
            {
                _logger.log(Logger.DEBUG, "Input " + tsType
                    + " MOD time series is empty. Removed it now so not using it anymore.");

                ite.remove(); //delete the RTS from the list
            }
            else
            {
                _modsTsList.add(inputRts.clone());

                ite.remove(); //delete the MOD RTS from the list
            }

        } // close while loop

        // validate required time series: RAIM
        if(_drivingTs == null)
        {
            throw new ModelException("Input time series does not have " + DataType.RAIM_DATATYPE);
        }

        if(_mapeTs != null && (_mapeTs.getIntervalInHours() != getDrivingTsInterval()))
        {//if _mapeTs interval is greater than _raim(but must be even multiple of it), convert _mapeTs to that smaller interval

            _mapeTs.disIntegrateToIntervalAccum(getDrivingTsInterval());
        }

        // validate that all the ts' interval equal to RAIM(drivingTs) interval
        for(int i = 0; i < _tsList.size(); i++)
        {
            if(getDrivingTsInterval() != _tsList.get(i).getIntervalInHours())
            {
                throw new ModelException("Inputs time series " + _tsList.get(i).getTimeSeriesType()
                    + " does not have same interval time as RAIM ts.");
            }

        }

        // RAIM or MAP are the only ones necessary if it's not the full frozen ground algorithm
        if(!_sacHTData.getSacSmaHtParameters().getModelRunType().equals(SacSmaHTConstants.ONLY_SOIL_RUN_STR))
        {
            if(_matTs == null)
            {
                throw new ModelException("Input time series does not have " + DataType.MAT_DATATYPE);
            }
            if(_snsgTs == null)
            {
                throw new ModelException("Input time series does not have " + DataType.SNSG_DATATYPE);
            }
            if(_sweTs == null)
            {
                throw new ModelException("Input time series does not have " + DataType.SWE_DATATYPE);
            }
            if(_sascTs == null)
            {
                throw new ModelException("Input time series does not have " + DataType.SASC_DATATYPE);
            }
        }

        super.runModelDriverValidation();

        if(_modsTsList.size() > 0)
        {
            super.checkInputTsInSync(_modsTsList);
        }

    } // close runModelDriverValidation()

    /**
     * executes the model. workDirName and outputDirName are directory names for input and output, which are only needed
     * for running legacy Fortran mode. Bubble up exceptions, if any
     */
    @Override
    public void execute() throws Exception
    {

        _logger.log(Logger.DEBUG, "Enter execute()");

        this.runModelDriverValidation();

        _resultMap.put(DataType.CHANNEL_INFLOW_DATATYPE, null);
        _resultMap.put(DataType.FROZEN_DEPTH_DATATYPE, null);
        _resultMap.put(DataType.FGIX_DATATYPE, null);

        calculateRunoff(_resultMap);

    } // close execute()

    private void prepareData() throws Exception
    {
        _logger.log(Logger.DEBUG, "Enter prepareData()");

        final double supm = _sacHTData.getSacSmaHtParameters().getUztwm()
            + _sacHTData.getSacSmaHtParameters().getUzfwm();
        final double slwm = _sacHTData.getSacSmaHtParameters().getLztwm()
            + _sacHTData.getSacSmaHtParameters().getLzfsm() + _sacHTData.getSacSmaHtParameters().getLzfpm();

        frdFg1(getDrivingTsInterval(), supm, slwm);

        if(super.needCarryoverTransfer())
        {
            //do carryover transfer                        
            _sacHTData.getSacSmaHtState().doCarryOverTransfer(_savedParameters, (ModelParameters)_parameters);

            // wipe out soil moisture states.
            estFst1();
        }

        _logger.log(Logger.DEBUG, "Exit prepareData()");
    }

    private void calculateRunoff(final Map<String, RegularTimeSeries> resultMap) throws Exception
    {
        // stores latest values of temp, swe and sh - hardcode ta to 10.0 to be
        // above freezing
        // for the soil moisture only case - if this is not the case, this will
        // be overridden by
        // what's read from the file..
        double ta = 10., we = 0.;
        double sh = 0.;
        double aesc = 0.;
        String requestedOutputDataType;

        prepareData();

        final double pxadj = _sacHTData.getSacSmaHtParameters().getPxAdj(); // adj[1]
        final double peadj = _sacHTData.getSacSmaHtParameters().getPeAdj(); // adj[2]
        final double timeStep = getDrivingTsInterval() / 24.;
        // Define the maximum number of Soil Layer, this value will change if
        // user defined soil layer depth is defined.
        int maxNumberOfSoilLayers = _sacHTData.getSacSmaHtState().getNumberOfSoilLayers();

        final Calendar startingTime = new GregorianCalendar(OHDConstants.GMT_TIMEZONE);
        startingTime.setTimeInMillis(getComputationStartTime());

        Iterator<String> setIter = resultMap.keySet().iterator();
        RegularTimeSeries resultantTimeSeries;

        while(setIter.hasNext())
        {
            requestedOutputDataType = setIter.next();
            /*
             * resultantTimeSeries = new RegularTimeSeries(getDrivingTsStartTime(), getDrivingTsEndTime(),
             * getDrivingTsInterval(),
             * OHDModelUtilities._initializeNwsrfsDataTypeToMeasuringUnitMap().get(requestedOutputDataType));
             */

            String rfsUnitString = null;
            rfsUnitString = NwsrfsDataTypeMappingReader.getNwsrfsUnit(requestedOutputDataType, _logger);
            resultantTimeSeries = new RegularTimeSeries(getComputationStartTime(),
                                                        getComputationEndTime(),
                                                        getDrivingTsInterval(),
                                                        MeasuringUnit.getMeasuringUnit(rfsUnitString));

            resultantTimeSeries.setLocationId(_outputLocationId);
            resultantTimeSeries.setTimeSeriesType(requestedOutputDataType);
            resultMap.put(requestedOutputDataType, resultantTimeSeries);
        }

        /**
         * ---------------------------initialize 6 state variable TS ---------------------
         */
        final RegularTimeSeries uztwcTS = new RegularTimeSeries(getComputationStartTime(),
                                                                getComputationEndTime(),
                                                                getDrivingTsInterval(),
                                                                MeasuringUnit.mm);
        uztwcTS.setLocationId(_outputLocationId);
        uztwcTS.setTimeSeriesType(OHDConstants.SAC_STATE_UZTWC);

        final RegularTimeSeries uzfwcTS = new RegularTimeSeries(getComputationStartTime(),
                                                                getComputationEndTime(),
                                                                getDrivingTsInterval(),
                                                                MeasuringUnit.mm);
        uzfwcTS.setLocationId(_outputLocationId);
        uzfwcTS.setTimeSeriesType(OHDConstants.SAC_STATE_UZFWC);

        final RegularTimeSeries lztwcTS = new RegularTimeSeries(getComputationStartTime(),
                                                                getComputationEndTime(),
                                                                getDrivingTsInterval(),
                                                                MeasuringUnit.mm);
        lztwcTS.setLocationId(_outputLocationId);
        lztwcTS.setTimeSeriesType(OHDConstants.SAC_STATE_LZTWC);

        final RegularTimeSeries lzfscTS = new RegularTimeSeries(getComputationStartTime(),
                                                                getComputationEndTime(),
                                                                getDrivingTsInterval(),
                                                                MeasuringUnit.mm);
        lzfscTS.setLocationId(_outputLocationId);
        lzfscTS.setTimeSeriesType(OHDConstants.SAC_STATE_LZFSC);

        final RegularTimeSeries lzfpcTS = new RegularTimeSeries(getComputationStartTime(),
                                                                getComputationEndTime(),
                                                                getDrivingTsInterval(),
                                                                MeasuringUnit.mm);
        lzfpcTS.setLocationId(_outputLocationId);
        lzfpcTS.setTimeSeriesType(OHDConstants.SAC_STATE_LZFPC);

        final RegularTimeSeries adimcTS = new RegularTimeSeries(getComputationStartTime(),
                                                                getComputationEndTime(),
                                                                getDrivingTsInterval(),
                                                                MeasuringUnit.mm);
        adimcTS.setLocationId(_outputLocationId);
        adimcTS.setTimeSeriesType(OHDConstants.SAC_STATE_ADIMC);

        _ssmTsList = new ArrayList<RegularTimeSeries>();
        _sstTsList = new ArrayList<RegularTimeSeries>();

        if(_sacHTData.getSacSmaHtParameters().getUserDefinedSoilLayerDepths() != null
            && _sacHTData.getSacSmaHtParameters().getUserDefinedSoilLayerDepths().length > 0)
        {
            maxNumberOfSoilLayers = _sacHTData.getSacSmaHtParameters().getUserDefinedSoilLayerDepths().length;
        }

        for(int numberOfSoilLayers = 0; numberOfSoilLayers < maxNumberOfSoilLayers; numberOfSoilLayers++)
        {

            final RegularTimeSeries ssmTS = new RegularTimeSeries(getComputationStartTime(),
                                                                  getComputationEndTime(),
                                                                  getDrivingTsInterval(),
                                                                  MeasuringUnit.unitlessReal);
            ssmTS.setLocationId(_outputLocationId);
            ssmTS.setTimeSeriesType(SacSmaHTConstants.SOIL_LAYER_MOISTURE_CONTENT_TS_TAG + numberOfSoilLayers);

            _ssmTsList.add(ssmTS);

            final RegularTimeSeries sstTS = new RegularTimeSeries(getComputationStartTime(),
                                                                  getComputationEndTime(),
                                                                  getDrivingTsInterval(),
                                                                  MeasuringUnit.degreesCelsius);
            sstTS.setLocationId(_outputLocationId);
            sstTS.setTimeSeriesType(SacSmaHTConstants.SOIL_LAYER_TEMPERATURE_TS_TAG + numberOfSoilLayers);
            _sstTsList.add(sstTS);
        }

        final SacSmaHTModel sacSmaHTModel = new SacSmaHTModel(_sacHTData);

        this.applySacSmaMods(super.getInitialStateTime()); //apply mods

        final MonthlyValues monthlyValues = new MonthlyValues(_sacHTData.getSacSmaHtParameters().getEtDemand());

        final double intervalPortionOfDay = getDrivingTsInterval() * 1.0 / OHDConstants.HOURS_PER_DAY;

        for(int i = 0; i < _drivingTs.getMeasurementCount(); i++)
        {
            /*
             * To account for the fact that the intervals of the input time series may be different than that of RAIM,
             * adjust the others
             */
            final long curDateTime = _drivingTs.getMeasurementTimeByIndex(i);

            final double pxv = _drivingTs.getMeasurementByIndex(i).getValue(MeasuringUnit.mm) * pxadj; // model expect unit:mm

            if(!_sacHTData.getSacSmaHtParameters().getModelRunType().equals(SacSmaHTConstants.ONLY_SOIL_RUN_STR))
            {

                ta = _matTs.getMeasurementByTime(curDateTime).getValue(MeasuringUnit.degreesCelsius); // model expect oC
                we = _sweTs.getMeasurementByTime(curDateTime).getValue(MeasuringUnit.mm); // model expect unit:mm
                sh = _snsgTs.getMeasurementByTime(curDateTime).getValue(MeasuringUnit.cm); // model expect unit:mm
                aesc = _sascTs.getMeasurementByTime(curDateTime).getValue(); // unitless
            }

            // update SacSmaHTModelState 3 states' values
            _sacHTData.getSacSmaHtState().setInitAirTemperature(ta);
            _sacHTData.getSacSmaHtState().setInitSnowDepth(sh);
            _sacHTData.getSacSmaHtState().setInitSnowWaterEquivalent(we);

            // set SacSmaHTModelState time
            _sacHTData.getSacSmaHtState().setDateTime(getComputationStartTime() + i * getDrivingTsIntervalInMillis());

            // apply mods
            this.applySacSmaMods(curDateTime);

            // The lines are for generating input state TS xml file(using the
            // same mechanism of generating output state TS xml):
            // Normal precison is 3 digits after decimal point, to get longer
            // precision, edit writeTimeSeriesToFewsXML()
            // SacSmaHTModelState accumulatedState =
            // (SacSmaHTModelState)(_sacHTData.getSacSmaHtState().clone());
            // super._listOfOutputStates.add(accumulatedState);

            /* This is actually the object created in SacSmaHTModel, not a copy */
            startingTime.add(Calendar.HOUR, i);

            double etDemand;
            if(_mapeTs == null)
            {
                etDemand = monthlyValues.getValue(curDateTime) * intervalPortionOfDay * peadj;
            }
            else
            {
                etDemand = _mapeTs.getMeasurementValueByTime(curDateTime, MeasuringUnit.mm)
                    * monthlyValues.getValue(curDateTime) * peadj;

            }

            //if using the optional input SASC TS, further adjust evaporationAmount(inside fLand1(..))
            if(_sascTs != null)
            {
                aesc = _sascTs.getMeasurementValueByTime(curDateTime, MeasuringUnit.percentDecimal);
            }

            /* -----------------------running model calculation------------------- */
            final SacSmaHTResults currentResult = sacSmaHTModel.fLand1(timeStep, pxv, ta, we, aesc, sh, etDemand, i);

            // re-set iterator to the beginning
            setIter = resultMap.keySet().iterator();
            while(setIter.hasNext())
            {
                requestedOutputDataType = setIter.next();
                final MeasuringUnit tsUnit = resultMap.get(requestedOutputDataType).getMeasuringUnit();
                resultMap.get(requestedOutputDataType)
                         .setMeasurementByIndex(new Measurement(currentResult.get(requestedOutputDataType),
                                                                tsUnit,
                                                                true),
                                                i);
            }

            // save the state into TS
            uztwcTS.setMeasurementByIndex(_sacHTData.getSacSmaHtState().getUztwc(), i);
            uzfwcTS.setMeasurementByIndex(_sacHTData.getSacSmaHtState().getUzfwc(), i);
            lztwcTS.setMeasurementByIndex(_sacHTData.getSacSmaHtState().getLztwc(), i);
            lzfscTS.setMeasurementByIndex(_sacHTData.getSacSmaHtState().getLzfsc(), i);
            lzfpcTS.setMeasurementByIndex(_sacHTData.getSacSmaHtState().getLzfpc(), i);
            adimcTS.setMeasurementByIndex(_sacHTData.getSacSmaHtState().getAdimc(), i);

            for(int numberOfSoilLayers = 0; numberOfSoilLayers < maxNumberOfSoilLayers; numberOfSoilLayers++)
            {

                if(_sacHTData.getSacSmaHtParameters().getUserDefinedSoilLayerDepths() != null
                    && _sacHTData.getSacSmaHtParameters().getUserDefinedSoilLayerDepths().length > 0)
                {
                    _ssmTsList.get(numberOfSoilLayers)
                              .setMeasurementByIndex(_sacHTData.getInterpolatedTotalSoilMoisture()[numberOfSoilLayers],
                                                     i);
                    if(!_sacHTData.getSacSmaHtParameters()
                                  .getModelRunType()
                                  .equals(SacSmaHTConstants.ONLY_SOIL_RUN_STR))
                    {
                        _sstTsList.get(numberOfSoilLayers)
                                  .setMeasurementByIndex(_sacHTData.getInterpolatedSoilTemperatures()[numberOfSoilLayers],
                                                         i);
                    }
                }
                else
                {
                    _ssmTsList.get(numberOfSoilLayers)
                              .setMeasurementByIndex(_sacHTData.getSacSmaHtState()
                                                               .getSoilLayerMoistureContent(numberOfSoilLayers),
                                                     i);
                    _sstTsList.get(numberOfSoilLayers)
                              .setMeasurementByIndex(_sacHTData.getSacSmaHtState()
                                                               .getSoilLayerTemperature(numberOfSoilLayers),
                                                     i);
                }
            }

        } // close for loop

        // copy the results so that you can put them back with the new keytype
        // we need, the above was kept the same
        // so this didn't affect the internal sacHT calculations
        final Map<String, RegularTimeSeries> tempResultMap = new HashMap<String, RegularTimeSeries>();
        setIter = resultMap.keySet().iterator();
        RegularTimeSeries tempResultTS;
        tempResultTS = resultMap.get("FRZD");
        while(setIter.hasNext())
        {
            requestedOutputDataType = setIter.next();
            tempResultTS = resultMap.get(requestedOutputDataType);
            // we don't need to store SFGD time series if model run type is
            // "only soil". as values are not calculated.
            if(!(_sacHTData.getSacSmaHtParameters().getModelRunType().equals(SacSmaHTConstants.ONLY_SOIL_RUN_STR) && tempResultTS.getTimeSeriesType()
                                                                                                                                 .equals(DataType.FROZEN_DEPTH_DATATYPE)))
            {
                tempResultMap.put(tempResultTS.getLocationId() + tempResultTS.getTimeSeriesType()
                                      + tempResultTS.getIntervalInHours(),
                                  tempResultTS);
            }
        }
        resultMap.clear();
        setIter = tempResultMap.keySet().iterator();
        while(setIter.hasNext())
        {
            final String tempTSKey = setIter.next();
            resultMap.put(tempTSKey, tempResultMap.get(tempTSKey));
        }

        _logger.log(Logger.DEBUG, "Finished iterating over time");

        final long stateTime = _drivingTs.getMeasurementTimeByIndex(_drivingTs.getMeasurementCount() - 1);
        _sacHTData.getSacSmaHtState().setDateTime(stateTime);
        _sacHTData.getSacSmaHtState().setName("No Name" + stateTime);
        _sacHTData.getSacSmaHtState().setId("No Name" + stateTime);

        /**
         * --------------------Add all the 6 state variable TS into the mods map ---------------------
         */
        resultMap.put(OHDUtilities.getCompositeKeyInResultMap(uztwcTS), uztwcTS);
        resultMap.put(OHDUtilities.getCompositeKeyInResultMap(uzfwcTS), uzfwcTS);
        resultMap.put(OHDUtilities.getCompositeKeyInResultMap(lztwcTS), lztwcTS);
        resultMap.put(OHDUtilities.getCompositeKeyInResultMap(lzfscTS), lzfscTS);
        resultMap.put(OHDUtilities.getCompositeKeyInResultMap(lzfpcTS), lzfpcTS);
        resultMap.put(OHDUtilities.getCompositeKeyInResultMap(adimcTS), adimcTS);

        /* ----------------- Add 6 state percentage TSs----------------------- */
        final RegularTimeSeries stateUztwcPerc = uztwcTS.clone();
        stateUztwcPerc.scaleRegularTimeSeries(1.0 / ((SacSmaHTModelParameters)_parameters).getUztwm());
        stateUztwcPerc.setMeasuringUnitNoChangeData(MeasuringUnit.percentDecimal);
        stateUztwcPerc.setTimeSeriesType(SacSmaModelConstants.UZTWC_PERC);
        addTStoResultMap(stateUztwcPerc);

        final RegularTimeSeries stateUzfwcPerc = uzfwcTS.clone();
        stateUzfwcPerc.scaleRegularTimeSeries(1.0 / ((SacSmaHTModelParameters)_parameters).getUzfwm());
        stateUzfwcPerc.setMeasuringUnitNoChangeData(MeasuringUnit.percentDecimal);
        stateUzfwcPerc.setTimeSeriesType(SacSmaModelConstants.UZFWC_PERC);
        addTStoResultMap(stateUzfwcPerc);

        final RegularTimeSeries stateLztwcPerc = lztwcTS.clone();
        stateLztwcPerc.scaleRegularTimeSeries(1.0 / ((SacSmaHTModelParameters)_parameters).getLztwm());
        stateLztwcPerc.setMeasuringUnitNoChangeData(MeasuringUnit.percentDecimal);
        stateLztwcPerc.setTimeSeriesType(SacSmaModelConstants.LZTWC_PERC);
        addTStoResultMap(stateLztwcPerc);

        final RegularTimeSeries stateLzfscPerc = lzfscTS.clone();
        stateLzfscPerc.scaleRegularTimeSeries(1.0 / ((SacSmaHTModelParameters)_parameters).getLzfsm());
        stateLzfscPerc.setMeasuringUnitNoChangeData(MeasuringUnit.percentDecimal);
        stateLzfscPerc.setTimeSeriesType(SacSmaModelConstants.LZFSC_PERC);
        addTStoResultMap(stateLzfscPerc);

        final RegularTimeSeries stateLzfpcPerc = lzfpcTS.clone();
        stateLzfpcPerc.scaleRegularTimeSeries(1.0 / ((SacSmaHTModelParameters)_parameters).getLzfpm());
        stateLzfpcPerc.setMeasuringUnitNoChangeData(MeasuringUnit.percentDecimal);
        stateLzfpcPerc.setTimeSeriesType(SacSmaModelConstants.LZFPC_PERC);
        addTStoResultMap(stateLzfpcPerc);

        final RegularTimeSeries stateAdimcPerc = adimcTS.clone();
        stateAdimcPerc.scaleRegularTimeSeries(1.0 / (((SacSmaHTModelParameters)_parameters).getUztwm() + ((SacSmaHTModelParameters)_parameters).getLztwm()));
        stateAdimcPerc.setMeasuringUnitNoChangeData(MeasuringUnit.percentDecimal);
        stateAdimcPerc.setTimeSeriesType(SacSmaModelConstants.ADIMC_PERC);
        addTStoResultMap(stateAdimcPerc);

        for(int i = 0; i < _sstTsList.size(); i++)
        {
            if(!_sacHTData.getSacSmaHtParameters().getModelRunType().equals(SacSmaHTConstants.ONLY_SOIL_RUN_STR))
            {
                resultMap.put(OHDUtilities.getCompositeKeyInResultMap(_sstTsList.get(i)), _sstTsList.get(i));
            }

            resultMap.put(OHDUtilities.getCompositeKeyInResultMap(_ssmTsList.get(i)), _ssmTsList.get(i));
        }

        _logger.log(Logger.DEBUG, "Exit calculateRunoff()");
    }

    private void fstFg1()
    {
        /*
         * CVK ESTIMATE INITIAL SAC-SMA UNFROZEN WATER STORAGES CVK IF THEY ARE NOT INPUTED
         */
        if(_sacHTData.getSacSmaHtState().isColdStart())
        {
            estFst1();
        }

        // CVK CHECKING CONSISTENCY OF UNFROZEN WATER STATES
        int l = 0;
        double sfi = 0., xx1, xx2;

        if(_sacHTData.getSacSmaHtState().getUztwh() > _sacHTData.getSacSmaHtState().getUztwc())
        // if(MathHelper.isGreater(_sacHTData.getSacSmaHtState().getUztwh(),
        // _sacHTData.getSacSmaHtState().getUztwc()))
        {
            _sacHTData.getSacSmaHtState().setUztwh(_sacHTData.getSacSmaHtState().getUztwc());
            l = 1;
        }
        sfi = sfi + _sacHTData.getSacSmaHtState().getUztwc() - _sacHTData.getSacSmaHtState().getUztwh();
        if(_sacHTData.getSacSmaHtState().getUzfwh() > _sacHTData.getSacSmaHtState().getUzfwc())
        // if(MathHelper.isGreater(_sacHTData.getSacSmaHtState().getUzfwh(),
        // _sacHTData.getSacSmaHtState().getUzfwc()))
        {
            _sacHTData.getSacSmaHtState().setUzfwh(_sacHTData.getSacSmaHtState().getUzfwc());
            l = 1;
        }
        sfi = sfi + _sacHTData.getSacSmaHtState().getUzfwc() - _sacHTData.getSacSmaHtState().getUzfwh();

        if(_sacHTData.getSacSmaHtState().getLztwh() > _sacHTData.getSacSmaHtState().getLztwc())
        // if(MathHelper.isGreater(_sacHTData.getSacSmaHtState().getLztwh(),
        // _sacHTData.getSacSmaHtState().getLztwc()))
        {
            _sacHTData.getSacSmaHtState().setLztwh(_sacHTData.getSacSmaHtState().getLztwc());
            l = 1;
        }
        sfi = sfi + _sacHTData.getSacSmaHtState().getLztwc() - _sacHTData.getSacSmaHtState().getLztwh();

        if(_sacHTData.getSacSmaHtState().getLzfsh() > _sacHTData.getSacSmaHtState().getLzfsc())
        // if(MathHelper.isGreater(_sacHTData.getSacSmaHtState().getLzfsh(),
        // _sacHTData.getSacSmaHtState().getLzfsc()))
        {
            _sacHTData.getSacSmaHtState().setLzfsh(_sacHTData.getSacSmaHtState().getLzfsc());
            l = 1;
        }
        sfi = sfi + _sacHTData.getSacSmaHtState().getLzfsc() - _sacHTData.getSacSmaHtState().getLzfsh();

        if(_sacHTData.getSacSmaHtState().getLzfph() > _sacHTData.getSacSmaHtState().getLzfpc())
        // if(MathHelper.isGreater(_sacHTData.getSacSmaHtState().getLzfph(),
        // _sacHTData.getSacSmaHtState().getLzfpc()))
        {
            _sacHTData.getSacSmaHtState().setLzfph(_sacHTData.getSacSmaHtState().getLzfpc());
            l = 1;
        }
        sfi = sfi + _sacHTData.getSacSmaHtState().getLzfpc() - _sacHTData.getSacSmaHtState().getLzfph();

        xx1 = Math.abs(_sacHTData.getSacSmaHtState().getFindex() * SacSmaHTConstants.FRST_FACT);
        xx2 = 1.1 * sfi;

        if(xx1 > xx2)
        // if(MathHelper.isGreater(xx1, xx2))
        {
            _sacHTData.getSacSmaHtState().setFindex(-(float)sfi / (float)SacSmaHTConstants.FRST_FACT);
            l = 1;
        }
        xx1 = Math.abs(_sacHTData.getSacSmaHtState().getFindex() * SacSmaHTConstants.FRST_FACT);
        xx2 = 0.9 * sfi;

        if(xx1 < xx2)
        // if(MathHelper.isLessThan(xx1, xx2))
        {
            _sacHTData.getSacSmaHtState().setFindex(0.0f - (float)sfi / (float)SacSmaHTConstants.FRST_FACT);
            l = 1;
        }

        if(l != 0)
        {
            _logger.log(Logger.WARNING, "WARN: initial state variables contain impossible values");
        }
    }

    private void estFst1()
    {
        // GENERATE INITIAL UNFROZEN WATER STATES IF NOT INPUTED
        // SUBROUTINE ESTFST1(CF,IC,FGCO,TSOIL,FGPM)
        // SUBROUTINE ESTFST1(FGCO,TSOIL,FGPM,state)
        // REAL
        // ZSOIL(MAX_NUMBER_OF_SOIL_LAYERS),SMC(MAX_NUMBER_OF_SOIL_LAYERS),SH2O(MAX_NUMBER_OF_SOIL_LAYERS)
        final double[] zSoil = new double[SacSmaHTConstants.MAX_NUMBER_OF_SOIL_LAYERS];
        double suz, slz, smcuz, smclz, tbup;
        double sup, slw, tbdn, dz, ts, dsw, alp, alp1, frost;
        int frozenGround, nSoil;

        // COMMON/FRDSTFG/SMAX,PSISAT,BRT,SWLT,QUARTZ,STYPE,NUPL,NSAC,RTUP,RTLW,DZUP,DZLOW
        // COMMON/FRZCNST/ FRST_FACT,ZBOT

        nSoil = (int)(_sacHTData.getSacSmaHtState().getNumberOfSoilLayers() + 0.1);

        frozenGround = 0;
        for(int i = 0; i < nSoil; i++)
        {
            if(_sacHTData.getSacSmaHtState().getSoilLayerTemperature(i) < 0.)
            // if(MathHelper.isLessThanZero(_sacHTData.getSacSmaHtState().getSoilLayerTemperature(i)))
            {
                frozenGround = 1;
            }
        }

        _sacHTData.getSacSmaHtState()
                  .setSoilLayerMoistureContent(0, (float)SacSmaHTConstants.MAX_SOIL_POROSITY * 0.15f);
        _sacHTData.getSacSmaHtState().setSoilLayerUnfrozenWater(0,
                                                                _sacHTData.getSacSmaHtState()
                                                                          .getSoilLayerMoistureContent(0));

        // The following just for convenience, so don't have to repeatedly get
        // from state
        double uztwc, uzfwc, lztwc, lzfsc, lzfpc;
        uztwc = _sacHTData.getSacSmaHtState().getUztwc();
        uzfwc = _sacHTData.getSacSmaHtState().getUzfwc();
        lztwc = _sacHTData.getSacSmaHtState().getLztwc();
        lzfsc = _sacHTData.getSacSmaHtState().getLzfsc();
        lzfpc = _sacHTData.getSacSmaHtState().getLzfpc();

        // c ESTIMATE UNFROZEN WATER STORAGES
        for(int i = 0; i < nSoil; i++)
        {
            zSoil[i] = _sacHTData.getSacSmaHtState().getSoilLayerDepth(i);
        }

        _sacHTData.getFrdStfg().dzUp = zSoil[0] - zSoil[_sacHTData.getFrdStfg().nUpl - 1];
        _sacHTData.getFrdStfg().dzLow = zSoil[_sacHTData.getFrdStfg().nUpl - 1] - zSoil[nSoil - 1];
        suz = uztwc + uzfwc;
        slz = lztwc + lzfsc + lzfpc;
        smcuz = 0.001 * _sacHTData.getFrdStfg().rTuz * suz / _sacHTData.getFrdStfg().dzUp
            + _sacHTData.getFrdStfg().soilWiltingPoint;
        smclz = 0.001 * _sacHTData.getFrdStfg().rTlz * slz / _sacHTData.getFrdStfg().dzLow
            + _sacHTData.getFrdStfg().soilWiltingPoint;

        tbup = SacSmaHtCommonMethods.calculateBoundaryLayerTemperature(_sacHTData.getSacSmaHtState().getSoilLayerTemperature(0)
                                                                   + SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS,
                                                               _sacHTData.getSacSmaHtState().getSoilLayerTemperature(1)
                                                                   + SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS,
                                                               zSoil,
                                                               SacSmaHTConstants.DEPTH_AT_BOTTOM_LAYER_IN_METERS,
                                                               0,
                                                               nSoil);
        sup = 0.;
        slw = 0.;
        for(int i = 1; i < nSoil; i++)
        {
            // CALCULATE AVERAGE SOIL TEMPERATURE OF I-TH LAYER
            if(i != nSoil - 1)
            {
                tbdn = SacSmaHtCommonMethods.calculateBoundaryLayerTemperature(_sacHTData.getSacSmaHtState()
                                                                                 .getSoilLayerTemperature(i)
                                                                           + SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS,
                                                                       _sacHTData.getSacSmaHtState()
                                                                                 .getSoilLayerTemperature(i + 1)
                                                                           + SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS,
                                                                       zSoil,
                                                                       SacSmaHTConstants.DEPTH_AT_BOTTOM_LAYER_IN_METERS,
                                                                       i,
                                                                       nSoil);
            }
            else
            {
                tbdn = SacSmaHtCommonMethods.calculateBoundaryLayerTemperature(_sacHTData.getSacSmaHtState()
                                                                                 .getSoilLayerTemperature(i)
                                                                           + SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS,
                                                                       _sacHTData.getSacSmaHtParameters().getTbot(),
                                                                       zSoil,
                                                                       SacSmaHTConstants.DEPTH_AT_BOTTOM_LAYER_IN_METERS,
                                                                       i,
                                                                       nSoil);
            }

            dz = zSoil[i - 1] - zSoil[i];
            ts = SacSmaHtCommonMethods.calculateAverageTemperature(tbup,
                                                           _sacHTData.getSacSmaHtState().getSoilLayerTemperature(i)
                                                               + SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS,
                                                           tbdn,
                                                           dz);
            tbup = tbdn;

            // CALCULATE POTENTIAL UNFROZEN WATER CONTENT
            if(i < _sacHTData.getFrdStfg().nUpl)
            {
                _sacHTData.getSacSmaHtState().setSoilLayerMoistureContent(i, smcuz);
                //if(ts <= SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS)
                if(MathHelper.isLessOrEqual(ts, SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS))
                {
                    _sacHTData.getSacSmaHtState()
                              .setSoilLayerUnfrozenWater(i,
                                                         SacSmaHtCommonMethods.frH2o(ts,
                                                                             _sacHTData.getSacSmaHtState()
                                                                                       .getSoilLayerMoistureContent(i),
                                                                             _sacHTData.getSacSmaHtState()
                                                                                       .getSoilLayerMoistureContent(i),
                                                                             _sacHTData.getFrdStfg().maximumSoilMoisture,
                                                                             _sacHTData.getFrdStfg().brt,
                                                                             _sacHTData.getFrdStfg().psisat,
                                                                             _sacHTData.getSacSmaHtParameters()
                                                                                       .getRunoffEffect()));

                }
                else
                {
                    _sacHTData.getSacSmaHtState().setSoilLayerUnfrozenWater(i,
                                                                            _sacHTData.getSacSmaHtState()
                                                                                      .getSoilLayerMoistureContent(i));
                }

                dsw = 1000
                    * (_sacHTData.getSacSmaHtState().getSoilLayerUnfrozenWater(i) - _sacHTData.getFrdStfg().soilWiltingPoint)
                    * (zSoil[i - 1] - zSoil[i]) / _sacHTData.getFrdStfg().rTuz;

                //if(dsw > 0.)
                if(MathHelper.isGreaterThanZero(dsw))
                {
                    sup = sup + dsw;
                }
            }
            else
            {
                _sacHTData.getSacSmaHtState().setSoilLayerMoistureContent(i, smclz);
                //if(ts <= SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS)
                if(MathHelper.isLessOrEqual(ts, SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS))
                {
                    _sacHTData.getSacSmaHtState()
                              .setSoilLayerUnfrozenWater(i,
                                                         SacSmaHtCommonMethods.frH2o(ts,
                                                                             _sacHTData.getSacSmaHtState()
                                                                                       .getSoilLayerMoistureContent(i),
                                                                             _sacHTData.getSacSmaHtState()
                                                                                       .getSoilLayerMoistureContent(i),
                                                                             _sacHTData.getFrdStfg().maximumSoilMoisture,
                                                                             _sacHTData.getFrdStfg().brt,
                                                                             _sacHTData.getFrdStfg().psisat,
                                                                             _sacHTData.getSacSmaHtParameters()
                                                                                       .getRunoffEffect()));
                }
                else
                {
                    _sacHTData.getSacSmaHtState().setSoilLayerUnfrozenWater(i,
                                                                            _sacHTData.getSacSmaHtState()
                                                                                      .getSoilLayerMoistureContent(i));
                }
                dsw = 1000
                    * (_sacHTData.getSacSmaHtState().getSoilLayerUnfrozenWater(i) - _sacHTData.getFrdStfg().soilWiltingPoint)
                    * (zSoil[i - 1] - zSoil[i]) / _sacHTData.getFrdStfg().rTlz;
                //if(dsw > 0.)
                if(MathHelper.isGreaterThanZero(dsw))
                {
                    slw = slw + dsw;
                }
            }
        } // end do

        // original condition - if ((_sacHTData.getSacSmaHtState().getFindex()
        // == 999.0) && (l != 0))
        // no longer use condition above because fIndex == 999.0 indicates a
        // cold start state, the whole
        // of estFst1 is only done on cold starts..
        if(frozenGround != 0)
        {
            //if(sup > suz)
            if(MathHelper.isGreater(sup, suz))
            {
                sup = suz;
            }
            //if(slw > slz)
            if(MathHelper.isGreater(slw, slz))
            {
                slw = slz;
            }
            alp = uztwc / suz;
            _sacHTData.getSacSmaHtState().setUztwh(sup * alp);
            _sacHTData.getSacSmaHtState().setUzfwh(sup * (1 - alp));
            alp = lztwc / slz;
            _sacHTData.getSacSmaHtState().setLztwh(slw * alp);

            alp1 = lzfsc / slz;
            _sacHTData.getSacSmaHtState().setLzfsh(slw * alp1);
            _sacHTData.getSacSmaHtState().setLzfph(slw * (1 - alp - alp1));
            frost = 0.;

            frost = frost + uztwc - _sacHTData.getSacSmaHtState().getUztwh();
            frost = frost + uzfwc - _sacHTData.getSacSmaHtState().getUzfwh();
            frost = frost + lztwc - _sacHTData.getSacSmaHtState().getLztwh();
            frost = frost + lzfsc - _sacHTData.getSacSmaHtState().getLzfsh();
            frost = frost + lzfpc - _sacHTData.getSacSmaHtState().getLzfph();
            _sacHTData.getSacSmaHtState().setFindex(0.f - frost / SacSmaHTConstants.FRST_FACT);
        }
        else
        {
            // ALL WATER UNFROZEN
            _sacHTData.getSacSmaHtState().setUztwh(uztwc);
            _sacHTData.getSacSmaHtState().setUzfwh(uzfwc);
            _sacHTData.getSacSmaHtState().setLztwh(lztwc);
            _sacHTData.getSacSmaHtState().setLzfsh(lzfsc);
            _sacHTData.getSacSmaHtState().setLzfph(lzfpc);
            _sacHTData.getSacSmaHtState().setFindex(0.);
        }
    }

    /* return nsoil - soil layer selected */
    private int soilPar1(final double supm, final double[] soilLayerDepthInMeters) throws Exception
    {
        /*
         * C MEMBER SOILPAR1 SUBROUTINE SOILPAR1(STXT,SUPM,SLWM,SMAX,PSISAT,BRT,SWLT,
         * QUARTZ,STYPE,NSOIL,NUPL,NSAC,ZSOIL,RTUP,RTLOW)
         */

        /*
         * CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC PURPOSE:CALCULATE SOIL PARAMETERS BASED ON SOIL
         * TEXTURE (STXT) RITTEN BY VICTOR KOREN - HRL JANUARY 2000
         * ------------------------------------------------------ SMAX - SOIL POROSITY - Important - same as fgpm[0]
         * PSISAT - SATURATION MATRIC POTENTIAL BRT - SLOPE OF THE RETENTION CURVE FRT = LOG(PSISAT)+BRTLOG(SMAX)+2.
         * QUARTZ - FRACTION OF QUARTZ STYPE - SOIL TYPE (11 - IF COARSE SOIL, 12 - IF FINE) NSOIL - TOTAL NUMBER OF
         * SELECTED LAYERS NUPL - NUMBER OF UPPER ZONE LAYERS + RESIDUE LAYER RTUP - =1, AND >1 ONLY WHEN UPPER ZONE
         * DEEPPER THAN MAX SOIL DEPTH RTLOW - THE SAME FOR LOWER ZONE
         * CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
         */

        _sacHTData.getLogger().log(Logger.DEBUG, "Enter soilPar1()");
        double zup, zlow, ztot;
        int numberOfSoilLayers;

        final double[] layerDepths = new double[SacSmaHTConstants.MAX_NUMBER_OF_SOIL_LAYERS];

        /*
         * C DEFINITION OF DEFAULT SOIL LAYER THICKNESS FIVE LAYERS ARE DIFINED. IF IT'S DESIRABLE TO USE MORE OR LESS
         * LAYERS, SDPTH SHOULD BE CHANGED, HOWEVER, THE FIRST LAYER (VERY THIN RESIDUE LAYER) SHOULD NOT BE CHANGED
         */

        final double[] defaultLayerDepths = {0.03, 0.14, 0.41, 0.81, 1.51, 0., 0., 0., 0., 0.};
        final double[] percentSandBySoilTextureNumber = {0.92, 0.82, 0.58, 0.17, 0.09, 0.43, 0.58, 0.10, 0.32, 0.52,
            0.06, 0.22};
        final double[] percentClayBySoilTextureNumber = {0.03, 0.06, 0.10, 0.13, 0.05, 0.18, 0.27, 0.34, 0.34, 0.42,
            0.47, 0.58};
        double x, zmax, xx1;
        int nLow = 0;
        /*
         * C SAND AND CLAY FRACTION TABLES DEPENDING ON SOIL TEXTURE CLASS (FROM CLASS 1 - SAND TO CLASS 12 - CLAY),
         * FROM COSBY ET AL. [WWR, 20(6), 1984] 1 - SAND 5.2% (IN USA) 2 - LOAMY SAND 5.6% 3 - SANDY LOAM 18.1% 4 -
         * SILTY LOAM 19.4% 5 - SILT 6 - LOAM 12.7% 7 - SANDY CLAY LOAM 3.9% 8 - SILTY CLAY LOAM 12.0% 9 - CLAY LOAM
         * 8.5% 10 - SANDY CLAY 0.7% 11 - SILTY CLAY 6.4% 12 - CLAY 7.5% FRACTION OF QUARTZ ASSUMED TO BE EQUAL FRACTION
         * OF SAND AND ALL SOILS ASSUMED AS FINE, FROM PETERS-LIDARD ET AL. [J. ATM. SCI., 55, 1998]
         */

        /*
         * C COSBY RELATIONSHIPS SMAX IS POROSITY PSISAT IS 'SATURATION ' MATRIC POTENTIAL, IN M BRT IS SLOPE OF
         * RETENTION CURVE SWLT IS WILTING POINT QUARTZ IS QUARTZ FRACTION STYPE IS SOIL TYPE: 11 - NATURAL COARSE, 12 -
         * NATURAL FINE, 21 - CRUSHED COARSE, 22 - CRUSHED FINE
         */

        // int soilTextureNumber=
        // (int)(_sacHTData.getSacSmaHtParameters().getStxt() + .01);
        final int soilTextureNumber = _sacHTData.getSacSmaHtParameters().getStxt();
        _sacHTData.getFrdStfg().maximumSoilMoisture = -0.126 * percentSandBySoilTextureNumber[soilTextureNumber - 1]
            + 0.489;
        // cvk 12/05 Equation for PSISAT wrong: should be 7.74
        // coefficient not 0.759
        // cvk PSISAT = 0.759*EXP(-3.02*SAND(ITXT))
        _sacHTData.getFrdStfg().psisat = 7.74 * Math.exp(-3.02 * percentSandBySoilTextureNumber[soilTextureNumber - 1]);
        _sacHTData.getFrdStfg().brt = 15.9 * percentClayBySoilTextureNumber[soilTextureNumber - 1] + 2.91;
        _sacHTData.getFrdStfg().soilWiltingPoint = _sacHTData.getFrdStfg().maximumSoilMoisture
            * Math.pow(1500. / _sacHTData.getFrdStfg().psisat, -1. / _sacHTData.getFrdStfg().brt);
        _sacHTData.getFrdStfg().quartz = percentSandBySoilTextureNumber[soilTextureNumber - 1];
        _sacHTData.getFrdStfg().soilType = 12;

        // C DEFINE SOIL LAYER THICKNESS & SOIL
        // MOISTURE STATES OF THE LAYERS
        int tempNumOfSoilLayers;
        for(tempNumOfSoilLayers = 0; tempNumOfSoilLayers < SacSmaHTConstants.MAX_NUMBER_OF_SOIL_LAYERS; tempNumOfSoilLayers++)
        {
            layerDepths[tempNumOfSoilLayers] = defaultLayerDepths[tempNumOfSoilLayers];
            if(layerDepths[tempNumOfSoilLayers] == 0.)
            // if(MathHelper.isEqualToZero(layerDepths[tempNumOfSoilLayers]))
            {
                break;
                // go to 20
            }
        }

        // label 20
        numberOfSoilLayers = tempNumOfSoilLayers;

        zup = 0.001 * supm / (_sacHTData.getFrdStfg().maximumSoilMoisture - _sacHTData.getFrdStfg().soilWiltingPoint);
        final double slwm = _sacHTData.getSacSmaHtParameters().getLztwm()
            + _sacHTData.getSacSmaHtParameters().getLzfsm() + _sacHTData.getSacSmaHtParameters().getLzfpm();
        zlow = 0.001 * slwm / (_sacHTData.getFrdStfg().maximumSoilMoisture - _sacHTData.getFrdStfg().soilWiltingPoint);
        ztot = zup + zlow;
        if(zup > layerDepths[numberOfSoilLayers - 1] - layerDepths[0] || Double.isNaN(zup))
        // if(MathHelper.isGreater(zup, layerDepths[numberOfSoilLayers - 1] -
        // layerDepths[0]) || Double.isNaN(zup))
        {
            _logger.log(Logger.ERROR, "Error in soilPar: Too high upper storages" + supm + " " + zup);
            _logger.log(Logger.ERROR, "Reduce UZTWM or UZFWM or their upper bounds");
            throw new ModelException("Too high upper storages");
        }
        // anywhere where we subtract -1 from numberOfSoilLayers-1 is because
        // the model really intends
        // to get to the previous layer (the FORTRAN version was NSOIL-1 - since
        // Java indices begin at 0....

        x = 1.5 * (layerDepths[numberOfSoilLayers - 1 - 1] - layerDepths[0]);
        if(zup > x)
        // if(MathHelper.isGreater(zup, x))
        {
            layerDepths[numberOfSoilLayers - 1 - 1] = zup / 1.5 + layerDepths[0];
        }

        // C SPLIT THE UPPER ZONE INTO SOIL LAYERS
        _sacHTData.getFrdStfg().rTuz = 1.;
        _sacHTData.getFrdStfg().rTlz = 1.;
        soilLayerDepthInMeters[0] = 0.0 - layerDepths[0];
        zmax = 1.5 * (layerDepths[numberOfSoilLayers - 1] - layerDepths[0]);

        for(int i = 1; i < numberOfSoilLayers; i++)
        {
            xx1 = 1.5 * (layerDepths[i] - layerDepths[0]);
            if(zup <= xx1 && !Double.isNaN(zup) || i == numberOfSoilLayers - 1)
            // if(MathHelper.isLessOrEqual(zup, xx1) || i == numberOfSoilLayers
            // - 1)
            {
                nLow = i + 1;
                _sacHTData.getFrdStfg().nUpl = i + 1;
                if(zup > zmax || Double.isNaN(zup))
                // if(MathHelper.isGreater(zup, zmax) || Double.isNaN(zup))
                {
                    _sacHTData.getFrdStfg().rTuz = zmax / zup;
                    zup = zmax;
                }
                for(int j = 1; j <= i; j++)
                {
                    soilLayerDepthInMeters[j] = soilLayerDepthInMeters[j - 1] - zup
                        * (layerDepths[j] - layerDepths[j - 1]) / (layerDepths[i] - layerDepths[0]);
                }
                // go to 1
                break;
            }
        }

        // 1 CONTINUE
        // C SPLIT THE LOWER ZONE INTO SOIL LAYERS
        if(_sacHTData.getFrdStfg().nUpl < numberOfSoilLayers)
        {
            for(int i = _sacHTData.getFrdStfg().nUpl; i < numberOfSoilLayers; i++)
            {
                xx1 = 1.5 * (layerDepths[i] - layerDepths[0]);
                if(ztot <= xx1 && !Double.isNaN(ztot) || i == numberOfSoilLayers - 1)
                // if(MathHelper.isLessOrEqual(ztot, xx1) && !Double.isNaN(ztot)
                // || i == numberOfSoilLayers - 1)
                {
                    if(ztot > zmax || Double.isNaN(ztot))
                    // if(MathHelper.isGreater(ztot, zmax) ||
                    // Double.isNaN(ztot))
                    {
                        _sacHTData.getFrdStfg().rTlz = (zmax - zup) / (ztot - zup);
                        ztot = zmax;
                    }

                    for(int j = _sacHTData.getFrdStfg().nUpl; j <= i; j++)
                    {
                        soilLayerDepthInMeters[j] = soilLayerDepthInMeters[j - 1] - (ztot - zup)
                            * (layerDepths[j] - layerDepths[j - 1])
                            / (layerDepths[i] - layerDepths[_sacHTData.getFrdStfg().nUpl - 1]);
                    }
                    // GOTO 2
                    nLow = i + 1;
                    break;
                }
            }
        }

        // 2 CONTINUE

        xx1 = 0.1 * layerDepths[numberOfSoilLayers - 1];

        if(nLow < numberOfSoilLayers && -soilLayerDepthInMeters[nLow - 1] < xx1)
        // if(nLow < numberOfSoilLayers &&
        // MathHelper.isLessThan(-soilLayerDepthInMeters[nLow - 1], xx1))
        {
            _sacHTData.getFrdStfg().nsac = nLow;
            soilLayerDepthInMeters[nLow - 1 + 1] = soilLayerDepthInMeters[_sacHTData.getFrdStfg().nsac - 1]
                + soilLayerDepthInMeters[_sacHTData.getFrdStfg().nsac - 1]
                - soilLayerDepthInMeters[_sacHTData.getFrdStfg().nsac - 1 - 1];
            if(-soilLayerDepthInMeters[nLow - 1 + 1] < layerDepths[numberOfSoilLayers - 1])
            // if(MathHelper.isLessThan(-soilLayerDepthInMeters[nLow - 1 + 1],
            // layerDepths[numberOfSoilLayers - 1]))
            {
                soilLayerDepthInMeters[nLow - 1 + 1] = -layerDepths[numberOfSoilLayers - 1];
            }
            numberOfSoilLayers = nLow + 1;
        }
        else
        {
            numberOfSoilLayers = nLow;
            _sacHTData.getFrdStfg().nsac = numberOfSoilLayers;
        }

        if(soilLayerDepthInMeters[numberOfSoilLayers - 1] < SacSmaHTConstants.DEPTH_AT_BOTTOM_LAYER_IN_METERS)
        // if(MathHelper.isLessThan(soilLayerDepthInMeters[numberOfSoilLayers -
        // 1],Constants.DEPTH_AT_BOTTOM_LAYER_IN_METERS))
        {
            SacSmaHTConstants.DEPTH_AT_BOTTOM_LAYER_IN_METERS = soilLayerDepthInMeters[numberOfSoilLayers - 1];
        }

        if(_sacHTData.getFrdStfg().nUpl == nLow)
        {
            _logger.log(Logger.ERROR, "Number of upper layers and lower layers are the same");
            throw new ModelException("Number of upper and lower layers are the same");
        }
        _sacHTData.getLogger().log(Logger.DEBUG,
                                   "In soilPar1(): number of soil layers computed is " + numberOfSoilLayers);

        for(int i = 0; i < numberOfSoilLayers; i++)
        {
            _sacHTData.getLogger().log(Logger.DEBUG,
                                       "In soilPar1(): depth to layer " + i + " is " + soilLayerDepthInMeters[i]);
        }
        return numberOfSoilLayers;
    }

    // instead of modifying parfrz1, 2 and 3 from caller, directly modify
    // fgpm[3], fgpm[4] and fgpm[5]
    // from main_frz func
    // public void frdFg1(int idt, double supm, double slwm, double [] state,
    // double parfrz1, double parfrz2, double parfrz3)
    private void frdFg1(final int idt, final double supm, final double slwm) throws Exception
    {
        /*
         * C....................................... C THIS SUBROUTINE READS 'SAC-SMA ' INPUT CARDS RELATED TO FROZEN C
         * GROUND.
         */

        final double[] zSoil = new double[SacSmaHTConstants.MAX_NUMBER_OF_SOIL_LAYERS];
        // common/finfg0/
        // nsinp,zs(MAX_NUMBER_OF_SOIL_LAYERS),ts(MAX_NUMBER_OF_SOIL_LAYERS),fgco0(6),pta0,pwe0,psh0

        /*
         * Initialize few frozen ground constant to pass through common statement: FRST_FACT is a factor to reduce
         * frozen soil moisture to match closer Eric's temperature index; CKSOIL is a Kulik's parameter of frozen ground
         * conductivity reduction ZBOT is a depth of the bottom bound of the soil profile DATA FRST_FACT/5.0/,
         * CKSOIL/8.0/, ZBOT/-3.0/ DATA FRST_FACT/5.0/, ZBOT/-3.0/: tested Root with -2.0
         */

        final double zd;
        // SacSmaHTData._depthAtBottomLayer =-2.5;
        double zx, sh;

        final int nsl = SacSmaHTConstants.MAX_NUMBER_OF_SOIL_LAYERS;
        int nSoil = 0;

        // CVK DEFAULT STATE VALUES
        for(int i = 0; i < nsl; i++)
        {
            zSoil[i] = -999.;
            // _sacSmaHtState.setSoilLayerTemperature(i,-999.);
        }

        /*
         * C....................................... C READ FROZEN GROUND CARD NUMBER 1.
         */

        /*
         * READ FROZEN GROUND CARD NUMBER 2. CVK ARRAY FGPM IS INCREASD BY 5 VARIABLES
         */

        _finFg0.nsinp = _sacHTData.getSacSmaHtState().getNumberOfSoilLayers();

        if(_finFg0.nsinp < 0 || _finFg0.nsinp > 7)
        // if(MathHelper.isLessThanZero(_finFg0.nsinp) ||
        // MathHelper.isGreater(_finFg0.nsinp, 7))
        {
            _logger.log(Logger.ERROR, _finFg0.nsinp + "is an invalid number of soil layers");
            throw new ModelException(_finFg0.nsinp + "is an invalid number of soil layers");
        }

        // read depths of soil layers into zs
        if(_finFg0.nsinp > 0)
        // if(MathHelper.isGreaterThanZero(_finFg0.nsinp))
        {
            for(int i = 0; i < _finFg0.nsinp; i++)
            {
                // _finFg0.zs[i]
                // =((Double)_sacSmaHtState.getState("layerDepth"+i)).doubleValue();
                _finFg0.zs[i] = _sacHTData.getSacSmaHtState().getSoilLayerDepth(i);
            }
        }

        // CVK SET DEFAULT NEW FROZEN GROUND PARAMETERS

        // CVK SOIL TEXTURE */
        if(_sacHTData.getSacSmaHtParameters().getStxt() < 0)
        {
            _sacHTData.getSacSmaHtParameters().setStxt(4);
        }

        // CVK BOTTOM BOUNDARY (3m) SOIL TEMPERATURE (ASSUMED = CLIMATE ANNUAL
        // AIR TEMP)
        if(_sacHTData.getSacSmaHtParameters().getTbot() < 0.)
        // if(MathHelper.isLessThanZero(_sacHTData.getSacSmaHtParameters().getTbot()))
        {
            _sacHTData.getSacSmaHtParameters().setTbot(12.5 + SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS);
        }
        else
        {
            _sacHTData.getSacSmaHtParameters().setTbot(_sacHTData.getSacSmaHtParameters().getTbot()
                + SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS);
        }

        /*
         * CVK PARAMETER THAT ACCOUNTS FOR ICE EFFECT ON PERC./INFILTR. USUALLY CONST=8 CVK THERE ARE TWO OTHER
         * POTENTIAL PARAMETERS THAT ACCOUNT FOR CVK STATISTICAL PROPERTY OF ICE DISTRIBUTION INDUCED BY IMPERMEABLE
         * SOIL LAYER: CVK CRITICAL ICE CONTENT ABOVE WHICH SOIL BECOMES IMPERMEABLE, CVK ICE DISTRIBUTION PARAMETER
         * (ASSUMES GAMMA DISTRIBUTION) CVK USUALLY THEY NOT NEEDED. DEFAULT OPTION EXCLUDES THESE EFFECT.
         */

        // CVK DEFINE SOIL LAYERS & STATES
        nSoil = soilPar1(supm, zSoil);

        _sacHTData.getFrdStfg().dzUp = (float)zSoil[0] - (float)zSoil[_sacHTData.getFrdStfg().nUpl - 1];
        _sacHTData.getFrdStfg().dzLow = (float)zSoil[_sacHTData.getFrdStfg().nUpl - 1]
            - (float)zSoil[_sacHTData.getFrdStfg().nsac - 1];

        // code below commented out; for warm starts; we currently dont' support feature of using
        // soil temperatures or depths different from what model read
        if(_finFg0.nsinp > 0)
        // if(MathHelper.isGreaterThanZero(_finFg0.nsinp))
        {

//            // CVK...........................................
//            // CVK READ CARD NUMBER 3A (SOIL TEMP.) IF NSINP NE 0
//            for(int i = 0; i < _finFg0.nsinp; i++)
//            {
//                _finFg0.ts[i] = (float)_sacHTData.getSacSmaHtState().getSoilLayerTemperature(i);
//            }
//
//            // CVK IF STATES ARE IN INPUT CARD ZS(1) SHOULD = -ZSOIL(1),
//            // CVK AND TS(1) SHOULD = TSOIL(1).
//
//            _sacHTData.getSacSmaHtState().setSoilLayerTemperature(0, _finFg0.ts[0]);
//            // c03/07 - following line replaced by the one after by victor
//            // zd = zSoil[nSoil-1]-_finFg0.zs[(int)_finFg0.nsinp-1];
//            zd = -(float)zSoil[nSoil - 1] - _finFg0.zs[(int)_finFg0.nsinp - 1];
//
//            // CVK IF INPUT SOIL DEPTH LESS SOIL DEFINED DEPTH, USE
//            // CVK A BOTTOM BOUNDARY SOIL TEMPERATURE TO INTERPOLATE
//            if(zd > 0.)
//            // if(MathHelper.isGreaterThanZero(zd))
//            {
//                _finFg0.ts[(int)_finFg0.nsinp] = (float)_sacHTData.getSacSmaHtParameters().getTbot()
//                    - (float)SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS;
//                // 03/07 victor's change to following line
//                _finFg0.zs[(int)_finFg0.nsinp] = -SacSmaHTConstants.DEPTH_AT_BOTTOM_LAYER_IN_METERS;
//            }
//            boolean continueFor = true;
//            for(int i = 1; i < nSoil; i++)
//            {
//                // c03/07 - following line replaced by one after by victor
//                // zx = zSoil[0] - zSoil[i];
//                zx = -(float)zSoil[i];
//                // attention - again, what to do if j == 0?
//                for(int j = 0; j <= (int)_finFg0.nsinp; j++)
//                {
//                    if(zx <= _finFg0.zs[j])
//                    // if(MathHelper.isLessOrEqual(zx, _finFg0.zs[j]))
//                    {
//                        if(j != 0)
//                        {
//                            _sacHTData.getSacSmaHtState()
//                                      .setSoilLayerTemperature(i,
//                                                               _finFg0.ts[j - 1] + (_finFg0.ts[j] - _finFg0.ts[j - 1])
//                                                                   * (zx - _finFg0.zs[j - 1])
//                                                                   / (_finFg0.zs[j] - _finFg0.zs[j - 1]));
//                            System.out.println("initial zx zs " + j + " " + zx + " " + _finFg0.zs[j - 1] + " "
//                                + _finFg0.zs[j]);
//                            System.out.println("initial ts " + j + " " + " " + _finFg0.ts[j - 1] + " " + _finFg0.ts[j]);
//                            System.out.println("initial tsoil " + j
//                                + _sacHTData.getSacSmaHtState().getSoilLayerTemperature(i));
//                            // c03/07
//                            continueFor = false;
//                            System.out.println("before break in for. SoilLayerTemp = "
//                                + _sacHTData.getSacSmaHtState().getSoilLayerTemperature(i));
//                            break;
//                            //TODO: This only break the inner loop but not the external loop
//                        }
//                    }
//                }
//                System.out.println("after break in for");
//                if(!continueFor)
//                {
//                    break;
//                }
//           }
        }
        else
        {
            // for the only soil moisture mode, init air temperature is set to
            // 10 and the init we and depth to 0.
            if(_sacHTData.getSacSmaHtParameters().getModelRunType().equals(SacSmaHTConstants.ONLY_SOIL_RUN_STR))
            {
                _sacHTData.getSacSmaHtState().setInitAirTemperature(10.);
                _sacHTData.getSacSmaHtState().setInitSnowDepth(0.0);
                _sacHTData.getSacSmaHtState().setInitSnowWaterEquivalent(0.0);
            }

            // CVK IF STATES ARE NOT IN INPUT CARD INTERPOLATE BETWEEN AIR
            // CVK AND BOTTOM BOUNDARY SOIL TEMPERATURE
            // CVK AIR TEMPERATURE AND SNOW WATER EQUIVALENT STATES 0 BE
            // CVK IN INPUT DECK.
            _finFg0.zs[0] = 0.;
            _finFg0.ts[0] = _sacHTData.getSacSmaHtState().getInitAirTemperature();
            // CVK ASSUMED SNOW CONDUCTIVITY IS LESS THAN SOIL CONDUCTIVITY
            // CVK BY RATIO OF INVERSE SNOW DENSITY =0.2
            sh = 0.01f * (float)_sacHTData.getSacSmaHtState().getInitSnowDepth() / 0.2f;
            _finFg0.zs[1] = -(float)SacSmaHTConstants.DEPTH_AT_BOTTOM_LAYER_IN_METERS + (float)sh;
            _finFg0.ts[1] = (float)_sacHTData.getSacSmaHtParameters().getTbot()
                - (float)SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS;

            for(int i = 0; i < nSoil; i++)
            {
                if(i == 0)
                {
                    zx = -0.5f * (float)zSoil[0] + (float)sh;
                }
                else
                {
                    zx = -0.5f * ((float)zSoil[i - 1] + (float)zSoil[i]) + (float)sh;
                }

                _sacHTData.getSacSmaHtState().setSoilLayerTemperature(i,
                                                                      _finFg0.ts[0] + (_finFg0.ts[1] - _finFg0.ts[0])
                                                                          * zx / _finFg0.zs[1]);
            }

        } // if nsinp > 0

        // cc TSOIL(NSOIL+1)=FGPM(2)
        _sacHTData.getSacSmaHtState().setNumberOfSoilLayers(nSoil);
        for(int i = 0; i < nSoil; i++)
        {
            _sacHTData.getSacSmaHtState().setSoilLayerDepth(i, (float)zSoil[i]);
            // cc FGPM(7+NSOIL+1)=ZBOT
        }

        // c save initial frozen ground variables

        _finFg0.pta0 = _sacHTData.getSacSmaHtState().getInitAirTemperature();
        _finFg0.pwe0 = _sacHTData.getSacSmaHtState().getInitSnowWaterEquivalent();
        _finFg0.psh0 = _sacHTData.getSacSmaHtState().getInitSnowDepth();

        // c Generate frozen states if not defined
        // c in optimization mode fstfg1 will be called in loop from frdfg0
        fstFg1();

    }

    /**
     * If {@link #_inputModRtsList} size() > 0 (means that there are input MOD TSs), this method then apply those MODS
     * modifications. It takes the effect of SACCO MOD and FGIX MOD first, then SACBASEF MOD in the end. SACBASEF MOD
     * will be ignored if LZFPC or LZFSC MOD happened at this time step.
     * <p>
     * Note: the state period is from (time - interval) to time. So modify the state after the computation has been done
     * at this step. Get ready for next time step.
     */
    private void applySacSmaMods(final long time)
    {
        if(_modsTsList.size() == 0)
        {
            return; // no input MOD TSs, do nothing
        }

        boolean baseflowStatesModified = false;

        RegularTimeSeries sacbasefModRTS = null;

        /*
         * This loop is for potential SACCO MOD or FGIX MOD
         */
        for(final RegularTimeSeries modRTS: _modsTsList)
        {
            if(modRTS.getTimeSeriesType().equals(OHDConstants.SAC_MOD_BASEF) == false)
            {// sacbasef mod will be treated separately later

                try
                {
                    final double modValue = modRTS.getMeasurementValueByTime(time, modRTS.getMeasuringUnit());

                    //if is missing value, do not apply
                    if(modValue == modRTS.getMissingValue())
                    {
                        continue;
                    }

                    final String stateName = modRTS.getTimeSeriesType();
                    _sacHTData.getSacSmaHtState().updateState(stateName, modValue);

                    _logger.log(Logger.DEBUG,
                                "Applying state MODS " + stateName + " at time "
                                    + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE) + " GMT");

                    this.checkModsMaxValues(time); //needs before moving into estFst1()

                    // make sure to wipe out soil moisture states.
                    estFst1();

                    if(stateName.equals(SacSmaHTConstants.LOWER_ZONE_FREE_PRIMARY_WATER_CONTENTS)
                        || stateName.equals(SacSmaHTConstants.LOWER_ZONE_FREE_SECONDARY_WATER_CONTENTS))
                    {
                        baseflowStatesModified = true;
                    }

                }
                catch(final NullPointerException e)
                {
                    // no MODS at this time, loop to next rts
                }
            }
            else
            { // now this RTS is SACBASEF type
                sacbasefModRTS = modRTS;
            }

        } // close for loop

        /*
         * Apply SACBASE MOD after SACCO MOD. If the baseflow has been explicitly changed by LZFPC MOD or LZFSC MOD
         * above, ignore SACBASE MOD now. (SACBASE MOD cannot be at the initial state time either)
         */
        // if(time == _initStateTime)
        if(time == super.getInitialStateTime())
        {
            return; // SACBASEF MOD is ignored at the initial state time
        }
        else if(baseflowStatesModified == false && sacbasefModRTS != null)
        {
            try
            {
                final double multiplier = sacbasefModRTS.getMeasurementValueByTime(time, MeasuringUnit.unitlessReal);
                _sacHTData.getSacSmaHtState().setBaseflowStatesUsingMultiplier(multiplier);

                _logger.log(Logger.DEBUG, "Applying state MODS " + OHDConstants.SAC_MOD_BASEF + " at time "
                    + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE));

                this.checkModsMaxValues(time); //needs before moving into estFst1()

                // make sure to wipe out soil moisture states.
                estFst1();
            }
            catch(final NullPointerException e)
            {
                // do nothing, no MODS at this time
            }
        }

    } // close method applySacSmaMods(final long time)

    /**
     * Modify states as in fckco1.f. Cap the states to their allowed maximum values: the max of UZTWC is UZTWM, the max
     * of LZFPC is LZFPM etc.
     * 
     * @param time
     */
    private void checkModsMaxValues(final long time)
    {

        /** -------------modify states as in fckco1.f ------------------ */
        if(((SacSmaHTModelState)_state).getUztwc() > ((SacSmaHTModelParameters)_parameters).getUztwm())
        {
            _logger.log(Logger.WARNING, "The value for Upper Zone Tension Water Contents was changed from "
                + ((SacSmaHTModelState)_state).getUztwc() + " to " + ((SacSmaHTModelParameters)_parameters).getUztwm()
                + " on " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE) + ".");
            ((SacSmaHTModelState)_state).setUztwc(((SacSmaHTModelParameters)_parameters).getUztwm());
        }

        if(((SacSmaHTModelState)_state).getUzfwc() > ((SacSmaHTModelParameters)_parameters).getUzfwm())
        {
            _logger.log(Logger.WARNING, "The value for Upper Zone Free Water Contents was changed from "
                + ((SacSmaHTModelState)_state).getUzfwc() + " to " + ((SacSmaHTModelParameters)_parameters).getUzfwm()
                + " on " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE) + ".");
            ((SacSmaHTModelState)_state).setUzfwc(((SacSmaHTModelParameters)_parameters).getUzfwm());
        }

        if(((SacSmaHTModelState)_state).getLztwc() > ((SacSmaHTModelParameters)_parameters).getLztwm())
        {
            _logger.log(Logger.WARNING, "The value for Lower Zone Tension Water Contents was changed from "
                + ((SacSmaHTModelState)_state).getLztwc() + " to " + ((SacSmaHTModelParameters)_parameters).getLztwm()
                + " on " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE) + ".");
            ((SacSmaHTModelState)_state).setLztwc(((SacSmaHTModelParameters)_parameters).getLztwm());
        }

        if(((SacSmaHTModelState)_state).getLzfsc() > ((SacSmaHTModelParameters)_parameters).getLzfsm())
        {
            _logger.log(Logger.WARNING, "The value for Lower Zone Free Secondary Contents was changed from "
                + ((SacSmaHTModelState)_state).getLzfsc() + " to " + ((SacSmaHTModelParameters)_parameters).getLzfsm()
                + " on " + DateTime.getDateTimeStringFromLong(time, OHDConstants.GMT_TIMEZONE) + ".");
            ((SacSmaHTModelState)_state).setLzfsc(((SacSmaHTModelParameters)_parameters).getLzfsm());
        }

        if(((SacSmaHTModelState)_state).getLzfpc() > ((SacSmaHTModelParameters)_parameters).getLzfpm())
        {
            _logger.log(Logger.WARNING, "The value for Lower Zone Free Primary Contents was changed from "
                + ((SacSmaHTModelState)_state).getLzfpc() + " to " + ((SacSmaHTModelParameters)_parameters).getLzfpm()
                + " on " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE) + ".");
            ((SacSmaHTModelState)_state).setLzfpc(((SacSmaHTModelParameters)_parameters).getLzfpm());
        }

        if(((SacSmaHTModelState)_state).getAdimc() > (((SacSmaHTModelParameters)_parameters).getUztwm() + ((SacSmaHTModelParameters)_parameters).getLztwm()))
        {
            _logger.log(Logger.WARNING,
                        "The value for additional impervious contents was changed from "
                            + ((SacSmaHTModelState)_state).getAdimc()
                            + " to "
                            + (((SacSmaHTModelParameters)_parameters).getUztwm() + ((SacSmaHTModelParameters)_parameters).getLztwm())
                            + " on " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE) + ".");
            ((SacSmaHTModelState)_state).setAdimc(((SacSmaHTModelParameters)_parameters).getUztwm()
                + ((SacSmaHTModelParameters)_parameters).getLztwm());
        }

        if(((SacSmaHTModelState)_state).getAdimc() < ((SacSmaHTModelState)_state).getUztwc())
        {
            _logger.log(Logger.WARNING, "The value for additional impervious contents was changed from "
                + ((SacSmaHTModelState)_state).getAdimc() + " to " + ((SacSmaHTModelState)_state).getUztwc() + " on "
                + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE) + ".");
            ((SacSmaHTModelState)_state).setAdimc(((SacSmaHTModelState)_state).getUztwc());
        }
    }

    /*
     * (non-Javadoc)
     * @see ohd.hseb.fewsadapter.model.ModelDriver#setSpecificLogger(ohd.hseb.util.Logger) Sets the logger on _sacHTData
     * as well, so that things can be logged there.
     */

    @Override
    public void setLogger(final Logger logger)
    {
        super.setLogger(logger);
        _sacHTData.setLogger(logger);
        // Set the logger to parameters as it is required now.
        _sacHTData.getSacSmaHtParameters().setLogger(logger);
    }

    /*
     * (non-Javadoc)
     * @see ohd.hseb.fewsadapter.model.ModelDriver#setSpecificLogger(ohd.hseb.util.Logger) Sets the logger on _sacHTData
     * as well, so that things can be logged there.
     */

    @Override
    public Logger getLogger()
    {
        return _sacHTData.getLogger();

    }

    /*
     * (non-Javadoc)
     * @see ohd.hseb.fewsadapter.model.ModelDriver#setSpecificLogger(ohd.hseb.util.Logger) Sets the logger on _sacHTData
     * as well, so that things can be logged there.
     */

    public void setState(final ModelState state)
    {//override Driver method

        _state = state;
        _sacHTData.setSacSmaHtState((SacSmaHTModelState)_state);
    }

    /*
     * (non-Javadoc)
     * @see ohd.hseb.fewsadapter.model.ModelDriver#setSpecificLogger(ohd.hseb.util.Logger) Sets the logger on _sacHTData
     * as well, so that things can be logged there.
     */

    /*
     * (non-Javadoc)
     * @see ohd.hseb.fewsadapter.model.ModelDriver#setSpecificLogger(ohd.hseb.util.Logger) Sets the logger on _sacHTData
     * as well, so that things can be logged there.
     */

    @Override
    public SacSmaHTModelState getState()
    {
        return _sacHTData.getSacSmaHtState();
    }

    public void setModelParameters(final ModelParameters params)
    {
        _parameters = params;
        _sacHTData.setSacSmaHtParameters((SacSmaHTModelParameters)_parameters);
    }

    /*
     * (non-Javadoc)
     * @see ohd.hseb.fewsadapter.model.ModelDriver#setSpecificLogger(ohd.hseb.util.Logger) Sets the logger on _sacHTData
     * as well, so that things can be logged there.
     */

    @Override
    public SacSmaHTModelParameters getParameters()
    {
        return _sacHTData.getSacSmaHtParameters();
    }

    /**
     * @param uztwc
     * @param uzfwc
     * @param lztwc
     * @param lzfsc
     * @param lzfpc
     * @param adimc
     * @param fgco
     * @param rsum
     * @param ppe
     * @param psc
     * @param pta
     * @param pwe
     * @param psh
     * @param tsoil
     */
    public void createFsmCo1(final double uztwc,
                             final double uzfwc,
                             final double lztwc,
                             final double lzfsc,
                             final double lzfpc,
                             final double adimc,
                             final double[] fgco,
                             final double rsum,
                             final double ppe,
                             final double psc,
                             final double pta,
                             final double pwe,
                             final double psh,
                             final double[] tsoil)
    {

    }

    public double getmaximumSoilMoisture()
    {
        return _sacHTData.getFrdStfg().getmaximumSoilMoisture();
    }

    public double getpsisat()
    {
        return _sacHTData.getFrdStfg().getpsisat();
    }

    public double getbrt()
    {
        return _sacHTData.getFrdStfg().getbrt();
    }

    public double getsoilWiltingPoint()
    {
        return _sacHTData.getFrdStfg().getsoilWiltingPoint();
    }

    public double getquartz()
    {
        return _sacHTData.getFrdStfg().getquartz();
    }

    public double getsoilType()
    {
        return _sacHTData.getFrdStfg().getsoilType();
    }

    public int getnUpl()
    {
        return _sacHTData.getFrdStfg().getnUpl();
    }

    public int getnsac()
    {
        return _sacHTData.getFrdStfg().getnsac();
    }

    public double getrTuz()
    {
        return _sacHTData.getFrdStfg().getrTuz();
    }

    public double getrTlz()
    {
        return _sacHTData.getFrdStfg().getrTlz();
    }

    public double getdzUp()
    {
        return _sacHTData.getFrdStfg().getdzUp();
    }

    public double getdzLow()
    {
        return _sacHTData.getFrdStfg().getdzLow();
    }

    /*
     * CVK New fg version parameters & soil layer definition: FGPM(1) - soil texture class FGPM(2) - optional, soil
     * temperature at the 3m depth FGPM(3) - optional, porosity of residue layer PAR(18) [if no calb=FGPM(4)] - runoff
     * reduction parameter 1 PAR(19) [if no calb=FGPM(5)] - runoff reduction parameter 2 PAR(20) [if no calb=FGPM(6)] -
     * runoff reduction parameter 3 FGPM(6) - runoff reduction parameter 3 (for eric's version only) FGPM(7) - number of
     * soil layers FGPM(8)-FGPM(15) - depths of soil layers (m), negative. first layer (residue) depth=-0.03m
     */
    public void createFrzCnst(final double FRST_FACT, final double ZBOT)
    {

    }

    public void createFinFg0(final double nsinp,
                             final double[] zs,
                             final double[] ts,
                             final double[] fgco0,
                             final double pta0,
                             final double psh0,
                             final double pwe0)
    {
        _finFg0 = new FinFg0(nsinp, zs, ts, fgco0, pta0, psh0, pwe0);
    }

    private class FinFg0
    {
        double nsinp;
        double[] zs;
        double[] ts;
        double pta0;
        double psh0;
        double pwe0;

        FinFg0()
        {
            zs = new double[8];
            ts = new double[8];
        }

        FinFg0(final double nsinp,
               final double[] zs,
               final double[] ts,
               final double[] fgco0,
               final double pta0,
               final double psh0,
               final double pwe0)
        {
            this.nsinp = nsinp;
            this.zs = zs;
            this.ts = ts;
            this.pta0 = pta0;
            this.psh0 = psh0;
            this.pwe0 = pwe0;
        }
    }

} // close class
