package ohd.hseb.ohdmodels.sacsmaHT;

import ohd.hseb.ohdmodels.sacsma.SacSmaModelConstants;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.ModelException;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.ohdmodels.ModelParameters;

/**
 * This class represents the parameters for the SacSmaHTModel. Values are initially loaded into super._parasmMap, then
 * extracted into instance variables to avoid frequent accessing the Map.
 * 
 * @author FewsPilot Team
 */
final public class SacSmaHTModelParameters extends ModelParameters
{

    public final static String STXT_TAG = "STXT";
    public final static String TBOT_TAG = "TBOT";
    //public final static String CRSL_TAG = "CRSL";
    //NSOIL is here only to convert OPT files which contain this, it is not necessary for the current Java model
    public final static String NSOIL_TAG = "NSOIL";
//    private final int _numberOfSoilLayers = 0; //nsoil
    public final static String USER_DEFINE_SOIL_TAG = "USER_DEFINED_SOIL_LAYER_DEPTH";
    public final static String MODEL_RUN_TYPE_TAG = "MODEL_RUN_TYPE";

    private double _adimp;
    private double _efc;
    private double[] _etDemand = new double[OHDConstants.NUMBER_OF_MONTHS_IN_YEAR];
    private double _lzfpm;
    private double _lzfsm;
    private double _lzpk;
    private double _lzsk;
    private double _lztwm;
    /*
     * Set the default model run mode to a full run if user did not specify it where frozen ground calculations are done
     * and runoff is affected
     */
    private String _modelRunType = SacSmaHTConstants.FULL_RUN_STR;
    private double _pctim;
    private double _peadj;
    private double _pfree;
    private double _pxadj;
    private double _rexp;
    private double _riva;
    private double _rserv;
    private double _side;
    private int _stxt;
    private double _tbot;
    private double _uzfwm;
    private double _uzk;
    private double _uztwm;
    private double _zperc;

    private double[] _userDefinedSoilLayerDepths = new double[0]; //default value, if there are not values in the Parameter object

    /*
     * Default is false, which means that MAPE ts is not used and etDemand 12 values represent ET-demand 16th of each
     * month, daily values are computed by linear interpolation; In the condition of true, etDemand 12 values becomes
     * PE-adjustment factor for the input MATE ts
     */
    private boolean _useMapeInput = false;

    public SacSmaHTModelParameters()
    {
        super();
    }

    /**
     * Extract values from super._parametersMap into instance variables. The values in super._parametersMap are set by
     * parsing parameter xml file in OHDFewsAdapter.java.
     * 
     * @throws Exception
     */
    @Override
    public void extractValuesFromMap() throws Exception
    {

        _adimp = getDoubleDataParameter(SacSmaModelConstants.ADIMP_TAG);
        _efc = getDoubleDataParameter(SacSmaModelConstants.EFC_TAG);

        //fill _etDemand[]
        if(isParamExisting(SacSmaModelConstants.ET_DEMAND_CURVE_TAG))
        {
            _etDemand = super.getDoubleArrayParameter(SacSmaModelConstants.ET_DEMAND_CURVE_TAG);
        }
        else
        {
            _etDemand[0] = getDoubleDataParameter(SacSmaModelConstants.ETTJAN_TAG);
            _etDemand[1] = getDoubleDataParameter(SacSmaModelConstants.ETTFEB_TAG);
            _etDemand[2] = getDoubleDataParameter(SacSmaModelConstants.ETTMAR_TAG);
            _etDemand[3] = getDoubleDataParameter(SacSmaModelConstants.ETTAPR_TAG);
            _etDemand[4] = getDoubleDataParameter(SacSmaModelConstants.ETTMAY_TAG);
            _etDemand[5] = getDoubleDataParameter(SacSmaModelConstants.ETTJUN_TAG);
            _etDemand[6] = getDoubleDataParameter(SacSmaModelConstants.ETTJUL_TAG);
            _etDemand[7] = getDoubleDataParameter(SacSmaModelConstants.ETTAUG_TAG);
            _etDemand[8] = getDoubleDataParameter(SacSmaModelConstants.ETTSEP_TAG);
            _etDemand[9] = getDoubleDataParameter(SacSmaModelConstants.ETTOCT_TAG);
            _etDemand[10] = getDoubleDataParameter(SacSmaModelConstants.ETTNOV_TAG);
            _etDemand[11] = getDoubleDataParameter(SacSmaModelConstants.ETTDEC_TAG);
        }

        _lzfpm = getDoubleDataParameter(SacSmaModelConstants.LZFPM_TAG);
        _lzfsm = getDoubleDataParameter(SacSmaModelConstants.LZFSM_TAG);
        _lzpk = getDoubleDataParameter(SacSmaModelConstants.LZPK_TAG);
        _lzsk = getDoubleDataParameter(SacSmaModelConstants.LZSK_TAG);
        _lztwm = getDoubleDataParameter(SacSmaModelConstants.LZTWM_TAG);

        if(isParamExisting(MODEL_RUN_TYPE_TAG))
        {
            _modelRunType = getStringDataParameter(SacSmaHTModelParameters.MODEL_RUN_TYPE_TAG);
        }

        _pctim = getDoubleDataParameter(SacSmaModelConstants.PCTIM_TAG);
        _peadj = getDoubleDataParameter(SacSmaModelConstants.PEADJ_TAG);
        _pfree = getDoubleDataParameter(SacSmaModelConstants.PFREE_TAG);
        _pxadj = getDoubleDataParameter(SacSmaModelConstants.PXADJ_TAG);
        _rexp = getDoubleDataParameter(SacSmaModelConstants.REXP_TAG);
        _riva = getDoubleDataParameter(SacSmaModelConstants.RIVA_TAG);
        _rserv = getDoubleDataParameter(SacSmaModelConstants.RSERV_TAG);
        _side = getDoubleDataParameter(SacSmaModelConstants.SIDE_TAG);
        _stxt = getIntDataParameter(SacSmaHTModelParameters.STXT_TAG);
        _tbot = getDoubleDataParameter(SacSmaHTModelParameters.TBOT_TAG);
        _uzfwm = getDoubleDataParameter(SacSmaModelConstants.UZFWM_TAG);
        _uzk = getDoubleDataParameter(SacSmaModelConstants.UZK_TAG);
        _uztwm = getDoubleDataParameter(SacSmaModelConstants.UZTWM_TAG);
        _zperc = getDoubleDataParameter(SacSmaModelConstants.ZPERC_TAG);

        if(isParamExisting(SacSmaModelConstants.MAPE_INPUT_TAG))
        {
            _useMapeInput = getBooleanDataParameter(SacSmaModelConstants.MAPE_INPUT_TAG);
        }

        if(isParamExisting(USER_DEFINE_SOIL_TAG))
        {
            _userDefinedSoilLayerDepths = getDoubleArrayParameter(SacSmaHTModelParameters.USER_DEFINE_SOIL_TAG);
        }

    }

    /**
     * This method does a little more than just validate, it also adjusts the parameters.
     */
    public void validateParams() throws ModelException, Exception
    {

        adjust();

        if(!(_modelRunType.equals(SacSmaHTConstants.FULL_RUN_STR))
            && !(_modelRunType.equals(SacSmaHTConstants.FULL_RUN_UNAFFECTED_RUNOFF_STR))
            && !(_modelRunType.equals(SacSmaHTConstants.ONLY_SOIL_RUN_STR)))
        {
            throw new ModelException("Invalid MODEL_RUN_TYPE_TAG - valid values  are SOIL_AND_HEAT_TRANSFER,ONLY_SOIL and SOIL_AND_HEAT_TRANSFER_UNAFFECTED_RUNOFF");
        }

        _logger.log(Logger.DEBUG, "Model Run Type used = " + getModelRunType());
    }

    private void adjust()
    {

        double oldValue;

        if((oldValue = getPctim()) > 1.0)
        {
            setPctim(1.0);
            _logger.log(Logger.DEBUG, "Value of PCTIM was changed from " + oldValue + " to " + 1.0);
        }

        if((oldValue = getAdimp()) > 1.0 - getPctim())
        {
            setAdimp(1.0 - getPctim());
            _logger.log(Logger.DEBUG, "Value of ADIMP was changed from " + oldValue + " to " + (1.0 - getPctim()));
        }

        if((oldValue = getRiva()) > 1.0)
        {
            setRiva(1.0);
            _logger.log(Logger.DEBUG, "Value of RIVA was changed from " + oldValue + " to " + 1.0);
        }

        if((oldValue = getEfc()) > 1.0)
        {
            setEfc(1.0);
            _logger.log(Logger.DEBUG, "Value of EFC was changed from " + oldValue + " to " + 1.0);
        }

        if((oldValue = getUztwm()) < 0.1)
        {
            setUztwm(0.1);
            _logger.log(Logger.DEBUG, "Value of UZTWM was changed from " + oldValue + " to " + 0.1);
        }

        if((oldValue = getUzfwm()) < 0.1)
        {
            setUzfwm(0.1);
            _logger.log(Logger.DEBUG, "Value of UZFWM was changed from " + oldValue + " to " + 0.1);
        }

        if((oldValue = getLztwm()) < 0.1)
        {
            setLztwm(0.1);
            _logger.log(Logger.DEBUG, "Value of LZTWM was changed from " + oldValue + " to " + 0.1);
        }

        if((oldValue = getLzfsm()) < 0.1)
        {
            setLzfsm(0.1);
            _logger.log(Logger.DEBUG, "Value of LZFSM was changed from " + oldValue + " to " + 0.1);
        }

        if((oldValue = getLzfpm()) < 0.1)
        {
            setLzfpm(0.1);
            _logger.log(Logger.DEBUG, "Value of LZFPM was changed from " + oldValue + " to " + 0.1);
        }

        if((oldValue = getUzk()) > 1.0)
        {
            setUzk(1.0);
            _logger.log(Logger.DEBUG, "Value of UZK was changed from " + oldValue + " to " + 1.0);
        }

        if((oldValue = getLzsk()) > 1.0)
        {
            setLzsk(1.0);
            _logger.log(Logger.DEBUG, "Value of LZSK was changed from " + oldValue + " to " + 1.0);
        }

        if((oldValue = getLzpk()) > 1.0)
        {
            setLzpk(1.0);
            _logger.log(Logger.DEBUG, "Value of LZPK was changed from " + oldValue + " to " + 1.0);
        }

        if((oldValue = getPfree()) > 1.0)
        {
            setPfree(1.0);
            _logger.log(Logger.DEBUG, "Value of PFREE was changed from " + oldValue + " to " + 1.0);
        }

        if((oldValue = getRserv()) > 1.0)
        {
            setRserv(1.0);
            _logger.log(Logger.DEBUG, "Value of RSERV was changed from " + oldValue + " to " + 1.0);
        }
    }

    public double getAdimp()
    {
        return _adimp;
    }

    public void setAdimp(final double adimp)
    {
        _adimp = adimp;
    }

    /**
     * @return the _efc
     */
    public double getEfc()
    {
        return _efc;
    }

    public void setEfc(final double efc)
    {
        _efc = efc;
    }

//    /**
//     * Set the user defined soil layers. If not values are set in the array it will calculate the default values and set
//     * them.
//     * 
//     * @param userDefinedSoilLayer
//     * @throws SacSmaHTInputException
//     */
//    public void setUserDefinedSoilLayerDepths(double[] userDefinedSoilLayer) throws SacSmaHTInputException
//    {
//        if(userDefinedSoilLayer != null && userDefinedSoilLayer.length > 0)
//        {
//            _numberOfSoilLayers = userDefinedSoilLayer.length;
//            if(userDefinedSoilLayer.length > SacSmaHTConstants.MAX_NUMBER_OF_USER_DEFINED_PROFILES)
//            {
//                throw new SacSmaHTInputException("Error: Maximum number of user defined soil layers exceeded ");
//            }
//            Arrays.sort(userDefinedSoilLayer);
//            // Validate values.
//            for(int i = 0; i < userDefinedSoilLayer.length; i++)
//            {
//                if(!((java.lang.Math.abs(userDefinedSoilLayer[i]) >= 0.0) && (java.lang.Math.abs(userDefinedSoilLayer[i]) <= -SacSmaHTConstants.DEPTH_AT_BOTTOM_LAYER_IN_METERS)))
//                {
//                    throw new SacSmaHTInputException("Error: Value of user defined soil layers exceeded not in range  0 <= "
//                        + java.lang.Math.abs(userDefinedSoilLayer[i])
//                        + " <= "
//                        + SacSmaHTConstants.DEPTH_AT_BOTTOM_LAYER_IN_METERS);
//                }
//            } //close for block
//            super.insertParameter(SacSmaHTModelParameters.USER_DEFINE_SOIL_TAG, userDefinedSoilLayer);
//        } //close if block
//        else
//        {
//            final double supm = getUztwm() + getUzfwm();
//            final int nsl = SacSmaHTConstants.MAX_NUMBER_OF_USER_DEFINED_PROFILES;
//            final double[] zSoil = new double[SacSmaHTConstants.MAX_NUMBER_OF_USER_DEFINED_PROFILES];
//
//            for(int i = 0; i < nsl; i++)
//            {
//                zSoil[i] = -999.;
//            }
//            //
//            final SacSmaHTModelDriver modelDriver = new SacSmaHTModelDriver();
//            try
//            {
//                modelDriver.setModelParameters(this);
//                _numberOfSoilLayers = modelDriver.soilPar1(supm, zSoil);
//                userDefinedSoilLayer = new double[_numberOfSoilLayers];
//                userDefinedSoilLayer[0] = -0.5 * zSoil[0];
//
//                for(int i = 1; i < _numberOfSoilLayers; i++)
//                {
//                    userDefinedSoilLayer[i] = -0.5 * (zSoil[i - 1] + zSoil[i]);
//                }
//            }
//            catch(final Exception e)
//            {
//                throw new SacSmaHTInputException("Error" + e.toString());
//            }
//
//            super.insertParameter(SacSmaHTModelParameters.USER_DEFINE_SOIL_TAG, userDefinedSoilLayer);
//        }
//    }

    /**
     * This method return an array of doubles, if there are not values in the Parameter object it return an array with 0
     * elements. This returned array is used without validation, assuming it will contain 0 or more elements. Returning
     * a null value will break existing code.
     * 
     * @return a double array of 0 or more elements.
     */
    public double[] getUserDefinedSoilLayerDepths()
    {
        return _userDefinedSoilLayerDepths;
    }

    public double getLzfpm()
    {
        return _lzfpm;
    }

    public void setLzfpm(final double lzfpm)
    {
        _lzfpm = lzfpm;
    }

    public double getLzfsm()
    {
        return _lzfsm;
    }

    public void setLzfsm(final double lzfsm)
    {
        _lzfsm = lzfsm;
    }

    public double getLzpk()
    {
        return _lzpk;
    }

    public void setLzpk(final double lzpk)
    {
        _lzpk = lzpk;
    }

    public double getLzsk()
    {
        return _lzsk;
    }

    public void setLzsk(final double lzsk)
    {
        _lzsk = lzsk;
    }

    public double getLztwm()
    {
        return _lztwm;
    }

    public void setLztwm(final double lztwm)
    {
        _lztwm = lztwm;
    }

    /**
     * @return - one of SOIL_AND_HEAT_TRANSFER, ONLY_SOIL, SOIL_AND_HEAT_TRANSFER_UNAFFECTED_RUNOFF
     */
    public String getModelRunType()
    {
        return _modelRunType;
    }

    public void setModelRunType(final String modelRunType)
    {
        _modelRunType = modelRunType;
    }

    public double getPctim()
    {
        return _pctim;
    }

    public void setPctim(final double pctim)
    {
        _pctim = pctim;
    }

    /**
     * @return the _peAdj
     */
    public double getPeAdj()
    {
        return _peadj;
    }

    public void setPeAdj(final double peAdj)
    {
        _peadj = peAdj;
    }

    public double getPfree()
    {
        return _pfree;
    }

    public void setPfree(final double pfree)
    {
        _pfree = pfree;
    }

    /**
     * @return the _pxAdj
     */
    public double getPxAdj()
    {
        return _pxadj;
    }

    public void setPxAdj(final double pxAdj)
    {
        _pxadj = pxAdj;
    }

    public double getRexp()
    {
        return _rexp;
    }

    public void setRexp(final double rexp)
    {
        _rexp = rexp;
    }

    /**
     * @return the _riva
     */
    public double getRiva()
    {
        return _riva;
    }

    public void setRiva(final double riva)
    {
        _riva = riva;
    }

    public double getRserv()
    {
        return _rserv;
    }

    public void setRserv(final double rserv)
    {
        _rserv = rserv;
    }

    public double getSide()
    {
        return _side;
    }

    public void setSide(final double side)
    {
        _side = side;
    }

    /**
     * @return the _stxt as int.
     */
    public int getStxt()
    {
        return _stxt;
    }

    public void setStxt(final int stxt)
    {
        _stxt = stxt;
    }

    /**
     * @return the _tbot
     */
    public double getTbot()
    {
        return _tbot;
    }

    public void setTbot(final double tbot)
    {
        _tbot = tbot;
    }

    public double getRunoffEffect()
    {
        if(this.getModelRunType().equals(SacSmaHTConstants.FULL_RUN_UNAFFECTED_RUNOFF_STR))
        {
            return 0.0;
        }

        return 8.0;
    }

    public double getUzfwm()
    {
        return _uzfwm;
    }

    public void setUzfwm(final double uzfwm)
    {
        _uzfwm = uzfwm;
    }

    public double getUzk()
    {
        return _uzk;
    }

    public void setUzk(final double uzk)
    {
        _uzk = uzk;
    }

    public double getUztwm()
    {
        return _uztwm;
    }

    public void setUztwm(final double uztwm)
    {
        _uztwm = uztwm;
    }

    public double getZperc()
    {
        return _zperc;
    }

    public void setZperc(final double zperc)
    {
        _zperc = zperc;
    }

    public double getEtDemand(final int i)
    {
        return _etDemand[i];
    }

    public double[] getEtDemand()
    {
        return _etDemand;
    }

    /**
     * Default is false, which means that MAPE ts is not used and etDemand 12 values represent ET-demand 16th of each
     * month, daily values are computed by linear interpolation; In the condition of true, etDemand 12 values becomes
     * PE-adjustment factor for the input MATE ts
     */
    public boolean useMAPEInput()
    {
        return _useMapeInput;
    }

    public void setMapeUseOption(final boolean choice)
    {
        _useMapeInput = choice;
    }
} //close class
