package ohd.hseb.ohdmodels.sacsmaHT;

import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.ModelException;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.Parameters;
import ohd.hseb.util.fews.ohdmodels.ModelParameters;
import ohd.hseb.util.fews.ohdmodels.ModelState;

/**
 * This class holds all the SacSmaHT Model state variables. It is a subclass of the abstract class ModelState and
 * interface IModelState. The state variable values are loaded from a text file with properties format through
 * {@link #loadState(String, Logger)}. The values are are initially loaded into the Map<String, Object> _statesMap in
 * ModelState, then the values are extracted to the corresponding instance variables.
 * 
 * @author CHPS
 */

final public class SacSmaHTModelState extends ModelState
{
    //the following states should be always present:
    private double _uztwc;
    private double _uzfwc;
    private double _lztwc;
    private double _lzfsc;
    private double _lzfpc;
    private double _adimc;
    private double _initTemp;
    private double _initSnowDepth;
    private double _initSwe;
    private int _numSoilLayer;

    private final double[] _soilLayerDepth = new double[SacSmaHTConstants.MAX_NUMBER_OF_SOIL_LAYERS];
    private final double[] _soilLayerMoistContent = new double[SacSmaHTConstants.MAX_NUMBER_OF_SOIL_LAYERS];
    private final double[] _soilLayerTemp = new double[SacSmaHTConstants.MAX_NUMBER_OF_SOIL_LAYERS];
    private final double[] _soilLayerUnfrozenWater = new double[SacSmaHTConstants.MAX_NUMBER_OF_SOIL_LAYERS];

    //these states may not always be present:
    private double _uztwh;
    private double _uzfwh;
    private double _lztwh;
    private double _lzfsh;
    private double _lzfph;
    private double _fgix;

    /*
     * This member is set if the client knows that this state is meant to be a cold start state, regardless of layer
     * information stored with the state. Currently used by the NWSRFS XML parser when it reads the Default (or
     * coldstart) state. There is then no dependency on the layer information re. the determination of cold vs. warm
     * start
     */
    private boolean _isColdStart = true;

    public SacSmaHTModelState()
    {
        super();
    }

    /**
     * Extract values from super._statesMap into instance variables. The values in super._statessMap are set by parsing
     * the initial state file in OHDFewsAdapter.java.
     */
    @Override
    public void extractValuesFromMap() throws Exception
    {
        _uztwc = getDoubleDataState(SacSmaHTConstants.UPPER_ZONE_TENSION_WATER_CONTENTS);

        _uzfwc = getDoubleDataState(SacSmaHTConstants.UPPER_ZONE_FREE_WATER_CONTENTS);

        _lztwc = getDoubleDataState(SacSmaHTConstants.LOWER_ZONE_TENSION_WATER_CONTENTS);

        _lzfsc = getDoubleDataState(SacSmaHTConstants.LOWER_ZONE_FREE_SECONDARY_WATER_CONTENTS);

        _lzfpc = getDoubleDataState(SacSmaHTConstants.LOWER_ZONE_FREE_PRIMARY_WATER_CONTENTS);

        _adimc = getDoubleDataState(SacSmaHTConstants.ADDITIONAL_IMPERVIOUS_WATER_CONTENTS);

        _initTemp = getDoubleDataState(SacSmaHTConstants.INIT_AIR_TEMPERATURE_TAG);

        _numSoilLayer = getIntDataState(SacSmaHTConstants.NUMBER_SOIL_LAYER_TAG);

        //if reach here, all states above are present in statesI.txt. If not cold start, the following are mandatory
        if(_numSoilLayer > 0)
        {
            _uztwh = getDoubleDataState(SacSmaHTConstants.UZTWH_TAG);

            _uzfwh = getDoubleDataState(SacSmaHTConstants.UZFWH_TAG);

            _lztwh = getDoubleDataState(SacSmaHTConstants.LZTWH_TAG);

            _lzfsh = getDoubleDataState(SacSmaHTConstants.LZFSH_TAG);

            _lzfph = getDoubleDataState(SacSmaHTConstants.LZFPH_TAG);

            _fgix = getDoubleDataState(OHDConstants.FINDEX_TAG);

            for(int i = 0; i < _numSoilLayer; i++)
            {
                _soilLayerDepth[i] = getDoubleDataState(SacSmaHTConstants.LAYER_DEPTH_TAG + i);

                _soilLayerMoistContent[i] = getDoubleDataState(SacSmaHTConstants.SOIL_LAYER_MOISTURE_CONTENT_TAG + i);

                _soilLayerTemp[i] = getDoubleDataState(SacSmaHTConstants.SOIL_LAYER_TEMPERATURE_TAG + i);

                _soilLayerUnfrozenWater[i] = getDoubleDataState(SacSmaHTConstants.SOIL_LAYER_WATER_CONTENT_TAG + i);

            }

        }

        //at least one of the two(INIT_SNOW_DEPTH_TAG, INIT_SNOW_WATER_EQUIVALENT_TAG) must be present; if both missing, validateState() will throw Exception
        if(isStateExisting(SacSmaHTConstants.INIT_SNOW_DEPTH_TAG))
        {
            _initSnowDepth = getDoubleDataState(SacSmaHTConstants.INIT_SNOW_DEPTH_TAG);
        }

        if(isStateExisting(SacSmaHTConstants.INIT_SNOW_WATER_EQUIVALENT_TAG))
        {
            _initSwe = getDoubleDataState(SacSmaHTConstants.INIT_SNOW_WATER_EQUIVALENT_TAG);
        }
    }

    public void validateState(final Parameters params) throws Exception
    {
        final SacSmaHTModelParameters sacParams = (SacSmaHTModelParameters)params;
        Object sweValue, snowDepthValue;

        sweValue = getStateMap().get(SacSmaHTConstants.INIT_SNOW_WATER_EQUIVALENT_TAG);
        snowDepthValue = getStateMap().get(SacSmaHTConstants.INIT_SNOW_DEPTH_TAG);

        // validation specific to SAC-HT - both Snow depth and SWE cannot be
        // missing, if one is the other is inferred
        // (logic moved here from frdFg1
        if(sweValue == null && snowDepthValue == null)
        {
            throw new ModelException("Error: Provide at least one of INIT_SNOW_DEPTH and INIT_SNOW_WATER_EQUIVALENT in the input states file");
        }
        else if(snowDepthValue == null)
        {
            // CVK set default snow depth assuming density =0.2
            setInitSnowDepth(0.1 * Double.parseDouble(sweValue.toString()) / 0.2);
            _logger.log(Logger.WARNING, "Initial Snow Depth was set to " + getInitSnowDepth() + " using the value of "
                + sweValue + " given for initial snow water equivalent");
        }
        else
        // sweValue == null
        {
            // get SWE from snowdepth by inverting equation above
            setInitSnowWaterEquivalent(2 * Double.parseDouble(snowDepthValue.toString()));
            _logger.log(Logger.WARNING, "Initial Snow Water Equivalent was set to " + getInitSnowWaterEquivalent()
                + " using the the value of " + snowDepthValue + " given for initial snow depth");
        }

        if(isStateExisting(SacSmaHTConstants.NUMBER_SOIL_LAYER_TAG) && getNumberOfSoilLayers() > 0)
        {
            _isColdStart = false;
        }

        if(validateNumberOfLayers() == false)
        {
            throw new ModelException("Error: The number of soil layers is different for each attribute");
        }

        _logger.log(Logger.DEBUG, "Number of soil layers = " + getNumberOfSoilLayers());

        // what used to be in adjustStates() - first check that sacParams valid
        // in case this is called without setting
        if(sacParams != null)
        {
            if(getUztwc() > sacParams.getUztwm())
            {
                _logger.log(Logger.WARNING,
                            "For the initial state, the value for Upper Zone Tension Water Contents was changed from "
                                + getUztwc() + " to " + sacParams.getUztwm());
                setUztwc(sacParams.getUztwm());
            }

            if(getUzfwc() > sacParams.getUzfwm())
            {
                _logger.log(Logger.WARNING,
                            "For the initial state, the value for Upper Zone Free Water Contents was changed from "
                                + getUzfwc() + " to " + sacParams.getUzfwm());
                setUzfwc(sacParams.getUzfwm());

            }

            if(getLztwc() > sacParams.getLztwm())
            {
                _logger.log(Logger.WARNING,
                            "For the initial state, the value for Lower Zone Tension Water Contents was changed from "
                                + getLztwc() + " to " + sacParams.getLztwm());
                setLztwc(sacParams.getLztwm());
            }

            if(getLzfsc() > sacParams.getLzfsm())
            {
                _logger.log(Logger.WARNING,
                            "For the initial state, the value for Lower Zone Free Secondary Contents was changed from "
                                + getLzfsc() + " to " + sacParams.getLzfsm());
                setLzfsc(sacParams.getLzfsm());
            }

            if(getLzfpc() > sacParams.getLzfpm())
            {
                _logger.log(Logger.WARNING,
                            "For the initial state, the value for Lower Zone Free Primary Contents was changed from "
                                + getLzfpc() + " to " + sacParams.getLzfpm());
                setLzfpc(sacParams.getLzfpm());
            }

            if(getAdimc() > sacParams.getUztwm() + sacParams.getLztwm())
            {
                _logger.log(Logger.WARNING,
                            "For the initial state, the value for additional impervious contents was changed from "
                                + getAdimc() + " to " + (sacParams.getUztwm() + sacParams.getLztwm()));
                setAdimc(sacParams.getUztwm() + sacParams.getLztwm());
            }

            if(getAdimc() < getUztwc())
            {
                _logger.log(Logger.WARNING,
                            "For the initial state, the value for additional impervious contents was changed from "
                                + getAdimc() + " to " + getUztwc());
                setAdimc(getUztwc());
            }
        }
        else
        {
            throw new ModelException("The paramter SacSmaModelParameter in validateState() is null.");
        }
    }

    /**
     * Over-written the super class method. This method is used to output states into the generated state txt file. The
     * units are always in METRIC.
     */
    @Override
    protected String getStringsFromState()
    {
        final StringBuilder resultStr = new StringBuilder();

        resultStr.append(SacSmaHTConstants.ADDITIONAL_IMPERVIOUS_WATER_CONTENTS + "=" + _adimc + OHDConstants.NEW_LINE);

        resultStr.append(OHDConstants.FINDEX_TAG + "=" + _fgix + OHDConstants.NEW_LINE);

        resultStr.append(SacSmaHTConstants.INIT_AIR_TEMPERATURE_TAG + "=" + _initTemp + OHDConstants.NEW_LINE);

        resultStr.append(SacSmaHTConstants.INIT_SNOW_DEPTH_TAG + "=" + _initSnowDepth + OHDConstants.NEW_LINE);

        resultStr.append(SacSmaHTConstants.INIT_SNOW_WATER_EQUIVALENT_TAG + "=" + _initSwe + OHDConstants.NEW_LINE);

        resultStr.append(SacSmaHTConstants.INIT_SNOW_DEPTH_TAG + "=" + _initSnowDepth + OHDConstants.NEW_LINE);

        resultStr.append(SacSmaHTConstants.LZFPH_TAG + "=" + _lzfph + OHDConstants.NEW_LINE);

        resultStr.append(SacSmaHTConstants.LOWER_ZONE_FREE_PRIMARY_WATER_CONTENTS + "=" + _lzfpc
            + OHDConstants.NEW_LINE);

        resultStr.append(SacSmaHTConstants.LOWER_ZONE_FREE_SECONDARY_WATER_CONTENTS + "=" + _lzfsc
            + OHDConstants.NEW_LINE);

        resultStr.append(SacSmaHTConstants.LZFSH_TAG + "=" + _lzfsh + OHDConstants.NEW_LINE);

        resultStr.append(SacSmaHTConstants.LZTWH_TAG + "=" + _lztwh + OHDConstants.NEW_LINE);

        resultStr.append(SacSmaHTConstants.LOWER_ZONE_TENSION_WATER_CONTENTS + "=" + _lztwc + OHDConstants.NEW_LINE);

        resultStr.append(SacSmaHTConstants.UZFWH_TAG + "=" + _uzfwh + OHDConstants.NEW_LINE);

        resultStr.append(SacSmaHTConstants.UPPER_ZONE_FREE_WATER_CONTENTS + "=" + _uzfwc + OHDConstants.NEW_LINE);

        resultStr.append(SacSmaHTConstants.UZTWH_TAG + "=" + _uztwh + OHDConstants.NEW_LINE);

        resultStr.append(SacSmaHTConstants.UPPER_ZONE_TENSION_WATER_CONTENTS + "=" + _uztwc + OHDConstants.NEW_LINE);

        resultStr.append(SacSmaHTConstants.NUMBER_SOIL_LAYER_TAG + "=" + _numSoilLayer + OHDConstants.NEW_LINE);

        for(int i = 0; i < _numSoilLayer; i++)
        {
            resultStr.append(SacSmaHTConstants.LAYER_DEPTH_TAG + i + "=" + _soilLayerDepth[i] + OHDConstants.NEW_LINE);

            resultStr.append(SacSmaHTConstants.SOIL_LAYER_MOISTURE_CONTENT_TAG + i + "=" + _soilLayerMoistContent[i]
                + OHDConstants.NEW_LINE);

            resultStr.append(SacSmaHTConstants.SOIL_LAYER_TEMPERATURE_TAG + i + "=" + _soilLayerTemp[i]
                + OHDConstants.NEW_LINE);

            resultStr.append(SacSmaHTConstants.SOIL_LAYER_WATER_CONTENT_TAG + i + "=" + _soilLayerUnfrozenWater[i]
                + OHDConstants.NEW_LINE);

        }

        //units are in METRIC always
        resultStr.append(OHDConstants.UNIT_TAG + "=" + OHDConstants.UNIT_METRIC + OHDConstants.NEW_LINE);

        return resultStr.toString();
    }

    /**
     * A generic method to re-set 7 kinds of state values: 6 SacSma state plus frozen ground state FGIX.
     * 
     * @param stateName - state string name
     * @param newValue - new value of the state
     */
    public void updateState(final String stateName, final double newValue)
    {

        if(SacSmaHTConstants.UPPER_ZONE_TENSION_WATER_CONTENTS.contains(stateName.toUpperCase()))
        {
            this.setUztwc(newValue);
        }
        else if(SacSmaHTConstants.UPPER_ZONE_FREE_WATER_CONTENTS.contains(stateName.toUpperCase()))
        {
            this.setUzfwc(newValue);
        }
        else if(SacSmaHTConstants.LOWER_ZONE_TENSION_WATER_CONTENTS.contains(stateName.toUpperCase()))
        {
            this.setLztwc(newValue);
        }
        else if(SacSmaHTConstants.LOWER_ZONE_FREE_SECONDARY_WATER_CONTENTS.contains(stateName.toUpperCase()))
        {
            this.setLzfsc(newValue);
        }
        else if(SacSmaHTConstants.LOWER_ZONE_FREE_PRIMARY_WATER_CONTENTS.contains(stateName.toUpperCase()))
        {
            this.setLzfpc(newValue);
        }
        else if(SacSmaHTConstants.ADDITIONAL_IMPERVIOUS_WATER_CONTENTS.contains(stateName.toUpperCase()))
        {
            this.setAdimc(newValue);
        }
        else if(stateName.equals(OHDConstants.FINDEX_TAG))
        {
            this.setFindex(newValue);
        }

        // if state name does not equal to one of the 7 names above, this method
        // cannot re-set its value.
    }

    public void setNumberOfSoilLayers(final int numberOfSoilLayers)
    {
        _numSoilLayer = numberOfSoilLayers;
    }

    public int getNumberOfSoilLayers()
    {
        return _numSoilLayer;
    }

    /**
     * @param i Layer for which depth to be returned
     * @return Depth at layer
     */
    public double getSoilLayerDepth(final int i)
    {
        return _soilLayerDepth[i];
    }

    /**
     * Return the array of soil layer depths. Even though the number of elements of the array is
     * {@link SacSmaHTConstants#MAX_NUMBER_OF_SOIL_LAYERS}, only {@link #_numSoilLayer} elements are correct, the rest
     * are just 0.0.
     */
    double[] getSoilLayerDepthArray()
    {
        return _soilLayerDepth;
    }

    /**
     * @param i Layer for which temperature to be returned
     * @return Temperature at layer
     */
    public double getSoilLayerTemperature(final int i)
    {
        return _soilLayerTemp[i];
    }

    /**
     * @param i Layer for which soil layer moisture content to be returned
     * @return Soil layer moisture content at layer
     */
    public double getSoilLayerMoistureContent(final int i)
    {
        return _soilLayerMoistContent[i];
    }

    /**
     * @param i Layer for which soil layer unfrozen water to be returned
     * @return Unfrozen water at layer
     */
    public double getSoilLayerUnfrozenWater(final int i)
    {
        return _soilLayerUnfrozenWater[i];
    }

    /**
     * Sets the temperature for a given soil layer
     * 
     * @param i Layer for which to set the temperature
     * @param Temperature value to set
     */
    public void setSoilLayerTemperature(final int i, final double temp)
    {
        _soilLayerTemp[i] = temp;
    }

    /**
     * Sets the unfrozen water content for a given soil layer
     * 
     * @param i Layer for which to set the unfrozen water content
     * @param Unfrozen water content value to set
     */
    public void setSoilLayerUnfrozenWater(final int i, final double water)
    {
        _soilLayerUnfrozenWater[i] = water;
    }

    /**
     * Sets the total soil layer moisture content for a given soil layer
     * 
     * @param i Layer for which to set the total moisture content
     * @param Total Moisture content value to set
     */
    public void setSoilLayerMoistureContent(final int i, final double moisture)
    {
        _soilLayerMoistContent[i] = moisture;
    }

    /**
     * Sets the total layer depth for a given soil layer
     * 
     * @param i Layer for which to set the depth
     * @param Depth value to set
     */
    public void setSoilLayerDepth(final int i, final double depth)
    {
        _soilLayerDepth[i] = depth;
    }

    public void setUztwc(final double uztwc)
    {
        _uztwc = uztwc;
    }

    public double getUztwc()
    {
        return _uztwc;
    }

    public void setUzfwc(final double uzfwc)
    {
        _uzfwc = uzfwc;
    }

    public double getUzfwc()
    {
        return _uzfwc;
    }

    public void setLztwc(final double lztwc)
    {
        _lztwc = lztwc;
    }

    public double getLztwc()
    {
        return _lztwc;
    }

    public void setLzfsc(final double lzfsc)
    {
        _lzfsc = lzfsc;
    }

    public double getLzfsc()
    {
        return _lzfsc;
    }

    public void setLzfpc(final double lzfpc)
    {
        _lzfpc = lzfpc;
    }

    public double getLzfpc()
    {
        return _lzfpc;
    }

    public void setAdimc(final double adimc)
    {
        _adimc = adimc;
    }

    public double getAdimc()
    {
        return _adimc;
    }

    public void setFindex(final double findex)
    {
        _fgix = findex;
    }

    public double getFindex()
    {
        return _fgix;
    }

    public void setUztwh(final double uztwh)
    {
        _uztwh = uztwh;
    }

    public double getUztwh()
    {
        return _uztwh;
    }

    public void setUzfwh(final double uzfwh)
    {
        _uzfwh = uzfwh;
    }

    public double getUzfwh()
    {
        return _uzfwh;
    }

    public void setLztwh(final double lztwh)
    {
        _lztwh = lztwh;
    }

    public double getLztwh()
    {
        return _lztwh;
    }

    public void setLzfsh(final double lzfsh)
    {
        _lzfsh = lzfsh;
    }

    public double getLzfsh()
    {
        return _lzfsh;
    }

    public void setLzfph(final double lzfph)
    {
        _lzfph = lzfph;
    }

    public double getLzfph()
    {
        return _lzfph;
    }

    public void setInitAirTemperature(final double at)
    {
        _initTemp = at;
    }

    public double getInitAirTemperature()
    {
        return _initTemp;
    }

    public void setInitSnowWaterEquivalent(final double swe)
    {
        _initSwe = swe;
    }

    public double getInitSnowWaterEquivalent()
    {
        return _initSwe;
    }

    public void setInitSnowDepth(final double sd)
    {
        _initSnowDepth = sd;
    }

    public double getInitSnowDepth()
    {
        return _initSnowDepth;
    }

    @Override
    public void insertStateAsString(final String tag, final String valueStr)
    {

        if(valueStr.equals(String.valueOf(OHDConstants.MISSING_DATA)))
        {
            return;
        }
        super.insertStateAsString(tag, valueStr);
    }

    /**
     * Validate the four entries {@link SacSmaHTConstants#LAYER_DEPTH_TAG},
     * {@link SacSmaHTConstants#SOIL_LAYER_TEMPERATURE_TAG}, {@link SacSmaHTConstants#SOIL_LAYER_MOISTURE_CONTENT_TAG},
     * {@link SacSmaHTConstants#SOIL_LAYER_WATER_CONTENT_TAG} to make sure that either none of them or equal number of
     * them show up in statesI.txt.
     */
    private boolean validateNumberOfLayers() throws Exception
    {
        int countOfParametersAtLayer;

        /*------------check the 4 entries are always present in statesI.txt --------- */
        for(int i = 0; i < SacSmaHTConstants.MAX_NUMBER_OF_SOIL_LAYERS; i++)
        {
            countOfParametersAtLayer = 0;

            if(isStateExisting(SacSmaHTConstants.LAYER_DEPTH_TAG + i))
            {
                countOfParametersAtLayer++;
            }
            if(isStateExisting(SacSmaHTConstants.SOIL_LAYER_TEMPERATURE_TAG + i))
            {
                countOfParametersAtLayer++;
            }
            if(isStateExisting(SacSmaHTConstants.SOIL_LAYER_MOISTURE_CONTENT_TAG + i))
            {
                countOfParametersAtLayer++;
            }
            if(isStateExisting(SacSmaHTConstants.SOIL_LAYER_WATER_CONTENT_TAG + i))
            {
                countOfParametersAtLayer++;
            }

            if(countOfParametersAtLayer != 0 && countOfParametersAtLayer != 4)
            {
                return false; //some of the 4 entries are missing in statesI.txt
            }

        }

        /*
         * 2 cases reach here: 1)all those 4 type entries are completely missing(countOfParametersAtLayer = 0); 2) all 4
         * type entries are present. Both cases are acceptable, so return true.
         */
        return true;

    }

    public boolean isColdStart()
    {
        return _isColdStart;
    }

    public void setIsColdStart(final boolean coldStart)
    {
        _isColdStart = coldStart;
    }

//    public void removeWarmStartStates()
//    {
//        deleteState(SacSmaHTConstants.UZTWH_TAG);
//        deleteState(SacSmaHTConstants.UZTWH_TAG);
//        deleteState(SacSmaHTConstants.UZFWH_TAG);
//        deleteState(SacSmaHTConstants.LZTWH_TAG);
//        deleteState(SacSmaHTConstants.LZFPH_TAG);
//        deleteState(SacSmaHTConstants.FINDEX_TAG);
//
//        for(int i = 0; i < SacSmaHTConstants.MAX_NUMBER_OF_SOIL_LAYERS; i++)
//        {
//            if(getStateAsString(SacSmaHTConstants.LAYER_DEPTH_TAG + i) != null)
//            {
//                deleteState(SacSmaHTConstants.LAYER_DEPTH_TAG + i);
//            }
//            if(getStateAsString(SacSmaHTConstants.SOIL_LAYER_TEMPERATURE_TAG + i) != null)
//            {
//                deleteState(SacSmaHTConstants.SOIL_LAYER_TEMPERATURE_TAG + i);
//            }
//            if(getStateAsString(SacSmaHTConstants.SOIL_LAYER_MOISTURE_CONTENT_TAG + i) != null)
//            {
//                deleteState(SacSmaHTConstants.SOIL_LAYER_MOISTURE_CONTENT_TAG + i);
//            }
//            if(getStateAsString(SacSmaHTConstants.SOIL_LAYER_WATER_CONTENT_TAG + i) != null)
//            {
//                deleteState(SacSmaHTConstants.SOIL_LAYER_WATER_CONTENT_TAG + i);
//            }
//        }
//
//    }

    /**
     * Set a multiplier to LZFSC and LZFPC values as part of mods logic.
     * 
     * @param multiplier
     */
    public void setBaseflowStatesUsingMultiplier(final double multiplier)
    {
        if(multiplier == OHDConstants.MISSING_DATA)
        {
            return;
        }

        this.setLzfsc(this.getLzfsc() * multiplier);
        this.setLzfpc(this.getLzfpc() * multiplier);
    }

    /**
     * Add carryover transfer logic to sacsmaHT model
     */
    @Override
    public void doCarryOverTransfer(final ModelParameters savedParams, final ModelParameters params)
    {
        final SacSmaHTModelParameters savedParameters = (SacSmaHTModelParameters)savedParams;
        final SacSmaHTModelParameters parameters = (SacSmaHTModelParameters)params;

        _logger.log(Logger.DEBUG, "SacSmaHTModelState before carry over transfer " + this.getStateMap());

        if(this.getUzfwc() > parameters.getUzfwm())
        {
            this.setUzfwc(parameters.getUzfwm());
        }
        final double uztwc = this.getUztwc();

        this.setUztwc(parameters.getUztwm() - savedParameters.getUztwm() + uztwc);
        if(this.getUztwc() < 0.0)
        {
            this.setUztwc(0.0);
        }
        final double lztwc = this.getLztwc();
        this.setLztwc(parameters.getLztwm() - savedParameters.getLztwm() + lztwc);
        if(this.getLztwc() < 0.0)
        {
            this.setLztwc(0.0);
        }
        if(parameters.getLzsk() != 0.0)
        {
            final double lzfsc = this.getLzfsc();
            this.setLzfsc(lzfsc * (savedParameters.getLzsk() / parameters.getLzsk())
                * ((1.0 + parameters.getSide() / (1.0 + savedParameters.getSide()))));
        }
        if(this.getLzfsc() > parameters.getLzfsm())
        {
            this.setLzfsc(parameters.getLzfsm());
        }
        if(parameters.getLzpk() != 0.0)
        {
            final double lzfpc = this.getLzfpc();
            this.setLzfpc(lzfpc * (savedParameters.getLzpk() / parameters.getLzpk())
                * ((1.0 + parameters.getSide() / (1.0 + savedParameters.getSide()))));
        }
        if(this.getLzfpc() > parameters.getLzfpm())
        {
            this.setLzfpc(parameters.getLzfpm());
        }
        final double adimc = this.getAdimc();
        this.setAdimc(parameters.getUztwm() + parameters.getLztwm() - savedParameters.getUztwm()
            - savedParameters.getLztwm() + adimc);
        if(this.getAdimc() < this.getUztwc())
        {
            this.setAdimc(this.getUztwc());
        }

        _logger.log(Logger.DEBUG, "SacSmaHTModelState after carry over transfer " + this.getStateMap());
    }

}
