package ohd.hseb.ohdmodels.sacsmaHT;

import java.util.ArrayList;

import ohd.hseb.ohdmodels.sacsmaHT.HeatTransferSimulation.CommonVariables;
import ohd.hseb.util.Logger;
import ohd.hseb.util.MathHelper;
import ohd.hseb.util.fews.ModelException;

final class SacSmaHtCommonMethods
{
    private SacSmaHtCommonMethods()
    {
        /*
         * This class only contains static constants and methods, so an instance of this class should never be created.
         * If not providing this private constructor, users still can create objects of this class by using the
         * invisible default constructor provided by Java, which is wrong. Now, with this private constructor, the
         * default constructor by Java is destroyed. Now, nobody from outside of this class can create the object of
         * this class
         */
    }

    /*******************************************************************************************************************
     * *** New calculation of average temperature (TAVG) *** in freezing/thawing layer using UP, DOWN, and MIDDLE ***
     * *** layer temperatures (TUP, TDN, TM) ********** c!! Ken's version and early my version was wrong because all IF
     * c statements compared to 0. not T0 temperature. However, results were c reasonably good due to a reasonable
     * assumption that TAVG in all c combinations of TUP, TM & TDN equals (TUP+2*TM+TDN)/4. c In this version I replaced
     * all IF statements to compare to T0, c and corrected one old statement (marked c!!!) that was wrong.
     */

    /*
     * dwt - tension water change per time interval, mm dwf - free water change per time interval, mm smc - total
     * moisture content of frozen ground model layers sH2O - unfrozen water content of frozen ground model layers nUpl -
     * upper soil layer to distribute sac-sma water nLowl - lower soil layer to distribute sac-sma water zSoil - soil
     * layer depths, m sMax - soil porosity
     */

    public static double calculateAverageTemperature(final double upperLayerBoundaryTemperature,
                                                     final double middlLayerBoundaryTemperature,
                                                     final double lowerLayerBoundaryTemperature,
                                                     final double incrementalSoilLayerDepth)
    {
        final double midPointInMeters = incrementalSoilLayerDepth * 0.5f;
        double averageTemperature, x0, lowerLayerTemperature, upperLayerTemperature;

        if(upperLayerBoundaryTemperature < SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS)
        {
            if(middlLayerBoundaryTemperature < SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS)
            {
                if(lowerLayerBoundaryTemperature < SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS)
                {
                    /* C1 ****** TUP, TM, TDN < 0. CASE ************ */
                    averageTemperature = (upperLayerBoundaryTemperature + 2 * middlLayerBoundaryTemperature + lowerLayerBoundaryTemperature) / 4.;
                }
                else
                {
                    //C2  ******   TUP & TM < 0.,  TDN > 0. CASE   *****
                    x0 = (SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS - middlLayerBoundaryTemperature)
                        * midPointInMeters / (lowerLayerBoundaryTemperature - middlLayerBoundaryTemperature);
                    averageTemperature = 0.5
                        * (upperLayerBoundaryTemperature * midPointInMeters + middlLayerBoundaryTemperature
                            * (midPointInMeters + x0) + SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS
                            * (2. * midPointInMeters - x0)) / incrementalSoilLayerDepth;
                }
            }
            else
            {
                if(lowerLayerBoundaryTemperature < SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS)
                {
                    //C3  *******  TUP < 0.  TM > 0.  TDN < 0. CASE   ****
                    upperLayerTemperature = (SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS - upperLayerBoundaryTemperature)
                        * midPointInMeters / (middlLayerBoundaryTemperature - upperLayerBoundaryTemperature);
                    lowerLayerTemperature = midPointInMeters
                        - (SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS - middlLayerBoundaryTemperature)
                        * midPointInMeters / (lowerLayerBoundaryTemperature - middlLayerBoundaryTemperature);
                    averageTemperature = 0.5
                        * (upperLayerBoundaryTemperature * upperLayerTemperature
                            + SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS
                            * (2. * incrementalSoilLayerDepth - upperLayerTemperature - lowerLayerTemperature) + lowerLayerBoundaryTemperature
                            * lowerLayerTemperature) / incrementalSoilLayerDepth;
                    //C       C121   ****   END TDN_121  THEN  *****
                }
                else
                {
                    //C4   ******  TUP < 0  TM > 0  TDN > 0  CASE   ******
                    upperLayerTemperature = (SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS - upperLayerBoundaryTemperature)
                        * midPointInMeters / (middlLayerBoundaryTemperature - upperLayerBoundaryTemperature);
                    averageTemperature = 0.5
                        * (upperLayerBoundaryTemperature * upperLayerTemperature + SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS
                            * (2. * incrementalSoilLayerDepth - upperLayerTemperature)) / incrementalSoilLayerDepth;
                }

            }
        }
        else
        {
            if(middlLayerBoundaryTemperature < SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS)
            {
                if(lowerLayerBoundaryTemperature < SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS)
                {
                    upperLayerTemperature = midPointInMeters
                        - (SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS - upperLayerBoundaryTemperature)
                        * midPointInMeters / (middlLayerBoundaryTemperature - upperLayerBoundaryTemperature);
                    averageTemperature = 0.5
                        * (SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS
                            * (incrementalSoilLayerDepth - upperLayerTemperature) + middlLayerBoundaryTemperature
                            * (midPointInMeters + upperLayerTemperature) + lowerLayerBoundaryTemperature
                            * midPointInMeters) / incrementalSoilLayerDepth;
                }
                else
                {
                    upperLayerTemperature = midPointInMeters
                        - (SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS - upperLayerBoundaryTemperature)
                        * midPointInMeters / (middlLayerBoundaryTemperature - upperLayerBoundaryTemperature);
                    lowerLayerTemperature = (SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS - middlLayerBoundaryTemperature)
                        * midPointInMeters / (lowerLayerBoundaryTemperature - middlLayerBoundaryTemperature);
                    averageTemperature = 0.5
                        * (SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS
                            * (2. * incrementalSoilLayerDepth - upperLayerTemperature - lowerLayerTemperature) + middlLayerBoundaryTemperature
                            * (upperLayerTemperature + lowerLayerTemperature)) / incrementalSoilLayerDepth;
                }
            }
            else
            {
                if(lowerLayerBoundaryTemperature < SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS)
                {
                    lowerLayerTemperature = midPointInMeters
                        - (SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS - middlLayerBoundaryTemperature)
                        * midPointInMeters / (lowerLayerBoundaryTemperature - middlLayerBoundaryTemperature);
                    averageTemperature = (SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS
                        * (incrementalSoilLayerDepth - lowerLayerTemperature) + 0.5
                        * (SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS + lowerLayerBoundaryTemperature)
                        * lowerLayerTemperature)
                        / incrementalSoilLayerDepth;
                }
                else
                {
                    averageTemperature = (upperLayerBoundaryTemperature + 2. * middlLayerBoundaryTemperature + lowerLayerBoundaryTemperature) / 4.;
                }
            }
        }

        return averageTemperature;
    }

    /**
     * CALCULATE FREE WATER CONTENT if TEMPERATURE IS BELOW 273.16K (T0). REQUIRES NEWTON-TYPE ITERATION TO SOLVE THE
     * NONLINEAR IMPLICIT EQUATION GIVEN IN EQN 17 OF KOREN ET AL. (1999, JGR, VOL 104(D16), 19569-19585).
     */
    public static double frH2o(final double averageTemperature,
                               final double totalSoilMoisture,
                               final double liquidSoilMoisture,
                               final double maximumSoilMoistureContent,
                               final double B,
                               final double soilMatricPotential,
                               final double CK)
    {

        double freeWaterContent, frozenSoilMoistureAsPercent, totalSoilMoistureAsPercent;
        double newGuess, difference, DENOM, localCopyOfBParameter, localAverageTemperature, maximumSoilMoistureAsPercent, DF;
        int iterationCounter;
        final int maxNumberOfIterations = 900;
        final double latentHeatOfFusionForIce = 3.335 * 100000;
        final double tolerance = 0.005f;

        //  ***   LIMITS ON PARAMETER B: B < 5.5              ****
        //  ***   SIMULATIONS SHOWED if B > 5.5 UNFROZEN WATER CONTENT  ****
        //  ***   IS NON-REALISTICALLY HIGH AT VERY LOW TEMPERATURES    ****
        //******************************************************************

        localCopyOfBParameter = (float)B;
        if(B > 5.5)
        {
            localCopyOfBParameter = 5.5f;
        }

        localAverageTemperature = (float)averageTemperature;

        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
        //  if TEMPERATURE ABOVE or EQUAL 273.16K SH2O = SMC
        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

        if(localAverageTemperature >= SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS)
        {
            freeWaterContent = (float)totalSoilMoisture;
            return freeWaterContent;
        }

        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
        //  CONVERT SOIL MOISTURE VARIABLES INTO %
        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

        // initial guess for frozen water content
        frozenSoilMoistureAsPercent = ((float)totalSoilMoisture - (float)liquidSoilMoisture) * 100.f;
        maximumSoilMoistureAsPercent = (float)maximumSoilMoistureContent * 100.f;
        totalSoilMoistureAsPercent = (float)totalSoilMoisture * 100.f;

        iterationCounter = 0;

        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
        //  START OF ITERRATIONS
        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
        boolean flag = true;

        while(flag)
        {
            DF = (-(float)soilMatricPotential * (float)SacSmaHTConstants.GRAVITY_CONSTANT / (float)latentHeatOfFusionForIce)
                * (float)Math.pow(1 + (float)CK * (float)frozenSoilMoistureAsPercent * 0.01f, 2)
                * Math.pow((float)maximumSoilMoistureAsPercent
                               / ((float)totalSoilMoistureAsPercent - (float)frozenSoilMoistureAsPercent),
                           (float)localCopyOfBParameter);

            DENOM = 2 * (float)CK / (1 + (float)CK * (float)frozenSoilMoistureAsPercent * 0.01f)
                + (float)localCopyOfBParameter
                / ((float)totalSoilMoistureAsPercent - (float)frozenSoilMoistureAsPercent);

            newGuess = (float)frozenSoilMoistureAsPercent
                - (1 - ((float)localAverageTemperature - (float)SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS)
                    / ((float)localAverageTemperature * (float)DF)) / (float)DENOM;

            // Iteration will be performed only when newGuess > 0 and iterationCoutner < 100
            if(newGuess <= 0. || iterationCounter > 100)
            {
                // Use Flerchinger Equation
                final double flerchingerValue1 = -(float)latentHeatOfFusionForIce
                    / ((float)SacSmaHTConstants.GRAVITY_CONSTANT * (float)soilMatricPotential);
                final double flerchingerValue2 = ((float)localAverageTemperature - (float)SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS)
                    / (float)localAverageTemperature;
                final double flerchingerValue3 = (float)flerchingerValue1 * (float)flerchingerValue2;
                final double flerchingerValue4 = (float)Math.pow(flerchingerValue3, -1 / (float)B);
                final double flerchingerValue = (float)flerchingerValue4 * (float)maximumSoilMoistureContent;

                freeWaterContent = (float)Math.min(flerchingerValue, totalSoilMoisture);
                if(freeWaterContent < 0.)
                {
                    freeWaterContent = 0.2f;
                }

                return freeWaterContent;
            }
            // next iteration
            else
            {

                difference = Math.abs((float)newGuess - (float)frozenSoilMoistureAsPercent);
                frozenSoilMoistureAsPercent = (float)newGuess;

                if(frozenSoilMoistureAsPercent >= totalSoilMoistureAsPercent)
                {
                    frozenSoilMoistureAsPercent = (float)totalSoilMoistureAsPercent * 0.8f;
                    totalSoilMoistureAsPercent = (float)totalSoilMoistureAsPercent * 0.97f;
                }
            }

            iterationCounter = iterationCounter + 1;

            //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
            //  EXIT if MORE THAN 100 ITERRATIONS. FREE WATER CONTENT
            //  WILL BE ESTIMATED FROM FLERCHIGER EQUATION
            //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

            //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
            //  END OF ITERRATIONS if LAST ESTIMATE DIFFERENCE IS LESS THAN 
            //  AN ERROR
            //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

            if(difference <= tolerance)
            {
                flag = false;
            }
        }

        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
        //  RECALCULATE ICE CONTENT IN % INTO VOLUMETRIC FREE WATER
        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

        freeWaterContent = Math.max(((float)totalSoilMoisture - (float)frozenSoilMoistureAsPercent * 0.01f), 0.02f);
        return freeWaterContent;
    }

    /**
     * CALCULATE TEMPERATURE ON THE BOUNDARY OF THE LAYER BY INTERPOLATION OF THE MIDDLE LAYER TEMPERATURES
     */
    public static double calculateBoundaryLayerTemperature(final double upperSoilLayerTemperature,
                                                           final double lowerSoilLayerTemperature,
                                                           final double soilLayerDepths[],
                                                           final double depthAtBottomLayer,
                                                           final int soilLayer,
                                                           final int numberOfSoilLayers)
    {
        double temperatureOfBounary, upperLayerDepth, lowerLayerDepth;

        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
        //   USE SURFACE TEMPERATURE ON THE TOP OF THE FIRST LAYER
        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

        if(soilLayer == 0)
        {
            upperLayerDepth = 0.;
        }
        else
        {
            upperLayerDepth = soilLayerDepths[soilLayer - 1];
        }

        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
        //   USE DEPTH OF THE CONSTANT BOTTOM TEMPERATURE WHEN INTERPOLATE
        //   TEMPERATURE INTO THE LAST LAYER BOUNDARY
        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

        if(soilLayer == (numberOfSoilLayers - 1))
        {
            lowerLayerDepth = 2. * depthAtBottomLayer - soilLayerDepths[soilLayer];
        }
        else
        {
            lowerLayerDepth = soilLayerDepths[soilLayer + 1];
        }

        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
        //   LINEAR INTERPOLATION BETWEEN THE AVERAGE LAYER TEMPERATURES
        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

        temperatureOfBounary = upperSoilLayerTemperature + (lowerSoilLayerTemperature - upperSoilLayerTemperature)
            * (upperLayerDepth - soilLayerDepths[soilLayer]) / (upperLayerDepth - lowerLayerDepth);

        return temperatureOfBounary;
    }

    /*
     * Interpolate soil variables from a non-regular profile into user-defined profile
     */

    public static double[] interpolateSoilVariableValue(final double[] modelDefinedValue,
                                                        final double[] modelDefinedSoilProfile,
                                                        final double[] userDefinedSoilProfile,
                                                        final Logger logger) throws Exception
    {
        final ArrayList<Double> valuesAtUserDefinedProfile = new ArrayList<Double>();

        for(int i = 0; i < userDefinedSoilProfile.length; i++)
        {
            int j;
            for(j = 0; j < modelDefinedSoilProfile.length; j++)
            {

                if(userDefinedSoilProfile[i] < modelDefinedSoilProfile[j])
                {
                    if(j > 0)
                    {
                        valuesAtUserDefinedProfile.add((modelDefinedValue[j - 1]
                            * (modelDefinedSoilProfile[j] - userDefinedSoilProfile[i]) + modelDefinedValue[j]
                            * (userDefinedSoilProfile[i] - modelDefinedSoilProfile[j - 1]))
                            / (modelDefinedSoilProfile[j] - modelDefinedSoilProfile[j - 1]));

                        break;
                    }
                    else
                    {
                        logger.log(Logger.ERROR, "The user defined layer " + userDefinedSoilProfile[i]
                            + " is less than the minimum model defined layer " + modelDefinedSoilProfile[0]);
                        //actually these are all errors, exception should not be thrown to stop, instead return code should be used so that we return to fews w/o a result
                        throw new ModelException("No soil layers selected: The user defined layer "
                            + userDefinedSoilProfile[i] + " is less than the minimum model defined layer "
                            + modelDefinedSoilProfile[0]);
                    }
                }
            }

            // If you got to the end of the modeling soil layers without a match and
            // you're at the first desired layer, there's an error. Also, reduce the number
            // of desired soil layers by 1

            if(j == modelDefinedSoilProfile.length)
            {
                if(i == 0)
                {
                    logger.log(Logger.ERROR,
                               "The model defined layer is greater than the maxiumum model defined layer "
                                   + modelDefinedSoilProfile[j]);
                    throw new ModelException("No soil layers selected");
                }
                else
                {
                    logger.log(Logger.WARNING, "Desired soil layer: " + userDefinedSoilProfile[i]
                        + " out of model range");
                    logger.log(Logger.WARNING, "Number of desired soil layers was changed from "
                        + userDefinedSoilProfile.length + " to " + Integer.toString(userDefinedSoilProfile.length - 1));
                    break;
                }
            }
        }

        final double[] returnValues = new double[valuesAtUserDefinedProfile.size()];
        for(int i = 0; i < valuesAtUserDefinedProfile.size(); i++)
        {
            returnValues[i] = valuesAtUserDefinedProfile.get(i).doubleValue();
        }
        return returnValues;
    }

    /**
     * JOHNSON'S METHOD TO CALCULATE SOIL THERMAL CONDUCTIVITY (FROZEN OR NON-FROZEN)
     */
    public static double calculateSoilThermalConductivity(final double soilType,
                                                          final double unfrozenWaterContent,
                                                          final double maximumIceContent,
                                                          final double maximumPorosity,
                                                          final double QUARTZ)
    {
        double soilThermalConductivity, DRY, SR, SKER, S_DENS, SK, saturatedSoilConductivity;
        //double S_QUARZ=7.7;
        //final static double SKW=0.57;
        //double SKICE=2.2;
        //     S_QUARZ IS QUARTZ CONDUCTIVITY IN W/M.K
        //     SKW IS WATER CONDUCTIVITY IN W/M.K
        //     SKICE IS ICE CONDUCTIVITY IN W/M.K

        //     S_TYPE =
        //      11 - NATURAL COARSE
        //      12 - NATURAL FINE
        //      21 - CRUSHED COARSE
        //      22 - CRUSHED FINE

        //     DRY CONDUCTIVITY

        S_DENS = (1 - maximumPorosity) * 2700.;
        if(soilType < 20)
        {
            DRY = (0.137 * S_DENS + 64.7) / (2700 - 0.947 * S_DENS);
        }
        else
        {
            DRY = 0.39 * Math.pow(maximumPorosity, -2.2);
        }

        //     KERSTEN NUMBER
        SR = (unfrozenWaterContent + maximumIceContent) / maximumPorosity;

        if(SR <= 0.)
        {
            SKER = 0.;
        }
        else
        {
            if(MathHelper.isEqualToZero(maximumIceContent))
            // lc code changed to match fortran precision
            //if(maximumIceContent == 0.)
            {
                if((soilType % 10.) == 1.)
                {
                    SKER = 0.7 * Math.log(SR) / Math.log(10) + 1.0;
                }
                else
                {
                    SKER = Math.log(SR) / Math.log(10) + 1.0;
                }

                if(SKER < 0.)
                {
                    SKER = 0.;
                }
            }
            else
            {
                SKER = SR;
            }
        }

        //     PARTICLE CONDUCTIVITY
        if(((soilType % 10.) == 1.) && (QUARTZ < 0.2))
        {
            SK = Math.pow(SacSmaHTConstants.QUARTZ_CONDUCTIVITY, QUARTZ) * Math.pow(3.0, (1. - QUARTZ));
        }
        else
        {
            SK = Math.pow(SacSmaHTConstants.QUARTZ_CONDUCTIVITY, QUARTZ) * Math.pow(2.0, (1. - QUARTZ));
        }

        //     SATURATED CONDUCTIVITY
        if(MathHelper.isEqualToZero(maximumIceContent))
        // lc code changed to match fortran precision
        //if(maximumIceContent == 0.)
        {
            saturatedSoilConductivity = Math.pow(SK, (1. - maximumPorosity))
                * Math.pow(SacSmaHTConstants.WATER_CONDUCTIVITY, maximumPorosity);
        }
        else
        {
            saturatedSoilConductivity = Math.pow(SK, (1. - maximumPorosity))
                * Math.pow(SacSmaHTConstants.WATER_CONDUCTIVITY, unfrozenWaterContent)
                * Math.pow(SacSmaHTConstants.ICE_CONDUCTIVITY, maximumPorosity - unfrozenWaterContent);
        }

        //     ACTUAL SOIL THERMAL CONDUCTIVITY
        soilThermalConductivity = (saturatedSoilConductivity - DRY) * SKER + DRY;

        return soilThermalConductivity;
    }

    public static double calculateSnowThermalConductivity(final double snowDensity)
    {
        final double UNIT = 0.11631;
        double C, snowThermalConductivity;

        //   ****   SIMULATION OF THERMAL SNOW CONDUCTIVITY                   
        //   ****  SIMULATION UNITS OF CSNOW IS CAL/(CM*HR* C) 
        //   ****  AND IT WILL BE RETURND IN W/(M* C)
        //   ****  BASIC VERSION IS DYACHKOVA EQUATION                                

        // *****   Dyachkova equation (1960), for range 0.1-0.4

        C = 0.328 * Math.pow(10, (2.25 * snowDensity));
        snowThermalConductivity = UNIT * C;

        return snowThermalConductivity;
    }

    /**
     * TO CALCULATE SINK/SOURCE TERM OF THE TERMAL DIFFUSION
     */
    //      write(*,*)'SNKSRC input:',TUP,TM,TDN,SMC,SH2O,HCPCT,SMCMAX,
    //     +      PSISAT,B, DT, K,STYPE,QUARTZ,CK
    //      write(*,*)'SNKSRC zsoil input:',ZSOIL(1:K)
    //      write(*,*)'SNKSRC output:',TUP,TM,TDN,SMC,SH2O,HCPCT,SMCMAX,
    //     +  PSISAT,B, DT, K,STYPE,QUARTZ,CK,SNKSRC
    //      write(*,*)'SNKSRC zsoil output:',ZSOIL(1:K)
    /*
     * SNKSRC input: 268.3272 271.4908 272.4384 0.3633357 0.2625487 1985916. 0.4675800 4.632060 4.977000 1800.000 4
     * 12.00000 0.1700000 8.000000 SNKSRC zsoil input: -2.9999999E-02 -0.1381460 -0.4035953 -0.9951212 SNKSRC output:
     * 268.3272 271.4908 272.4384 0.3633357 0.2625000 1985916. 0.4675800 4.632060 4.977000 1800.000 4 12.00000 0.1700000
     * 8.000000 5.333759 SNKSRC zsoil output: -2.9999999E-02 -0.1381460 -0.4035953 -0.9951212
     */
    public static double snkSrc(final double upperLayerBoundaryTemperature,
                                final double soilTemperature,
                                final double lowerLayerBoundaryTemperature,
                                final double totalSoilMoisture,
                                final CommonVariables unfrozenWaterContent,
                                final double soilLayerHeatCapacity,
                                final double soilLayerDepths[],
                                final double maximumSoilMoisture,
                                final double PSISAT,
                                final double soilTypeBParameter,
                                final double modelTimeStep,
                                final int soilLayer,
                                final double soilType,
                                final double QUARTZ,
                                final double iceEffectParameter)
    {
        double heatContent, frozenSoilMoisture, incrementalSoilDepth, unfrozenConductivity;
        double frozenConductivity, upperBoundaryHeatFlux, totalHeatFlux, averageTemperature;
        double superCooledLiquidWater, potentialLiquidWaterContent, conductivity, lowerBoundaryHeatFlux;
        final double HLICE = 3.335 * 100000;

        frozenSoilMoisture = totalSoilMoisture - unfrozenWaterContent.val;

        if(soilLayer == 0)
        {
            incrementalSoilDepth = -soilLayerDepths[0];
        }
        else
        {
            incrementalSoilDepth = soilLayerDepths[soilLayer - 1] - soilLayerDepths[soilLayer];
        }

        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
        //  CALCULATE DIFFUSIVITIES OF THAWED AND FROZEN GROUND
        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

        unfrozenConductivity = SacSmaHtCommonMethods.calculateSoilThermalConductivity(soilType,
                                                                                      totalSoilMoisture,
                                                                                      0.,
                                                                                      maximumSoilMoisture,
                                                                                      QUARTZ);
        frozenConductivity = unfrozenConductivity;

        if(frozenSoilMoisture > 0.)
        {
            frozenConductivity = SacSmaHtCommonMethods.calculateSoilThermalConductivity(soilType,
                                                                                        unfrozenWaterContent.val,
                                                                                        frozenSoilMoisture,
                                                                                        maximumSoilMoisture,
                                                                                        QUARTZ);
        }

        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
        //  HEAT FLUX FROM THE TOP BOUNDARY OF LAYER
        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

        if(upperLayerBoundaryTemperature <= SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS)
        {
            conductivity = frozenConductivity;
        }
        else
        {
            conductivity = unfrozenConductivity;
        }

        upperBoundaryHeatFlux = -conductivity
            * (SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS - upperLayerBoundaryTemperature)
            / (0.5 * incrementalSoilDepth);

        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
        //  HEAT FLUX FROM THE BOTTOM BONDARY OF LAYER
        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

        if(lowerLayerBoundaryTemperature <= SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS)
        {
            conductivity = frozenConductivity;
        }
        else
        {
            conductivity = unfrozenConductivity;
        }

        lowerBoundaryHeatFlux = -conductivity
            * (SacSmaHTConstants.ABSOLUTE_ZERO_IN_KELVINS - lowerLayerBoundaryTemperature)
            / (0.5 * incrementalSoilDepth);

        totalHeatFlux = upperBoundaryHeatFlux + lowerBoundaryHeatFlux;

        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
        //     CALCULATE POTENTIAL INCREASE/REDUCTION OF LIQUED WATER CONTENT
        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

        potentialLiquidWaterContent = totalHeatFlux * modelTimeStep
            / (SacSmaHTConstants.DENSITY_OF_WATER * HLICE * incrementalSoilDepth) + unfrozenWaterContent.val;

        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
        //     ESTIMATE UNFROZEN WATER AT TEMPERATURE TAVG,
        //     AND CHECK if CALCULATED WATER CONTENT IS REASONABLE 
        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC 

        averageTemperature = SacSmaHtCommonMethods.calculateAverageTemperature(upperLayerBoundaryTemperature,
                                                                               soilTemperature,
                                                                               lowerLayerBoundaryTemperature,
                                                                               incrementalSoilDepth);

        superCooledLiquidWater = SacSmaHtCommonMethods.frH2o(averageTemperature,
                                                             totalSoilMoisture,
                                                             unfrozenWaterContent.val,
                                                             maximumSoilMoisture,
                                                             soilTypeBParameter,
                                                             PSISAT,
                                                             iceEffectParameter);

        if((potentialLiquidWaterContent < unfrozenWaterContent.val)
            && (potentialLiquidWaterContent < superCooledLiquidWater))
        {
            if(superCooledLiquidWater > unfrozenWaterContent.val)
            {
                potentialLiquidWaterContent = unfrozenWaterContent.val;
            }
            else
            {
                potentialLiquidWaterContent = superCooledLiquidWater;
            }
        }

        if((potentialLiquidWaterContent > unfrozenWaterContent.val)
            && (potentialLiquidWaterContent > superCooledLiquidWater))
        {
            if(superCooledLiquidWater < unfrozenWaterContent.val)
            {
                potentialLiquidWaterContent = unfrozenWaterContent.val;
            }
            else
            {
                potentialLiquidWaterContent = superCooledLiquidWater;
            }
        }

        if(potentialLiquidWaterContent < 0.)
        {
            potentialLiquidWaterContent = 0.;
        }
        if(potentialLiquidWaterContent > totalSoilMoisture)
        {
            potentialLiquidWaterContent = totalSoilMoisture;
        }

        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
        //     CALCULATE HEAT SINK/SOURCE TERM FROM SOIL ICE
        //     PHASE CHANGE AND REPLACE PREVIOUS WATER CONTENT 
        //CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

        heatContent = -SacSmaHTConstants.DENSITY_OF_WATER * HLICE * incrementalSoilDepth
            * (potentialLiquidWaterContent - unfrozenWaterContent.val) / modelTimeStep;

        unfrozenWaterContent.val = potentialLiquidWaterContent;

        return heatContent;
    }
}
