package ohd.hseb.ohdmodels.snow17;

import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.DoubleHolder;
import ohd.hseb.util.Logger;
import ohd.hseb.util.MathHelper;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.OHDUtilities;

/**
 * This class does Snow17 computation for each MAT (air temperature data) time step data through
 * {@link #pack19(double, double[], double[], double, long, double, double)} and its subsequent calling other methods.
 * MAT TS is driving. So for each MAT time interval step, pack19 is executed once. If MAP time interval is half of MAT,
 * then within each pack19 execution, MAP 2 data points were processed.
 * <p>
 * To run {@link #pack19(double, double[], double[], double, long, double, double)}, {@link Snow17ModelParameters} and
 * {@link Snow17ModelState} need to be loaded at first.
 * <p>
 * 
 * @author FewsPilot Team
 */
final public class Snow17Model
{

    private final Snow17ModelParameters _snow17ModelParameters;

    private final Snow17ModelState _snow17ModelState;

    private final Logger _logger;

    //input MODS ts, by default, not using MODS
    private RegularTimeSeries _inputUadjModTs = null;
    private RegularTimeSeries _inputMfcModTs = null;

    //these variables are for logging at hour=24
    private double _dqnet = 0.0; //daily net energy exchange, MM
    private double _dsfall = 0.0; //daily snow fall, MM
    private double _drain = 0.0; //daily rain on snow, MM
    private double _aesc; //current snow cover, percentage
    private double _twe; //current total water equivalent, MM
    private double _powe; //observed water equivalent, MM
    private double _posc; //observed areal snow cover
    private double _podpt; //observed snow depth, CM
    private double _drsl = 0.0; //sum of rain-snow elevation during periods with snowfall during the day, M
    private int _ndrsp = 0; //number of periods during the day with snowfall

    static double _spx = 0.0; //log purpose
    static double _chgwe = 0.0; //log purpose

    private final StringBuilder _tableLogMessage = new StringBuilder();

    private String _currDateTimeStr;
    private final DateTime _currentDateTime;
    private boolean _printLog = false;

    private boolean _printTableLog = false; //TECHNIQUE PRINTSNW: if true, print table log. Default is false

    private final int _startHourOfDay;

    /**
     * Instantiates a new Snow17Model object in normal mode.
     * 
     * @param snow17ModelParams the snow17 model params
     * @param snow17ModelState the snow17 model state
     * @param uadjModTs the input UADJ MODS time series. If not using the MODS, pass in NULL.
     * @param mfcModTs the input MFC MODS time series. If not using the MODS, pass in NULL.
     */
    public Snow17Model(final Snow17ModelParameters snow17ModelParams,
                       final Snow17ModelState snow17ModelState,
                       final RegularTimeSeries uadjModTs,
                       final RegularTimeSeries mfcModTs,
                       final int startHourOfDay,
                       final Logger logger)
    {
        _snow17ModelParameters = snow17ModelParams; // alias
        _snow17ModelState = snow17ModelState; // alias

        _inputUadjModTs = uadjModTs;
        _inputMfcModTs = mfcModTs;

        _currentDateTime = new DateTime();

        _currentDateTime.setStartOfDay(startHourOfDay);

        _startHourOfDay = startHourOfDay;

        _logger = logger;
        if(_logger.getPrintDebugInfo() > 0)
        {
            _printLog = true;
        }

    }

    /**
     * Pack19() corresponds to Fortran subroutine PACK19(pack19.f). It does all the snow17 model computation. For each
     * time step in MAT, pack19() is called once. Inside pack19(), there is a big loop: it loops that many times of the
     * ratio between MAT/MAP interval. <br>
     * If no snow, Pack19() ends quickly. If there is snow, Pack19() precedes to evoke other methods, Melt19(),
     * Aesc19(), Rout19() etc. Before returning from the method, check if needs to run Sndepth19().
     * 
     * @param airTemp the air temperature
     * @param precipArray the precip array, if precip interval equals to air temperature interval, the array size is 1
     * @param percentSnowFallArray the perecent snow fall array, its size equals to precipArray. This is optional input.
     *            If not needed, its values are set to MISSING_DATA.
     * @param rainSnowLatitude the rain snow latitude
     * @param currTimeInMillis the epoch time. Only used in a very rare situation: when card 6A is included(user
     *            specified SMFV for calculating melt from glaciers).
     * @return the snow17 results
     */
    public Snow17Results pack19(final double airTemp,
                                final double[] precipArray,
                                final double[] percentSnowFallArray,
                                final double rainSnowLatitude,
                                final long currTimeInMillis,
                                final double owe,
                                final double osc,
                                final double odpt,
                                final RAINSNOW_MOD_VALUE rainSnowMod)
    {
        // SUBROUTINE
        // PACK19(KDA,KHR,NDT,TA,PX,PCTS,RSL,OWE,OSC,ODPT,PGM,RM,TWE,COVER,CWE,CAESC,IFUT,IDT,IBUG,IDN,IMN,IYR,IOUTYP,OPNAME,IVER)

        _currentDateTime.setTime(currTimeInMillis, null);

        if(_printLog)
        {
            _currDateTimeStr = DateTime.getDateTimeStringFromLong(currTimeInMillis, null);

            _logger.log(Logger.DEBUG,
                        "PACK19 DEBUG--" + _currDateTimeStr + " NDT=" + _snow17ModelParameters.getRatioMatMapInterval()
                            + " IDT=" + _snow17ModelParameters.getAirTempInterval());

            _logger.log(Logger.DEBUG,
                        "INPUT DATA--TA="
                            + airTemp
                            + " RSL="
                            + rainSnowLatitude
                            + " OWE="
                            + owe
                            + " OSC="
                            + osc
                            + " PGM="
                            + OHDUtilities.getFortranPrecison(_snow17ModelParameters.getDailyGroundMelt()
                                / (OHDConstants.HOURS_PER_DAY / _snow17ModelParameters.getAirTempInterval())) + " TWE="
                            + OHDUtilities.getFortranPrecison(_snow17ModelState.calculateTweFromState()) + " ODPT="
                            + odpt);

            for(int i = 0; i < precipArray.length; i++)
            {
                _logger.log(Logger.DEBUG, "PX AND PCTS= " + precipArray[i] + "  " + percentSnowFallArray[i]);
            }

            _logger.log(Logger.DEBUG,
                        "WE=" + OHDUtilities.getFortranPrecison(_snow17ModelState.getWaterEquivalentOfSolidSnow())
                            + " NEGHS=" + OHDUtilities.getFortranPrecison(_snow17ModelState.getNegHeatStorage())
                            + " LIQW=" + OHDUtilities.getFortranPrecison(_snow17ModelState.getLiquidWaterInSnowPack())
                            + " TINDEX=" + OHDUtilities.getFortranPrecison(_snow17ModelState.getTempIndex())
                            + " ACCMAX=" + OHDUtilities.getFortranPrecison(_snow17ModelState.getMaxWeSinceSnow())
                            + " SB="
                            + OHDUtilities.getFortranPrecison(_snow17ModelState.getArealWaterEquivBeforeSnow())
                            + " SBAESC=" + OHDUtilities.getFortranPrecison(_snow17ModelState.getAescBeforeSnow())
                            + " SBWS="
                            + OHDUtilities.getFortranPrecison(_snow17ModelState.getWaterEquivalentUnderSnowCover())
                            + " STORGE="
                            + OHDUtilities.getFortranPrecison(_snow17ModelState.getExcessLiquidWaterInStorage())
                            + " AEADJ="
                            + OHDUtilities.getFortranPrecison(_snow17ModelState.getSnowArealExtentAdjustment())
                            + " SNDPT=" + OHDUtilities.getFortranPrecison(_snow17ModelState.getSnowDepth()) + " SNTMP="
                            + OHDUtilities.getFortranPrecison(_snow17ModelState.getAverageSnowTemp()) + " TAPREV="
                            + _snow17ModelState.getTaprev() + " EXLAG="
                            + OHDUtilities.getStringLineFromDoubleArray(_snow17ModelState.getExlagArray()));
        }
        _powe = owe;
        _posc = osc;
        _podpt = odpt;
        _aesc = 0.0;

        double prain = 0.0;
        double psfall = 0.0;

//        System.out.println("Pack19 States:\n" + _snow17ModelState.toString());

//        System.out.println("Pack19 Params:\n" + _snow17ModelParameters.toStringInstanceValues());

        /* --------------Declare local variables ------------ */
        final double[] raimArray = new double[_snow17ModelParameters.getRatioMatMapInterval()];

        // OTHER VARIABLES
        double deltaAirTemp = 0.0;
        double liqwmx = 0.0;

        double totalLagExcessWater = 0.0;

        _snow17ModelState.setSnowDensity(-9.99); // hard-coded as pack19.f

//        if(_snow17ModelState.getWaterEquivalentOfSolidSnow() > 0.0)
        if(MathHelper.isGreaterThanZero(_snow17ModelState.getWaterEquivalentOfSolidSnow()))
        {

            _snow17ModelState.setSnowDensity(0.1 * _snow17ModelState.getWaterEquivalentOfSolidSnow()
                / _snow17ModelState.getSnowDepth());
        }

//        if(_snow17ModelState.getWaterEquivalentOfSolidSnow() == 0)
        if(MathHelper.isEqualToZero(_snow17ModelState.getWaterEquivalentOfSolidSnow()))
        {
            deltaAirTemp = airTemp;
        }
        else
        {
            deltaAirTemp = airTemp - _snow17ModelState.getTaprev();

//            if(_snow17ModelState.getTaprev() > 0. && airTemp > 0.)
            if(MathHelper.isGreaterThanZero(_snow17ModelState.getTaprev()) && MathHelper.isGreaterThanZero(airTemp))
            {
                deltaAirTemp = Math.abs(deltaAirTemp);
            }

//            if(_snow17ModelState.getTaprev() > 0. && airTemp <= 0.)
            if(MathHelper.isGreaterThanZero(_snow17ModelState.getTaprev()) && MathHelper.isLessOrEqual(airTemp, 0.))
            {
                deltaAirTemp = airTemp; // extra "="
            }
        }

        final double groundMeltDuringInterval = _snow17ModelParameters.getDailyGroundMelt()
            / (OHDConstants.HOURS_PER_DAY / _snow17ModelParameters.getPrecipInterval());
        // DAYGM: daily melt at the snow-soil interface; units of mm; GM: melt during the MAP interval period
        double totalNewSnowFall = 0.0;
        double totalSurfaceMelt = 0.0;
        double totalGroundMelt = 0.0;
        double totalRefrozenWater = 0.0;

        double robg = 0.0;
        double probg = 0.0;
        double snowPackOutFlow = 0.0; //PACKRO
        double psnwro = 0.0;
        double gmro = 0.0;
        double melt = 0.0;

        // the following two variables store and pass values from one loop to next loop
        double pmelt = 0.0;
        double pcnhs = 0.0;

        boolean melt19Called = false;
        // within pack19(), melt19() no need to be called more than one times;

        double qnet = 0.0;
        double pqnet = 0.0;

        /* -----------------a big for loop --------------------------- */
        mainForLoop: for(int loopCount = 1; loopCount <= _snow17ModelParameters.getRatioMatMapInterval(); loopCount++)
        {

            final double currentPrecip = precipArray[loopCount - 1]; // PXI is like a pointer pointing to PX

            blockBefore160:
            {
//                if(currentPrecip != 0.0 || _snow17ModelState.getWaterEquivalentOfSolidSnow() != 0.0)
                if(MathHelper.notEqualToZero(currentPrecip)
                    || MathHelper.notEqualToZero(_snow17ModelState.getWaterEquivalentOfSolidSnow()))
                { // either has precipitation or has ice

                    double sfall = 0.0; // snowfall
                    double cnhspx = 0.0;
                    double rain = 0.0;
                    double rainm = 0.0;
                    double fracs = 0.0; // percentage of precip as snow
                    double fracr = 0.0; // percentage of precip as rain

//                    if(currentPrecip != 0.0)
                    if(MathHelper.notEqualToZero(currentPrecip))
                    {
                        /*
                         * if precipitation != 0.0, then determine what percentage of precip is snow or rain, based on
                         * PXTEMP and area_elevation curve if using it; or modified by RAISNOW MOD
                         */

                        double snowPercentage = percentSnowFallArray[loopCount - 1];

//                        if(snowPercentage > 1.0)
                        if(MathHelper.isGreater(snowPercentage, 1.0))
                        {
                            snowPercentage = 1.0;
                        }

                        if(rainSnowMod == RAINSNOW_MOD_VALUE.NO_MOD)
                        {
//                            if(snowPercentage >= 0.0)
                            if(MathHelper.isGreaterOrEqual(snowPercentage, 0.0))
                            {
                                fracs = snowPercentage;
                                fracr = 1.0 - snowPercentage;
                            }
                            // else if (LAEC != 0)
                            else if(_snow17ModelParameters.useRainSnowElevationInput() == true)
                            { // card3B is included (ae_pairs(area_elevation curve data) is available)

                                final double[][] aePairs = _snow17ModelParameters.getAe_pairs(); //two rows, multiple columns: 1st row is elevation, 2nd row is percentage

                                final int aeNum = aePairs[0].length;

                                _drsl += rainSnowLatitude;

                                _ndrsp++;

                                if(rainSnowLatitude > aePairs[0][0])
                                {//_aePairs[0][0] == EMIN, minimum elevation

                                    int j = 0;

                                    //find the elevation(first row of aePairs[][]) just above rainSnowLatitude
                                    try
                                    {
                                        for(j = 2; j <= aeNum; j++)
                                        {
                                            if(rainSnowLatitude <= aePairs[0][j - 1])
                                            {
                                                throw new Exception(); // GOTO 1063
                                            }
                                        }

                                        //when rainSnowLatitude is higher than highest elevation in aePairs[][], all are rain
                                        fracr = 1.0;
                                    }
                                    // GOTO 30;
                                    catch(final Exception e)
                                    {
                                        // 1063
                                        fracr = aePairs[1][j - 2] + (aePairs[1][j - 1] - aePairs[1][j - 2])
                                            * (rainSnowLatitude - aePairs[0][j - 2])
                                            / (aePairs[0][j - 1] - aePairs[0][j - 2]);
                                    }
                                }
                                else
                                {//rainSnowLatitude <= EMIN, all are snow, not rain

                                    fracr = 0.0;
                                }

                                // 30
                                fracs = 1.0 - fracr; // partial rain and partial snow
                            }
                            else if(airTemp > _snow17ModelParameters.getDelineationTemperature())
                            { //if not using RAIN-SNOW ELEVATION curve, then determine rain & snow faction by PXTEMP: warm

                                fracs = 0.0; // no snow
                                fracr = 1.0; // rain
                            }
                            else
                            { //if not using RAIN-SNOW ELEVATION curve, then determine rain & snow faction by PXTEMP: cold

                                fracs = 1.0; // snow
                                fracr = 0.0; // no rain
                            }
                        } //close RAINSNOW NO_MOD
                        else if(rainSnowMod == RAINSNOW_MOD_VALUE.RAIN)
                        { // has RAINSNOW MOD RAIN
                            fracs = 0.0; // no snow
                            fracr = 1.0; // rain
                        }
                        else
                        { // has RAINSNOW MOD SNOW
                            fracs = 1.0; // snow
                            fracr = 0.0; // no rain
                        }
                        // the percentage of precip as snow or rain has been determined

//                        if(fracs != 0.0)
                        if(MathHelper.notEqualToZero(fracs))
                        { // if there is snow in precipitation, update states

                            double snowPackBottomTemp = airTemp; // TS: bottom of snowpack temperature(DEGC)

//                            if(snowPackBottomTemp > 0.0)
                            if(MathHelper.isGreaterThanZero(snowPackBottomTemp))
                            {
                                snowPackBottomTemp = 0.0;
                            }

                            sfall = currentPrecip * fracs * _snow17ModelParameters.getSnowCorrectionFactor();

                            _spx += sfall;

                            _dsfall += sfall;

                            psfall += sfall;

                            final double weliqw = (_snow17ModelState.getWaterEquivalentOfSolidSnow() + _snow17ModelState.getLiquidWaterInSnowPack()); // ice + water

                            if(MathHelper.isLessOrEqual(weliqw, _snow17ModelState.getWaterEquivalentUnderSnowCover()))
                            {

                                boolean goTo107 = false;

                                if(MathHelper.isGreater(weliqw, _snow17ModelState.getArealWaterEquivBeforeSnow()))
                                {
                                    if(sfall < Snow17ModelConstants.HSNOF * _snow17ModelParameters.getPrecipInterval())
                                    { // if snow fall in each hour is less than 0.2

                                        //SBWS = SBWS + 0.75*SFALL
                                        _snow17ModelState.setWaterEquivalentUnderSnowCover(_snow17ModelState.getWaterEquivalentUnderSnowCover()
                                            + 0.75 * sfall);

                                        // GOTO 107;
                                        goTo107 = true;
                                    }
                                }
                                else if(sfall < Snow17ModelConstants.HSNOF * _snow17ModelParameters.getPrecipInterval())
                                { // if snow fall in each hour is less than 0.2

                                    //SB = SB + SFALL
                                    _snow17ModelState.setArealWaterEquivBeforeSnow(_snow17ModelState.getArealWaterEquivBeforeSnow()
                                        + sfall);

                                    //SBWS = SB
                                    _snow17ModelState.setWaterEquivalentUnderSnowCover(_snow17ModelState.getArealWaterEquivBeforeSnow());

                                    // GOTO 107;
                                    goTo107 = true;
                                }

                                if(goTo107 == false)
                                {
                                    //SBWS = WELIQW + 0.75*SFALL
                                    _snow17ModelState.setWaterEquivalentUnderSnowCover(weliqw + 0.75 * sfall);
                                }
                            }
                            else
                            {
                                //SBWS = SBWS + 0.75*SFALL
                                _snow17ModelState.setWaterEquivalentUnderSnowCover(_snow17ModelState.getWaterEquivalentUnderSnowCover()
                                    + 0.75 * sfall);
                            }

                            // 107
                            //WE = WE + SFALL
                            _snow17ModelState.setWaterEquivalentOfSolidSnow(_snow17ModelState.getWaterEquivalentOfSolidSnow()
                                + sfall);

                            if(MathHelper.isGreaterOrEqual(_snow17ModelState.getWaterEquivalentOfSolidSnow()
                                                               + _snow17ModelState.getLiquidWaterInSnowPack(),
                                                           3.0 * _snow17ModelState.getArealWaterEquivBeforeSnow()))
                            {

                                _snow17ModelState.setMaxWeSinceSnow(_snow17ModelState.getWaterEquivalentOfSolidSnow()
                                    + _snow17ModelState.getLiquidWaterInSnowPack());
                                _snow17ModelState.setSnowArealExtentAdjustment(0.0);
                            }

                            cnhspx = -snowPackBottomTemp * sfall / 160.0;

                            if(sfall > 1.5 * _snow17ModelParameters.getPrecipInterval())
                            {
                                _snow17ModelState.setTempIndex(snowPackBottomTemp);
                            }
                        } // end of if (FRACS != 0.0)

                        rain = currentPrecip * fracr;

                        _spx += rain;

                        prain += rain;

//                        if(_snow17ModelState.getWaterEquivalentOfSolidSnow() == 0.0)
                        if(MathHelper.isEqualToZero(_snow17ModelState.getWaterEquivalentOfSolidSnow()))
                        { // if there is no ice, jump out

                            break blockBefore160; // GOTO 160;
                        }

                        _drain += rain;

                        double tr = airTemp;

                        if(tr < 0.0)
                        {
                            tr = 0.0;
                        }

                        rainm = 0.0125 * rain * tr;

                    } // end of for loop if(PXI != 0)

                    // If didn't jump to 160 previously and has come to here, it is going to evoke MELT19, AESC19,
                    // ROUT19
                    blockBefore150:
                    {
                        // calculate melt
//                        if(_snow17ModelState.getWaterEquivalentOfSolidSnow() <= groundMeltDuringInterval)
                        if(MathHelper.isLessOrEqual(_snow17ModelState.getWaterEquivalentOfSolidSnow(),
                                                    groundMeltDuringInterval))
                        {
                            gmro = _snow17ModelState.getWaterEquivalentOfSolidSnow()
                                + _snow17ModelState.getLiquidWaterInSnowPack();

                            melt = 0.0;
                            robg = rain;
                            rain = 0.0;
                        }
                        else
                        {// WE > GM

                            double gmwlos = groundMeltDuringInterval
                                / _snow17ModelState.getWaterEquivalentOfSolidSnow()
                                * _snow17ModelState.getLiquidWaterInSnowPack();
                            double gmslos = groundMeltDuringInterval;

                            if(melt19Called == false)
                            { // enter here if has not been here yet

                                // preparing for calling method Melt19()
                                final DoubleHolder pmeltHolder = new DoubleHolder(pmelt);
                                final DoubleHolder pcnhsHolder = new DoubleHolder(pcnhs);

                                // CALL MELT19(IDN,IMN,ALAT,TA,PMELT,MFMAX,MFMIN,MBASE,TINDEX,TIPM,PCNHS,NMF,LMFV,SMFV);
                                melt19(airTemp, pmeltHolder, pcnhsHolder);

                                // retrieve values after calling
                                pmelt = pmeltHolder.getValue();
                                pcnhs = pcnhsHolder.getValue();

                                melt19Called = true;
                            } // end of if(melt19Called == false)

                            double cnhs = pcnhs / _snow17ModelParameters.getRatioMatMapInterval();

                            if(rain > 0.25 * _snow17ModelParameters.getPrecipInterval())
                            { //heavy rain(if _itpx==6, the threshold is 1.5). "rain" is not necessarily equal to MAP ts value, it has been processed

                                double ea = 2.7489E8 * Math.exp(-4278.63 / (airTemp + 242.792));

                                ea = 0.90 * ea;

                                final double tak = (airTemp + 273) * 0.01;
                                final double tak4 = tak * tak * tak * tak;
                                final double qn = .0612 * _snow17ModelParameters.getPrecipInterval() * (tak4 - 55.55);

                                double uadjc = 1.0;
                                if(_inputUadjModTs != null)
                                { //has UADJ MODS

                                    final double uadjcMod = _inputUadjModTs.getMeasurementValueByTimeNoException(currTimeInMillis,
                                                                                                                 MeasuringUnit.unitlessReal);

                                    if(uadjcMod != OHDConstants.MISSING_DATA) //input UADJ TS could have missing data(-999.0)
                                    {
                                        uadjc = uadjcMod; //now, MOD takes an effect: re-set uadjc to MOD value
                                        if(_printLog)
                                        {
                                            _logger.log(Logger.DEBUG, "UADJ MODS applied at " + _currDateTimeStr);
                                        }
                                    }
                                }

                                final double qe = 8.5 * (ea - 6.11) * _snow17ModelParameters.getWindFuncParameter()
                                    * uadjc;
                                final double elev = _snow17ModelParameters.getElevation() * 0.01; // convert from Meter to 100Meter
                                final double pa = (29.9 - 0.335 * elev + 0.00022 * Math.pow(elev, 2.4)) * 33.86;
                                final double qh = 7.5 * 0.000646 * pa * _snow17ModelParameters.getWindFuncParameter()
                                    * uadjc * airTemp;

                                /*
                                 * The Fortran code below is not really used in NWSRFS. So I ignored them. IFUT is not
                                 * used.
                                 */
                                //IF (IFUT .EQ. 0) QE = QE*WINDC
                                //IF (IFUT .EQ. 0) QH = QH*WINDC
                                melt = qn + qe + qh + rainm;

//                                if(melt < 0.0)
                                if(MathHelper.isLessThanZero(melt))
                                {
                                    melt = 0.0;
                                }

                            }
                            else
                            { //NON-RAIN OR LIGHT DIZZLE INTERVAL

                                melt = pmelt / _snow17ModelParameters.getRatioMatMapInterval();

                                if(_inputMfcModTs != null)
                                { //using MFC MODS: MELT=MELT*MFC

                                    final double mfcMod = _inputMfcModTs.getMeasurementValueByTimeNoException(currTimeInMillis,
                                                                                                              MeasuringUnit.unitlessReal);
                                    if(mfcMod != OHDConstants.MISSING_DATA) //_inputMfcModTs could have missing value(-999.0)
                                    {
                                        melt *= mfcMod; //modify melt by MFC MOD value
                                        if(_printLog)
                                        {
                                            _logger.log(Logger.DEBUG, "MFC MODS applied at " + _currDateTimeStr);
                                        }
                                    }
                                }

                                melt = melt + rainm;
                            }

                            // CALL AESC19(WE,LIQW,ACCMAX,SB,SBAESC,SBWS,SI,ADC,AEADJ,AESC);
                            _aesc = aesc19();

//                            if(_aesc == 1.0)
                            if(MathHelper.isEqual(_aesc, 1.0))
                            {
                                robg = 0.0;
                            }
                            else
                            {
                                melt = melt * _aesc;
                                cnhs = cnhs * _aesc;
                                gmwlos = gmwlos * _aesc;
                                gmslos = gmslos * _aesc;
                                robg = (1.0 - _aesc) * rain;
                                rain = rain - robg;
                            }

//                            if(cnhs + _snow17ModelState.getNegHeatStorage() < 0.0)
                            if(MathHelper.isLessThanZero(cnhs + _snow17ModelState.getNegHeatStorage()))
                            {
                                cnhs = -1.0 * _snow17ModelState.getNegHeatStorage();
                            }

                            totalNewSnowFall = totalNewSnowFall + sfall;
                            totalSurfaceMelt = totalSurfaceMelt + melt;
                            totalGroundMelt = totalGroundMelt + gmslos;
                            _snow17ModelState.setWaterEquivalentOfSolidSnow(_snow17ModelState.getWaterEquivalentOfSolidSnow()
                                - gmslos);
                            _snow17ModelState.setLiquidWaterInSnowpack(_snow17ModelState.getLiquidWaterInSnowPack()
                                - gmwlos);
                            gmro = gmslos + gmwlos;

//                            if(melt > 0.0)
                            if(MathHelper.isGreaterThanZero(melt))
                            {

//                                if(melt < _snow17ModelState.getWaterEquivalentOfSolidSnow())
                                if(MathHelper.isLessThan(melt, _snow17ModelState.getWaterEquivalentOfSolidSnow()))
                                {
                                    //136
                                    _snow17ModelState.setWaterEquivalentOfSolidSnow(_snow17ModelState.getWaterEquivalentOfSolidSnow()
                                        - melt);
                                }
                                else
                                {

                                    melt = _snow17ModelState.getWaterEquivalentOfSolidSnow()
                                        + _snow17ModelState.getLiquidWaterInSnowPack();

                                    qnet = melt;
                                    _dqnet += qnet;

                                    // GOTO 150;
                                    break blockBefore150;
                                }
                            }
                            // ground melt has been obtained

                            qnet = melt - cnhs - cnhspx;
                            _dqnet += qnet;
                            pqnet += qnet;

                            final double water = melt + rain;
                            final double heat = cnhs + cnhspx;

                            final double plwhc = _snow17ModelParameters.getPercentLiquidWaterHoldingCap();

                            liqwmx = plwhc * _snow17ModelState.getWaterEquivalentOfSolidSnow();

                            _snow17ModelState.setNegHeatStorage(_snow17ModelState.getNegHeatStorage() + heat);

//                            if(_snow17ModelState.getNegHeatStorage() < 0.0)
                            if(MathHelper.isLessThanZero(_snow17ModelState.getNegHeatStorage()))
                            {
                                _snow17ModelState.setNegHeatStorage(0.0);
                            }

//                            if(_snow17ModelState.getNegHeatStorage() > (0.33 * _snow17ModelState.getWaterEquivalentOfSolidSnow()))
                            if(MathHelper.isGreater(_snow17ModelState.getNegHeatStorage(),
                                                    (0.33 * _snow17ModelState.getWaterEquivalentOfSolidSnow())))
                            {
                                _snow17ModelState.setNegHeatStorage(0.33 * _snow17ModelState.getWaterEquivalentOfSolidSnow());
                            }

                            // calculate excess water
                            double excessLiquidWater = 0.0;

                            final double leftSide = water + _snow17ModelState.getLiquidWaterInSnowPack();
                            final double rightSide = liqwmx + _snow17ModelState.getNegHeatStorage() + plwhc
                                * _snow17ModelState.getNegHeatStorage();

//                            if(leftSide >= rightSide)
                            if(MathHelper.isGreaterOrEqual(leftSide, rightSide))
                            {

                                excessLiquidWater = water + _snow17ModelState.getLiquidWaterInSnowPack() - liqwmx
                                    - _snow17ModelState.getNegHeatStorage() - plwhc
                                    * _snow17ModelState.getNegHeatStorage();
                                _snow17ModelState.setLiquidWaterInSnowpack(liqwmx + plwhc
                                    * _snow17ModelState.getNegHeatStorage());
                                _snow17ModelState.setWaterEquivalentOfSolidSnow(_snow17ModelState.getWaterEquivalentOfSolidSnow()
                                    + _snow17ModelState.getNegHeatStorage());
                                totalRefrozenWater = totalRefrozenWater + _snow17ModelState.getNegHeatStorage();
                                _snow17ModelState.setNegHeatStorage(0.0);
                            }
                            else if(MathHelper.isLessThan(water, _snow17ModelState.getNegHeatStorage()))
                            {

                                _snow17ModelState.setWaterEquivalentOfSolidSnow(_snow17ModelState.getWaterEquivalentOfSolidSnow()
                                    + water);
                                _snow17ModelState.setNegHeatStorage(_snow17ModelState.getNegHeatStorage() - water);

                                excessLiquidWater = 0.0;
                                totalRefrozenWater = totalRefrozenWater + water;
                            }
                            else
                            {

                                _snow17ModelState.setLiquidWaterInSnowpack(_snow17ModelState.getLiquidWaterInSnowPack()
                                    + water - _snow17ModelState.getNegHeatStorage());
                                _snow17ModelState.setWaterEquivalentOfSolidSnow(_snow17ModelState.getWaterEquivalentOfSolidSnow()
                                    + _snow17ModelState.getNegHeatStorage());
                                totalRefrozenWater = totalRefrozenWater + _snow17ModelState.getNegHeatStorage();
                                _snow17ModelState.setNegHeatStorage(0.0);
                                excessLiquidWater = 0.0;
                            }

//                            if(_snow17ModelState.getNegHeatStorage() == 0.0)
                            if(MathHelper.isEqualToZero(_snow17ModelState.getNegHeatStorage()))
                            {
                                _snow17ModelState.setTempIndex(0.0);
                            }

                            // CALL ROUT19(ITPX,EXCESS,WE,AESC,STORGE,NEXLAG,EXLAG,PACKRO);
                            snowPackOutFlow = rout19(excessLiquidWater, _aesc);

                            snowPackOutFlow = snowPackOutFlow + gmro;
                            if(_printLog)
                            {
                                _logger.log(Logger.DEBUG,
                                            "SFALL=" + OHDUtilities.getFortranPrecison(sfall) + " RAIN="
                                                + OHDUtilities.getFortranPrecison(rain) + " MELT="
                                                + OHDUtilities.getFortranPrecison(melt) + " CNHS="
                                                + OHDUtilities.getFortranPrecison(cnhs) + " CNHSPX="
                                                + OHDUtilities.getFortranPrecison(cnhspx) + " AESC="
                                                + OHDUtilities.getFortranPrecison(_aesc) + " QNET="
                                                + OHDUtilities.getFortranPrecison(qnet) + " EXCESS="
                                                + OHDUtilities.getFortranPrecison(excessLiquidWater) + " GMRO="
                                                + OHDUtilities.getFortranPrecison(gmro) + " PACKRO="
                                                + OHDUtilities.getFortranPrecison(snowPackOutFlow) + " ROBG="
                                                + OHDUtilities.getFortranPrecison(robg));
                            }
                            // GOTO 190;
                            // 190
                            raimArray[loopCount - 1] = snowPackOutFlow + robg;
                            psnwro += snowPackOutFlow;
                            probg += robg;

                            continue mainForLoop;
                        } // end of if (WE > GM) block

                    } // end of blockBefore150

                    // 150
                    totalLagExcessWater = _snow17ModelState.getExLagTotal();

                    snowPackOutFlow = gmro + melt + totalLagExcessWater
                        + _snow17ModelState.getExcessLiquidWaterInStorage() + rain;

                    // CALL ZERO19();
                    _snow17ModelState.zero19();

                    _aesc = 0.0;
                    // SNDEN = -9.99;
                    _snow17ModelState.setSnowDensity(-9.99);
                    // GOTO 190;
                    // 190
                    raimArray[loopCount - 1] = snowPackOutFlow + robg;
                    psnwro += snowPackOutFlow;
                    probg += robg;

                    continue mainForLoop;
                } // end of if (!((PXI == 0.0) && (WE == 0.0)))

            } // end of blockBefore160

            // 160
            robg = currentPrecip;
            snowPackOutFlow = 0.0;
            _aesc = 0.0;
            // 190
            raimArray[loopCount - 1] = snowPackOutFlow + robg;
            psnwro += snowPackOutFlow;
            probg += robg;

        } // end of for loop, also the end of mainForLoop

        _snow17ModelState.setPqnet(pqnet);

        totalLagExcessWater = _snow17ModelState.getExLagTotal();

        _twe = _snow17ModelState.calculateTweFromState();

        blockBefore215:
        {
//            if(_twe == 0.0)
            if(MathHelper.isEqualToZero(_twe))
            {
                // SNDPT = 0.0;
                _snow17ModelState.setSnowDepth(0.0);
                // SNTMP = 0.0;
                _snow17ModelState.setAverageSnowTemp(0.0);
                // GOTO 215;
                break blockBefore215;
            }

            final double SLIQ = _snow17ModelState.getLiquidWaterInSnowPack() + totalLagExcessWater
                + _snow17ModelState.getExcessLiquidWaterInStorage();

            // CALL SNDEPTH19(WE,SLIQ,SXFALL,SXMELT,SXGMLOS,SXRFRZ,TA,DTA,IDT,SNDPT,SNDEN,SNTMP,IBUG);
            sndepth19(SLIQ,
                      totalNewSnowFall,
                      totalSurfaceMelt,
                      totalGroundMelt,
                      totalRefrozenWater,
                      airTemp,
                      deltaAirTemp);

            // CALL AESC19(WE,LIQW,ACCMAX,SB,SBAESC,SBWS,SI,ADC,AEADJ,AESC);
            _aesc = aesc19();

        } // end of blockBefore215

        // 215

        //stores the value before it is modified by updt19()
        final double cwe = _twe;
        final double caesc = _aesc;
        // CALL UPDT19:
        if(_powe >= 0.0 || _posc >= 0.0)
        {

            final DoubleHolder tweHolder = new DoubleHolder(_twe);
            final DoubleHolder aescHolder = new DoubleHolder(_aesc);

            this.updt19(_powe, _posc, tweHolder, aescHolder);

            _twe = tweHolder.getValue();
            _aesc = aescHolder.getValue();

//            if(_twe != cwe)
            if(MathHelper.isNotEqual(_twe, cwe))
            {
                _chgwe += _twe - cwe; //accumulates the change of twe
            }
        }

        // save the results into the Snow17Results object
        final Snow17Results snow17Results = new Snow17Results(raimArray,
                                                              _twe,
                                                              _aesc,
                                                              _snow17ModelState.getSnowDepth(),
                                                              prain,
                                                              psfall,
                                                              psnwro,
                                                              probg);

        this._snow17ModelState.setDateTime(currTimeInMillis);
        this._snow17ModelState.setTaprev(airTemp);

        if(_printLog)
        {
            _logger.log(Logger.DEBUG,
                        "OUTPUT DATA--TWE=" + OHDUtilities.getFortranPrecison(_twe) + " COVER="
                            + OHDUtilities.getFortranPrecison(_aesc) + " CWE=" + OHDUtilities.getFortranPrecison(cwe)
                            + " CAESC=" + OHDUtilities.getFortranPrecison(caesc) + " SNDPT="
                            + OHDUtilities.getFortranPrecison(_snow17ModelState.getSnowDepth()) + " SNDEN="
                            + OHDUtilities.getFortranPrecison(_snow17ModelState.getSnowDensity()) + " SNTMP="
                            + OHDUtilities.getFortranPrecison(_snow17ModelState.getAverageSnowTemp()));

            for(int i = 0; i < raimArray.length; i++)
            {
                _logger.log(Logger.DEBUG, "RM=" + OHDUtilities.getFortranPrecison(raimArray[i]) + " "
                    + _currDateTimeStr);
            }
        }

        return snow17Results;

    } // end of Pack19()

    /**
     * THIS METHOD COMPUTES THE AREAL EXTENT OF SNOW COVER USING THE AREAL DEPLETION CURVE FOR THE SNOW-17 OPERATION.
     * The method corresponds to Fortran subroutine AESC19(aesc19.f)
     * 
     * @return the double
     */
    public double aesc19()
    {
        // SUBROUTINE AESC19(WE,LIQW,ACCMAX,SB,SBAESC,SBWS,SI,ADC,AEADJ,AESC)
        double resultAESC = 0.0;
        double R = 0.0, areaIndex = 0.0;

        final double waterEquivalentWhenAllSnow = _snow17ModelParameters.getArealWeUnderFullSnowCover();

        // TWE = WE + LIQW
        final double tweAesc = _snow17ModelState.getWaterEquivalentOfSolidSnow()
            + _snow17ModelState.getLiquidWaterInSnowPack();

//        if(_snow17ModelState.getSnowArealExtentAdjustment() != 0.0)
        if(MathHelper.notEqualToZero(_snow17ModelState.getSnowArealExtentAdjustment()))
        {
            double oldAreaIndex = _snow17ModelState.getMaxWeSinceSnow();

            if(MathHelper.isGreater(_snow17ModelState.getMaxWeSinceSnow(), waterEquivalentWhenAllSnow))
            {
                oldAreaIndex = waterEquivalentWhenAllSnow;
            }

//            if(_snow17ModelState.getSnowArealExtentAdjustment() < oldAi && tweAesc >= oldAi)
            if(MathHelper.isLessThan(_snow17ModelState.getSnowArealExtentAdjustment(), oldAreaIndex)
                && MathHelper.isGreaterOrEqual(tweAesc, oldAreaIndex))
            {
                _snow17ModelState.setSnowArealExtentAdjustment(0.0);
            }

//            if(_snow17ModelState.getSnowArealExtentAdjustment() >= oldAi
//                && tweAesc >= _snow17ModelState.getSnowArealExtentAdjustment())
            if(MathHelper.isGreaterOrEqual(_snow17ModelState.getSnowArealExtentAdjustment(), oldAreaIndex)
                && MathHelper.isGreaterOrEqual(tweAesc, _snow17ModelState.getSnowArealExtentAdjustment()))
            {
                _snow17ModelState.setSnowArealExtentAdjustment(0.0);
            }
        }

        // if (TWE > ACCMAX) ACCMAX = TWE;
//        if(tweAesc > _snow17ModelState.getMaxWeSinceSnow())
        if(MathHelper.isGreater(tweAesc, _snow17ModelState.getMaxWeSinceSnow()))
        {
            _snow17ModelState.setMaxWeSinceSnow(tweAesc);
        }

        areaIndex = _snow17ModelState.getMaxWeSinceSnow();

        if(MathHelper.isGreater(_snow17ModelState.getMaxWeSinceSnow(), waterEquivalentWhenAllSnow))
        {
            areaIndex = waterEquivalentWhenAllSnow;
        }

//        if(_snow17ModelState.getSnowArealExtentAdjustment() > 0.0)
        if(MathHelper.isGreaterThanZero(_snow17ModelState.getSnowArealExtentAdjustment()))
        {
            areaIndex = _snow17ModelState.getSnowArealExtentAdjustment();
        }

        int indexForSearchingCurve = 0;

        int block = 0; //for logging purpose
        if(MathHelper.isGreaterOrEqual(tweAesc, areaIndex))
        { // IF (TWE .GE. AI) THEN

            block = 1;
            _snow17ModelState.setArealWaterEquivBeforeSnow(tweAesc);
            _snow17ModelState.setWaterEquivalentUnderSnowCover(tweAesc);

            resultAESC = 1.0;
        }
        else if(MathHelper.isLessOrEqual(tweAesc, _snow17ModelState.getArealWaterEquivBeforeSnow()))
        { // ELSEIF (TWE .LE. SB) THEN

            block = 2;

            R = tweAesc / areaIndex * 10.0 + 1.0;
            indexForSearchingCurve = (int)Math.floor(R);
            R = R - Math.floor(R);

            final double[] adc = _snow17ModelParameters.getAdc();

            resultAESC = adc[indexForSearchingCurve - 1]
                + (adc[indexForSearchingCurve] - adc[indexForSearchingCurve - 1]) * R;

            _snow17ModelState.setArealWaterEquivBeforeSnow(tweAesc);
            _snow17ModelState.setWaterEquivalentUnderSnowCover(tweAesc);
            _snow17ModelState.setAescBeforeSnow(resultAESC);

            //GOTO 120
        }
//        else if(tweAesc < _snow17ModelState.getWaterEquivalentUnderSnowCover())
        else if(MathHelper.isLessThan(tweAesc, _snow17ModelState.getWaterEquivalentUnderSnowCover()))
        { // ELSEIF (TWE .LT. SBWS) THEN

            block = 3;

            resultAESC = _snow17ModelState.getAescBeforeSnow()
                + (1.0 - _snow17ModelState.getAescBeforeSnow())
                * (tweAesc - _snow17ModelState.getArealWaterEquivBeforeSnow())
                / (_snow17ModelState.getWaterEquivalentUnderSnowCover() - _snow17ModelState.getArealWaterEquivBeforeSnow());

            //GOTO 120
        }
        else
        {
            resultAESC = 1.0;
        }

        // 120
//    if(resultAESC < 0.05)
        if(MathHelper.isLessThan(resultAESC, 0.05))
        {
            resultAESC = 0.05;
        }

//      if(resultAESC > 1.0)
        if(MathHelper.isGreater(resultAESC, 1.0))
        {
            resultAESC = 1.0;
        }
        if(_printLog)
        {
            _logger.log(Logger.DEBUG, "aesc() returns " + resultAESC + " at " + _currDateTimeStr + " by block " + block);
            _logger.log(Logger.DEBUG,
                        "TWE=" + tweAesc + " AI=" + areaIndex + " SB="
                            + _snow17ModelState.getArealWaterEquivBeforeSnow() + " SBWS="
                            + _snow17ModelState.getWaterEquivalentUnderSnowCover() + " WE="
                            + _snow17ModelState.getWaterEquivalentOfSolidSnow() + " LIQW="
                            + _snow17ModelState.getLiquidWaterInSnowPack());
        }

        return resultAESC;
    } // end of method Aesc19

    /**
     * SUBROUTINES COMPUTES SURFACE MELT BASED ON 100 PERCENT SNOW COVER AND NON-RAIN CONDITIONS. The method corresponds
     * to Fortran subroutine MELT19(melt19.f)
     * 
     * @param airTemp the air temp
     * @param meltHolder the melt holder, will return the value to the caller
     * @param cnhsHolder the cnhs holder, will return the value to the caller
     * @param timeLong the epoch time
     */
    private void melt19(final double airTemp, final DoubleHolder meltHolder, final DoubleHolder cnhsHolder)
    {
        // SUBROUTINE MELT19(IDN,IMN,ALAT,TA,MELT,MFMAX,MFMIN,MBASE,TINDEX, TIPM,CNHS,NMF,LMFV,SMFV)

        /* ------------Declare local variables and set the values--------- */
        double melt = meltHolder.getValue();
        double cnhs = cnhsHolder.getValue();

        // OTHER VARIABLES
        double meltFactor = 0.0;
        double nmRate = 0.0;

        if(_snow17ModelParameters.getMeltFactorVariationIndicator() == 0) // card6A is included
        {
            meltFactor = this.getMeltFactorWithNormalSmfv();
        }
        else
        {
            meltFactor = this.getMeltFactorWithSpecifiedSmfv();
        }

        final double ratio = meltFactor / _snow17ModelParameters.getMaxMeltFactor();

        //COMPUTE MELT AND NEGATIVE HEAT EXCHANGE INDEX TEMPERATURES
        double tmx = airTemp - _snow17ModelParameters.getBaseTemperature();

//        if(tmx < 0.0)
        if(MathHelper.isLessThanZero(tmx))
        {
            tmx = 0.0;
        }

        double tsur = airTemp;
//        if(tsur > 0.0)
        if(MathHelper.isGreaterThanZero(tsur))
        {
            tsur = 0.0;
        }

        final double tnmx = _snow17ModelState.getTempIndex() - tsur;

        //NEGATIVE HEAT EXCHANGE
        nmRate = ratio * _snow17ModelParameters.getNegMeltFactor();
        cnhs = nmRate * tnmx;

        //UPDATE TINDEX
        final double newTindex = _snow17ModelState.getTempIndex() + _snow17ModelParameters.getTemperatureIndex()
            * (airTemp - _snow17ModelState.getTempIndex());
        _snow17ModelState.setTempIndex(newTindex);

//        if(_snow17ModelState.getTempIndex() > 0.0)
        if(MathHelper.isGreaterThanZero(_snow17ModelState.getTempIndex()))
        {
            _snow17ModelState.setTempIndex(0.0);
        }

        //SURFACE MELT
//        if(tmx > 0.0)
        if(MathHelper.isGreaterThanZero(tmx))
        {
            melt = meltFactor * tmx;
        }

        /* ------------set the values to the Holders--------- */
        meltHolder.setValue(melt);
        cnhsHolder.setValue(cnhs);

        return;
    } // end of method Melt19()

    /**
     * Calculate melt factor when using normal seasonal melt-factor variation.
     */
    private double getMeltFactorWithNormalSmfv()
    {
        final int year = _currentDateTime.year();

        DateTime springDateTime = new DateTime(year, 3, 21, _startHourOfDay); //current year March 21st

        int numDaysSinceSpring;
        if(springDateTime.getTimeInMillis() < _currentDateTime.getTimeInMillis())
        {//current time is after March 21st
            numDaysSinceSpring = springDateTime.daysUntil(_currentDateTime);
        }
        else
        {//current time is before or equal to March 21st
            springDateTime = new DateTime(year - 1, 3, 21, _startHourOfDay); //last year March 21st

            numDaysSinceSpring = springDateTime.daysUntil(_currentDateTime);

            final DateTime beginMarchDateTime = new DateTime(year, 3, 1, _startHourOfDay);//Starting point of March

            /*
             * if time is in March and is not leap year, 2/29 still needs to be counted one. So increment IDN by one.
             * xxxx-03-01 00:00:00 or 12:00:00 is considered Feb. last day last time step, not in March yet, to be
             * consistent with NWSRFS.
             */
            if(OHDUtilities.isLeapYear(year) == false
                && _currentDateTime.getTimeInMillis() > beginMarchDateTime.getTimeInMillis())
            {
                numDaysSinceSpring += 1;
            }
        }

        //0 in Java is NWSRFS 24 of previous day
        if(_currentDateTime.hour() == _startHourOfDay)
        {
            numDaysSinceSpring -= 1;
        }

        double meltFactor;

        final double meltFactorDiff = _snow17ModelParameters.getMaxMeltFactor()
            - _snow17ModelParameters.getMinMeltFactor();

        if(_snow17ModelParameters.getLatitude() < 54.0)
        { // All the US states except Alaska. So this block is the one got really used, except basins in Alaska

            meltFactor = Math.sin(numDaysSinceSpring * 2.0 * 3.1416 / 366.0) * meltFactorDiff * 0.5
                + (_snow17ModelParameters.getMaxMeltFactor() + _snow17ModelParameters.getMinMeltFactor()) * 0.5;

        }
        else
        { // Alaska state
            double X = 0.0, XX = 0.0;
            double adjMf = 0.0;

            if(numDaysSinceSpring >= 275)
            {
                X = (numDaysSinceSpring - 275.0) / (458.0 - 275.0);
            }
            else if(numDaysSinceSpring >= 92)
            {
                X = (275.0 - numDaysSinceSpring) / (275.0 - 92.0);
            }
            else
            {
                X = (91.0 + numDaysSinceSpring) / 183.0;
            }
            XX = Math.sin(numDaysSinceSpring * 2.0 * 3.1416 / 366.0) * 0.5 + 0.5;

            //            if(X <= 0.48)
            if(MathHelper.isLessOrEqual(X, 0.48))
            {
                adjMf = 0.0;
            }
            //            else if(X >= 0.70)
            else if(MathHelper.isGreaterOrEqual(X, 0.70))
            {
                adjMf = 1.0;
            }
            else
            {
                adjMf = (X - 0.48) / (0.70 - 0.48);
            }

            meltFactor = XX * adjMf * meltFactorDiff + _snow17ModelParameters.getMinMeltFactor();
        }

        return meltFactor;
    }

    /**
     * Calculate melt factor when using user specified seasonal melt-factor variation.
     */
    private double getMeltFactorWithSpecifiedSmfv()
    {

        final int year = _currentDateTime.year();

        final int month = _currentDateTime.month(); //1: Jan

        final double[] smfv = _snow17ModelParameters.getSeasonalMeltFactorVariationArray();

        DateTime startDateTime = new DateTime(year, month, 16, _startHourOfDay); //current year, current month, 16th 12Z
        startDateTime.setStartOfDay(_startHourOfDay);

        if(_currentDateTime.getTimeInMillis() <= startDateTime.getTimeInMillis())
        {//first half of the month(prior to and including 16th 12Z), shift one month back

            startDateTime = startDateTime.addMonthsTo(-1);
        }

        int numDaysToInterpolate = startDateTime.daysUntil(_currentDateTime);

        /*
         * for time step hitting _startHourOfDay, daysUntil returns a number incremented by 1, but NWSRFS consider it
         * being the last time step of previous day
         */
        if(_currentDateTime.hour() == _startHourOfDay)
        {
            numDaysToInterpolate -= 1;
        }

        final DateTime endDateTime = startDateTime.addMonthsTo(1);

        final int totalNumDays = startDateTime.daysUntil(endDateTime);//from one month middle to next month's middle

        final double range = smfv[endDateTime.month() - 1] - smfv[startDateTime.month() - 1];

        final double adjMf = smfv[startDateTime.month() - 1] + (range / totalNumDays) * numDaysToInterpolate;

        final double meltFactorDiff = _snow17ModelParameters.getMaxMeltFactor()
            - _snow17ModelParameters.getMinMeltFactor();

        final double meltFactor = _snow17ModelParameters.getMinMeltFactor() + adjMf * meltFactorDiff;

        return meltFactor;

    }

    /**
     * THIS SUBROUTINE ROUTES EXCESS WATER THROUGH THE SNOW COVER FOR THE 'SNOW-17 ' OPERATION. The method corresponds
     * to Fortran subroutine ROUT19(rout19.f)
     * 
     * @param excessLiquidWater the excess liquid water
     * @param aesc the aesc
     * @return the double
     */
    private double rout19(final double excessLiquidWater, final double aesc)
    {
        // SUBROUTINE ROUT19(IT,EXCESS,WE,AESC,STORGE,NEXLAG,EXLAG,PACKRO)

        double snowPackOutFlow = 0.0;

        final double[] exlagArray = _snow17ModelState.getExlagArray(); //it's an alias to Snow17ModelState _exlag array
        final double CL = 0.03 * _snow17ModelParameters.getPrecipInterval() / 6.0; // don't know why fortran hard-coded 6.0 here

        double term = 0.0;

        // LAG EXCESS WATER FIRST - FUNCTION OF EXCESS AND WE.
        blockBefore150:
        {
//            if(excessLiquidWater != 0.0)
            if(MathHelper.notEqualToZero(excessLiquidWater))
            {
//                if(excessLiquidWater >= 0.1)
                if(MathHelper.isGreaterOrEqual(excessLiquidWater, 0.1))
                {
//                    if(_snow17ModelState.getWaterEquivalentOfSolidSnow() >= 1.0)
                    if(MathHelper.isGreaterOrEqual(_snow17ModelState.getWaterEquivalentOfSolidSnow(), 1.0))
                    {
                        // COMPUTE LAG IN HOURS AND PRORATE EXCESS.
                        int N = (int)(Math.pow(excessLiquidWater * 4, 0.3) + 0.5);

                        if(N == 0)
                        {
                            N = 1;
                        }

                        final double FN = N;

                        for(int I = 1; I <= N; I++)
                        {

                            final double FI = I;
                            term = CL * _snow17ModelState.getWaterEquivalentOfSolidSnow() * FN
                                / (excessLiquidWater * (FI - 0.5));

//                            if(term > 150.0)
                            if(MathHelper.isGreater(term, 150.0))
                            {
                                term = 150.0;
                            }

                            final double FLAG = 5.33 * (1.0 - Math.exp(-term));
                            // L2 = (FLAG+FIT)/FIT + 1.0
                            final int L2 = (int)((FLAG + _snow17ModelParameters.getPrecipInterval())
                                / _snow17ModelParameters.getPrecipInterval() + 1.0);
                            final int L1 = L2 - 1;
                            final double ENDL1 = L1 * _snow17ModelParameters.getPrecipInterval();
                            final double POR2 = (FLAG + _snow17ModelParameters.getPrecipInterval() - ENDL1)
                                / _snow17ModelParameters.getPrecipInterval();
                            final double POR1 = 1.0 - POR2;
                            exlagArray[L2 - 1] = exlagArray[L2 - 1] + POR2 * excessLiquidWater / FN;
                            exlagArray[L1 - 1] = exlagArray[L1 - 1] + POR1 * excessLiquidWater / FN;

                        }
                        // GOTO 150
                        break blockBefore150;
                    }
                } // end of if (EXCESS >= 0.1)
                  // EXCESS OR WE SMALL, THUS NO LAG.
                exlagArray[0] = exlagArray[0] + excessLiquidWater;
            } // end of if (EXCESS != 0.0)
        } // end of blockBefore150

        // 150 ATTENUATE LAGGED EXCESS WATER - FUNCTION OF STORGE AND WE.
//        if((_snow17ModelState.getExcessLiquidWaterInStorage() + exlagArray[0]) != 0.0)

        if(MathHelper.notEqualToZero(_snow17ModelState.getExcessLiquidWaterInStorage() + exlagArray[0]))
        {

//            if((_snow17ModelState.getExcessLiquidWaterInStorage() + exlagArray[0]) >= 0.1)
            if(MathHelper.isGreaterOrEqual((_snow17ModelState.getExcessLiquidWaterInStorage() + exlagArray[0]), 0.1))
            {
                // EFFECT OF ATTENUATION COMPUTED USING A ONE-HOUR TEME STEP.
                final double el = exlagArray[0] / _snow17ModelParameters.getPrecipInterval();
                final double els = el / (25.4 * aesc);
                final double wes = _snow17ModelState.getWaterEquivalentOfSolidSnow() / (25.4 * aesc);
                term = 500.0 * els / Math.pow(wes, 1.3);

//                if(term > 150.0)
                if(MathHelper.isGreater(term, 150.0))
                {
                    term = 150.0;
                }

                final double R1 = 1.0 / (5.0 * Math.exp(-term) + 1.0);

                for(int I = 1; I <= _snow17ModelParameters.getPrecipInterval(); I++)
                {

                    final double OS = (_snow17ModelState.getExcessLiquidWaterInStorage() + el) * R1;
                    snowPackOutFlow = snowPackOutFlow + OS;
                    _snow17ModelState.setExcessLiquidWaterInStorage(_snow17ModelState.getExcessLiquidWaterInStorage()
                        + el - OS);
                }

//                if(_snow17ModelState.getExcessLiquidWaterInStorage() <= 0.001)
                if(MathHelper.isLessOrEqual(_snow17ModelState.getExcessLiquidWaterInStorage(), 0.001))
                {
                    snowPackOutFlow = snowPackOutFlow + _snow17ModelState.getExcessLiquidWaterInStorage();
                    _snow17ModelState.setExcessLiquidWaterInStorage(0.0);
                }

            }
            else
            {
                // NO ATTENUATION
                snowPackOutFlow = _snow17ModelState.getExcessLiquidWaterInStorage() + exlagArray[0];
                _snow17ModelState.setExcessLiquidWaterInStorage(0.0);
            }
        }

        // DOWNSHIFT WATER IN EXLAG().
        for(int I = 2; I <= _snow17ModelState.getExlagArray().length; I++)
        {

            exlagArray[I - 2] = exlagArray[I - 1];
        }

        exlagArray[_snow17ModelState.getExlagArray().length - 1] = 0.0;

        return snowPackOutFlow;
    } // end of method Rout19()

    /**
     * SUBROUTINE CALCULATES SNOW DEPTH AND SNOW TEMPERATURE FOR THE COMPUTATIONAL TIME PERIOD. The method corresponds
     * to Fortran subroutine SNDEPTH19(sndepth19.f)
     * 
     * @param liquidWaterContent the liquid water content
     * @param newSnowFall the new snow fall
     * @param surfaceMelt the surface melt
     * @param groundMelt the ground melt
     * @param refrozenWater the refrozen water
     * @param airTemp the air temp
     * @param deltaAirTemp the delta air temp
     */
    private void sndepth19(final double liquidWaterContent,
                           final double newSnowFall,
                           final double surfaceMelt,
                           final double groundMelt,
                           final double refrozenWater,
                           final double airTemp,
                           final double deltaAirTemp)
    {
        // SUBROUTINE SNDEPTH19(WE,SLIQ,SFALL,SMELT,SGMLOS,SRFRZ,TA,DTA,IDT,SNDPT,SNDEN,SNTMP,IBUG)

        // OTHER VARIABLES
        double newSnowTemp = airTemp; // TEMP OF NEW SNOW, C
        double newSnowDepth = 0.0; // DEPTH OF NEW SNOW, CM
        final DoubleHolder dptnewHolder = new DoubleHolder(newSnowDepth);
        double newSnowDensity = 0.0; // DENSITY OF NEW SNOW, G/CM3
        final DoubleHolder dennewHolder = new DoubleHolder(newSnowDensity);
        double densityIncrease = 0.0; // DENSITY INCREASE DUE TO MELT METAPHORISM (REFREEZING OF LIQUID WATER),
        double weBegin = 0.0; // VALUE OF WE AT THE START OF THE PERIOD, MM
        double waterEquivalentOfExistingSnow = 0.0; // WATER EQUIVALENT OF EXISTING SNOW AT END OF PERIOD, MM
        double oldSnowDepth = 0.0; // DEPTH OF OLD SNOW AFTER ANY MELT, CM
        double totalSnowDepth = 0.0; // TOTAL DEPTH (OLD AND NEW SNOW), CM
        double totalSnowDensity = 0.0; // TOTAL DENSITY (OLD AND NEW SNOW), G/CM3

        final double totalMelt = surfaceMelt + groundMelt;
        //PLOSS: TOTAL MELT DURING THE PERIOD (SURFACE MELT PLUS GROUNDMELT), MM

        final double snowFallMinusMelt = newSnowFall - totalMelt;
        //SFALLX: SNOWFALL MINUS MELT, MM

        if(_printLog)
        {
            _logger.log(Logger.DEBUG,
                        "SNDEPTH19 INPUT--SNDPT=" + OHDUtilities.getFortranPrecison(_snow17ModelState.getSnowDepth())
                            + " SNDEN=" + OHDUtilities.getFortranPrecison(_snow17ModelState.getSnowDensity())
                            + " SNTMP=" + OHDUtilities.getFortranPrecison(_snow17ModelState.getAverageSnowTemp())
                            + " TA=" + airTemp + " DTA=" + deltaAirTemp + " WE="
                            + OHDUtilities.getFortranPrecison(_snow17ModelState.getWaterEquivalentOfSolidSnow())
                            + " SLIQ=" + OHDUtilities.getFortranPrecison(liquidWaterContent) + " SFALL="
                            + OHDUtilities.getFortranPrecison(newSnowFall) + " SMELT="
                            + OHDUtilities.getFortranPrecison(surfaceMelt) + " SGMLOS="
                            + OHDUtilities.getFortranPrecison(groundMelt) + " SRFRZ="
                            + OHDUtilities.getFortranPrecison(refrozenWater));
        }
//        if(_snow17ModelState.getSnowDepth() > 0.0)
        if(MathHelper.isGreater(_snow17ModelState.getSnowDepth(), 0.0))
        {
            //PLOSS=SMELT+SGMLOS(PLOSS - totalMelt: code moved to the top of the method)

            //compute WE value at the start of the period - existing snow
            weBegin = _snow17ModelState.getWaterEquivalentOfSolidSnow() - newSnowFall + totalMelt - refrozenWater;

            //check for various cases:
//            if(totalMelt < weBegin)
            if(MathHelper.isLessThan(totalMelt, weBegin))
            {

                /*
                 * Amount of melt < existing snow at start of period. Assuming all melt came from exsiting snow(typical
                 * case). Check for any new snowfall during period. If any, get depth and density
                 */
//                if(newSnowFall > 0.0)
                if(MathHelper.isGreaterThanZero(newSnowFall))
                {
                    dptnewHolder.setValue(newSnowDepth);
                    dennewHolder.setValue(newSnowDensity);

                    snew(newSnowTemp, newSnowFall, dptnewHolder, dennewHolder);

                    // retrieve values after calling
                    newSnowDepth = dptnewHolder.getValue();
                    newSnowDensity = dennewHolder.getValue();
                }

                //compute water equivalent of existing snow at end of period - prior to any refreeze
                waterEquivalentOfExistingSnow = weBegin - totalMelt;
                //compute the depth of old snow
                oldSnowDepth = 0.1 * waterEquivalentOfExistingSnow / _snow17ModelState.getSnowDensity();
                totalSnowDepth = oldSnowDepth + newSnowDepth;
                totalSnowDensity = (oldSnowDepth * _snow17ModelState.getSnowDensity() + newSnowDepth * newSnowDensity)
                    / totalSnowDepth;

                //compute new temperature of old snow
                // CALL SNOWT(TSNDPT,TSNDEN,WE,SLIQ,TA,DTA,SNTMP,DPTNEW,IDT);
                snowt(totalSnowDepth, totalSnowDensity, liquidWaterContent, deltaAirTemp, newSnowDepth);

                //compute new depth and density of old snow
                // CALL SNOWPACK(WEX,DT,SNDPT,SNDEN,SLIQ,SNTMP,IBUG);
                snowpack(waterEquivalentOfExistingSnow, liquidWaterContent);
                if(_printLog)
                {
                    _logger.log(Logger.DEBUG,
                                "MELT FROM OLD SNOW--SNDPT="
                                    + OHDUtilities.getFortranPrecison(_snow17ModelState.getSnowDepth()) + " SNDEN="
                                    + OHDUtilities.getFortranPrecison(_snow17ModelState.getSnowDensity()) + " DPTNEW="
                                    + OHDUtilities.getFortranPrecison(newSnowDepth) + " DENNEW="
                                    + OHDUtilities.getFortranPrecison(newSnowDensity) + " SNTMP="
                                    + OHDUtilities.getFortranPrecison(_snow17ModelState.getAverageSnowTemp())
                                    + " TSNEW=" + OHDUtilities.getFortranPrecison(newSnowTemp));
                }
            }
            else
            {//totalMelt >= weBegin

                //SFALLX=SFALL-PLOSS (code moved to the top)

//                if(snowFallMinusMelt > 0.0)
                if(MathHelper.isGreater(snowFallMinusMelt, 0.0))
                {
                    // preparing for calling method Snew()
                    dptnewHolder.setValue(newSnowDepth);
                    dennewHolder.setValue(newSnowDensity);

                    // CALL SNEW(TSNEW,SFALLX,DPTNEW,DENNEW);
                    snew(newSnowTemp, snowFallMinusMelt, dptnewHolder, dennewHolder);

                    // retrieve values after calling
                    newSnowDepth = dptnewHolder.getValue();
                    newSnowDensity = dennewHolder.getValue();

                    totalSnowDepth = _snow17ModelState.getSnowDepth() + newSnowDepth;
                    totalSnowDensity = (_snow17ModelState.getSnowDepth() * _snow17ModelState.getSnowDensity() + newSnowDepth
                        * newSnowDensity)
                        / totalSnowDepth;

                    // CALL SNOWT(TSNDPT,TSNDEN,WE,SLIQ,TA,DTA,SNTMP,DPTNEW,IDT)
                    snowt(totalSnowDepth, totalSnowDensity, liquidWaterContent, deltaAirTemp, newSnowDepth);

                    // CALL SNOWPACK(WE1,DT,SNDPT,SNDEN,SLIQ,SNTMP,IBUG)
                    snowpack(weBegin, liquidWaterContent);
                    if(_printLog)
                    {
                        _logger.log(Logger.DEBUG,
                                    "MELT FROM NEW SNOW--SNDPT="
                                        + OHDUtilities.getFortranPrecison(_snow17ModelState.getSnowDepth()) + " SNDEN="
                                        + OHDUtilities.getFortranPrecison(_snow17ModelState.getSnowDensity())
                                        + " DPTNEW=" + OHDUtilities.getFortranPrecison(newSnowDepth) + " DENNEW="
                                        + OHDUtilities.getFortranPrecison(newSnowDensity) + " SNTMP="
                                        + OHDUtilities.getFortranPrecison(_snow17ModelState.getAverageSnowTemp())
                                        + " TSNEW=" + OHDUtilities.getFortranPrecison(newSnowTemp));
                    }

                } // close if (SFALLX > 0.0)
                else
                {//snowFallMinusMelt <= 0.0

                    waterEquivalentOfExistingSnow = _snow17ModelState.getWaterEquivalentOfSolidSnow() - refrozenWater;
                    totalSnowDepth = 0.1 * waterEquivalentOfExistingSnow / _snow17ModelState.getSnowDensity();

                    // CALL SNOWT(TSNDPT,SNDEN,WE,SLIQ,TA,DTA,SNTMP,DPTNEW,IDT);
                    snowt(totalSnowDepth,
                          _snow17ModelState.getSnowDensity(),
                          liquidWaterContent,
                          deltaAirTemp,
                          newSnowDepth);

                    // CALL SNOWPACK(WEX,DT,SNDPT,SNDEN,SLIQ,SNTMP,IBUG);
                    snowpack(waterEquivalentOfExistingSnow, liquidWaterContent);
                    if(_printLog)
                    {
                        _logger.log(Logger.DEBUG,
                                    "ALL NEW SNOW MELTS--SNDPT="
                                        + OHDUtilities.getFortranPrecison(_snow17ModelState.getSnowDepth()) + " SNDEN="
                                        + OHDUtilities.getFortranPrecison(_snow17ModelState.getSnowDensity())
                                        + " DPTNEW=" + OHDUtilities.getFortranPrecison(newSnowDepth) + " DENNEW="
                                        + OHDUtilities.getFortranPrecison(newSnowDensity) + " SNTMP="
                                        + OHDUtilities.getFortranPrecison(_snow17ModelState.getAverageSnowTemp())
                                        + " TSNEW=" + OHDUtilities.getFortranPrecison(newSnowTemp));
                    }
                }
            }

//            if(newSnowTemp > 0.0)
            if(MathHelper.isGreaterThanZero(newSnowTemp))
            {
                newSnowTemp = 0.0;
            }

            _snow17ModelState.setAverageSnowTemp((_snow17ModelState.getAverageSnowTemp()
                * _snow17ModelState.getSnowDepth() + newSnowTemp * newSnowDepth)
                / (_snow17ModelState.getSnowDepth() + newSnowDepth));
            _snow17ModelState.setSnowDepth(_snow17ModelState.getSnowDepth() + newSnowDepth);
            _snow17ModelState.setSnowDensity(0.1 * (_snow17ModelState.getWaterEquivalentOfSolidSnow() - refrozenWater)
                / _snow17ModelState.getSnowDepth());

//            if(refrozenWater != 0.0)
            if(MathHelper.notEqualToZero(refrozenWater))
            {
                densityIncrease = _snow17ModelState.getWaterEquivalentOfSolidSnow()
                    / (_snow17ModelState.getWaterEquivalentOfSolidSnow() - refrozenWater);
                _snow17ModelState.setSnowDensity(_snow17ModelState.getSnowDensity() * densityIncrease);
            }

//            if(_snow17ModelState.getSnowDensity() > 0.6)
            if(MathHelper.isGreater(_snow17ModelState.getSnowDensity(), 0.6))
            {
                _snow17ModelState.setSnowDensity(0.6);
            }

//            if(_snow17ModelState.getSnowDensity() < 0.05)
            if(MathHelper.isLessThan(_snow17ModelState.getSnowDensity(), 0.05))
            {
                _snow17ModelState.setSnowDensity(0.05);
            }

            _snow17ModelState.setSnowDepth(0.1 * _snow17ModelState.getWaterEquivalentOfSolidSnow()
                / _snow17ModelState.getSnowDensity());
        } //close if(_snow17ModelState.getSnowDepth() > 0.0) 
        else
        {
            /*
             * SNDPT <= 0.0: no snow cover at start of period. But WE > 0.0(if not, the method would not be called);
             * SFALL > 0.0
             */

            //compute net snowfall during the period - snowfall minus any surface and ground melt
            //SFALLX = SFALL - SMELT - SGMLOS (SFALLX - snowFallMinusMelt: code moved to the top of the method)
            // preparing for calling method Snew(): compute the depth and density of net snowfall
            dptnewHolder.setValue(newSnowDepth);
            dennewHolder.setValue(newSnowDensity);

            // CALL SNEW(TSNEW,SFALLX,DPTNEW,DENNEW);
            snew(newSnowTemp, snowFallMinusMelt, dptnewHolder, dennewHolder);

            // retrieve values after calling
            newSnowDepth = dptnewHolder.getValue();
            newSnowDensity = dennewHolder.getValue();

//            if(refrozenWater > 0.0)
            if(MathHelper.isGreaterThanZero(refrozenWater))
            {
                densityIncrease = (snowFallMinusMelt + refrozenWater) / snowFallMinusMelt;
                newSnowDensity = densityIncrease * newSnowDensity;
                newSnowDepth = 0.1 * _snow17ModelState.getWaterEquivalentOfSolidSnow() / newSnowDensity;
            }
            _snow17ModelState.setSnowDepth(newSnowDepth);
            _snow17ModelState.setSnowDensity(newSnowDensity);
            _snow17ModelState.setAverageSnowTemp(newSnowTemp);

//            if(_snow17ModelState.getAverageSnowTemp() > 0.0)
            if(MathHelper.isGreaterThanZero(_snow17ModelState.getAverageSnowTemp()))
            {
                _snow17ModelState.setAverageSnowTemp(0.0);
            }
            if(_printLog)
            {
                _logger.log(Logger.DEBUG,
                            "NEW SNOW ONLY OUTPUT--SFALLX=" + OHDUtilities.getFortranPrecison(snowFallMinusMelt)
                                + " SNDPT=" + OHDUtilities.getFortranPrecison(_snow17ModelState.getSnowDepth())
                                + " SNDEN=" + OHDUtilities.getFortranPrecison(_snow17ModelState.getSnowDensity())
                                + " SNTMP=" + OHDUtilities.getFortranPrecison(_snow17ModelState.getAverageSnowTemp()));
            }
        }
        if(_printLog)
        {
            _logger.log(Logger.DEBUG,
                        "SNDEPTH19 OUTPUT--SNDPT=" + OHDUtilities.getFortranPrecison(_snow17ModelState.getSnowDepth())
                            + " SNDEN=" + OHDUtilities.getFortranPrecison(_snow17ModelState.getSnowDensity())
                            + " SNTMP=" + OHDUtilities.getFortranPrecison(_snow17ModelState.getAverageSnowTemp()));
        }

        return;
    } // end of method Sndepth19()

    /**
     * SUBROUTINE CALCULATES AVERAGE SNOW TEMPERATURE. The method corresponds to Fortran subroutine SNOWT, located in
     * sndepth.f
     * 
     * @param snDpt the snow depth
     * @param snDen the snow density
     * @param sliq the sliq
     * @param dta the dta
     * @param dptNew the dpt new
     */
    private void snowt(final double snDpt, final double snDen, final double sliq, final double dta, final double dptNew)
    {
        // SUBROUTINE SNOWT(SNDPT,SNDEN,WE,SLIQ,TA,DTA,SNTMP,DPTNEW,IDT)

        // SPECIFIC HEAT CAPACITIES OF ICE, WATER, AND AIR. UNITS ARE WATTS
        // SEC/(M**3 C)
        // PARAMETER (CICE=2.1E06,CH2O=4.2E06,CAIR=1E03)
        final double cice = 2.1E06;
        final double CH2O = 4.2E06;
        final double cAir = 1E03;

        // HEAT WAVE LENGTH, IN SECONDS
        final double waveLength = 2.0 * 3600. * _snow17ModelParameters.getAirTempInterval();

        // CONVERT DEPTHS TO METERS
        final double snDptMeter = 0.01 * snDpt;
        final double dptNewMeter = 0.01 * dptNew;

        // GET TOTAL WATER EQUIVALENT AND DENSITY (SOLID PLUS LIQUID WATER)
        final double weTotal = _snow17ModelState.getWaterEquivalentOfSolidSnow() + sliq;
        final double denTotal = 0.1 * weTotal / snDpt;

        // COMPUTE THERMAL CONDUCTIVITY
        final double thermalConduct = 0.0442 * Math.exp(5.181 * denTotal);

        // COMPUTE FRACTION LIQUID
        final double fractLiquid = sliq / weTotal;

        // COMPUTE EFFECTIVE SPECIFIC VOLUMETRIC HEAT CAPACITY
        final double heatCapacity = cice * snDen + cAir * (1. - snDen - fractLiquid) + CH2O * fractLiquid;

        // COMPUTE NEW SNOW COVER TEMPERATURE
        final double snowCoverTemp = Math.sqrt(3.14 * heatCapacity / (waveLength * thermalConduct));

        double snowTemp;
//        if(dptNew > 0.)
        if(MathHelper.isGreaterThanZero(dptNew))
        {

            snowTemp = _snow17ModelState.getAverageSnowTemp() + dta
                * (Math.exp(-snowCoverTemp * dptNewMeter) - Math.exp(-snowCoverTemp * snDptMeter))
                / (snowCoverTemp * (snDptMeter - dptNewMeter));

        }
        else
        {
            snowTemp = _snow17ModelState.getAverageSnowTemp() + dta * (1. - Math.exp(-snowCoverTemp * snDptMeter))
                / (snowCoverTemp * snDptMeter);
        }

//        if(_snow17ModelState.getAverageSnowTemp() > 0.)
        if(MathHelper.isGreaterThanZero(snowTemp))
        {
            snowTemp = 0.0;
        }

        _snow17ModelState.setAverageSnowTemp(snowTemp);

        return;
    } // end of method Snowt()

    /**
     * CALCULATES DEPTH AND DENSITY OF THE NEW SNOWFALL<br>
     * (INPUT) SFALL - NET NEW SNOWFALL, MM (INPUT) DPTNEW - SNOW DEPTH, CM<br>
     * (OUTPUT) DENNEW - SNOW DENSITY (OUTPUT)<br>
     * ** CALCULATE NEW SNOWFALL DENSITY DEPENDING ON TEMPERATURE ** EQUATION FROM ANDERSON - NOAA TECH REPORT NWS 19 **
     * BASED ON DATA FROM ALTA, UTAH<br>
     * The method corresponds to Fortran subroutine SNEW, located in sndepth.f
     * 
     * @param tsNew the ts new
     * @param snowFall the snow fall
     * @param dptnewHolder the dptnew holder
     * @param dennewHolder the dennew holder
     */
    private void snew(final double tsNew,
                      final double snowFall,
                      final DoubleHolder dptnewHolder,
                      final DoubleHolder dennewHolder)
    {
        // SUBROUTINE SNEW (TSNEW,SFALL,DPTNEW,DENNEW )

        /* ---------Declare local variables and set the values from Holders ------ */
        double dptNew = dptnewHolder.getValue();
        double denNew = dennewHolder.getValue();

        // CALCULATE NEW SNOW DENSITY BASED ON TEMPERATURE
//        if(tsNew <= -15.)
        if(MathHelper.isLessOrEqual(tsNew, -15.))
        {
            denNew = 0.05;
        }
        else
        {
            denNew = 0.05 + 0.0017 * Math.pow((tsNew + 15.), 1.5);
        }

        dptNew = 0.1 * snowFall / denNew;

        /* ---------------Set values back to Holders --------------- */
        dptnewHolder.setValue(dptNew);
        dennewHolder.setValue(denNew);

        return;
    } // end of method Snew()

    /**
     * SUBROUTINE CALCULATES SNOW COMPACTION AND DESTRUCTIVE METAMORPHISM<br>
     * INPUT -SNDPT AND SNDEN AT BEGINNING OF PERIOD<br>
     * OUTPUT - SNDPT AND SNDEN - VALUES FOR REMAINING EXISTING SNOW AFTER COMPACTION AND DESTRUCTIVE METAMORPHISM<br>
     * The method corresponds to Fortran subroutine SNOWPACK, located in sndepth.f
     * 
     * @param we the we
     * @param sliq the sliq
     */
    private void snowpack(final double we, final double sliq)
    {
        // SUBROUTINE SNOWPACK (WE,DT,SNDPT,SNDEN,SLIQ,SNTMP,IBUG)
        // PARAMETER (C1=0.026, C2=21.0)
        // PARAMETER (C3=0.005, C4=0.10, THRESD=0.15, C5=2.,CX=23.)

        //stores the original value for logging purpose
        final double sndpt1 = _snow17ModelState.getSnowDepth();
        final double snden1 = _snow17ModelState.getSnowDensity();

        final double wecm = we * 0.1;

        double B = 0.0;
        final double C1 = 0.026, C2 = 21.0, C3 = 0.005, C4 = 0.10, C5 = 2.0, CX = 23.0;
        final double thresd = 0.15;
        double dcc = 1.;

//        if(wecm > 0.01)
        if(MathHelper.isGreater(wecm, 0.01))
        {

            B = _snow17ModelParameters.getAirTempInterval() * C1
                * Math.exp(0.08 * _snow17ModelState.getAverageSnowTemp() - C2 * _snow17ModelState.getSnowDensity());
            dcc = (Math.exp(B * wecm) - 1.) / (B * wecm);
        }

        double A1 = C3;

//        if(sliq > 0.)
        if(MathHelper.isGreaterThanZero(sliq))
        {
            A1 = A1 * C5;
        }

        double A2 = C4 * _snow17ModelState.getAverageSnowTemp();

//        if(_snow17ModelState.getSnowDensity() > thresd)
        if(MathHelper.isGreater(_snow17ModelState.getSnowDensity(), thresd))
        {
            A2 = A2 - CX * (_snow17ModelState.getSnowDensity() - thresd);
        }

        final double A = A1 * _snow17ModelParameters.getAirTempInterval() * Math.exp(A2);
        final double dcd = Math.exp(A);

        _snow17ModelState.setSnowDensity(_snow17ModelState.getSnowDensity() * dcc * dcd);

        _snow17ModelState.setSnowDepth(wecm / _snow17ModelState.getSnowDensity());
        if(_printLog)
        {
            _logger.log(Logger.DEBUG, "SNOWPACK INPUT--WE=" + OHDUtilities.getFortranPrecison(we) + " SNDPT="
                + OHDUtilities.getFortranPrecison(sndpt1) + " SNDEN=" + OHDUtilities.getFortranPrecison(snden1)
                + " SNTMP=" + OHDUtilities.getFortranPrecison(_snow17ModelState.getAverageSnowTemp()) + " SLIQ="
                + OHDUtilities.getFortranPrecison(sliq));

            _logger.log(Logger.DEBUG,
                        "   OUTPUT - SNDPT=" + OHDUtilities.getFortranPrecison(_snow17ModelState.getSnowDepth())
                            + " SNDEN=" + OHDUtilities.getFortranPrecison(_snow17ModelState.getSnowDensity()) + " DCC="
                            + OHDUtilities.getFortranPrecison(dcc) + " DCD=" + OHDUtilities.getFortranPrecison(dcd));
        }

        return;
    } // end of method Snowpack()

    /**
     * This method updates water-equivalent(twe) and snow cover(aesc) based on observation data(owe, osc). Affected
     * carryover values are adjusted. If owe is available, when exceeding WE tolerance(WETOL), set twe to owe.
     */
    public void updt19(double owe, double osc, final DoubleHolder tweHolder, final DoubleHolder aescHolder)
    {
//        if(owe < 0.0 && osc < 0.0)
        if(MathHelper.isLessThanZero(owe) && MathHelper.isLessThanZero(osc))
        {
            return; //neither owe or osc is available, return quickly
        }

        //if reach here, owe or osc or both are available

        // CALL UPDT19:
        double twe = tweHolder.getValue();
        double aesc = aescHolder.getValue();

        final double[] adc = _snow17ModelParameters.getAdc();

        final double si = _snow17ModelParameters.getArealWeUnderFullSnowCover();

        // update swe if observed we(owe) is available
//        if(owe >= 0.0)
        if(MathHelper.isGreaterOrEqual(owe, 0.0))
        {
//            if(owe < 1.0)
            if(MathHelper.isLessThan(owe, 1.0))
            {
                owe = 0.0;
            }

            // update swe(twe) if difference exceeds tolerance
//            if(Math.abs(twe - owe) > (twe * _snow17ModelParameters.getWeToleranceFaction()))
            if(MathHelper.isGreater(Math.abs(twe - owe), (twe * _snow17ModelParameters.getWeToleranceFaction())))
            {
                twe = owe; //since owe >= 0.0, so twe >= 0.0

                // update carryover values to be consistent with this change
//                if(twe > 0.0)
                if(MathHelper.isGreaterThanZero(twe))
                {
                    // CALL ADJC19
                    // adjust carryover values based on new water-equivalent
                    _snow17ModelState.adjc19(twe, si, adc);

                    // CALL AESC19
                    aesc = aesc19(); // compute areal extent of snow based on new we

                }
                else
                { // no snow remains after update - cannot update areal snow cover even if observed value exists
                  // CALL ZERO19
                    _snow17ModelState.zero19();
                    aesc = 0.0;
                }
            }
        } // close if(owe >= 0.0)

        // update areal extent of snow-cover(aesc) if observed aesc(osc) available and a snow-cover exists
//        if(osc >= 0.0 && twe >= 0.1)
        if(MathHelper.isGreaterOrEqual(osc, 0.0) && MathHelper.isGreaterOrEqual(twe, 0.1))
        {
            // update aesc if difference exceeds tolerance - also change aeadj
//            if(Math.abs(osc - aesc) > _snow17ModelParameters.getScToleranceFaction() && osc > 0.05)
            if(MathHelper.isGreater(Math.abs(osc - aesc), _snow17ModelParameters.getScToleranceFaction())
                && MathHelper.isGreater(osc, 0.05))
            {
//                if(osc > 1.0)
                if(MathHelper.isGreater(osc, 1.0))
                {
                    osc = 1.0;
                }

                aesc = osc;

                // CALL AECO19
                _snow17ModelState.aeco19(aesc, twe, si, adc);
            }
        }

        tweHolder.setValue(twe);
        aescHolder.setValue(aesc);

    }//close method

    /**
     * Same as in Fortran code, at hour=24, log the values, which are from the period of 24 hours prior to the current
     * time.
     * 
     * @param dayOfMonth - the next day of dayOfMonth after the current time
     */
    public void logDailyValues(final int dayOfMonth)
    {
        if(_printTableLog)
        {
            //change liquid water content to a percentage
            final double pliqw = 100.0 * _snow17ModelState.getLiquidWaterInSnowPack()
                / _snow17ModelState.getWaterEquivalentOfSolidSnow();

            //if RAIN-SNOW-ELEVATION has never been used on this day, print out blank string
            String rslStr;
            if(_ndrsp == 0)
            {
                rslStr = OHDUtilities.getBlankString(10);
            }
            else
            {
                rslStr = OHDUtilities.getFormatString(_drsl / _ndrsp, 0, 10);
            }
            //construct the String as the table format shown in Fortran output file, same precision
            _tableLogMessage.append(OHDUtilities.getFormatString(dayOfMonth, 7))
                            .append(OHDUtilities.getFormatString(_dsfall, 1, 14))
                            .append(OHDUtilities.getFormatString(_drain, 1, 17))
                            .append(OHDUtilities.getFormatString(_dqnet, 1, 18))
                            .append(OHDUtilities.getFormatString(_aesc, 2, 12))
                            .append(OHDUtilities.getFormatString(pliqw, 1, 14))
                            .append(OHDUtilities.getFormatString(_snow17ModelState.getNegHeatStorage(), 1, 15))
                            .append(OHDUtilities.getFormatString(_snow17ModelState.getAverageSnowTemp(), 1, 14))
                            .append(OHDUtilities.getFormatString(_snow17ModelState.getSnowDepth(), 1, 14))
                            .append(OHDUtilities.getFormatString(_podpt, 1, 15))
                            .append(OHDUtilities.getFormatString(_twe, 0, 11))
                            .append(OHDUtilities.getFormatString(_powe, 0, 13))
                            .append(OHDUtilities.getFormatString(_posc, 2, 13))
                            .append(rslStr)
                            .append(OHDConstants.NEW_LINE);
        }
        //reset the daily values to 0.0
        _dqnet = 0.0;
        _dsfall = 0.0;
        _drain = 0.0;
        _drsl = 0.0;
        _ndrsp = 0;
    }

    /**
     * Print out the table format logging message at DEBUG level after all the computation has been done. Only need to
     * do this when in forcasting mode, not MCP3 mode.
     */
    public void logTableMessage()
    {
        if(_printTableLog)
        {
            _logger.log(Logger.DEBUG, _tableLogMessage.toString());

//            System.out.println(_tableLogMessage.toString()); //for visual inspection
        }
    }

    /**
     * Set the first two lines of the table log: 1st line has locationId, time zone info(always Z time); 2nd line is the
     * columns header. Also automatically set the boolean {@link #_printTableLog} to true.
     */
    void setPrintTableLogHeader(final String header)
    {
        _printTableLog = true;

        _tableLogMessage.append(header);
        _tableLogMessage.append(OHDConstants.NEW_LINE);

        //construct the String as the table format shown in Fortran output file, same precision
        _tableLogMessage.append("DAY SNOWFALL[MM] RAIN_ON_SNOW[MM] ENERGY_EXCHANGE[MM] SIM.COVER PCT.LIQ.WATER HEAT_DEFICIT[MM] SNOW_TEMP.[C]"
            + " SIM.DEPTH[CM] OBS.DEPTH[CM]  SIM.WE[MM]  OBS.WE[MM]  OBS.COVER RAIN-SNOW-ELEVATION[M]");

        _tableLogMessage.append(OHDConstants.NEW_LINE);
    }
} // close class
