package ohd.hseb.ohdmodels.snow17;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;

import javax.management.timer.Timer;

import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.DoubleHolder;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.DataType;
import ohd.hseb.util.fews.ModelException;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.OHDUtilities;
import ohd.hseb.util.fews.ohdmodels.ModelDriver;

/**
 * This ModelDriver subclass is called by {@link FewsAdapter} through
 * {@link FewsAdapter#setUpAndExecuteModel(Properties)} to conduct Snow17 computation for the time period in the input
 * fews.xml. super._parameters and super._state in the super class ModelDriver are also loaded inside the method
 * {@link FewsAdapter#setUpAndExecuteModel(Properties)}. The Snow17ModelDriver method
 * {@link #execute(String, String, Properties, Map)} is the entry point from FewsAdapter. The results, four output time
 * series(raim, swe, sasc and snsg), are stored in the argument Map and returned. Finally, they are written to the xml
 * file {@link FewsAdapter#OUTPUT_FILE_NAME}.
 * <p>
 * The method {@link #ex19()} loops through the whole computation time period, with step of MAT input time series
 * interval. During each loop, {@link #ex19()} calls {@link Snow17Model}'s method
 * {@link Snow17Model#pack19(double, double[], double[], double, long, double, double)} once, which does the Snow17
 * computation.
 * <p>
 * All the date and time are in GMT timezone. For input data in non-GMT time zone, the time is converted to GMT time in
 * FewsAdapter.java. When needs to output the computation results, the time is switched back to the user's timezone.
 * 
 * @author FewsPilot Team
 */
final public class Snow17ModelDriver extends ModelDriver
{

    /** alias to super._parameters to save type casting */
    private Snow17ModelParameters _snow17ModelParameters;

    /** alias to super._state to save type casting */
    private Snow17ModelState _snow17ModelState;

    /**
     * 2 required input time series: MAP and MAT TS, with MAT TS being super._drivingTs. MAT TS unit can be oF or oC,
     * model runs in unit of oC; MAP TS unit can be any length unit, model runs in mm
     */

    private RegularTimeSeries _inputMapTs;

    /** --------------------------------optional input time series ------------------------- */

    //percent snowfall time series, interval must be same as _inputMapTs
    private RegularTimeSeries _inputPtpsTs;

    // rain snow elevation time series, interval must be same as _inputMatTs
    private RegularTimeSeries _inputRselTs = null;

    // input observed water-equivalent TS
    private RegularTimeSeries _inputSnweTs = null;

    //input observed areal extent of snow cover TS
    private RegularTimeSeries _inputAescTs = null;

    //input observed snow depth TS; unlike other two observed input TSs, this one is only used in logging table
    private RegularTimeSeries _inputSnogTs = null;

    private int _itpx; //input MAP ts interval
    private int _idt; //input MAT ts interval
    private int _ratioMatMap;
    private boolean _printLog = false;

    private volatile Snow17Technique _snow17Technique;

    //MODS optional input TSs:
    private RegularTimeSeries _inputAescchngModTs = null;
    private RegularTimeSeries _inputMfcModTs = null;
    private RegularTimeSeries _inputRainSnowModTs = null;
    private RegularTimeSeries _inputUadjModTs = null;
    private RegularTimeSeries _inputWeAddModTs = null;
    private RegularTimeSeries _inputWechngModTs = null;

    /**
     * Default constructor. From the view of Fews, it is implicitly called by {@link FewsAdapter} in the method
     * {@link FewsAdapter#setUpAndExecuteModel(Properties arguments)}:
     * <p>
     * Class modelClass = Class.forName(arguments.getProperty(_MODEL_NAME_TAG));<br>
     * Object obj = modelClass.newInstance(); //calls ModelDriver subclass(e.g. Snow17ModelDriver.java) constructor
     * super._parameters and super._state are created by default constructors. They are not loaded yet (from params xml
     * file and statesI file)!
     */
    public Snow17ModelDriver()
    {
        super(); // calling ModelDriver constructor

        super._parameters = new Snow17ModelParameters();
        super._savedParameters = new Snow17ModelParameters();
        super._state = new Snow17ModelState();

    } // close constructor

    /**
     * Implement the interface ModelDriver method. Here is the entry point from {@link FewsAdapter}'s method
     * setUpAndExecuteModel():<br>
     * _model.execute(_fewsDataWorkLocation,_fewsDataOutputLocation, _logger, arguments, resultMap);<br>
     * super._parameters and super._state in the super class ModelDriver have been already loaded inside the method
     * {@link FewsAdapter#setUpAndExecuteModel(Properties)}. It calls {@link #ex19()}, which does all the computation
     * through looping. At the end, outputs the states variables to ./testoutput/snow17/statesO.txt
     * 
     * @param workDirName the work dir name
     * @param outputDirName the output dir name
     * @param modelSpecificArguments the model specific arguments
     * @param resultMap the result map. It has four entries. Time series type("RAIM", "SWE", "SASC" and "SNSG") are the
     *            keys. The corresponding values are the output time series.
     * @throws Exception the exception
     */
    @Override
    public void execute() throws Exception
    {
        //use alias to save casting
        _snow17ModelParameters = (Snow17ModelParameters)super._parameters;
        _snow17ModelState = (Snow17ModelState)super._state;

        if(_logger.getPrintDebugInfo() > 0)
        {
            _printLog = true;
        }

        runModelDriverValidation();

        if(_printLog)
        {
            _logger.log(Logger.DEBUG, "DEBUG--CONTENTS OF PARAMETERS AND STATES IN THE BEGINNING:");
            _logger.log(Logger.DEBUG, _snow17ModelParameters.toString());
            _logger.log(Logger.DEBUG, _snow17ModelState.toString());
        }

        if(_snow17Technique.getBoolValue(Snow17ModelConstants.TECHNIQUE_SNOW))
        {
            // perform snow17 computation
            ex19();
        }
        else
        {//not run the model, raim equal to map, SASC all values are 0.0; No other secondary TSs; statesO.txt is zero

            _inputMapTs.trimTimeSeriesAtStart(getComputationStartTime());
            _inputMapTs.trimTimeSeriesAtEnd(getComputationEndTime());
            _inputMapTs.setTimeSeriesType(DataType.RAIM_DATATYPE);
            addTStoResultMap(_inputMapTs);

            final RegularTimeSeries sascTs = new RegularTimeSeries(getComputationStartTime(),
                                                                   getComputationEndTime(),
                                                                   getDrivingTsInterval(),
                                                                   MeasuringUnit.percentDecimal,
                                                                   0.0);
            sascTs.setTimeSeriesType(DataType.SASC_DATATYPE);
            sascTs.setLocationId(_outputLocationId);
            addTStoResultMap(sascTs);

            _snow17ModelState.zero19();

            _snow17ModelState.setDateTime(getComputationEndTime());

            _logger.log(Logger.DEBUG, "TECHNIQUE SNOW is set to be false. So the Snow17 model is not executed.");
        }

        _snow17ModelState.setITPX(_inputMapTs.getIntervalInHours()); //will be output to statesO.txt for COX

        _logger.log(Logger.DEBUG, "Snow17ModelDriver has finished");

        if(_printLog)
        {
            _logger.log(Logger.DEBUG, "DEBUG--CONTENTS OF PARAMETERS AND STATES IN THE END:");
            _logger.log(Logger.DEBUG, _snow17ModelParameters.toString());
            _logger.log(Logger.DEBUG, _snow17ModelState.toString());
        }

    }// close execute()

    /**
     * Ex19() corresponds to Fortran subroutine EX19(ex19.f). It loops through the whole calculation period, each step
     * equals to MAT time step. For each MAT(air temperature) time step, it calls
     * {@link Snow17Model#pack19(double, double[], double[], double, long, double, double)} once.
     */
    private void ex19() throws Exception
    {
        if(_printLog)
        {
            _logger.log(Logger.DEBUG,
                        "Running Snow17 program from GMT time: "
                            + DateTime.getDateTimeStringFromLong(getComputationStartTime(),OHDConstants.GMT_TIMEZONE) + " to "
                            + DateTime.getDateTimeStringFromLong(getComputationEndTime(),OHDConstants.GMT_TIMEZONE));
        }
        final double[] precipArrayInMatInterval = new double[_ratioMatMap];

        final double[] percentSnowArrayInMatInterval = new double[_ratioMatMap];

        final double[] lapseRateArray = new double[OHDConstants.HOURS_PER_DAY / _idt];
        // chop 24hr into several periods, with each period equal to MAT interval;

        //calculate lapse rates when TAELEV != ELEV
        if(_snow17ModelParameters.getElevationWithAirTemp() != _snow17ModelParameters.getElevation())
        {
            final double talMax = _snow17ModelParameters.getTalmax();
            final double talMin = _snow17ModelParameters.getTalmin();
            if(_printLog)
            {
                _logger.log(Logger.DEBUG, "TEMPERATURE LAPSE RATES:");
            }

            // divide a day(24 hrs) into several periods, each period is MAT interval; calculate lapseRate of each period in a day
            for(int i = 1; i <= OHDConstants.HOURS_PER_DAY / _idt; i++)
            {
                double tl = (i - 0.5) * _idt;

                if(getStartHourOfDay() != 0)
                { // so in forcasting mode: only in forcasting mode, tl needs to converted to local time zone
                    final TimeZone runInfoTZ = getRunInfo().getTimeZone();

                    if(_runInfo.getTimeZoneRawOffsetInHours() == 0 && (_printLog))
                    {//run_info.xml is GMT time zone, log it as warning

                        _logger.log(Logger.WARNING,
                                    "run_info.xml is in GMT time zone. Please confirm that this is correct. "
                                        + "The output results is affected by the local time zone when the two parameters ELEV and TAELEV are not equal.");
                    }

                    tl += OHDUtilities.getLocalTimeAt12Z(runInfoTZ);
                }

                if(tl > 24.0)
                {
                    tl = tl - 24.0;
                }

                if(tl >= 15.0)
                {
                    lapseRateArray[i - 1] = talMax - (tl - 15.0) / 15.0 * (talMax - talMin);

                }
                else if(tl > 6.0)
                {
                    lapseRateArray[i - 1] = talMin + (tl - 6.0) / 9.0 * (talMax - talMin);
                }
                else
                {
                    lapseRateArray[i - 1] = talMin + (6.0 - tl) / 15.0 * (talMax - talMin);
                }
                if(_printLog)
                {
                    _logger.log(Logger.DEBUG, String.valueOf(lapseRateArray[i - 1]));
                }

            } // end of for loop

        } // end of if (_snow17ModelParameters.getTaelev() != _snow17ModelParameters.getElev())

        /* ---------WEADD, WECHNG and AESCCHNG MODS in the initial state time -------- */
        double owe = OHDConstants.MISSING_DATA;
        double osc = OHDConstants.MISSING_DATA;

        // try to detect if any of WECHNG, AESCCHNG and WEADD MODS existing at the initial state time
        if(_inputWechngModTs != null)
        {
            owe = _inputWechngModTs.getMeasurementValueByTimeNoException(getInitialStateTime(), MeasuringUnit.mm);

            if(owe != OHDConstants.MISSING_DATA && _printLog)
            {
                _logger.log(Logger.DEBUG,
                            "WECHNG MODS applied at the initial state time "
                                + DateTime.getDateTimeStringFromLong(getInitialStateTime(), null));
            }
        }

        if(_inputAescchngModTs != null)
        {
            osc = _inputAescchngModTs.getMeasurementValueByTimeNoException(getInitialStateTime(),
                                                                           MeasuringUnit.percentDecimal);

            if(osc != OHDConstants.MISSING_DATA && _printLog)
            {
                _logger.log(Logger.DEBUG,
                            "AESCCHNG MODS applied at the initial state time "
                                + DateTime.getDateTimeStringFromLong(getInitialStateTime(), null));
            }
        }

        /*
         * WEADD MODS is ignored when the date equals to the initial state time(see mweadd.f), even though from ex19.f,
         * it looks like WEADD code is there and it is active even at this time
         */

        // 119
        final Snow17Model snow17Model = new Snow17Model(_snow17ModelParameters,
                                                        _snow17ModelState,
                                                        _inputUadjModTs,
                                                        _inputMfcModTs,
                                                        getStartHourOfDay(),
                                                        _logger);

        final double twe = _snow17ModelState.calculateTweFromState();

        final double twe1 = twe; //stores the original value for the end comparison

        if(owe >= 0.0 || osc >= 0.0)
        {
            double aesc = 0.0;

            if(twe != 0.0)
            {
                aesc = snow17Model.aesc19();
            }

            // CALL UPDT19:

            final DoubleHolder tweHolder = new DoubleHolder(twe);
            final DoubleHolder aescHolder = new DoubleHolder(aesc);

            snow17Model.updt19(owe, osc, tweHolder, aescHolder);
            //here, only for updating initial state(adjust carryover), the returned twe and aesc are not used

        }

        if(_snow17Technique.getBoolValue(Snow17ModelConstants.PRINTSNW)) //running in full version
        {
            final String headerLine = "'" + _drivingTs.getLocationId() + "' DAILY OUTPUT IS FOR HOUR "
                + getStartHourOfDay() + "    TIME ZONE=Z"; //DAILY OUTPUT is Z time, so hard-coded as Z 

            snow17Model.setPrintTableLogHeader(headerLine);
        }

        /* ---------------End of MODS at Initial State Time----------- */

        /** --------------create empty output RTS, with initial value of 0.0 -------------------- */
        /* ----------- primary output RTS(two, generated in slim version) -------------- */
        /*
         * getITSWE(), getITSSC() and getITSDPT() must be an even multiple of getIDT() (MAT TS interval); If params.xml
         * does not specify the output TS intervals, use MAT time interval(e.g. driving TS interval)
         */

        // rain + melt result time series, == RM[], unit: mm
        final RegularTimeSeries rmTs = new RegularTimeSeries(getComputationStartTime(),
                                                             getComputationEndTime(),
                                                             _itpx,
                                                             MeasuringUnit.mm,
                                                             0.0);
        rmTs.setTimeSeriesType(DataType.RAIM_DATATYPE);
        rmTs.setLocationId(_outputLocationId);
        addTStoResultMap(rmTs);

        //COVER[], or SASC, simulated area snow cover(percentage)
        final RegularTimeSeries sascTs = new RegularTimeSeries(getComputationStartTime(),
                                                               getComputationEndTime(),
                                                               getDrivingTsInterval(),
                                                               MeasuringUnit.percentDecimal,
                                                               0.0);
        sascTs.setTimeSeriesType(DataType.SASC_DATATYPE);
        sascTs.setLocationId(_outputLocationId);
        addTStoResultMap(sascTs);
	
	//SNow Water Equivalent (SWE) 
	
        final RegularTimeSeries sweTs = new RegularTimeSeries(getComputationStartTime(),
                                          getComputationEndTime(),
                                          getDrivingTsInterval(),
                                          MeasuringUnit.mm,
                                          0.0);
        sweTs.setLocationId(_outputLocationId);
        sweTs.setTimeSeriesType(DataType.SWE_DATATYPE);
        addTStoResultMap(sweTs);


        /* ----------- secondary output RTS ------------------------ */
        /*av RegularTimeSeries sweTs = null; //snow water equivalent */
        RegularTimeSeries snsgTs = null; //snow depth
        RegularTimeSeries prainTs = null; //precip as rain: unit of mm, output at _drivingTs interval
        RegularTimeSeries psfallTs = null; //precip as snow fall TS: unit of mm, output at _drivingTs interval
        RegularTimeSeries psnwroTs = null; //snow pack outflow within the period -- MAT interval: unit of mm, output at _drivingTs interval
        RegularTimeSeries probgTs = null; //Rain on bare ground within the period -- MAT interval: unit of mm, output at _drivingTs interval

        /* -------------------prepare for the output state time series ------------------------ */
        RegularTimeSeries accmaxStateTs = null;
        RegularTimeSeries aeadjStateTs = null;
        RegularTimeSeries liqwStateTs = null;
        RegularTimeSeries neghsStateTs = null;
        RegularTimeSeries sbStateTs = null;
        RegularTimeSeries sbaescStateTs = null;
        RegularTimeSeries sbwsStateTs = null;
        RegularTimeSeries sndptStateTs = null;
        RegularTimeSeries sntmpStateTs = null;
        RegularTimeSeries storgeStateTs = null;
        RegularTimeSeries taprevStateTs = null;
        RegularTimeSeries tindexStateTs = null;
        RegularTimeSeries weStateTs = null;
        RegularTimeSeries pqnetStateTs = null;

        if(_snow17Technique.getBoolValue(Snow17ModelConstants.SACSNOW_FULL_VERSION)) //running in full version
        {
            /*------------initiate secondary output TSs -----------------*/
            /*sweTs = new RegularTimeSeries(getComputationStartTime(),
                                          getComputationEndTime(),
                                          getDrivingTsInterval(),
                                          MeasuringUnit.mm,
                                          0.0);
            sweTs.setLocationId(_outputLocationId);
            sweTs.setTimeSeriesType(DataType.SWE_DATATYPE);
            addTStoResultMap(sweTs);
            */
            /*
             * SDPT[], or SNSG, simulated snow depth, unit: CM. Snow17ModelState SNDPT is also CM. When outputting to
             * outputs.xml, ohdfewsadapter switches _snsgTs from CM to unit of MM, because FEWS expects MM unit.
             */
            snsgTs = new RegularTimeSeries(getComputationStartTime(),
                                           getComputationEndTime(),
                                           getDrivingTsInterval(),
                                           MeasuringUnit.cm,
                                           0.0);
            snsgTs.setTimeSeriesType(DataType.SNSG_DATATYPE);
            snsgTs.setLocationId(_outputLocationId);
            addTStoResultMap(snsgTs);

            prainTs = new RegularTimeSeries(getComputationStartTime(),
                                            getComputationEndTime(),
                                            getDrivingTsInterval(),
                                            MeasuringUnit.mm,
                                            0.0);
            prainTs.setTimeSeriesType(DataType.PRAIN_DATATYPE);
            prainTs.setLocationId(_outputLocationId);
            addTStoResultMap(prainTs);

            psfallTs = new RegularTimeSeries(getComputationStartTime(),
                                             getComputationEndTime(),
                                             getDrivingTsInterval(),
                                             MeasuringUnit.mm,
                                             0.0);
            psfallTs.setTimeSeriesType(DataType.PSFALL_DATATYPE);
            psfallTs.setLocationId(_outputLocationId);
            addTStoResultMap(psfallTs);

            psnwroTs = new RegularTimeSeries(getComputationStartTime(),
                                             getComputationEndTime(),
                                             getDrivingTsInterval(),
                                             MeasuringUnit.mm,
                                             0.0);
            psnwroTs.setTimeSeriesType(DataType.PSNWRO_DATATYPE);
            psnwroTs.setLocationId(_outputLocationId);
            addTStoResultMap(psnwroTs);

            probgTs = new RegularTimeSeries(getComputationStartTime(),
                                            getComputationEndTime(),
                                            getDrivingTsInterval(),
                                            MeasuringUnit.mm,
                                            0.0);
            probgTs.setTimeSeriesType(DataType.PROBG_DATATYPE);
            probgTs.setLocationId(_outputLocationId);
            addTStoResultMap(probgTs);

            /*------------initiate output state TSs -----------------*/
            //1
            accmaxStateTs = new RegularTimeSeries(getComputationStartTime(),
                                                  getComputationEndTime(),
                                                  getDrivingTsInterval(),
                                                  MeasuringUnit.mm);
            accmaxStateTs.setTimeSeriesType(Snow17ModelState.ACCMAX_TAG);
            accmaxStateTs.setLocationId(_outputLocationId);
            addTStoResultMap(accmaxStateTs);
            //2
            aeadjStateTs = new RegularTimeSeries(getComputationStartTime(),
                                                 getComputationEndTime(),
                                                 getDrivingTsInterval(),
                                                 MeasuringUnit.mm);
            aeadjStateTs.setTimeSeriesType(Snow17ModelState.AEADJ_TAG);
            aeadjStateTs.setLocationId(_outputLocationId);
            addTStoResultMap(aeadjStateTs);
            //3
            liqwStateTs = new RegularTimeSeries(getComputationStartTime(),
                                                getComputationEndTime(),
                                                getDrivingTsInterval(),
                                                MeasuringUnit.mm);
            liqwStateTs.setTimeSeriesType(Snow17ModelState.LIQW_TAG);
            liqwStateTs.setLocationId(_outputLocationId);
            addTStoResultMap(liqwStateTs);
            //4
            neghsStateTs = new RegularTimeSeries(getComputationStartTime(),
                                                 getComputationEndTime(),
                                                 getDrivingTsInterval(),
                                                 MeasuringUnit.mm);
            neghsStateTs.setTimeSeriesType(Snow17ModelState.NEGHS_TAG);
            neghsStateTs.setLocationId(_outputLocationId);
            addTStoResultMap(neghsStateTs);
            //5
            sbStateTs = new RegularTimeSeries(getComputationStartTime(),
                                              getComputationEndTime(),
                                              getDrivingTsInterval(),
                                              MeasuringUnit.mm);
            sbStateTs.setTimeSeriesType(Snow17ModelState.SB_TAG);
            sbStateTs.setLocationId(_outputLocationId);
            addTStoResultMap(sbStateTs);
            //6
            sbaescStateTs = new RegularTimeSeries(getComputationStartTime(),
                                                  getComputationEndTime(),
                                                  getDrivingTsInterval(),
                                                  MeasuringUnit.percentDecimal);
            sbaescStateTs.setTimeSeriesType(Snow17ModelState.SBAESC_TAG);
            sbaescStateTs.setLocationId(_outputLocationId);
            addTStoResultMap(sbaescStateTs);
            //7
            sbwsStateTs = new RegularTimeSeries(getComputationStartTime(),
                                                getComputationEndTime(),
                                                getDrivingTsInterval(),
                                                MeasuringUnit.mm);
            sbwsStateTs.setTimeSeriesType(Snow17ModelState.SBWS_TAG);
            sbwsStateTs.setLocationId(_outputLocationId);
            addTStoResultMap(sbwsStateTs);
            //8
            sndptStateTs = new RegularTimeSeries(getComputationStartTime(),
                                                 getComputationEndTime(),
                                                 getDrivingTsInterval(),
                                                 MeasuringUnit.cm);
//      SNDPT is with unit CM, it will be converted to MM when writing the TS out to xml file
            sndptStateTs.setTimeSeriesType(Snow17ModelState.SNDPT_TAG);
            sndptStateTs.setLocationId(_outputLocationId);
            addTStoResultMap(sndptStateTs);
            //9
            sntmpStateTs = new RegularTimeSeries(getComputationStartTime(),
                                                 getComputationEndTime(),
                                                 getDrivingTsInterval(),
                                                 MeasuringUnit.degreesCelsius);
            sntmpStateTs.setTimeSeriesType(Snow17ModelState.SNTMP_TAG);
            sntmpStateTs.setLocationId(_outputLocationId);
            addTStoResultMap(sntmpStateTs);
            //10
            storgeStateTs = new RegularTimeSeries(getComputationStartTime(),
                                                  getComputationEndTime(),
                                                  getDrivingTsInterval(),
                                                  MeasuringUnit.mm);
            storgeStateTs.setTimeSeriesType(Snow17ModelState.STORGE_TAG);
            storgeStateTs.setLocationId(_outputLocationId);
            addTStoResultMap(storgeStateTs);
            //11
            taprevStateTs = new RegularTimeSeries(getComputationStartTime(),
                                                  getComputationEndTime(),
                                                  getDrivingTsInterval(),
                                                  MeasuringUnit.degreesCelsius);
            taprevStateTs.setTimeSeriesType(Snow17ModelState.TAPREV_TAG);
            taprevStateTs.setLocationId(_outputLocationId);
            addTStoResultMap(taprevStateTs);
            //12
            tindexStateTs = new RegularTimeSeries(getComputationStartTime(),
                                                  getComputationEndTime(),
                                                  getDrivingTsInterval(),
                                                  MeasuringUnit.degreesCelsius);
            tindexStateTs.setTimeSeriesType(Snow17ModelState.TINDEX_TAG);
            tindexStateTs.setLocationId(_outputLocationId);
            addTStoResultMap(tindexStateTs);
            //13
            weStateTs = new RegularTimeSeries(getComputationStartTime(),
                                              getComputationEndTime(),
                                              getDrivingTsInterval(),
                                              MeasuringUnit.mm);
            weStateTs.setTimeSeriesType(Snow17ModelState.WE_TAG);
            weStateTs.setLocationId(_outputLocationId);
            addTStoResultMap(weStateTs);
            //14
            pqnetStateTs = new RegularTimeSeries(getComputationStartTime(),
                                                 getComputationEndTime(),
                                                 getDrivingTsInterval(),
                                                 MeasuringUnit.mm);
            pqnetStateTs.setTimeSeriesType(Snow17ModelState.PQNET_TAG);
            pqnetStateTs.setLocationId(_outputLocationId);
            addTStoResultMap(pqnetStateTs);
        }

        //TAELEV - ELEV
        final double diffTaElevAndElev = _snow17ModelParameters.getElevationWithAirTemp()
            - _snow17ModelParameters.getElevation();

        final DateTime dateTime = new DateTime(getComputationStartTime(), getStartHourOfDay()); //forecasting mode(12Z) and mcp3 mode(0Z)

        double srm = 0.0; //sum up all the rain+melt during the whole period

        /** -------------------------the giant loop -------------------------- */

        // starting from the first timestep, each loop, forward one MAT interval. MAT is driving TS
        for(long time = getComputationStartTime(); time <= getComputationEndTime(); time += getDrivingTsIntervalInMillis())
        {
            dateTime.setTime(time); // redundant for the first run, but needed afterwards

            /** --fill precipArray and optional percentSnowArray. PTPS TS must have the same interval as MAP-- */
            /*
             * If MAP, MAT have same time interval, the following only loop once; if MAT interval is multiple of MAT,
             * thetimestep for MAP steps backward multiple times
             */
            for(int i = 0; i < _ratioMatMap; i++)
            {
                final long mapTimeStep = time - (_ratioMatMap - 1 - i) * _itpx * Timer.ONE_HOUR;

                // input MAP TS could be any unit, but model runs in unit of mm
                precipArrayInMatInterval[i] = _inputMapTs.getMeasurementValueByTime(mapTimeStep, MeasuringUnit.mm)
                    * _snow17ModelParameters.getPxadj();

                if(_snow17ModelParameters.usePercentSnowFallInput())
                {
                    percentSnowArrayInMatInterval[i] = _inputPtpsTs.getMeasurementValueByTime(mapTimeStep,
                                                                                              MeasuringUnit.percentDecimal)
                        * _snow17ModelParameters.getPxadj();
                }
                else
                {
                    percentSnowArrayInMatInterval[i] = OHDConstants.MISSING_DATA;
                }

            } // close for loop

            // need to be oC for model calculation, if input air temperature TS unit is oF, now will be converted
            double airTemperature = _drivingTs.getMeasurementValueByTime(time, MeasuringUnit.degreesCelsius);

            // adjust airTemperature using lapse rate, which is only needed when TAELEV != ELEV
            if(_snow17ModelParameters.getElevationWithAirTemp() != _snow17ModelParameters.getElevation())
            {
                /*
                 * get index of which period in a day. A day(24hrs) is divided into several periods, with each period
                 * equal to MAT interval
                 */
                final int i = dateTime.getNwsrfsHour() / _idt;

                airTemperature += diffTaElevAndElev * 0.01 * lapseRateArray[i - 1];
            }

            // the original programmer uses -99.0 as missing value; the following is testing if TAPREV is missing
            if(_snow17ModelState.getTaprev() < -98.0)
            {
                _snow17ModelState.setTaprev(airTemperature);
            }

            double rainSnowLatitude; // model runs in unit(meters). Input data can be any unit, here convert to m

            // only use RSEL input TS values when specified; otherwise, even though _inputRselTs is available,
            // if no specification, just set as MISSING_DATA
            if(_snow17ModelParameters.useRainSnowElevationInput())
            {
                // rain-snow elevation TS interval must be equal to MAT interval
                rainSnowLatitude = _inputRselTs.getMeasurementValueByTime(time, MeasuringUnit.meters);
            }
            else
            {
                rainSnowLatitude = OHDConstants.MISSING_DATA;
            }

            /*
             * First, get observed WE or AESC from observed input TS if available, then next, if there is WEADD, WECHNG
             * or AESCCHNG MODS at this time step too, owe or osc are changed according to the MODS
             */
            owe = OHDConstants.MISSING_DATA;
            if(_inputSnweTs != null)
            {
                owe = _inputSnweTs.getMeasurementValueByTimeNoException(time, MeasuringUnit.mm); //could be -999.0
                // do nothing, when no measurement at this time step
            }

            osc = OHDConstants.MISSING_DATA;
            if(_inputAescTs != null)
            {
                osc = _inputAescTs.getMeasurementValueByTimeNoException(time, MeasuringUnit.percentDecimal); //could be -999.0            
                // do nothing, when no measurement at this time step
            }

            double odpt = OHDConstants.MISSING_DATA;
            if(_inputSnogTs != null)
            {
                odpt = _inputSnogTs.getMeasurementValueByTimeNoException(time, MeasuringUnit.cm); //could be -999.0            
                // do nothing, when no measurement at this time step
            }

            // try to detect if any of WECHNG, AESCCHNG and WEADD MODS occurs at this time step
            if(_inputWechngModTs != null)
            {
                final double wechng = _inputWechngModTs.getMeasurementValueByTimeNoException(time, MeasuringUnit.mm); //could be -999.0
                if(wechng != OHDConstants.MISSING_DATA)
                {
                    owe = wechng;
                    if(_printLog)
                    {
                        _logger.log(Logger.DEBUG,
                                    "WECHNG MODS applied at " + DateTime.getDateTimeStringFromLong(time, null));
                    }
                }
            }

            if(_inputAescchngModTs != null)
            {
                final double aescchng = _inputAescchngModTs.getMeasurementValueByTimeNoException(time,
                                                                                                 MeasuringUnit.percentDecimal); //could be -999.0
                if(aescchng != OHDConstants.MISSING_DATA)
                {
                    osc = aescchng;
                    if(_printLog)
                    {
                        _logger.log(Logger.DEBUG,
                                    "AESCCHNG MODS applied at " + DateTime.getDateTimeStringFromLong(time, null));
                    }
                }
            }

            if(_inputWeAddModTs != null)
            {
                final double weAdd = _inputWeAddModTs.getMeasurementValueByTimeNoException(time, MeasuringUnit.mm); //could be -999.0
                if(weAdd != OHDConstants.MISSING_DATA)
                {
                    if(_printLog)
                    {
                        _logger.log(Logger.DEBUG,
                                    "WEADD MODS applied at " + DateTime.getDateTimeStringFromLong(time, null));
                    }

                    if(owe == OHDConstants.MISSING_DATA)
                    {// no WECHNG MODS, but has WEADD MODS

                        owe = _snow17ModelState.calculateTweFromState() + weAdd;
                        //can NOT use the local variable twe
                    }
                    else
                    {// has WECHNG MODS & WEADD MODS

                        owe += weAdd;
                    }

                } //close if(weAdd != OHDModelConstants.MISSING_DATA)
            }

            RAINSNOW_MOD_VALUE rainSnowMod = RAINSNOW_MOD_VALUE.NO_MOD;

            if(_inputRainSnowModTs != null)
            {// there is RAINSNOW mods, no LSTCMPDY issue

                final double rainSnow = _inputRainSnowModTs.getMeasurementValueByTimeNoException(time,
                                                                                                 MeasuringUnit.unitlessInt);

                rainSnowMod = RAINSNOW_MOD_VALUE.getModFromValue(rainSnow);
            }

            //an optional property in run info: default is false(use pxtemp to determine precip is 100% rain or 100% snow)
            if(_snow17Technique.getBoolValue(Snow17ModelConstants.PRECIP_IS_ALL_RAIN))
            {//forcing all precip to be 100% rain
                rainSnowMod = RAINSNOW_MOD_VALUE.RAIN;

                if(_inputRainSnowModTs != null)
                {// there is RAINSNOW mods, conflict with the presence of the property "PRECIP_IS_ALL_RAIN"

                    _logger.log(Logger.WARNING,
                                "Both the property PRECIP_IS_ALL_RAIN(value=true) and RAINSNOW Mod exist and they may cause conflict. "
                                    + "Set all precip to be 100% rain according to the property PRECIP_IS_ALL_RAIN");

                }
                else
                {
                    _logger.log(Logger.DEBUG, "The property " + Snow17ModelConstants.PRECIP_IS_ALL_RAIN
                        + " exists and its value is true. Set all precip to be 100% rain.");
                }

            }

            // CALL PACK19(KDA,KHR,NDT,PTA,PPX,PPCTS,PRSL,POWE,POSC, PODPT,PGM,PRM,TWE,PCOVER,CWE,CAESC,IFUT,IDT,IBUG,
            // IDN,IMN,IYR,IOUTYP,OPNAME,IVER)
            final Snow17Results timeStepSnow17Results = snow17Model.pack19(airTemperature,
                                                                           precipArrayInMatInterval,
                                                                           percentSnowArrayInMatInterval,
                                                                           rainSnowLatitude,
                                                                           time,
                                                                           owe,
                                                                           osc,
                                                                           odpt,
                                                                           rainSnowMod);

            // save raim result, ts interval equal to MAP interval
            final double[] raimArrayInInterval = timeStepSnow17Results.getRaim();

            for(int i = 0; i < _ratioMatMap; i++)
            {
                final long mapTimeStep = time - (_ratioMatMap - 1 - i) * _itpx * Timer.ONE_HOUR;

                rmTs.setMeasurementByTime(raimArrayInInterval[i], mapTimeStep);

                srm += raimArrayInInterval[i];
            }

            // save sasc result at MAT interval for now
            sascTs.setMeasurementByTime(timeStepSnow17Results.getSasc(), time);
	    /*av bug 535  */
            sweTs.setMeasurementByTime(timeStepSnow17Results.getSwe(), time);
	    
            //at the end of day, log the daily values in the past 24 hours; only useful if in forcasting mode(not mcp3 mode)
            if(getStartHourOfDay() != 0 && dateTime.getNwsrfsHour() == 24)
            {
                final DateTime nextDay = new DateTime(dateTime.getTimeInMillis() + Timer.ONE_HOUR);
                snow17Model.logDailyValues(nextDay.dayOfMonth());
                /*
                 * get the next day after dayOfMonth because the values are from the period of 24 hours prior to the
                 * current time.
                 */
            }

            if(_snow17Technique.getBoolValue(Snow17ModelConstants.SACSNOW_FULL_VERSION)) //running in full version
            {
                /*---------------- storing secondary TSs --------------------*/
                // save swe result at MAT interval for now
                /*av sweTs.setMeasurementByTime(timeStepSnow17Results.getSwe(), time); */

                // save snsg result at MAT interval for now
                snsgTs.setMeasurementByTime(timeStepSnow17Results.getSnsg(), time);

                //at MAT interval
                prainTs.setMeasurementByTime(timeStepSnow17Results.getPrain(), time);

                //at MAT interval
                psfallTs.setMeasurementByTime(timeStepSnow17Results.getPsfall(), time);

                probgTs.setMeasurementByTime(timeStepSnow17Results.getProbg(), time);

                psnwroTs.setMeasurementByTime(timeStepSnow17Results.getPsnwro(), time);

                //storing state at every time step
                accmaxStateTs.setMeasurementByTime(_snow17ModelState.getMaxWeSinceSnow(), time); //1
                aeadjStateTs.setMeasurementByTime(_snow17ModelState.getSnowArealExtentAdjustment(), time); //2
                liqwStateTs.setMeasurementByTime(_snow17ModelState.getLiquidWaterInSnowPack(), time); //3
                neghsStateTs.setMeasurementByTime(_snow17ModelState.getNegHeatStorage(), time); //4
                sbStateTs.setMeasurementByTime(_snow17ModelState.getArealWaterEquivBeforeSnow(), time); //5
                sbaescStateTs.setMeasurementByTime(_snow17ModelState.getAescBeforeSnow(), time); //6
                sbwsStateTs.setMeasurementByTime(_snow17ModelState.getWaterEquivalentUnderSnowCover(), time); //7
                sndptStateTs.setMeasurementByTime(_snow17ModelState.getSnowDepth(), time); //8
                sntmpStateTs.setMeasurementByTime(_snow17ModelState.getAverageSnowTemp(), time); //9
                storgeStateTs.setMeasurementByTime(_snow17ModelState.getExcessLiquidWaterInStorage(), time); //10
                taprevStateTs.setMeasurementByTime(_snow17ModelState.getTaprev(), time); //11
                tindexStateTs.setMeasurementByTime(_snow17ModelState.getTempIndex(), time); //12
                weStateTs.setMeasurementByTime(_snow17ModelState.getWaterEquivalentOfSolidSnow(), time); //13
                pqnetStateTs.setMeasurementByTime(_snow17ModelState.getPqnet(), time); //14
            }

        } // end of giant for loop

        //SBAL = SPX - SRM - (TWE-TWE1) + CHGWE
        final double sbal = Snow17Model._spx - srm - (_snow17ModelState.calculateTweFromState() - twe1)
            + Snow17Model._chgwe;

        if(Math.abs(sbal) > 1.0 && (_printLog))
        {
            _logger.log(Logger.WARNING, "Snow balance residual exceeds 1 MM. SBAL=" + sbal + " SPX=" + Snow17Model._spx
                + " SRM=" + srm + " TWE=" + _snow17ModelState.calculateTweFromState() + " TWE1=" + twe1 + " CHGWE="
                + Snow17Model._chgwe);
        }

        /*
         * getITSWE(), getITSSC() and getITSDPT() must be an even multiple of getIDT() (MAT TS interval); If params.xml
         * does not specify the output TS intervals, use MAT time interval(e.g. driving TS interval)
         */
        /*
         * change output TS(_sweTs, _coverTs and _sdptTs) interval if needed(they are at MAT interval, default value,
         * now). User can specify them to bigger interval. _rmTs interval is fixed to MAP interval
         */
        if(sascTs.getIntervalInHours() != _snow17ModelParameters.getSascInterval())
        {
            sascTs.integrateToIntervalInst(_snow17ModelParameters.getSascInterval());
        }
        /*av 535 fb */
	if(sweTs.getIntervalInHours() != _snow17ModelParameters.getSweInterval())
        {
            sweTs.integrateToIntervalInst(_snow17ModelParameters.getSweInterval());
        }
        if(_snow17Technique.getBoolValue(Snow17ModelConstants.SACSNOW_FULL_VERSION))
        {
            /*av if(sweTs.getIntervalInHours() != _snow17ModelParameters.getSweInterval())
            {
                sweTs.integrateToIntervalInst(_snow17ModelParameters.getSweInterval());
            }
	    */

            if(snsgTs.getIntervalInHours() != _snow17ModelParameters.getSnsgInterval())
            {
                snsgTs.integrateToIntervalInst(_snow17ModelParameters.getSnsgInterval());
            }
        }

        //if in forecast mode(not in MCP3 mode), print out the table like logging message and only when PRINTSNW is true
        if(getStartHourOfDay() != 0)
        {
            snow17Model.logTableMessage();
        }

        //198
        return;
    }// end of Ex19() method

    /**
     * Actually does more than validation: assign individual input TS to a reference with more meaningful name; Check if
     * the needed TS(s) are present; validate snow17Parameters and snow17ModelState.
     */
    @Override
    final protected void runModelDriverValidation() throws Exception
    {
        //create the immutable Snow17Technique object
        _snow17Technique = new Snow17Technique(_runInfo.getProperties(), _logger);

        if(_logger.getPrintDebugInfo() > 0)
        {
            _printLog = true;
        }

        /** ---- assign individual RTS from getTsList() to its reference ---------- */

        final Iterator<RegularTimeSeries> ite = getTsList().iterator();

        while(ite.hasNext())
        {
            final RegularTimeSeries ts = ite.next();
            final String inputTsType = ts.getTimeSeriesType();

            if(inputTsType.equalsIgnoreCase(DataType.MAP_DATATYPE)) // required input
            {
                _inputMapTs = ts;
                _snow17ModelParameters.setPrecipInterval(_inputMapTs.getIntervalInHours());
            }
            else if(inputTsType.equalsIgnoreCase(DataType.MAT_DATATYPE)) // required input
            {
                _drivingTs = ts; // unit allowed to be oC or oF, will be converted to oC in
                _snow17ModelParameters.setAirTempInterval(_drivingTs.getIntervalInHours());
            }
            else if(inputTsType.equalsIgnoreCase(DataType.PTPS_DATATYPE)) // optional input
            {
                if(_snow17ModelParameters.usePercentSnowFallInput() == true)
                {
                    _inputPtpsTs = ts;
                    _snow17ModelParameters.setInputPercentSnowFallInterval(_inputPtpsTs.getIntervalInHours());
                }
                else
                {// since not using it, delete it from _tsList
                    ite.remove();
                }
            }
            else if(inputTsType.equalsIgnoreCase(DataType.RSEL_DATATYPE)) // optional input
            {
                if(_snow17ModelParameters.useRainSnowElevationInput() == true)
                {
                    _inputRselTs = ts;
                    _snow17ModelParameters.setInputRainSnowInterval(_inputRselTs.getIntervalInHours());
                }
                else
                {// since not using it, delete it from _tsList
                    ite.remove();
                }
            }
            else if(inputTsType.equalsIgnoreCase(DataType.SNWE_DATATYPE)) // optional input
            {
                _inputSnweTs = ts;
            }
            else if(inputTsType.equalsIgnoreCase(DataType.AESC_DATATYPE)) // optional input
            {
                _inputAescTs = ts;
            }
            else if(inputTsType.equalsIgnoreCase(DataType.SNOG_DATATYPE)) // optional input
            {
                _inputSnogTs = ts;
            }
            /*
             * FEWS will always provide input MODS time series xml file regardless of having MODS or not: when not using
             * a MOD, that MOD time series still have a header in the xml file. It is parsed and stored in getTsList()
             * with all values be -999.0. To let model computation not read these empty time series, they are removed
             * now.
             */
            else if(ts.isEmpty())//if passed all "if" checks above, must be MOD ts
            {
                if(_printLog)
                {
                    _logger.log(Logger.DEBUG, "Input " + inputTsType
                        + " MOD time series is empty. Removed it now so not using it anymore.");
                }

                ite.remove(); //delete the RTS from the list
            }
            else if(inputTsType.equalsIgnoreCase(SNOW17_MODS.AESCCHNG.getDataType())) // optional input
            {
                _inputAescchngModTs = ts.clone();

                ite.remove();

                if(_printLog)
                {
                    _logger.log(Logger.DEBUG, "Input " + inputTsType + " MOD time series is present.");
                }

            }
            else if(inputTsType.equalsIgnoreCase(SNOW17_MODS.MFC.getDataType())) // optional input
            {
                _inputMfcModTs = ts.clone();

                ite.remove();

                if(_printLog)
                {
                    _logger.log(Logger.DEBUG, "Input " + inputTsType + " MOD time series is present.");
                }
            }
            else if(inputTsType.equalsIgnoreCase(SNOW17_MODS.RAINSNOW.getDataType())) // optional input
            {
                _inputRainSnowModTs = ts.clone();

                ite.remove();

                if(_printLog)
                {
                    _logger.log(Logger.DEBUG, "Input " + inputTsType + " MOD time series is present.");
                }
            }
            else if(inputTsType.equalsIgnoreCase(SNOW17_MODS.UADJ.getDataType())) // optional input
            {
                _inputUadjModTs = ts.clone();

                ite.remove();

                if(_printLog)
                {
                    _logger.log(Logger.DEBUG, "Input " + inputTsType + " MOD time series is present.");
                }
            }
            else if(inputTsType.equalsIgnoreCase(SNOW17_MODS.WEADD.getDataType())) // optional input
            {
                _inputWeAddModTs = ts.clone();

                ite.remove();

                if(_printLog)
                {
                    _logger.log(Logger.DEBUG, "Input " + inputTsType + " MOD time series is present.");
                }
            }
            else if(inputTsType.equalsIgnoreCase(SNOW17_MODS.WECHNG.getDataType())) // optional input
            {
                _inputWechngModTs = ts.clone();

                ite.remove();

                if(_printLog)
                {
                    _logger.log(Logger.DEBUG, "Input " + inputTsType + " MOD time series is present.");
                }
            }

        } // close while loop

        if(_inputWechngModTs != null && _snow17Technique.getBoolValue(Snow17ModelConstants.TECHNIQUE_UPWE) == false)
        {
            //TECHNIQUE UPWE is set to false so canceling out WECHNG MOD: _inputWechngModTs remains null
            _logger.log(Logger.DEBUG, "TECHNIQUE UPWE is false. So WECHNG Mod is ignored.");

            _inputWechngModTs = null;
        }

        if(_inputAescchngModTs != null && _snow17Technique.getBoolValue(Snow17ModelConstants.TECHNIQUE_UPSC) == false)
        {
            //TECHNIQUE UPSC is set to false so canceling out AESCHNG MOD: _inputAescchngModTs remains null
            _logger.log(Logger.DEBUG, "TECHNIQUE UPSC is false. So AESCHNG Mod is ignored.");

            _inputAescchngModTs = null;
        }

        final long lstcmpdy = getSwitchFromObservedToForecastTime();

        final List<RegularTimeSeries> modsTsList = new ArrayList<RegularTimeSeries>(6); //max 6 mod TSs

        /*--------------------validate individual mod TS against lstcmpdy(MOD AESCCHNG, WEADD and WECHNG) and other checks ----------*/
        if(_inputAescchngModTs != null)
        {
            if(_inputAescchngModTs.getMeasuringUnit() != MeasuringUnit.percentDecimal)
            {
                throw new Exception("Error: input AESCCHNG MOD time series unit is not "
                    + MeasuringUnit.percentDecimal.getName());
            }

            if(lstcmpdy < _inputAescchngModTs.getStartTime())
            {
                _inputAescchngModTs = null;
            }
            else
            {
                _inputAescchngModTs.trimTimeSeriesAtEnd(lstcmpdy);

                modsTsList.add(_inputAescchngModTs);
            }

        }

        if(_inputMfcModTs != null)
        {
            if(_inputMfcModTs.getMeasuringUnit() != MeasuringUnit.unitlessReal)
            {
                throw new Exception("Error: input MFC MOD time series unit is not "
                    + MeasuringUnit.unitlessReal.getName());
            }

            modsTsList.add(_inputMfcModTs);
        }

        if(_inputRainSnowModTs != null)
        {
            if(_inputRainSnowModTs.getMeasuringUnit() != MeasuringUnit.unitlessInt)
            {
                throw new Exception("Error: input RAINSNOW MOD time series unit is not "
                    + MeasuringUnit.unitlessInt.getName());
            }

            modsTsList.add(_inputRainSnowModTs);
        }

        if(_inputUadjModTs != null)
        {
            if(_inputUadjModTs.getMeasuringUnit() != MeasuringUnit.unitlessReal)
            {
                throw new Exception("Error: input UADJ MOD time series unit is not "
                    + MeasuringUnit.unitlessReal.getName());
            }

            for(int i = 0; i < _inputUadjModTs.getMeasurementCount(); i++)
            {
                final double uadj = _inputUadjModTs.getMeasurementByIndex(i).getValue();

                if(uadj < 0.0 || uadj > 10.0)
                {
                    //is allowed to have missing value
                    if(uadj != OHDConstants.MISSING_DATA)
                    {
                        throw new ModelException("Error: Input UADJ MODS value must be >= 0.0 and =< 10.0 or being missing value -999.0.");
                    }
                }
            } //close for loop

            modsTsList.add(_inputUadjModTs);
        }

        if(_inputWeAddModTs != null)
        {
            if(lstcmpdy < _inputWeAddModTs.getStartTime())
            {
                _inputWeAddModTs = null;
            }
            else
            {
                _inputWeAddModTs.trimTimeSeriesAtEnd(lstcmpdy);

                modsTsList.add(_inputWeAddModTs);
            }

        }

        if(_inputWechngModTs != null)
        {
            if(lstcmpdy < _inputWechngModTs.getStartTime())
            {
                _inputWechngModTs = null;
            }
            else
            {
                _inputWechngModTs.trimTimeSeriesAtEnd(lstcmpdy);

                modsTsList.add(_inputWechngModTs);
            }

        }

        super.checkInputTsInSync(modsTsList);

        /** ----------- check two mandatory input TS are in the input xml file ------- */
        if(_drivingTs == null)
        {
            throw new ModelException("No input air temperature time series(MAT) in the input xml file!");
        }

        if(_inputMapTs == null)
        {
            throw new ModelException("No input precip time series(MAP) in the input xml file!");
        }

        _itpx = _snow17ModelParameters.getPrecipInterval();
        _idt = _snow17ModelParameters.getAirTempInterval();
        _ratioMatMap = _idt / _itpx;

        /** -------------------- check input MAP TS and trim it if needed------------------- */
        final int extraMapTimeSteps = (int)((getComputationStartTime() - _inputMapTs.getStartTime()) / _inputMapTs.getIntervalInMillis());

        if(extraMapTimeSteps < _ratioMatMap - 1)
        { // >= is ok(if ">", _inputMapTs starts too early, too much data; if "==", exactly amount of data)

            final String message = "Input MAP time series starting time is too late: "
                + DateTime.getDateTimeStringFromLong(_inputMapTs.getStartTime(),OHDConstants.GMT_TIMEZONE);

            throw new ModelException(message);
        }
        else if(extraMapTimeSteps > _ratioMatMap - 1)
        { // MAP starts too early, trim MAP TS to the exact expected start time

            // output raim TS start time, end time and interval are equal to MAP TS'
            final long expectedMapStartTime = getComputationStartTime() - (_ratioMatMap - 1)
                * _inputMapTs.getIntervalInMillis();
            if(_printLog)
            {
                _logger.log(Logger.DEBUG,
                            "Input MAP time series is truncated from "
                                + DateTime.getDateTimeStringFromLong(_inputMapTs.getStartTime(),OHDConstants.GMT_TIMEZONE) + " to "
                                + DateTime.getDateTimeStringFromLong(expectedMapStartTime,OHDConstants.GMT_TIMEZONE));
            }
            _inputMapTs.trimTimeSeriesAtStartWithCheck(expectedMapStartTime);
        }
        // now _inputMapTs has exactly the correct start time

        // _inputRainSnowTs could have different start time, end time and interval as _inputMapTs

        /** -- if optional input TS specified in params.xml, check to see the corresponding TS available in fews.xml - */

        if(_snow17ModelParameters.useRainSnowElevationInput() && _inputRselTs == null)
        {
            final String message = "Parameter file specifies to use rain-snow elevation TS, but it is not available.";

            throw new ModelException(message);
        }

        if(_snow17ModelParameters.usePercentSnowFallInput() && _inputPtpsTs == null)
        {
            final String message = "Parameter file specifies to use percent snowfall TS, but it is not available.";

            throw new ModelException(message);
        }

        /*
         * if OWE, OSC and ODPT intervals > MAT interval, disintegrate the input TS to MAT(driving TS) interval, for the
         * time steps added, -999.0 is used
         */
        if(_inputSnweTs != null)
        {//there is input OWE TS

            if(_inputSnweTs.getIntervalInHours() != getDrivingTsInterval())
            {
                _inputSnweTs.disIntegrateToIntervalWithMissing(getDrivingTsInterval());
                //if OWE interval is not multiple of MAT, an exception will be thrown
            }
            if(_printLog)
            {
                _logger.log(Logger.DEBUG,
                            "Update with observed water-equivalent time series " + _inputSnweTs.getTimeSeriesType());
            }
        } //close if(_inputOscTs != null)

        if(_inputAescTs != null)
        {//there is input OSC TS

            if(_inputAescTs.getIntervalInHours() != getDrivingTsInterval())
            {
                _inputAescTs.disIntegrateToIntervalWithMissing(getDrivingTsInterval());
                //if OSC interval is not multiple of MAT, an exception will be thrown
            }
            if(_printLog)
            {
                _logger.log(Logger.DEBUG,
                            "Update with observed areal snow cover time series " + _inputAescTs.getTimeSeriesType());
            }
        } //close if(_inputOscTs != null)

        if(_inputSnogTs != null)
        {//there is input ODPT TS

            if(_inputSnogTs.getIntervalInHours() != getDrivingTsInterval())
            {
                _inputSnogTs.disIntegrateToIntervalWithMissing(getDrivingTsInterval());
                //if OSC interval is not multiple of MAT, an exception will be thrown
            }
            if(_printLog)
            {
                _logger.log(Logger.DEBUG, "Use observed snow depth time series " + _inputSnogTs.getTimeSeriesType());
            }
        } //close if(_inputOdptTs != null)

        // validate Parameter and State
        _snow17ModelParameters.validateParams();

        if(super.needCarryoverTransfer() || _snow17ModelState.getITPX() != _snow17ModelParameters.getPrecipInterval())
        {//do carryover transfer when parameters have been modified or previous ITPX != current one 

            _snow17ModelState.validateState(super._savedParameters);
            _snow17ModelState.doCarryOverTransfer(super._savedParameters, _snow17ModelParameters);
        }
        else
        {//no need to do carryover transfer

            if(_snow17ModelState.getStateMap().size() != 0)
            {
                _snow17ModelState.validateState(_snow17ModelParameters);
            }
        }

        super.runModelDriverValidation();

    } // close runDriverValidation()
} // close class
