package ohd.hseb.ohdmodels.snow17;

import java.util.Arrays;

import ohd.hseb.util.fews.ohdmodels.ModelException;
import ohd.hseb.util.fews.ohdmodels.ModelParameters;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.OHDUtilities;
import ohd.hseb.util.fews.ParameterType.Row;
import ohd.hseb.util.fews.ParameterType.Table;
import ohd.hseb.util.fews.Parameters;

/**
 * This class holds all the Snow17 Model parameters. It is a subclass of {@link ModelParameters} and interface
 * {@link IModelParameters}. They are specified in the opt file:
 * <p>
 * <ul>
 * <li>Card #1,
 * <li>Card #2,
 * <li>Card #3,
 * <li>Card #3A(optional),
 * <li>Card #3B(optional),
 * <li>Card #4(optional),
 * <li>Card #5(optional),
 * <li>Card #6,
 * <li>Card #6A(optional),
 * <li>Card #7,
 * <li>Card #8
 * </ul>
 * <p>
 * Values are initially loaded into super._parasmMap, then extracted into instance variables to avoid frequent accessing
 * the Map. Most setter methods are not needed.
 * <p>
 * All the stringValue in the parameter xml file are case-insensitive.
 * 
 * @author FewsPilot Team
 */
final public class Snow17ModelParameters extends ModelParameters
{

    /** -----------------------------Tags------------------------- */
    // INPUT CARD NUMBER 1
    final public static String ELEV_TAG = "ELEV";
    private double _elev;

    final public static String ALAT_TAG = "ALAT";
    private double _alat;

    // INPUT CARD NUMBER 2
    private int _inputTSMapInterval; //this will NOT be in params xml, will be set in Snow17ModelDriver 
    final public static String PXADJ_TAG = "PXADJ";
    private double _pxadj;
    private int _inputPercentSnowFallInterval; //this will be used to confirm to be equal to precip TS interval
    /*
     * optional parameter, default value is false. If the parameter is set to true, use the input percent-snow-fall TS,
     * which is required to be present then; if absent or set to false, ignore the input percent-snow-fall TS, even
     * though it is present.
     */
    final public static String PERCENT_SNOWFALL_INPUT_TAG = "PERCENT_SNOWFALL_INPUT_OPTION";
    private boolean _usePercentSnowFall = false; //default value

    // INPUT CARD NUMBER 3
    private int _inputTSMatInterval; //NOT in params xml, will be set in Snow17ModelDriver based on input MAT TS
    final public static String TAELEV_TAG = "TAELEV";
    /*
     * originally, in the opt file, TAELEV could be absent(default is ELEV). However, Snow17OptFileConverter always
     * generate TAELEV element in params.xml
     */
    private double _taelev;

    public final static String TALMAX_TAG = "TALMAX"; // optional _talmax: only needed when _taelev != _elev
    private double _talmax;
    public final static String TALMIN_TAG = "TALMIN"; // optional _talmin: only needed when _taelev != _elev
    private double _talmin;

    // INPUT CARD NUMBER 3A (Optional: only included if _aec == "AVSE"); no mandatory tags in card3A
    private int _inputRainSnowInterval;

    /*
     * optional parameter, default value is false. If the parameter is set to true, use the input rain-snow-elevation
     * TS, which is required to be present then; if absent or set to false, ignore the input rain-snow-elevation TS,
     * even though it is present.
     */
    public final static String RAIN_SNOW_ELEV_INPUT_TAG = "RAIN_SNOW_ELEV_INPUT_OPTION";
    private boolean _useRainSnowElevInput = false; //default value

    // INPUT CARD NUMBER 3B (Optional: only included if _aec == "AVSE". At minimal, (emin, 0.0), (emax, 1.0) )
    public final static String AE_PAIRS_METR_TAG = "AREA_ELEV_CURVE(METR)"; // _ae_pairs
    public final static String AE_PAIRS_ENGL_TAG = "AREA_ELEV_CURVE(ENGL)"; // _ae_pairs
    private double[][] _ae_pairs = null; //only available when using RAIN-SNOW-ELEVATION; always in METRIC unit

    // INPUT CARD NUMBER 4 (Optional). Observation input TSs(optional). See comment above in card #1.

    // INPUT CARD NUMBER 5 (originally optional in opt files; but they are definitely present due to Snow17OptFileConverter)
    public final static String ITSWE_TAG = "SWE_OUTPUT_TS_INTERVAL";
    private int _itswe;
    public final static String ITSASC_TAG = "SASC_OUTPUT_TS_INTERVAL";
    private int _itsasc;
    public final static String ITSH_TAG = "SNSG_OUTPUT_TS_INTERVAL";
    private int _itsh;
//    private final static String[] _CARD5_TAGS = {ITSWE_TAG, ITSASC_TAG, ITSH_TAG};

    // INPUT CARD NUMBER 6
    public final static String SCF_TAG = "SCF";
    private double _scf;
    public final static String MFMAX_TAG = "MFMAX";
    private double _mfmax;
    public final static String MFMIN_TAG = "MFMIN";
    private double _mfmin;
    public final static String UADJ_TAG = "UADJ";
    private double _uadj;
    public final static String SI_TAG = "SI";
    private double _si;
    public final static String MV_TAG = "MV";
    private int _mv = 0; //default

    // INPUT CARD NUMBER 6A (Optional: only needed if _mv == 1)
    public final static String SMFV_TAG = "SMFV";
    private double[] _smfv;

    // INPUT CARD NUMBER 7
    public final static String NMF_TAG = "NMF";
    private double _nmf;
    public final static String TIPM_TAG = "TIPM";
    private double _tipm;
    public final static String MBASE_TAG = "MBASE";
    private double _mbase;
    public final static String PXTEMP_TAG = "PXTEMP";
    private double _pxtemp;
    public final static String PLWHC_TAG = "PLWHC";
    private double _plwhc;
    public final static String DAYGM_TAG = "DAYGM";
    private double _daygm;

    // In forecasting mode, the following two parameters are definitely needed; In MCP3 mode, they are only needed when card#1 UPS set 'UPDATE'
    public final static String WETOL_TAG = "WETOL[0., 1.0]";
    private double _wetol;

    public final static String SCTOL_TAG = "SCTOL[0., 1.0]";
    private double _sctol;

    private double[] _adc;

    //END OF ALL CARD TAGS

    /**
     * Instantiates a new snow17 model parameters. Values are set by parsing params.xml file by FewsXMLParser.java(in
     * package ohdfewsadapter) method public void parseParameters(final String paramFileName).
     */
    public Snow17ModelParameters()
    {
        super();
    }

    /**
     * Extract values from super._parametersMap into instance variables. The values in super._parametersMap are set by
     * parsing parameter xml file in OHDFewsAdapter.java.
     * 
     * @throws Exception
     */
    @Override
    public void extractValuesFromMap() throws Exception
    {
        _elev = getDoubleDataParameter(ELEV_TAG);

        _taelev = getDoubleDataParameter(TAELEV_TAG); //TAELEV should be present due to Snow17OptFileConverter

        _alat = getDoubleDataParameter(ALAT_TAG);

        _pxadj = getDoubleDataParameter(PXADJ_TAG);

        if(isParamExisting(RAIN_SNOW_ELEV_INPUT_TAG))
        {
            _useRainSnowElevInput = getBooleanDataParameter(RAIN_SNOW_ELEV_INPUT_TAG);
        }

        if(isParamExisting(PERCENT_SNOWFALL_INPUT_TAG))
        {
            _usePercentSnowFall = getBooleanDataParameter(PERCENT_SNOWFALL_INPUT_TAG);
        }

        _scf = getDoubleDataParameter(SCF_TAG);

        _mfmax = getDoubleDataParameter(MFMAX_TAG);

        _mfmin = getDoubleDataParameter(MFMIN_TAG);

        _uadj = getDoubleDataParameter(UADJ_TAG);

        _si = getDoubleDataParameter(SI_TAG);

        _mv = getIntDataParameter(MV_TAG);

        if(_mv == 1)
        {
            _smfv = getDoubleArrayParameter(SMFV_TAG);
        }

        _nmf = getDoubleDataParameter(NMF_TAG);

        _tipm = getDoubleDataParameter(TIPM_TAG);

        _mbase = getDoubleDataParameter(MBASE_TAG);

        _pxtemp = getDoubleDataParameter(PXTEMP_TAG);

        _plwhc = getDoubleDataParameter(PLWHC_TAG);

        _daygm = getDoubleDataParameter(DAYGM_TAG);

        if(isParamExisting(WETOL_TAG))
        {
            _wetol = getDoubleDataParameter(WETOL_TAG);
        }

        if(isParamExisting(SCTOL_TAG))
        {
            _sctol = getDoubleDataParameter(SCTOL_TAG);
        }

        _adc = getDoubleArrayParameter(OHDConstants.ADC_TAG);

    } //close extractValuesFromMap()

    /**
     * Set values from instance variables into super._parametersMap Map. The original values in super._parametersMap are
     * set by parsing parameter xml file in OHDFewsAdapter.java.
     * 
     * @throws Exception
     */
    @Override
    public void setValuesToMap() throws Exception
    {
        super.insertParameter(ELEV_TAG, _elev);

        super.insertParameter(TAELEV_TAG, _taelev); //TAELEV should be present due to Snow17OptFileConverter

        super.insertParameter(ALAT_TAG, _alat);

        super.insertParameter(PXADJ_TAG, _pxadj);

        if(isParamExisting(RAIN_SNOW_ELEV_INPUT_TAG))
        {
            super.insertParameter(RAIN_SNOW_ELEV_INPUT_TAG, _useRainSnowElevInput);
        }

        if(isParamExisting(PERCENT_SNOWFALL_INPUT_TAG))
        {
            super.insertParameter(PERCENT_SNOWFALL_INPUT_TAG, _usePercentSnowFall);
        }

        super.insertParameter(SCF_TAG, _scf);

        super.insertParameter(MFMAX_TAG, _mfmax);

        super.insertParameter(MFMIN_TAG, _mfmin);

        super.insertParameter(UADJ_TAG, _uadj);

        super.insertParameter(SI_TAG, _si);

        super.insertParameter(MV_TAG, _mv);

        if(_mv == 1)
        {
            super.insertParameter(SMFV_TAG, _smfv);
        }

        super.insertParameter(NMF_TAG, _nmf);

        super.insertParameter(TIPM_TAG, _tipm);

        super.insertParameter(MBASE_TAG, _mbase);

        super.insertParameter(PXTEMP_TAG, _pxtemp);

        super.insertParameter(PLWHC_TAG, _plwhc);

        super.insertParameter(DAYGM_TAG, _daygm);

        if(isParamExisting(WETOL_TAG))
        {
            super.insertParameter(WETOL_TAG, _wetol);
        }

        if(isParamExisting(SCTOL_TAG))
        {
            super.insertParameter(SCTOL_TAG, _sctol);
        }

        super.insertParameter(OHDConstants.ADC_TAG, _adc);

    } //close setValuesToMap()

    /**
     * Validate several time series intervals fit the constraints.
     */
    public void validateParams() throws ModelException, Exception
    { //assuming isValidated() == false

        String errorMessage;

        /** --------------if TAELEV != ELEV, needs TALMAX & TALMIN ----------------- */
        if(_elev != _taelev)
        {
            if(isParamExisting(TALMAX_TAG) == false || isParamExisting(TALMIN_TAG) == false)
            {
                errorMessage = "TALMAX or TALMIN parameter is absent. They are needed when TAELEV != ELEV. TAELEV="
                    + getElevationWithAirTemp() + " ELEV=" + getElevation();

                throw new ModelException(errorMessage);
            }
            else
            {//both TALMAX and TALMIN exist

                _talmax = getDoubleDataParameter(TALMAX_TAG);
                _talmin = getDoubleDataParameter(TALMIN_TAG);
            }

            if(getTalmax() < 0.0 || getTalmin() < 0.0)
            {
                _logger.log(Logger.WARNING,
                            "At least one of the lapse rates are not positive. Check that values are correct.");
            }
        }

        if(_usePercentSnowFall)
        {
            _logger.log(Logger.DEBUG, "Using percent snowfall TS.");

            //          check the optional input percent snowfall TS interval: must be equal to MAP interval
            if(_inputPercentSnowFallInterval != _inputTSMapInterval)
            {
                errorMessage = "Error: Input percent snowfall TS interval must be equal to the input precipitation TS(MAP) "
                    + "interval. Currently, percent snowfall TS interval(HR)= "
                    + _inputPercentSnowFallInterval
                    + " precipitation TS interval(HR)= " + _inputTSMapInterval;

                throw new ModelException(errorMessage);
            }
        }
        else
        {
            _logger.log(Logger.DEBUG, "NOT using percent snowfall TS. ");
        }

        if(_useRainSnowElevInput)
        {
            _logger.log(Logger.DEBUG, "Using rain-snow elevation TS.");

            if(isParamExisting(AE_PAIRS_ENGL_TAG) == false && isParamExisting(AE_PAIRS_METR_TAG) == false)
            {
                throw new ModelException("Using rain-snow elevation input, but no area elevation curve parameters available");
            }

            //because of using RAIN-SNOW-ELEVATION, ae_pairs are present: extract data from _paramsMap and create _ae_pairs
            boolean metrUnit = true;

            Table table = null;
            if(isParamExisting(AE_PAIRS_METR_TAG) == true)
            {
                table = super.getTableDataParameter(AE_PAIRS_METR_TAG);
            }
            else
            { //default is English units
                table = super.getTableDataParameter(AE_PAIRS_ENGL_TAG);
                metrUnit = false;
            }

            if(table.getRow() != null)
            {
                /**
                 * The format in params.xml is like:<br>
                 * <table>
                 * <br>
                 * <columnTypes A="double" B="double"/><br>
                 * <row A="914.4" B="0.0"/><br>
                 * <row A="1219.2" B="0.6"/><br>
                 * <row A="1524.0" B="0.94"/><br>
                 * <row A="1766.316" B="1.0"/><br>
                 * </table>
                 * <br>
                 * However, _ae_pairs 2-D array is two rows, with many columns:first row is the first column in
                 * params.xml(elevation); second row is the second column in params.xml(percentage)
                 */
                _ae_pairs = new double[2][table.getRow().size()];
                int i = 0;
                for(final Row row: table.getRow())
                {
                    _ae_pairs[0][i] = Double.valueOf(row.getA()); //elevation(unit is foot(ENGL) or meter(METR)
                    _ae_pairs[1][i] = Double.valueOf(row.getB()); //percentage   
                    if(metrUnit == false)
                    {//the number is foot unit, needs to convert to meter

                        _ae_pairs[0][i] *= 0.3048; //1 foot = 0.3048 M
                    }
                    i++;
                }
            }

            //check the optional input Rain Snow TS interval: must be equal to MAT interval
            if(_inputRainSnowInterval != _inputTSMatInterval)
            {
                errorMessage = "Error: Input rain-snow-elevation TS interval must be equal to the input air temperature "
                    + "TS(MAT) interval. Currently, rain-snow-elevation TS interval(HR)= "
                    + _inputRainSnowInterval
                    + " air temperature TS interval(HR)= " + _inputTSMatInterval;

                throw new ModelException(errorMessage);
            }
        }
        else
        {//not using RAIN-SNOW-ELEVATION
            _logger.log(Logger.DEBUG, "NOT using rain-snow elevation TS. ");
        }

        /** ---------check the values consistent with constraints and log it--------------- */

        //check MAP and MAT interval: MAT interval must be equal or even multiple of MAP interval
        if(OHDUtilities.isEvenMultiple(_inputTSMatInterval, _inputTSMapInterval) == false)
        {
            errorMessage = "Error: Input air temperature TS(MAT) interval must be equal or an even multiple of the input "
                + "precip TS(MAP) interval. Currently, air temperature TS interval(HR)= "
                + _inputTSMatInterval
                + " precip TS interval(HR)= " + _inputTSMapInterval;

            throw new ModelException(errorMessage);
        }

        /*
         * check the 3 output TS(s)(SWE, SASC, and SNSG) interval: must be equal or even multiple of MAT interval.
         * Output RAIM TS interval is fixed to MAP interval
         */
        checkOptionalOutputTsInterval(ITSWE_TAG);
        checkOptionalOutputTsInterval(ITSASC_TAG);
        checkOptionalOutputTsInterval(ITSH_TAG);

        modifyParametersAsPin19();

        //if reach here, validation is successful
        _logger.log(Logger.DEBUG, "Snow17ModelParameters has been validated!");

        return;

    } //close validateParams

    /**
     * This private method serves for the 3 optional output TSs(SWE, SASC & SNSG). Two purposes: 1)If the parameter with
     * name of {@link #ITSWE_TAG}, {@link #ITSASC_TAG} or {@link #ITSH_TAG} is absent in the parameter file, log a
     * warning and set it with the default value, input MAT interval; 2)when the parameter is present, confirm it is
     * multiple of MAT interval. If not so, logs and throws an Exception.
     * 
     * @param paramsTag
     */
    private void checkOptionalOutputTsInterval(final String paramsTag) throws ModelException, Exception
    {
        int interval = _inputTSMatInterval; //initialize to the default value first

        if(super._validPeriodParametersMap.size() == 0 || isParamExisting(paramsTag) == false)
        {
            _logger.log(Logger.DEBUG, "The parameter " + paramsTag
                + " is absent. Use the default value, input MAT time series interval, " + _inputTSMatInterval + " hr.");

            //do not change interval, use the default value
        }
        else
        {// the parameter is existing
            final int tempInterval = getIntDataParameter(paramsTag);

            //sometimes fcinit punch out segment definition files with output TS interval of 0, needs to handle it
            if(tempInterval == 0)
            {
                _logger.log(Logger.WARNING, "The parameter " + paramsTag
                    + " is 0. Use the default value, input MAT time series interval, " + _inputTSMatInterval + " hr.");

                //do not change interval, use the default value
            }
            else if(OHDUtilities.isEvenMultiple(tempInterval, _inputTSMatInterval) == false)
            {
                final String errorMessage = "Error: " + paramsTag
                    + " must be equal or even multiple of the input air temperature " + "TS(MAT) interval. Currently, "
                    + paramsTag + "(HR)= " + tempInterval + " air temperature TS interval(HR)= " + _inputTSMatInterval;

                throw new ModelException(errorMessage);
            }
            else
            {//now tempInterval is not "0" and is multiple of MAT interval, set interval value

                interval = tempInterval;
            }

        } //close else "the parameter is existing"

        //now sets instance variables' values to interval
        if(paramsTag.equals(ITSWE_TAG))
        {
            _itswe = interval;
        }
        else if(paramsTag.equals(ITSASC_TAG))
        {
            _itsasc = interval;
        }
        else if(paramsTag.equals(ITSH_TAG))
        {
            _itsh = interval;
        }

    } //close method

    /**
     * Modify parameters as Fortran subroutine Pin19 does
     */
    public void modifyParametersAsPin19()
    {

        //re-adjust some values(need MAT interval info, so it has to be set already) 
        _mfmax *= _inputTSMatInterval / 6.0; // 6.0 is hard-coded in ex19.f

        _mfmin *= _inputTSMatInterval / 6.0;

        _nmf *= _inputTSMatInterval / 6.0;

        _uadj *= _inputTSMapInterval / 6.0;

        _tipm = 1.0 - Math.pow(1.0 - getTemperatureIndex(), _inputTSMatInterval / 6.0);

        if(_mv == 1) //Card#6A is included
        {
            for(int i = 0; i < _smfv.length; i++)
            {
                if(_smfv[i] < 0)
                {
                    _smfv[i] = 0.0;
                }
                if(_smfv[i] > 1.0)
                {
                    _smfv[i] = 1.0;
                }
            }

        }

        /** ---------------------- Card#7 ---------------------------- */

        //Lower limit of TIPM is 0.01; Max limit of TIPM is 1.0
        if(getTemperatureIndex() < 0.01)
        {
            _tipm = 0.01;
        }
        else if(getTemperatureIndex() > 1.0)
        {
            _tipm = 1.0;
        }

        if(getPercentLiquidWaterHoldingCap() > 1.0)
        {
            _plwhc *= 0.01;
        }

        //Make sure PLWHC <= (1.0 - MAXIMUM ALLOWED DENSITY), currently 0.6 in SNDEPTH19 
        if(getPercentLiquidWaterHoldingCap() > 0.4)
        {
            _plwhc = 0.4;
        }

    }

    /**
     * get ADC array
     */
    public double[] getAdc()
    {
        return _adc;
    }

    public void setAdc(final double[] adc)
    {
        _adc = adc.clone();
    }

    /**
     * Gets the two-D array(two rows and multiple columns) ae_pairs in METR unit, no matter in parameter xml file it is
     * in ENGL or METR unit. (In parameter xml, it is two columns and multiple rows.) First row value is the
     * elevation(feet or meter). Second row value is the decimal fraction of area below the elevation. The pair values
     * are specified in original opt file card 3B, except the first and last pairs, (EMIN, 0.0) and (EMAX, 1.0).
     * <p>
     * This parameter(ae_pairs in card3A or even 3B) is only available when using rain snow elevation. (
     * {@link #useRainSnowElevationInput()} is true). Returns null when not available
     * 
     * @return the ae_pairs
     */
    @SuppressWarnings("boxing")
    double[][] getAe_pairs()
    {
        return _ae_pairs;
    }

    /**
     * get ALAT
     */
    public double getLatitude()
    {
        return _alat;
    }

    public void setLatitude(final double alat)
    {
        _alat = alat;
    }

    /**
     * get DAYGM: daily constant amount of melt which occurs at the snow-soil interface whenever snow is present (mm)
     */
    public double getDailyGroundMelt()
    {
        return _daygm;
    }

    public void setDailyGroundMelt(final double daygm)
    {
        _daygm = daygm;
    }

    /**
     * get ELEV
     */
    public double getElevation()
    {
        return _elev;
    }

    public void setElevation(final double elev)
    {
        _elev = elev;
    }

    /**
     * get MBASE: base temperature for snowmelt computations during non-rain periods (DEGC)
     */
    public double getBaseTemperature()
    {
        return _mbase;
    }

    public void setBaseTemperature(final double mbase)
    {
        _mbase = mbase;
    }

    /**
     * get MFMAX: Maximum melt factor during non-rain periods, assumed to occur on June 21(mm/DEGC/6hr)
     */
    public double getMaxMeltFactor()
    {
        return _mfmax;
    }

    public void setMaxMeltFactor(final double mfmax)
    {
        _mfmax = mfmax;
    }

    /**
     * get MFMIN: Minimum melt factor during non-rain periods, assumed to occur on December 21(mm/DEGC/6hr)
     */
    public double getMinMeltFactor()
    {
        return _mfmin;
    }

    public void setMinMeltFactor(final double mfmin)
    {
        _mfmin = mfmin;
    }

    /**
     * get MV
     */
    public int getMeltFactorVariationIndicator()
    {
        return _mv;
    }

    public void setMeltFactorVariationIndicator(final int mv)
    {
        _mv = mv;
    }

    /**
     * get NMF
     */
    public double getNegMeltFactor()
    {
        return _nmf;
    }

    public void setNegMeltFactor(final double nmf)
    {
        _nmf = nmf;
    }

    /**
     * get NPATE + 2. Calculate based on the ae_pairs.
     */
    int getAreaElevCurveNum()
    {
        return _ae_pairs[0].length;
    }

    /**
     * get PLWHC: Percent (decimal) liquid water holding capacity, indicates the max amount of liquid water that can be
     * held against gravity drainage in the snow cover
     */
    public double getPercentLiquidWaterHoldingCap()
    {
        return _plwhc;
    }

    public void setPercentLiquidWaterHoldingCap(final double plwhc)
    {
        _plwhc = plwhc;
    }

    /**
     * get PXADJ
     */
    public double getPxadj()
    {
        return _pxadj;
    }

    public void setPxadj(final double pxadj)
    {
        _pxadj = pxadj;
    }

    /**
     * get PXTEMP: the temperature that delineates rain from snow(DEGC)
     */
    public double getDelineationTemperature()
    {
        return _pxtemp;
    }

    public void setDelineationTemperature(final double pxtemp)
    {
        _pxtemp = pxtemp;
    }

    /**
     * get SCF: multiplying factor that adjusts precipitation data for gage catch deficiencies, gains or losses due to
     * drifting.
     */
    public double getSnowCorrectionFactor()
    {
        return _scf;
    }

    public void setSnowCorrectionFactor(final double scf)
    {
        _scf = scf;
    }

    /**
     * get SI: the mean areal water-equivalent above which there is always 100% areal snow cover (mm)
     */
    public double getArealWeUnderFullSnowCover()
    {
        return _si;
    }

    public void setArealWeUnderFullSnowCover(final double si)
    {
        _si = si;
    }

    /**
     * get SMPV
     */
    double[] getSeasonalMeltFactorVariationArray()
    {
        return _smfv;
    }

    /**
     * get TAELEV
     */
    public double getElevationWithAirTemp()
    {
        return _taelev;
    }

    public void setElevationWithAirTemp(final double taelev)
    {
        _taelev = taelev;
    }

    /**
     * get TALMAX. Lapse rate at warmest temperature, assuming 3PM local time. Units of DEGC/100M.
     */
    public double getTalmax()
    {
        return _talmax;
    }

    /**
     * get TALMIN. Lapse rate at coldest temperature, assuming 6AM local time. Units of DEGC/100M.
     */
    public double getTalmin()
    {
        return _talmin;
    }

    /**
     * get TIPM
     */
    public double getTemperatureIndex()
    {
        return _tipm;
    }

    public void setTemperatureIndex(final double tipm)
    {
        _tipm = tipm;
    }

    /**
     * get UADJ: the average wind function during rain-on-snow periods(mm/mb/6hr)
     */
    public double getWindFuncParameter()
    {
        return _uadj;
    }

    public void setWindFuncParameter(final double uadj)
    {
        _uadj = uadj;
    }

    /**
     * Gets the interval(unit of HR) of precipitation time series(MAP), default is 6 HR.
     * 
     * @return the precip interval
     */
    public int getPrecipInterval()
    {
        return _inputTSMapInterval;
    }

    /**
     * Sets the interval(unit of HR) of precipitation time series(MAP). It can be any number.
     * 
     * @param itpx the new precip interval
     */
    public void setPrecipInterval(final int itpx)
    {
        _inputTSMapInterval = itpx;
    }

    /**
     * Gets the interval(unit of HR) of air temperature time series(MAT), default is 6 HR.
     * 
     * @return the air temp interval
     */
    public int getAirTempInterval()
    {
        return _inputTSMatInterval;
    }

    /**
     * Sets the interval(unit of HR) of air temperature time series(MAT), must be an even multiple of the data time
     * interval for precipitation data.
     * 
     * @param idt the new air temp interval
     */
    public void setAirTempInterval(final int idt)
    {
        _inputTSMatInterval = idt;
    }

    /**
     * Return wetol. In forecasting mode, this parameter is definitely needed; In MCP3 mode, it is only needed and
     * available when isUsingUpdate() == true
     * 
     * @return the tolerance used when updating water-equivalent-decimal fraction; updating only occurs if
     *         |Simulated-Observed| > tolerance * Observed
     */
    double getWeToleranceFaction()
    {
        return _wetol;
    }

    /**
     * Return sctol. In forecasting mode, this parameter is definitely needed; In MCP3 mode, it is only needed and
     * available when isUsingUpdate() == true
     * 
     * @return the tolerance used when updating areal extend of snow cover; updating only occurs if |Simulated-Observed|
     *         > tolerance
     */
    double getScToleranceFaction()
    {
        return _sctol;
    }

    /**
     * Gets the data time interval of simulated water equivalent time series(Units of HR). If specification does not
     * want the output time series and does not specify its data interval, take air temperature data interval -- done
     * during validation.
     * 
     * @return the swe interval
     */
    public int getSweInterval()
    {
        return _itswe;
    }

    /**
     * Gets the data time interval of simulated areal snow cover time series(Units of HR). If specification does not
     * want the output time series and does not specify its data interval, take air temperature data interval -- done
     * during validation.
     * 
     * @return the sasc interval
     */
    public int getSascInterval()
    {
        return _itsasc;
    }

    /**
     * Gets the data time interval of simulated snow depth time series(Units of HR). If specification does not want the
     * output time series and does not specify its data interval, take air temperature data interval -- done during
     * validation.
     * 
     * @return the snsg interval
     */
    public int getSnsgInterval()
    {
        return _itsh;
    }

    /**
     * If true, use the input rain-snow-elevation TS, which is required to be present then; if false, ignore the input
     * rain-snow-elevation TS, even though it is present. And only when it is true, {@link #getAe_pairs()} is available.
     */
    public boolean useRainSnowElevationInput()
    {
        return _useRainSnowElevInput;
    }

    public void setUseRainSnowElevationInput(final boolean b)
    {
        _useRainSnowElevInput = b;
    }

    /**
     * If true, use the input percent-snow-fall TS, which is required to be present then; if false, ignore the input
     * percent-snow-fall TS, even though it is present.
     */
    public boolean usePercentSnowFallInput()
    {
        return _usePercentSnowFall;
    }

    public void setUsePercentSnowFallInput(final boolean b)
    {
        _usePercentSnowFall = b;
    }

    int getInputPercentSnowFallInterval()
    {
        return _inputPercentSnowFallInterval;
    }

    public void setInputPercentSnowFallInterval(final int inputPercentSnowFallInterval)
    {
        _inputPercentSnowFallInterval = inputPercentSnowFallInterval;
    }

    int getInputRainSnowInterval()
    {
        return _inputRainSnowInterval;
    }

    void setInputRainSnowInterval(final int inputRainSnowInterval)
    {
        _inputRainSnowInterval = inputRainSnowInterval;
    }

    /**
     * MAT ts is driving TS and its interval could be larger than MAP interval(multiple times). At every time step of
     * MAT, there is a loop with fixed MAT value, but with different MAP value, if MAT interval is larger than MAP
     * interval. If they are equal, then only loop once.
     */
    int getRatioMatMapInterval()
    {
        return _inputTSMatInterval / _inputTSMapInterval;
    }

    /**
     * Override {@link Parameters#toString()} which bases on the values in its _paramsMap. Here, it is based on the
     * instance variables which might have been modified.
     */
    public String toStringInstanceValues()
    {
        final StringBuilder resultStr = new StringBuilder();

        resultStr.append(OHDConstants.NEW_LINE);

        resultStr.append(ELEV_TAG + "=" + _elev).append(OHDConstants.NEW_LINE);
        resultStr.append(TAELEV_TAG + "=" + _taelev).append(OHDConstants.NEW_LINE);
        resultStr.append(ALAT_TAG + "=" + _alat).append(OHDConstants.NEW_LINE);
        resultStr.append(PXADJ_TAG + "=" + _pxadj).append(OHDConstants.NEW_LINE);
        resultStr.append(RAIN_SNOW_ELEV_INPUT_TAG + "=" + _useRainSnowElevInput).append(OHDConstants.NEW_LINE);
        resultStr.append(PERCENT_SNOWFALL_INPUT_TAG + "=" + _usePercentSnowFall).append(OHDConstants.NEW_LINE);
        resultStr.append(SCF_TAG + "=" + _scf).append(OHDConstants.NEW_LINE);
        resultStr.append(MFMAX_TAG + "=" + _mfmax).append(OHDConstants.NEW_LINE);
        resultStr.append(MFMIN_TAG + "=" + _mfmin).append(OHDConstants.NEW_LINE);
        resultStr.append(UADJ_TAG + "=" + _uadj).append(OHDConstants.NEW_LINE);
        resultStr.append(SI_TAG + "=" + _si).append(OHDConstants.NEW_LINE);
        resultStr.append(MV_TAG + "=" + _mv).append(OHDConstants.NEW_LINE);
        resultStr.append(SMFV_TAG + "=" + Arrays.toString(_smfv)).append(OHDConstants.NEW_LINE);
        resultStr.append(NMF_TAG + "=" + _nmf).append(OHDConstants.NEW_LINE);
        resultStr.append(TIPM_TAG + "=" + _tipm).append(OHDConstants.NEW_LINE);
        resultStr.append(MBASE_TAG + "=" + _mbase).append(OHDConstants.NEW_LINE);
        resultStr.append(PXTEMP_TAG + "=" + _pxtemp).append(OHDConstants.NEW_LINE);
        resultStr.append(PLWHC_TAG + "=" + _plwhc).append(OHDConstants.NEW_LINE);
        resultStr.append(DAYGM_TAG + "=" + _daygm).append(OHDConstants.NEW_LINE);

        resultStr.append(WETOL_TAG + "=" + _wetol).append(OHDConstants.NEW_LINE);

        resultStr.append(SCTOL_TAG + "=" + _sctol).append(OHDConstants.NEW_LINE);

        resultStr.append(OHDConstants.ADC_TAG + "=" + Arrays.toString(_adc)).append(OHDConstants.NEW_LINE);

        return resultStr.toString();
    }

	public void modifyParametersForDifferentTimestep(
			final int modelInterval) throws Exception {
		
		setAirTempInterval(modelInterval);
	
		setPrecipInterval(modelInterval);
	
		setInputPercentSnowFallInterval(modelInterval);
	
		modifyParametersAsPin19(); //modify parameters based on new model interval     
	     
	}

} // close class
