/**
 * 
 */
package ohd.hseb.ohdmodels.snow17;

import java.util.Arrays;

import ohd.hseb.util.Logger;
import ohd.hseb.util.MathHelper;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.Parameters;
import ohd.hseb.util.fews.State;
import ohd.hseb.util.fews.ohdmodels.ModelParameters;
import ohd.hseb.util.fews.ohdmodels.ModelState;

/**
 * This class holds all the Snow17 Model state variables. It is a subclass of the abstract class ModelState and
 * interface IModelState. The state variable values are loaded from a text file with properties format through
 * {@link #loadState(String, Logger)}. The values are are initially loaded into the Map<String, Object> _statesMap in
 * ModelState, then the values are extracted to the corresponding instance variables.
 * 
 * @author CHPS
 */
public final class Snow17ModelState extends ModelState
{

    //CARD #9 TAGS:
    public final static String WE_TAG = "WE"; // _we [MM]
    private double _we;
    public final static String NEGHS_TAG = "NEGHS"; // _neghs [MM]
    private double _neghs;
    public final static String LIQW_TAG = "LIQW"; // _liqw [MM]
    private double _liqw;
    public final static String TINDEX_TAG = "TINDEX"; // _tindex [DEGC]
    private double _tindex;
    public final static String ACCMAX_TAG = "ACCMAX"; // _accmax [MM]
    private double _accmax;
    public final static String SNDPT_TAG = "SNDPT";
    private double _sndpt;
    /*
     * _sndpt, in model computation is with unit of CM. Both input state txt file and output state txt file are with
     * unit of CM. The output TS _snsgTs in Snow17ModelDriver is in unit of CM too. However, when being written out to
     * outputs.xml, ohdfewsadapter automatically converts it to MM according to nwsrfs_datatype_mapping_file.txt.
     */
    public final static String SNTMP_TAG = "SNTMP"; // _sntmp [DEGC]
    private double _sntmp;
    public final static String TAPREV_TAG = "TAPREV"; // _taprev [DEGC]
    private double _taprev;

    //CARD #10 TAGS(optional in NWSRFS), but for Java model they are required to be present in the state file too
    public final static String SB_TAG = "SB"; // _sb [MM]
    private double _sb;
    public final static String SBAESC_TAG = "SBAESC"; // _sbaesc [PCTD]
    private double _sbaesc;
    public final static String SBWS_TAG = "SBWS"; // _sbws [MM]
    private double _sbws;
    public final static String STORGE_TAG = "STORGE"; // _storge [MM]
    private double _storge;
    public final static String AEADJ_TAG = "AEADJ"; // _aeadj [MM]
    private double _aeadj;
    private double[] _exlag;

    //the following two states property names can be absent in the state file
    private final static String ITPX_TAG = "MAP_INTERVAL"; // _itpx
    private int _itpx = 6; //previous MAP interval default value is set to 6, could be re-set if present in statesI.txt

    public final static String PQNET_TAG = "PQNET";
    private double _pqnet; //not in _statesMap, so will not in statesO.txt, but in outputStateTS.xml

    private double _snden; //not in _statesMap

    /**
     * Default constructor. It gets all its values in _statesMap by {@link State#loadState(String, Logger)}.
     */
    public Snow17ModelState()
    {
        super();
    }

    /**
     * Extract values from super._statesMap into instance variables. The values in super._statessMap are set by parsing
     * the initial state file in OHDFewsAdapter.java.
     */
    @Override
    public void extractValuesFromMap() throws Exception
    {
        //Card #9
        _we = getDoubleDataState(WE_TAG);
        _neghs = getDoubleDataState(NEGHS_TAG);
        _liqw = getDoubleDataState(LIQW_TAG);
        _tindex = getDoubleDataState(TINDEX_TAG);
        _accmax = getDoubleDataState(ACCMAX_TAG);
        _sndpt = getDoubleDataState(SNDPT_TAG);
        _sntmp = getDoubleDataState(SNTMP_TAG);
        _taprev = getDoubleDataState(TAPREV_TAG);

        //Card #10
        _sb = getDoubleDataState(SB_TAG);
        _sbaesc = getDoubleDataState(SBAESC_TAG);
        _sbws = getDoubleDataState(SBWS_TAG);
        _storge = getDoubleDataState(STORGE_TAG);
        _aeadj = getDoubleDataState(AEADJ_TAG);
        _exlag = getDoubleArrayDataState(OHDConstants.EXLAGARRAY_TAG);

        if(isStateExisting(ITPX_TAG))
        {
            _itpx = getIntDataState(ITPX_TAG);
        }
    }

    /**
     * Call super class {@link State} to validate all required tags present in _statesMap; confirm the input states unit
     * is METRIC.
     */
    public void validateState(final Parameters params) throws Exception
    {
        super.confirmUnitMetric();
    }

    /**
     * Over-written the super class method. This method is used in {@link State#writeState(String, Logger)} to output
     * states into the generated state txt file.
     */
    @Override
    protected String getStringsFromState()
    {
        final StringBuilder resultStr = new StringBuilder();

        resultStr.append(OHDConstants.NEW_LINE);

        //Card #9
        resultStr.append(WE_TAG + "=" + _we).append(OHDConstants.NEW_LINE);
        resultStr.append(NEGHS_TAG + "=" + _neghs).append(OHDConstants.NEW_LINE);
        resultStr.append(LIQW_TAG + "=" + _liqw).append(OHDConstants.NEW_LINE);
        resultStr.append(TINDEX_TAG + "=" + _tindex).append(OHDConstants.NEW_LINE);
        resultStr.append(ACCMAX_TAG + "=" + _accmax).append(OHDConstants.NEW_LINE);
        resultStr.append(SNDPT_TAG + "=" + _sndpt).append(OHDConstants.NEW_LINE);
        resultStr.append(SNTMP_TAG + "=" + _sntmp).append(OHDConstants.NEW_LINE);
        resultStr.append(TAPREV_TAG + "=" + _taprev).append(OHDConstants.NEW_LINE);

        //Card #10
        resultStr.append(SB_TAG + "=" + _sb).append(OHDConstants.NEW_LINE);
        resultStr.append(SBAESC_TAG + "=" + _sbaesc).append(OHDConstants.NEW_LINE);
        resultStr.append(SBWS_TAG + "=" + _sbws).append(OHDConstants.NEW_LINE);
        resultStr.append(STORGE_TAG + "=" + _storge).append(OHDConstants.NEW_LINE);
        resultStr.append(AEADJ_TAG + "=" + _aeadj).append(OHDConstants.NEW_LINE);

//        resultStr.append(EXLAGARRAY_TAG + "=" + Arrays.toString(_exlag)).append(OHDConstants.NEW_LINE);
        resultStr.append(OHDConstants.EXLAGARRAY_TAG + "=");
        for(int i = 0; i < _exlag.length; i++)
        {
            resultStr.append(_exlag[i] + " ");
        }
        resultStr.append(OHDConstants.NEW_LINE);
        resultStr.append(ITPX_TAG + "=" + getITPX() + OHDConstants.NEW_LINE);
        resultStr.append(OHDConstants.UNIT_TAG + "=" + OHDConstants.UNIT_METRIC + OHDConstants.NEW_LINE);

        return resultStr.toString();
    }

    /**
     * get ACCMAX
     */
    public double getMaxWeSinceSnow()
    {
        return _accmax;
    }

    public void setMaxWeSinceSnow(final double accmax)
    {
        _accmax = accmax;
    }

    /**
     * get AEADJ
     */
    public double getSnowArealExtentAdjustment()
    {
        return _aeadj;
    }

    public void setSnowArealExtentAdjustment(final double aeadj)
    {
        _aeadj = aeadj;
    }

    /**
     * Gets the EXLAG[]. EXLAG array is stored as one String line in _statesMap, with each item separated by one space.
     * Need to convert the line into double []
     */
    public double[] getExlagArray()
    {
        return _exlag;
    }

    /**
     * Sets the EXLAG[]. EXLAG array is stored as one String line in _statesMap, with each item separated by one space.
     * Need to convert the double array into one String line first, then insert it into _statesMap
     */
    public void setExlagArray(final double[] exlagArray)
    {
        _exlag = exlagArray;
    }

    /**
     * Calculate the total sum of exlag array.
     */
    double getExLagTotal()
    {
        double total = 0.0;

        final double[] exlag = getExlagArray();

        for(int i = 0; i < exlag.length; i++)
        {
            total += exlag[i];
        }

        return total;
    }

    /**
     * get LIQW
     */
    public double getLiquidWaterInSnowPack()
    {
        return _liqw;
    }

    public void setLiquidWaterInSnowpack(final double liqw)
    {
        _liqw = liqw;
    }

    /**
     * get NEGHS
     */
    public double getNegHeatStorage()
    {
        return _neghs;
    }

    public void setNegHeatStorage(final double neghs)
    {
        _neghs = neghs;
    }

    /**
     * get SB
     */
    public double getArealWaterEquivBeforeSnow()
    {
        return _sb;
    }

    public void setArealWaterEquivBeforeSnow(final double sb)
    {
        _sb = sb;
    }

    /**
     * get SBAESC
     */
    public double getAescBeforeSnow()
    {
        return _sbaesc;
    }

    public void setAescBeforeSnow(final double sbaesc)
    {
        _sbaesc = sbaesc;
    }

    /**
     * get SBWS
     */
    public double getWaterEquivalentUnderSnowCover()
    {
        return _sbws;
    }

    public void setWaterEquivalentUnderSnowCover(final double sbws)
    {
        _sbws = sbws;
    }

    /**
     * get SNDPT, units of CM.
     */
    public double getSnowDepth()
    {
        return _sndpt;
    }

    /**
     * Set snow depth, units of CM.
     */
    public void setSnowDepth(final double sndpt)
    {
        _sndpt = sndpt;
    }

    /**
     * get SNTMP
     */
    public double getAverageSnowTemp()
    {
        return _sntmp;
    }

    public void setAverageSnowTemp(final double sntmp)
    {
        _sntmp = sntmp;
    }

    void setSnowDensity(final double snden)
    {
        _snden = snden;
    }

    /**
     * get SNDEN
     */
    double getSnowDensity()
    {

        if(_snden == Double.NEGATIVE_INFINITY)
        {
            return 0.0;
        }

        return _snden;
    }

    /**
     * get STORGE
     */
    public double getExcessLiquidWaterInStorage()
    {
        return _storge;
    }

    public void setExcessLiquidWaterInStorage(final double storge)
    {
        _storge = storge;
    }

    public double getTaprev()
    {
        return _taprev;
    }

    public void setTaprev(final double taprev)
    {
        _taprev = taprev;
    }

    /**
     * get TINDEX
     */
    public double getTempIndex()
    {
        return _tindex;
    }

    public void setTempIndex(final double tindex)
    {
        _tindex = tindex;
    }

    /**
     * get WE
     */
    public double getWaterEquivalentOfSolidSnow()
    {
        return _we;
    }

    public void setWaterEquivalentOfSolidSnow(final double we)
    {
        _we = we;
    }

    /**
     * TWE = WE + LIQW + TEX + STORGE. Calculate TWE from states values.
     */
    public double calculateTweFromState()
    {
        return getWaterEquivalentOfSolidSnow() + getLiquidWaterInSnowPack() + getExLagTotal()
            + getExcessLiquidWaterInStorage();
    }

    /**
     * Return MAP interval when this Snow17ModelState was calculated and saved. If "MAP_INTERVAL=.." is not present in
     * the statesI.txt file, will return the default value, 6.
     */
    public int getITPX()
    {
        return _itpx;
    }

    /**
     * Set MAP interval.
     */
    public void setITPX(final int itpx)
    {
        _itpx = itpx;
    }

    /**
     * This method is same as Fortran subroutine ZERO19(zero19.f). It re-sets all carryover values to no snow
     * conditions. Assuming EXLAG array has been created, so cannot be called in the constructor.
     */
    void zero19()
    {
        setWaterEquivalentOfSolidSnow(0.0);
        setNegHeatStorage(0.0);
        setLiquidWaterInSnowpack(0.0);
        setTempIndex(0.0);
        setMaxWeSinceSnow(0.0);
        setArealWaterEquivBeforeSnow(0.0);
        setAescBeforeSnow(0.0);
        setWaterEquivalentUnderSnowCover(0.0);
        setExcessLiquidWaterInStorage(0.0);
        setSnowArealExtentAdjustment(0.0);

        setSnowDepth(0.0);
        setAverageSnowTemp(0.0);

        //set EXLAG array to 0.0
        final double[] zeroArray = new double[getExlagArray().length];//null pointer exception if EXLAG array has not been created

        setExlagArray(zeroArray);

        setTaprev(0.0);

    }

    /**
     * This method is same as Fortran subroutine ADJC19(adjc19.f). Based on the new value of total
     * water-equivalent(twe), It adjusts state: WE, LIQW, ACCMAX, SB, SBAESC, SBWS, AEADJ, SNDPT, & SNTMP
     * 
     * @param ADC - a double array. It is not modified inside this method.
     */
    void adjc19(final double twe, final double si, final double[] ADC)
    {
        final double oldwe = getWaterEquivalentOfSolidSnow() + getLiquidWaterInSnowPack();

        double ai = getMaxWeSinceSnow();

//        if(getMaxWeSinceSnow() > si)
        if(MathHelper.isGreater(getMaxWeSinceSnow(), si))
        {
            ai = si;
        }

        double ainorm = 0.0;

//        if(getSnowArealExtentAdjustment() != 0.0)
        if(MathHelper.notEqualToZero(getSnowArealExtentAdjustment()))
        {
            ainorm = ai;
            ai = getSnowArealExtentAdjustment();
        }

        final double oldai = ai;

        final double tex = getExLagTotal();

        final double freew = getExcessLiquidWaterInStorage() + tex;

        double qual = 1.0;

//        if(getWaterEquivalentOfSolidSnow() > 0.0)
        if(MathHelper.isGreaterThanZero(getWaterEquivalentOfSolidSnow()))
        {
            qual = 1.0 + (getLiquidWaterInSnowPack() / getWaterEquivalentOfSolidSnow());
        }

        setWaterEquivalentOfSolidSnow((twe - freew) / qual);

        //snow depth is adjusted by ratio of the we change
//        if(getWaterEquivalentOfSolidSnow() > 0.0 && oldwe != getLiquidWaterInSnowPack())
        if(MathHelper.isGreaterThanZero(getWaterEquivalentOfSolidSnow())
            && MathHelper.isNotEqual(oldwe, getLiquidWaterInSnowPack()))
        {
            final double sndpt = getSnowDepth() * getWaterEquivalentOfSolidSnow()
                / (oldwe - getLiquidWaterInSnowPack());
            setSnowDepth(sndpt);
        }
//        else if(getWaterEquivalentOfSolidSnow() > 0.0)
        else if(MathHelper.isGreaterThanZero(getWaterEquivalentOfSolidSnow()))
        {
            setSnowDepth(0.5 * getWaterEquivalentOfSolidSnow());
        }
        else
        {
            setSnowDepth(0.0);
            setAverageSnowTemp(0.0);
        }

        setLiquidWaterInSnowpack((qual - 1.0) * getWaterEquivalentOfSolidSnow());

        final double swe = getWaterEquivalentOfSolidSnow() + getLiquidWaterInSnowPack();

        //if swe >= 3*SB, start new accumulation period and set aeadj to zero
//        if(swe < (3.0 * getArealWaterEquivBeforeSnow()))
        if(MathHelper.isLessThan(swe, (3.0 * getArealWaterEquivBeforeSnow())))
        {
//            if(getSnowArealExtentAdjustment() != 0.0)
            if(MathHelper.notEqualToZero(getSnowArealExtentAdjustment()))
            {
                //determine if aeadj should be removed - set to zero
//                if(getSnowArealExtentAdjustment() < ainorm && swe >= ainorm)
                if(MathHelper.isLessThan(getSnowArealExtentAdjustment(), ainorm)
                    && MathHelper.isGreaterOrEqual(swe, ainorm))
                {
                    setSnowArealExtentAdjustment(0.0);
                }

//                if(getSnowArealExtentAdjustment() >= ainorm && swe >= getSnowArealExtentAdjustment())
                if(MathHelper.isGreaterOrEqual(getSnowArealExtentAdjustment(), ainorm)
                    && MathHelper.isGreaterOrEqual(swe, getSnowArealExtentAdjustment()))
                {
                    setSnowArealExtentAdjustment(0.0);
                }

            }

//            if(oldwe > (0.8 * getMaxWeSinceSnow()))
            if(MathHelper.isGreater(oldwe, (0.8 * getMaxWeSinceSnow())))
            {
                setMaxWeSinceSnow(swe * getMaxWeSinceSnow() / oldwe);
            }
//            else if(swe > getMaxWeSinceSnow())
            else if(MathHelper.isGreater(swe, getMaxWeSinceSnow()))
            {
                setMaxWeSinceSnow(swe);
            }

            ai = getMaxWeSinceSnow();

//            if(getMaxWeSinceSnow() > si)
            if(MathHelper.isGreater(getMaxWeSinceSnow(), si))
            {
                ai = si;
            }

//            if(getSnowArealExtentAdjustment() > 0.0)
            if(MathHelper.isGreaterThanZero(getSnowArealExtentAdjustment()))
            {
                ai = getSnowArealExtentAdjustment();
            }

//            if(swe < ai)
            if(MathHelper.isLessThan(swe, ai))
            {
                double R;

                if(MathHelper.isLessThan(oldwe, oldai) && MathHelper.isGreater(oldwe, getArealWaterEquivBeforeSnow()))
                {

                    R = swe / oldwe;

                    setArealWaterEquivBeforeSnow(getArealWaterEquivBeforeSnow() * R);

                    setWaterEquivalentUnderSnowCover(getWaterEquivalentUnderSnowCover() * R);

                    R = (getArealWaterEquivBeforeSnow() / ai) * 10.0 + 1.0;

                }
                else
                {
                    setArealWaterEquivBeforeSnow(swe);

                    setWaterEquivalentUnderSnowCover(swe);

                    R = (swe / ai) * 10.0 + 1.0;
                }

                final int N = (int)Math.floor(R);

                R = R - Math.floor(R);

                double sbaesc = ADC[N - 1] + (ADC[N] - ADC[N - 1]) * R;

//                if(sbaesc > 1.0)
                if(MathHelper.isGreater(sbaesc, 1.0))
                {
                    sbaesc = 1.0;
                }

                setAescBeforeSnow(sbaesc);

                return; //note: there are two exits in this method
            }

        } //close if (swe < 3.0 *SB)
        else
        {
            setMaxWeSinceSnow(swe);
            setSnowArealExtentAdjustment(0.0);
        }

        setArealWaterEquivBeforeSnow(swe);
        setWaterEquivalentUnderSnowCover(swe);

        return; //note: there are two exits in this method

    } //close adjc19

    /**
     * This method corresponds to Fortran subroutine AECO19(aeco19.f). Adjusts snow model carryover values for a change
     * in the areal extent of the snow cover.
     * 
     * @param cover
     * @param twe
     * @param si
     * @param ADC
     */
    void aeco19(final double cover, final double twe, final double si, final double[] ADC)
    {
        //determine if currently on depletion curve or new snow line
        final double tex = getExLagTotal();

        final double freew = getExcessLiquidWaterInStorage() + tex;

        final double swe = twe - freew;

        double ai = getMaxWeSinceSnow();

//        if(getMaxWeSinceSnow() > si)
        if(MathHelper.isGreater(getMaxWeSinceSnow(), si))
        {
            ai = si;
        }

//        if(getSnowArealExtentAdjustment() > 0.0)
        if(MathHelper.isGreaterThanZero(getSnowArealExtentAdjustment()))
        {
            ai = getSnowArealExtentAdjustment();
        }

//        if(swe < ai)
        if(MathHelper.isLessThan(swe, ai))
        {

            if(MathHelper.isGreater(swe, getArealWaterEquivBeforeSnow()))
            {
                //currently on new snow line
//                if(cover > getAescBeforeSnow())
                if(MathHelper.isGreater(cover, getAescBeforeSnow()))
                {
                    double R = (swe / ai) * 10.0 + 1.0;

                    final int N = (int)Math.floor(R);

                    R = R - N;

                    final double aesc = ADC[N - 1] + (ADC[N] - ADC[N - 1]) * R;

//                    if(cover > aesc)
                    if(MathHelper.isGreater(cover, aesc))
                    {
                        //adjust sbws, leave aeadj as is
                        final double sbws = ((1.0 - getAescBeforeSnow()) / (cover - getAescBeforeSnow()) * (swe - getArealWaterEquivBeforeSnow()))
                            + getArealWaterEquivBeforeSnow();

                        setWaterEquivalentUnderSnowCover(sbws);

                        return;
                    }
                }
            }

        }

        //currently or should be on the depletion curve
        double weai;

        for(int i = 1; i < ADC.length; i++)
        {

            if(cover < ADC[i])
            {

                //106
                final int J = i - 1;
                final int FJ = J;
                //In Fortran scenario, x-value of ADC[1] is 0, of ADC[2] is 0.1, of ADC[3] is 0.2 ..
                //In Java scenario, x-value of ADC[0] is 0, of ADC[1] is 0.1, of ADC[2] is 0.2 ..
                //So, in Fortran: FJ = J - 1; But in Java: FJ = J;

                weai = 0.1 * (FJ + (cover - ADC[J]) / (ADC[i] - ADC[J]));

                //105
                setSnowArealExtentAdjustment(swe / weai);
                setAescBeforeSnow(cover);

                return;
            }

        } //close for loop

        weai = 1.0;

        //105
        setSnowArealExtentAdjustment(swe / weai);
        setAescBeforeSnow(cover);

        return;

    } //close aeco19

    @Override
    public void doCarryOverTransfer(final ModelParameters savedParams, final ModelParameters params)
    {
        final Snow17ModelParameters snow17Params = (Snow17ModelParameters)params;

        final String oldStateStr = this.getStringsFromState(); //save the old state before being modified

        final double we = getWaterEquivalentOfSolidSnow();

        double liqw = getLiquidWaterInSnowPack();

        final double plwhc = snow17Params.getPercentLiquidWaterHoldingCap();

        double wlos = 0.0;

        if(liqw > (plwhc * we))
        {
            wlos = liqw - plwhc * we;
            liqw = plwhc * we;
        }

        setLiquidWaterInSnowpack(liqw);

        double sb = getArealWaterEquivBeforeSnow() - wlos;

        setArealWaterEquivBeforeSnow(sb);

        final double aeadj = getSnowArealExtentAdjustment();

        double sbaesc = getAescBeforeSnow();

        final double si = snow17Params.getArealWeUnderFullSnowCover();

        final double twe = we + liqw;

        double ai = getMaxWeSinceSnow();

//        if(ai > si)
        if(MathHelper.isGreater(ai, si))
        {
            ai = si;
        }

//        if(aeadj > 0.0)
        if(MathHelper.isGreaterThanZero(aeadj))
        {
            ai = aeadj;
        }

//        if(twe < ai)
        if(MathHelper.isLessThan(twe, ai))
        {
//            if(sb > twe)
            if(MathHelper.isGreater(sb, twe))
            {
                sb = twe; //sb could be modified here, but the state sb was set above already
            }
            //dealing with ADC()

            final double[] adc = snow17Params.getAdc();

            double r = (sb / ai) * 10.0 + 1.0;

            final int n = (int)Math.floor(r);
            r -= n;

            sbaesc = adc[n - 1] + (adc[n] - adc[n - 1]) * r;

            if(sbaesc > 1.0)
            {
                sbaesc = 1.0;
            }
            else if(sbaesc < 0.05)
            {
                sbaesc = 0.05;
            }
        } //close if(twe < ai)

        setAescBeforeSnow(sbaesc);

        final double sbws = getWaterEquivalentUnderSnowCover() - wlos;

        setWaterEquivalentUnderSnowCover(sbws);

        //ADC stuff
        final int itpxO = getITPX(); //the MAP interval when the states were saved
        final int itpxN = snow17Params.getPrecipInterval(); //the current(New) MAP interval

        if(itpxO != itpxN)
        {//when MAP interval is changed, modify EXLAG array

            final int nlold = 5 / itpxO + 2;
            final int nlnew = 5 / itpxN + 2;

            final double[] prevExlagArray = getExlagArray();

            final double[] newExlagArray = new double[nlnew];

            final int NHR = nlold * itpxO;

            final double fhr = 1.0 / itpxO;

            for(int N = 0; N < NHR; N++)
            {
                final int NO = (N - 1) / itpxO + 1; // starting from 0,
                final int NN = (N - 1) / itpxN + 1; // starting from 0,

                if(NN >= nlnew)
                {
                    break;
                }

                newExlagArray[NN] += fhr * prevExlagArray[NO];

            }

            // break reaches here
            setExlagArray(newExlagArray);

        } //close if(itpxO != itpxN)

        final String newStateStr = this.getStringsFromState();

        _logger.log(Logger.DEBUG, "Carryover Transfer is successfully finished.");

        if(oldStateStr.equals(newStateStr) == false)
        {//states have been changed by cox

            final StringBuilder strBuilder = new StringBuilder();
            strBuilder.append("Old: " + OHDConstants.NEW_LINE + oldStateStr);
            strBuilder.append(OHDConstants.NEW_LINE);
            strBuilder.append("New: " + OHDConstants.NEW_LINE + newStateStr);
            _logger.log(Logger.DEBUG, strBuilder.toString());

            //print out to console for visual inspection
//            System.out.println(strBuilder.toString());
        }
        else
        {//states were not modified
            _logger.log(Logger.DEBUG, "No states have been changed by Carryover Transfer.");
        }

    } //close doCarryOver()

    public double getPqnet()
    {
        return _pqnet;
    }

    void setPqnet(final double pqnet)
    {
        _pqnet = pqnet;
    }

    public String toStringFromStates()
    {
        final StringBuilder resultStr = new StringBuilder();

        resultStr.append("Snow17ModelStates:" + OHDConstants.NEW_LINE);
        resultStr.append("WE=" + _we).append(OHDConstants.NEW_LINE);
        resultStr.append("NEGHS=" + _neghs).append(OHDConstants.NEW_LINE);
        resultStr.append("LIQW=" + _liqw).append(OHDConstants.NEW_LINE);
        resultStr.append("TINDEX=" + _tindex).append(OHDConstants.NEW_LINE);
        resultStr.append("ACCMAX=" + _accmax).append(OHDConstants.NEW_LINE);
        resultStr.append("SB=" + _sb).append(OHDConstants.NEW_LINE);
        resultStr.append("SBAESC=" + _sbaesc).append(OHDConstants.NEW_LINE);
        resultStr.append("SBWS=" + _sbws).append(OHDConstants.NEW_LINE);
        resultStr.append("AEADJ=" + _aeadj).append(OHDConstants.NEW_LINE);
        resultStr.append("STORGE=" + _storge).append(OHDConstants.NEW_LINE);
        resultStr.append("SNDPT=" + _sndpt).append(OHDConstants.NEW_LINE);
        resultStr.append("SNTMP=" + _sntmp).append(OHDConstants.NEW_LINE);
        resultStr.append("EXLAG=" + Arrays.toString(_exlag)).append(OHDConstants.NEW_LINE);

        return resultStr.toString();
    }

	public void modifyStatesForDifferentTimestep(
			final int modelInterval) throws Exception {
				      
	     //need to re-compute exlag array
	
	        final int nlold = 5 / getITPX() + 2;
	        final int nlnew = 5 / modelInterval + 2;
	
	        final int nhr = nlold * getITPX();
	
	        final double fhr = 1.0 / getITPX();
	
	        final double[] exlag = new double[nlnew];
	
	        for(int n = 0; n < nhr; n++)
	        {
	            final int no = n / getITPX();
	            final int nn = n / modelInterval;
	
	            if(nn >= nlnew)
	            {
	                break;
	            }
	
	            exlag[nn] += fhr * getExlagArray()[no];
	        }
	
	        //break reaches here
	        setExlagArray(exlag);
	
	        setITPX(modelInterval);         
	}

} // close class
