package ohd.hseb.ohdmodels.ssarresv;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import ohd.hseb.db.DbTimeHelper;
import ohd.hseb.measurement.AbsTimeMeasurement;
import ohd.hseb.measurement.Measurement;
import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.SortedProperties;
import ohd.hseb.util.fews.NwsrfsDataTypeMappingReader;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.State;
import ohd.hseb.util.fews.ohdmodels.LegacyModelAdapter;
import ohd.hseb.util.fews.ohdmodels.ModValues;
import ohd.hseb.util.fews.ohdmodels.ModelDriver;

/**
 * Created: March, 2008
 * 
 * @author CHPS SSARRESV team
 * @since SSARRESV
 * @see ModelDriver
 * @version 1.0
 */
final public class SsarresvModelDriver extends ModelDriver
{

    private static final String _SSARESV_EXE_NAME = "ssarresv";
    private static String _MOD_NAME = "SSARREG";

    private final List<RegularTimeSeries> _modsTsList;

    private boolean _printLog = false;
    //MOD TSs: 7 types
    private static final String[] _SSARRESV_MOD_TYPES = new String[]{"FREEFLOW", "SETQ", "SETH", "SETS", "SETDQ",
        "SETDH", "SETDS"};

    public SsarresvModelDriver()
    {
        super(); //calling ModelDriver constructor

        _state = new SsarresvModelState();
        _parameters = new SsarresvModelParameters();
        _parameters.setLogger(_logger);
        _savedParameters = new SsarresvModelParameters();
        _savedParameters.setLogger(_logger);

        _modsTsList = new ArrayList<RegularTimeSeries>();
    }

    @Override
    public void execute() throws Exception
    {

        if(_logger.getPrintDebugInfo() > 0)
        {
            _printLog = true;
        }
        final LegacyModelAdapter ssarresvAdapter = new LegacyModelAdapter(this, getWorkDir(), _logger);

        ssarresvAdapter.writeStandAloneFiles();

        //ssarresv needs an extra argument - mod file name, 
        //fill in a String [] array and send to executeModel as last argument
        final SortedProperties properties = new SortedProperties();
        properties.put(LegacyModelAdapter.MOD_FILE_NAME, getWorkDir() + "/" + LegacyModelAdapter.MODS_TXT_FILE);

        //    final String extraArgs[] = new String[]{getWorkDir() + "/" + LegacyModelAdapter.MODS_FILENAME};

        // write ssarresv mods if any to temp file
        ssarresvAdapter.writeModsToFile(_modsTsList);

        // TO DO: implement approach below
        //ssarresvAdapter.writeModsToFileUsingModValues(super.getInputModsListNew());

        //Clear out state Map before storing the new data (bug 244)
        ((State)_state).getStateMap().clear();

        //execute model
        ssarresvAdapter.executeModel(getDriverProperties().getProperty(OHDConstants.LEGACY_LOCATION_DIR) + "/"
            + _SSARESV_EXE_NAME, properties);

        //load output states from temp file
        getState().loadState(ssarresvAdapter.getOutputStateFileName(), _logger);

        //load the results from output temp file
        final List<RegularTimeSeries> resultsList = ssarresvAdapter.readOutputFlatFilesToLists();

        // qualifier id is part of unique key NOT location id
        // this is because ssarresv can have multiple outputs with same type

        for(int i = 0; i < resultsList.size(); i++)
        {
            addTStoResultMap(resultsList.get(i));
        }

        getState().setDateTime(getComputationEndTime());
    }

    /**
     * Overriding the method. Be called in {@link LegacyModelAdapter#LegacyModelAdapter(ModelDriver, String, Logger)}
     */
    @Override
    final protected void runModelDriverValidation() throws Exception
    {
        final Iterator<RegularTimeSeries> ite = getTsList().iterator();

        RegularTimeSeries inputRts;
        String tsType;
        boolean isModTs;

        while(ite.hasNext())
        {
            isModTs = false;
            inputRts = ite.next();

            tsType = inputRts.getTimeSeriesType();

            setDrivingTimeseriesUsingNonModInputs(ite, inputRts, tsType, isModTs);

            // added to address fogBugz 1450
            // extend "observed" timeseries which are not required to have data to convert entire period

            if(NwsrfsDataTypeMappingReader.areMissingValuesAllowed(tsType, _logger))
            {
                final double missingValue = inputRts.getMissingValue();
                inputRts.extendTimeSeries(super.getComputationEndTime(), missingValue);
            }

        } //close while loop

        if(_drivingTs == null)
        {
            if(_printLog)
            {
                _logger.log(Logger.DEBUG, "Driving Time Series is null...");
            }
        }

        super.runModelDriverValidation();
        if(_modsTsList.size() > 0)
        {
            super.checkInputTsInSync(_modsTsList);
        }
    }

    private void setDrivingTimeseriesUsingNonModInputs(final Iterator<RegularTimeSeries> ite,
                                                       final RegularTimeSeries inputRts,
                                                       final String tsType,
                                                       boolean isModTs)
    {
        //fish out MOD TSs from _tsList and put them in _modsTsList
        for(final String type: _SSARRESV_MOD_TYPES)
        {
            if(tsType.equalsIgnoreCase(type))
            {
                _modsTsList.add(inputRts.clone());

                ite.remove();

                isModTs = true;

                break; //jump out of loop
            }
        }

        //set _drivingTs to any non-Mod TS, since all input ts have the same time step. Any ts is a driver.
        if(_drivingTs == null && isModTs == false)
        {
            _drivingTs = inputRts;

            if(_printLog)
            {
                _logger.log(Logger.DEBUG,
                            "Driving Time Series set to " + _drivingTs.getLocationId() + " "
                                + _drivingTs.getTimeSeriesType() + " " + _drivingTs.getIntervalInHours());
            }
        }
    }

    @Override
    /*
     * The following method is used to create a SSARRESV .txt mod file each value in the inputModList time series is
     * converted into a line in the text file For example: SSARREG_DS_5_817416=0.0 In the code below, numIdentifiers is
     * the Number of fields separated by underscore not including the mod name and date as julian hr in the above line
     * numIdentifiers is 2 which are DS and '5' and it does not include SSARREG and 817416
     */
    public String getModsStringFromTS(final List<RegularTimeSeries> inputModsList) throws Exception
    {
        final int numIdentifiers = 2;
        final long effDate = new DateTime(_runInfo.getRunLastObservationTimeLong(), getStartHourOfDay()).convertToJulianHours();
        //final String header = numIdentifiers + " " + Long.toString(effDate) + "\n";
        final StringBuffer header = new StringBuffer();
        header.append(numIdentifiers).append(" ").append(effDate).append("\n");

        final List<SsarresvModEntry> allSsarresvMods = new ArrayList<SsarresvModEntry>();

        for(final RegularTimeSeries ts: inputModsList)
        {

            createEntriesInModFileForOneTimeseriesType(_MOD_NAME, allSsarresvMods, ts);
        }

        Collections.sort(allSsarresvMods);

        handleModsWithSameJulianHour(allSsarresvMods);

        //String modsString = "";
        final StringBuffer modsString = new StringBuffer("");

        // when all values are missing return an empty string
        // NO MODS
        if(allSsarresvMods.size() == 0)
        {
            //return modsString;
            return modsString.toString();
        }

        //modsString = header;
        modsString.append(header);
        for(int i = 0; i < allSsarresvMods.size(); i++)
        {
            //modsString = modsString + allSsarresvMods.get(i).getModLine();
            modsString.append(allSsarresvMods.get(i).getModLine());
        }

        //System.out.println("ModsString   = " + modsString);
        //System.out.println("ModsString2  = " + modsString2);

        return modsString.toString();
    }

    @Override
    /*
     * The following method is used to create a SSARRESV .txt mod file each value in the inputModList time series is
     * converted into a line in the text file For example: SSARREG_DS_5_817416=0.0 In the code below, numIdentifiers is
     * the Number of fields separated by underscore not including the mod name and date as julian hr in the above line
     * numIdentifiers is 2 which are DS and '5' and it does not include SSARREG and 817416
     */
    public String getModsStringFromModValues(final List<ModValues> inputModsList) throws Exception
    {
        final int numIdentifiers = 2;
        final long effDate = new DateTime(_runInfo.getRunStartTimeLong(), getStartHourOfDay()).convertToJulianHours();
        //final String header = numIdentifiers + " " + Long.toString(effDate) + "\n";
        final StringBuffer header = new StringBuffer();
        header.append(numIdentifiers).append(" ").append(effDate).append("\n");

        final List<SsarresvModEntry> allSsarresvMods = new ArrayList<SsarresvModEntry>();

        for(final ModValues oneSsarresvModType: inputModsList)
        {

            createEntriesInModFileForOneModType(_MOD_NAME, allSsarresvMods, oneSsarresvModType);
        }

        Collections.sort(allSsarresvMods);

        handleModsWithSameJulianHour(allSsarresvMods);

        //String modsString = header;
        final StringBuffer modsString = new StringBuffer(header);
        for(int i = 0; i < allSsarresvMods.size(); i++)
        {
            //modsString = modsString + allSsarresvMods.get(i).getModLine();
            modsString.append(allSsarresvMods.get(i).getModLine());
        }

        //System.out.println("ModsString   = " + modsString);
        //System.out.println("ModsString2  = " + modsString2);
        //return modsString;
        return modsString.toString();
    }

    private void createEntriesInModFileForOneModType(final String modName,
                                                     final List<SsarresvModEntry> ssarresvMods,
                                                     final ModValues mods) throws Exception
    {
        StringBuffer oneModLine;
        double modValue;
        DateTime sdt;

        int dateAsInteger;
        SsarresvModEntry modEntry;
        final HashMap<Long, Measurement> modsValuesMap = mods.getModValuesMap();
        final SsarresvRegulationTypes regType = SsarresvRegulationTypes.getRegulationUsingName(mods.getTimeSeriesType());
        final int regulationCode = regType.getReguationNumber();
        final MeasuringUnit regulationUnit = regType.getRegulationUnit();

        for(final Long date: modsValuesMap.keySet())
        {
            oneModLine = new StringBuffer();
            oneModLine.append(modName).append("_").append(mods.getQualifierId()).append("_").append(regulationCode);
            modValue = modsValuesMap.get(date).getValue(regulationUnit);

            sdt = new DateTime(date, getStartHourOfDay());
            dateAsInteger = sdt.convertToJulianHours();
            oneModLine.append("_").append(dateAsInteger).append("=").append(modValue).append("\n");

            modEntry = new SsarresvModEntry(oneModLine.toString(), dateAsInteger, regulationCode);
            ssarresvMods.add(modEntry);
        }
    }

    private void createEntriesInModFileForOneTimeseriesType(final String modName,
                                                            final List<SsarresvModEntry> myStringsCollection,
                                                            final RegularTimeSeries ts) throws Exception
    {
        StringBuffer oneModLine;
        Double modValue;
        long time;
        DateTime sdt;

        int dateAsInteger;
        SsarresvModEntry modEntry;
        final int measureMentCount = ts.getMeasurementCount();

        final SsarresvRegulationTypes regType = SsarresvRegulationTypes.getRegulationUsingName(ts.getTimeSeriesType());
        final int regulationCode = regType.getReguationNumber();
        final MeasuringUnit regulationUnit = regType.getRegulationUnit();

        for(int index = 0; index < measureMentCount; index++)
        {
            //oneModLine = modName + "_" + ts.getQualifierId() + "_" + regulationCode;
            oneModLine = new StringBuffer();
            oneModLine.append(modName).append("_").append(ts.getQualifierIds().get(0)).append("_").append(regulationCode);

            modValue = ts.getMeasurementValueByIndex(index, regulationUnit);

            if(modValue == ts.getMissingValue())
            {
                continue;
            }

            final AbsTimeMeasurement absTime = ts.getAbsTimeMeasurementByIndex(index);
            time = absTime.getTime();
            sdt = new DateTime(time, getStartHourOfDay());

            dateAsInteger = sdt.convertToJulianHours();
            //oneModLine = oneModLine + "_" + dateAsInteger + "=" + modValue + "\n";
            oneModLine.append("_").append(dateAsInteger).append("=").append(modValue).append("\n");

            //modEntry = new SsarresvModEntry(oneModLine, dateAsInteger, regulationCode);
            modEntry = new SsarresvModEntry(oneModLine.toString(), dateAsInteger, regulationCode);
            myStringsCollection.add(modEntry);
        }

    }

    private void handleModsWithSameJulianHour(final List<SsarresvModEntry> ssarresvMods)
    {

        for(int i = 2; i < ssarresvMods.size(); i++)
        {
            final SsarresvModEntry tieBreakerEntry = ssarresvMods.get(i - 2);
            final SsarresvModEntry previousEntry = ssarresvMods.get(i - 1);
            final SsarresvModEntry currentEntry = ssarresvMods.get(i);

            if(currentEntry.getJulianHour() == previousEntry.getJulianHour())
            {
                final long time = DateTime.convertFromJulianHoursRelativeTo12Z(currentEntry.getJulianHour())
                                          .getTimeInMillis();
                if(_printLog)
                {
                    _logger.log(Logger.DEBUG,
                                "found two ssarresv mods with matching dates on "
                                    + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE) + "i = " + i + " count = "
                                    + ssarresvMods.size());
                }

                if(previousEntry.getRegulationCode() == tieBreakerEntry.getRegulationCode())
                {
                    // original sort is ok; don't need to re-sort
                }
                else if(currentEntry.getRegulationCode() == tieBreakerEntry.getRegulationCode())
                {
                    ssarresvMods.set(i - 1, currentEntry);
                    ssarresvMods.set(i, previousEntry);
                }
                else
                {
                    if(_printLog)
                    {
                        _logger.log(Logger.DEBUG,
                                    "Could not determine proper order for two ssarresv mods with matching dates on "
                                        + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE) + ".Entries removed");
                    }

                    // this is tricky; must remove current one (i) before previous one (i-1)
                    ssarresvMods.remove(i);
                    ssarresvMods.remove(i - 1);
                }

            }

        }

    }

    @Override
    public SsarresvModelParameters getParameters()
    {
        return (SsarresvModelParameters)_parameters;
    }

    @Override
    public SsarresvModelParameters getPreviousParameters()
    {
        return (SsarresvModelParameters)_savedParameters;
    }

    @Override
    public SsarresvModelState getState()
    {
        return (SsarresvModelState)_state;
    }

}
