package ohd.hseb.ohdmodels.statq;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Properties;

import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.MeasuringUnitType;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.SortedProperties;
import ohd.hseb.util.fews.NwsrfsDataFileExtractor;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.ParameterType.Row;
import ohd.hseb.util.fews.ParameterType.Table;
import ohd.hseb.util.fews.ohdmodels.LegacyModelAdapter;
import ohd.hseb.util.fews.ohdutilities.UtilityDriver;

final public class StatQDriver extends UtilityDriver
{

    private static final String STATQ_EXE_NAME = "statq";
    private static final String ARGUMENTS_FILENAME = "arguments.txt";;

    public StatQDriver()
    {
        super(); //calling ModelDriver constructor

        _state = new StatQModelState();
        _parameters = new StatQModelParameters();
        _runInfo = super.getRunInfo();
    }

    @Override
    public StatQModelParameters getParameters()
    {
        return (StatQModelParameters)_parameters;
    }

    @Override
    public StatQModelState getState()
    {
        return (StatQModelState)_state;
    }

    @Override
    public void execute() throws Exception
    {
        this.runLocalDriverValidation();

        // Even this is not a model per se. We are using some of the methods of LegacyModelAdpater to pass information to the STAT-Q legacy code.
        final LegacyModelAdapter statQAdapter = new LegacyModelAdapter(getWorkDir(), _logger);

        if(!this.getParameters().isParamExisting(OHDConstants.OPERATION_CONTENTS_TAG))
        {
            this.getParameters().setOperationContentFromParameters();
        }

        // Create the text files from the XML files.
        // Write the parameter information
        statQAdapter.writeParamsToFile(this.getParameters(), StatQModelConstants.PARAMS_FILENAME);
        // set information required for the time series from the run info 
        statQAdapter.setComputationEndTime(_runInfo.getRunEndTimeLong());
        statQAdapter.setInitialStateTime(_runInfo.getRunStartTimeLong());
        // Write the time series information
        statQAdapter.writeRegularTimeSeriesDataToFlatFile(_tsList);

        final SortedProperties properties = new SortedProperties();
        this.createModelProperties(properties);
        //if there are any model specific arguments to be sent, 
        //fill in a String [] array and send to executeModel as last argument

        final String[] arguments = {getWorkDir() + File.separator + ARGUMENTS_FILENAME};

        // write out properties to arguments.txt file
        final FileOutputStream argsFile = new FileOutputStream(getWorkDir() + File.separator + ARGUMENTS_FILENAME);
        properties.store(argsFile, null);
        argsFile.close();

        statQAdapter.executeLegacy(getDriverProperties().getProperty(OHDConstants.LEGACY_LOCATION_DIR) + File.separator
            + STATQ_EXE_NAME, arguments);
        // Copy the legacy logs inputs to CHPS logs. So far they are appended at the end.
        statQAdapter.readFromFlatFileToDiagnosticsList();

    }

    /**
     * Create the arguments information being passed to the legacy code.
     * 
     * @param properties
     * @throws IOException
     */
    private void createModelProperties(final Properties properties) throws IOException
    {

        final FileOutputStream diagTxtFile = new FileOutputStream(getWorkDir() + File.separator
            + StatQModelConstants.DIAG_FILENAME);
        diagTxtFile.close();

        properties.put(StatQModelConstants.DATA_TYPE_FILENAME_TAG, getWorkDir() + File.separator
            + NwsrfsDataFileExtractor.DATATYPE_MAPPING_FILENAME);
        properties.put(StatQModelConstants.PARAM_FILENAME_TAG, getWorkDir() + File.separator
            + StatQModelConstants.PARAMS_FILENAME);
        properties.put(StatQModelConstants.OUTPUT_TS_FILENAME_TAG, getWorkDir() + File.separator
            + StatQModelConstants.OUTPUT_TS_FILENAME);
        properties.put(StatQModelConstants.HTML_CSS_FILENAME_TAG,
                       super._runInfo.getProperties().getProperty(StatQModelConstants.HTML_CSS_FILENAME_TAG));
        properties.put(StatQModelConstants.OUTPUT_HTML_FILENAME_TAG,
                       super._runInfo.getProperties().getProperty(StatQModelConstants.OUTPUT_HTML_FILENAME_TAG));
        properties.put(StatQModelConstants.TS_FILE_NAME_TAG, getWorkDir() + File.separator
            + StatQModelConstants.INPUT_TS_FILENAME);
        properties.put(StatQModelConstants.DIAG_FILENAME_TAG, getWorkDir() + File.separator
            + StatQModelConstants.DIAG_FILENAME);
        properties.put(StatQModelConstants.DEBUG_FLAG_TAG,
                       super._runInfo.getProperties().getProperty(OHDConstants.PRINT_DEBUG_INFO));

    }

    /**
     * @throws Exception
     */
    final protected void runLocalDriverValidation() throws Exception
    {

        this.validateTimeSeries();
        this.validateParameters();

    }

    /**
     * validate the time series have
     * 
     * @throws Exception
     */
    private void validateTimeSeries() throws Exception
    {
        // must have only 2 timeseries in input.xml 
        if(getTsList().size() != 2)
        {

            throw new Exception("The number of time series must be 2. One observed and one Simulated time series");
        }

        boolean observedFirst = true;

        final RegularTimeSeries inputRts1 = getTsList().get(0).clone();
        final RegularTimeSeries inputRts2 = getTsList().get(1).clone();

        final String observedParameterId = super._runInfo.getProperties()
                                                         .getProperty(StatQModelConstants.PARAMETER_ID_FOR_OBSERVED_TS);
        final String simulatedParameterId = super._runInfo.getProperties()
                                                          .getProperty(StatQModelConstants.PARAMETER_ID_FOR_SIMULATED_TS);

        if(observedParameterId == null || simulatedParameterId == null)
        {
            throw new Exception("Both " + StatQModelConstants.PARAMETER_ID_FOR_OBSERVED_TS + " and "
                + StatQModelConstants.PARAMETER_ID_FOR_SIMULATED_TS + "are required properties");

        }

        if(observedParameterId.equals(inputRts1.getTimeSeriesType()))
        {
            observedFirst = true;
        }
        else
        {
            if(observedParameterId.equals(inputRts2.getTimeSeriesType()))
            {
                observedFirst = false;
            }
            else
            {
                throw new Exception("It must be 2 timeseries. One observed and one Simulated time series");
            }
        }
        if(simulatedParameterId.equals(inputRts2.getTimeSeriesType()))
        {
            observedFirst = true;
        }
        else
        {
            if(simulatedParameterId.equals(inputRts1.getTimeSeriesType()))
            {
                observedFirst = false;
            }
            else
            {
                throw new Exception("It must be 2 timeseries. One observed and one Simulated time series");
            }
        }

        // We need the first timeseries in list to be the observed timeseries and the 2nd timeseries as the simulated timeseries.
        if(!observedFirst)
        {
            getTsList().clear();
            getTsList().add(inputRts2);
            getTsList().add(inputRts1);
        }

        // simulated and observed time series must have same units
        if(!inputRts1.getMeasuringUnit().getName().equals(inputRts2.getMeasuringUnit().getName()))
        {
            throw new Exception("The simulated "
                + simulatedParameterId
                + " and observed "
                + observedParameterId
                + " time series must have same units. Please check the workflow property at XXXXX_Stats_calibration.xml");
        }

        //if discharge data is present. The timeseries must have units of CMS
        if(inputRts1.getMeasuringUnit().getType() == MeasuringUnitType.discharge
            && inputRts1.getMeasuringUnit().getName() != MeasuringUnit.cms.getName())
        {
            throw new Exception("Discharge data must have units of " + MeasuringUnit.cms.getName());
        }
        //if discharge data is present. The timeseries must have units of CMS
        if(inputRts2.getMeasuringUnit().getType() == MeasuringUnitType.discharge
            && inputRts2.getMeasuringUnit().getName() != MeasuringUnit.cms.getName())
        {
            throw new Exception("Discharge data must have units of " + MeasuringUnit.cms.getName());
        }
        //if rainfall data is present. The timeseries must have units of MM
        if(inputRts1.getMeasuringUnit().getType() == MeasuringUnitType.length
            && inputRts1.getMeasuringUnit().getName() != MeasuringUnit.mm.getName())
        {
            throw new Exception("Rainfall data must have units in " + MeasuringUnit.mm.getName());
        }
        //if rainfall data is present. The timeseries must have units of MM
        if(inputRts2.getMeasuringUnit().getType() == MeasuringUnitType.length
            && inputRts2.getMeasuringUnit().getName() != MeasuringUnit.mm.getName())
        {
            throw new Exception("Rainfall data must have units in " + MeasuringUnit.mm.getName());
        }
    }

    /**
     * @throws Exception
     */
    private void validateParameters() throws Exception
    {
        /**
         * Parameter validation
         */

        final RegularTimeSeries inputRts1 = getTsList().get(0);

        if(!this.getParameters().isParamExisting(OHDConstants.OPERATION_CONTENTS_TAG))
        {
            if(this.getParameters().getIntDataParameter(StatQModelParameters.PROCESSING_TIMESTEP) >= inputRts1.getIntervalInHours())
            {
                final int processingTimeStep = this.getParameters()
                                                   .getIntDataParameter(StatQModelParameters.PROCESSING_TIMESTEP);
                final int intervalInHours = inputRts1.getIntervalInHours();
                if(processingTimeStep % intervalInHours != 0)
                {
                    throw new Exception("The parameter " + StatQModelParameters.PROCESSING_TIMESTEP
                        + " must be divisible by the timeseries interval in hours");
                }
            }
            else
            {
                throw new Exception("The parameter " + StatQModelParameters.PROCESSING_TIMESTEP
                    + " must be greater or equal to the timeseries interval in hours");
            }

            int startMonth = 0;
            int startYear = 0;

            if(this.getParameters().isParamExisting(StatQModelParameters.START_MONTH_YEAR))
            {
                final String startDate = this.getParameters()
                                             .getStringDataParameter(StatQModelParameters.START_MONTH_YEAR);
                startMonth = Integer.parseInt(startDate.substring(0, 2));
                startYear = Integer.parseInt(startDate.substring(2));

                Calendar calendar = new GregorianCalendar(startYear, startMonth - 1, 0);
                calendar.set(Calendar.DAY_OF_MONTH, calendar.getActualMinimum(Calendar.DAY_OF_MONTH));
                final int firstDayOfMonth = calendar.getActualMinimum(Calendar.DAY_OF_MONTH);
                final DateTime startDateParams = new DateTime(startYear, startMonth, firstDayOfMonth, 01);

                DateTime startDateRunInfo = new DateTime(super.getRunInfo().getRunStartTimeLong());

                if(startDateRunInfo.getTimeInMillis() > startDateParams.getTimeInMillis())
                {
                    throw new Exception("The start date from the control file " + StatQModelParameters.START_MONTH_YEAR
                        + " " + startDateParams.toString(OHDConstants.DATE_TIME_FORMAT_STR)
                        + " must be greater or equal to the timeseries start Date time "
                        + startDateRunInfo.toString(OHDConstants.DATE_TIME_FORMAT_STR)
                        + " Check parameter file in ModuleParFiles directory");
                }

                //This part is optional as there is more data than needed by the timeseries.
                if(startDateRunInfo.dayOfMonth() > 1)
                {
                    _logger.log(Logger.WARNING,
                                "StartDate must be the first of the month, not " + startDateRunInfo.dayOfMonth()
                                    + " moving date to first of next month.");
                    calendar = new GregorianCalendar(startDateRunInfo.year(),
                                                     startDateRunInfo.month(),
                                                     startDateRunInfo.dayOfMonth());
                    calendar.set(Calendar.DAY_OF_MONTH, calendar.getActualMaximum(Calendar.DAY_OF_MONTH));
                    //int lastDayOfMonth = calendar.getActualMaximum(Calendar.DAY_OF_MONTH);
                    startDateRunInfo = startDateRunInfo.addDaysTo(1 + calendar.getActualMaximum(Calendar.DAY_OF_MONTH)
                        - startDateRunInfo.dayOfMonth());
                    if(startDateRunInfo.getTimeInMillis() > startDateParams.getTimeInMillis())
                    {
                        throw new Exception("The start date from the control file "
                            + StatQModelParameters.START_MONTH_YEAR + " "
                            + startDateParams.toString(OHDConstants.DATE_TIME_FORMAT_STR)
                            + " must be greater or equal to the timeseries start Date time "
                            + startDateRunInfo.toString(OHDConstants.DATE_TIME_FORMAT_STR)
                            + " Check parameter file in ModuleParFiles directory");
                    }
                    else
                    {
                        super.getRunInfo().setRunStartTimeLong(startDateRunInfo);
                    }
                }

            }
            else
            {
                DateTime startDateRunInfo = new DateTime(super.getRunInfo().getRunStartTimeLong());
                if(startDateRunInfo.dayOfMonth() > 1)
                {
                    _logger.log(Logger.WARNING,
                                "StartDate " + startDateRunInfo.toString(OHDConstants.DATE_TIME_FORMAT_STR)
                                    + " must be the first day of the month, not " + startDateRunInfo.dayOfMonth()
                                    + " moving date to first of next month.");
                    final Calendar calendar = new GregorianCalendar(startDateRunInfo.year(),
                                                                    startDateRunInfo.month() - 1,
                                                                    startDateRunInfo.dayOfMonth());
                    calendar.set(Calendar.DAY_OF_MONTH, calendar.getActualMaximum(Calendar.DAY_OF_MONTH));
                    final int lastDayOfMonth = calendar.getActualMaximum(Calendar.DAY_OF_MONTH);
                    startDateRunInfo = startDateRunInfo.addDaysTo(1 + lastDayOfMonth - startDateRunInfo.dayOfMonth());
                    super.getRunInfo().setRunStartTimeLong(startDateRunInfo);
                    _logger.log(Logger.WARNING,
                                "StartDate moved to " + startDateRunInfo.toString(OHDConstants.DATE_TIME_FORMAT_STR));
                }
                startMonth = startDateRunInfo.month();
                startYear = startDateRunInfo.year();
                final StringBuffer startMonthYear = new StringBuffer();
                if(startMonth < 10)
                    startMonthYear.append("0");
                startMonthYear.append(startMonth);
                startMonthYear.append(startYear);
                this.getParameters().insertParameter(StatQModelParameters.START_MONTH_YEAR, startMonthYear.toString());
            }

            int endMonth = 0;
            int endYear = 0;

            if(this.getParameters().isParamExisting(StatQModelParameters.END_MONTH_YEAR))
            {
                final String endDate = this.getParameters().getStringDataParameter(StatQModelParameters.END_MONTH_YEAR);
                endMonth = Integer.parseInt(endDate.substring(0, 2));
                endYear = Integer.parseInt(endDate.substring(2));

                final Calendar calendar = new GregorianCalendar(endYear, (endMonth - 1), 1);
                calendar.set(Calendar.DAY_OF_MONTH, calendar.getActualMaximum(Calendar.DAY_OF_MONTH));
                final int lastDayOfMonth = calendar.getActualMaximum(Calendar.DAY_OF_MONTH);
                final DateTime lastDateParams = new DateTime(endYear, endMonth, lastDayOfMonth, 23);
                final DateTime endDateRunInfo = new DateTime(super.getRunInfo().getRunEndTimeLong());

                if(endDateRunInfo.getTimeInMillis() < lastDateParams.getTimeInMillis())
                {
                    throw new Exception("The end date from the control file " + StatQModelParameters.END_MONTH_YEAR
                        + " " + lastDateParams.toString(OHDConstants.DATE_TIME_FORMAT_STR)
                        + " must be less or equal to the timeseries End Date time "
                        + endDateRunInfo.toString(OHDConstants.DATE_TIME_FORMAT_STR));
                }
            }
            else
            {
                DateTime endDateRunInfo = new DateTime(super.getRunInfo().getRunEndTimeLong());

                final Calendar calendar = new GregorianCalendar(endDateRunInfo.year(),
                                                                endDateRunInfo.month() - 1,
                                                                endDateRunInfo.dayOfMonth());

                if(calendar.getActualMaximum(Calendar.DAY_OF_MONTH) != endDateRunInfo.dayOfMonth())
                {
                    endDateRunInfo = endDateRunInfo.addHoursTo(-getTsList().get(0).getIntervalInHours());
                }

                super.getRunInfo().setRunEndTimeLong(endDateRunInfo);
                endMonth = endDateRunInfo.month();
                endYear = endDateRunInfo.year();

                final StringBuffer endMonthYear = new StringBuffer();
                if(endMonth < 10)
                    endMonthYear.append("0");
                endMonthYear.append(endMonth);
                endMonthYear.append(endYear);
                this.getParameters().insertParameter(StatQModelParameters.END_MONTH_YEAR, endMonthYear.toString());
            }

            if(startYear > endYear)
            {
                throw new Exception("The date Parameter " + StatQModelParameters.START_MONTH_YEAR
                    + " is bigger than date parameter " + StatQModelParameters.END_MONTH_YEAR);
            }
            else
            {
                if(startYear == endYear)
                    if(startMonth > endMonth)
                    {
                        throw new Exception("The date Parameter " + StatQModelParameters.START_MONTH_YEAR
                            + " is bigger than date parameter " + StatQModelParameters.END_MONTH_YEAR);
                    }
            }

            //  http://docs.oracle.com/javase/7/docs/api/java/util/GregorianCalendar.html#getActualMaximum%28int%29

            if(this.getParameters().isParamExisting(StatQModelParameters.PAIR_OF_FLOW_INTERVALS))
            {
                final Table pairFlowIntervalTable = this.getParameters()
                                                        .getTableDataParameter(StatQModelParameters.PAIR_OF_FLOW_INTERVALS);
                final List<Row> pairFlowIntervalRowList = pairFlowIntervalTable.getRow();
                for(final Row pairFlowIntervalRow: pairFlowIntervalRowList)
                {
                    if(pairFlowIntervalRow.getA().isEmpty() || pairFlowIntervalRow.getB().isEmpty())
                    {
                        throw new Exception("One of the values of the " + StatQModelParameters.PAIR_OF_FLOW_INTERVALS
                            + " parameter is missing. It must have exactly 2 columns per row");
                    }
                }
            }

            if(this.getParameters().isParamExisting(StatQModelParameters.EVENT_START_AND_END_DATES))
            {
                final Table tableEventStartEndDates = this.getParameters()
                                                          .getTableDataParameter(StatQModelParameters.EVENT_START_AND_END_DATES);

                final DateTime startDateRunInfo = new DateTime(super.getRunInfo().getRunStartTimeLong());
                final DateTime endDateRunInfo = new DateTime(super.getRunInfo().getRunEndTimeLong());

                final List<Row> eventDateOutOfRange = new ArrayList<Row>();
                final List<Row> eventStartEndDatesRowList = tableEventStartEndDates.getRow();
                for(final Row eventstartEndDatesRow: eventStartEndDatesRowList)
                {
                    if((eventstartEndDatesRow.getA() == null || eventstartEndDatesRow.getA().isEmpty())
                        || (eventstartEndDatesRow.getB() == null || eventstartEndDatesRow.getB().isEmpty()))
                    {
                        throw new Exception("One of the values of the "
                            + StatQModelParameters.EVENT_START_AND_END_DATES
                            + " parameter is missing. It must have exactly 2 columns per row");
                    }

                    DateTime eventStartDate = null;
                    try
                    {
                        eventStartDate = DateTime.parseMMDDYYYYHHZ(eventstartEndDatesRow.getA());
                    }
                    catch(final Exception e)
                    {
                        throw new Exception("The value " + eventstartEndDatesRow.getA() + " of the "
                            + StatQModelParameters.EVENT_START_AND_END_DATES
                            + " parameter is wrong. It must have a valid date using format "
                            + OHDConstants.DATE_TIME_FORMAT_STR2);
                    }
                    DateTime eventEndDate = null;
                    try
                    {
                        eventEndDate = DateTime.parseMMDDYYYYHHZ(eventstartEndDatesRow.getB());
                    }
                    catch(final Exception e)
                    {
                        throw new Exception("The value " + eventstartEndDatesRow.getB() + " of the "
                            + StatQModelParameters.EVENT_START_AND_END_DATES
                            + " parameter is wrong. It must have a valid date using format "
                            + OHDConstants.DATE_TIME_FORMAT_STR2);
                    }

                    if(eventStartDate.getTimeInMillis() < startDateRunInfo.getTimeInMillis())
                    {
                        eventDateOutOfRange.add(eventstartEndDatesRow);
                        _logger.log(Logger.WARNING,
                                    "For the parameter " + StatQModelParameters.EVENT_START_AND_END_DATES + " the "
                                        + StatQModelParameters.EVENT_START_DATES + " date "
                                        + eventStartDate.toString(OHDConstants.DATE_TIME_FORMAT_STR)
                                        + " will be removed as it is outside the analysis period. From: "
                                        + startDateRunInfo.toString(OHDConstants.DATE_TIME_FORMAT_STR) + " to: "
                                        + endDateRunInfo.toString(OHDConstants.DATE_TIME_FORMAT_STR));
                    }
                    if(eventEndDate.getTimeInMillis() > endDateRunInfo.getTimeInMillis())
                    {
                        eventDateOutOfRange.add(eventstartEndDatesRow);
                        _logger.log(Logger.WARNING,
                                    "For the parameter " + StatQModelParameters.EVENT_START_AND_END_DATES + " the "
                                        + StatQModelParameters.EVENT_END_DATES + " date "
                                        + eventEndDate.toString(OHDConstants.DATE_TIME_FORMAT_STR)
                                        + " will be removed as it is outside the analysis period. From: "
                                        + startDateRunInfo.toString(OHDConstants.DATE_TIME_FORMAT_STR) + " to: "
                                        + endDateRunInfo.toString(OHDConstants.DATE_TIME_FORMAT_STR));
                    }
                } //end for
                if(eventDateOutOfRange != null)
                {
                    eventStartEndDatesRowList.removeAll(eventDateOutOfRange);
                }
            }

            if(this.getParameters().isParamExisting(StatQModelParameters.PARTITION_NUMBER_MONTHS_FOR_ACCUMULATIVE_FLOW))
            {
                final int partitionNumberMonthsForAccumulativeFlow = this.getParameters()
                                                                         .getIntDataParameter(StatQModelParameters.PARTITION_NUMBER_MONTHS_FOR_ACCUMULATIVE_FLOW);
                if(partitionNumberMonthsForAccumulativeFlow < 1 || partitionNumberMonthsForAccumulativeFlow > 12)
                {
                    throw new Exception("The  Parameter "
                        + StatQModelParameters.PARTITION_NUMBER_MONTHS_FOR_ACCUMULATIVE_FLOW
                        + " must be greather or equal than 1 and less or equal than 12");
                }
            }
        }

    }
} //close method

