package ohd.hseb.ohdmodels.tatum;

import java.util.ArrayList;
import java.util.List;

import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.ohdmodels.ModelDriver;

public class TatumModelDriver extends ModelDriver
{

    private final TatumModelParameters _tatumModelParameters; //alias to super._parameters to save type casting

    private final TatumModelParameters _tatumModelSavedParameters; //alias to super._savedParameters to save type casting

    private final TatumModelState _tatumModelState; //alias to super._state to save type casting

    public TatumModelDriver()
    {
        super(); //calling ModelDriver constructor       

        super._parameters = new TatumModelParameters();
        _tatumModelParameters = (TatumModelParameters)super.getParameters(); //use alias to save casting

        super._savedParameters = new TatumModelParameters();
        _tatumModelSavedParameters = (TatumModelParameters)super._savedParameters; //use alias to save casting

        super._state = new TatumModelState();
        _tatumModelState = (TatumModelState)super._state; //use alias to save casting     

    }

    @Override
    public TatumModelParameters getParameters()
    {
        return _tatumModelParameters;
    }

    @Override
    public TatumModelParameters getPreviousParameters()
    {
        return _tatumModelSavedParameters;
    }

    @Override
    public TatumModelState getState()
    {

        return _tatumModelState;
    }

    @Override
    public void execute() throws Exception
    {

        // all input timeseries have the same time step. Any timeseries is a driver.
        _drivingTs = _tsList.get(0);

        if(_drivingTs == null)
        {
            _logger.log(Logger.FATAL, "Driving Time Series is null.Exiting...");

            throw new Exception();
        }

        _logger.log(Logger.DEBUG,
                    "Driving Time Series set to " + _drivingTs.getLocationId() + " " + _drivingTs.getTimeSeriesType()
                        + " " + _drivingTs.getIntervalInHours());

        try
        {
            runLocalModelDriverValidation();
            addTStoResultMap(this.computeTatum());
        }
        catch(final Exception e)
        {

            final Exception x = e;

            throw x;

        }

        _tatumModelState.setDateTime(getComputationEndTime());

    }

    private RegularTimeSeries computeTatum() throws Exception
    {

        // set this values and we don't need to retrieve several times in loop
        final long drivingTsIntervalInMillis = getDrivingTsIntervalInMillis();
        final long computationStartTime = getComputationStartTime();
        final long computationEndTime = getComputationEndTime();
        double flowin = 0.0;
        double flow = 0.0;
        final int numberOfLayers = _tatumModelParameters.getNumberLayers();
        final double[] upperLimitFlowLayer = _tatumModelParameters.getUpperLimitFlowByLayer();
        final double[] numberCoefficientsByLayer = _tatumModelParameters.getNumberCoefficientsByLayer();
        double[] coefficientsByLayers;
        double[] carryOverByLayers;
        double flowot = 0.0;

        // Output TS
        final RegularTimeSeries tatumOutputTs = new RegularTimeSeries(computationStartTime,
                                                                      computationEndTime,
                                                                      _tatumModelParameters.getOutflowTimeSerieTimeStep(),
                                                                      OHDConstants.DISCHARGE_UNIT);
        tatumOutputTs.setTimeSeriesType(_tatumModelParameters.getOutflowTimeSerieType());
        tatumOutputTs.setLocationId(_drivingTs.getLocationId());

        final List<String> qualifiersIds = new ArrayList<String>();
        qualifiersIds.add(_tatumModelParameters.getOutflowTimeSerieId());
        tatumOutputTs.setQualifierIds(qualifiersIds);

        for(long timeStep = computationStartTime; timeStep <= computationEndTime; timeStep += drivingTsIntervalInMillis)
        {
            flowot = 0.0;
            flowin = _drivingTs.getMeasurementValueByTime(timeStep, OHDConstants.DISCHARGE_UNIT);

            for(int layerNumber = 1; layerNumber <= numberOfLayers; layerNumber++)
            {
                coefficientsByLayers = _tatumModelParameters.getTatumCoefficientsLayer(numberOfLayers - layerNumber + 1);
                carryOverByLayers = _tatumModelState.getTatumCarryOverValuesByLayer(numberOfLayers - layerNumber);

                if(numberOfLayers <= 1)
                {
                    flow = flowin;
                }
                else
                {
                    if(layerNumber <= 1)
                    {
                        if(flowin > upperLimitFlowLayer[0])
                        {
                            flow = upperLimitFlowLayer[0];
                        }
                        if(flowin <= upperLimitFlowLayer[0])
                        {
                            flow = flowin;
                        }
                    }
                    else
                    {
                        if(layerNumber == numberOfLayers)
                        {
                            if(upperLimitFlowLayer[layerNumber - 2] > flowin)
                            {
                                flow = 0;
                            }
                            if(upperLimitFlowLayer[layerNumber - 2] <= flowin)
                            {
                                flow = flowin - upperLimitFlowLayer[layerNumber - 2];
                            }
                        }
                        else
                        {
                            if(upperLimitFlowLayer[layerNumber - 1] < flowin)
                            {
                                flow = upperLimitFlowLayer[layerNumber - 1] - upperLimitFlowLayer[layerNumber - 2];
                            }
                            if(upperLimitFlowLayer[layerNumber - 1] >= flowin
                                && upperLimitFlowLayer[layerNumber - 2] < flowin)
                            {
                                flow = flowin - upperLimitFlowLayer[layerNumber - 2];
                            }
                            if(upperLimitFlowLayer[layerNumber - 2] > flowin)
                            {
                                flow = 0.0;
                            }
                        }
                    }
                }

                flowot = flowot + flow * coefficientsByLayers[0];
                // Apply carryover if any.
                // fixed FB case 984 by RHC, cast the double to integer and # coef layer >= 1
                if(((int)numberCoefficientsByLayer[layerNumber - 1]) >= 1 && carryOverByLayers.length > 0)
                {
                    for(int carryOVerNumber = 0; carryOVerNumber < carryOverByLayers.length; carryOVerNumber++)
                    {
                        flowot = flowot + coefficientsByLayers[carryOVerNumber + 1]
                            * carryOverByLayers[carryOVerNumber];
                    }
                    // Update carry over value
                    // No need check this again, commented out by RHC for FB case 984                    
                    for(int carryOVerNumber = carryOverByLayers.length - 1; carryOVerNumber > 0; carryOVerNumber--)
                    {
                    	carryOverByLayers[carryOVerNumber] = carryOverByLayers[carryOVerNumber - 1];
                    }

                    carryOverByLayers[0] = flow;
                    //update the carry over values
                    _tatumModelState.updateTatumCarryOverValuesByLayer((numberOfLayers - layerNumber),
                                                                       carryOverByLayers);
                }
            }
            //save the value to output ts
            tatumOutputTs.setMeasurementByTime(flowot, timeStep);
        }
        /** --------------------- prepare carryover for storing-------------------- */
        _tatumModelState.setCarryOverValues();

        return tatumOutputTs;
    }

    private void runLocalModelDriverValidation() throws Exception
    {

        //validate TatumModelParamters and TatumModelState
        _tatumModelParameters.validateParams(_drivingTs.getIntervalInHours());

        if(super.needCarryoverTransfer())
        {
            _tatumModelState.validateState(_tatumModelSavedParameters);
            _tatumModelState.doCarryOverTransfer(_tatumModelSavedParameters, _tatumModelParameters);
        }
        else
        {
            _tatumModelState.validateState(_tatumModelParameters);

        }

        super.runModelDriverValidation();
    }
}
