package ohd.hseb.ohdmodels.tatum;

import java.util.ArrayList;
import java.util.List;

import ohd.hseb.measurement.MeasuringUnitType;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.ModelException;
import ohd.hseb.util.fews.NwsrfsDataTypeMappingReader;
import ohd.hseb.util.fews.ohdmodels.LegacyModelParameters;

public class TatumModelParameters extends LegacyModelParameters
{
    // card 1
    public final static String GENERAL_INFO_TAG = "GENERAL_INFO";
    public final static String TS_INFLOW_ID_TAG = "TS_INFLOW_ID";
    public final static String TS_INFLOW_TYPE_TAG = "TS_INFLOW_TYPE";
    public final static String TS_INFLOW_TIMESTEP_TAG = "TS_INFLOW_TIMESTEP";
    public final static String TS_OUTFLOW_ID_TAG = "TS_OUTFLOW_ID";
    public final static String TS_OUTFLOW_TYPE_TAG = "TS_OUTFLOW_TYPE";
    public final static String TS_OUTFLOW_TIMESTEP_TAG = "TS_OUTFLOW_TIMESTEP";
    public final static String NUMBER_OF_LAYERS_TAG = "NUMBER_OF_LAYERS";
    public final static String CARRYOVER_FLAG = "CARRYOVER_FLAG";
    // card 2
    public final static String LAYER_COEFFICIENTS_TAG = "LAYER_COEFFICIENTS";
    // card 3
    public final static String LAYER_UPPER_LIMIT_FLOW_TAG = "LAYER_UPPER_LIMIT_FLOW";
    // card 4
    public final static String LAYER_CARRYOVER_TAG = "LAYER_CARRYOVER";

    public final static String[] LAYER_DATA_TAG = {"LAYER_COEFFICIENTS", "LAYER_UPPER_LIMIT_FLOW"};
    public final static int COEFFICIENT_INDEX = 0;
    public final static int UPPER_LIMIT_INDEX = 1;
    public final static int CARRYOVER_INDEX = 1;

    public final static String[] EACH_LAYER_DATA = {"LAYER_COEFFICIENTS_", "LAYER_CARRYOVER_"};

    private String generalInfo;
    private String inflowTimeSerieId;
    private String inflowTimeSerieType;
    private int inflowTimeSerieTimeStep;
    private String outflowTimeSerieId;
    private String outflowTimeSerieType;
    private int outflowTimeSerieTimeStep;
    private int numberLayers;
    private boolean carryOver;
    private double[] numberCoefficientsByLayer;
    private double[] upperLimitFlowByLayer;
    private final List<double[]> coefficientsByLayerList = new ArrayList<double[]>();
    double[] limitFlowByLayer;

    @Override
    public void extractValuesFromMap() throws Exception
    {

        generalInfo = getStringDataParameter(GENERAL_INFO_TAG);
        inflowTimeSerieId = getStringDataParameter(TS_INFLOW_ID_TAG);
        inflowTimeSerieType = getStringDataParameter(TS_INFLOW_TYPE_TAG);
        inflowTimeSerieTimeStep = getIntDataParameter(TS_INFLOW_TIMESTEP_TAG);
        outflowTimeSerieId = getStringDataParameter(TS_OUTFLOW_ID_TAG);
        outflowTimeSerieType = getStringDataParameter(TS_OUTFLOW_TYPE_TAG);
        outflowTimeSerieTimeStep = getIntDataParameter(TS_OUTFLOW_TIMESTEP_TAG);
        numberLayers = getIntDataParameter(NUMBER_OF_LAYERS_TAG);
        carryOver = getIntDataParameter(CARRYOVER_FLAG) == 1 ? true : false;
        numberCoefficientsByLayer = getDoubleArrayParameter(LAYER_COEFFICIENTS_TAG);
        if(isParamExisting(LAYER_UPPER_LIMIT_FLOW_TAG))
        {
            limitFlowByLayer = getDoubleArrayParameter(LAYER_UPPER_LIMIT_FLOW_TAG);

            //invert values as they are in the wrong order in the params.xml file
            upperLimitFlowByLayer = new double[limitFlowByLayer.length];
            for(int i = 0; i < limitFlowByLayer.length; i++)
            {
                upperLimitFlowByLayer[(limitFlowByLayer.length - 1) - i] = limitFlowByLayer[i];
            }
        }
        if(numberLayers > 0)
        {
            for(int layerId = 0; layerId < numberLayers; layerId++)
                coefficientsByLayerList.add(this.getTatumCoefficientsLayer(layerId + 1));
        }

    }

    /**
     * @return the generalInfo
     */
    public String getGeneralInfo()
    {
        return generalInfo;
    }

    /**
     * @param generalInfo the generalInfo to set
     */
    public void setGeneralInfo(final String generalInfo)
    {
        this.generalInfo = generalInfo;
    }

    /**
     * @return the inflowTimeSerieId
     */
    public String getInflowTimeSerieId()
    {
        return inflowTimeSerieId;
    }

    /**
     * @param inflowTimeSerieId the inflowTimeSerieId to set
     */
    public void setInflowTimeSerieId(final String inflowTimeSerieId)
    {
        this.inflowTimeSerieId = inflowTimeSerieId;
    }

    /**
     * @return the inflowTimeSerieType
     */
    public String getInflowTimeSerieType()
    {
        return inflowTimeSerieType;
    }

    /**
     * @param inflowTimeSerieType the inflowTimeSerieType to set
     */
    public void setInflowTimeSerieType(final String inflowTimeSerieType)
    {
        this.inflowTimeSerieType = inflowTimeSerieType;
    }

    /**
     * @return the inflowTimeSerieTimeStep
     */
    public int getInflowTimeSerieTimeStep()
    {
        return inflowTimeSerieTimeStep;
    }

    /**
     * @param inflowTimeSerieTimeStep the inflowTimeSerieTimeStep to set
     */
    public void setInflowTimeSerieTimeStep(final int inflowTimeSerieTimeStep)
    {
        this.inflowTimeSerieTimeStep = inflowTimeSerieTimeStep;
    }

    /**
     * @return the outflowTimeSerieId
     */
    public String getOutflowTimeSerieId()
    {
        return outflowTimeSerieId;
    }

    /**
     * @param outflowTimeSerieId the outflowTimeSerieId to set
     */
    public void setOutflowTimeSerieId(final String outflowTimeSerieId)
    {
        this.outflowTimeSerieId = outflowTimeSerieId;
    }

    /**
     * @return the outflowTimeSerieType
     */
    public String getOutflowTimeSerieType()
    {
        return outflowTimeSerieType;
    }

    /**
     * @param outflowTimeSerieType the outflowTimeSerieType to set
     */
    public void setOutflowTimeSerieType(final String outflowTimeSerieType)
    {
        this.outflowTimeSerieType = outflowTimeSerieType;
    }

    /**
     * @return the outflowTimeSerieTimeStep
     */
    public int getOutflowTimeSerieTimeStep()
    {
        return outflowTimeSerieTimeStep;
    }

    /**
     * @param outflowTimeSerieTimeStep the outflowTimeSerieTimeStep to set
     */
    public void setOutflowTimeSerieTimeStep(final int outflowTimeSerieTimeStep)
    {
        this.outflowTimeSerieTimeStep = outflowTimeSerieTimeStep;
    }

    /**
     * @return the numberLayers
     */
    public int getNumberLayers()
    {
        return numberLayers;
    }

    /**
     * @param numberLayers the numberLayers to set
     */
    public void setNumberLayers(final int numberLayers)
    {
        this.numberLayers = numberLayers;
    }

    /**
     * @return the carryOver
     */
    public boolean isCarryOver()
    {
        return carryOver;
    }

    /**
     * @param carryOver the carryOver to set
     */
    public void setCarryOver(final boolean carryOver)
    {
        this.carryOver = carryOver;
    }

    /**
     * @return the numberCoefficientsByLayer
     */
    public double[] getNumberCoefficientsByLayer()
    {
        return numberCoefficientsByLayer;
    }

    /**
     * @param numberCoefficientsByLayer the numberCoefficientsByLayer to set
     */
    public void setNumberCoefficientsByLayer(final double[] numberCoefficientsByLayer)
    {
        this.numberCoefficientsByLayer = numberCoefficientsByLayer;
    }

    /**
     * @return the upperLimitFlowByLayer
     */
    public double[] getUpperLimitFlowByLayer()
    {
        return upperLimitFlowByLayer;
    }

    /**
     * @param upperLimitFlowByLayer the upperLimitFlowByLayer to set
     */
    public void setUpperLimitFlowByLayer(final double[] upperLimitFlowByLayer)
    {
        this.upperLimitFlowByLayer = upperLimitFlowByLayer;
    }

    /**
     * @param layerId The Id of the layer number to retrieve
     * @return the tatum coefficients for a Layer
     * @throws Exception
     */
    public double[] getTatumCoefficientsLayer(final int layerId) throws Exception
    {
        return getDoubleArrayParameter(TatumModelParameters.LAYER_COEFFICIENTS_TAG + "_" + layerId);
    }

    /**
     * @param layerId
     * @return
     * @throws Exception
     */
    public double[] getTatumCoefficientsValuesByLayer(final int layerId) throws Exception
    {
        return this.coefficientsByLayerList.get(layerId);

    }

    /**
     * @param layerId
     * @param tatumCarryOverValues
     */
    public void updateTatumCoefficientsValuesByLayer(final int layerId, final double[] tatumCarryOverValues)
    {
        coefficientsByLayerList.set(layerId, tatumCarryOverValues);
    }

    public void validateParams(final int drivingTsInterval) throws ModelException, Exception
    {

        if(numberLayers <= 0)
        {
            _logger.log(Logger.FATAL, "THE NUMBER OF LAYERS IS ZERO");
            throw new ModelException("THE NUMBER OF LAYERS IS ZERO");
        }

        final String dimension = NwsrfsDataTypeMappingReader.getNwsrfsDim(inflowTimeSerieType, _logger);
        if(!dimension.equals(MeasuringUnitType.discharge.getName()))
        {
            _logger.log(Logger.FATAL, "INFLOW DIMENSION " + inflowTimeSerieType + ",IS NOT EQUAL TO "
                + MeasuringUnitType.discharge.getName());
            throw new ModelException("INFLOW DIMENSION " + inflowTimeSerieType + ",IS NOT EQUAL TO "
                + MeasuringUnitType.discharge.getName());
        }
        int layer = 1;
        for(final double[] coefficients: coefficientsByLayerList)
        {
            if(coefficients.length <= 0)
            {
                _logger.log(Logger.FATAL, "THE NUMBER OF TATUM COEFFICIENTS FOR LAYER " + layer + " IS ZERO");
                throw new ModelException("THE NUMBER OF TATUM COEFFICIENTS FOR LAYER " + layer + "IS ZERO");

            }
            else
            {
                double total = 0;
                for(int i = 0; i < coefficients.length; i++)
                {
                    total += coefficients[i];
                }
                if(Math.abs(total - 1.0) > 0.001)
                {
                    _logger.log(Logger.DEBUG, "COEFFICIENTS DO NOT SUM TO 1.0 FOR LAYER," + layer
                        + " A GAIN OR LOSS OCCURS IN THE REACH.");
                }
            }
            layer++;
        }

        if(this.getInflowTimeSerieTimeStep() != this.getOutflowTimeSerieTimeStep())
        {
            _logger.log(Logger.FATAL,
                        "The time interval of inflow timeseries parameter is different from time interval of outflow time series parameter");
            throw new ModelException("The time interval of inflow timeseries parameter is different from time interval of outflow time series parameter");
        }
        if(drivingTsInterval != this.getInflowTimeSerieTimeStep())
        {
            _logger.log(Logger.FATAL,
                        "The time interval of input timeseries is different from time interval in parameters data");
            throw new ModelException("The time interval of inflow time series is different from time interval of inflow time series parameter");
        }

        _logger.log(Logger.DEBUG, "TatumModelParameters has been validated!");
    }
}
