package ohd.hseb.ohdmodels.unithg;

import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.util.fews.OHDConstants;

public class UHGMod extends UHGModelDriver
{
    double _uhgOrdinates[] = null;
    private int _currOrdinates;
    private long _currTimeStep;
    long _startTimeMod = 0;
    long _endTimeMod = 0;
    UHGModelParameters _uhgModelParameters = null;
    RegularTimeSeries _ts = null;

    /**
     * This method computes total discharge with Mod.
     * 
     * @param uhgModelParameters
     * @param ts
     * @param currOrdinates
     * @param currTimeStep
     * @param modelStartTime
     * @param modelEndTime
     * @param timeBackShift
     * @param intervalCount
     * @return the total discharge at current time step with applying Mod.
     * @throws Exception
     */
    double computeTotalDischargeWithMod(final UHGModelParameters uhgModelParameters,
                                        final RegularTimeSeries ts,
                                        final int currOrdinates,
                                        final long currTimeStep,
                                        final long modelStartTime,
                                        final long modelEndTime,
                                        final long timeBackShift,
                                        final int intervalCount) throws Exception
    {
        _uhgModelParameters = uhgModelParameters;
        _ts = ts;

        // Get Mod start and end date/time 
        for(int i = 0; i < _uhgModelParameters.getNumMods(); i++)
        {
            _startTimeMod = _uhgModelParameters.getStartDateMod()[i];
            _endTimeMod = _uhgModelParameters.getEndDateMod()[i];

            final long lastTime = currTimeStep + (currOrdinates * _ts.getIntervalInMillis());

            if(lastTime >= _startTimeMod - _ts.getIntervalInMillis() && lastTime <= _endTimeMod)
            {
                break;
            }
        }
        //FB 1724
        //_uhgOrdinates = _uhgModelParameters.getUHGOrdinates(_uhgModelParameters.getStartDateMod()[0] - timeBackShift);
        _uhgOrdinates = _uhgModelParameters.getUHGOrdinates(_uhgModelParameters.getStartDateMod()[0]);

        // The start date of MOD is before or the same as model start run 
        if(_uhgModelParameters.getStartDateMod()[0] <= modelStartTime)
        {
            return (this.computeQWhenStartDateBeforeEqualModelStartDate(currOrdinates,
                                                                        currTimeStep,
                                                                        modelStartTime,
                                                                        modelEndTime,
                                                                        timeBackShift,
                                                                        intervalCount));
        }
        // The start date of MOD starts after the model start run.
        else
        {
            return (this.computeQWhenModStartDateAfterModelStartDate(currOrdinates,
                                                                     currTimeStep,
                                                                     modelStartTime,
                                                                     modelEndTime,
                                                                     timeBackShift,
                                                                     intervalCount));
        }

    }

    /*-
     * This method calculates discharge values with applying MOD. If the start date of MOD is
     * before or the same date as model start run, then MOD applied to the carryover runoff and 
     * runoff values until the MOD have expired. After the end date of MOD, the remaining
     * volumes back into the default UNITHG.  
     * It returns the total discharge at current time step.
     * 
     *    
     *        MOD0 and Model      End              Start      End 
     *        Start run are       date of          date of    date of
     *        the same            MOD0             MOD1       MOD1
     *             |               |               |          |   
     * cccccccccccc|RRRRRRRRRRRRRRR|rrrrrrrrrrrrr|R|RRRRRRRRRR|rrrrrrrrrrrrr 
     * <---MOD-----|-------Mod---->|<---no mod-->^<---Mod---->|<--no mod-->
     *                                           ^
     *                                          Apply mod
     *                                          one time step early
     * 
     *  Note: cccccc - MOD applied to runoff carryover
     *        RRRRRR - MOD applied to runoff
     *        rrrrrr - no MOD applied to runoff
     *        
     */
    private double computeQWhenStartDateBeforeEqualModelStartDate(final int currOrdinates,
                                                                  final long currTimeStep,
                                                                  final long modelStartTime,
                                                                  final long modelEndTime,
                                                                  final long timeBackShift,
                                                                  final int intervalCount) throws Exception
    {
        double tempResults = 0.0;

        _currOrdinates = currOrdinates;
        _currTimeStep = currTimeStep;

        while((_currOrdinates >= 0) && (_currTimeStep <= modelEndTime))
        {
            // Take the runoff value from drivingTs
            final double tempUhgRunOff = _ts.getMeasurementValueByTime(_currTimeStep, OHDConstants.RUNOFF_UNIT);

            // No MOD (using default unithg ordinates) applied to the runoff value after the end date of MOD 
            if(_currTimeStep > _uhgModelParameters.getEndDateMod()[0])
            {
                // Apply mod one time step early
                if(_currTimeStep == _startTimeMod - _ts.getIntervalInMillis())
                    //FB 1724
                    //_uhgOrdinates = _uhgModelParameters.getUHGOrdinates(_currTimeStep + _ts.getIntervalInMillis()
                    //    - timeBackShift);
                    _uhgOrdinates = _uhgModelParameters.getUHGOrdinates(_currTimeStep + _ts.getIntervalInMillis());
                else
                    //FB 1724
                    // _uhgOrdinates = _uhgModelParameters.getUHGOrdinates(_currTimeStep - timeBackShift);
                    _uhgOrdinates = _uhgModelParameters.getUHGOrdinates(_currTimeStep);

                tempResults += tempUhgRunOff * _uhgOrdinates[_currOrdinates];

            }
            else
            // MOD applied to the carryover runoff and runoff values until the MOD time have expired.
            {
                tempResults += tempUhgRunOff * _uhgOrdinates[_currOrdinates];
            }

            _currOrdinates -= intervalCount;
            _currTimeStep += _ts.getIntervalInMillis();

        } // end while

        return tempResults;
    }

    /*-
     * This method calculates discharge values with applying MOD. When the start date of MOD is after the
     * Model start run, the MOD get applied one time step early and the runoff values are only applied
     * the MOD ordinates during the MOD window period. No MOD applied to the carryover runoff values and
     * after the end date of MOD, the remaining volumes back into the default UNITHG.      
     * It returns the total discharge at current time step.
     *  
     *         Model
     *         start      MOD1            MOD1            MOD2      MOD2
     *         run        Start           End             Start     End      
     *         |        | |               |               |         |  
     * cccccccc|rrrrrrrr|R|RRRRRRRRRRRRRRR|rrrrrrrrrrrrr|R|RRRRRRRRR|rrrrrrrrrr 
     * <-----NO MOD---->|<------Mod------>|<--no mod--->|<---Mod--->|<-no mod--> 
     *                  ^                               ^
     *                  ^                               ^
     *              MOD apply                           MOD apply
     *              one time                            one time
     *              step early                          step early
     *              
     *  Note: cccccc - No MOD applied to runoff carryover
     *        rrrrrr - No MOD applied to runoff
     *        RRRRRR - MOD applied to runoff
     * 
     */
    private double computeQWhenModStartDateAfterModelStartDate(final int currOrdinates,
                                                               final long currTimeStep,
                                                               final long modelStarTime,
                                                               final long modelEndTime,
                                                               final long timeBackShift,
                                                               final int intervalCount) throws Exception
    {
        double tempResults = 0.0;

        _currOrdinates = currOrdinates;
        _currTimeStep = currTimeStep;

        while((_currOrdinates >= 0) && (_currTimeStep <= modelEndTime))
        {
            // Take the runoff value from drivingTs
            final double tempUhgRunOff = _ts.getMeasurementValueByTime(_currTimeStep, OHDConstants.RUNOFF_UNIT);

            // MOD applied to the runoff during the MOD time window and one time step early. 
            if(_currTimeStep >= _uhgModelParameters.getStartDateMod()[0] - _ts.getIntervalInMillis()
                && _currTimeStep <= _uhgModelParameters.getEndDateMod()[0])
            {
                tempResults += tempUhgRunOff * _uhgOrdinates[_currOrdinates];
            }
            else if(_currTimeStep > _uhgModelParameters.getEndDateMod()[0])
            {
                if(_currTimeStep >= _startTimeMod - _ts.getIntervalInMillis() && _currTimeStep <= _endTimeMod)
                    //FB 1724
                    //_uhgOrdinates = _uhgModelParameters.getUHGOrdinates(_startTimeMod - timeBackShift);
                    _uhgOrdinates = _uhgModelParameters.getUHGOrdinates(_startTimeMod);
                else
                    //FB 1724
                    //_uhgOrdinates = _uhgModelParameters.getUHGOrdinates(_currTimeStep - timeBackShift);
                    _uhgOrdinates = _uhgModelParameters.getUHGOrdinates(_currTimeStep);

                tempResults += tempUhgRunOff * _uhgOrdinates[_currOrdinates];
            }
            // No MOD applied to runoff before and after MOD time window.
            else
            {
                tempResults += tempUhgRunOff * _uhgModelParameters.getUHGOrdinates(null)[_currOrdinates];
            }

            _currOrdinates -= intervalCount;
            _currTimeStep += _ts.getIntervalInMillis();

        } // end while

        return tempResults;
    }
}
