package ohd.hseb.ohdmodels.unithg;

import java.util.ArrayList;
import java.util.List;

import javax.management.timer.Timer;

import ohd.hseb.db.DbTimeHelper;
import ohd.hseb.measurement.Measurement;
import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.DISCHARGE_DATATYPE;
import ohd.hseb.util.fews.ModelException;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.OHDUtilities;
import ohd.hseb.util.fews.RUNOFF_DATATYPE;
import ohd.hseb.util.fews.ohdmodels.ModelDriver;

/**
 * The unit hydrograph converts runoff volume into instantaneous discharge at a gage or flow-point. <br>
 * <br>
 * The unit hydrograph is based on a given unit (depth) of runoff occurring over a given watershed (area) during a given
 * data time interval. It produces the typical hydrograph defined at specific ordinates which this volume of runoff
 * would characteristically generate at a flow-point. In terms flow the unit hydrograph can be defined as the outflow
 * hydrograph in CMS which results from 1 MM of runoff from a runoff period of given duration. In terms of volume it
 * represents 1000 M3 per KM2 since a depth of 1 MM over 1 square KM is equal to 1000 M3.<br>
 * <br>
 * The unit hydrograph is a useful hydrologic tool because it is an effective yet simple method of distributing runoff.
 * In technical terms it represents a linear and time invariant system. It continuously takes into consideration channel
 * storage effects above a flow-point and the travel time or areal distribution of runoff. Unit hydrographs also have
 * the important characteristic of proportionality. For runoff intervals of a constant length, changes in runoff volume
 * do not alter the base length of the unit hydrograph. Ordinates are raised or lowered in proportion to the volume of
 * runoff produced. In addition the time distribution of runoff from a given computational interval is independent of
 * concurrent runoff (or the lack thereof) from antecedent or future intervals. Runoff distributed to ordinates from one
 * computational interval may be added to or superposed upon distributed runoff from adjacent intervals. <br>
 * Convert the input runoff TS(type "TCI") to discharge TS(type "SQIN), with interval equals to UHG ordinates interval.
 * Input runoff TS interval must be equal to UHG duration; could be different from UHG ordinates interval, but must be
 * even multiple of it. If the optional old discharge TS is present in the input fews.xml and its interval equals to UHG
 * ordinates interval, the old discharge will be added to the computed discharge TS.
 * 
 * @author FewsPilot Team
 */
public class UHGModelDriver extends ModelDriver
{

    private final UHGModelParameters _uhgModelParameters; //alias to super._parameters to save type casting

    private final UHGModelParameters _uhgModelSavedParameters; //alias to super._savedParameters to save type casting

    private final UHGModelState _uhgModelState; //alias to super._state to save type casting

    private boolean _printLog = false;
    private double[] _uhgOrdinates = null;
    private double[] _defaultUhgOrdinates = null;
    protected int _tempOrdinates = 0;
    protected long _tempTimeStep = 0;
    private long _timeBackShift = 0;
    private int _intervalCount = 1;
    private long _modelStartRun;
    private long _drivingTsIntervalInMillis;
    private long _computationStartTime;
    private long _computationEndTime;

    public UHGModelDriver()
    {
        super(); //calling ModelDriver constructor

        super.setParameters(new UHGModelParameters());
        super._savedParameters = new UHGModelParameters();

        _uhgModelParameters = (UHGModelParameters)super.getParameters(); //use alias to save casting
        _uhgModelSavedParameters = (UHGModelParameters)super._savedParameters; //use alias to save casting

        super._state = new UHGModelState();
        _uhgModelState = (UHGModelState)super._state; //use alias to save casting

    }

    /*
     * (non-Javadoc)
     * @see ohd.hseb.ohdmodels.ModelDriver#execute(java.util.Map)
     */
    @Override
    final public void execute() throws Exception
    {
        double[] _carryOverRunOff = null;
        final List<double[]> carryOverRunOff = new ArrayList<double[]>();
        RegularTimeSeries _totalDischargeRTS = null;
        RegularTimeSeries _oldDischargeTS = null;

        if(_logger.getPrintDebugInfo() > 0)
        {
            _printLog = true;
        }

        // validate params and state too; assign _drivingTs and _oldDischargeTs(if there)
        _oldDischargeTS = this.runLocalModelDriverValidation();
        // set this values and we don't need to retrieve several times in loop
        _drivingTsIntervalInMillis = getDrivingTsIntervalInMillis();
        _computationStartTime = getComputationStartTime();
        _computationEndTime = getComputationEndTime();

        if(_printLog)
        {
            _logger.log(Logger.DEBUG,
                        "Running UNIT-HG model. The input runoff TS is from(GMT time) "
                            + DateTime.getDateTimeStringFromLong(_computationStartTime,OHDConstants.GMT_TIMEZONE) + " to "
                            + DateTime.getDateTimeStringFromLong(_computationEndTime,OHDConstants.GMT_TIMEZONE));

            _logger.log(Logger.DEBUG, "INPUT CARRYOVER VALUES(NUM=" + _uhgModelState.getCarryOverNum() + "):");
            _logger.log(Logger.DEBUG,
                        OHDUtilities.getStringLineFromDoubleArray(_uhgModelState.getCarryOverValuesInMM()));
        }

        /** -------------------run model ------------------------ */

        _totalDischargeRTS = computeUHGModel(carryOverRunOff, _oldDischargeTS);

        _carryOverRunOff = carryOverRunOff.get(0);

        /** -------------------store result RTS into resultMap ------------------------ */
        addTStoResultMap(_totalDischargeRTS);

        /** --------------------- prepare carryover array for storing-------------------- */
        final long carryOverStartTime = _computationEndTime - (_uhgModelState.getCarryOverNum() - 1)
            * _drivingTsIntervalInMillis;

        final double[] newCarryOver = new double[_uhgModelState.getCarryOverNum()];

        int i = 0;
        for(long time = carryOverStartTime; time <= _computationEndTime; time += _drivingTsIntervalInMillis, i++)
        {

            if(time <= getInitialStateTime())
            { //use the old carryover until(including) the initial state time

                final int numStep = (int)((getInitialStateTime() - time) / _drivingTsIntervalInMillis);

                final int oldCarryOverIndex = _uhgModelState.getCarryOverNum() - numStep - 1;

                newCarryOver[i] = _carryOverRunOff[oldCarryOverIndex];

            }
            else
            {
                newCarryOver[i] = _drivingTs.getMeasurementValueByTime(time, OHDConstants.RUNOFF_UNIT);
            }

        }

        //update state
        _uhgModelState.setCarryOverValues(newCarryOver, OHDConstants.RUNOFF_UNIT);

        //set the state current time to the last data point time
        _uhgModelState.setDateTime(_computationEndTime);

        _logger.log(Logger.DEBUG, "UHGModelDriver has finished");

        if(_printLog)
        {
            _logger.log(Logger.DEBUG, "RUNOFF VALUES:");
            _logger.log(Logger.DEBUG, OHDUtilities.getStringOfValuesFromRTS(_drivingTs));
            _logger.log(Logger.DEBUG, "DISCHARGE VALUES:");
            _logger.log(Logger.DEBUG, OHDUtilities.getStringOfValuesFromRTS(_totalDischargeRTS));
            _logger.log(Logger.DEBUG, "CARRYOVER VALUES:");
            _logger.log(Logger.DEBUG,
                        OHDUtilities.getStringLineFromDoubleArray(_uhgModelState.getCarryOverValuesInMM()));
        }

    } //close execute()

    /*-
     *
     * Does UNIT-HG calculation in this method. 
     * This method get each Instantaneous Discharge compute when iterate the RunOff values and Ordinates values.
     *
     *         Table 2. Runoff applied to Unit Hydrograph
     *   Runoff                 |                       Day 1                           |             Day 2                                     |
     *   Period Interval        24      3      6      9     12     15     18     21     24      3      6      9     12     15     18     21     24      3
     *   ------------------------------------------------------------------------------------------------------------------------------------------------
     *   c/o carryover   --   0.46   0.32   0.23   0.16   0.11   0.07   0.04   0.02   0.01   0.00
     *   1   2400-0600 7.05         11.63  16.50  11.42   8.60   6.35   4.58   3.31   2.19   1.41   0.78   0.28
     *   2   0600-1200 2.04                        3.37   4.77   3.30   2.49   1.84   1.33   0.96   0.63   0.41   0.22   0.08
     *   3   1200-1800 1.50                                      2.48   3.51   2.43   1.83   1.35   0.98   0.71   0.47   0.30   0.17   0.06
     *   4   1800-2400 1.14                                                    1.88   2.67   1.85   1.39   1.03   0.74   0.54   0.35   0.23   0.13   0.05
     *   5   2400-0600 0.87                                                                  1.44   2.04   1.41   1.06   0.78   0.57   0.41   0.27   0.17
     *   6   0600-1200 0.69                                                                                1.14   1.61   1.12   0.84   0.62   0.45   0.32
     *   7   1200-1800 0.56                                                                                              0.92   1.31   0.91   0.68   0.50
     *   8   1800-2400 0.47                                                                                                            0.78   1.10   0.76
     *   Instantaneous
     *   Discharge (CMS)      0.46  11.95  16.73  14.95  13.48  12.20  10.62   9.48   8.03   7.01   5.82   4.98   4.10   3.74   3.24   3.01   2.63   --
     *   ------------------------------------------------------------------------------------------------------------------------------------------------
     *   Number of Ordinates = 11 Ordinate spacing = 3 hours Area = 102.7 KM2
     *   Ordinates = 1.65 2.34 1.62 1.22 .90 .65 .47 .31 .20 .11 .04        
     *
     *  The Instantaneous Discharge values (Q) are computed base in Ordinates Values (Y) and Runoff values (X) as follow:
     *  Q0 = 0.46 (FROM CARRYOVER (CO))
     *  Q1 = 0.32 + (7.05)*(1.65) = 0.32 + 11.63 = 11.95 
     *  Q1 = CO + (X1)(Y1)
     *  
     *  Q2 = 0.23 + (7.05)*(2.34) = 0.23 + 16.50 = 16.73
     *  Q2 = CO + (X1)(Y2)
     *  
     *  Q3 = 0.16 + (7.05)*(1.62) + (2.04)*(1.65) = 0.16 + 11.42 + 3.37 = 14.95
     *  Q3 = CO + (X1)(Y3) + (X2)(Y1)
     *  ...   
     *
     *   Algorithm:
     *    X = RunOff values in array
     *    Y = Ordinates values in array
     *    Q = Instantaneous Discharge (result of compute)
     *    Results are computed base on the following example, where X and Y are position in array
     *    Q  = (X)(Y)
     *    ------------
     *    Q0 = (X0)(Y0)
     *    Q1 = (X0)(Y1) + (X1)(Y0)
     *    Q2 = (X0)(Y2) + (X1)(Y1) + (X2)(Y0)
     *    Q3 =            (X1)(Y2) + (X2)(Y1) + (X3)(Y0)
     *    Q4 =                       (X2)(Y2) + (X3)(Y1)
     *    Q5 =                                  (X3)(Y2)
     *    
     *    Notice that in order to compute Q3 value, X positions increase and Y position decrease.
     *    to get values to be computed.  
     *    
     *    For instance:
     *    If X = 2, 1, 3, 4 and Y = 1, 4, 2
     *    
     *    Q1 = (2)*(1)                                 = 2
     *    Q2 = (2)*(4) + (1)*(1)                       = 9
     *    Q3 = (2)*(2) + (1)*(4) + (3)*(1)             = 11
     *    Q4 =           (1)*(2) + (3)*(4) + (4)*(1)   = 18
     *    Q5 =                     (3)*(2) + (4)*(4)   = 22
     *    Q6 =                               (4)*(2)   = 8
     *       
     *    This method get each Instantaneous Discharge (Q value) compute when iterate the RunOff values and Ordinates values (X and Y array)
     *    using temporary pointers to X and Y to get values required to get the complete Q results when iterating X and Y.    
     *    
     *    Note: the results are affected by MODS and time back shifts
     *    
     * UPDATE:
     *    A change was introduce when MODS get applied. For a giving Q Time MODS will apply for all RunOff values and previous computed values also.
     *    For instance if Q3 have MODS all Ordinates values in the Q3 will be from Ordinates MODS values. Also the values in Q1 and Q2 where Q3 intersect will be updated. 
     *    In this example { (X1)(Y0) and (X1)(Y1)} will be calculated again using Ordinates MODS values.   
     *    
     *    We append to the input time series the carryover values to ease and reduce the computation steps. 
     * 
     */
    private RegularTimeSeries computeUHGModel(final List<double[]> carryOverRunOff,
                                              final RegularTimeSeries _oldDischargeTS) throws Exception
    {
        /** --------------------------convert input runoff into discharge ------------------ */
        final UHGMod _mod = new UHGMod();

        double instantDischarge = 0.0;
        double tempResults = 0.0;
        int coRunOffCount = 0;
        double uhgRunOff = 0.0;
        /*
         * IDTR is equal to or multiple of IDTQ. If IDTR and IDTQ are equal, no time shift; If not equal, needs to
         * consider that the measurement of runoff value covers back IDTR hours, which is (IDTR/IDTQ -1) number of IDTQ
         * time steps.
         */
        RegularTimeSeries _totalDischargeRTS = null;

        // Get default ordinates 
        _defaultUhgOrdinates = _uhgModelParameters.getUHGOrdinates(null);

        _timeBackShift = (_uhgModelParameters.getUHGDurationInHours() / _uhgModelParameters.getUHGIntervalInHours() - 1)
            * _uhgModelParameters.getUHGIntervalInHours() * Timer.ONE_HOUR;

        final long carryOverStartTime = _computationStartTime - _uhgModelState.getCarryOverNum()
            * _drivingTsIntervalInMillis;

        // Use to save the measurements in the time series
        long count = carryOverStartTime - _timeBackShift;

        final double[] _carryOverRunOff = _uhgModelState.getCarryOverValuesInMM();

        // Prepend carryover values to the driving timeseries.
        coRunOffCount = _carryOverRunOff.length - 1;
        Measurement m = null;

        for(long timeStep = _computationStartTime - _drivingTsIntervalInMillis; timeStep >= carryOverStartTime; timeStep -= _drivingTsIntervalInMillis)
        {
            m = new Measurement(_carryOverRunOff[coRunOffCount], MeasuringUnit.mm);
            _drivingTs.setMeasurementByTime(m, timeStep);
            coRunOffCount--;
        } // end Prepend  carryover values 

        // This interval count will tell us how many records we need to switch back the maximum size of ordinates 
        // to compute values Q4 to Q7 from our example
        if(_timeBackShift > 0)
        {
            _intervalCount = (int)(_drivingTsIntervalInMillis / (_drivingTsIntervalInMillis - _timeBackShift));
        }

        // TimeSeries where the results will be saved.
        _totalDischargeRTS = new RegularTimeSeries(carryOverStartTime,
                                                   _computationEndTime,
                                                   _uhgModelParameters.getUHGIntervalInHours(),
                                                   OHDConstants.DISCHARGE_UNIT);
        _totalDischargeRTS.setTimeSeriesType(DISCHARGE_DATATYPE.SQIN.getTypeName());

        // Get model start run
        _modelStartRun = this.getRunStartTimeLong();

        // Get group UHG_ORDINATES dates
        _uhgModelParameters.getModDates();

        // This loop will iterate all the runOff values. We need to iterate throw all the time steps in the time series.
        // and get the ordinates for each time step as they can change based on if they have mods or not.
        // Basically this algorithm will obtain each the complete Q value in each iteration of the loop. It will moved through all the row 
        // Q to obtain the required values even if they are in a next (or previous) run off.

        for(long timeStep = carryOverStartTime; timeStep <= _computationEndTime; timeStep += _drivingTsIntervalInMillis)
        {
            //take the runoff value from drivingTs
            uhgRunOff = _drivingTs.getMeasurementValueByTime(timeStep, OHDConstants.RUNOFF_UNIT);

            //get the ordinates for each time step.
            _uhgOrdinates = _uhgModelParameters.getUHGOrdinates(timeStep - _timeBackShift);

            // If this is the first iteration then we start in the first ordinate. 
            int initOrd = 0;
            if(timeStep != carryOverStartTime)
            {
                initOrd = _uhgOrdinates.length - _intervalCount;
            }

            // Only first time we need to iterate all ordinates values,  
            // first time will get us values for Q1, Q2, and Q3, next iterations will get us remaining values.
            // This is controlled by the initOrd variable.

            for(int ord = initOrd; ord < _uhgOrdinates.length; ord++)
            {
                _uhgOrdinates = _uhgModelParameters.getUHGOrdinates(timeStep - _timeBackShift);

                instantDischarge = uhgRunOff * _uhgOrdinates[ord];

                _tempOrdinates = ord;
                _tempTimeStep = timeStep;
                _tempOrdinates -= _intervalCount;
                _tempTimeStep += _drivingTsIntervalInMillis;

                // Compute Discharge with MOD ========================.  
                if(_uhgModelParameters.applyMod())
                {
                    tempResults = _mod.computeTotalDischargeWithMod(_uhgModelParameters,
                                                                    _drivingTs,
                                                                    _tempOrdinates,
                                                                    _tempTimeStep,
                                                                    _modelStartRun,
                                                                    _computationEndTime,
                                                                    _timeBackShift,
                                                                    _intervalCount);
                }
                // Compute discharge without MOD ===============
                else
                {
                    tempResults = this.computeTotalDischarge();
                }

                // Save the values to TimeSeries.
                _totalDischargeRTS.setMeasurementByTime((instantDischarge + tempResults), count);

                count += (_drivingTsIntervalInMillis - _timeBackShift);

            } // close for loop
        } //close for loop

        //the location id HAS to be specified, for the fewsadapter to be able to find this in the results
        _totalDischargeRTS.setLocationId(_outputLocationId);

        /** -------------------trim the output TS starting time ------------------------ */
        _totalDischargeRTS.trimTimeSeriesAtStartWithCheck(_computationStartTime - _timeBackShift);

        /** ------------------------- UHG is additive -------------------------- */
        if(_oldDischargeTS != null)
        {
            _oldDischargeTS.trimTimeSeriesAtStartWithCheck(_totalDischargeRTS.getStartTime());
            _oldDischargeTS.trimTimeSeriesAtEnd(_totalDischargeRTS.getEndTime());
            _totalDischargeRTS.addToRegularTimeSeries(_oldDischargeTS);
        }

        //add constant base flow value with unit of CMS to each discharge value
        _totalDischargeRTS.addToRegularTimeSeries(_uhgModelParameters.getConstantBaseFlow());

        /** -------------------trim the output TS end time(Optional) ------------------------ */
        _totalDischargeRTS.trimTimeSeriesAtEndWithCheck(_computationEndTime);

        carryOverRunOff.add(_carryOverRunOff);

        return _totalDischargeRTS;
    }

    /*-
     * This method computes total discharge without MOD.
     * - Uses default ordinates
     *    
     *    
     *            Model
     *            start
     *            run
     *             |               
     * cccccccccccc|rrrrrrrrrrrrrrrrrrrrrrrrrrrr 
     * <-----------default ordinates-----------> 
     * 
     * 
     *  Note: cccccc - No MOD applied to runoff carryover
     *        rrrrrr - No MOD applied to runoff
     *
     *        
     */
    double computeTotalDischarge()
    {
        double tempResults = 0.0;

        while((_tempOrdinates >= 0) && (_tempTimeStep <= _computationEndTime))
        {
            // Take the runoff value from drivingTs
            final double tempUhgRunOff = _drivingTs.getMeasurementValueByTime(_tempTimeStep, OHDConstants.RUNOFF_UNIT);

            tempResults += tempUhgRunOff * _defaultUhgOrdinates[_tempOrdinates];

//            System.out.println(DateTime.getDateTimeStringFromLong(_tempTimeStep, null) + " RO= " + tempUhgRunOff
//                + " DEFAULT[" + _tempOrdinates + "]= " + _defaultUhgOrdinates[_tempOrdinates] + " ->Q = " + tempResults);

            _tempOrdinates -= _intervalCount;

            _tempTimeStep += _drivingTsIntervalInMillis;
        }
        return tempResults;
    }

    /*
     * (non-Javadoc)
     * @see ohd.hseb.ohdmodels.ModelDriver#runDriverValidation()
     */

    private RegularTimeSeries runLocalModelDriverValidation() throws Exception
    {
        RegularTimeSeries _oldDischargeTS = null;
        //assign _drivingTs and _oldDischargeTs, not assuming any order in the input fews.xml
        for(int i = 0; i < getTsList().size(); i++)
        {
            final String tsType = getTsList().get(i).getTimeSeriesType();

            if(RUNOFF_DATATYPE.isRunOffType(tsType))
            {
                _drivingTs = getTsList().get(i); //alias

                _uhgModelParameters.setRunOffTsInterval(getDrivingTsInterval());
            } //close if
            else if(DISCHARGE_DATATYPE.isDischargeType(tsType))
            {
                _oldDischargeTS = getTsList().get(i); //alias

                _uhgModelParameters.setOldDischargeTsInterval(_oldDischargeTS.getIntervalInHours());
                if(_printLog)
                {
                    _logger.log(Logger.DEBUG, "Using UHG additive feature.");
                }

            }
            else
            {
                if(_printLog)
                {
                    _logger.log(Logger.DEBUG, "The input TS type \"" + getTsList().get(i).getTimeSeriesType()
                        + "\" is not recognized and is ignored in UHG computation.");
                }

                _tsList.remove(i);

                i--; //because after removing, next obj shifted up
            }

        } //close for block

        if(_drivingTs == null)
        {
            throw new ModelException("No input runoff time series in the input xml file!");
        }

        super.runModelDriverValidation();

        //validate UHGModelParamters and UHGModelState
        _uhgModelParameters.validateParams();

        if(super.needCarryoverTransfer())
        {
            _uhgModelState.validateState(_uhgModelSavedParameters);
            _uhgModelState.doCarryOverTransfer(_uhgModelSavedParameters, _uhgModelParameters);
        }
        else
        {
            _uhgModelState.validateState(_uhgModelParameters);

        }
        return _oldDischargeTS;

    } //close runDriverValidation
} //close class
