package ohd.hseb.ohdmodels.unithg;

import java.text.ParseException;
import java.util.Arrays;
import java.util.Map;

import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.MathHelper;
import ohd.hseb.util.fews.GroupOfParameters;
import ohd.hseb.util.fews.GroupOfParameters.ValidPeriod;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.ParameterType.Row;
import ohd.hseb.util.fews.ParameterType.Table;
import ohd.hseb.util.fews.Parameters;
import ohd.hseb.util.fews.ohdmodels.ModelException;
import ohd.hseb.util.fews.ohdmodels.ModelParameters;
import ohd.hseb.util.fileConverter.UHGOptFileConverter;

/**
 * This class holds all the UHG Model parameters. It is a subclass of {@link ModelParameters} and interface
 * {@link IModelParameters}. They are specified in the opt file:
 * <p>
 * <ul>
 * <li>Card #1,
 * <li>Card #2(correspond to card#3 in PDF document, Fortran code, pin2.f, completely ignores card#2 in document),
 * <li>Card #3(correspond to card#4 in PDF document),
 * </ul>
 * Values are initially loaded into super._parasmMap, then extracted into instance variables to avoid frequent accessing
 * the Map. Setter methods are not needed, since the model does not change parameters.
 * <p>
 * the unit type either be Metric or English.If Metric, both ordinates and constant base flow unit is CMS, drainage area
 * unit is KM2(square kilometer); If English, both ordinates and constant base flow unit is CFS, drainage area unit is
 * MI2(square mile). The unit is determined by parameter xml file.
 * <p>
 * {@link UHGOptFileConverter} sets the parameter values by using various insertParameter(..) methods in
 * {@link Parameters}.
 * 
 * @author FewsPilot Team
 */
final public class UHGModelParameters extends ModelParameters
{

    public final static String _UHG_ORD_TAG = "UHG_ORDINATES";

    public final static String _CONST_BASE_FLOW_TAG = "CONSTANT_BASE_FLOW";

    /**
     * also called Runoff Duration(IDTR): the period in which runoff occurs that cause the discharge. e.g. 6 hr UHG
     * means that 6hr runoff of 1mm will produce such, such discharge TS. Input runoff TS interval should be equal to
     * this interval
     */
    public final static String _UHG_DURATION_TAG = "UHG_DURATION";

    /**
     * It is UHG spacing time interval, it is also the discharge time series interval(IDTQ)
     */
    public final static String _UHG_SPACING_INTERVAL_TAG = "UHG_INTERVAL";

    public final static String _DRAINAGE_AREA_TAG = "DRAINAGE_AREA";

    /*
     * the following TS intervals are not in parameter xml file, they are set in UHGModelDriver. "0" is used as a flag
     * for absence of the corresponding TS. So, if a TS is not available, its interval remains as "0".
     */
    private int _runOffTsInterval = 0; //required input TS
    private int _oldDischargeTsInterval = 0; //for additive feature. If present in input file, use it. No parameter specification for it. 

    private double[] _defaultUhgOrdinates;
    private double[] _computedUhgOrdinates;
    private Map<ValidPeriod, double[]> _ordinatesMap;
    private double _constantBaseFlow;
    private int _durationInHours;
    private int _intervalInHours;

    private long[] _endTimeMod;
    private long[] _startTimeMod;
    private int _numMods = 0;
    /*
     * the unit type either be Metric or English.If Metric, both ordinates and constant base flow unit is CMS, drainage
     * area unit is KM2(square kilometer); If English, both ordinates and constant base flow unit is CFS, drainage area
     * unit is MI2(square mile). default is Metric
     */
    private boolean _unitTypeMetric = true;

    /**
     * Instantiates a new UHG model parameters. Values are set by parsing params.xml file by FewsXMLParser.java(in
     * package ohdfewsadapter) method public void parseParameters(final String paramFileName).
     */
    public UHGModelParameters()
    {
        super();
    }

    /**
     * For performance we set the values of from the parent class to local variables that allow a fast response.
     * 
     * @throws Exception
     */
    @Override
    protected void extractValuesFromMap() throws Exception
    {
        this._durationInHours = getIntDataParameter(UHGModelParameters._UHG_DURATION_TAG);
        this._intervalInHours = getIntDataParameter(UHGModelParameters._UHG_SPACING_INTERVAL_TAG);
        this._constantBaseFlow = getDoubleDataParameter(UHGModelParameters._CONST_BASE_FLOW_TAG);
        this._ordinatesMap = super.getDoubleArrayParameterMap(UHGModelParameters._UHG_ORD_TAG);

        if(isParamExisting(OHDConstants.UNIT_TAG)
            && getStringDataParameter(OHDConstants.UNIT_TAG).trim().equalsIgnoreCase(OHDConstants.UNIT_ENGLISH))
        {
            _unitTypeMetric = false; //default is Metric
        } 
        else 
        if (isParamExisting(OHDConstants.UNIT_TAG)
                && !getStringDataParameter(OHDConstants.UNIT_TAG).trim().equalsIgnoreCase(OHDConstants.UNIT_METRIC)){
            _logger.log(Logger.WARNING,
                    "THE ENGLISH METRIC SWITCH MUST BE '"+OHDConstants.UNIT_METRIC+"', '"+OHDConstants.UNIT_ENGLISH+"'," + 
                    " OR BLANK. THE VALUE ENTERED IS " + getStringDataParameter(OHDConstants.UNIT_TAG) +". METRIC IS ASSUMED");
        }

        _defaultUhgOrdinates = super.getDoubleArrayParameter(UHGModelParameters._UHG_ORD_TAG);

        _computedUhgOrdinates = this.calculateUHGOrdinates(null);

    }

    /**
     * For performance we set the values of from the parent class to local variables that allow a fast response.
     * 
     * @throws Exception
     */
    @Override
    protected void setValuesToMap() throws Exception
    {
        super.insertParameter(UHGModelParameters._UHG_DURATION_TAG, this._durationInHours);
        super.insertParameter(UHGModelParameters._UHG_SPACING_INTERVAL_TAG, this._intervalInHours);
        super.insertParameter(UHGModelParameters._CONST_BASE_FLOW_TAG, this._constantBaseFlow);

        for(final ValidPeriod validPeriod: _ordinatesMap.keySet())
        {
            final Table table = new Table();
            final Row columnTypes = new Row();
            Row row = null;
            final double[] ordinates = _ordinatesMap.get(validPeriod);
            for(int i = 0; i < ordinates.length; i++)
            {
                row = new Row();
                row.setA(Double.toString(ordinates[i]));
                table.setRow(row);
            }
            columnTypes.setA("double");
            table.setColumnTypes(columnTypes);

            super.insertParameter(UHGModelParameters._UHG_ORD_TAG, table, validPeriod);
        }
    }

    public void validateParams() throws ModelException, Exception
    {

        //UHG duration must be equal or (even) multiple of UHG spacing
        if(getUHGDurationInHours() % getUHGIntervalInHours() != 0)
        {
            final StringBuffer message = new StringBuffer();
            message.append("Error: unit hydrograph duration must be equal or multiple of the UHG data interval.")
                   .append("Currently, unit hydrograph duration(HR)= ")
                   .append(getUHGDurationInHours())
                   .append(" UHG data time interval(HR)= ")
                   .append(getUHGIntervalInHours());

            throw new ModelException(message.toString());
        }

        //check the input runoff TS interval must be equal to the UHG duration
        if(getRunOffTsInterval() != getUHGDurationInHours())
        {
            throw new ModelException("Input runoff time series interval (" + getRunOffTsInterval()
                + ") not equal to the unit hydrograph (" + getUHGDurationInHours() + ") duration!");
        }

        //check the old discharge TS interval equal to UHG ordinates interval, e.g. IDTQ
        if(getOldDischargeTsInterval() != 0 && getOldDischargeTsInterval() != getUHGIntervalInHours())
        {//!=0 means: old discharge TS is used

            throw new ModelException("To use the additive feature, the old discharge time series interval must be equal"
                + " to UHG ordinates interval");
        }

        //compute area represented by UHG & compare with user specified drainage area
        final Long startTime = null;
        final double[] uhgOrdinates = getUHGOrdinates(startTime); //with unit of CMS per MM

        double sumOrd = 0.0;

        for(int i = 0; i < uhgOrdinates.length; i++)
        {
            sumOrd += uhgOrdinates[i];
        }

        final double area_from_ordinates = sumOrd / (24 / getUHGIntervalInHours()) * 86.4; //86400 seconds per day / 1000 mm per meter

        double drainageArea; //in unit of square kilometer

        if(_unitTypeMetric == true)
        {
            drainageArea = getDoubleDataParameter(UHGModelParameters._DRAINAGE_AREA_TAG);
        }
        else
        {
            drainageArea = getDoubleDataParameter(UHGModelParameters._DRAINAGE_AREA_TAG) * 2.58998811; // 1 square mile = 2.59 square kilometers
        }

        if(Math.abs(drainageArea - area_from_ordinates) / area_from_ordinates > 0.01)
        {
        	// Change the variables place as they were in the wrong order. 
        	// Documented in REDMINE #80048: Drainage Areas getting reversed in CHPS warning message 
            _logger.log(Logger.WARNING,
                        "Area represented by the unit hydrograph (" + MathHelper.roundToNDecimalPlaces(area_from_ordinates, 2)
                            + " square kilometers) differs from the user specified area " + "("
                            + MathHelper.roundToNDecimalPlaces(drainageArea, 2)
                            + " square kilometers) by more than one percent");
        }

        if(_logger.getPrintDebugInfo() > 0)
        {
            if(_unitTypeMetric == true)
            {
                _logger.log(Logger.DEBUG, "Input parameters use Metric units");
            }
            else
            {
                _logger.log(Logger.DEBUG, "Input parameters use English units");
            }

            _logger.log(Logger.DEBUG, "UHGModelParamters has been validated!");
        }

    }

    /**
     * This method returns the flag to be applied Mod or not
     * 
     * @return true - mod applied false - no mod
     */
    boolean applyMod()
    {
        if(_ordinatesMap.size() > 1)
            return true;

        return false;
    }

    /**
     * This method get Mod date and time of each event
     * 
     * @throws Exception
     */
    void getModDates() throws Exception
    {
        int idxMod = 0;
        _numMods = _ordinatesMap.size() - 1;
        final long[] startTimeMod = new long[_numMods];
        final long[] endTimeMod = new long[_numMods];

        for(final ValidPeriod validPeriod: _ordinatesMap.keySet())
        {
            if(validPeriod != OHDConstants.DEFAULT_VALID_PERIOD)
            {
                startTimeMod[idxMod] = new DateTime(validPeriod.getStartDate().getDate(), validPeriod.getStartDate()
                                                                                                     .getTime()).getTimeInMillis();

                endTimeMod[idxMod] = new DateTime(validPeriod.getEndDate().getDate(), validPeriod.getEndDate()
                                                                                                 .getTime()).getTimeInMillis();
                idxMod++;
            }
        }

        if(_numMods > 0)
        {
            Arrays.sort(startTimeMod);
            _startTimeMod = startTimeMod;

            Arrays.sort(endTimeMod);
            _endTimeMod = endTimeMod;
        }
    }

    /**
     * Get number of mod event
     * 
     * @return the number of mod event
     */
    int getNumMods()
    {
        return _numMods;
    }

    /**
     * Retrieve the UH Ordinates; if startTime is a null value it will retrieve the default Ordinates parameters.
     * Otherwise, it will retrieve the mods Ordinates parameters
     * 
     * @param startTime the time we need to retrieve, it can be null value if we want the default value.
     * @return an double array containing the values of the ordinates unit is CMS per MM
     * @throws Exception
     */

    double[] getUHGOrdinates(final Long startTime) throws Exception
    {
        if(_ordinatesMap.size() > 1)
        {
            return this.calculateUHGOrdinates(startTime);
        }
        else
        {
            return _computedUhgOrdinates;
        }
    }

    double[] calculateUHGOrdinates(final Long startTime) throws Exception
    {

        GroupOfParameters.ValidPeriod validPeriod = OHDConstants.DEFAULT_VALID_PERIOD;
        // If date is not null and it is found we will use the validPeriod object otherwise the "Default" valid period will be used.
        // "Default" validPeriod is an null valid period. It could be change later to any other value defined.
        if(startTime != null)
        {
            try
            {
                for(final ValidPeriod period: _ordinatesMap.keySet())
                {
                    if(period != OHDConstants.DEFAULT_VALID_PERIOD)
                    {
                        if(period.getStartDate() != null && period.getEndDate() != null
                            && startTime >= super.asLong(period.getStartDate())
                            && startTime <= super.asLong(period.getEndDate()))
                        {
                            validPeriod = period;
                            //break;
                        }
                        else if(period.getValidBeforeDate() != null
                            && this.asLong(period.getValidBeforeDate()) >= startTime)
                        {
                            validPeriod = period;
                            //break;
                        }
                        else if(period.getValidAfterDate() != null
                            && this.asLong(period.getValidAfterDate()) <= startTime)
                        {
                            validPeriod = period;
                            //break;
                        }
                        else if(period.getStartMonthDay() != null && period.getEndMonthDay() != null)
                        {
                            validPeriod = period;
                            //break;
                        }
                        else if(period.getMonthDay() != null && period.getMonthDay().size() > 0)
                        {
                            validPeriod = period;
                            //break;
                        }
                        else if(period.getMonth() != null && period.getMonth().size() > 0)
                        {
                            validPeriod = period;
                            //break;
                        }
                        else if(period.getDay() != null && period.getDay().size() > 0)
                        {
                            validPeriod = period;
                            //break;
                        }
                    }
                } // end for loop

            }
            catch(final ParseException pe)
            {
                throw new Exception("Error when getting parameter values for " + UHGModelParameters._UHG_ORD_TAG);
            }
        }

        final double[] uhgOrdinates = _ordinatesMap.get(validPeriod).clone();

        if(startTime != null && uhgOrdinates != null && uhgOrdinates.length > 0)
        {
            double defaultOrdinates = 0.0;
            double modsOrdinates = 0.0;
            for(int i = 0; i < uhgOrdinates.length; i++)
            {
                defaultOrdinates += _defaultUhgOrdinates[i];
                modsOrdinates += uhgOrdinates[i];
            }

            if(modsOrdinates != 0.0)
            {//to avoid dividing by 0.0, generateing NaN
                final double scale = defaultOrdinates / modsOrdinates;
                for(int i = 0; i < uhgOrdinates.length; i++)
                {
                    uhgOrdinates[i] *= scale;
                }
            }
        }

        if(_unitTypeMetric == false) //convert ordinates from CFS per INCH to CMS per MM
        {
            for(int i = 0; i < uhgOrdinates.length; i++)
            {
                uhgOrdinates[i] /= 35.3147; // 35.3147 FT3 = 1 M3
                uhgOrdinates[i] /= 25.4; // 1 IN = 25.4 MM
            }
        }
        return uhgOrdinates;
    }

    long[] getStartDateMod()
    {
        return _startTimeMod;
    }

    long[] getEndDateMod()
    {
        return _endTimeMod;
    }

    /**
     * @return
     */
    public int getUHGOrdinatesAmount()
    {
        return _defaultUhgOrdinates.length;
    }

    //the returned value is with unit of CMS
    double getConstantBaseFlow()
    {
        if(_unitTypeMetric == true)
        {
            return this._constantBaseFlow;
        }

        return this._constantBaseFlow / 35.3147;
        //35.3147 FT3 = 1 M3
    }

    void setConstantBaseFlow(final double constantBaseFlow)
    {
        this._constantBaseFlow = constantBaseFlow;
    }

    /**
     * IDTR
     */
    public int getUHGDurationInHours()
    {
        return this._durationInHours;
    }

    void setUHGDurationInHours(final int idtr)
    {
        this._durationInHours = idtr;
    }

    /**
     * IDTQ
     */
    public int getUHGIntervalInHours()
    {
        return this._intervalInHours;
    }

    void setUHGIntervalInHours(final int idtq)
    {
        this._intervalInHours = idtq;
    }

    int getRunOffTsInterval()
    {
        return _runOffTsInterval;
    }

    void setRunOffTsInterval(final int runOffTsInterval)
    {
        _runOffTsInterval = runOffTsInterval;
    }

    /**
     * For additive feature to the output discharge TS
     */
    int getOldDischargeTsInterval()
    {
        return _oldDischargeTsInterval;
    }

    /**
     * For additive feature to the output discharge TS
     */
    void setOldDischargeTsInterval(final int oldDischargeTsInterval)
    {
        _oldDischargeTsInterval = oldDischargeTsInterval;
    }

} //close class
