package ohd.hseb.ohdmodels.unithg;

import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.Parameters;
import ohd.hseb.util.fews.State;
import ohd.hseb.util.fews.ohdmodels.ModelException;
import ohd.hseb.util.fews.ohdmodels.ModelParameters;
import ohd.hseb.util.fews.ohdmodels.ModelState;

/**
 * Stores the state(carryover, old runoff TS) with interval too. 1)load state from a xml file to get a)carryover TS and
 * b)the unit hydrograph for the old runoff(optional); 2)generate the discharge TS based on the old runoff TS and the
 * corresponding UHG. 3)save the runoff and the corresponding UHG into a xml file, for next time warm start Because not
 * using _stateMap to store state variable values, so no need to derive from ModelState, instead implement IModelState
 * remember the unit
 */
final public class UHGModelState extends ModelState
{

    private final static String _RUNOFF_PREFIX = "RUNOFF#";
    //in input state file, carryover runoff array is like: RUNOFF#0=2.2 etc.

    private final static String _UHG_ORDINATE_NUM = "UHG_ORDINATE_NUM";

    /** Metric system unit: {@link MeasuringUnit#mm} */
    public final static MeasuringUnit _DEFAULT_RUNOFF_UNIT = OHDConstants.RUNOFF_UNIT;

    /** English system unit: {@link MeasuringUnit#inches} */
    public final static MeasuringUnit _ENGLIGH_RUNOFF_UNIT = MeasuringUnit.inches;

    private int _durationinHours;
    private int _intervalInHours;
    private int _ordinateNumber;
    private double[] _carryOver;
    private String _unitType;

    /**
     * Default constructor. The values are loaded through {@link State#loadState(String, Logger)} from input state file.
     */
    UHGModelState()
    {
        super();
    }

    /**
     * This constructor is only used in UHGOptFileConverter.java
     * 
     * @throws ModelException
     */
    public UHGModelState(final double[] carryOverValues,
                         final MeasuringUnit unit,
                         final int ordNum,
                         final int idtr,
                         final int idtq) throws ModelException
    {
        this();

        _durationinHours = idtr;

        _intervalInHours = idtq;

        _ordinateNumber = ordNum;

        setCarryOverValues(carryOverValues, unit);

    }

    /**
     * calculated on the fly: (n-1) / (IDTR / IDTQ)
     */
    int getCarryOverNum()
    {
        return (this._ordinateNumber - 1) / (_durationinHours / _intervalInHours);
    }

    /**
     * Returns carryover values in MM.
     */
    double[] getCarryOverValuesInMM()
    {
        return _carryOver;
    }

    /**
     * The number of carryover should be: (n-1) * IDTQ/IDTR. n: number of ordinates; detal_t: IDTQ; delta_T: IDTR. But
     * does not check it in this method. It is checked in validateState(). Because UHGOptFileConverter will use this
     * method through fillUHGModelState() and the state is not loaded yet.
     * <p>
     * The unit can only be MM or IN, one is Metric system unit, one is English system unit.
     * 
     * @param carryOverValues
     */
    void setCarryOverValues(final double[] carryOverValues, final MeasuringUnit unit) throws ModelException
    {
        /** --------------------------check the unit first ---------------------- */
        if(unit != UHGModelState._DEFAULT_RUNOFF_UNIT && unit != UHGModelState._ENGLIGH_RUNOFF_UNIT)
        {
            throw new ModelException("Carryover(old runoff) unit can only be MM or IN.");
        }

        if(unit == UHGModelState._DEFAULT_RUNOFF_UNIT)
        {//unit is MM
            this._unitType = OHDConstants.UNIT_METRIC;
        }
        else
        {//unit is INCH
            this._unitType = OHDConstants.UNIT_ENGLISH;
        }
        this._carryOver = carryOverValues;

    }

    /**
     * Creates a tag, like "RUNOFF_#0", "RUNOFF_#1", ..., etc. When writing out the state, the integer after "#" is used
     * for sorting.
     * <p>
     * This method is used in both saving the state into and retrieving the state from _statesMap. So the tag needs to
     * be consistent.
     */
    private String getRunOffTag(final int valueIndex)
    {
        final StringBuilder sb = new StringBuilder(_RUNOFF_PREFIX);
        return sb.append(valueIndex).toString();

        //return _RUNOFF_PREFIX + String.valueOf(valueIndex);
    }

    /**
     * Returns true if the unit type is "METRIC", otherwise returns false, meaning the unit type is "ENGLISH". If there
     * is no entry with key of {@link OHDConstants#UNIT_TAG}, the default is "METRIC" and return true.
     */
    private boolean isUnitTypeMetric()
    {
        if(_unitType == null || this._unitType.equals(OHDConstants.UNIT_METRIC))
        {
            return true;
        }

        return false;
    }

    /**
     * Only MM or INCH. One is the unit in Metric system, one is the unit in English system
     */
    private MeasuringUnit getRunoffUnit()
    {
        if(isUnitTypeMetric() == true)
        {
            return UHGModelState._DEFAULT_RUNOFF_UNIT;
        }

        return UHGModelState._ENGLIGH_RUNOFF_UNIT;
    }

    /**
     * Check IDTR and IDTQ to be consistent with {@link UHGModelParameters}. Make sure UHG ordinates number in
     * states/statesI.txt and the parameter xml file are the same. Make sure the number of carryover values in
     * statesI.txt is enough.
     */
    public void validateState(final Parameters params) throws Exception
    {

        final UHGModelParameters uhgParams = (UHGModelParameters)params;

        /** ----------- set local variables values -------------- * */
        boolean convertValue = false;
        this._durationinHours = super.getIntDataState(UHGModelParameters._UHG_DURATION_TAG);
        this._intervalInHours = super.getIntDataState(UHGModelParameters._UHG_SPACING_INTERVAL_TAG);
        this._ordinateNumber = super.getIntDataState(UHGModelState._UHG_ORDINATE_NUM);
        this._unitType = super.getStateAsString(OHDConstants.UNIT_TAG);

        if(getRunoffUnit() != UHGModelState._DEFAULT_RUNOFF_UNIT)
        {
            convertValue = true;
        }

        /** ------------check state IDTR equal to parameter specification ------------ */
        if(_durationinHours != uhgParams.getUHGDurationInHours())
        {
            throw new ModelException("Input state UHG Duration(" + _durationinHours
                + "hr) is not equal to parameter specification(" + uhgParams.getUHGDurationInHours() + "hr)");
        }

        /** ------------check state IDTQ equal to parameter specification ------------ */
        if(_intervalInHours != uhgParams.getUHGIntervalInHours())
        {
            throw new ModelException("Input state UHG spacing interval(" + _intervalInHours
                + "hr) is not equal to parameter specification(" + uhgParams.getUHGIntervalInHours() + "hr)");
        }

        /** -----make sure UHG ordinates number in states/statesI.txt and the parameter xml file are the same --- */
        boolean changeStateRunoff = false;
        final int originalNumberOfOrdinates = this._ordinateNumber;

        if(this._ordinateNumber != uhgParams.getUHGOrdinatesAmount())

        {

            //tolerate this case: somehow "UHG_ORDINATE_NUM" in states/statesI.txt can be less than or greather than 
            //the num of "UHG_ORDINATES" in params.xml

            if(this._ordinateNumber < uhgParams.getUHGOrdinatesAmount())
            {
                for(int i = this._ordinateNumber; i < uhgParams.getUHGOrdinatesAmount(); i++)
                    this._ordinateNumber++; //increment to match

                changeStateRunoff = true;

                _logger.log(Logger.WARNING,
                            "UHG_ORDINATE_NUM in states is less than the num of UHG_ORDINATES in the parameters "
                                + "Number of ordinates changed from " + originalNumberOfOrdinates + " to "
                                + _ordinateNumber + ". Insert extra runoff value as 0.0");
            }
            else
            {
                for(int i = this._ordinateNumber; i > uhgParams.getUHGOrdinatesAmount(); i--)
                    this._ordinateNumber--; //decrement to match            	            	 

                _logger.log(Logger.WARNING,
                            "UHG_ORDINATE_NUM in states is greater than the num of UHG_ORDINATES in the parameters"
                                + "Number of ordinates changed from " + originalNumberOfOrdinates + " to "
                                + _ordinateNumber);
            }
        }

        _carryOver = new double[getCarryOverNum()];

        /*
         * check the number of carryover, based on "UHG_ORDINATE_NUM" in statesI.txt, allows it to be more than needed.
         * This does not check "UHG_ORDINATE_NUM" in statesI.txt match with the num of "UHG_ORDINATES" in params.xml
         */
        int inputCarryOverCounter = 0;

        final TreeMap<String, Object> treeMap = new TreeMap<String, Object>(getStateMap());

        final Set<Map.Entry<String, Object>> mySet = treeMap.entrySet();

        for(final Map.Entry<String, Object> curEntry: mySet)
        {
            if(curEntry.getKey().contains(_RUNOFF_PREFIX))
            {
                inputCarryOverCounter++;
            }
        }

        if(inputCarryOverCounter < _carryOver.length && changeStateRunoff == false)
        {
            throw new ModelException("The number of carryover(" + inputCarryOverCounter
                + ") is not equal to the expected(" + _carryOver.length + ")");
        }

        //allows inputCarryOver greater than or equal to expected carryover number, (n-1) * delta_t /delta_T,
        if(_logger.getPrintDebugInfo() > 0)
        {
            if(inputCarryOverCounter > _carryOver.length)
            {
                final String message = "The number of carryover(" + inputCarryOverCounter
                    + ") is greater than the expected(" + _carryOver.length + "). The extra ones will be ignored.";

                _logger.log(Logger.DEBUG, message);

            }

            //if reach here, validation is fine
            _logger.log(Logger.DEBUG, "UHGModelState has been validated. The number of carryover is "
                + inputCarryOverCounter);
        }

        //after the check above, we know there is enough carryover values in statesI.txt to fill _carryover array
        for(int i = 0; i < _carryOver.length; i++)
        {
            if(i >= (originalNumberOfOrdinates - 1) && changeStateRunoff)
            {//set extra runoffs to 0.0
                _carryOver[i] = 0.0;

                break; //jump out of loop
            }

            _carryOver[i] = super.getDoubleDataState(getRunOffTag(i));

            if(convertValue == true)
            { //convert INCH to MM
                _carryOver[i] *= OHDConstants.MM_PER_INCH; //1 inch = 25.4 mm
            }
        } //close for loop
    }

    @Override
    public void doCarryOverTransfer(final ModelParameters savedParams, final ModelParameters params) throws ModelException
    {
        final UHGModelParameters savedParameters = (UHGModelParameters)savedParams;
        final UHGModelParameters parameters = (UHGModelParameters)params;

        _logger.log(Logger.DEBUG, "UHGModelState before carry over transfer " + this.getStateMap());

        // here goes the carry over logic
        final int idtrn = parameters.getUHGDurationInHours();
        final int idtr = savedParameters.getUHGDurationInHours();

        final int idtqn = parameters.getUHGIntervalInHours();
        final int idtq = savedParameters.getUHGIntervalInHours();

        final int nvn = parameters.getUHGOrdinatesAmount();
        final int nv = savedParameters.getUHGOrdinatesAmount();

        final int nroold = (int)((nv - 1) * (((double)idtq) / idtr));
        final int nronew = (int)((nvn - 1) * (((double)idtqn) / idtrn));

        int nroadj = nroold;
        int l = 0;
        int idiv = 0;
        int ldiv = 0;

        final double[] carryOverValues = this.getCarryOverValuesInMM(); //array length is getCarryOverNum()--based on statesI.txt
        double[] newCarryOverValues = new double[nroold]; //based on information from parameter xml file

        for(int i = 0; i < nroold; i++)
        {
            newCarryOverValues[i] = carryOverValues[i];
        }

        if(idtr != idtrn)
        {
            if((idtr / idtrn) * idtrn != idtr)
            {
                if((idtrn / idtr) * idtr == idtrn)
                {
                    nroadj = (nv - 1) / (idtrn / idtq);
                    final int nr = nroadj;
                    final int n = nroadj + 1;
                    newCarryOverValues = new double[n]; //initialized to 0.0

                    if(nroold > (nroadj * idtrn / idtq))
                    {
                        final int idiff = nroold - nroadj * idtrn / idtq;
                        nroadj = nroadj + 1;
                        for(int i = 0; i < idiff; i++)
                        {
                            newCarryOverValues[0] = newCarryOverValues[0] + carryOverValues[i];
                        }
                        newCarryOverValues[0] = newCarryOverValues[0] / (idtrn / idtq);
                    }
                    final int iratio = idtrn / idtr;
                    l = 1;
                    for(int m = 0; m < nr; m++)
                    {
                        final int lratio = l + iratio - 1;
                        final int j = nroadj - m + 1;
                        for(int i = l; i < lratio; i++)
                        {
                            final int k = nroold - i + 1;
                            newCarryOverValues[j] = carryOverValues[k] + newCarryOverValues[j];
                        }
                        l = l + iratio;
                    }
                }
            }
            else
            {
                nroadj = nroold * (idtr / idtrn);
                newCarryOverValues = new double[nroadj]; //initialized to 0.0

                idiv = idtr / idtrn;
                l = 0;
                for(int m = 0; m < nroold; m++)
                {
                    ldiv = l + idiv;
                    final int i = (nroold - 1) - m;
                    for(int j = l; j < ldiv; j++)
                    {
                        final int k = (nroadj - 1) - j;
                        newCarryOverValues[k] = carryOverValues[i] / idiv;
                    }
                    l = l + idiv;
                }
            }
        }
        //  Compute new carryover        
        final double[] carryOverValuesFinal = new double[nronew]; //initialized to 0.0

        int jro = nronew;
        if(nronew > nroadj)
        {
            jro = nroadj;
        }
        for(int i = 0; i < jro; i++)
        {
            final int n = (nroadj - 1) - i;
            final int k = (nronew - 1) - i;

            carryOverValuesFinal[k] = newCarryOverValues[n];
        }

        _durationinHours = idtrn;
        _intervalInHours = idtqn;
        _ordinateNumber = nvn;

        this.setCarryOverValues(carryOverValuesFinal, UHGModelState._DEFAULT_RUNOFF_UNIT);

        if(_logger.getPrintDebugInfo() > 0)
        {
            _logger.log(Logger.DEBUG, "UHGModelState after carry over transfer " + this.getStateMap());
        }
    }

    /**
     * Over-written the super class method. This method is used to output states into the generated state txt file.
     */
    @Override
    protected String getStringsFromState()
    {
        final StringBuilder resultStr = new StringBuilder();

        for(int i = 0; i < _carryOver.length; i++)
        {
            resultStr.append(_RUNOFF_PREFIX).append(i).append("=").append(_carryOver[i]).append(OHDConstants.NEW_LINE);
        }
        resultStr.append(OHDConstants.UNIT_TAG)
                 .append("=")
                 .append(isUnitTypeMetric() ? OHDConstants.UNIT_METRIC : OHDConstants.UNIT_ENGLISH)
                 .append(OHDConstants.NEW_LINE);
        resultStr.append(UHGModelParameters._UHG_DURATION_TAG)
                 .append("=")
                 .append(_durationinHours)
                 .append(OHDConstants.NEW_LINE);
        resultStr.append(UHGModelParameters._UHG_SPACING_INTERVAL_TAG)
                 .append("=")
                 .append(_intervalInHours)
                 .append(OHDConstants.NEW_LINE);
        resultStr.append(UHGModelState._UHG_ORDINATE_NUM)
                 .append("=")
                 .append(this._ordinateNumber)
                 .append(OHDConstants.NEW_LINE);

        return resultStr.toString();
    }
} //close class
