package ohd.hseb.ohdutilities.ffg;

import java.util.List;
import java.util.Map;
import java.util.Properties;

import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.DISCHARGE_DATATYPE;
import ohd.hseb.util.fews.DataType;
import ohd.hseb.util.fews.Diagnostics;
import ohd.hseb.util.fews.FewsRegularTimeSeries;
import ohd.hseb.util.fews.FewsXMLParser;
import ohd.hseb.util.fews.Parameters;
import ohd.hseb.util.fews.ohdutilities.UtilityDriver;
import ohd.hseb.util.fews.ohdutilities.UtilityState;

public abstract class FFGDriver extends UtilityDriver
{
    protected String[] _areaIdArray; //stores each area Id running by FFH; Gridded FFG only has one area id
    protected FewsRegularTimeSeries _dischargeTs; //required when high flow adjust option is true("FFG_HIGH_FLOW_ADJUST_OPTION" and "FFG_USER_CTRL_HIGH_FLOW_ADJUST")
    protected String _dischargeTslocationIdString="";

    public FFGDriver()
    {
        super(); //calling UtilityDriver constructor

        super._state = new UtilityState();
        //FFH does not have a state, initiation to avoid NPE; Gridded FFG, 1st grid's state has input netcdf file, last grid's state has output netcdf file

    }

    /**
     * Parse input time series file and parameter file. Check every ts in _tsList has enough data. Validate properties
     * in run_info.xml.
     */
    @Override
    protected void runDriverValidation() throws Exception
    {
        /*
         * over-write runStart time with LSTCMPDY, because FFG only cares about events after LSTCMPDY. If without
         * re-setting runStart time, if it is earlier than LSTCMPDY, the validation might find that the input
         * timeseries(MAT for SNOW17, RAIM for API-CONT) not having enough data to cover up to runStart time.
         */
        _runInfo.setRunStartTimeLong(new DateTime(_runInfo.getRunLastObservationTimeLong()));

        this.parseInputTimeSeriesFiles();

        super._state.setDateTime(getInitialStateTime());
        super.runDriverValidation(); //UtilityDriver runDriverValidation() checks every ts in _tsList has enough data

        this.parseParams();

        /*--------------------validate properties section in run_info.xml------------------*/
        final Properties properties = _runInfo.getProperties();

        if(!properties.containsKey(FfgConstants.FFG_PARAMETER_FILE_NAME) //"ffgParameters"
            || !properties.containsKey(FfgConstants.RUNOFF_MODEL_PARAMETERS_ROOT_DIR) //"runoffModelParametersRootDir"
            || !properties.containsKey(FfgConstants.RUNOFF_MODEL_STATES_ROOT_DIR)) //"runoffModelStatesRootDir"
        {//property "printDebugInfo" or "floodFlowId" is optional AND only when running SNOW-17, "snowModelParametersRootDir" & "snowModelStatesRootDir" are required

            throw new Exception("The properties in run_info.xml is missing one of the required elements: ["
                + FfgConstants.FFG_PARAMETER_FILE_NAME + ", " + FfgConstants.RUNOFF_MODEL_PARAMETERS_ROOT_DIR + ", "
                + FfgConstants.RUNOFF_MODEL_STATES_ROOT_DIR + "]");
        }

        if(((FfgParameters)getParameters()).runSnow17())
        {// only when running SNOW-17, "snowModelParametersRootDir" & "snowModelStatesRootDir" are required

            if(!properties.containsKey(FfgConstants.SNOW_MODEL_PARAMETERS_ROOT_DIR) //"snowModelParametersRootDir"
                || !properties.containsKey(FfgConstants.SNOW_MODEL_STATES_ROOT_DIR))//"snowModelStatesRootDir"
            {
                throw new Exception("To run SNOW17, either " + FfgConstants.SNOW_MODEL_PARAMETERS_ROOT_DIR + " or "
                    + FfgConstants.SNOW_MODEL_STATES_ROOT_DIR
                    + " is missing in the properties section in run_info.xml.");
            }
        }

        if(getInitialStateTime() < getSwitchFromObservedToForecastTime())
        {
            final String runStart = DateTime.getDateTimeStringFromLong(getInitialStateTime(), null);
            final String lstcmpdy = DateTime.getDateTimeStringFromLong(getSwitchFromObservedToForecastTime(), null);
            _logger.log(Logger.WARNING, "The run start date and time[" + runStart
                + "] is before the current date and time[" + lstcmpdy
                + "]. FFG does not use the data before the current date and time.");
        }
    }

    /**
     * Parse input time series files and set each area's MAT ts location id to its parent directory name, which is area
     * id. Assuming each input time series file contains only one time series. So _runInfo.getInputTimeSeriesFileList()
     * and getTsList() have equal size and same sequence.<br>
     * Also set _dischargeTs if it exists.
     */
    private void parseInputTimeSeriesFiles() throws Exception
    {
        final FewsXMLParser xmlParser = new FewsXMLParser(_logger);

        xmlParser.parseTimeSeries(_runInfo.getInputTimeSeriesFileList(),
                                  getTsList(),
                                  getAreMissingValuesAlwaysAllowed(),
                                  getInitialStateTime());

        if(_runInfo.getInputTimeSeriesFileList().size() != getTsList().size())
        {
            throw new Exception("For FFG program, each input time series file should only contains one time series");
        }

        String parentDir = "";
        
        for(int i = 0; i < _runInfo.getInputTimeSeriesFileList().size(); i++)
        {

            final RegularTimeSeries rts = getTsList().get(i);

            final String tsType = rts.getTimeSeriesType();

            if(tsType.equalsIgnoreCase(DataType.MAT_DATATYPE))
            {

                final String fileName = _runInfo.getInputTimeSeriesFileList().get(i);

                final String[] path = fileName.split("/"); //both in linux and Windows, the file name is separated by "/", so hard-coded here

                parentDir = path[path.length - 2]; //second String from the last

                rts.setLocationId(parentDir);
            }
            else if(DISCHARGE_DATATYPE.isDischargeType(tsType))
            {
                _dischargeTs = (FewsRegularTimeSeries)rts;
                _dischargeTslocationIdString = _dischargeTs.getLocationId();
            }
        } //close for loop
                        
        if(_dischargeTslocationIdString.isEmpty() == false)
        {//adding locationId to prepend string in logging; if location id information is not available from input TSs, then don't bother

            final String logPrependStr = ((Diagnostics)_logger).getPrependStr() + " LocationID("
                + _dischargeTslocationIdString + ")";

            ((Diagnostics)_logger).setPrependStr(logPrependStr);
        }
        
    }

    /**
     * Parse the parameter xml file specified in the run info properties, extract values from map. Also set some
     * variables to avoid frequent access.
     */
    private void parseParams() throws Exception
    {
        final FewsXMLParser xmlParser = new FewsXMLParser(_logger);

        final String paramsFile = _runInfo.getProperties().getProperty(FfgConstants.FFG_PARAMETER_FILE_NAME);

        xmlParser.parseParameters(paramsFile, (Parameters)super._parameters);

        ((FfgParameters)_parameters).extractValuesFromMapFFG();

        _outputLocationId = ((FfgParameters)_parameters).getFfgLocationId();

        _areaIdArray = ((FfgParameters)_parameters).getFfgAreaIdArray();
    }

    /**
     * Returns a map: key is the area id, value is the list of FfgRainfallRunoffCurve for various durations. For Gridded
     * FFG, there is only one area.<br>
     * This procedure is equivalent to NWSRFS FCST and this is common for both FFH and Gridded FFG. It may run
     * SNOW17(optional), then one rainfall runoff model(SACSMA or API-Cont).
     */
    protected Map<String, List<FfgRainfallRunoffCurve>> getRainFallRunoffCurveMap() throws Exception
    {
        final FfgRainFallRunoffCurveGenerator ffgRainFallRunoffCurveGenerator = new FfgRainFallRunoffCurveGenerator(this);

        return ffgRainFallRunoffCurveGenerator.calculateAndReturnFfgRainFallRunOffCurveMap();
    }

}
