package ohd.hseb.ohdutilities.ffg;

import java.util.Arrays;

import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.ohdutilities.UtilityParameters;

public abstract class FfgParameters extends UtilityParameters
{
    private FFG_HIGH_FLOW_ADJUST_OPTION _highFlowAdjustOption = FFG_HIGH_FLOW_ADJUST_OPTION.NO_ADJUSTMENT_AS_0; //default 

    private FFG_RUNOFF_ADJUST_OPTION _runOffAdjustOption = FFG_RUNOFF_ADJUST_OPTION.NO_ADJUSTMENT_AS_0; //default

    private double _ffgPercentImpervious = 0.0; //default
    protected String _ffgLocationId;
    protected String[] _ffgAreaIdArray;
    protected double[] _ffgAreaWeightArray;

    protected int[] _modelIntervalArray;

    private boolean _ffgUserCtrlCheckDecreasingFFG;

    /**
     * This method replace the implementation of the abstract method extractValuesFromMap. We will be manually calling
     * the extractValuesFromMapFFG method in the Driver as the parser parameters could be from different models. We need
     * to be sure that we only call this method when the parameter object is FFG. As extractValuesFromMap is called
     * after parsing the parameter xml file and we can not control the kind of xml file being parse we can not assume
     * that the extracted values will be from FFG.
     * 
     * @throws Exception
     */
    protected void extractValuesFromMapFFG() throws Exception
    {

        //user control parameters are required to be present in params xml file.
        final boolean ffgUserCtrlHighFlowAdjust = getBooleanDataParameter(FfgConstants.FFG_USER_CTRL_HIGH_FLOW_ADJUST_TAG); //default value is true
        final boolean ffgUserCtrlRunoffAdjust = getBooleanDataParameter(FfgConstants.FFG_USER_CTRL_RUNOFF_ADJUST_TAG); //default value is true

        _ffgUserCtrlCheckDecreasingFFG = getBooleanDataParameter(FfgConstants.FFG_USER_CTRL_CHECK_DECREASING_FFG_TAG); //default value is false

        _ffgLocationId = super.getStringDataParameter(FfgConstants.FFG_LOCATION_ID_TAG);

        if(isParamExisting(FfgConstants.FFG_PERCENT_IMPERVIOUS_TAG))
        {
            _ffgPercentImpervious = super.getDoubleDataParameter(FfgConstants.FFG_PERCENT_IMPERVIOUS_TAG);
        }

        /*--------------set high flow adjust parameter -------------------*/
        if(isParamExisting(FFG_HIGH_FLOW_ADJUST_OPTION.paramName()))
        {
            final String highFlowAdjustOption = super.getStringDataParameter((FFG_HIGH_FLOW_ADJUST_OPTION.paramName()));
            _highFlowAdjustOption = FFG_HIGH_FLOW_ADJUST_OPTION.convertFromString(highFlowAdjustOption);

            _logger.log(Logger.DEBUG, FFG_HIGH_FLOW_ADJUST_OPTION.paramName() + " is " + highFlowAdjustOption);

            if(ffgUserCtrlHighFlowAdjust == false)
            {
                _highFlowAdjustOption = FFG_HIGH_FLOW_ADJUST_OPTION.NO_ADJUSTMENT_AS_0;

                _logger.log(Logger.DEBUG, FfgConstants.FFG_USER_CTRL_HIGH_FLOW_ADJUST_TAG + " is false, so "
                    + FFG_HIGH_FLOW_ADJUST_OPTION.paramName() + " is turned off.");
            }

        }
        else
        {
            _logger.log(Logger.DEBUG, FFG_HIGH_FLOW_ADJUST_OPTION.paramName() + " is absent. The default is "
                + FFG_HIGH_FLOW_ADJUST_OPTION.NO_ADJUSTMENT_AS_0);
        }

        /*--------------set runoff adjust parameter -------------------*/
        if(isParamExisting(FFG_RUNOFF_ADJUST_OPTION.paramName()))
        {
            final String runOffAdjustOption = super.getStringDataParameter(FFG_RUNOFF_ADJUST_OPTION.paramName());
            _runOffAdjustOption = FFG_RUNOFF_ADJUST_OPTION.convertFromString(runOffAdjustOption);

            _logger.log(Logger.DEBUG, FFG_RUNOFF_ADJUST_OPTION.paramName() + " is " + runOffAdjustOption);

            if(ffgUserCtrlRunoffAdjust == false)
            {
                _runOffAdjustOption = FFG_RUNOFF_ADJUST_OPTION.NO_ADJUSTMENT_AS_0;

                _logger.log(Logger.DEBUG, FfgConstants.FFG_USER_CTRL_RUNOFF_ADJUST_TAG + " is false, so "
                    + FFG_RUNOFF_ADJUST_OPTION.paramName() + " is turned off.");
            }
        }
        else
        {
            _logger.log(Logger.DEBUG, FFG_RUNOFF_ADJUST_OPTION.paramName() + " is absent. The default is "
                + FFG_RUNOFF_ADJUST_OPTION.NO_ADJUSTMENT_AS_0);
        }
    }

    /**
     * Based on the passed in String, set {@link #_modelIntervalArray}.
     */
    protected void setModelDurationInterval(final String durStr) throws Exception
    {
        //now set model interval array
        if(durStr.equalsIgnoreCase(FfgConstants.FFG_DURATION_1_3_6_HR_TAG))
        {
            _modelIntervalArray = OHDConstants.FFG_MODEL_1_3_6_INTERVAL_ARRAY; //1, 3, 6 HR model interval
        }
        else if(durStr.equalsIgnoreCase(FfgConstants.FFG_DURATION_1_3_6_12_HR_TAG))
        {
            _modelIntervalArray = OHDConstants.FFG_MODEL_1_3_6_12_INTERVAL_ARRAY; //1, 3, 6, 12 HR model interval
        }
        else if(durStr.equalsIgnoreCase(FfgConstants.FFG_DURATION_1_3_6_12_24_HR_TAG))
        {
            _modelIntervalArray = OHDConstants.FFG_MODEL_1_3_6_12_24_INTERVAL_ARRAY; //1, 3, 6, 12, 24 HR model interval
        }
        else
        {//cannot recognized

            throw new Exception("The parameter value of " + FfgConstants.FFG_DURATION_TAG + " is " + durStr
                + " and it cannot be recognized.");

        }

        _logger.log(Logger.DEBUG, "Model intervals have: " + Arrays.toString(_modelIntervalArray) + " HRs.");

    }

    /**
     * Return parameter value {@link FfgConstants#FFG_HIGH_FLOW_ADJUST_TAG} as one of the enum types in
     * {@link FFG_HIGH_FLOW_ADJUST_OPTION}.
     * 
     * @return - one of the enum types in {@link FFG_HIGH_FLOW_ADJUST_OPTION}.
     */
    public FFG_HIGH_FLOW_ADJUST_OPTION getHightFlowAdjustOption()
    {
        return _highFlowAdjustOption;
    }

    public String[] getFfgAreaIdArray()
    {
        return _ffgAreaIdArray;
    }

    /**
     * Returns the parameter {@link FfgConstants#FFG_RUNOFF_ADJUST_TAG} as one of the enum types in
     * {@link FFG_RUNOFF_ADJUST_OPTION}.
     * 
     * @return - one of the enum types in {@link FFG_RUNOFF_ADJUST_OPTION}.
     */
    public FFG_RUNOFF_ADJUST_OPTION getRunOffAdjustOption()
    {
        return _runOffAdjustOption;
    }

    /**
     * Return an double array containing the percentage of each area. The number of array elements equal to the number
     * of areas in FFG. The numbers have been converted to decimal double values. No need to be divided by 100.0
     * anymore.
     */
    public double[] getFfgAreaWeightArray()
    {
        return _ffgAreaWeightArray;
    }

    /**
     * Retrieve the optional parameter {@link FfgConstants#FFG_SNOW_MODEL_NAME_TAG}. If it does not exist, returns null
     * String. If it exists, it can only be {@link FfgConstants#SNOW_MODEL_NAME}.
     */
    public String getFfgSnowModelName(final String areaId) throws Exception
    {
        String model = null;

        if(isParamExistingInGroup(areaId, FfgConstants.FFG_SNOW_MODEL_NAME_TAG))
        {
            model = getStringDataParameter(areaId, FfgConstants.FFG_SNOW_MODEL_NAME_TAG);
        }

        return model;
    }

    /**
     * Retrieve the parameter {@link FfgConstants#FFG_RAINFALL_RUNOFF_MODEL_NAME_TAG}. The value can only be
     * {@link FfgConstants#API_CONT_MODEL_NAME} or {@link FfgConstants#SAC_MODEL_NAME}.
     */
    public String getFfgRainfallRunoffModelName(final String areaId) throws Exception
    {
        return getStringDataParameter(areaId, FfgConstants.FFG_RAINFALL_RUNOFF_MODEL_NAME_TAG);
    }

    /**
     * If one area runs SNOW-17, returns TRUE; if none of the areas runs SNOW-17, returns FALSE.
     */
    boolean runSnow17() throws Exception
    {
        for(final String areaId: _ffgAreaIdArray)
        {
            if(this.getFfgSnowModelName(areaId) != null)
            {
                return true;
            }
        }

        return false;
    }

    public double getFfgMinimumThresholdRunOff(final String areaId) throws Exception
    {
        double minR = FfgConstants.FFG_MIN_THRESHOLD_RUNOFF_DEFAULT_VALUE; //initialized to default value

        if(isParamExistingInGroup(areaId, FfgConstants.FFG_MINIMUM_THRESHOLD_RUNOFF_TAG))
        {

            final double minR2 = super.getDoubleDataParameter(areaId, FfgConstants.FFG_MINIMUM_THRESHOLD_RUNOFF_TAG); //if it is 0.0 or negative, discard it, 

            if(minR2 > 0.)
            {//the value is a valid value

                minR = minR2;
            }
            else
            {
                _logger.log(Logger.DEBUG, FfgConstants.FFG_MINIMUM_THRESHOLD_RUNOFF_TAG + " parameter for area "
                    + areaId + " is less than or equal to 0.0. So use the default value "
                    + FfgConstants.FFG_MIN_THRESHOLD_RUNOFF_DEFAULT_VALUE);
            }
        }
        else
        {
            _logger.log(Logger.DEBUG, FfgConstants.FFG_MINIMUM_THRESHOLD_RUNOFF_TAG
                + " parameter does not exist for area " + areaId + ". So use the default value "
                + FfgConstants.FFG_MIN_THRESHOLD_RUNOFF_DEFAULT_VALUE);
        }

        return minR;

    }

    public double getFfgMaximumThresholdRunOff(final String areaId) throws Exception
    {
        double maxR = FfgConstants.FFG_MAX_THRESHOLD_RUNOFF_DEFAULT_VALUE;

        if(isParamExistingInGroup(areaId, FfgConstants.FFG_MAXIMUM_THRESHOLD_RUNOFF_TAG))
        {
            final double maxR2 = super.getDoubleDataParameter(areaId, FfgConstants.FFG_MAXIMUM_THRESHOLD_RUNOFF_TAG);

            if(maxR2 > 0.)
            {
                maxR = maxR2;
            }
            else
            {
                _logger.log(Logger.DEBUG, FfgConstants.FFG_MAXIMUM_THRESHOLD_RUNOFF_TAG + " parameter for area "
                    + areaId + " is less than or equal to 0.0. So use the default value "
                    + FfgConstants.FFG_MAX_THRESHOLD_RUNOFF_DEFAULT_VALUE);
            }
        }
        else
        {
            _logger.log(Logger.DEBUG, FfgConstants.FFG_MAXIMUM_THRESHOLD_RUNOFF_TAG
                + " parameter does not exist for area " + areaId + ". So use the default value "
                + FfgConstants.FFG_MAX_THRESHOLD_RUNOFF_DEFAULT_VALUE);
        }

        return maxR;
    }

    /**
     * Returns the array of Intensity values, which is needed only when {@link #getRunOffAdjustOption()} is equivalent
     * to 1(the Intensity values are multiplier to runoff), 2 (the Intensity values used as ffg) or 5(adjust ffg -- for
     * Gridded FFG only).
     * 
     * @throws Exception
     */
    public double[] getIntensityArray() throws Exception
    {
        final double hinten[] = new double[_modelIntervalArray.length]; //initialized to 0.0 values by default
        
                
        if(getRunOffAdjustOption() == FFG_RUNOFF_ADJUST_OPTION.ADJUST_RUNOFF_AS_1
            || getRunOffAdjustOption() == FFG_RUNOFF_ADJUST_OPTION.USE_FIELDS_AS_FFG_AS_2
            || getRunOffAdjustOption() == FFG_RUNOFF_ADJUST_OPTION.ADJUST_FFG_AS_5)
        {

            for(int i = 0; i < hinten.length; i++)
            {
                hinten[i] = getDoubleDataParameter(FfgConstants.FFG_INTENSITY_TAGS[i]);
            }
        }

        return hinten;
    }

    /**
     * Return the array of model intervals.
     */
    public int[] getModelIntervalArray() throws Exception
    {
        return _modelIntervalArray;
    }

    public String getFfgLocationId()
    {
        return _ffgLocationId;
    }

    public double getFfgPercentImpervious()
    {
        return _ffgPercentImpervious;
    }

    /**
     * Returns an integer array with 3, 4 or 5 elements, depending on FFG_DURATION(e.g. {@link #_modelIntervalArray}.
     * This array is only available when {@link #getHightFlowAdjustOption()} is not equivalent to legacy 0, 3 or 4
     * value. Otherwise, throw an Exception.<br>
     * Message the data in case the converter didn't do it.
     */
    final int[] getAdjFlowTimeArrayInHours() throws Exception
    {
        if(adjustHighFlow() == false //
            || getHightFlowAdjustOption() == FFG_HIGH_FLOW_ADJUST_OPTION.HIGHEST_FORECAST_FLOW_IN_TIMESERIES_AS_3)
        {
            throw new Exception("The parameter " + FFG_HIGH_FLOW_ADJUST_OPTION.paramName() + " value is "
                + getHightFlowAdjustOption() + ". So FFG_ADJUST_FLOW_TIME is not available.");
        }

        //if reach here, HIGH FLOW ADJUST USER CONTROL is true and HIGH FLOW ADJUST OPTION is 1 or 2
        final int taq[] = new int[_modelIntervalArray.length];

        for(int i = 0; i < taq.length; i++)
        {
            taq[i] = super.getIntDataParameter(FfgConstants.FFG_ADJUST_FLOW_TIME_TAGS[i]);
        }

        //message the data based on the rules of default. These message supposed to be done by converter. But, re-do it here just in case.
        if(taq[0] < 1.0)
        {
            taq[0] = 12; //Time to adjust flow for 1-hour duration - default 12 hours
        }

        //Time to adjust flow for 3-hour and other duration - default is TAQ1
        for(int i = 1; i < taq.length; i++)
        {
            if(taq[i] < 1.0)
            {
                taq[i] = taq[0];
            }
        }

        return taq;
    }

    /**
     * Returns false when "FFG_HIGH_FLOW_ADJUST_OPTION" is "NO_ADJUSTMENT" or "REDUCE_RUNOFF_BY_STORM_RUNOFF".
     */
    public boolean adjustHighFlow()
    {
        if(getHightFlowAdjustOption() == FFG_HIGH_FLOW_ADJUST_OPTION.NO_ADJUSTMENT_AS_0 //
            || getHightFlowAdjustOption() == FFG_HIGH_FLOW_ADJUST_OPTION.REDUCE_RUNOFF_BY_STORM_RUNOFF_AS_4)
        {
            return false;
        }

        return true;

    }

    public boolean isFfgUserCtrlCheckDecreasingFFG()
    {
        return _ffgUserCtrlCheckDecreasingFFG;
    }
}
