package ohd.hseb.ohdutilities.ffg;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import ohd.hseb.ohdutilities.ffg.model.FFGRainfallRunoffModelController;
import ohd.hseb.ohdutilities.ffg.model.FFGSnowModelController;
import ohd.hseb.util.DataPoint;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.Diagnostics;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.OHDUtilities;
import ohd.hseb.util.misc.HString;

/**
 * This class generates rainfall_runoff curves for individual durations(1HR, 3HR, 6HR etc) for each area. The model(s)
 * to be run include optional SNOW-17 model and the rainfall runoff model(SAC-SMA or API-Cont). If running SNOW-17
 * model, its output(rain plus melt) is passed to the rainfall runoff model.<br>
 * By repeated looping with various precip value, the precips which generate
 * {@link FfgConstants#FFG_MINIMUM_THRESHOLD_RUNOFF_TAG}(1HR interval) and
 * {@link FfgConstants#FFG_MAXIMUM_THRESHOLD_RUNOFF_TAG}(the longest interval, e.g. 6, 12 or 24HR) are
 * identified(precipA for minimum threshold runoff, precipB for maximum threshold runoff). precipA +
 * 0.3(precipB-precipA) generates 2nd point; precipA + 0.6(precipB-precipA) generates 3rd point; precipB + 0.1 IN
 * generates 4th point.
 */
class FfgRainFallRunoffCurveGenerator
{
    private final FFGSnowModelController _snowModelController;
    private final FFGRainfallRunoffModelController _rainFallModelController;

    private final FfgParameters _ffgParams;
    private final Logger _logger;
	private FFGDriver _ffgDriver;

    FfgRainFallRunoffCurveGenerator(final FFGDriver ffgDriver) throws Exception
    {
       
    	_ffgDriver = ffgDriver;
    	_rainFallModelController = new FFGRainfallRunoffModelController(ffgDriver);

        _snowModelController = new FFGSnowModelController(ffgDriver);

        _ffgParams = (FfgParameters)ffgDriver.getParameters();

        _logger = ffgDriver.getLogger();
    }

    /**
     * Calculates each area's FFG Rainfall Runoff Curve for individual duration(1HR, 3Hr, 6HR etc). For Gridded FFG, it
     * only has one area.
     */
    Map<String, List<FfgRainfallRunoffCurve>> calculateAndReturnFfgRainFallRunOffCurveMap() throws Exception
    {
    	// variables for FogBugz 1616 change
    	String currentPrePendString = ((Diagnostics)_logger).getPrependStr();
    	String newPrependString = "" ;
    	// end
    	
    	final String[] areaIdArray = _ffgParams.getFfgAreaIdArray(); //stores each area Id running by FFH

        final Map<String, List<FfgRainfallRunoffCurve>> rainfallRunoffCurveMap = new HashMap<String, List<FfgRainfallRunoffCurve>>(areaIdArray.length);

        final int[] modelIntervalArray = _ffgParams.getModelIntervalArray();      
        
        for(int areaIdIndex = 0; areaIdIndex < areaIdArray.length; areaIdIndex++)
        {
            final String areaId = areaIdArray[areaIdIndex];

            
            // for FogBugz 1616: Prepend logs with location info
            if (areaIdIndex==0){
            	newPrependString = currentPrePendString+"("+areaId+")";                	
            }
            else{                	            	   
        	   newPrependString = HString.replaceSubstring(currentPrePendString, areaIdArray[areaIdIndex-1], areaIdArray[areaIdIndex]);
            }                
            // only add another id if not equal to the default id                               
            if(! areaId.equals(_ffgDriver._dischargeTslocationIdString)){	                
            	((Diagnostics)_logger).setPrependStr(newPrependString);
                 currentPrePendString= newPrependString;
            }
             
            // end for FogBugz 1616
            
            final List<FfgRainfallRunoffCurve> curveListInMap = new ArrayList<FfgRainfallRunoffCurve>();

            rainfallRunoffCurveMap.put(areaId, curveListInMap);

            _logger.log(Logger.DEBUG, "Calculating the RainFall Runoff Curve for the area: " + areaId);

            //the controller will decide to run SNOW model or not
            _snowModelController.initControllerForSegment(areaId);

            //the controller will decide to run SAC-SMA or API-Cont
            _rainFallModelController.initControllerForSegment(areaId);

            /*----------- p1 & p4 calculate min precip and max precip ---------*/
            final FfgRainfallRunoffCurve curve = new FfgRainfallRunoffCurve();
            curve.setMinThresholdRunoffAndGuessedPrecip(_ffgParams.getFfgMinimumThresholdRunOff(areaId));
            this.calculateExpectedPrecipForTargetRunoff(curve.getP1(), 1, 0.0, 0.0, 1 * OHDConstants.MM_PER_INCH);

            curve.setMaxThresholdRunoffAndGuessedPrecip(_ffgParams.getFfgMaximumThresholdRunOff(areaId));
            this.calculateExpectedPrecipForTargetRunoff(curve.getP4(),
                                                        modelIntervalArray[modelIntervalArray.length - 1],
                                                        curve.getP1().getX() * OHDConstants.MM_PER_INCH,
                                                        0.0,
                                                        2.0 * OHDConstants.MM_PER_INCH);

            _logger.log(Logger.DEBUG, "precipA(IN)=" + curve.getPrecipA() + " precipB(IN, after incrementing 0.1 IN)="
                + curve.getPrecipB());

            curve.setPrecipValuesBasedOnPrecipAB();
            //now, all 4 points' X values(precip) have been set, their Y values(runoff) need to be calculated

            for(int durationIndex = 0; durationIndex < modelIntervalArray.length; durationIndex++)
            {
                final FfgRainfallRunoffCurve curveAtOneDuration = curve.clone();

                //generates FFG Rainfall/Runoff Curve at this model interval
                generateFfgRainfallRunoffCurveAtInterval(curveAtOneDuration, modelIntervalArray[durationIndex]);

                curveListInMap.add(curveAtOneDuration);

                _logger.log(Logger.DEBUG, "Area=" + areaId + ", model interval=" + modelIntervalArray[durationIndex]
                    + "HR. The FFG RainFall Runoff Curve: ");
                _logger.log(Logger.DEBUG, curveAtOneDuration.toString());
            }
        }

        return rainfallRunoffCurveMap;
    }

    /**
     * Looping through the model(s), potential SNOW-17 and the rainfall runoff model(SAC-SMA or API-Cont) with various
     * input precipitation amount(unit of MM), to try to find the "correct" precipitation which produces the expected
     * runoff(point's Y value). At the beginning of the call, point's X value(precipitation) is the first guessed value.
     * At the end of the method, its X value is the correct precipitation.
     * 
     * @param point - its X value is precipitation(unit of INCH), its Y value is the expected runoff value(unit of
     *            INCH).
     * @param modelTimeInterval - the model interval
     * @param rmin - low limit of the precip range, used for narrowing the precip. same as in ex32.f
     * @param rmax - up limit of the precip range, used for narrowing the precip. same as in ex32.f
     * @param rinc - increment step, used for getting a new precip. same as in ex32.f
     */
    private void calculateExpectedPrecipForTargetRunoff(final DataPoint point,
                                                        final int modelTimeInterval,
                                                        double rmin, //MM
                                                        double rmax, //MM
                                                        final double rinc) throws Exception
    {

        double precipAmount = point.getX() * OHDConstants.MM_PER_INCH; //first guessed precip

        int i = 1;
        //the following code based on ex32.f
        while(true)
        {

            if(i > 20) //MXITER = 20
            {
                _logger.log(Logger.DEBUG,
                            "Jump out of the looping of SAC-SMA model after 100 loops. The expected runnoff has not been achieved yet.");
                break;
            }

            /* ----------------calculate runoff by model or models in the time step---------------- */
            final double tci = getTciBasedOnPrecip(precipAmount, modelTimeInterval);

            final double rdif = tci - point.getY() * OHDConstants.MM_PER_INCH; //in unit of MM  RDIF=ROFF - RO
            _logger.log(Logger.DEBUG,
                        "RAINE(Inch)="
                            + String.valueOf(OHDUtilities.getFortranPrecison(precipAmount * OHDConstants.INCHES_PER_MM))
                            + " ROFF(Inch)="
                            + String.valueOf(OHDUtilities.getFortranPrecison(tci * OHDConstants.INCHES_PER_MM))
                            + " TARG RO(Inch)=" + point.getY());

            if(Math.abs(rdif) < 0.254) //MM
            {
                break; //jump out of while(true) loop
            }

            if(rdif > 0.0)
            {
                /*
                 * RUNOFF GREATER THAN TARGETED THRESHOLD RUNOFF...SET RMAX AND USE BISECTION
                 */
                rmax = precipAmount;

                precipAmount = (rmin + rmax) / 2.0;

                if(Math.abs(rmax - rmin) < 0.000254)
                {
                    break;
                }

            }
            else if(Math.abs(rdif) <= rinc)
            {
                /*
                 * RUNOFF LESS THAN MINIMUM THRESHOLD RUNOFF...SET RMIN AND USE BISECTION
                 */

                rmin = precipAmount;

                if(rmax <= 0.00254)
                {
                    precipAmount += rinc / 2.0;
                }

                if(rmax > 0.0)
                {
                    precipAmount = (rmin + rmax) / 2.0;
                }

                if(Math.abs(rmax - rmin) < 0.000254)
                {
                    break;
                }
            }
            else
            {
                /* RUNOFF LESS THAN MINIMUM THRESHOLD RUNOFF ... INCREMENT BY RINC INCH */
                rmin = precipAmount;
                precipAmount += rinc;
            }

            i++;

        } //close while loop

        point.setX(precipAmount * OHDConstants.INCHES_PER_MM);
    }

    /**
     * Generates FFG Rainfall Runoff(Y axis, unit of Inch) versus Rainfall(X axis, unit of Inch) curve at this model
     * interval(HR). All the 4 points' X values(Rainfall, e.g. precip) have been pre-set. This method gets their
     * corresponding Y values(Runoff, e.g. TCI).
     * 
     * @param modelInterval - HR
     * @throws Exception
     */
    private void generateFfgRainfallRunoffCurveAtInterval(final FfgRainfallRunoffCurve curve, final int modelInterval) throws Exception
    {

        this.setPointRunoffBasedOnPrecip(curve.getP1(), modelInterval);
        this.setPointRunoffBasedOnPrecip(curve.getP2(), modelInterval);
        this.setPointRunoffBasedOnPrecip(curve.getP3(), modelInterval);
        this.setPointRunoffBasedOnPrecip(curve.getP4(), modelInterval);

        //now, the curve for this modelInterval is generated
        _logger.log(Logger.DEBUG, "model interval=" + modelInterval);
        _logger.log(Logger.DEBUG, curve.toString());
    }

    /**
     * The point x value(precip, unit of Inch) has been set. This method sets its Y value(Runoff, unit of Inch).
     * 
     * @param point
     * @param modelInterval
     * @throws Exception
     */
    private void setPointRunoffBasedOnPrecip(final DataPoint point, final int modelInterval) throws Exception
    {
        final double runoff = this.getTciBasedOnPrecip(point.getX() * OHDConstants.MM_PER_INCH, modelInterval);
        point.setY(runoff * OHDConstants.INCHES_PER_MM);

    }

    /**
     * Calculate the runoff(e.g. TCI, unit of MM) with the input precipitation(MM) by using SAC or SNOW/SAC or API-CONT
     * or SNOW/API-CONT, at the time step set by model interval.
     * 
     * @param precip - in unit of MM
     * @return runoff(TCI) in unit of MM
     */
    private double getTciBasedOnPrecip(double precip, final int modelInterval) throws Exception
    {
        double generatedTci = 0.0;

        precip = _snowModelController.getSnowRaim(precip, modelInterval); //change precip due to SNOW model

        _logger.log(Logger.DEBUG, "RainFall Model Input: RAIM(mm)=" + precip);
        generatedTci = _rainFallModelController.getRunoff(precip, modelInterval);
        _logger.log(Logger.DEBUG,
                    "RainFall Model Output: ROFF=" + generatedTci + "[MM] or "
                        + String.valueOf(generatedTci * OHDConstants.INCHES_PER_MM) + "[INCH]");

        return generatedTci; //MM
    }

}
