package ohd.hseb.ohdutilities.ffg;

import ohd.hseb.util.DataPoint;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.OHDUtilities;

/**
 * FFG Rainfall Runoff(Y axis, unit of Inch) versus Rainfall(X axis, unit of Inch) curve at this model interval(HR). It
 * holds 4 DataPoint objects.
 */
public class FfgRainfallRunoffCurve
{
    //the following 4 points Y values are runoff(INCH), X values are precip(INCH).
    private DataPoint _p1 = new DataPoint(); //lowest end
    private DataPoint _p2 = new DataPoint(); //0.3 of precip range
    private DataPoint _p3 = new DataPoint(); //0.6 of precip range
    private DataPoint _p4 = new DataPoint(); //highest end

    public FfgRainfallRunoffCurve()
    {
        //empty;
    }

    /**
     * Set the first point the minimal threshold runoff(unit of Inch) and the first guessed precip as 1.0 inch.
     * 
     * @param minRunoff - unit of Inch
     */
    void setMinThresholdRunoffAndGuessedPrecip(final double minRunoff)
    {
        _p1.setY(minRunoff);
        _p1.setX(1.0); //set first guessed precip
    }

    /**
     * PrecipA should have been obtained at {@link #_p1} before calling this method. Set the last point the maximal
     * threshold runoff (unit of Inch) and the first guessed precip as precipA + 3.0 inch.
     */
    void setMaxThresholdRunoffAndGuessedPrecip(final double maxRunoff)
    {
        _p4.setY(maxRunoff);
        _p4.setX((int)(getPrecipA() + 3.0)); //set first guessed precip based on precipA
    }

    /**
     * Returns the precip(unit of Inch) that produces the runoff close to the minimal runoff within the error range.
     */
    double getPrecipA()
    {
        return _p1.getX();
    }

    /**
     * Returns the precip(unit of Inch) that produces the runoff close to the maximal runoff within the error range,
     * incremented by 0.1 inch.
     */
    double getPrecipB()
    {
        return _p4.getX() + 0.1; //increasing the max precip by 0.1 inch
    }

    /**
     * After precipA and precipB have been obtained, set each point's precip values based on the range.
     */
    void setPrecipValuesBasedOnPrecipAB()
    {
        final double precipA = getPrecipA();
        final double precipB = getPrecipB();

        _p4.setX(precipB);

        _p2.setX(_p1.getX() + 0.3 * (precipB - precipA));

        _p3.setX(_p1.getX() + 0.6 * (precipB - precipA));
    }

    /**
     * Use interpolation and extrapolation to get the corresponding precip value(FFH Value, unit of Inch) based on the
     * magicRunoff(unit of Inch) passed in.
     */
    public double getFFGValueByInterpolation(final double magicRunoff)
    {
        double ffhPrecip = 0.0;

        if(magicRunoff < _p1.getY())
        { //extrapolation at the lower end
            ffhPrecip = _p1.getX() - (_p1.getY() - magicRunoff) * (_p2.getX() - _p1.getX()) / (_p2.getY() - _p1.getY());

        }
        else if(magicRunoff > _p4.getY())
        { //extrapolation at the upper end
            ffhPrecip = _p4.getX() + (magicRunoff - _p4.getY()) * (_p4.getX() - _p3.getX()) / (_p4.getY() - _p3.getY());
        }
        else
        { //magicRunoff in the range of the curve. interpolation

            final DataPoint leftPoint;
            final DataPoint rightPoint;

            if(magicRunoff > _p3.getY())
            {
                leftPoint = _p3;
                rightPoint = _p4;
            }
            else if(magicRunoff > _p2.getY())
            {
                leftPoint = _p2;
                rightPoint = _p3;
            }
            else
            {
                leftPoint = _p1;
                rightPoint = _p2;
            }

            //interpolate: x = [(x2-x1)/(y2-y1)]*(y-y1)+x1 
            ffhPrecip = ((rightPoint.getX() - leftPoint.getX()) / (rightPoint.getY() - leftPoint.getY()))
                * (magicRunoff - leftPoint.getY()) + leftPoint.getX();

        }

        return ffhPrecip;
    }

    DataPoint getP1()
    {
        return _p1;
    }

    DataPoint getP2()
    {
        return _p2;
    }

    DataPoint getP3()
    {
        return _p3;
    }

    DataPoint getP4()
    {
        return _p4;
    }

    @Override
    public FfgRainfallRunoffCurve clone()
    {
        final FfgRainfallRunoffCurve curveClone = new FfgRainfallRunoffCurve();

        curveClone._p1 = _p1.clone();
        curveClone._p2 = _p2.clone();
        curveClone._p3 = _p3.clone();
        curveClone._p4 = _p4.clone();

        return curveClone;
    }

    @Override
    public String toString()
    {
        final StringBuffer strBuffer = new StringBuffer();

        strBuffer.append("p1: ").append(OHDUtilities.getFortranPrecison(_p1.getX())).append(OHDConstants.NEW_LINE);
        strBuffer.append("p1: ").append(OHDUtilities.getFortranPrecison(_p1.getY())).append(OHDConstants.NEW_LINE);
        strBuffer.append("p2: ").append(OHDUtilities.getFortranPrecison(_p2.getX())).append(OHDConstants.NEW_LINE);
        strBuffer.append("p2: ").append(OHDUtilities.getFortranPrecison(_p2.getY())).append(OHDConstants.NEW_LINE);
        strBuffer.append("p3: ").append(OHDUtilities.getFortranPrecison(_p3.getX())).append(OHDConstants.NEW_LINE);
        strBuffer.append("p3: ").append(OHDUtilities.getFortranPrecison(_p3.getY())).append(OHDConstants.NEW_LINE);
        strBuffer.append("p4: ").append(OHDUtilities.getFortranPrecison(_p4.getX())).append(OHDConstants.NEW_LINE);
        strBuffer.append("p4: ").append(OHDUtilities.getFortranPrecison(_p4.getY())).append(OHDConstants.NEW_LINE);

        return strBuffer.toString();

    }
}
