package ohd.hseb.ohdutilities.ffg;

import java.util.Arrays;

import javax.management.timer.Timer;

import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.ohdutilities.ffg.ffh.FFHParameters;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.FewsRegularTimeSeries;

public class FfgThreshRCalculator
{
    private final FfgParameters _ffgParams;
    private final FewsRegularTimeSeries _dischargeTs;
    private final long _lastComputedTimeInMillis; //lstcmdy

    private final Logger _logger;

    public FfgThreshRCalculator(final FfgParameters params,
                                final FewsRegularTimeSeries dischargeTs,
                                final long lastComputedTimeInMillis,
                                final Logger log)
    {
        _ffgParams = params;
        _dischargeTs = dischargeTs;
        _lastComputedTimeInMillis = lastComputedTimeInMillis;
        _logger = log;
    }

    /**
     * Translated from gethiq.f Fortran code. Returns highest forecast flow array(unit of CFS, one element for one model
     * interval). Returns null if threshold runoff is a function of static parameters(= flood flow / unit hydrograph
     * peak). Used by both FFH and Gridded FFG.
     */
    public double[] getMaxForecastFlowArrayInCFS() throws Exception
    {
        _logger.log(Logger.DEBUG, "High Flow Adjust Option=" + _ffgParams.getHightFlowAdjustOption().paramValue());

        if(_ffgParams.adjustHighFlow() == false)
        {//FFH threshold runoff will be calculated statically: = flood flow / unit hydrograph peak

            _logger.log(Logger.DEBUG, "No base flow.");
            return null;
        }

        //FFH threshold runoff will be calculated dynamically: = (flood flow - max forecast flow) / unit hydrograph peak
        final double[] baseFlow = new double[_ffgParams.getModelIntervalArray().length]; //initialize baseFlow for each duration to zero

        if(_ffgParams.getHightFlowAdjustOption() == FFG_HIGH_FLOW_ADJUST_OPTION.HIGHEST_FORECAST_FLOW_IN_TIMESERIES_AS_3)
        {
            //(3) determine highest forecast Q in time series
            final double maxForecastFlow = _dischargeTs.getMaxMeasurement().getValue(MeasuringUnit.cfs);

            for(int i = 0; i < baseFlow.length; i++)
            {
                baseFlow[i] = maxForecastFlow;
            }

        }
        else
        {//case 1 and 2, needs "Time to adjust flow for 1-hour", "3-hour" etc

            final int[] timeAdjFlowInHours = _ffgParams.getAdjFlowTimeArrayInHours();

            for(int i = 0; i < baseFlow.length; i++)
            {
                final long time = _lastComputedTimeInMillis + timeAdjFlowInHours[i] * Timer.ONE_HOUR;

                if(_ffgParams.getHightFlowAdjustOption() == FFG_HIGH_FLOW_ADJUST_OPTION.FORECAST_FLOW_AT_HOURS_AS_1)
                {
                    //(1) forecast Q at time equal to timeHighFlow[i] hours
                    baseFlow[i] = _dischargeTs.getMeasurementValueByTime(time, MeasuringUnit.cfs);
                }
                else if(_ffgParams.getHightFlowAdjustOption() == FFG_HIGH_FLOW_ADJUST_OPTION.HIGHEST_FORECAST_FLOW_OVER_NEXT_HOURS_AS_2)
                {

                    //(2) determine highest forecast Q over next timeHighFlow[i] hour
                    baseFlow[i] = _dischargeTs.getMaxMeasurement(_lastComputedTimeInMillis, time)
                                              .getValue(MeasuringUnit.cfs);
                }

            }//end for loop
        }

        //For debug purpose
        for(int i = 0; i < baseFlow.length; i++)
        {
            _logger.log(Logger.DEBUG, "baseFlow[" + i + "]=" + baseFlow[i]);
        }

        return baseFlow;
    }

    /**
     * Only used in FFH: calculate each magic THRESH-R(unit of INCH) for each model interval. Stores them in an array
     * and returns it. If unit hydrograph peak for this duration is negative, save it in the returned array to let
     * FFHDriver to handle the special case.
     */
    public double[] getFfhThreshRArrayInInch() throws Exception
    {

        double[] threshRArrayInInch = new double[_ffgParams.getModelIntervalArray().length];

        if(((FFHParameters)_ffgParams).isThreshRFromParams() == false)
        {//calculate the magic numbers: threshold runoff = flood flow / unit hydrograph peak or runoff = (flood flow - max forecast flow) / unit hydrograph peak

            if(_ffgParams.adjustHighFlow())
            {//runoff = (flood flow - max forecast flow) / unit hydrograph peak
                _logger.log(Logger.DEBUG, "When calculating FFH threshold runoff array, adjust high flow.");
            }
            else
            {//runoff = flood flow / unit hydrograph peak
                _logger.log(Logger.DEBUG, "When calculating FFH threshold runoff array, does not adjust high flow.");
            }

            // Get high(max) base (forecast) flow(unit of CFS) based on QINE timeseries for each duration
            final double[] maxForecastFlowInCFS = getMaxForecastFlowArrayInCFS();

            final double[] uhgPeakInCfsPerInch = ((FFHParameters)_ffgParams).getUHGPeakArrayInCfsPerInch();

            for(int i = 0; i < uhgPeakInCfsPerInch.length; i++)
            {
                if(uhgPeakInCfsPerInch[i] < 0.01) //check if it is negative or not. If negative, threshR is set to the negative value, which will be percentage of 3HR FFG value.
                {
                    threshRArrayInInch[i] = uhgPeakInCfsPerInch[i];
                    continue;
                }

                double q = ((FFHParameters)_ffgParams).getFlowAtFloodStage(); //bankfull flow [cfs]

                if(_ffgParams.adjustHighFlow())
                {//reduce the bankfull flow by the max flow

                    q -= maxForecastFlowInCFS[i]; //threshold runoff = (flood flow - max forecast flow) / unit hydrograph peak

                    if(q < 0.0)
                    {
                        q = 0.0;
                    }
                }

                threshRArrayInInch[i] = q / uhgPeakInCfsPerInch[i];
            }

        }
        else
        {//isThreshRFromParams() == true
            threshRArrayInInch = ((FFHParameters)_ffgParams).getThreshRunOffArrayInInch();
            _logger.log(Logger.DEBUG, "Read the threshold runoff values directly from the parameter xml file.");
        }

        _logger.log(Logger.DEBUG,
                    "The array of threshold runoff THRESH-R(INCH): " + Arrays.toString(threshRArrayInInch));

        return threshRArrayInInch;

    } //close method

}
