package ohd.hseb.ohdutilities.ffg.ffh;

import java.util.Arrays;
import java.util.List;
import java.util.Map;

import javax.management.timer.Timer;

import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.ohdutilities.ffg.FFGDriver;
import ohd.hseb.ohdutilities.ffg.FFG_RUNOFF_ADJUST_OPTION;
import ohd.hseb.ohdutilities.ffg.FfgConstants;
import ohd.hseb.ohdutilities.ffg.FfgRainfallRunoffCurve;
import ohd.hseb.ohdutilities.ffg.FfgThreshRCalculator;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.Diagnostics;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.misc.HString;

public class FFHDriver extends FFGDriver
{
    private final FFHParameters _ffhParams;

    private double[] _intensityArray;//Intensity values are optional record 4, only needed when Runoff Adjust Option is 1 or 2

    private boolean _skipComputation = false; //When thresholds in QINE header is missing value(-999.0) or negative value, skip model computation. Default is not skipping

    public FFHDriver()
    {
        super(); //calling FFGDriver constructor

        _ffhParams = new FFHParameters();

        super._parameters = _ffhParams;
    }

    @Override
    protected void execute() throws Exception
    {
    	
    	this.runDriverValidation();
    	
    	// variables for FogBugz 1616 change
    	String currentPrePendString = ((Diagnostics)_logger).getPrependStr();
    	String originalPrePendString = currentPrePendString;
    	String newPrependString = "" ;
    	// end

        final int[] modelIntervalArray = _ffhParams.getModelIntervalArray();

        final double[] ffgArray = new double[modelIntervalArray.length];

        if(_skipComputation == false)
        {
            //calculate threshR array 
            final FfgThreshRCalculator threshRCal = new FfgThreshRCalculator(_ffhParams,
                                                                             _dischargeTs,
                                                                             _runInfo.getRunLastObservationTimeLong(),
                                                                             _logger);

            final double[] threshRArrayInInch = threshRCal.getFfhThreshRArrayInInch(); //THRESH-R for different model time interval

            _logger.log(Logger.DEBUG, "number of weighted areas: " + _areaIdArray.length);

            final Map<String, List<FfgRainfallRunoffCurve>> rainFallRunoffCurveMap = getRainFallRunoffCurveMap();

            final double[] sumPrecipOfAreas = new double[ffgArray.length];
            //for multiple areas: 1st element is the sum of multiple areas' precip at 1HR model interval, 2nd element is 3HR, etc.

            final double[] minPrecipOfAreas = new double[ffgArray.length];//1st element is the smallest precip among the multiple areas' precip values at 1HR model interval, 2nd element is 3HR, etc.

            final double arbitraryBigNumber = 300.0;

            for(int i = 0; i < minPrecipOfAreas.length; i++)
            {
                minPrecipOfAreas[i] = arbitraryBigNumber; //initialize with a great number
            }

            
            
            for(int areaIdIndex = 0; areaIdIndex < _areaIdArray.length; areaIdIndex++)
            {
                final String areaId = _areaIdArray[areaIdIndex];

                // for FogBugz 1616: Prepend logs with location info                
                                 
                if (areaIdIndex==0){
                	newPrependString = currentPrePendString+"("+areaId+")";                	
                }
                else{                	            	   
            	   newPrependString = HString.replaceSubstring(currentPrePendString, _areaIdArray[areaIdIndex-1], _areaIdArray[areaIdIndex]);
                }
                
                // only add another id if not equal to the default id                               
                if(! areaId.equals(this._dischargeTslocationIdString)){	                
                	((Diagnostics)_logger).setPrependStr(newPrependString);
	                 currentPrePendString= newPrependString;
                }
                // end for FogBugz 1616
                
                final List<FfgRainfallRunoffCurve> listOfCurvesForArea = rainFallRunoffCurveMap.get(areaId);

                for(int durationIndex = 0; durationIndex < modelIntervalArray.length; durationIndex++)
                {
                    int changedDurationIndex; //1, 0, 2, 3, etc. So 3HR ffg is calculated first
                    if(durationIndex == 0)
                    {
                        changedDurationIndex = 1;
                    }
                    else if(durationIndex == 1)
                    {
                        changedDurationIndex = 0;
                    }
                    else
                    {
                        changedDurationIndex = durationIndex;
                    }

                    double ffg;

                    //check if the magic THRESH-R is negative. If so, it is the percent of 3HR ffg value.
                    if(threshRArrayInInch[changedDurationIndex] < 0.01)
                    {
                        ffg = -0.01 * threshRArrayInInch[changedDurationIndex] * ffgArray[1];
                        //ffgArray[1] has been calculated during first looping

                        _logger.log(Logger.DEBUG, "Unit hydrograph peak is negative["
                            + threshRArrayInInch[changedDurationIndex] + "] for the duration index[" + durationIndex
                            + "]. Its ffg is set to percent of 3 HR ffg[" + ffgArray[1] + "]");
                    }
                    else
                    {
                        //get ffg value
                        final FfgRainfallRunoffCurve rainFallRunOffCurve = listOfCurvesForArea.get(changedDurationIndex);

                        ffg = getFfgValueAtDurationIndex(rainFallRunOffCurve,
                                                         threshRArrayInInch[changedDurationIndex],
                                                         changedDurationIndex);
                    }

                    if(minPrecipOfAreas[changedDurationIndex] > ffg)
                    {
                        minPrecipOfAreas[changedDurationIndex] = ffg;
                    }

                    if(_areaIdArray.length > 1)
                    {//multiple areas
                        ffgArray[changedDurationIndex] += ffg * _ffhParams.getFfgAreaWeightArray()[areaIdIndex];
                    }
                    else
                    {
                        ffgArray[changedDurationIndex] = ffg;
                    }

                    sumPrecipOfAreas[changedDurationIndex] += ffg;

                }//end inner loop, finished one area of all durations

            }//end outer loop, one area
            
            //for FogBugz 1616 reset prepend string
            ((Diagnostics)_logger).setPrependStr(originalPrePendString);
           
            //applies weight technique
            _ffhParams.applyWeightTechnique(ffgArray, minPrecipOfAreas, sumPrecipOfAreas);

            //applies max/min limits to guidance
            _ffhParams.applyMinMax(ffgArray);

            //check for decreasing FFG values
            _ffhParams.applyUserControlDecreasingFfgValues(ffgArray);
        }
        else
        {//skip computation, all ffg values are set to -9.8(missing value)

            for(int i = 0; i < ffgArray.length; i++)
            {
                ffgArray[i] = FfgConstants.FFG_MISSING_VALUE;
            }
        }

      
        
        //create one TS for one model interval and add it to result map. Each ts only has one time step.
        for(int i = 0; i < ffgArray.length; i++)
        {

            final long timeStepInLong = _runInfo.getRunLastObservationTimeLong() + modelIntervalArray[i]
                * Timer.ONE_HOUR;

            final RegularTimeSeries ffhTs = new RegularTimeSeries(timeStepInLong,
                                                                  timeStepInLong,
                                                                  modelIntervalArray[i],
                                                                  MeasuringUnit.inches,
                                                                  ffgArray[i]); //only has one time step
            ffhTs.setTimeSeriesType(FFHConstants.FFH_DATATYPE);
            ffhTs.setLocationId(_outputLocationId);
            ffhTs.setQualifierIds(Arrays.asList(_outputLocationId));

            super.addTStoResultMap(ffhTs);

        } //close for i loop

        // Generates output/outputs.xml
        super.writeOutputTimeseries();
        
        
    }

    /**
     * Parse FFH parameter file and input time series files(they were skipped in OHDFewsAdapter). Then call super class
     * runDriverValidation. Initialize variables, particularly some variables to avoid repeating that within a loop.
     * Retrieve {@link #_dischargeTs}, which is always present no matter it is used or not.
     */
    @Override
    final protected void runDriverValidation() throws Exception
    {
        super.runDriverValidation(); //parse input time series file and parameter file. Check every ts in _tsList has enough data

        //intensity values are only needed when Runoff Adjust Option is 1 or 2
        _intensityArray = _ffhParams.getIntensityArray();

        if(_ffhParams.isDischargeTsNeeded())
        {//input discharge TS is needed

            if(_dischargeTs == null)
            {
                throw new Exception("Input discharge time series is missing.");
            }

            if(_ffhParams.useThresholdForFloodFlow())
            {//need thresholds element in the header, check its presence
                final String floodFlowId = _runInfo.getProperties().getProperty(FfgConstants.FLOOD_FLOW_ID_TAG);

                if(floodFlowId == null || floodFlowId.isEmpty())
                {
                    throw new Exception("In run info property, " + FfgConstants.FLOOD_FLOW_ID_TAG
                        + " entry is missing. Run info property= " + _runInfo.getProperties().toString());
                }

                final double floodFlowInCFS = _dischargeTs.getThreshold(floodFlowId).getValue(MeasuringUnit.cfs);
                /*
                 * this getter will throw an exception if not find the corresponding threshold. When thresholds in QINE
                 * header is missing value(-999.0) or negative value, skip model computation. Default is not skipping
                 */

                _logger.log(Logger.DEBUG, "Using threshold from input discharge ts header as flood flow. It is: "
                    + floodFlowInCFS + "CFS");

                if(floodFlowInCFS < 0.0)
                {
                    _skipComputation = true;
                    _logger.log(Logger.WARNING, "Flood flow at stage(" + floodFlowInCFS
                        + ") is missing value or less than 0.0. Skip the computation.");
                }

                _ffhParams.setFlowAtFloodStageInCFS(floodFlowInCFS);
            }

            if(_ffhParams.adjustHighFlow())
            {//need discharge TS, trim it to the exact needed period
                _logger.log(Logger.DEBUG,
                            "Using input forecast discharge time series(" + _dischargeTs.getTimeSeriesType() + ").");

                _dischargeTs.trimTimeSeriesAtStartWithCheck(getSwitchFromObservedToForecastTime());

                _dischargeTs.trimTimeSeriesAtEndWithCheck(getComputationEndTime());
            }

        }
        else
        {
            _dischargeTs = null;
        }

        if(_ffhParams.isThreshRFromParams() == false)
        {//computation will need flood flow
            if(_ffhParams.getFlowAtFloodStage() == OHDConstants.MISSING_DATA)
            {
                throw new Exception("Flood flow has not been set.");
            }
        }
        /*
         * SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm");
         * sdf.setTimeZone(TimeZone.getTimeZone("GMT")); System.out.println("From FFHDriver Model state start at " +
         * sdf.format(getState().getDateTime()));
         */
    } //close runDriverValidation()

    /**
     * get ffg values(precip, unit of Inch) at this model interval. If running multiple areas, the returned ffg value
     * needs to be multiplied with the area weight(percentage).<br>
     * This method uses 3 possible mechanisms to get ffg values: 1) the most common one: use the magic threshR and the
     * method {@link #getFFHValueByInterpolation(double)}; 2)ffg value is just the intensity value; 3)ffg value is just
     * the threshR value. Which mechanism to use is determined by User Control and other parameters.
     * 
     * @param rainFallRunOffCurve - the Rainfall Runoff curve should have been established already.
     * @param thresholdRunOffInInch - the magic THRESH-R(unit of INCH) for this duration.
     */
    private double getFfgValueAtDurationIndex(final FfgRainfallRunoffCurve rainFallRunOffCurve,
                                              double thresholdRunOffInInch,
                                              final int durationIndex) throws Exception
    {
        if(_ffhParams.getRunOffAdjustOption() == FFG_RUNOFF_ADJUST_OPTION.ADJUST_RUNOFF_AS_1) //Adjust Runoff = 1
        {
            if(_intensityArray[durationIndex] >= 0.01 && _intensityArray[durationIndex] < 5.0)
            {
                _logger.log(Logger.DEBUG, "By the parameter " + FfgConstants.FFG_USER_CTRL_RUNOFF_ADJUST_TAG
                    + ", use  " + FfgConstants.FFG_INTENSITY_TAGS[durationIndex] + " as factor applied to runoff.");

                thresholdRunOffInInch *= _intensityArray[durationIndex]; //rotot
            }
        }
        else if(_ffhParams.getRunOffAdjustOption() == FFG_RUNOFF_ADJUST_OPTION.USE_FIELDS_AS_FFG_AS_2) //Adjust Runoff = 2
        {
            //380
            _logger.log(Logger.DEBUG, "By the parameter " + FfgConstants.FFG_USER_CTRL_RUNOFF_ADJUST_TAG + ", use  "
                + FfgConstants.FFG_INTENSITY_TAGS[durationIndex] + " as precip(ffg). totprecip = "
                + _intensityArray[durationIndex]);

            return _intensityArray[durationIndex];
        }
        else if(_ffhParams.getRunOffAdjustOption() == FFG_RUNOFF_ADJUST_OPTION.USE_THRESHOLD_RUNOFF_AS_FFG_AS_3) //Adjust Runoff = 3
        {
            //380
            _logger.log(Logger.DEBUG, "By the parameter " + FfgConstants.FFG_USER_CTRL_RUNOFF_ADJUST_TAG
                + ", use threshold runoff as precip(ffg). totprecip = " + thresholdRunOffInInch);

            return thresholdRunOffInInch;
        }

        //no block for Adjust Runoff = 0, falls to 360 code

//        several scenarios reach here: getUserControlRunoffAdjust() is false OR getUserControlRunoffAdjust() is true AND Runoff Adjust Option is 1 or 0

        //360
        final double runoffFromImperviousArea = thresholdRunOffInInch * _ffhParams.getFfgPercentImpervious();//pim
        final double runoffFromPerviousArea = thresholdRunOffInInch * (1.0 - _ffhParams.getFfgPercentImpervious()); //rop, will be applied to the curve

        //calcp(ffgid, idr, fx, runoffFromPreviousArea, pp, istat);
        final double pp = rainFallRunOffCurve.getFFGValueByInterpolation(runoffFromPerviousArea); //unit of INCH

        _logger.log(Logger.DEBUG, " in calcp - tro= " + runoffFromPerviousArea + " ffg=" + pp);
        _logger.log(Logger.DEBUG, " in cphed after calcp - idr= " + String.valueOf(durationIndex + 1) + "  rop="
            + runoffFromPerviousArea + "  pp=" + pp);

        final double ffg = pp + runoffFromImperviousArea;

        final double ppchk = 20.0;

        if(pp > ppchk)
        {
            _logger.log(Logger.WARNING, "WARNING: precip exceeds " + ppchk + " and low runoff: Head_id = "
                + _outputLocationId + " dur=" + (durationIndex + 1) + " pp=" + pp + " tro=" + thresholdRunOffInInch);
        }

        //380
        _logger.log(Logger.DEBUG, "totprecip = " + ffg);

        _logger.log(Logger.DEBUG, " in cphed - idr= " + String.valueOf(durationIndex + 1) + " tro(idr)= "
            + thresholdRunOffInInch + " hinten(idr)= " + _intensityArray[durationIndex] + " rotot= "
            + thresholdRunOffInInch + " pcipmv=" + _ffhParams.getFfgPercentImpervious());

        return ffg;
    }

}
