package ohd.hseb.ohdutilities.ffg.ffh;

import ohd.hseb.ohdutilities.ffg.FfgConstants;
import ohd.hseb.ohdutilities.ffg.FfgParameters;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.OHDUtilities;
import ohd.hseb.util.fews.ParameterType.Row;
import ohd.hseb.util.fews.ParameterType.Table;

public class FFHParameters extends FfgParameters
{
    private boolean _useThresholdForFloodq;
    private double _flowAtFloodStageInCFS = OHDConstants.MISSING_DATA; //initialized to -999.0, for checking it has been set or not later 

    public FFHParameters()
    {
        super();
    }

    /**
     * Called during parsing input/paramsFFH.xml, OHDFewsAdapter.defineModelRunInfoAndSetupModel() method.
     */

    @Override
    protected void extractValuesFromMapFFG() throws Exception
    {
        super.extractValuesFromMapFFG();

        //FFH_WEIGHT_AREA_AND_BASIN_ID_PAIR_TAG
        final Table ffhWeigthAreaBasinIdtable = super.getTableDataParameter(FFHConstants.FFH_WEIGHT_AND_AREA_ID_PAIR_TAG);

        if(ffhWeigthAreaBasinIdtable.getRow() != null)
        {
            _ffgAreaWeightArray = new double[ffhWeigthAreaBasinIdtable.getRow().size()];
            _ffgAreaIdArray = new String[_ffgAreaWeightArray.length];
            int i = 0;
            for(final Row myRow: ffhWeigthAreaBasinIdtable.getRow())
            {
                _ffgAreaWeightArray[i] = Double.valueOf(myRow.getA()) / 100.0; //convert to decimal number
                _ffgAreaIdArray[i] = myRow.getB();
                i++;
            }
        }

        //check each group's parameter is present
        for(int i = 0; i < _ffgAreaIdArray.length; i++)
        {
            if(_groupParametersMap.containsKey(_ffgAreaIdArray[i]) == false)
            {
                throw new Exception("The parameter for the area id[" + _ffgAreaIdArray[i] + "] is missing.");
            }
        }

        //get the FFG Duration
        final String durStr = super.getStringDataParameter(_groupParametersMap.get(_ffgAreaIdArray[0]).getId(),
                                                           FfgConstants.FFG_DURATION_TAG);

        //now set model interval array
        super.setModelDurationInterval(durStr);

        if(isThreshRFromParams() == false)
        {//THRESH-R is calculated

            _useThresholdForFloodq = super.getBooleanDataParameter(FFHConstants.FFH_USE_THRESHOLD_FOR_FLOOD_FLOW_TAG);

            if(_useThresholdForFloodq == false)
            {//optional parameter, needed when _useThresholdForFloodq == false
                setFlowAtFloodStageInCFS(super.getDoubleDataParameter(FFHConstants.FFH_FLOW_AT_FLOOD_STAGE_TAG));
            }
        }

    }

    /**
     * Returns a double array with length equal to {@link #_modelIntervalArray}. This array holds UHG peak flows(unit of
     * CFS).
     * 
     * @throws Exception
     */
    public double[] getUHGPeakArrayInCfsPerInch() throws Exception
    {
        final double[] upk = new double[_modelIntervalArray.length];

        for(int i = 0; i < upk.length; i++)
        {
            upk[i] = super.getDoubleDataParameter(FFHConstants.FFH_UNIT_HG_PEAK_IN_CFS_PER_INCH_TAGS[i]);
        }

        return upk;

    }

    /**
     * Returns a double array with length equal to {@link #_modelIntervalArray}. This array holds threshold runoff(unit
     * of Inch). The values have already been divided by 100.0.
     * 
     * @throws Exception
     */
    public double[] getThreshRunOffArrayInInch() throws Exception
    {
        final double[] thresholdRunoffArray = new double[_modelIntervalArray.length];

        for(int i = 0; i < thresholdRunoffArray.length; i++)
        {
            thresholdRunoffArray[i] = super.getDoubleDataParameter(FfgConstants.THRESHOLD_RUNOFF_IN_INCHES_TAGS[i]);
        }

        return thresholdRunoffArray;

    }

    /**
     * Apply min/max algorithm to the ffg values(array): if it is less than the minimal values, change it to the minimal
     * value; if it is greater than the maximal value, change it to the maximal value. If minimal value is negative,
     * re-set ffg value to {@link FfgConstants#FFG_MISSING_VALUE}.
     * 
     * @param ffg - a double array containing ffg values. Its values could be modified by the method.
     * @throws Exception
     */
    void applyMinMax(final double[] ffg) throws Exception
    {
        double minValue = 0.0; //unit INCH
        double maxValue = 0.0; //unit INCH

        for(int i = 0; i < ffg.length; i++)
        {

            final Table ffhMaxMinValuesTable = super.getTableDataParameter(FfgConstants.FFG_MIN_MAX_VALUES_TAGS[i]);
            if(ffhMaxMinValuesTable != null && ffhMaxMinValuesTable.getRow() != null)
            {
                final Row row = ffhMaxMinValuesTable.getRow().get(0);

                minValue = Double.valueOf(row.getA());
                maxValue = Double.valueOf(row.getB());
            }

            //MIN-MAX algorithm
            if(minValue < 0.0)
            {
                ffg[i] = FfgConstants.FFG_MISSING_VALUE;
            }
            else
            { //userMaxValue >= 0.0

                if(ffg[i] > maxValue)
                {
                    _logger.log(Logger.WARNING, "Applies user max value to ffg value at interval of "
                        + OHDConstants.FFG_MODEL_1_3_6_12_24_INTERVAL_ARRAY[i] + "HR: Original ffg value="
                        + OHDUtilities.getFortranPrecison(ffg[i]) + "(INCH); max value=" + maxValue + "(INCH)");

                    ffg[i] = maxValue;
                }
                else if(ffg[i] < minValue)
                {
                    _logger.log(Logger.WARNING, "Applies user minimal value to ffg value at interval of "
                        + OHDConstants.FFG_MODEL_1_3_6_12_24_INTERVAL_ARRAY[i] + "HR: Original ffg value="
                        + OHDUtilities.getFortranPrecison(ffg[i]) + "(INCH); minimal value=" + minValue + "(INCH)");

                    ffg[i] = minValue;
                }
            }

        } //close for loop

    }

    /**
     * Check for decreasing FFG values: when the next value < the previous value and User Control
     * {@link FFHConstants#FFH_USER_CTRL_CHECK_DECREASING_FFG_TAG} == false(default value), send warning message only;
     * if the next value < the previous value and User Control value is true, set it equal to the previous value.
     */
    void applyUserControlDecreasingFfgValues(final double[] hffg)
    {

        for(int i = 1; i < _modelIntervalArray.length; i++)
        {

            if(hffg[i] < hffg[i - 1])
            {
                if(isFfgUserCtrlCheckDecreasingFFG() == false)
                {
                    _logger.log(Logger.WARNING, "WARNING: FFG value for " + _modelIntervalArray[i] + " HR duration("
                        + OHDUtilities.getFortranPrecison(hffg[i]) + ") is less than " + _modelIntervalArray[i - 1]
                        + " HR duration(" + OHDUtilities.getFortranPrecison(hffg[i - 1]) + ").");
                }
                else
                {
                    hffg[i] = hffg[i - 1];

                    _logger.log(Logger.DEBUG, "after decreasing values check - hffg(" + _modelIntervalArray[i]
                        + " HR)=" + OHDUtilities.getFortranPrecison(hffg[i]));

                }
            }
        } //close for loop

    }

    /**
     * Sum of weights (wts) determines weighting technique used: 0.98 <= wts <= 1.02 - use weighted p wts < 0.0 - use
     * _smallest p wts = 0.0 and other use average p
     */
    void applyWeightTechnique(final double[] ffg, final double[] minPrecipOfAreas, final double[] sumPrecipOfAreas)
    {
        //get the sum of each area percentage weight, which is uded to determine the average method
        double wtSum = 0.0;

        for(final double wt: _ffgAreaWeightArray)
        {
            wtSum += wt;
        }

        if(wtSum >= 0.98 && wtSum <= 1.02)
        {
            //do nothing, _hffg already has the weighted values
        }
        else if(wtSum < 0.0)
        {// use lowest (minimum) value
            for(int j = 0; j < _modelIntervalArray.length; j++)
            {
                ffg[j] = minPrecipOfAreas[j];
            }
        }
        else
        {// use average value
            for(int j = 0; j < _modelIntervalArray.length; j++)
            {
                _logger.log(Logger.DEBUG, "sumP[" + j + "]=" + sumPrecipOfAreas[j] + " numgroup = "
                    + _ffgAreaIdArray.length);
                ffg[j] = sumPrecipOfAreas[j] / _ffgAreaIdArray.length;
            }
        }

    } //close method

    /**
     * This boolean is the replacement for legacy rating curve id, which is blank or "none" or "nonenone" -- not use
     * thresholds, use {@link #_flowAtFloodStageInCFS}; otherwise, use threshold.
     */
    boolean useThresholdForFloodFlow()
    {
        return _useThresholdForFloodq;
    }

    /**
     * optional parameter, needed when _useThresholdForFloodq == false
     */
    public double getFlowAtFloodStage()
    {
        return _flowAtFloodStageInCFS;
    }

    void setFlowAtFloodStageInCFS(final double floodFlowInCFS)
    {
        _flowAtFloodStageInCFS = floodFlowInCFS;
    }

    /**
     * If the group parameter "THRESHOLD_RUNOFF_1_HR_IN_INCHES" etc exists, return true -- THRESH-R array will be
     * directly from the parameters; else if the group parameter "FFH_UNIT_HG_PEAK_1_HR_IN_CFS_PER_INCH" etc exists,
     * return false -- THRESH-R array will be calculated:<br>
     * threshold runoff = flood flow / unit hydrograph peak -- static formula or<br>
     * threshold runoff = (flood flow - max forecast flow) / unit hydrograph peak -- dynamic formula<br>
     * Note: if true, a lot of parameters are not be needed, like HIGH_FLOW_ADJUST_OPTION, useThresholdForFloodFlow(),
     * getFlowAtFloodStage()
     */
    public boolean isThreshRFromParams() throws Exception
    {
        if(isParamExisting(FfgConstants.THRESHOLD_RUNOFF_1_HR_IN_INCHES_TAG))
        {
            return true;
        }
        else if(isParamExisting(FFHConstants.FFH_UNIT_HG_PEAK_1_HR_IN_CFS_PER_INCH_TAG))
        {
            return false;
        }
        else
        {
            throw new Exception("One group parameters THRESHOLD_RUNOFF_1_HR_IN_INCHES etc or FFH_UNIT_HG_PEAK_1_HR_IN_CFS_PER_INCH etc must exist.");
        }
    }

    /**
     * When THRESH-R are not directly from the parameter xml file, there are two scenarios need discharge TS: one is
     * using thresholds element in the header as flood flow; another is adjusting flood flow by the base flow(i.e.
     * "max forecast flow").
     */
    boolean isDischargeTsNeeded() throws Exception
    {
        if(isThreshRFromParams() == true)
        {
            return false; //THRESH-R directly from parameter xml files, no need of discharge TS
        }

        /*
         * if reach here, isThreshRFromParams() == false, which means: THRESH-R is calculated by: threshold runoff =
         * flood flow / unit hydrograph peak or runoff = (flood flow - max forecast flow) / unit hydrograph peak.
         * Bottomline: need flood flow
         */

        if(useThresholdForFloodFlow())
        {//use <thresholds> as flood flow
            return true;
        }

        if(super.adjustHighFlow())
        {//need discharge TS to figure out "max forecast flow"
            return true;
        }

        //all else:
        return false;

    }

}
